// Copyright (C) 2021 The Qt Company Ltd.
// Copyright (C) 2013 John Layt <jlayt@kde.org>
// Copyright © 2004-2023 Unicode, Inc.
// SPDX-License-Identifier: Unicode-3.0

#ifndef QTIMEZONEPRIVATE_DATA_P_H
#define QTIMEZONEPRIVATE_DATA_P_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists for the convenience
// of internal files.  This header file may change from version to version
// without notice, or even be removed.
//
// We mean it.
//

#include <QtCore/private/qglobal_p.h>
#include "qbytearrayview.h"
#include "qstring.h"

QT_REQUIRE_CONFIG(timezone);
QT_BEGIN_NAMESPACE

namespace QtTimeZoneCldr {
/*
    Recognized UTC-offset zones and CLDR-derived data on timezone IDs.

    The UTC-offset zone table is provided for generic UTC±HH:mm format time
    zones. The UTC backend can support arbitrary offsets in seconds, but only
    advertises a limited repertoire of offsets as "available" in the normal
    sense.

    Windows Zone ID support is included in the default base class, QTZP, so can
    be used on all platforms, since an app running on Linux may need to
    communicate with a Windows Outlook server. These tables can also be used to
    look up Region Codes and UTC Offsets on platforms whose backends don't
    directly support them. For example, Darwin does not support
    availableTimeZones() filtering by region or offset. This table is
    auto-generated from the CLDR supplemental/windowsZones.xml data file, with
    IDs converted from CLDR-stable to IANA-current using bcp47/timezone.xml
    data.

    Please do not edit this data directly. See the generated section for details
    of its last update and how to update it.
*/

struct AliasData
{
    // Table is sorted on aliasId(), then on ianaId().
    // Indices in alias table, not IANA (group) table
    quint16 aliasIdIndex;
    quint16 ianaIdIndex;
    // Values for the actual IDs:
    constexpr QByteArrayView aliasId() const;
    constexpr QByteArrayView ianaId() const;
};

// FIXME: rename ianaIdData[] to ianaListData[], aliasIdData[] to ianaIdData[]
struct ZoneData
{
    // Keys (table is sorted in Windows ID, then on territory enum value):
    quint16 windowsIdKey;      // Windows ID sequence number
    quint16 territory;         // QLocale::Territory, AnyTerritory means No Territory
    // Values for this Windows zone and territory:
    quint16 ianaIdIndex;       // Index in ianaIdData of space-joined IANA IDs
    constexpr QLatin1StringView id() const; // Space-joined list of IANA IDs
    constexpr auto ids() const { return id().tokenize(u' '); } // Iterate IANA IDs
};

struct WindowsData
{
    // Table is sorted on key and this puts the windowsId()s in ascending order.
    quint16 windowsIdKey;      // Windows ID sequence number
    quint16 windowsIdIndex;    // Index of Windows ID in windowsIdData
    // Values for this Windows zone:
    quint16 ianaIdIndex;       // Index in ianaIdData of space-joined IANA IDs
    qint32 offsetFromUtc;      // Standard Time Offset from UTC, used for quick look-ups
    // FIXME: ianaIdIndex is actually always a single IANA ID, so re-route to alias table
    constexpr QByteArrayView windowsId() const;
    constexpr QByteArrayView ianaId() const; // Space-joined list of IANA IDs
};

struct UtcData
{
    quint16 ianaIdIndex;       // Index in ianaIdData of space-joined IANA IDs
    qint32 offsetFromUtc;      // Offset form UTC in seconds
    constexpr QByteArrayView id() const; // Space-joined list of IANA IDs
};

// GENERATED PART STARTS HERE

/*
    This part of the file was generated on 2024-05-31 from the
    Common Locale Data Repository v45

    http://www.unicode.org/cldr/

    Do not edit this section: instead regenerate it using
    cldr2qlocalexml.py and qlocalexml2cpp.py on updated (or
    edited) CLDR data; see qtbase/util/locale_database/.
*/

// Alias ID Index, Alias ID Index
static constexpr AliasData aliasMappingTable[] = {
    {      0,    14 }, // Africa/Asmera -> Africa/Asmara
    {     28,    44 }, // Africa/Timbuktu -> Africa/Bamako
    {     58,    91 }, // America/Argentina/ComodRivadavia -> America/Argentina/Catamarca
    {    119,   132 }, // America/Atka -> America/Adak
    {    145,   166 }, // America/Buenos_Aires -> America/Argentina/Buenos_Aires
    {    197,    91 }, // America/Catamarca -> America/Argentina/Catamarca
    {    215,   237 }, // America/Coral_Harbour -> America/Atikokan
    {    254,   270 }, // America/Cordoba -> America/Argentina/Cordoba
    {    296,   313 }, // America/Ensenada -> America/Tijuana
    {    329,   348 }, // America/Fort_Wayne -> America/Indiana/Indianapolis
    {    377,   393 }, // America/Godthab -> America/Nuuk
    {    406,   348 }, // America/Indianapolis -> America/Indiana/Indianapolis
    {    427,   441 }, // America/Jujuy -> America/Argentina/Jujuy
    {    465,   481 }, // America/Knox_IN -> America/Indiana/Knox
    {    502,   521 }, // America/Louisville -> America/Kentucky/Louisville
    {    549,   565 }, // America/Mendoza -> America/Argentina/Mendoza
    {    591,   608 }, // America/Montreal -> America/Toronto
    {    624,   608 }, // America/Nipigon -> America/Toronto
    {    640,   660 }, // America/Pangnirtung -> America/Iqaluit
    {    676,   695 }, // America/Porto_Acre -> America/Rio_Branco
    {    714,   734 }, // America/Rainy_River -> America/Winnipeg
    {    751,   270 }, // America/Rosario -> America/Argentina/Cordoba
    {    767,   313 }, // America/Santa_Isabel -> America/Tijuana
    {    788,   805 }, // America/Shiprock -> America/Denver
    {    820,   608 }, // America/Thunder_Bay -> America/Toronto
    {    840,   855 }, // America/Virgin -> America/St_Thomas
    {    873,   893 }, // America/Yellowknife -> America/Edmonton
    {    910,   932 }, // Antarctica/South_Pole -> Pacific/Auckland
    {    949,   964 }, // Asia/Ashkhabad -> Asia/Ashgabat
    {    978,   992 }, // Asia/Calcutta -> Asia/Kolkata
    {   1005,  1020 }, // Asia/Chongqing -> Asia/Shanghai
    {   1034,  1020 }, // Asia/Chungking -> Asia/Shanghai
    {   1049,  1060 }, // Asia/Dacca -> Asia/Dhaka
    {   1071,  1020 }, // Asia/Harbin -> Asia/Shanghai
    {   1083,  1097 }, // Asia/Istanbul -> Europe/Istanbul
    {   1113,  1126 }, // Asia/Kashgar -> Asia/Urumqi
    {   1138,  1152 }, // Asia/Katmandu -> Asia/Kathmandu
    {   1167,  1178 }, // Asia/Macao -> Asia/Macau
    {   1189,  1202 }, // Asia/Rangoon -> Asia/Yangon
    {   1214,  1226 }, // Asia/Saigon -> Asia/Ho_Chi_Minh
    {   1243,  1257 }, // Asia/Tel_Aviv -> Asia/Jerusalem
    {   1272,  1284 }, // Asia/Thimbu -> Asia/Thimphu
    {   1297,  1316 }, // Asia/Ujung_Pandang -> Asia/Makassar
    {   1330,  1346 }, // Asia/Ulan_Bator -> Asia/Ulaanbaatar
    {   1363,  1379 }, // Atlantic/Faeroe -> Atlantic/Faroe
    {   1394,  1413 }, // Atlantic/Jan_Mayen -> Arctic/Longyearbyen
    {   1433,  1447 }, // Australia/ACT -> Australia/Sydney
    {   1464,  1447 }, // Australia/Canberra -> Australia/Sydney
    {   1483,  1500 }, // Australia/Currie -> Australia/Hobart
    {   1517,  1531 }, // Australia/LHI -> Australia/Lord_Howe
    {   1551,  1447 }, // Australia/NSW -> Australia/Sydney
    {   1565,  1581 }, // Australia/North -> Australia/Darwin
    {   1598,  1619 }, // Australia/Queensland -> Australia/Brisbane
    {   1638,  1654 }, // Australia/South -> Australia/Adelaide
    {   1673,  1500 }, // Australia/Tasmania -> Australia/Hobart
    {   1692,  1711 }, // Australia/Victoria -> Australia/Melbourne
    {   1731,  1746 }, // Australia/West -> Australia/Perth
    {   1762,  1783 }, // Australia/Yancowinna -> Australia/Broken_Hill
    {   1805,   695 }, // Brazil/Acre -> America/Rio_Branco
    {   1817,  1834 }, // Brazil/DeNoronha -> America/Noronha
    {   1850,  1862 }, // Brazil/East -> America/Sao_Paulo
    {   1880,  1892 }, // Brazil/West -> America/Manaus
    {   1907,  1923 }, // Canada/Atlantic -> America/Halifax
    {   1939,   734 }, // Canada/Central -> America/Winnipeg
    {   1954,  1979 }, // Canada/East-Saskatchewan -> America/Regina
    {   1994,   608 }, // Canada/Eastern -> America/Toronto
    {   2009,   893 }, // Canada/Mountain -> America/Edmonton
    {   2025,  2045 }, // Canada/Newfoundland -> America/St_Johns
    {   2062,  2077 }, // Canada/Pacific -> America/Vancouver
    {   2095,  1979 }, // Canada/Saskatchewan -> America/Regina
    {   2115,  2128 }, // Canada/Yukon -> America/Whitehorse
    {   2147,  2165 }, // Chile/Continental -> America/Santiago
    {   2182,  2201 }, // Chile/EasterIsland -> Pacific/Easter
    {   2216,  2221 }, // Cuba -> America/Havana
    {   2236,  2240 }, // EST -> Etc/GMT+5
    {   2250,  2256 }, // Egypt -> Africa/Cairo
    {   2269,  2274 }, // Eire -> Europe/Dublin
    {   2288,  2298 }, // Etc/GMT+0 -> Etc/GMT
    {   2306,  2298 }, // Etc/GMT-0 -> Etc/GMT
    {   2316,  2298 }, // Etc/GMT0 -> Etc/GMT
    {   2325,  2298 }, // Etc/Greenwich -> Etc/GMT
    {   2339,  2347 }, // Etc/UCT -> Etc/UTC
    {   2355,  2347 }, // Etc/Universal -> Etc/UTC
    {   2369,  2347 }, // Etc/Zulu -> Etc/UTC
    {   2378,  2393 }, // Europe/Belfast -> Europe/London
    {   2407,  2419 }, // Europe/Kiev -> Europe/Kyiv
    {   2431,  2446 }, // Europe/Nicosia -> Asia/Nicosia
    {   2459,  2475 }, // Europe/Tiraspol -> Europe/Chisinau
    {   2491,  2419 }, // Europe/Uzhgorod -> Europe/Kyiv
    {   2507,  2419 }, // Europe/Zaporozhye -> Europe/Kyiv
    {   2525,  2393 }, // GB -> Europe/London
    {   2528,  2393 }, // GB-Eire -> Europe/London
    {   2536,  2298 }, // GMT -> Etc/GMT
    {   2540,  2298 }, // GMT+0 -> Etc/GMT
    {   2546,  2298 }, // GMT-0 -> Etc/GMT
    {   2552,  2298 }, // GMT0 -> Etc/GMT
    {   2557,  2298 }, // Greenwich -> Etc/GMT
    {   2567,  2571 }, // HST -> Etc/GMT+10
    {   2582,  2591 }, // Hongkong -> Asia/Hong_Kong
    {   2606,  2614 }, // Iceland -> Atlantic/Reykjavik
    {   2633,  2638 }, // Iran -> Asia/Tehran
    {   2650,  1257 }, // Israel -> Asia/Jerusalem
    {   2657,  2665 }, // Jamaica -> America/Jamaica
    {   2681,  2687 }, // Japan -> Asia/Tokyo
    {   2698,  2708 }, // Kwajalein -> Pacific/Kwajalein
    {   2726,  2732 }, // Libya -> Africa/Tripoli
    {   2747,  2751 }, // MST -> Etc/GMT+7
    {   2761,   313 }, // Mexico/BajaNorte -> America/Tijuana
    {   2778,  2793 }, // Mexico/BajaSur -> America/Mazatlan
    {   2810,  2825 }, // Mexico/General -> America/Mexico_City
    {   2845,   932 }, // NZ -> Pacific/Auckland
    {   2848,  2856 }, // NZ-CHAT -> Pacific/Chatham
    {   2872,   805 }, // Navajo -> America/Denver
    {   2879,  1020 }, // PRC -> Asia/Shanghai
    {   2883,  2901 }, // Pacific/Enderbury -> Pacific/Kanton
    {   2916,  2933 }, // Pacific/Johnston -> Pacific/Honolulu
    {   2950,  2965 }, // Pacific/Ponape -> Pacific/Pohnpei
    {   2981,  2995 }, // Pacific/Samoa -> Pacific/Pago_Pago
    {   3013,  3026 }, // Pacific/Truk -> Pacific/Chuuk
    {   3040,  3026 }, // Pacific/Yap -> Pacific/Chuuk
    {   3052,  3059 }, // Poland -> Europe/Warsaw
    {   3073,  3082 }, // Portugal -> Europe/Lisbon
    {   3096,  3100 }, // ROC -> Asia/Taipei
    {   3112,  3116 }, // ROK -> Asia/Seoul
    {   3127,  3137 }, // Singapore -> Asia/Singapore
    {   3152,  1097 }, // Turkey -> Europe/Istanbul
    {   3159,  2347 }, // UCT -> Etc/UTC
    {   3163,  3173 }, // US/Alaska -> America/Anchorage
    {   3191,   132 }, // US/Aleutian -> America/Adak
    {   3203,  3214 }, // US/Arizona -> America/Phoenix
    {   3230,  3241 }, // US/Central -> America/Chicago
    {   3257,   348 }, // US/East-Indiana -> America/Indiana/Indianapolis
    {   3273,  3284 }, // US/Eastern -> America/New_York
    {   3301,  2933 }, // US/Hawaii -> Pacific/Honolulu
    {   3311,   481 }, // US/Indiana-Starke -> America/Indiana/Knox
    {   3329,  3341 }, // US/Michigan -> America/Detroit
    {   3357,   805 }, // US/Mountain -> America/Denver
    {   3369,  3380 }, // US/Pacific -> America/Los_Angeles
    {   3400,  3380 }, // US/Pacific-New -> America/Los_Angeles
    {   3415,  2995 }, // US/Samoa -> Pacific/Pago_Pago
    {   3424,  2347 }, // UTC -> Etc/UTC
    {   3428,  2347 }, // Universal -> Etc/UTC
    {   3438,  3443 }, // W-SU -> Europe/Moscow
    {   3457,  2347 }, // Zulu -> Etc/UTC
};

// Windows ID Key, Territory Enum, IANA ID Index
static constexpr ZoneData zoneDataTable[] = {
    {      1,     1,     0 }, // Afghanistan Standard Time / Afghanistan
    {      2,   248,    11 }, // Alaskan Standard Time / United States
    {      3,   248,   106 }, // Aleutian Standard Time / United States
    {      4,   193,   119 }, // Altai Standard Time / Russia
    {      5,    19,   132 }, // Arab Standard Time / Bahrain
    {      5,   127,   145 }, // Arab Standard Time / Kuwait
    {      5,   190,   157 }, // Arab Standard Time / Qatar
    {      5,   205,   168 }, // Arab Standard Time / Saudi Arabia
    {      5,   259,   180 }, // Arab Standard Time / Yemen
    {      6,     0,   190 }, // Arabian Standard Time / AnyTerritory
    {      6,   176,   200 }, // Arabian Standard Time / Oman
    {      6,   245,   212 }, // Arabian Standard Time / United Arab Emirates
    {      7,   113,   223 }, // Arabic Standard Time / Iraq
    {      8,    11,   236 }, // Argentina Standard Time / Argentina
    {      9,   193,   559 }, // Astrakhan Standard Time / Russia
    {     10,    26,   593 }, // Atlantic Standard Time / Bermuda
    {     10,    41,   610 }, // Atlantic Standard Time / Canada
    {     10,    95,   678 }, // Atlantic Standard Time / Greenland
    {     11,    15,   692 }, // AUS Central Standard Time / Australia
    {     12,    15,   709 }, // Aus Central W. Standard Time / Australia
    {     13,    15,   725 }, // AUS Eastern Standard Time / Australia
    {     14,    17,   762 }, // Azerbaijan Standard Time / Azerbaijan
    {     15,    95,   772 }, // Azores Standard Time / Greenland
    {     15,   188,   793 }, // Azores Standard Time / Portugal
    {     16,    32,   809 }, // Bahia Standard Time / Brazil
    {     17,    20,   823 }, // Bangladesh Standard Time / Bangladesh
    {     17,    27,   834 }, // Bangladesh Standard Time / Bhutan
    {     18,    22,   847 }, // Belarus Standard Time / Belarus
    {     19,   182,   860 }, // Bougainville Standard Time / Papua New Guinea
    {     20,    41,   881 }, // Canada Central Standard Time / Canada
    {     21,     0,   918 }, // Cape Verde Standard Time / AnyTerritory
    {     21,    43,   928 }, // Cape Verde Standard Time / Cape Verde
    {     22,    12,   948 }, // Caucasus Standard Time / Armenia
    {     23,    15,   961 }, // Cen. Australia Standard Time / Australia
    {     24,     0,  1002 }, // Central America Standard Time / AnyTerritory
    {     24,    24,  1012 }, // Central America Standard Time / Belize
    {     24,    59,  1027 }, // Central America Standard Time / Costa Rica
    {     24,    70,  1046 }, // Central America Standard Time / Ecuador
    {     24,    72,  1064 }, // Central America Standard Time / El Salvador
    {     24,    99,  1084 }, // Central America Standard Time / Guatemala
    {     24,   106,  1102 }, // Central America Standard Time / Honduras
    {     24,   168,  1122 }, // Central America Standard Time / Nicaragua
    {     25,     0,  1138 }, // Central Asia Standard Time / AnyTerritory
    {     25,     9,  1148 }, // Central Asia Standard Time / Antarctica
    {     25,    33,  1166 }, // Central Asia Standard Time / British Indian Ocean Territory
    {     25,    50,  1180 }, // Central Asia Standard Time / China
    {     25,   128,  1192 }, // Central Asia Standard Time / Kyrgyzstan
    {     26,    32,  1205 }, // Central Brazilian Standard Time / Brazil
    {     27,     3,  1241 }, // Central Europe Standard Time / Albania
    {     27,    64,  1255 }, // Central Europe Standard Time / Czechia
    {     27,   108,  1269 }, // Central Europe Standard Time / Hungary
    {     27,   157,  1285 }, // Central Europe Standard Time / Montenegro
    {     27,   207,  1302 }, // Central Europe Standard Time / Serbia
    {     27,   212,  1318 }, // Central Europe Standard Time / Slovakia
    {     27,   213,  1336 }, // Central Europe Standard Time / Slovenia
    {     28,    29,  1353 }, // Central European Standard Time / Bosnia and Herzegovina
    {     28,    60,  1369 }, // Central European Standard Time / Croatia
    {     28,   140,  1383 }, // Central European Standard Time / Macedonia
    {     28,   187,  1397 }, // Central European Standard Time / Poland
    {     29,     0,  1411 }, // Central Pacific Standard Time / AnyTerritory
    {     29,     9,  1422 }, // Central Pacific Standard Time / Antarctica
    {     29,   153,  1439 }, // Central Pacific Standard Time / Micronesia
    {     29,   166,  1470 }, // Central Pacific Standard Time / New Caledonia
    {     29,   214,  1485 }, // Central Pacific Standard Time / Solomon Islands
    {     29,   252,  1505 }, // Central Pacific Standard Time / Vanuatu
    {     30,     0,  1519 }, // Central Standard Time / AnyTerritory
    {     30,    41,  1527 }, // Central Standard Time / Canada
    {     30,   152,  1582 }, // Central Standard Time / Mexico
    {     30,   248,  1616 }, // Central Standard Time / United States
    {     31,   152,  1784 }, // Central Standard Time (Mexico) / Mexico
    {     32,   167,  1878 }, // Chatham Islands Standard Time / New Zealand
    {     33,    50,  1894 }, // China Standard Time / China
    {     33,   107,  1908 }, // China Standard Time / Hong Kong
    {     33,   139,  1923 }, // China Standard Time / Macao
    {     34,    61,  1934 }, // Cuba Standard Time / Cuba
    {     35,     0,  1949 }, // Dateline Standard Time / AnyTerritory
    {     36,     0,  1960 }, // E. Africa Standard Time / AnyTerritory
    {     36,     9,  1970 }, // E. Africa Standard Time / Antarctica
    {     36,    55,  1987 }, // E. Africa Standard Time / Comoros
    {     36,    67,  2001 }, // E. Africa Standard Time / Djibouti
    {     36,    74,  2017 }, // E. Africa Standard Time / Eritrea
    {     36,    77,  2031 }, // E. Africa Standard Time / Ethiopia
    {     36,   124,  2050 }, // E. Africa Standard Time / Kenya
    {     36,   141,  2065 }, // E. Africa Standard Time / Madagascar
    {     36,   151,  2085 }, // E. Africa Standard Time / Mayotte
    {     36,   215,  2100 }, // E. Africa Standard Time / Somalia
    {     36,   230,  2117 }, // E. Africa Standard Time / Tanzania
    {     36,   243,  2138 }, // E. Africa Standard Time / Uganda
    {     37,    15,  2153 }, // E. Australia Standard Time / Australia
    {     38,   154,  2191 }, // E. Europe Standard Time / Moldova
    {     39,    32,  2207 }, // E. South America Standard Time / Brazil
    {     40,    49,  2225 }, // Easter Island Standard Time / Chile
    {     41,     0,  2240 }, // Eastern Standard Time / AnyTerritory
    {     41,    18,  2248 }, // Eastern Standard Time / Bahamas
    {     41,    41,  2263 }, // Eastern Standard Time / Canada
    {     41,   248,  2295 }, // Eastern Standard Time / United States
    {     42,   152,  2461 }, // Eastern Standard Time (Mexico) / Mexico
    {     43,    71,  2476 }, // Egypt Standard Time / Egypt
    {     44,   193,  2489 }, // Ekaterinburg Standard Time / Russia
    {     45,    82,  2508 }, // Fiji Standard Time / Fiji
    {     46,     2,  2521 }, // FLE Standard Time / Aland Islands
    {     46,    36,  2538 }, // FLE Standard Time / Bulgaria
    {     46,    75,  2551 }, // FLE Standard Time / Estonia
    {     46,    83,  2566 }, // FLE Standard Time / Finland
    {     46,   131,  2582 }, // FLE Standard Time / Latvia
    {     46,   137,  2594 }, // FLE Standard Time / Lithuania
    {     46,   244,  2609 }, // FLE Standard Time / Ukraine
    {     47,    90,  2621 }, // Georgian Standard Time / Georgia
    {     48,    81,  2634 }, // GMT Standard Time / Faroe Islands
    {     48,   100,  2649 }, // GMT Standard Time / Guernsey
    {     48,   114,  2665 }, // GMT Standard Time / Ireland
    {     48,   115,  2679 }, // GMT Standard Time / Isle of Man
    {     48,   121,  2698 }, // GMT Standard Time / Jersey
    {     48,   188,  2712 }, // GMT Standard Time / Portugal
    {     48,   220,  2743 }, // GMT Standard Time / Spain
    {     48,   246,  2759 }, // GMT Standard Time / United Kingdom
    {     49,    95,  2773 }, // Greenland Standard Time / Greenland
    {     50,    37,  2786 }, // Greenwich Standard Time / Burkina Faso
    {     50,    89,  2805 }, // Greenwich Standard Time / Gambia
    {     50,    92,  2819 }, // Greenwich Standard Time / Ghana
    {     50,    95,  2832 }, // Greenwich Standard Time / Greenland
    {     50,   101,  2853 }, // Greenwich Standard Time / Guinea-Bissau
    {     50,   102,  2867 }, // Greenwich Standard Time / Guinea
    {     50,   109,  2882 }, // Greenwich Standard Time / Iceland
    {     50,   118,  2901 }, // Greenwich Standard Time / Ivory Coast
    {     50,   134,  2916 }, // Greenwich Standard Time / Liberia
    {     50,   145,  2932 }, // Greenwich Standard Time / Mali
    {     50,   149,  2946 }, // Greenwich Standard Time / Mauritania
    {     50,   196,  2964 }, // Greenwich Standard Time / Saint Helena
    {     50,   206,  2983 }, // Greenwich Standard Time / Senegal
    {     50,   209,  2996 }, // Greenwich Standard Time / Sierra Leone
    {     50,   233,  3012 }, // Greenwich Standard Time / Togo
    {     51,    63,  3024 }, // GTB Standard Time / Cyprus
    {     51,    94,  3052 }, // GTB Standard Time / Greece
    {     51,   192,  3066 }, // GTB Standard Time / Romania
    {     52,   104,  3083 }, // Haiti Standard Time / Haiti
    {     53,     0,  3106 }, // Hawaiian Standard Time / AnyTerritory
    {     53,    58,  3117 }, // Hawaiian Standard Time / Cook Islands
    {     53,    86,  3135 }, // Hawaiian Standard Time / French Polynesia
    {     53,   248,  3150 }, // Hawaiian Standard Time / United States
    {     54,   110,  3167 }, // India Standard Time / India
    {     55,   112,  3180 }, // Iran Standard Time / Iran
    {     56,   116,  3192 }, // Israel Standard Time / Israel
    {     57,   122,  3207 }, // Jordan Standard Time / Jordan
    {     58,   193,  3218 }, // Kaliningrad Standard Time / Russia
    {     59,   218,  3237 }, // Korea Standard Time / South Korea
    {     60,   135,  3248 }, // Libya Standard Time / Libya
    {     61,     0,  3263 }, // Line Islands Standard Time / AnyTerritory
    {     61,   125,  3274 }, // Line Islands Standard Time / Kiribati
    {     62,    15,  3293 }, // Lord Howe Standard Time / Australia
    {     63,   193,  3313 }, // Magadan Standard Time / Russia
    {     64,    49,  3326 }, // Magallanes Standard Time / Chile
    {     65,    86,  3347 }, // Marquesas Standard Time / French Polynesia
    {     66,   150,  3365 }, // Mauritius Standard Time / Mauritius
    {     66,   191,  3382 }, // Mauritius Standard Time / Reunion
    {     66,   208,  3397 }, // Mauritius Standard Time / Seychelles
    {     67,   132,  3409 }, // Middle East Standard Time / Lebanon
    {     68,   250,  3421 }, // Montevideo Standard Time / Uruguay
    {     69,   159,  3440 }, // Morocco Standard Time / Morocco
    {     69,   257,  3458 }, // Morocco Standard Time / Western Sahara
    {     70,     0,  3474 }, // Mountain Standard Time / AnyTerritory
    {     70,    41,  3482 }, // Mountain Standard Time / Canada
    {     70,   152,  3536 }, // Mountain Standard Time / Mexico
    {     70,   248,  3558 }, // Mountain Standard Time / United States
    {     71,   152,  3587 }, // Mountain Standard Time (Mexico) / Mexico
    {     72,    53,  3604 }, // Myanmar Standard Time / Cocos Islands
    {     72,   161,  3617 }, // Myanmar Standard Time / Myanmar
    {     73,   193,  3629 }, // N. Central Asia Standard Time / Russia
    {     74,   162,  3646 }, // Namibia Standard Time / Namibia
    {     75,   164,  3662 }, // Nepal Standard Time / Nepal
    {     76,     9,  3677 }, // New Zealand Standard Time / Antarctica
    {     76,   167,  3696 }, // New Zealand Standard Time / New Zealand
    {     77,    41,  3713 }, // Newfoundland Standard Time / Canada
    {     78,   172,  3730 }, // Norfolk Standard Time / Norfolk Island
    {     79,   193,  3746 }, // North Asia East Standard Time / Russia
    {     80,   193,  3759 }, // North Asia Standard Time / Russia
    {     81,   174,  3794 }, // North Korea Standard Time / North Korea
    {     82,   193,  3809 }, // Omsk Standard Time / Russia
    {     83,    49,  3819 }, // Pacific SA Standard Time / Chile
    {     84,     0,  3836 }, // Pacific Standard Time / AnyTerritory
    {     84,    41,  3844 }, // Pacific Standard Time / Canada
    {     84,   248,  3862 }, // Pacific Standard Time / United States
    {     85,   152,  3882 }, // Pacific Standard Time (Mexico) / Mexico
    {     86,   178,  3898 }, // Pakistan Standard Time / Pakistan
    {     87,   183,  3911 }, // Paraguay Standard Time / Paraguay
    {     88,   123,  3928 }, // Qyzylorda Standard Time / Kazakhstan
    {     89,    23,  3943 }, // Romance Standard Time / Belgium
    {     89,    65,  3959 }, // Romance Standard Time / Denmark
    {     89,    84,  3977 }, // Romance Standard Time / France
    {     89,   220,  3990 }, // Romance Standard Time / Spain
    {     90,   193,  4017 }, // Russia Time Zone 10 / Russia
    {     91,   193,  4036 }, // Russia Time Zone 11 / Russia
    {     92,   193,  4063 }, // Russia Time Zone 3 / Russia
    {     93,   193,  4077 }, // Russian Standard Time / Russia
    {     93,   244,  4104 }, // Russian Standard Time / Ukraine
    {     94,     0,  4122 }, // SA Eastern Standard Time / AnyTerritory
    {     94,     9,  4132 }, // SA Eastern Standard Time / Antarctica
    {     94,    32,  4169 }, // SA Eastern Standard Time / Brazil
    {     94,    80,  4248 }, // SA Eastern Standard Time / Falkland Islands
    {     94,    85,  4265 }, // SA Eastern Standard Time / French Guiana
    {     94,   223,  4281 }, // SA Eastern Standard Time / Suriname
    {     95,     0,  4300 }, // SA Pacific Standard Time / AnyTerritory
    {     95,    32,  4310 }, // SA Pacific Standard Time / Brazil
    {     95,    41,  4346 }, // SA Pacific Standard Time / Canada
    {     95,    45,  4363 }, // SA Pacific Standard Time / Cayman Islands
    {     95,    54,  4378 }, // SA Pacific Standard Time / Colombia
    {     95,    70,  4393 }, // SA Pacific Standard Time / Ecuador
    {     95,   119,  4411 }, // SA Pacific Standard Time / Jamaica
    {     95,   181,  4427 }, // SA Pacific Standard Time / Panama
    {     95,   184,  4442 }, // SA Pacific Standard Time / Peru
    {     96,     0,  4455 }, // SA Western Standard Time / AnyTerritory
    {     96,     8,  4465 }, // SA Western Standard Time / Anguilla
    {     96,    10,  4482 }, // SA Western Standard Time / Antigua and Barbuda
    {     96,    13,  4498 }, // SA Western Standard Time / Aruba
    {     96,    21,  4512 }, // SA Western Standard Time / Barbados
    {     96,    28,  4529 }, // SA Western Standard Time / Bolivia
    {     96,    32,  4544 }, // SA Western Standard Time / Brazil
    {     96,    34,  4597 }, // SA Western Standard Time / British Virgin Islands
    {     96,    41,  4613 }, // SA Western Standard Time / Canada
    {     96,    44,  4634 }, // SA Western Standard Time / Caribbean Netherlands
    {     96,    62,  4653 }, // SA Western Standard Time / Curacao
    {     96,    68,  4669 }, // SA Western Standard Time / Dominica
    {     96,    69,  4686 }, // SA Western Standard Time / Dominican Republic
    {     96,    96,  4708 }, // SA Western Standard Time / Grenada
    {     96,    97,  4724 }, // SA Western Standard Time / Guadeloupe
    {     96,   103,  4743 }, // SA Western Standard Time / Guyana
    {     96,   148,  4758 }, // SA Western Standard Time / Martinique
    {     96,   158,  4777 }, // SA Western Standard Time / Montserrat
    {     96,   189,  4796 }, // SA Western Standard Time / Puerto Rico
    {     96,   195,  4816 }, // SA Western Standard Time / Saint Barthelemy
    {     96,   197,  4838 }, // SA Western Standard Time / Saint Kitts and Nevis
    {     96,   198,  4855 }, // SA Western Standard Time / Saint Lucia
    {     96,   199,  4872 }, // SA Western Standard Time / Saint Martin
    {     96,   201,  4888 }, // SA Western Standard Time / Saint Vincent and Grenadines
    {     96,   211,  4907 }, // SA Western Standard Time / Sint Maarten
    {     96,   236,  4929 }, // SA Western Standard Time / Trinidad and Tobago
    {     96,   249,  4951 }, // SA Western Standard Time / United States Virgin Islands
    {     97,   200,  4969 }, // Saint Pierre Standard Time / Saint Pierre and Miquelon
    {     98,   193,  4986 }, // Sakhalin Standard Time / Russia
    {     99,   202,  5000 }, // Samoa Standard Time / Samoa
    {    100,   204,  5013 }, // Sao Tome Standard Time / Sao Tome and Principe
    {    101,   193,  5029 }, // Saratov Standard Time / Russia
    {    102,     0,  5044 }, // SE Asia Standard Time / AnyTerritory
    {    102,     9,  5054 }, // SE Asia Standard Time / Antarctica
    {    102,    39,  5071 }, // SE Asia Standard Time / Cambodia
    {    102,    51,  5087 }, // SE Asia Standard Time / Christmas Island
    {    102,   111,  5104 }, // SE Asia Standard Time / Indonesia
    {    102,   129,  5132 }, // SE Asia Standard Time / Laos
    {    102,   231,  5147 }, // SE Asia Standard Time / Thailand
    {    102,   255,  5160 }, // SE Asia Standard Time / Vietnam
    {    103,     0,  5177 }, // Singapore Standard Time / AnyTerritory
    {    103,    35,  5187 }, // Singapore Standard Time / Brunei
    {    103,   111,  5199 }, // Singapore Standard Time / Indonesia
    {    103,   143,  5213 }, // Singapore Standard Time / Malaysia
    {    103,   185,  5244 }, // Singapore Standard Time / Philippines
    {    103,   210,  5256 }, // Singapore Standard Time / Singapore
    {    104,     0,  5271 }, // South Africa Standard Time / AnyTerritory
    {    104,    30,  5281 }, // South Africa Standard Time / Botswana
    {    104,    38,  5297 }, // South Africa Standard Time / Burundi
    {    104,    57,  5314 }, // South Africa Standard Time / Congo - Kinshasa
    {    104,    76,  5332 }, // South Africa Standard Time / Eswatini
    {    104,   133,  5347 }, // South Africa Standard Time / Lesotho
    {    104,   142,  5361 }, // South Africa Standard Time / Malawi
    {    104,   160,  5377 }, // South Africa Standard Time / Mozambique
    {    104,   194,  5391 }, // South Africa Standard Time / Rwanda
    {    104,   216,  5405 }, // South Africa Standard Time / South Africa
    {    104,   260,  5425 }, // South Africa Standard Time / Zambia
    {    104,   261,  5439 }, // South Africa Standard Time / Zimbabwe
    {    105,   219,  5453 }, // South Sudan Standard Time / South Sudan
    {    106,   221,  5465 }, // Sri Lanka Standard Time / Sri Lanka
    {    107,   222,  5478 }, // Sudan Standard Time / Sudan
    {    108,   227,  5494 }, // Syria Standard Time / Syria
    {    109,   228,  5508 }, // Taipei Standard Time / Taiwan
    {    110,    15,  5520 }, // Tasmania Standard Time / Australia
    {    111,    32,  5558 }, // Tocantins Standard Time / Brazil
    {    112,     0,  5576 }, // Tokyo Standard Time / AnyTerritory
    {    112,   111,  5586 }, // Tokyo Standard Time / Indonesia
    {    112,   120,  5600 }, // Tokyo Standard Time / Japan
    {    112,   179,  5611 }, // Tokyo Standard Time / Palau
    {    112,   232,  5625 }, // Tokyo Standard Time / Timor-Leste
    {    113,   193,  5635 }, // Tomsk Standard Time / Russia
    {    114,   235,  5646 }, // Tonga Standard Time / Tonga
    {    115,   193,  5664 }, // Transbaikal Standard Time / Russia
    {    116,   239,  5675 }, // Turkey Standard Time / Turkey
    {    117,   241,  5691 }, // Turks And Caicos Standard Time / Turks and Caicos Islands
    {    118,   156,  5710 }, // Ulaanbaatar Standard Time / Mongolia
    {    119,   248,  5743 }, // US Eastern Standard Time / United States
    {    120,     0,  5818 }, // US Mountain Standard Time / AnyTerritory
    {    120,    41,  5828 }, // US Mountain Standard Time / Canada
    {    120,   152,  5885 }, // US Mountain Standard Time / Mexico
    {    120,   248,  5904 }, // US Mountain Standard Time / United States
    {    121,     0,  5920 }, // UTC / AnyTerritory
    {    122,     0,  5936 }, // UTC+12 / AnyTerritory
    {    122,   125,  5947 }, // UTC+12 / Kiribati
    {    122,   147,  5962 }, // UTC+12 / Marshall Islands
    {    122,   163,  5995 }, // UTC+12 / Nauru
    {    122,   242,  6009 }, // UTC+12 / Tuvalu
    {    122,   247,  6026 }, // UTC+12 / United States Outlying Islands
    {    122,   256,  6039 }, // UTC+12 / Wallis and Futuna
    {    123,     0,  6054 }, // UTC+13 / AnyTerritory
    {    123,   125,  6065 }, // UTC+13 / Kiribati
    {    123,   234,  6080 }, // UTC+13 / Tokelau
    {    124,     0,  6096 }, // UTC-02 / AnyTerritory
    {    124,    32,  6106 }, // UTC-02 / Brazil
    {    124,   217,  6122 }, // UTC-02 / South Georgia and South Sandwich Islands
    {    125,     0,  6145 }, // UTC-08 / AnyTerritory
    {    125,   186,  6155 }, // UTC-08 / Pitcairn
    {    126,     0,  6172 }, // UTC-09 / AnyTerritory
    {    126,    86,  6182 }, // UTC-09 / French Polynesia
    {    127,     0,  6198 }, // UTC-11 / AnyTerritory
    {    127,     5,  6209 }, // UTC-11 / American Samoa
    {    127,   171,  6227 }, // UTC-11 / Niue
    {    127,   247,  6240 }, // UTC-11 / United States Outlying Islands
    {    128,   254,  6255 }, // Venezuela Standard Time / Venezuela
    {    129,   193,  6271 }, // Vladivostok Standard Time / Russia
    {    130,   193,  6302 }, // Volgograd Standard Time / Russia
    {    131,    15,  6319 }, // W. Australia Standard Time / Australia
    {    132,     0,  6335 }, // W. Central Africa Standard Time / AnyTerritory
    {    132,     4,  6345 }, // W. Central Africa Standard Time / Algeria
    {    132,     7,  6360 }, // W. Central Africa Standard Time / Angola
    {    132,    25,  6374 }, // W. Central Africa Standard Time / Benin
    {    132,    40,  6392 }, // W. Central Africa Standard Time / Cameroon
    {    132,    46,  6406 }, // W. Central Africa Standard Time / Central African Republic
    {    132,    48,  6420 }, // W. Central Africa Standard Time / Chad
    {    132,    56,  6436 }, // W. Central Africa Standard Time / Congo - Brazzaville
    {    132,    57,  6455 }, // W. Central Africa Standard Time / Congo - Kinshasa
    {    132,    73,  6471 }, // W. Central Africa Standard Time / Equatorial Guinea
    {    132,    88,  6485 }, // W. Central Africa Standard Time / Gabon
    {    132,   169,  6503 }, // W. Central Africa Standard Time / Nigeria
    {    132,   170,  6516 }, // W. Central Africa Standard Time / Niger
    {    132,   238,  6530 }, // W. Central Africa Standard Time / Tunisia
    {    133,     6,  6543 }, // W. Europe Standard Time / Andorra
    {    133,    16,  6558 }, // W. Europe Standard Time / Austria
    {    133,    91,  6572 }, // W. Europe Standard Time / Germany
    {    133,    93,  6602 }, // W. Europe Standard Time / Gibraltar
    {    133,   117,  6619 }, // W. Europe Standard Time / Italy
    {    133,   136,  6631 }, // W. Europe Standard Time / Liechtenstein
    {    133,   138,  6644 }, // W. Europe Standard Time / Luxembourg
    {    133,   146,  6662 }, // W. Europe Standard Time / Malta
    {    133,   155,  6675 }, // W. Europe Standard Time / Monaco
    {    133,   165,  6689 }, // W. Europe Standard Time / Netherlands
    {    133,   175,  6706 }, // W. Europe Standard Time / Norway
    {    133,   203,  6718 }, // W. Europe Standard Time / San Marino
    {    133,   224,  6736 }, // W. Europe Standard Time / Svalbard and Jan Mayen
    {    133,   225,  6756 }, // W. Europe Standard Time / Sweden
    {    133,   226,  6773 }, // W. Europe Standard Time / Switzerland
    {    133,   253,  6787 }, // W. Europe Standard Time / Vatican City
    {    134,   156,  6802 }, // W. Mongolia Standard Time / Mongolia
    {    135,     0,  6812 }, // West Asia Standard Time / AnyTerritory
    {    135,     9,  6822 }, // West Asia Standard Time / Antarctica
    {    135,    87,  6840 }, // West Asia Standard Time / French Southern Territories
    {    135,   123,  6857 }, // West Asia Standard Time / Kazakhstan
    {    135,   144,  6928 }, // West Asia Standard Time / Maldives
    {    135,   229,  6944 }, // West Asia Standard Time / Tajikistan
    {    135,   240,  6958 }, // West Asia Standard Time / Turkmenistan
    {    135,   251,  6972 }, // West Asia Standard Time / Uzbekistan
    {    136,   180,  7001 }, // West Bank Standard Time / Palestinian Territories
    {    137,     0,  7023 }, // West Pacific Standard Time / AnyTerritory
    {    137,     9,  7034 }, // West Pacific Standard Time / Antarctica
    {    137,    98,  7060 }, // West Pacific Standard Time / Guam
    {    137,   153,  7073 }, // West Pacific Standard Time / Micronesia
    {    137,   173,  7087 }, // West Pacific Standard Time / Northern Mariana Islands
    {    137,   182,  7102 }, // West Pacific Standard Time / Papua New Guinea
    {    138,   193,  7123 }, // Yakutsk Standard Time / Russia
    {    139,    41,  7150 }, // Yukon Standard Time / Canada
};

// Windows ID Key, Windows ID Index, IANA ID Index, UTC Offset
static constexpr WindowsData windowsDataTable[] = {
    {      1,     0,     0, 16200 }, // Afghanistan Standard Time
    {      2,    26,  7184,-32400 }, // Alaskan Standard Time
    {      3,    48,   106,-36000 }, // Aleutian Standard Time
    {      4,    71,   119, 25200 }, // Altai Standard Time
    {      5,    91,   168, 10800 }, // Arab Standard Time
    {      6,   110,   212, 14400 }, // Arabian Standard Time
    {      7,   132,   223, 10800 }, // Arabic Standard Time
    {      8,   153,  7202,-10800 }, // Argentina Standard Time
    {      9,   177,  7233, 14400 }, // Astrakhan Standard Time
    {     10,   201,  7250,-14400 }, // Atlantic Standard Time
    {     11,   224,   692, 34200 }, // AUS Central Standard Time
    {     12,   250,   709, 31500 }, // Aus Central W. Standard Time
    {     13,   279,  7266, 36000 }, // AUS Eastern Standard Time
    {     14,   305,   762, 14400 }, // Azerbaijan Standard Time
    {     15,   330,   793, -3600 }, // Azores Standard Time
    {     16,   351,   809,-10800 }, // Bahia Standard Time
    {     17,   371,   823, 21600 }, // Bangladesh Standard Time
    {     18,   396,   847, 10800 }, // Belarus Standard Time
    {     19,   418,   860, 39600 }, // Bougainville Standard Time
    {     20,   445,  7283,-21600 }, // Canada Central Standard Time
    {     21,   474,   928, -3600 }, // Cape Verde Standard Time
    {     22,   499,   948, 14400 }, // Caucasus Standard Time
    {     23,   522,  7298, 34200 }, // Cen. Australia Standard Time
    {     24,   551,  1084,-21600 }, // Central America Standard Time
    {     25,   581,  1192, 21600 }, // Central Asia Standard Time
    {     26,   608,  7317,-14400 }, // Central Brazilian Standard Time
    {     27,   640,  1269,  3600 }, // Central Europe Standard Time
    {     28,   669,  1397,  3600 }, // Central European Standard Time
    {     29,   700,  1485, 39600 }, // Central Pacific Standard Time
    {     30,   730,  7332,-21600 }, // Central Standard Time
    {     31,   752,  7348,-21600 }, // Central Standard Time (Mexico)
    {     32,   783,  1878, 45900 }, // Chatham Islands Standard Time
    {     33,   813,  1894, 28800 }, // China Standard Time
    {     34,   833,  1934,-18000 }, // Cuba Standard Time
    {     35,   852,  1949,-43200 }, // Dateline Standard Time
    {     36,   875,  2050, 10800 }, // E. Africa Standard Time
    {     37,   899,  7368, 36000 }, // E. Australia Standard Time
    {     38,   926,  2191,  7200 }, // E. Europe Standard Time
    {     39,   950,  2207,-10800 }, // E. South America Standard Time
    {     40,   981,  2225,-21600 }, // Easter Island Standard Time
    {     41,  1009,  7387,-18000 }, // Eastern Standard Time
    {     42,  1031,  2461,-18000 }, // Eastern Standard Time (Mexico)
    {     43,  1062,  2476,  7200 }, // Egypt Standard Time
    {     44,  1082,  2489, 18000 }, // Ekaterinburg Standard Time
    {     45,  1109,  2508, 43200 }, // Fiji Standard Time
    {     46,  1128,  2609,  7200 }, // FLE Standard Time
    {     47,  1146,  2621, 14400 }, // Georgian Standard Time
    {     48,  1169,  2759,     0 }, // GMT Standard Time
    {     49,  1187,  2773,-10800 }, // Greenland Standard Time
    {     50,  1211,  2882,     0 }, // Greenwich Standard Time
    {     51,  1235,  3066,  7200 }, // GTB Standard Time
    {     52,  1253,  3083,-18000 }, // Haiti Standard Time
    {     53,  1273,  3150,-36000 }, // Hawaiian Standard Time
    {     54,  1296,  3167, 19800 }, // India Standard Time
    {     55,  1316,  3180, 12600 }, // Iran Standard Time
    {     56,  1335,  3192,  7200 }, // Israel Standard Time
    {     57,  1356,  3207,  7200 }, // Jordan Standard Time
    {     58,  1377,  3218,  7200 }, // Kaliningrad Standard Time
    {     59,  1403,  3237, 32400 }, // Korea Standard Time
    {     60,  1423,  3248,  7200 }, // Libya Standard Time
    {     61,  1443,  3274, 50400 }, // Line Islands Standard Time
    {     62,  1470,  3293, 37800 }, // Lord Howe Standard Time
    {     63,  1494,  3313, 36000 }, // Magadan Standard Time
    {     64,  1516,  3326,-10800 }, // Magallanes Standard Time
    {     65,  1541,  3347,-34200 }, // Marquesas Standard Time
    {     66,  1565,  3365, 14400 }, // Mauritius Standard Time
    {     67,  1589,  3409,  7200 }, // Middle East Standard Time
    {     68,  1615,  3421,-10800 }, // Montevideo Standard Time
    {     69,  1640,  3440,     0 }, // Morocco Standard Time
    {     70,  1662,  7404,-25200 }, // Mountain Standard Time
    {     71,  1685,  3587,-25200 }, // Mountain Standard Time (Mexico)
    {     72,  1717,  3617, 23400 }, // Myanmar Standard Time
    {     73,  1739,  3629, 21600 }, // N. Central Asia Standard Time
    {     74,  1769,  3646,  3600 }, // Namibia Standard Time
    {     75,  1791,  3662, 20700 }, // Nepal Standard Time
    {     76,  1811,  3696, 43200 }, // New Zealand Standard Time
    {     77,  1837,  3713,-12600 }, // Newfoundland Standard Time
    {     78,  1864,  3730, 39600 }, // Norfolk Standard Time
    {     79,  1886,  3746, 28800 }, // North Asia East Standard Time
    {     80,  1916,  7419, 25200 }, // North Asia Standard Time
    {     81,  1941,  3794, 30600 }, // North Korea Standard Time
    {     82,  1967,  3809, 21600 }, // Omsk Standard Time
    {     83,  1986,  3819,-10800 }, // Pacific SA Standard Time
    {     84,  2011,  3862,-28800 }, // Pacific Standard Time
    {     85,  2033,  3882,-28800 }, // Pacific Standard Time (Mexico)
    {     86,  2064,  3898, 18000 }, // Pakistan Standard Time
    {     87,  2087,  3911,-14400 }, // Paraguay Standard Time
    {     88,  2110,  3928, 18000 }, // Qyzylorda Standard Time
    {     89,  2134,  3977,  3600 }, // Romance Standard Time
    {     90,  2156,  4017, 39600 }, // Russia Time Zone 10
    {     91,  2176,  7436, 43200 }, // Russia Time Zone 11
    {     92,  2196,  4063, 14400 }, // Russia Time Zone 3
    {     93,  2215,  7451, 10800 }, // Russian Standard Time
    {     94,  2237,  4265,-10800 }, // SA Eastern Standard Time
    {     95,  2262,  4378,-18000 }, // SA Pacific Standard Time
    {     96,  2287,  4529,-14400 }, // SA Western Standard Time
    {     97,  2312,  4969,-10800 }, // Saint Pierre Standard Time
    {     98,  2339,  4986, 39600 }, // Sakhalin Standard Time
    {     99,  2362,  5000, 46800 }, // Samoa Standard Time
    {    100,  2382,  5013,     0 }, // Sao Tome Standard Time
    {    101,  2405,  5029, 14400 }, // Saratov Standard Time
    {    102,  2427,  5147, 25200 }, // SE Asia Standard Time
    {    103,  2449,  5256, 28800 }, // Singapore Standard Time
    {    104,  2473,  5405,  7200 }, // South Africa Standard Time
    {    105,  2500,  5453,  7200 }, // South Sudan Standard Time
    {    106,  2526,  5465, 19800 }, // Sri Lanka Standard Time
    {    107,  2550,  5478,  7200 }, // Sudan Standard Time
    {    108,  2570,  5494,  7200 }, // Syria Standard Time
    {    109,  2590,  5508, 28800 }, // Taipei Standard Time
    {    110,  2611,  7465, 36000 }, // Tasmania Standard Time
    {    111,  2634,  5558,-10800 }, // Tocantins Standard Time
    {    112,  2658,  5600, 32400 }, // Tokyo Standard Time
    {    113,  2678,  5635, 25200 }, // Tomsk Standard Time
    {    114,  2698,  5646, 46800 }, // Tonga Standard Time
    {    115,  2718,  5664, 32400 }, // Transbaikal Standard Time
    {    116,  2744,  5675,  7200 }, // Turkey Standard Time
    {    117,  2765,  5691,-14400 }, // Turks And Caicos Standard Time
    {    118,  2796,  7482, 28800 }, // Ulaanbaatar Standard Time
    {    119,  2822,  7499,-18000 }, // US Eastern Standard Time
    {    120,  2847,  5904,-25200 }, // US Mountain Standard Time
    {    121,  2873,  7528,     0 }, // UTC
    {    122,  2877,  5936, 43200 }, // UTC+12
    {    123,  2884,  6054, 46800 }, // UTC+13
    {    124,  2891,  6096, -7200 }, // UTC-02
    {    125,  2898,  6145,-28800 }, // UTC-08
    {    126,  2905,  6172,-32400 }, // UTC-09
    {    127,  2912,  6198,-39600 }, // UTC-11
    {    128,  2919,  6255,-16200 }, // Venezuela Standard Time
    {    129,  2943,  7536, 36000 }, // Vladivostok Standard Time
    {    130,  2969,  6302, 14400 }, // Volgograd Standard Time
    {    131,  2993,  6319, 28800 }, // W. Australia Standard Time
    {    132,  3020,  6503,  3600 }, // W. Central Africa Standard Time
    {    133,  3052,  7553,  3600 }, // W. Europe Standard Time
    {    134,  3076,  6802, 25200 }, // W. Mongolia Standard Time
    {    135,  3102,  7567, 18000 }, // West Asia Standard Time
    {    136,  3126,  7581,  7200 }, // West Bank Standard Time
    {    137,  3150,  7102, 36000 }, // West Pacific Standard Time
    {    138,  3177,  7593, 32400 }, // Yakutsk Standard Time
    {    139,  3199,  7606,-25200 }, // Yukon Standard Time
};

// IANA ID Index, UTC Offset
static constexpr UtcData utcDataTable[] = {
    {   7625,-50400 }, // UTC-14:00
    {   7635,-46800 }, // UTC-13:00
    {   7645,-43200 }, // UTC-12:00
    {   7655,-39600 }, // UTC-11:00
    {   7665,-36000 }, // UTC-10:00
    {   7675,-34200 }, // UTC-09:30
    {   7685,-32400 }, // UTC-09:00
    {   7695,-28800 }, // UTC-08:00
    {   7705,-25200 }, // UTC-07:00
    {   7715,-21600 }, // UTC-06:00
    {   7725,-18000 }, // UTC-05:00
    {   7735,-16200 }, // UTC-04:30
    {   7745,-14400 }, // UTC-04:00
    {   7755,-12600 }, // UTC-03:30
    {   7765,-10800 }, // UTC-03:00
    {   7775, -7200 }, // UTC-02:00
    {   7785, -3600 }, // UTC-01:00
    {   7795,     0 }, // UTC
    {   7819,  3600 }, // UTC+01:00
    {   7829,  7200 }, // UTC+02:00
    {   7839, 10800 }, // UTC+03:00
    {   7849, 12600 }, // UTC+03:30
    {   7859, 14400 }, // UTC+04:00
    {   7869, 16200 }, // UTC+04:30
    {   7879, 18000 }, // UTC+05:00
    {   7889, 19800 }, // UTC+05:30
    {   7899, 20700 }, // UTC+05:45
    {   7909, 21600 }, // UTC+06:00
    {   7919, 23400 }, // UTC+06:30
    {   7929, 25200 }, // UTC+07:00
    {   7939, 28800 }, // UTC+08:00
    {   7949, 30600 }, // UTC+08:30
    {   7959, 31500 }, // UTC+08:45
    {   7969, 32400 }, // UTC+09:00
    {   7979, 34200 }, // UTC+09:30
    {   7989, 36000 }, // UTC+10:00
    {   7999, 37800 }, // UTC+10:30
    {   8009, 39600 }, // UTC+11:00
    {   8019, 43200 }, // UTC+12:00
    {   8029, 45900 }, // UTC+12:45
    {   8039, 46800 }, // UTC+13:00
    {   8049, 50400 }, // UTC+14:00
};

static constexpr char windowsIdData[] = {
0x41, 0x66, 0x67, 0x68, 0x61, 0x6e, 0x69, 0x73, 0x74, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x6c, 0x61, 0x73, 0x6b, 0x61,
0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x41, 0x6c, 0x65, 0x75, 0x74, 0x69, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x6c, 0x74, 0x61, 0x69, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x72, 0x61, 0x62, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x72,
0x61, 0x62, 0x69, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x41, 0x72, 0x61, 0x62, 0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69,
0x6e, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x41, 0x73, 0x74, 0x72, 0x61, 0x6b, 0x68, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69,
0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x41, 0x55, 0x53, 0x20, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x75, 0x73, 0x20, 0x43, 0x65,
0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x57, 0x2e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x55, 0x53, 0x20, 0x45, 0x61, 0x73, 0x74, 0x65,
0x72, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x41, 0x7a, 0x65, 0x72, 0x62, 0x61, 0x69, 0x6a, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x7a, 0x6f, 0x72, 0x65, 0x73,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x42,
0x61, 0x68, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x42, 0x61, 0x6e, 0x67, 0x6c, 0x61, 0x64, 0x65, 0x73, 0x68, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x42, 0x65, 0x6c, 0x61,
0x72, 0x75, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d,
0x65, 0x0, 0x42, 0x6f, 0x75, 0x67, 0x61, 0x69, 0x6e, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x61, 0x6e,
0x61, 0x64, 0x61, 0x20, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x61, 0x70, 0x65, 0x20, 0x56,
0x65, 0x72, 0x64, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x43, 0x61, 0x75, 0x63, 0x61, 0x73, 0x75, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x2e, 0x20, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41, 0x73, 0x69,
0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x42, 0x72, 0x61, 0x7a, 0x69, 0x6c, 0x69, 0x61,
0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e,
0x74, 0x72, 0x61, 0x6c, 0x20, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x61, 0x6e, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x74,
0x72, 0x61, 0x6c, 0x20, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61,
0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x28, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x29, 0x0, 0x43,
0x68, 0x61, 0x74, 0x68, 0x61, 0x6d, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x68, 0x69,
0x6e, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x43, 0x75, 0x62, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x44, 0x61, 0x74, 0x65, 0x6c, 0x69, 0x6e, 0x65, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x2e, 0x20, 0x41, 0x66,
0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x45, 0x2e, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x2e,
0x20, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x2e, 0x20, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x20, 0x49, 0x73, 0x6c, 0x61,
0x6e, 0x64, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x28, 0x4d, 0x65,
0x78, 0x69, 0x63, 0x6f, 0x29, 0x0, 0x45, 0x67, 0x79, 0x70, 0x74, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x6b, 0x61, 0x74, 0x65, 0x72,
0x69, 0x6e, 0x62, 0x75, 0x72, 0x67, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x46, 0x69, 0x6a, 0x69, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x46, 0x4c, 0x45, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x47, 0x65, 0x6f, 0x72, 0x67, 0x69,
0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x47, 0x4d, 0x54, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x47, 0x72, 0x65, 0x65, 0x6e, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x47, 0x72, 0x65, 0x65, 0x6e,
0x77, 0x69, 0x63, 0x68, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x47, 0x54, 0x42, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x48, 0x61, 0x69, 0x74, 0x69, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x48, 0x61, 0x77, 0x61, 0x69, 0x69, 0x61,
0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x49, 0x6e, 0x64, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x49, 0x72, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x49, 0x73, 0x72, 0x61, 0x65, 0x6c, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4a, 0x6f, 0x72, 0x64,
0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x4b, 0x61, 0x6c, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x72, 0x61, 0x64, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4b, 0x6f, 0x72, 0x65, 0x61,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4c,
0x69, 0x62, 0x79, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x4c, 0x69, 0x6e, 0x65, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4c, 0x6f,
0x72, 0x64, 0x20, 0x48, 0x6f, 0x77, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x61, 0x67, 0x61, 0x64, 0x61, 0x6e, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x61, 0x67, 0x61,
0x6c, 0x6c, 0x61, 0x6e, 0x65, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x61, 0x72, 0x71, 0x75, 0x65, 0x73, 0x61, 0x73, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x61, 0x75,
0x72, 0x69, 0x74, 0x69, 0x75, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x69, 0x64, 0x64, 0x6c, 0x65, 0x20, 0x45, 0x61, 0x73, 0x74,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d,
0x6f, 0x6e, 0x74, 0x65, 0x76, 0x69, 0x64, 0x65, 0x6f, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x6f, 0x72, 0x6f, 0x63, 0x63, 0x6f, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x6f,
0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x28, 0x4d, 0x65, 0x78,
0x69, 0x63, 0x6f, 0x29, 0x0, 0x4d, 0x79, 0x61, 0x6e, 0x6d, 0x61, 0x72, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x2e, 0x20, 0x43, 0x65,
0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41, 0x73, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x61, 0x6d, 0x69, 0x62, 0x69, 0x61,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e,
0x65, 0x70, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x4e, 0x65, 0x77, 0x20, 0x5a, 0x65, 0x61, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x65, 0x77,
0x66, 0x6f, 0x75, 0x6e, 0x64, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x6f, 0x72, 0x66, 0x6f, 0x6c, 0x6b, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x6f,
0x72, 0x74, 0x68, 0x20, 0x41, 0x73, 0x69, 0x61, 0x20, 0x45, 0x61, 0x73, 0x74, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x6f, 0x72, 0x74,
0x68, 0x20, 0x41, 0x73, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x20, 0x4b, 0x6f, 0x72, 0x65, 0x61,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4f,
0x6d, 0x73, 0x6b, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d,
0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x53, 0x41, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x28, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x29, 0x0,
0x50, 0x61, 0x6b, 0x69, 0x73, 0x74, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x50, 0x61, 0x72, 0x61, 0x67, 0x75, 0x61, 0x79, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x51, 0x79,
0x7a, 0x79, 0x6c, 0x6f, 0x72, 0x64, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x52, 0x6f, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x52, 0x75, 0x73, 0x73,
0x69, 0x61, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x5a, 0x6f, 0x6e, 0x65, 0x20, 0x31, 0x30, 0x0,
0x52, 0x75, 0x73, 0x73, 0x69, 0x61, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x5a, 0x6f, 0x6e, 0x65,
0x20, 0x31, 0x31, 0x0, 0x52, 0x75, 0x73, 0x73, 0x69, 0x61, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20,
0x5a, 0x6f, 0x6e, 0x65, 0x20, 0x33, 0x0, 0x52, 0x75, 0x73, 0x73, 0x69, 0x61, 0x6e, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x41, 0x20,
0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x41, 0x20, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53,
0x41, 0x20, 0x57, 0x65, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x61, 0x69, 0x6e, 0x74, 0x20, 0x50, 0x69,
0x65, 0x72, 0x72, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x53, 0x61, 0x6b, 0x68, 0x61, 0x6c, 0x69, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x61, 0x6d, 0x6f, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x61,
0x6f, 0x20, 0x54, 0x6f, 0x6d, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x61, 0x72, 0x61, 0x74, 0x6f, 0x76, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x45, 0x20, 0x41, 0x73,
0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x53, 0x69, 0x6e, 0x67, 0x61, 0x70, 0x6f, 0x72, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x53, 0x75, 0x64, 0x61, 0x6e, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x72,
0x69, 0x20, 0x4c, 0x61, 0x6e, 0x6b, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x75, 0x64, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x79, 0x72, 0x69, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x61,
0x69, 0x70, 0x65, 0x69, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x54, 0x61, 0x73, 0x6d, 0x61, 0x6e, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x6f, 0x63, 0x61, 0x6e, 0x74,
0x69, 0x6e, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d,
0x65, 0x0, 0x54, 0x6f, 0x6b, 0x79, 0x6f, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x6f, 0x6d, 0x73, 0x6b, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x6f, 0x6e, 0x67, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x72,
0x61, 0x6e, 0x73, 0x62, 0x61, 0x69, 0x6b, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x75, 0x72, 0x6b, 0x65, 0x79, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x75, 0x72,
0x6b, 0x73, 0x20, 0x41, 0x6e, 0x64, 0x20, 0x43, 0x61, 0x69, 0x63, 0x6f, 0x73, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x55, 0x6c, 0x61, 0x61,
0x6e, 0x62, 0x61, 0x61, 0x74, 0x61, 0x72, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x55, 0x53, 0x20, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x55,
0x53, 0x20, 0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x55, 0x54, 0x43, 0x0, 0x55, 0x54, 0x43,
0x2b, 0x31, 0x32, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x33, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30,
0x32, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x38, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x39, 0x0,
0x55, 0x54, 0x43, 0x2d, 0x31, 0x31, 0x0, 0x56, 0x65, 0x6e, 0x65, 0x7a, 0x75, 0x65, 0x6c, 0x61,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x56,
0x6c, 0x61, 0x64, 0x69, 0x76, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x56, 0x6f, 0x6c, 0x67, 0x6f, 0x67, 0x72,
0x61, 0x64, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x57, 0x2e, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x2e, 0x20, 0x43,
0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x2e, 0x20, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x57, 0x2e, 0x20, 0x4d, 0x6f, 0x6e, 0x67, 0x6f, 0x6c, 0x69, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x65,
0x73, 0x74, 0x20, 0x41, 0x73, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x65, 0x73, 0x74, 0x20, 0x42, 0x61, 0x6e, 0x6b, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x65,
0x73, 0x74, 0x20, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x59, 0x61, 0x6b, 0x75, 0x74, 0x73, 0x6b,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x59,
0x75, 0x6b, 0x6f, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0
};

static constexpr char ianaIdData[] = {
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x62, 0x75, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x41, 0x6e, 0x63, 0x68, 0x6f, 0x72, 0x61, 0x67, 0x65, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4a, 0x75, 0x6e, 0x65, 0x61, 0x75, 0x20, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x74, 0x6c, 0x61, 0x6b, 0x61, 0x74, 0x6c, 0x61, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x6d, 0x65, 0x20, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x53, 0x69, 0x74, 0x6b, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x59, 0x61, 0x6b, 0x75, 0x74, 0x61, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x41, 0x64, 0x61, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x72, 0x6e,
0x61, 0x75, 0x6c, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x68, 0x72, 0x61, 0x69, 0x6e,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x75, 0x77, 0x61, 0x69, 0x74, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x51, 0x61, 0x74, 0x61, 0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x52, 0x69, 0x79,
0x61, 0x64, 0x68, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x64, 0x65, 0x6e, 0x0, 0x45, 0x74,
0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x34, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x75, 0x73,
0x63, 0x61, 0x74, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x75, 0x62, 0x61, 0x69, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x67, 0x68, 0x64, 0x61, 0x64, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x42, 0x75,
0x65, 0x6e, 0x6f, 0x73, 0x5f, 0x41, 0x69, 0x72, 0x65, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4c, 0x61, 0x5f,
0x52, 0x69, 0x6f, 0x6a, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72,
0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x52, 0x69, 0x6f, 0x5f, 0x47, 0x61, 0x6c, 0x6c,
0x65, 0x67, 0x6f, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67,
0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x53, 0x61, 0x6c, 0x74, 0x61, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x53,
0x61, 0x6e, 0x5f, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x5f, 0x4c, 0x75,
0x69, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e,
0x74, 0x69, 0x6e, 0x61, 0x2f, 0x54, 0x75, 0x63, 0x75, 0x6d, 0x61, 0x6e, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x55,
0x73, 0x68, 0x75, 0x61, 0x69, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41,
0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x43, 0x61, 0x74, 0x61, 0x6d, 0x61, 0x72,
0x63, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e,
0x74, 0x69, 0x6e, 0x61, 0x2f, 0x43, 0x6f, 0x72, 0x64, 0x6f, 0x62, 0x61, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4a,
0x75, 0x6a, 0x75, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67,
0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4d, 0x65, 0x6e, 0x64, 0x6f, 0x7a, 0x61, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41, 0x73, 0x74, 0x72, 0x61, 0x6b, 0x68, 0x61, 0x6e, 0x20,
0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x55, 0x6c, 0x79, 0x61, 0x6e, 0x6f, 0x76, 0x73, 0x6b,
0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x42, 0x65, 0x72, 0x6d, 0x75, 0x64,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x6c, 0x69, 0x66, 0x61,
0x78, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x6c, 0x61, 0x63, 0x65, 0x5f,
0x42, 0x61, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x6f, 0x6f, 0x73,
0x65, 0x5f, 0x42, 0x61, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f,
0x6e, 0x63, 0x74, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x68,
0x75, 0x6c, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x44, 0x61,
0x72, 0x77, 0x69, 0x6e, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x45,
0x75, 0x63, 0x6c, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x53,
0x79, 0x64, 0x6e, 0x65, 0x79, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f,
0x4d, 0x65, 0x6c, 0x62, 0x6f, 0x75, 0x72, 0x6e, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42,
0x61, 0x6b, 0x75, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x63, 0x6f, 0x72,
0x65, 0x73, 0x62, 0x79, 0x73, 0x75, 0x6e, 0x64, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69,
0x63, 0x2f, 0x41, 0x7a, 0x6f, 0x72, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x42, 0x61, 0x68, 0x69, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x68, 0x61, 0x6b,
0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x68, 0x69, 0x6d, 0x70, 0x68, 0x75, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x69, 0x6e, 0x73, 0x6b, 0x0, 0x50, 0x61, 0x63, 0x69,
0x66, 0x69, 0x63, 0x2f, 0x42, 0x6f, 0x75, 0x67, 0x61, 0x69, 0x6e, 0x76, 0x69, 0x6c, 0x6c, 0x65,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x65, 0x67, 0x69, 0x6e, 0x61, 0x20,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x77, 0x69, 0x66, 0x74, 0x5f, 0x43, 0x75,
0x72, 0x72, 0x65, 0x6e, 0x74, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x0,
0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x43, 0x61, 0x70, 0x65, 0x5f, 0x56, 0x65,
0x72, 0x64, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x65, 0x72, 0x65, 0x76, 0x61, 0x6e,
0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x41, 0x64, 0x65, 0x6c, 0x61,
0x69, 0x64, 0x65, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x42, 0x72,
0x6f, 0x6b, 0x65, 0x6e, 0x5f, 0x48, 0x69, 0x6c, 0x6c, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d,
0x54, 0x2b, 0x36, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x65, 0x6c, 0x69,
0x7a, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x6f, 0x73, 0x74, 0x61,
0x5f, 0x52, 0x69, 0x63, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x61,
0x6c, 0x61, 0x70, 0x61, 0x67, 0x6f, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x45, 0x6c, 0x5f, 0x53, 0x61, 0x6c, 0x76, 0x61, 0x64, 0x6f, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x47, 0x75, 0x61, 0x74, 0x65, 0x6d, 0x61, 0x6c, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x65, 0x67, 0x75, 0x63, 0x69, 0x67, 0x61, 0x6c, 0x70,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x6e, 0x61, 0x67, 0x75,
0x61, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x36, 0x0, 0x41, 0x6e, 0x74, 0x61,
0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x56, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x0, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x68, 0x61, 0x67, 0x6f, 0x73, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x55, 0x72, 0x75, 0x6d, 0x71, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x69, 0x73,
0x68, 0x6b, 0x65, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x75, 0x69,
0x61, 0x62, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x6d, 0x70,
0x6f, 0x5f, 0x47, 0x72, 0x61, 0x6e, 0x64, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x54, 0x69, 0x72, 0x61, 0x6e, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x50, 0x72,
0x61, 0x67, 0x75, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x75, 0x64, 0x61,
0x70, 0x65, 0x73, 0x74, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x50, 0x6f, 0x64, 0x67,
0x6f, 0x72, 0x69, 0x63, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x65, 0x6c,
0x67, 0x72, 0x61, 0x64, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x72, 0x61,
0x74, 0x69, 0x73, 0x6c, 0x61, 0x76, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c,
0x6a, 0x75, 0x62, 0x6c, 0x6a, 0x61, 0x6e, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x53, 0x61, 0x72, 0x61, 0x6a, 0x65, 0x76, 0x6f, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x5a, 0x61, 0x67, 0x72, 0x65, 0x62, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x6b,
0x6f, 0x70, 0x6a, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x57, 0x61, 0x72, 0x73,
0x61, 0x77, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x31, 0x0, 0x41, 0x6e,
0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x73, 0x65, 0x79, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x6f, 0x68, 0x6e, 0x70, 0x65, 0x69, 0x20, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4b, 0x6f, 0x73, 0x72, 0x61, 0x65, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4e, 0x6f, 0x75, 0x6d, 0x65, 0x61, 0x0, 0x50, 0x61, 0x63,
0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x75, 0x61, 0x64, 0x61, 0x6c, 0x63, 0x61, 0x6e, 0x61, 0x6c,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x45, 0x66, 0x61, 0x74, 0x65, 0x0, 0x43,
0x53, 0x54, 0x36, 0x43, 0x44, 0x54, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57,
0x69, 0x6e, 0x6e, 0x69, 0x70, 0x65, 0x67, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x52, 0x61, 0x6e, 0x6b, 0x69, 0x6e, 0x5f, 0x49, 0x6e, 0x6c, 0x65, 0x74, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x65, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x65, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x74, 0x61, 0x6d, 0x6f, 0x72, 0x6f, 0x73, 0x20,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4f, 0x6a, 0x69, 0x6e, 0x61, 0x67, 0x61, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x68, 0x69, 0x63, 0x61, 0x67, 0x6f, 0x20,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f,
0x4b, 0x6e, 0x6f, 0x78, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64,
0x69, 0x61, 0x6e, 0x61, 0x2f, 0x54, 0x65, 0x6c, 0x6c, 0x5f, 0x43, 0x69, 0x74, 0x79, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x6e, 0x6f, 0x6d, 0x69, 0x6e, 0x65, 0x65,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x5f, 0x44,
0x61, 0x6b, 0x6f, 0x74, 0x61, 0x2f, 0x42, 0x65, 0x75, 0x6c, 0x61, 0x68, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x5f, 0x44, 0x61, 0x6b, 0x6f, 0x74,
0x61, 0x2f, 0x43, 0x65, 0x6e, 0x74, 0x65, 0x72, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x5f, 0x44, 0x61, 0x6b, 0x6f, 0x74, 0x61, 0x2f, 0x4e, 0x65,
0x77, 0x5f, 0x53, 0x61, 0x6c, 0x65, 0x6d, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x5f, 0x43, 0x69, 0x74, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x68, 0x69, 0x61, 0x5f, 0x42, 0x61, 0x6e, 0x64, 0x65, 0x72,
0x61, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x72, 0x69, 0x64,
0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x65, 0x72,
0x72, 0x65, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x68, 0x69, 0x68,
0x75, 0x61, 0x68, 0x75, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x43, 0x68,
0x61, 0x74, 0x68, 0x61, 0x6d, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x68, 0x61, 0x6e, 0x67,
0x68, 0x61, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x6e, 0x67, 0x5f, 0x4b, 0x6f,
0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x63, 0x61, 0x75, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x76, 0x61, 0x6e, 0x61, 0x0, 0x45, 0x74, 0x63,
0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x32, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d,
0x33, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x79, 0x6f,
0x77, 0x61, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x6f, 0x6d, 0x6f, 0x72, 0x6f,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x6a, 0x69, 0x62, 0x6f, 0x75, 0x74, 0x69,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x73, 0x6d, 0x61, 0x72, 0x61, 0x0, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x64, 0x64, 0x69, 0x73, 0x5f, 0x41, 0x62, 0x61, 0x62,
0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x61, 0x69, 0x72, 0x6f, 0x62, 0x69,
0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x41, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x6e, 0x61,
0x72, 0x69, 0x76, 0x6f, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x79, 0x6f,
0x74, 0x74, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x67, 0x61, 0x64,
0x69, 0x73, 0x68, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x72, 0x5f,
0x65, 0x73, 0x5f, 0x53, 0x61, 0x6c, 0x61, 0x61, 0x6d, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4b, 0x61, 0x6d, 0x70, 0x61, 0x6c, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c,
0x69, 0x61, 0x2f, 0x42, 0x72, 0x69, 0x73, 0x62, 0x61, 0x6e, 0x65, 0x20, 0x41, 0x75, 0x73, 0x74,
0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x69, 0x6e, 0x64, 0x65, 0x6d, 0x61, 0x6e, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x43, 0x68, 0x69, 0x73, 0x69, 0x6e, 0x61, 0x75, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6f, 0x5f, 0x50, 0x61, 0x75, 0x6c, 0x6f,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x0,
0x45, 0x53, 0x54, 0x35, 0x45, 0x44, 0x54, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4e, 0x61, 0x73, 0x73, 0x61, 0x75, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54,
0x6f, 0x72, 0x6f, 0x6e, 0x74, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49,
0x71, 0x61, 0x6c, 0x75, 0x69, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e,
0x65, 0x77, 0x5f, 0x59, 0x6f, 0x72, 0x6b, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x44, 0x65, 0x74, 0x72, 0x6f, 0x69, 0x74, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x50, 0x65, 0x74, 0x65, 0x72, 0x73, 0x62, 0x75,
0x72, 0x67, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61,
0x6e, 0x61, 0x2f, 0x56, 0x69, 0x6e, 0x63, 0x65, 0x6e, 0x6e, 0x65, 0x73, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x57, 0x69, 0x6e,
0x61, 0x6d, 0x61, 0x63, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x65, 0x6e,
0x74, 0x75, 0x63, 0x6b, 0x79, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x69, 0x63, 0x65, 0x6c, 0x6c, 0x6f,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x65, 0x6e, 0x74, 0x75, 0x63, 0x6b,
0x79, 0x2f, 0x4c, 0x6f, 0x75, 0x69, 0x73, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x6e, 0x63, 0x75, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x43, 0x61, 0x69, 0x72, 0x6f, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x65,
0x6b, 0x61, 0x74, 0x65, 0x72, 0x69, 0x6e, 0x62, 0x75, 0x72, 0x67, 0x0, 0x50, 0x61, 0x63, 0x69,
0x66, 0x69, 0x63, 0x2f, 0x46, 0x69, 0x6a, 0x69, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x4d, 0x61, 0x72, 0x69, 0x65, 0x68, 0x61, 0x6d, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x53, 0x6f, 0x66, 0x69, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x54, 0x61,
0x6c, 0x6c, 0x69, 0x6e, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x48, 0x65, 0x6c,
0x73, 0x69, 0x6e, 0x6b, 0x69, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x52, 0x69, 0x67,
0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x69, 0x6c, 0x6e, 0x69, 0x75, 0x73,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x79, 0x69, 0x76, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x54, 0x62, 0x69, 0x6c, 0x69, 0x73, 0x69, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74,
0x69, 0x63, 0x2f, 0x46, 0x61, 0x72, 0x6f, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x47, 0x75, 0x65, 0x72, 0x6e, 0x73, 0x65, 0x79, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x44, 0x75, 0x62, 0x6c, 0x69, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x49, 0x73,
0x6c, 0x65, 0x5f, 0x6f, 0x66, 0x5f, 0x4d, 0x61, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x4a, 0x65, 0x72, 0x73, 0x65, 0x79, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c,
0x69, 0x73, 0x62, 0x6f, 0x6e, 0x20, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x4d,
0x61, 0x64, 0x65, 0x69, 0x72, 0x61, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f,
0x43, 0x61, 0x6e, 0x61, 0x72, 0x79, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c, 0x6f,
0x6e, 0x64, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x75, 0x75,
0x6b, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4f, 0x75, 0x61, 0x67, 0x61, 0x64, 0x6f,
0x75, 0x67, 0x6f, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x6e, 0x6a,
0x75, 0x6c, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x63, 0x63, 0x72, 0x61, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x6e, 0x6d, 0x61, 0x72, 0x6b, 0x73,
0x68, 0x61, 0x76, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x69, 0x73, 0x73,
0x61, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x6f, 0x6e, 0x61, 0x6b, 0x72,
0x79, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x52, 0x65, 0x79, 0x6b, 0x6a,
0x61, 0x76, 0x69, 0x6b, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x62, 0x69, 0x64,
0x6a, 0x61, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x72, 0x6f,
0x76, 0x69, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x6d, 0x61, 0x6b,
0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x75, 0x61, 0x6b, 0x63, 0x68,
0x6f, 0x74, 0x74, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x53, 0x74, 0x5f,
0x48, 0x65, 0x6c, 0x65, 0x6e, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61,
0x6b, 0x61, 0x72, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x46, 0x72, 0x65, 0x65, 0x74,
0x6f, 0x77, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f, 0x6d, 0x65, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4e, 0x69, 0x63, 0x6f, 0x73, 0x69, 0x61, 0x20, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x46, 0x61, 0x6d, 0x61, 0x67, 0x75, 0x73, 0x74, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f,
0x70, 0x65, 0x2f, 0x41, 0x74, 0x68, 0x65, 0x6e, 0x73, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x42, 0x75, 0x63, 0x68, 0x61, 0x72, 0x65, 0x73, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x2d, 0x61, 0x75, 0x2d, 0x50, 0x72, 0x69, 0x6e, 0x63,
0x65, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x30, 0x0, 0x50, 0x61, 0x63,
0x69, 0x66, 0x69, 0x63, 0x2f, 0x52, 0x61, 0x72, 0x6f, 0x74, 0x6f, 0x6e, 0x67, 0x61, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x54, 0x61, 0x68, 0x69, 0x74, 0x69, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x48, 0x6f, 0x6e, 0x6f, 0x6c, 0x75, 0x6c, 0x75, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4b, 0x6f, 0x6c, 0x6b, 0x61, 0x74, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x54, 0x65, 0x68, 0x72, 0x61, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a, 0x65, 0x72,
0x75, 0x73, 0x61, 0x6c, 0x65, 0x6d, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6d, 0x6d, 0x61,
0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x61, 0x6c, 0x69, 0x6e, 0x69, 0x6e,
0x67, 0x72, 0x61, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x65, 0x6f, 0x75, 0x6c, 0x0,
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x72, 0x69, 0x70, 0x6f, 0x6c, 0x69, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x34, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x4b, 0x69, 0x72, 0x69, 0x74, 0x69, 0x6d, 0x61, 0x74, 0x69, 0x0, 0x41, 0x75, 0x73,
0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x6f, 0x72, 0x64, 0x5f, 0x48, 0x6f, 0x77, 0x65,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x67, 0x61, 0x64, 0x61, 0x6e, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x75, 0x6e, 0x74, 0x61, 0x5f, 0x41, 0x72, 0x65, 0x6e,
0x61, 0x73, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4d, 0x61, 0x72, 0x71, 0x75,
0x65, 0x73, 0x61, 0x73, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x75, 0x72,
0x69, 0x74, 0x69, 0x75, 0x73, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x52, 0x65, 0x75,
0x6e, 0x69, 0x6f, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x68, 0x65,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x65, 0x69, 0x72, 0x75, 0x74, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x65, 0x76, 0x69, 0x64, 0x65, 0x6f, 0x0,
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x73, 0x61, 0x62, 0x6c, 0x61, 0x6e, 0x63,
0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x45, 0x6c, 0x5f, 0x41, 0x61, 0x69, 0x75,
0x6e, 0x0, 0x4d, 0x53, 0x54, 0x37, 0x4d, 0x44, 0x54, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x45, 0x64, 0x6d, 0x6f, 0x6e, 0x74, 0x6f, 0x6e, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x5f, 0x42, 0x61, 0x79,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x75, 0x76, 0x69, 0x6b, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x69, 0x75, 0x64, 0x61, 0x64, 0x5f, 0x4a,
0x75, 0x61, 0x72, 0x65, 0x7a, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x65,
0x6e, 0x76, 0x65, 0x72, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6f, 0x69,
0x73, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x7a, 0x61, 0x74,
0x6c, 0x61, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x6f, 0x63, 0x6f, 0x73,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x61, 0x6e, 0x67, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x4e, 0x6f, 0x76, 0x6f, 0x73, 0x69, 0x62, 0x69, 0x72, 0x73, 0x6b, 0x0, 0x41, 0x66,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x69, 0x6e, 0x64, 0x68, 0x6f, 0x65, 0x6b, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x4b, 0x61, 0x74, 0x68, 0x6d, 0x61, 0x6e, 0x64, 0x75, 0x0, 0x41, 0x6e, 0x74,
0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x63, 0x4d, 0x75, 0x72, 0x64, 0x6f, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x41, 0x75, 0x63, 0x6b, 0x6c, 0x61, 0x6e, 0x64,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x74, 0x5f, 0x4a, 0x6f, 0x68, 0x6e,
0x73, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4e, 0x6f, 0x72, 0x66, 0x6f, 0x6c,
0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x49, 0x72, 0x6b, 0x75, 0x74, 0x73, 0x6b, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4b, 0x72, 0x61, 0x73, 0x6e, 0x6f, 0x79, 0x61, 0x72, 0x73, 0x6b, 0x20,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4e, 0x6f, 0x76, 0x6f, 0x6b, 0x75, 0x7a, 0x6e, 0x65, 0x74, 0x73,
0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x50, 0x79, 0x6f, 0x6e, 0x67, 0x79, 0x61, 0x6e, 0x67,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4f, 0x6d, 0x73, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x67, 0x6f, 0x0, 0x50, 0x53, 0x54, 0x38,
0x50, 0x44, 0x54, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x56, 0x61, 0x6e, 0x63,
0x6f, 0x75, 0x76, 0x65, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f,
0x73, 0x5f, 0x41, 0x6e, 0x67, 0x65, 0x6c, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x54, 0x69, 0x6a, 0x75, 0x61, 0x6e, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b,
0x61, 0x72, 0x61, 0x63, 0x68, 0x69, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41,
0x73, 0x75, 0x6e, 0x63, 0x69, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x51, 0x79, 0x7a,
0x79, 0x6c, 0x6f, 0x72, 0x64, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x72,
0x75, 0x73, 0x73, 0x65, 0x6c, 0x73, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x43, 0x6f,
0x70, 0x65, 0x6e, 0x68, 0x61, 0x67, 0x65, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x50, 0x61, 0x72, 0x69, 0x73, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x61, 0x64,
0x72, 0x69, 0x64, 0x20, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x65, 0x75, 0x74, 0x61,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x72, 0x65, 0x64, 0x6e, 0x65, 0x6b, 0x6f, 0x6c, 0x79,
0x6d, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x6d, 0x63, 0x68, 0x61, 0x74,
0x6b, 0x61, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6e, 0x61, 0x64, 0x79, 0x72, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x6d, 0x61, 0x72, 0x61, 0x0, 0x45, 0x75, 0x72,
0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x6f, 0x73, 0x63, 0x6f, 0x77, 0x20, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x4b, 0x69, 0x72, 0x6f, 0x76, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53,
0x69, 0x6d, 0x66, 0x65, 0x72, 0x6f, 0x70, 0x6f, 0x6c, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d,
0x54, 0x2b, 0x33, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x52,
0x6f, 0x74, 0x68, 0x65, 0x72, 0x61, 0x20, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63,
0x61, 0x2f, 0x50, 0x61, 0x6c, 0x6d, 0x65, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x46, 0x6f, 0x72, 0x74, 0x61, 0x6c, 0x65, 0x7a, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x42, 0x65, 0x6c, 0x65, 0x6d, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x61, 0x63, 0x65, 0x69, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x52, 0x65, 0x63, 0x69, 0x66, 0x65, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x61, 0x6e, 0x74, 0x61, 0x72, 0x65, 0x6d, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63,
0x2f, 0x53, 0x74, 0x61, 0x6e, 0x6c, 0x65, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x43, 0x61, 0x79, 0x65, 0x6e, 0x6e, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x50, 0x61, 0x72, 0x61, 0x6d, 0x61, 0x72, 0x69, 0x62, 0x6f, 0x0, 0x45, 0x74, 0x63, 0x2f,
0x47, 0x4d, 0x54, 0x2b, 0x35, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x69,
0x6f, 0x5f, 0x42, 0x72, 0x61, 0x6e, 0x63, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x45, 0x69, 0x72, 0x75, 0x6e, 0x65, 0x70, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x41, 0x74, 0x69, 0x6b, 0x6f, 0x6b, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x43, 0x61, 0x79, 0x6d, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x42, 0x6f, 0x67, 0x6f, 0x74, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x47, 0x75, 0x61, 0x79, 0x61, 0x71, 0x75, 0x69, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4a, 0x61, 0x6d, 0x61, 0x69, 0x63, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x50, 0x61, 0x6e, 0x61, 0x6d, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4c, 0x69, 0x6d, 0x61, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x34,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x6e, 0x67, 0x75, 0x69, 0x6c, 0x6c,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x6e, 0x74, 0x69, 0x67, 0x75,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x75, 0x62, 0x61, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x72, 0x62, 0x61, 0x64, 0x6f, 0x73,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x61, 0x5f, 0x50, 0x61, 0x7a, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x6e, 0x61, 0x75, 0x73, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6f, 0x61, 0x5f, 0x56, 0x69, 0x73, 0x74, 0x61,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x6f, 0x5f, 0x56,
0x65, 0x6c, 0x68, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x6f, 0x72,
0x74, 0x6f, 0x6c, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6c, 0x61,
0x6e, 0x63, 0x2d, 0x53, 0x61, 0x62, 0x6c, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4b, 0x72, 0x61, 0x6c, 0x65, 0x6e, 0x64, 0x69, 0x6a, 0x6b, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x75, 0x72, 0x61, 0x63, 0x61, 0x6f, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x6f, 0x6d, 0x69, 0x6e, 0x69, 0x63, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x74, 0x6f, 0x5f, 0x44, 0x6f, 0x6d, 0x69,
0x6e, 0x67, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x72, 0x65, 0x6e,
0x61, 0x64, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x75, 0x61, 0x64,
0x65, 0x6c, 0x6f, 0x75, 0x70, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47,
0x75, 0x79, 0x61, 0x6e, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61,
0x72, 0x74, 0x69, 0x6e, 0x69, 0x71, 0x75, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x73, 0x65, 0x72, 0x72, 0x61, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x50, 0x75, 0x65, 0x72, 0x74, 0x6f, 0x5f, 0x52, 0x69, 0x63, 0x6f, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x74, 0x5f, 0x42, 0x61, 0x72, 0x74, 0x68,
0x65, 0x6c, 0x65, 0x6d, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x74,
0x5f, 0x4b, 0x69, 0x74, 0x74, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x74, 0x5f, 0x4c, 0x75, 0x63, 0x69, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4d, 0x61, 0x72, 0x69, 0x67, 0x6f, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x53, 0x74, 0x5f, 0x56, 0x69, 0x6e, 0x63, 0x65, 0x6e, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4c, 0x6f, 0x77, 0x65, 0x72, 0x5f, 0x50, 0x72, 0x69, 0x6e, 0x63, 0x65, 0x73,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x5f, 0x6f, 0x66,
0x5f, 0x53, 0x70, 0x61, 0x69, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x74, 0x5f, 0x54, 0x68, 0x6f, 0x6d, 0x61, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x69, 0x71, 0x75, 0x65, 0x6c, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53,
0x61, 0x6b, 0x68, 0x61, 0x6c, 0x69, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x41, 0x70, 0x69, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6f, 0x5f,
0x54, 0x6f, 0x6d, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x72, 0x61,
0x74, 0x6f, 0x76, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x37, 0x0, 0x41, 0x6e,
0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x76, 0x69, 0x73, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x50, 0x68, 0x6e, 0x6f, 0x6d, 0x5f, 0x50, 0x65, 0x6e, 0x68, 0x0, 0x49,
0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x68, 0x72, 0x69, 0x73, 0x74, 0x6d, 0x61, 0x73, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a, 0x61, 0x6b, 0x61, 0x72, 0x74, 0x61, 0x20, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x50, 0x6f, 0x6e, 0x74, 0x69, 0x61, 0x6e, 0x61, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x56, 0x69, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6e, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x42, 0x61, 0x6e, 0x67, 0x6b, 0x6f, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x5f,
0x43, 0x68, 0x69, 0x5f, 0x4d, 0x69, 0x6e, 0x68, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54,
0x2d, 0x38, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x72, 0x75, 0x6e, 0x65, 0x69, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x6b, 0x61, 0x73, 0x73, 0x61, 0x72, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x4b, 0x75, 0x61, 0x6c, 0x61, 0x5f, 0x4c, 0x75, 0x6d, 0x70, 0x75, 0x72, 0x20, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4b, 0x75, 0x63, 0x68, 0x69, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x4d, 0x61, 0x6e, 0x69, 0x6c, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x69, 0x6e,
0x67, 0x61, 0x70, 0x6f, 0x72, 0x65, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x32,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x61, 0x62, 0x6f, 0x72, 0x6f, 0x6e, 0x65,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x75, 0x6a, 0x75, 0x6d, 0x62, 0x75, 0x72,
0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x75, 0x62, 0x75, 0x6d, 0x62, 0x61,
0x73, 0x68, 0x69, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x62, 0x61, 0x62, 0x61,
0x6e, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x73, 0x65, 0x72, 0x75,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6c, 0x61, 0x6e, 0x74, 0x79, 0x72, 0x65,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x70, 0x75, 0x74, 0x6f, 0x0, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x69, 0x67, 0x61, 0x6c, 0x69, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4a, 0x6f, 0x68, 0x61, 0x6e, 0x6e, 0x65, 0x73, 0x62, 0x75, 0x72, 0x67,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x75, 0x73, 0x61, 0x6b, 0x61, 0x0, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x72, 0x61, 0x72, 0x65, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4a, 0x75, 0x62, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x43, 0x6f,
0x6c, 0x6f, 0x6d, 0x62, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x68, 0x61,
0x72, 0x74, 0x6f, 0x75, 0x6d, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x61, 0x6d, 0x61, 0x73,
0x63, 0x75, 0x73, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x61, 0x69, 0x70, 0x65, 0x69, 0x0,
0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x62, 0x61, 0x72, 0x74,
0x20, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x63, 0x71,
0x75, 0x61, 0x72, 0x69, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72,
0x61, 0x67, 0x75, 0x61, 0x69, 0x6e, 0x61, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d,
0x39, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a, 0x61, 0x79, 0x61, 0x70, 0x75, 0x72, 0x61, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x6f, 0x6b, 0x79, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x50, 0x61, 0x6c, 0x61, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x69,
0x6c, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x6f, 0x6d, 0x73, 0x6b, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x54, 0x6f, 0x6e, 0x67, 0x61, 0x74, 0x61, 0x70, 0x75, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x43, 0x68, 0x69, 0x74, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x49, 0x73, 0x74, 0x61, 0x6e, 0x62, 0x75, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x47, 0x72, 0x61, 0x6e, 0x64, 0x5f, 0x54, 0x75, 0x72, 0x6b, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x55, 0x6c, 0x61, 0x61, 0x6e, 0x62, 0x61, 0x61, 0x74, 0x61, 0x72, 0x20, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x43, 0x68, 0x6f, 0x69, 0x62, 0x61, 0x6c, 0x73, 0x61, 0x6e, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x49,
0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x70, 0x6f, 0x6c, 0x69, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x4d, 0x61, 0x72, 0x65,
0x6e, 0x67, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69,
0x61, 0x6e, 0x61, 0x2f, 0x56, 0x65, 0x76, 0x61, 0x79, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d,
0x54, 0x2b, 0x37, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x72, 0x65, 0x73,
0x74, 0x6f, 0x6e, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x77, 0x73,
0x6f, 0x6e, 0x5f, 0x43, 0x72, 0x65, 0x65, 0x6b, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x46, 0x6f, 0x72, 0x74, 0x5f, 0x4e, 0x65, 0x6c, 0x73, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x65, 0x72, 0x6d, 0x6f, 0x73, 0x69, 0x6c, 0x6c, 0x6f, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x68, 0x6f, 0x65, 0x6e, 0x69, 0x78, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x55, 0x54, 0x43, 0x20, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x32, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x54, 0x61, 0x72, 0x61, 0x77, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x4d, 0x61, 0x6a, 0x75, 0x72, 0x6f, 0x20, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x2f, 0x4b, 0x77, 0x61, 0x6a, 0x61, 0x6c, 0x65, 0x69, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x4e, 0x61, 0x75, 0x72, 0x75, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x2f, 0x46, 0x75, 0x6e, 0x61, 0x66, 0x75, 0x74, 0x69, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x57, 0x61, 0x6b, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x57,
0x61, 0x6c, 0x6c, 0x69, 0x73, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x33,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4b, 0x61, 0x6e, 0x74, 0x6f, 0x6e, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x46, 0x61, 0x6b, 0x61, 0x6f, 0x66, 0x6f, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x32, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x6f, 0x6e, 0x68, 0x61, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74,
0x69, 0x63, 0x2f, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x5f, 0x47, 0x65, 0x6f, 0x72, 0x67, 0x69, 0x61,
0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x38, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x50, 0x69, 0x74, 0x63, 0x61, 0x69, 0x72, 0x6e, 0x0, 0x45, 0x74, 0x63, 0x2f,
0x47, 0x4d, 0x54, 0x2b, 0x39, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x61,
0x6d, 0x62, 0x69, 0x65, 0x72, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x31,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x61, 0x67, 0x6f, 0x5f, 0x50, 0x61,
0x67, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4e, 0x69, 0x75, 0x65, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4d, 0x69, 0x64, 0x77, 0x61, 0x79, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x72, 0x61, 0x63, 0x61, 0x73, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x56, 0x6c, 0x61, 0x64, 0x69, 0x76, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x20,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x73, 0x74, 0x2d, 0x4e, 0x65, 0x72, 0x61, 0x0, 0x45, 0x75,
0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x6f, 0x6c, 0x67, 0x6f, 0x67, 0x72, 0x61, 0x64, 0x0, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x50, 0x65, 0x72, 0x74, 0x68, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x41, 0x6c, 0x67, 0x69, 0x65, 0x72, 0x73, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c,
0x75, 0x61, 0x6e, 0x64, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72,
0x74, 0x6f, 0x2d, 0x4e, 0x6f, 0x76, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44,
0x6f, 0x75, 0x61, 0x6c, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x6e,
0x67, 0x75, 0x69, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x64, 0x6a, 0x61, 0x6d,
0x65, 0x6e, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x72, 0x61, 0x7a, 0x7a,
0x61, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x69,
0x6e, 0x73, 0x68, 0x61, 0x73, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61,
0x6c, 0x61, 0x62, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x69, 0x62, 0x72,
0x65, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x61,
0x67, 0x6f, 0x73, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x69, 0x61, 0x6d, 0x65,
0x79, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x75, 0x6e, 0x69, 0x73, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41, 0x6e, 0x64, 0x6f, 0x72, 0x72, 0x61, 0x0, 0x45, 0x75,
0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x69, 0x65, 0x6e, 0x6e, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f,
0x70, 0x65, 0x2f, 0x42, 0x65, 0x72, 0x6c, 0x69, 0x6e, 0x20, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x42, 0x75, 0x73, 0x69, 0x6e, 0x67, 0x65, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x47, 0x69, 0x62, 0x72, 0x61, 0x6c, 0x74, 0x61, 0x72, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x52, 0x6f, 0x6d, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x61,
0x64, 0x75, 0x7a, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c, 0x75, 0x78, 0x65, 0x6d,
0x62, 0x6f, 0x75, 0x72, 0x67, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x61, 0x6c,
0x74, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x6f, 0x6e, 0x61, 0x63, 0x6f,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41, 0x6d, 0x73, 0x74, 0x65, 0x72, 0x64, 0x61,
0x6d, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4f, 0x73, 0x6c, 0x6f, 0x0, 0x45, 0x75,
0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x6e, 0x5f, 0x4d, 0x61, 0x72, 0x69, 0x6e, 0x6f, 0x0,
0x41, 0x72, 0x63, 0x74, 0x69, 0x63, 0x2f, 0x4c, 0x6f, 0x6e, 0x67, 0x79, 0x65, 0x61, 0x72, 0x62,
0x79, 0x65, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x74, 0x6f, 0x63, 0x6b,
0x68, 0x6f, 0x6c, 0x6d, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x5a, 0x75, 0x72, 0x69,
0x63, 0x68, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x61, 0x74, 0x69, 0x63, 0x61,
0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x76, 0x64, 0x0, 0x45, 0x74, 0x63, 0x2f,
0x47, 0x4d, 0x54, 0x2d, 0x35, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x61, 0x77, 0x73, 0x6f, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4b,
0x65, 0x72, 0x67, 0x75, 0x65, 0x6c, 0x65, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4f, 0x72,
0x61, 0x6c, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6c, 0x6d, 0x61, 0x74, 0x79, 0x20, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x41, 0x71, 0x74, 0x61, 0x75, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41,
0x71, 0x74, 0x6f, 0x62, 0x65, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x74, 0x79, 0x72, 0x61,
0x75, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x51, 0x6f, 0x73, 0x74, 0x61, 0x6e, 0x61, 0x79, 0x0,
0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x6c, 0x64, 0x69, 0x76, 0x65, 0x73, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x75, 0x73, 0x68, 0x61, 0x6e, 0x62, 0x65, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x41, 0x73, 0x68, 0x67, 0x61, 0x62, 0x61, 0x74, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x54, 0x61, 0x73, 0x68, 0x6b, 0x65, 0x6e, 0x74, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53,
0x61, 0x6d, 0x61, 0x72, 0x6b, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x65,
0x62, 0x72, 0x6f, 0x6e, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x47, 0x61, 0x7a, 0x61, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x30, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63,
0x74, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x75, 0x6d, 0x6f, 0x6e, 0x74, 0x44, 0x55, 0x72, 0x76, 0x69,
0x6c, 0x6c, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x75, 0x61, 0x6d,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x43, 0x68, 0x75, 0x75, 0x6b, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x53, 0x61, 0x69, 0x70, 0x61, 0x6e, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x5f, 0x4d, 0x6f, 0x72, 0x65, 0x73,
0x62, 0x79, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x61, 0x6b, 0x75, 0x74, 0x73, 0x6b, 0x20,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x68, 0x61, 0x6e, 0x64, 0x79, 0x67, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x68, 0x69, 0x74, 0x65, 0x68, 0x6f, 0x72, 0x73, 0x65,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x77, 0x73, 0x6f, 0x6e, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x6e, 0x63, 0x68, 0x6f, 0x72, 0x61, 0x67,
0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74,
0x69, 0x6e, 0x61, 0x2f, 0x42, 0x75, 0x65, 0x6e, 0x6f, 0x73, 0x5f, 0x41, 0x69, 0x72, 0x65, 0x73,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41, 0x73, 0x74, 0x72, 0x61, 0x6b, 0x68, 0x61,
0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x6c, 0x69, 0x66, 0x61,
0x78, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x53, 0x79, 0x64, 0x6e,
0x65, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x65, 0x67, 0x69, 0x6e,
0x61, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x41, 0x64, 0x65, 0x6c,
0x61, 0x69, 0x64, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x75, 0x69,
0x61, 0x62, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x68, 0x69, 0x63,
0x61, 0x67, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x78, 0x69,
0x63, 0x6f, 0x5f, 0x43, 0x69, 0x74, 0x79, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69,
0x61, 0x2f, 0x42, 0x72, 0x69, 0x73, 0x62, 0x61, 0x6e, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4e, 0x65, 0x77, 0x5f, 0x59, 0x6f, 0x72, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x44, 0x65, 0x6e, 0x76, 0x65, 0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x4b, 0x72, 0x61, 0x73, 0x6e, 0x6f, 0x79, 0x61, 0x72, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x4b, 0x61, 0x6d, 0x63, 0x68, 0x61, 0x74, 0x6b, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x4d, 0x6f, 0x73, 0x63, 0x6f, 0x77, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c,
0x69, 0x61, 0x2f, 0x48, 0x6f, 0x62, 0x61, 0x72, 0x74, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55,
0x6c, 0x61, 0x61, 0x6e, 0x62, 0x61, 0x61, 0x74, 0x61, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61,
0x6e, 0x61, 0x70, 0x6f, 0x6c, 0x69, 0x73, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x55, 0x54, 0x43, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x56, 0x6c, 0x61, 0x64, 0x69, 0x76, 0x6f, 0x73, 0x74, 0x6f, 0x6b,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x65, 0x72, 0x6c, 0x69, 0x6e, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x54, 0x61, 0x73, 0x68, 0x6b, 0x65, 0x6e, 0x74, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x48, 0x65, 0x62, 0x72, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x61,
0x6b, 0x75, 0x74, 0x73, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x68,
0x69, 0x74, 0x65, 0x68, 0x6f, 0x72, 0x73, 0x65, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x31, 0x34, 0x3a,
0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x31, 0x33, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43,
0x2d, 0x31, 0x32, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x31, 0x31, 0x3a, 0x30, 0x30,
0x0, 0x55, 0x54, 0x43, 0x2d, 0x31, 0x30, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30,
0x39, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x39, 0x3a, 0x30, 0x30, 0x0, 0x55,
0x54, 0x43, 0x2d, 0x30, 0x38, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x37, 0x3a,
0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x36, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43,
0x2d, 0x30, 0x35, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x34, 0x3a, 0x33, 0x30,
0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x34, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30,
0x33, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x33, 0x3a, 0x30, 0x30, 0x0, 0x55,
0x54, 0x43, 0x2d, 0x30, 0x32, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x31, 0x3a,
0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x20, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x30, 0x3a, 0x30, 0x30,
0x20, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x30, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30,
0x31, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x32, 0x3a, 0x30, 0x30, 0x0, 0x55,
0x54, 0x43, 0x2b, 0x30, 0x33, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x33, 0x3a,
0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x34, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43,
0x2b, 0x30, 0x34, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x35, 0x3a, 0x30, 0x30,
0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x35, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30,
0x35, 0x3a, 0x34, 0x35, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x36, 0x3a, 0x30, 0x30, 0x0, 0x55,
0x54, 0x43, 0x2b, 0x30, 0x36, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x37, 0x3a,
0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x38, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43,
0x2b, 0x30, 0x38, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x38, 0x3a, 0x34, 0x35,
0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x39, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30,
0x39, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x30, 0x3a, 0x30, 0x30, 0x0, 0x55,
0x54, 0x43, 0x2b, 0x31, 0x30, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x31, 0x3a,
0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x32, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43,
0x2b, 0x31, 0x32, 0x3a, 0x34, 0x35, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x33, 0x3a, 0x30, 0x30,
0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x34, 0x3a, 0x30, 0x30, 0x0
};

static constexpr char aliasIdData[] = {
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x73, 0x6d, 0x65, 0x72, 0x61, 0x0, 0x41, 0x66,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x73, 0x6d, 0x61, 0x72, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x54, 0x69, 0x6d, 0x62, 0x75, 0x6b, 0x74, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x42, 0x61, 0x6d, 0x61, 0x6b, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x43, 0x6f, 0x6d, 0x6f,
0x64, 0x52, 0x69, 0x76, 0x61, 0x64, 0x61, 0x76, 0x69, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x43, 0x61, 0x74,
0x61, 0x6d, 0x61, 0x72, 0x63, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41,
0x74, 0x6b, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x64, 0x61, 0x6b,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x75, 0x65, 0x6e, 0x6f, 0x73, 0x5f,
0x41, 0x69, 0x72, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72,
0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x42, 0x75, 0x65, 0x6e, 0x6f, 0x73, 0x5f, 0x41,
0x69, 0x72, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x74,
0x61, 0x6d, 0x61, 0x72, 0x63, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43,
0x6f, 0x72, 0x61, 0x6c, 0x5f, 0x48, 0x61, 0x72, 0x62, 0x6f, 0x75, 0x72, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x74, 0x69, 0x6b, 0x6f, 0x6b, 0x61, 0x6e, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x6f, 0x72, 0x64, 0x6f, 0x62, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f,
0x43, 0x6f, 0x72, 0x64, 0x6f, 0x62, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x45, 0x6e, 0x73, 0x65, 0x6e, 0x61, 0x64, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x54, 0x69, 0x6a, 0x75, 0x61, 0x6e, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x46, 0x6f, 0x72, 0x74, 0x5f, 0x57, 0x61, 0x79, 0x6e, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69,
0x61, 0x6e, 0x61, 0x70, 0x6f, 0x6c, 0x69, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x47, 0x6f, 0x64, 0x74, 0x68, 0x61, 0x62, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4e, 0x75, 0x75, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x61, 0x70, 0x6f, 0x6c, 0x69, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4a, 0x75, 0x6a, 0x75, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4a, 0x75, 0x6a, 0x75, 0x79,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x6e, 0x6f, 0x78, 0x5f, 0x49, 0x4e,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61,
0x2f, 0x4b, 0x6e, 0x6f, 0x78, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f,
0x75, 0x69, 0x73, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4b, 0x65, 0x6e, 0x74, 0x75, 0x63, 0x6b, 0x79, 0x2f, 0x4c, 0x6f, 0x75, 0x69, 0x73, 0x76,
0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x6e,
0x64, 0x6f, 0x7a, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67,
0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4d, 0x65, 0x6e, 0x64, 0x6f, 0x7a, 0x61, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x72, 0x65, 0x61, 0x6c, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x6f, 0x72, 0x6f, 0x6e, 0x74, 0x6f, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x69, 0x70, 0x69, 0x67, 0x6f, 0x6e, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x61, 0x6e, 0x67, 0x6e, 0x69, 0x72, 0x74,
0x75, 0x6e, 0x67, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x71, 0x61, 0x6c,
0x75, 0x69, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74,
0x6f, 0x5f, 0x41, 0x63, 0x72, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52,
0x69, 0x6f, 0x5f, 0x42, 0x72, 0x61, 0x6e, 0x63, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x52, 0x61, 0x69, 0x6e, 0x79, 0x5f, 0x52, 0x69, 0x76, 0x65, 0x72, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x69, 0x6e, 0x6e, 0x69, 0x70, 0x65, 0x67, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x6f, 0x73, 0x61, 0x72, 0x69, 0x6f, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x74, 0x61, 0x5f, 0x49, 0x73, 0x61,
0x62, 0x65, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x68, 0x69, 0x70,
0x72, 0x6f, 0x63, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x65, 0x6e,
0x76, 0x65, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x68, 0x75, 0x6e,
0x64, 0x65, 0x72, 0x5f, 0x42, 0x61, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x56, 0x69, 0x72, 0x67, 0x69, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x74, 0x5f, 0x54, 0x68, 0x6f, 0x6d, 0x61, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x59, 0x65, 0x6c, 0x6c, 0x6f, 0x77, 0x6b, 0x6e, 0x69, 0x66, 0x65, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x45, 0x64, 0x6d, 0x6f, 0x6e, 0x74, 0x6f, 0x6e, 0x0, 0x41, 0x6e,
0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x5f, 0x50,
0x6f, 0x6c, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x41, 0x75, 0x63, 0x6b,
0x6c, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x73, 0x68, 0x6b, 0x68, 0x61,
0x62, 0x61, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x73, 0x68, 0x67, 0x61, 0x62, 0x61,
0x74, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x43, 0x61, 0x6c, 0x63, 0x75, 0x74, 0x74, 0x61, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x6f, 0x6c, 0x6b, 0x61, 0x74, 0x61, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x43, 0x68, 0x6f, 0x6e, 0x67, 0x71, 0x69, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x53, 0x68, 0x61, 0x6e, 0x67, 0x68, 0x61, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x43,
0x68, 0x75, 0x6e, 0x67, 0x6b, 0x69, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x61,
0x63, 0x63, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x68, 0x61, 0x6b, 0x61, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x48, 0x61, 0x72, 0x62, 0x69, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x49, 0x73, 0x74, 0x61, 0x6e, 0x62, 0x75, 0x6c, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x49, 0x73, 0x74, 0x61, 0x6e, 0x62, 0x75, 0x6c, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61,
0x73, 0x68, 0x67, 0x61, 0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x72, 0x75, 0x6d, 0x71,
0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x74, 0x6d, 0x61, 0x6e, 0x64, 0x75, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x74, 0x68, 0x6d, 0x61, 0x6e, 0x64, 0x75, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x63, 0x61, 0x6f, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d,
0x61, 0x63, 0x61, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x52, 0x61, 0x6e, 0x67, 0x6f, 0x6f,
0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x61, 0x6e, 0x67, 0x6f, 0x6e, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x53, 0x61, 0x69, 0x67, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48,
0x6f, 0x5f, 0x43, 0x68, 0x69, 0x5f, 0x4d, 0x69, 0x6e, 0x68, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x54, 0x65, 0x6c, 0x5f, 0x41, 0x76, 0x69, 0x76, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a, 0x65,
0x72, 0x75, 0x73, 0x61, 0x6c, 0x65, 0x6d, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x68, 0x69,
0x6d, 0x62, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x68, 0x69, 0x6d, 0x70, 0x68, 0x75,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x6a, 0x75, 0x6e, 0x67, 0x5f, 0x50, 0x61, 0x6e, 0x64,
0x61, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x6b, 0x61, 0x73, 0x73, 0x61,
0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x6c, 0x61, 0x6e, 0x5f, 0x42, 0x61, 0x74, 0x6f,
0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x6c, 0x61, 0x61, 0x6e, 0x62, 0x61, 0x61, 0x74,
0x61, 0x72, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x46, 0x61, 0x65, 0x72,
0x6f, 0x65, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x46, 0x61, 0x72, 0x6f,
0x65, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x4a, 0x61, 0x6e, 0x5f, 0x4d,
0x61, 0x79, 0x65, 0x6e, 0x0, 0x41, 0x72, 0x63, 0x74, 0x69, 0x63, 0x2f, 0x4c, 0x6f, 0x6e, 0x67,
0x79, 0x65, 0x61, 0x72, 0x62, 0x79, 0x65, 0x6e, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c,
0x69, 0x61, 0x2f, 0x41, 0x43, 0x54, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61,
0x2f, 0x53, 0x79, 0x64, 0x6e, 0x65, 0x79, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69,
0x61, 0x2f, 0x43, 0x61, 0x6e, 0x62, 0x65, 0x72, 0x72, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72,
0x61, 0x6c, 0x69, 0x61, 0x2f, 0x43, 0x75, 0x72, 0x72, 0x69, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74,
0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x62, 0x61, 0x72, 0x74, 0x0, 0x41, 0x75, 0x73,
0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x48, 0x49, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72,
0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x6f, 0x72, 0x64, 0x5f, 0x48, 0x6f, 0x77, 0x65, 0x0, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4e, 0x53, 0x57, 0x0, 0x41, 0x75, 0x73,
0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x0, 0x41, 0x75, 0x73,
0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x44, 0x61, 0x72, 0x77, 0x69, 0x6e, 0x0, 0x41, 0x75,
0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x51, 0x75, 0x65, 0x65, 0x6e, 0x73, 0x6c, 0x61,
0x6e, 0x64, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x42, 0x72, 0x69,
0x73, 0x62, 0x61, 0x6e, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f,
0x53, 0x6f, 0x75, 0x74, 0x68, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f,
0x41, 0x64, 0x65, 0x6c, 0x61, 0x69, 0x64, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c,
0x69, 0x61, 0x2f, 0x54, 0x61, 0x73, 0x6d, 0x61, 0x6e, 0x69, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74,
0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x56, 0x69, 0x63, 0x74, 0x6f, 0x72, 0x69, 0x61, 0x0, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4d, 0x65, 0x6c, 0x62, 0x6f, 0x75, 0x72,
0x6e, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x57, 0x65, 0x73,
0x74, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x50, 0x65, 0x72, 0x74,
0x68, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x59, 0x61, 0x6e, 0x63,
0x6f, 0x77, 0x69, 0x6e, 0x6e, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61,
0x2f, 0x42, 0x72, 0x6f, 0x6b, 0x65, 0x6e, 0x5f, 0x48, 0x69, 0x6c, 0x6c, 0x0, 0x42, 0x72, 0x61,
0x7a, 0x69, 0x6c, 0x2f, 0x41, 0x63, 0x72, 0x65, 0x0, 0x42, 0x72, 0x61, 0x7a, 0x69, 0x6c, 0x2f,
0x44, 0x65, 0x4e, 0x6f, 0x72, 0x6f, 0x6e, 0x68, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x6f, 0x6e, 0x68, 0x61, 0x0, 0x42, 0x72, 0x61, 0x7a, 0x69, 0x6c,
0x2f, 0x45, 0x61, 0x73, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61,
0x6f, 0x5f, 0x50, 0x61, 0x75, 0x6c, 0x6f, 0x0, 0x42, 0x72, 0x61, 0x7a, 0x69, 0x6c, 0x2f, 0x57,
0x65, 0x73, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x6e, 0x61,
0x75, 0x73, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74,
0x69, 0x63, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x6c, 0x69, 0x66,
0x61, 0x78, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61,
0x6c, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x45, 0x61, 0x73, 0x74, 0x2d, 0x53, 0x61,
0x73, 0x6b, 0x61, 0x74, 0x63, 0x68, 0x65, 0x77, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x52, 0x65, 0x67, 0x69, 0x6e, 0x61, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61,
0x2f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f,
0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f,
0x4e, 0x65, 0x77, 0x66, 0x6f, 0x75, 0x6e, 0x64, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x74, 0x5f, 0x4a, 0x6f, 0x68, 0x6e, 0x73, 0x0, 0x43, 0x61,
0x6e, 0x61, 0x64, 0x61, 0x2f, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x56, 0x61, 0x6e, 0x63, 0x6f, 0x75, 0x76, 0x65, 0x72, 0x0, 0x43,
0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x53, 0x61, 0x73, 0x6b, 0x61, 0x74, 0x63, 0x68, 0x65, 0x77,
0x61, 0x6e, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x59, 0x75, 0x6b, 0x6f, 0x6e, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x68, 0x69, 0x74, 0x65, 0x68, 0x6f, 0x72,
0x73, 0x65, 0x0, 0x43, 0x68, 0x69, 0x6c, 0x65, 0x2f, 0x43, 0x6f, 0x6e, 0x74, 0x69, 0x6e, 0x65,
0x6e, 0x74, 0x61, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e,
0x74, 0x69, 0x61, 0x67, 0x6f, 0x0, 0x43, 0x68, 0x69, 0x6c, 0x65, 0x2f, 0x45, 0x61, 0x73, 0x74,
0x65, 0x72, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x2f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x0, 0x43, 0x75, 0x62, 0x61, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x76, 0x61, 0x6e, 0x61, 0x0, 0x45, 0x53, 0x54, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x35, 0x0, 0x45, 0x67, 0x79, 0x70, 0x74, 0x0,
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x69, 0x72, 0x6f, 0x0, 0x45, 0x69, 0x72,
0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x44, 0x75, 0x62, 0x6c, 0x69, 0x6e, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x30, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d,
0x54, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x30, 0x0, 0x45, 0x74, 0x63, 0x2f,
0x47, 0x4d, 0x54, 0x30, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x72, 0x65, 0x65, 0x6e, 0x77, 0x69,
0x63, 0x68, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x55, 0x43, 0x54, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x55,
0x54, 0x43, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x55, 0x6e, 0x69, 0x76, 0x65, 0x72, 0x73, 0x61, 0x6c,
0x0, 0x45, 0x74, 0x63, 0x2f, 0x5a, 0x75, 0x6c, 0x75, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x42, 0x65, 0x6c, 0x66, 0x61, 0x73, 0x74, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x4c, 0x6f, 0x6e, 0x64, 0x6f, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x69,
0x65, 0x76, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x79, 0x69, 0x76, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4e, 0x69, 0x63, 0x6f, 0x73, 0x69, 0x61, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x4e, 0x69, 0x63, 0x6f, 0x73, 0x69, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x54, 0x69, 0x72, 0x61, 0x73, 0x70, 0x6f, 0x6c, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x43, 0x68, 0x69, 0x73, 0x69, 0x6e, 0x61, 0x75, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x55, 0x7a, 0x68, 0x67, 0x6f, 0x72, 0x6f, 0x64, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x5a, 0x61, 0x70, 0x6f, 0x72, 0x6f, 0x7a, 0x68, 0x79, 0x65, 0x0, 0x47, 0x42, 0x0,
0x47, 0x42, 0x2d, 0x45, 0x69, 0x72, 0x65, 0x0, 0x47, 0x4d, 0x54, 0x0, 0x47, 0x4d, 0x54, 0x2b,
0x30, 0x0, 0x47, 0x4d, 0x54, 0x2d, 0x30, 0x0, 0x47, 0x4d, 0x54, 0x30, 0x0, 0x47, 0x72, 0x65,
0x65, 0x6e, 0x77, 0x69, 0x63, 0x68, 0x0, 0x48, 0x53, 0x54, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47,
0x4d, 0x54, 0x2b, 0x31, 0x30, 0x0, 0x48, 0x6f, 0x6e, 0x67, 0x6b, 0x6f, 0x6e, 0x67, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x6e, 0x67, 0x5f, 0x4b, 0x6f, 0x6e, 0x67, 0x0, 0x49, 0x63,
0x65, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x52,
0x65, 0x79, 0x6b, 0x6a, 0x61, 0x76, 0x69, 0x6b, 0x0, 0x49, 0x72, 0x61, 0x6e, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x54, 0x65, 0x68, 0x72, 0x61, 0x6e, 0x0, 0x49, 0x73, 0x72, 0x61, 0x65, 0x6c,
0x0, 0x4a, 0x61, 0x6d, 0x61, 0x69, 0x63, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4a, 0x61, 0x6d, 0x61, 0x69, 0x63, 0x61, 0x0, 0x4a, 0x61, 0x70, 0x61, 0x6e, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x54, 0x6f, 0x6b, 0x79, 0x6f, 0x0, 0x4b, 0x77, 0x61, 0x6a, 0x61, 0x6c,
0x65, 0x69, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4b, 0x77, 0x61, 0x6a,
0x61, 0x6c, 0x65, 0x69, 0x6e, 0x0, 0x4c, 0x69, 0x62, 0x79, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x54, 0x72, 0x69, 0x70, 0x6f, 0x6c, 0x69, 0x0, 0x4d, 0x53, 0x54, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x37, 0x0, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x2f,
0x42, 0x61, 0x6a, 0x61, 0x4e, 0x6f, 0x72, 0x74, 0x65, 0x0, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f,
0x2f, 0x42, 0x61, 0x6a, 0x61, 0x53, 0x75, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x61, 0x7a, 0x61, 0x74, 0x6c, 0x61, 0x6e, 0x0, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f,
0x2f, 0x47, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x5f, 0x43, 0x69, 0x74, 0x79, 0x0, 0x4e, 0x5a, 0x0,
0x4e, 0x5a, 0x2d, 0x43, 0x48, 0x41, 0x54, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x43, 0x68, 0x61, 0x74, 0x68, 0x61, 0x6d, 0x0, 0x4e, 0x61, 0x76, 0x61, 0x6a, 0x6f, 0x0, 0x50,
0x52, 0x43, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x45, 0x6e, 0x64, 0x65, 0x72,
0x62, 0x75, 0x72, 0x79, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4b, 0x61, 0x6e,
0x74, 0x6f, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4a, 0x6f, 0x68, 0x6e,
0x73, 0x74, 0x6f, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x48, 0x6f, 0x6e,
0x6f, 0x6c, 0x75, 0x6c, 0x75, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x6f,
0x6e, 0x61, 0x70, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x6f, 0x68,
0x6e, 0x70, 0x65, 0x69, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x53, 0x61, 0x6d,
0x6f, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x61, 0x67, 0x6f, 0x5f,
0x50, 0x61, 0x67, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x54, 0x72, 0x75,
0x6b, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x43, 0x68, 0x75, 0x75, 0x6b, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x59, 0x61, 0x70, 0x0, 0x50, 0x6f, 0x6c, 0x61,
0x6e, 0x64, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x57, 0x61, 0x72, 0x73, 0x61, 0x77,
0x0, 0x50, 0x6f, 0x72, 0x74, 0x75, 0x67, 0x61, 0x6c, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x4c, 0x69, 0x73, 0x62, 0x6f, 0x6e, 0x0, 0x52, 0x4f, 0x43, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x54, 0x61, 0x69, 0x70, 0x65, 0x69, 0x0, 0x52, 0x4f, 0x4b, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x53, 0x65, 0x6f, 0x75, 0x6c, 0x0, 0x53, 0x69, 0x6e, 0x67, 0x61, 0x70, 0x6f, 0x72, 0x65,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x69, 0x6e, 0x67, 0x61, 0x70, 0x6f, 0x72, 0x65, 0x0,
0x54, 0x75, 0x72, 0x6b, 0x65, 0x79, 0x0, 0x55, 0x43, 0x54, 0x0, 0x55, 0x53, 0x2f, 0x41, 0x6c,
0x61, 0x73, 0x6b, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x6e, 0x63,
0x68, 0x6f, 0x72, 0x61, 0x67, 0x65, 0x0, 0x55, 0x53, 0x2f, 0x41, 0x6c, 0x65, 0x75, 0x74, 0x69,
0x61, 0x6e, 0x0, 0x55, 0x53, 0x2f, 0x41, 0x72, 0x69, 0x7a, 0x6f, 0x6e, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x68, 0x6f, 0x65, 0x6e, 0x69, 0x78, 0x0, 0x55, 0x53,
0x2f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x43, 0x68, 0x69, 0x63, 0x61, 0x67, 0x6f, 0x0, 0x55, 0x53, 0x2f, 0x45, 0x61, 0x73, 0x74,
0x2d, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x0, 0x55, 0x53, 0x2f, 0x45, 0x61, 0x73, 0x74,
0x65, 0x72, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x65, 0x77, 0x5f,
0x59, 0x6f, 0x72, 0x6b, 0x0, 0x55, 0x53, 0x2f, 0x48, 0x61, 0x77, 0x61, 0x69, 0x69, 0x0, 0x55,
0x53, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2d, 0x53, 0x74, 0x61, 0x72, 0x6b, 0x65,
0x0, 0x55, 0x53, 0x2f, 0x4d, 0x69, 0x63, 0x68, 0x69, 0x67, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x65, 0x74, 0x72, 0x6f, 0x69, 0x74, 0x0, 0x55, 0x53, 0x2f,
0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x0, 0x55, 0x53, 0x2f, 0x50, 0x61, 0x63, 0x69,
0x66, 0x69, 0x63, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f, 0x73, 0x5f,
0x41, 0x6e, 0x67, 0x65, 0x6c, 0x65, 0x73, 0x0, 0x55, 0x53, 0x2f, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2d, 0x4e, 0x65, 0x77, 0x0, 0x55, 0x53, 0x2f, 0x53, 0x61, 0x6d, 0x6f, 0x61, 0x0,
0x55, 0x54, 0x43, 0x0, 0x55, 0x6e, 0x69, 0x76, 0x65, 0x72, 0x73, 0x61, 0x6c, 0x0, 0x57, 0x2d,
0x53, 0x55, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x6f, 0x73, 0x63, 0x6f, 0x77,
0x0, 0x5a, 0x75, 0x6c, 0x75, 0x0
};
// GENERATED PART ENDS HERE

constexpr QByteArrayView AliasData::aliasId() const { return aliasIdData + aliasIdIndex; }
constexpr QByteArrayView AliasData::ianaId() const { return aliasIdData + ianaIdIndex; }
constexpr QByteArrayView WindowsData::windowsId() const { return windowsIdData + windowsIdIndex; }
// Each of the following returns a space-joined sequence of IANA IDs:
constexpr QByteArrayView WindowsData::ianaId() const { return ianaIdData + ianaIdIndex; }
constexpr QByteArrayView UtcData::id() const { return ianaIdData + ianaIdIndex; }
constexpr QLatin1StringView ZoneData::id() const
{ return QLatin1StringView(ianaIdData + ianaIdIndex); }

} // namespace QtTimeZoneCldr

QT_END_NAMESPACE

#endif // QTIMEZONEPRIVATE_DATA_P_H
