/*-------------------------------------------------------------------------
 *
 * parser.c		- Parser part of the PL/pgPSM
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2012, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  src/pl/plpgpsm/parser.c
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"

#include "funcapi.h"

#include "access/sysattr.h"
#include "catalog/heap.h"
#include "catalog/namespace.h"
#include "catalog/pg_proc.h"
#include "catalog/pg_type.h"
#include "executor/spi.h"
#include "parser/parse_node.h"
#include "utils/builtins.h"
#include "utils/elog.h"
#include "utils/syscache.h"

static Node *post_column_ref(ParseState *pstate, ColumnRef *cref, Node *var);
static Node *param_ref(ParseState *pstate, ParamRef *pref);

static void second_stage(PLpgPSM_parser *pstate, PLpgPSM_astnode *astnode);
static List *make_params(HeapTuple procTup);
static PLpgPSM_astnode *get_labeled_statement(PLpgPSM_astnode *astnode, char *label);
static PLpgPSM_declare_param *new_declparam(PLpgPSM_variable *variable, Oid typeid,
								char param_mode,
								int param_offset);
static void check_condition_overlap(PLpgPSM_eb_stmt *stmt);


PLpgPSM_parser  parser;

#define parser_errposition1(pos)  plpgpsm_scanner_errposition(pos)

#define INIT_NODE(n, t, v, l) \
	do { \
		n->astnode.type = t; \
		n->astnode.is_visible = v; \
		n->lineno = plpgpsm_location_to_lineno(l); \
		n->location = l; \
	} while (0)

#define SET_PARENT(a1, a2) \
	do { \
		a1->astnode.parent = (PLpgPSM_astnode *) a2; \
	} while (0)


/* ----------
 * Lookup table for predefined condition names
 * ----------
 */
typedef struct
{
	const char *label;
	int			sqlerrstate;
} ConditionLabelMap;

static const ConditionLabelMap condition_label_map[] = {
#include "plerrcodes.h"	/* pgrminclude ignore */
	{NULL, 0}
};


PLpgPSM_function *
plpgpsm_compile(FunctionCallInfo fcinfo, bool forValidator)
{
	Oid		funcOid = fcinfo->flinfo->fn_oid;
	HeapTuple		procTup;
	char		*prosrc;
	Datum		prosrcdatum;
	bool			isnull;
	int	parse_rc;
	PLpgPSM_function	*result;
	List		*params;
	Form_pg_proc procStruct;
	char *proname;

	result = (PLpgPSM_function *) palloc0(sizeof(PLpgPSM_function));

	/* Lookup the pg_proc tuple by Oid */
	procTup = SearchSysCache1(PROCOID, ObjectIdGetDatum(funcOid));
	if (!HeapTupleIsValid(procTup))
		elog(ERROR, "cache lookup failed for function %u", funcOid);

	/* Take source code */
	prosrcdatum = SysCacheGetAttr(PROCOID, procTup,
							    Anum_pg_proc_prosrc, &isnull);
	if (isnull)
		elog(ERROR, "null prosrc");
	prosrc = TextDatumGetCString(prosrcdatum);

	procStruct = (Form_pg_proc) GETSTRUCT(procTup);
	proname = pstrdup(NameStr(procStruct->proname));

	parser.nvars = 0;
	parser.nargs = 0;
	parser.noutargs = 0;
	parser.is_void = procStruct->prorettype == VOIDOID;

	plpgpsm_scanner_init(prosrc);

	params = make_params(procTup);

	parse_rc = plpgpsm_yyparse();
	if (parse_rc != 0)
		elog(ERROR, "plpgpsm parser returned %d", parse_rc);

	if (params != NIL)
	{
		params = lappend(params, parser.asttree);
		parser.asttree = (PLpgPSM_astnode *) plpgpsm_eb_stmt(PLPGPSM_STMT_COMPOUND, NULL,
											    -1,
											    proname,
											    NULL,
											    params);
		parser.asttree->is_visible = false;
	}

	parser.asttree->parent = NULL;

	second_stage(&parser, parser.asttree);

	plpgpsm_scanner_finish();
	pfree(prosrc);

	ReleaseSysCache(procTup);

	result = (PLpgPSM_function *) palloc0(sizeof(PLpgPSM_function));
	result->astroot = parser.asttree;

	result->parser = (PLpgPSM_parser *) palloc(sizeof(PLpgPSM_parser));
	result->parser->nvars = parser.nvars;
	result->parser->nargs = parser.nargs;
	result->parser->noutargs = parser.noutargs;

	return result;
}

/*
 * get_near_compound_stmt - returns near compound statement
 *
 */
PLpgPSM_astnode *
plpgpsm_get_near_compound_stmt(PLpgPSM_astnode *astnode)
{
	if (astnode == NULL)
		return NULL;

	if (astnode->type != PLPGPSM_STMT_COMPOUND)
		return plpgpsm_get_near_compound_stmt(astnode->parent);
	else
		return astnode;
}

/*
 * returns a variable (or cursor) specified by name and label, returns NULL
 * when variable (or cursor) is not declared
 *
 */
PLpgPSM_astnode *
plpgpsm_search_variable(PLpgPSM_execstate *estate, PLpgPSM_ident *ident, PLpgPSM_astnode *scope, int *fnumber)
{
	if (scope == NULL)
		return NULL;

	if (fnumber != NULL)
		*fnumber = SPI_ERROR_NOATTRIBUTE;

	/* variables should be defined only inside compound statement */
	if (scope->type == PLPGPSM_STMT_COMPOUND)
	{
		PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) scope;
		ListCell	*lc;

		if (ident->scope != NULL)
		{
			if (stmt->label == NULL)
				return plpgpsm_search_variable(estate, ident, scope->parent, fnumber);

			if (strcmp(ident->scope, stmt->label) != 0)
				return plpgpsm_search_variable(estate, ident, scope->parent, fnumber);
		}

		foreach (lc, stmt->stmts)
		{
			PLpgPSM_astnode *iter = (PLpgPSM_astnode *) lfirst(lc);

			if (iter->type == PLPGPSM_DECLARE_CONDITION)
			{
				continue;
			}
			if (iter->type == PLPGPSM_DECLARE_VARIABLE)
			{
				PLpgPSM_declare_variable *dvar = (PLpgPSM_declare_variable *) iter;
				ListCell			*lcvar;

				foreach (lcvar, dvar->variables)
				{
					PLpgPSM_variable *var = (PLpgPSM_variable *) lfirst(lcvar);

					Assert (((PLpgPSM_astnode *) lfirst(lcvar))->type == PLPGPSM_VARIABLE);

					if (strcmp(ident->name, var->name) == 0)
						return (PLpgPSM_astnode *) var;
				}
			}
			else if (iter->type == PLPGPSM_DECLARE_PARAM)
			{
				PLpgPSM_declare_param *dpar = (PLpgPSM_declare_param *) iter;

				if (strcmp(ident->name, dpar->variable->name) == 0)
					return (PLpgPSM_astnode *) dpar->variable;
			}
			else if (iter->type == PLPGPSM_DECLARE_CURSOR)
			{
				PLpgPSM_declare_cursor *dcur = (PLpgPSM_declare_cursor *) iter;

				if (strcmp(ident->name, dcur->cursor->name) == 0)
					return (PLpgPSM_astnode *) dcur->cursor;
			}
			else
				/* variables should be declared before any other statements */
				return plpgpsm_search_variable(estate, ident, scope->parent, fnumber);
		}
	}
	else if (scope->type == PLPGPSM_STMT_FOR)
	{
		if (estate != NULL)
		{
			PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) scope;
			PLpgPSM_datum *datum;
			PLpgPSM_fetch_extra *extra;

			if (ident->scope != NULL)
			{
				/* leave when expected label is different than current */
				if (stmt->namespace == NULL)
					return plpgpsm_search_variable(estate, ident, scope->parent, fnumber);

				if (strcmp(ident->scope, stmt->namespace) != 0)
					return plpgpsm_search_variable(estate, ident, scope->parent, fnumber);
			}

			/* only in runtime, FOR statement can hold a TupleDesc */
			Assert(stmt->cursor != NULL);

			datum = &estate->datums[stmt->cursor->offset];
			Assert(datum->class == PLPGPSM_DATUM_CURSOR);

			/* cursor must be opened */
			if (!datum->isnull)
			{
				int	_fnumber;

				extra = (PLpgPSM_fetch_extra *) datum->extra;
				Assert(extra != NULL && extra->is_ready && extra->cursor_tupdesc != NULL);

				_fnumber = SPI_fnumber(extra->cursor_tupdesc, ident->name);
				if (_fnumber != SPI_ERROR_NOATTRIBUTE)
				{
					Assert(fnumber != NULL);
					*fnumber = _fnumber;

					return (PLpgPSM_astnode *) stmt->cursor;
				}
			}
		}
		return plpgpsm_search_variable(estate, ident, scope->parent, fnumber);
	}

	return plpgpsm_search_variable(estate, ident, scope->parent, fnumber);
}

/*
 * replace saved handler, when new distance is less than previous distance
 *
 */
static void
actualise_possible_handler(PLpgPSM_declare_handler *handler, int distance,
					PLpgPSM_declare_handler **possible_handler,
					int *min_distance)
{
	if (*min_distance == -1 || *min_distance > distance)
	{
		/* initialisation */
		*possible_handler = handler;
		*min_distance = distance;
	}
	else if (*min_distance == distance)
		elog(ERROR, "ambigonuous handlers");
}

/*
 * compare condition, returns true, when is exact match of conditions
 *
 */
static bool
compare_condition(PLpgPSM_condition *condition, PLpgPSM_condition *handled_condition,
							PLpgPSM_declare_handler *handler,
								PLpgPSM_declare_handler **possible_handler,
								int	*distance)
{
	if (condition->type == PLPGPSM_CONDITION_SQLSTATE)
	{
		if (handled_condition->type == PLPGPSM_CONDITION_SQLSTATE || handled_condition->type == PLPGPSM_CONDITION_NAMED)
		{
			/* leave when named condition hasnot attached SQLSTATE */
			if (handled_condition->type == PLPGPSM_CONDITION_NAMED && handled_condition->sqlstate == 0)
				return false;

			/* exact match */
			if (condition->sqlstate == handled_condition->sqlstate)
				return true;

			/* category match, when handled_condition is a category */
			if (ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(handled_condition->sqlstate) &&
			    ERRCODE_TO_CATEGORY(handled_condition->sqlstate) == handled_condition->sqlstate)
				actualise_possible_handler(handler, 1, possible_handler, distance);
		}
		else if (handled_condition->type == PLPGPSM_CONDITION_NOTFOUND)
		{
			/* exact match */
			if (condition->sqlstate == MAKE_SQLSTATE('0','2','0','0','0'))
				return true;

			/* category match when handled_condition is a category */
			if (ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(MAKE_SQLSTATE('0','2','0','0','0')))
				actualise_possible_handler(handler, 1, possible_handler, distance);
		}
		else if (handled_condition->type == PLPGPSM_CONDITION_SQLWARNING)
		{
			if (ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(MAKE_SQLSTATE('0','1','0','0','0')) ||
				ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(MAKE_SQLSTATE('0','2','0','0','0')))
				actualise_possible_handler(handler, 2, possible_handler, distance);
		}
	}
	else if (condition->type == PLPGPSM_CONDITION_NAMED)
	{
		if (handled_condition->type == PLPGPSM_CONDITION_NAMED)
		{
			return (strcmp(condition->name, handled_condition->name) == 0);
		}
		else if (condition->sqlstate != 0)
		{
			if (handled_condition->type == PLPGPSM_CONDITION_SQLSTATE || handled_condition->type == PLPGPSM_CONDITION_NAMED)
			{
				/* leave when named condition hasnot attached SQLSTATE */
				if (handled_condition->type == PLPGPSM_CONDITION_NAMED && handled_condition->sqlstate == 0)
					return false;

				/* exact match */
				if (condition->sqlstate == handled_condition->sqlstate)
					return true;

				/* category match, when handled_condition is a category */
				if (ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(handled_condition->sqlstate) &&
				    ERRCODE_TO_CATEGORY(handled_condition->sqlstate) == handled_condition->sqlstate)
					actualise_possible_handler(handler, 1, possible_handler, distance);
			}
			else if (handled_condition->type == PLPGPSM_CONDITION_NOTFOUND)
			{
				/* exact match */
				if (condition->sqlstate == MAKE_SQLSTATE('0','2','0','0','0'))
					return true;

				/* category match when handled_condition is a category */
				if (ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(MAKE_SQLSTATE('0','2','0','0','0')))
					actualise_possible_handler(handler, 1, possible_handler, distance);
			}
			else if (handled_condition->type == PLPGPSM_CONDITION_SQLWARNING)
			{
				if (ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(MAKE_SQLSTATE('0','1','0','0','0')) ||
					ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(MAKE_SQLSTATE('0','2','0','0','0')))
					actualise_possible_handler(handler, 2, possible_handler, distance);
			}
		}
	}

	return false;
}

PLpgPSM_declare_handler *
plpgpsm_search_handler(PLpgPSM_astnode *scope, PLpgPSM_condition *condition)
{
	ListCell	*l;
	PLpgPSM_declare_handler *possible_handler = NULL;
	int			distance = -1;

	if (scope == NULL)
		return NULL;

	/*
	 * We should to protect handler against recursive handlers calls.
	 *
	 *   BEGIN
	 *      DECLARE UNDO HANDLER FOR SQLSTATE '01001'
	 *        BEGIN
	 *          PRINT 'handled '01001 and forwarded';
	 *          SIGNAL SQLSTATE '01001';
	 *        END;
	 *      SIGNAL SQLSTATE '01001';
	 *      ...
	 * Barier is implement as skipping compound statement when we
	 * going back from any handler.
	 */
	if (scope->type == PLPGPSM_DECLARE_HANDLER)
		return plpgpsm_search_handler(scope->parent->parent, condition);

	if (scope->type != PLPGPSM_STMT_COMPOUND)
		return plpgpsm_search_handler(scope->parent, condition);

	foreach(l, ((PLpgPSM_eb_stmt *) scope)->stmts)
	{
		PLpgPSM_astnode *iter = (PLpgPSM_astnode *) lfirst(l);

		if (iter->type == PLPGPSM_DECLARE_VARIABLE || iter->type == PLPGPSM_DECLARE_CURSOR ||
				iter->type == PLPGPSM_DECLARE_CONDITION)
			continue;
		else if (iter->type == PLPGPSM_DECLARE_HANDLER)
		{
			PLpgPSM_declare_handler *current_handler = (PLpgPSM_declare_handler *) iter;
			ListCell	*lc;

			foreach(lc, current_handler->condition_list)
			{
				PLpgPSM_condition *c2 = (PLpgPSM_condition *) lfirst(lc);

				Assert(c2->astnode.type == PLPGPSM_CONDITION);
				if (compare_condition(condition, c2, current_handler,
									&possible_handler,
									&distance))
					return current_handler;
			}
		}
		else
			break;
	}

	if (possible_handler != NULL)
		return possible_handler;

	return plpgpsm_search_handler(scope->parent, condition);
}

/*
 * AST node constructors
 *
 */
PLpgPSM_eb_stmt *
plpgpsm_eb_stmt(PLpgPSM_astnode_enum type, PLpgPSM_astnode *parent,
								    int location,
								    char *label,
								    PLpgPSM_ESQL *expr,
								    List	*stmts)
{
	ListCell *lc;

	PLpgPSM_eb_stmt *estmt = palloc(sizeof(PLpgPSM_eb_stmt));
	estmt->astnode.type = type;
	estmt->astnode.parent = parent;
	estmt->astnode.is_visible = true;
	estmt->location = location,
	estmt->lineno = plpgpsm_location_to_lineno(location);
	estmt->label = label;
	estmt->expr = expr;
	if (estmt->expr != NULL)
		estmt->expr->astnode.parent = (PLpgPSM_astnode *) estmt;

	estmt->stmts = stmts;
	estmt->option = false;
	estmt->handler = NULL;
	estmt->namespace = NULL;
	estmt->cursor = NULL;

	foreach (lc, stmts)
	{
		PLpgPSM_astnode *astnode = (PLpgPSM_astnode *) lfirst(lc);
		astnode->parent = (PLpgPSM_astnode *) estmt;
	}

	return estmt;
}

/*
 * parser setup - setup hooks for dynamic parameters
 *
 */
void
plpgpsm_parser_setup(struct ParseState *pstate, PLpgPSM_ESQL *esql)
{
	pstate->p_post_columnref_hook = post_column_ref;
	pstate->p_paramref_hook = param_ref;
	pstate->p_ref_hook_state = (void *) esql;
}

static PLpgPSM_variable_ref *
new_varref(PLpgPSM_astnode *var, int nattr)
{
	PLpgPSM_variable_ref *varref = (PLpgPSM_variable_ref *) palloc(sizeof(PLpgPSM_variable_ref));

	varref->ref = (PLpgPSM_astnode *) var;
	varref->nattr = nattr;

	return varref;
}

static int32
gettypmod(TupleDesc tupdesc, int fnumber)
{
	SPI_result = 0;

	if (fnumber > tupdesc->natts || fnumber == 0 ||
		fnumber <= FirstLowInvalidHeapAttributeNumber)
	{
		SPI_result = SPI_ERROR_NOATTRIBUTE;
		return InvalidOid;
	}

	if (fnumber > 0)
		return tupdesc->attrs[fnumber - 1]->atttypmod;
	else
		return (SystemAttributeDefinition(fnumber, true))->atttypmod;
}

/*
 * creates node with variable placeholder
 *
 */
static Node *
make_datum_param(PLpgPSM_ESQL *esql, PLpgPSM_astnode *node, int location, int fnumber)
{
	Param	   *param;

	param = makeNode(Param);
	param->paramkind = PARAM_EXTERN;
	param->paramid = list_length(esql->var_refs) + 1;
	param->paramtype = 0;
	param->paramtypmod = -1;
	param->location = location;

	/*
	 * When expression is checked in first stage, then we have to work with
	 * faked estate. This estate has not initialized datums area.
	 *
	 */
	if (esql->curr_estate != NULL && ((PLpgPSM_execstate *) esql->curr_estate)->datums != NULL)
	{
		/* List must be allocated in function's execution context */
		PLpgPSM_execstate *estate = (PLpgPSM_execstate *) esql->curr_estate;
		MemoryContext		oldcontext;

		oldcontext = MemoryContextSwitchTo(estate->exec_cxt);
		esql->var_refs = lappend(esql->var_refs, new_varref(node, fnumber));
		MemoryContextSwitchTo(oldcontext);

		if (node->type == PLPGPSM_VARIABLE)
		{
			PLpgPSM_datum	*datum = &estate->datums[((PLpgPSM_variable *) node)->offset];

			if (datum->typeinfo != NULL)
			{
				param->paramtype = datum->typeinfo->typeid;
				param->paramtypmod = datum->typeinfo->typmod;
			}
		}
		else if (node->type == PLPGPSM_CURSOR)
		{
			PLpgPSM_cursor *cursor = (PLpgPSM_cursor *) node;
			PLpgPSM_datum	*datum = &estate->datums[cursor->offset];
			PLpgPSM_fetch_extra *extra;

			if (datum->isnull)
				elog(ERROR, "cursor \"%s\" is closed", cursor->name);

			extra = (PLpgPSM_fetch_extra *) datum->extra;

			if (!extra->is_ready)
				elog(ERROR, "cursor \"%s\" is not prepared", cursor->name);

			param->paramtype = SPI_gettypeid(extra->cursor_tupdesc, fnumber);
			param->paramtypmod = gettypmod(extra->cursor_tupdesc, fnumber);
		}
	}

	return (Node *) param;
}

/*
 * Allow access to parameters via $n notation
 *
 */
static Node *
param_ref(ParseState *pstate, ParamRef *pref)
{
	PLpgPSM_ESQL	*esql = (PLpgPSM_ESQL *) pstate->p_ref_hook_state;
	int	paramno = pref->number - 1;
	Param	   *param;

	if (esql->curr_estate != NULL)
	{
		PLpgPSM_execstate *estate = (PLpgPSM_execstate *) esql->curr_estate;
		PLpgPSM_eb_stmt *top_stmt = (PLpgPSM_eb_stmt *) estate->func->astroot;

		if (top_stmt->astnode.type == PLPGPSM_STMT_COMPOUND)
		{
			ListCell	*lc;

			foreach (lc, top_stmt->stmts)
			{
				PLpgPSM_declare_param *declpar;

				if (((PLpgPSM_astnode *) lfirst(lc))->type != PLPGPSM_DECLARE_PARAM)
					break;

				declpar = (PLpgPSM_declare_param *) lfirst(lc);
				if (declpar->param_offset == paramno)
					return make_datum_param(esql, (PLpgPSM_astnode *) declpar->variable, pref->location, 0);
			}
		}

		ereport(ERROR,
				(errcode(ERRCODE_UNDEFINED_PARAMETER),
				 errmsg("there is no parameter $%d", paramno + 1),
				 errposition(pref->location + 1)));
	}

	/* fake param node */
	param = makeNode(Param);
	param->paramkind = PARAM_EXTERN;
	param->paramid = paramno;
	param->paramtype = 0;
	param->paramtypmod = -1;
	param->location = pref->location;

	return (Node *) param;
}

static Node *
resolve_column_ref(ParseState *pstate, PLpgPSM_ESQL *esql,
					    ColumnRef *cref)
{
	PLpgPSM_variable *var;
	PLpgPSM_ident ident;
	int		fnumber;

	switch (list_length(cref->fields))
	{
		case 1:
			{
				Node *field1 = (Node *) linitial(cref->fields);

				Assert(IsA(field1, String));
				ident.name = strVal(field1);
				ident.scope = NULL;
				break;
			}
		case 2:
			{
				Node	*field1 = (Node *) linitial(cref->fields);
				Node	*field2 = (Node *) lsecond(cref->fields);

				ident.scope = strVal(field1);
				ident.name = strVal(field2);
				break;
			}
	}

	var = (PLpgPSM_variable *) plpgpsm_search_variable((PLpgPSM_execstate *) esql->curr_estate, &ident, esql->astnode.parent, &fnumber);

	if (var != NULL)
	{
		if (var->astnode.type != PLPGPSM_VARIABLE && var->astnode.type != PLPGPSM_CURSOR)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("\"%s\" is not variable", ident.name),
				 parser_errposition1(cref->location)));

		return make_datum_param(esql, (PLpgPSM_astnode *) var, cref->location, fnumber);
	}

	return NULL;
}

/*
 * parser callback after parsing a ColumnRef
 *
 */
static Node *
post_column_ref(ParseState *pstate, ColumnRef *cref, Node *var)
{
	Node *myvar;
	PLpgPSM_ESQL	*esql = (PLpgPSM_ESQL *) pstate->p_ref_hook_state;

	myvar = resolve_column_ref(pstate, esql, cref);

	if (myvar != NULL && var != NULL)
	{
		/*
		 * We could leave it to the core parser to throw this error, but we
		 * can add a more useful detail message than the core could.
		 */
		ereport(ERROR,
				(errcode(ERRCODE_AMBIGUOUS_COLUMN),
				 errmsg("column reference \"%s\" is ambiguous",
						NameListToString(cref->fields)),
				 errdetail("It could refer to either a PL/pgPSM variable or a table column."),
				 parser_errposition(pstate, cref->location)));
	}

	return myvar;
}

/*
 * Returns condition for scope if exists
 *
 */
static PLpgPSM_declare_condition *
search_condition(PLpgPSM_parser *pstate, PLpgPSM_astnode *scope, char *name, int *sqlstate,
							PLpgPSM_declare_condition *exclude_condition)
{
	ListCell	*l;

	if (scope == NULL)
	{
		int i;

		if (name == NULL || exclude_condition != NULL)
			return NULL;

		/* try search predefined conditions */
		for (i = 0; condition_label_map[i].label != NULL; i++)
		{
			if (strcmp(name, condition_label_map[i].label) == 0)
			{
				PLpgPSM_condition *condition;
				PLpgPSM_declare_condition *declcon;

				condition = (PLpgPSM_condition *) palloc(sizeof(PLpgPSM_condition));
				INIT_NODE(condition, PLPGPSM_CONDITION, false, -1);
				condition->type = PLPGPSM_CONDITION_NAMED;
				condition->name = (char *) condition_label_map[i].label;
				condition->sqlstate = condition_label_map[i].sqlerrstate;

				declcon = (PLpgPSM_declare_condition *) palloc(sizeof(PLpgPSM_declare_condition));
				INIT_NODE(declcon, PLPGPSM_DECLARE_CONDITION, false, -1);
				declcon->condition = condition;
				SET_PARENT(condition, declcon);

				/* creating new condition and appending to parse tree */
				if (pstate->asttree->type != PLPGPSM_STMT_COMPOUND)
				{
					pstate->asttree = (PLpgPSM_astnode *) plpgpsm_eb_stmt(PLPGPSM_STMT_COMPOUND,
												NULL, -1,
												NULL, NULL,
												list_make2(declcon, pstate->asttree));
				}
				else
				{
					PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) pstate->asttree;

					declcon->astnode.parent = pstate->asttree;
					stmt->stmts = lcons(declcon, stmt->stmts);
				}

				return declcon;
			}
		}

		return NULL;
	}

	if (scope->type != PLPGPSM_STMT_COMPOUND)
		return search_condition(pstate, scope->parent, name, sqlstate, exclude_condition);

	foreach(l, ((PLpgPSM_eb_stmt *) scope)->stmts)
	{
		PLpgPSM_astnode *iter = (PLpgPSM_astnode *) lfirst(l);

		if (iter->type == PLPGPSM_DECLARE_VARIABLE || iter->type == PLPGPSM_DECLARE_CURSOR)
			continue;
		else if (iter->type == PLPGPSM_DECLARE_CONDITION)
		{
			PLpgPSM_declare_condition *declcon = (PLpgPSM_declare_condition *) iter;
			PLpgPSM_condition *condition = declcon->condition;

			if (declcon == exclude_condition)
				continue;

			Assert(condition->name != NULL);
			if (name != NULL && strcmp(name, condition->name) == 0)
				return declcon;
			if (sqlstate != NULL && *sqlstate == condition->sqlstate)
				return declcon;
		}
		else
			break;
	}

	return search_condition(pstate, scope->parent, name, sqlstate, exclude_condition);
}

/*
 * Does:
 *    1. namespace building and label checking
 *    2. searching and joining static handlers
 *    3. searching and filling referers
 *
 */
static void
second_stage(PLpgPSM_parser *pstate, PLpgPSM_astnode *astnode)
{
	if (astnode->type == PLPGPSM_STMT_LEAVE || astnode->type == PLPGPSM_STMT_ITERATE)
	{
		PLpgPSM_stmt_leave_iterate *stmt = (PLpgPSM_stmt_leave_iterate *) astnode;

		stmt->stop_node = get_labeled_statement(astnode, stmt->label);
		if (stmt->stop_node == NULL)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("label \"%s\" is not accessible in current scope (or maybe not exists)", stmt->label),
								 parser_errposition1(stmt->location)));

		/* ITERATE cannot be used to loop over compound statement */
		if (astnode->type == PLPGPSM_STMT_ITERATE && stmt->stop_node->type == PLPGPSM_STMT_COMPOUND)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("cannot iterate over compound statement"),
					 errhint("Use LOOP statement."),
								 parser_errposition1(stmt->location)));
	}
	else if (plpgpsm_allow_cast_to_eb(astnode))
	{
		PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode;
		ListCell *lc;

		if (stmt->label != NULL)
		{
			/* label must be unique */
			if (get_labeled_statement(astnode, stmt->label) != NULL)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("label \"%s\" is not unique in hiearchy", stmt->label),
									 parser_errposition1(stmt->location)));
		}

		foreach  (lc, stmt->stmts)
		{
			second_stage(pstate, (PLpgPSM_astnode *) lfirst(lc));
		}

		if (astnode->type == PLPGPSM_STMT_COMPOUND)
		{
			check_condition_overlap(stmt);
		}
		if (astnode->type == PLPGPSM_STMT_CASE)
		{
			PLpgPSM_condition notfound;

			/* set not found handler */
			notfound.astnode.type = PLPGPSM_CONDITION;
			notfound.type = PLPGPSM_CONDITION_SQLSTATE;
			notfound.sqlstate = MAKE_SQLSTATE('0','2','0','0','0');
			notfound.name = NULL;

			stmt->handler = plpgpsm_search_handler(astnode->parent, &notfound);
		}
	}
	else if (astnode->type == PLPGPSM_REFERER)
	{
		PLpgPSM_referer *ref = (PLpgPSM_referer *) astnode;

		if (ref->ident->astnode.type == PLPGPSM_IDENT_VARIABLE)
		{
			PLpgPSM_variable *var;

			var = (PLpgPSM_variable *) plpgpsm_search_variable(NULL, ref->ident, ref->astnode.parent, NULL);
			if (var == NULL)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("not found variable \"%s\" in current scope",
								ref->ident->name),
							parser_errposition1(ref->ident->location)));

			if (var->astnode.type != PLPGPSM_VARIABLE)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("\"%s\" is not variable in current scope",
								ref->ident->name),
							parser_errposition1(ref->ident->location)));

			if (var->special == PLPGPSM_SPECIAL_SQLCODE)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("SQLCODE is read only"),
							parser_errposition1(ref->ident->location)));

			if (var->special == PLPGPSM_SPECIAL_SQLSTATE)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("SQLSTATE is read only"),
							parser_errposition1(ref->ident->location)));

			ref->ref = (PLpgPSM_astnode *) var;
		}
		else if (ref->ident->astnode.type == PLPGPSM_IDENT_CURSOR)
		{
			PLpgPSM_cursor *cur;

			cur = (PLpgPSM_cursor *) plpgpsm_search_variable(NULL, ref->ident, ref->astnode.parent, NULL);
			if (cur == NULL)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("not found cursor \"%s\" in current scope",
								ref->ident->name),
							parser_errposition1(ref->ident->location)));

			if (cur->astnode.type != PLPGPSM_CURSOR)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("\"%s\" is not cursor in current scope",
								ref->ident->name),
							parser_errposition1(ref->ident->location)));

			ref->ref = (PLpgPSM_astnode *) cur;
		}
	}
	else if (astnode->type == PLPGPSM_STMT_ASSIGN)
	{
		PLpgPSM_stmt_assign *stmt = (PLpgPSM_stmt_assign *) astnode;

		/* evaluate referers */
		if (stmt->target != NULL)
			second_stage(pstate, (PLpgPSM_astnode *) stmt->target);
		else
		{
			ListCell	*lc;

			foreach (lc, stmt->target_list)
			{
				second_stage(pstate, (PLpgPSM_astnode *) lfirst(lc));
			}
		}
	}
	else if (astnode->type == PLPGPSM_DECLARE_VARIABLE)
	{
		PLpgPSM_declare_variable *declvar = (PLpgPSM_declare_variable *) astnode;
		ListCell	*lc;

		foreach (lc, declvar->variables)
		{
			second_stage(pstate, (PLpgPSM_astnode *) lfirst(lc));
		}
	}
	else if (astnode->type == PLPGPSM_VARIABLE)
	{
		PLpgPSM_variable *var = (PLpgPSM_variable *) astnode;

		/*
		 * Identification of special variables like SQLCODE and SQLSTATE.
		 * These variables can be declared only once in function. Usual rules
		 * ensures uniquity in current scope, it should be unique in function.
		 */
		if (strcmp(var->name, "sqlcode") == 0)
		{
			PLpgPSM_astnode			*outer_compound;

			outer_compound = plpgpsm_get_near_compound_stmt(astnode->parent);
			if (outer_compound != NULL)
			{
				PLpgPSM_ident		ident;

				ident.scope = NULL;
				ident.name = "sqlcode";

				if (plpgpsm_search_variable(NULL, &ident, outer_compound->parent, NULL) != NULL)
					ereport(ERROR,
							(errcode(ERRCODE_SYNTAX_ERROR),
							 errmsg("SQLCODE can be declare only once in function"),
								parser_errposition1(var->location)));
			}

			var->special = PLPGPSM_SPECIAL_SQLCODE;
		}
		else if (strcmp(var->name, "sqlstate") == 0)
		{
			PLpgPSM_astnode			*outer_compound;

			outer_compound = plpgpsm_get_near_compound_stmt(astnode->parent);
			if (outer_compound != NULL)
			{
				PLpgPSM_ident		ident;

				ident.scope = NULL;
				ident.name = "sqlstate";

				if (plpgpsm_search_variable(NULL, &ident, outer_compound->parent, NULL) != NULL)
					ereport(ERROR,
							(errcode(ERRCODE_SYNTAX_ERROR),
							 errmsg("SQLSTATE can be declare only once in function"),
								parser_errposition1(var->location)));
			}

			var->special = PLPGPSM_SPECIAL_SQLSTATE;
		}
		else
			var->special = PLPGPSM_SPECIAL_NONE;
	}
	else if (astnode->type == PLPGPSM_STMT_OPEN)
	{
		second_stage(pstate, (PLpgPSM_astnode *) ((PLpgPSM_stmt_open *) astnode)->cursor);
	}
	else if (astnode->type == PLPGPSM_STMT_FETCH)
	{
		PLpgPSM_stmt_fetch *stmt = (PLpgPSM_stmt_fetch *) astnode;
		PLpgPSM_condition notfound;
		ListCell	*lc;

		second_stage(pstate, (PLpgPSM_astnode *) stmt->cursor);
		foreach(lc, stmt->target_list)
		{
			second_stage(pstate, (PLpgPSM_astnode *) lfirst(lc));
		}

		notfound.astnode.type = PLPGPSM_CONDITION;
		notfound.type = PLPGPSM_CONDITION_SQLSTATE;
		notfound.sqlstate = MAKE_SQLSTATE('0','2','0','0','0');
		notfound.name = NULL;

		stmt->handler = plpgpsm_search_handler(astnode->parent, &notfound);
	}
	else if (astnode->type == PLPGPSM_STMT_CLOSE)
	{
		second_stage(pstate, (PLpgPSM_astnode *) ((PLpgPSM_stmt_close *) astnode)->cursor);
	}
	else if (astnode->type == PLPGPSM_DECLARE_HANDLER)
	{
		PLpgPSM_declare_handler *handler = (PLpgPSM_declare_handler *) astnode;
		PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode->parent;
		ListCell	*l;

		/* verify named condition */
		foreach(l, handler->condition_list)
		{
			PLpgPSM_condition *condition = (PLpgPSM_condition *) lfirst(l);

			Assert(condition->astnode.type == PLPGPSM_CONDITION);
			if (condition->type == PLPGPSM_CONDITION_NAMED)
			{
				PLpgPSM_declare_condition *declcond = search_condition(pstate, astnode->parent, condition->name, NULL, NULL);
				if (declcond == NULL)
					ereport(ERROR,
							(errcode(ERRCODE_SYNTAX_ERROR),
							 errmsg("condition not found"),
									parser_errposition1(condition->location)));
				condition->sqlstate = declcond->condition->sqlstate;
			}
		}

		/* parent node - compound statement should be ATOMIC */
		Assert(stmt->astnode.type == PLPGPSM_STMT_COMPOUND);
		if (handler->handler_type == PLPGPSM_HANDLER_UNDO && !stmt->option)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("current compound statement is not ATOMIC"),
					 errhint("UNDO handler can be declared only in ATOMIC compound statement"),
								parser_errposition1(handler->location)));

		second_stage(pstate, (PLpgPSM_astnode *) handler->stmt);
	}
	else if (astnode->type == PLPGPSM_STMT_SIGNAL || astnode->type == PLPGPSM_STMT_RESIGNAL)
	{
		PLpgPSM_stmt_signal *stmt = (PLpgPSM_stmt_signal *) astnode;

		if (stmt->condition != NULL)
		{
			PLpgPSM_condition *condition = stmt->condition;

			if (condition->type == PLPGPSM_CONDITION_NAMED)
			{
				PLpgPSM_declare_condition *declcond;

				declcond = search_condition(pstate, astnode->parent, condition->name, NULL, NULL);
				if (declcond == NULL)
					ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("condition not found"),
									parser_errposition1(stmt->location)));

				condition->sqlstate = declcond->condition->sqlstate;

				/* when SQLSTATE is not defined use default '45000' */
				if (condition->sqlstate == 0)
					condition->sqlstate = MAKE_SQLSTATE('4','5','0','0','0');
			}

			if (ERRCODE_TO_CATEGORY(condition->sqlstate) == MAKE_SQLSTATE('0','0','0','0','0'))
			{
				stmt->elog_level = NOTICE;
			}
			else if (ERRCODE_TO_CATEGORY(condition->sqlstate) == MAKE_SQLSTATE('0','1','0','0','0') ||
				 ERRCODE_TO_CATEGORY(condition->sqlstate) == MAKE_SQLSTATE('0','2','0','0','0'))
			{
				stmt->elog_level = WARNING;
			}
			else
				stmt->elog_level = ERROR;

			/*
			 * handler is used only for NOTICE and WARNING levels, but we can use a routine
			 * for searching ambigonuous handlers.
			 */
			stmt->handler = plpgpsm_search_handler(astnode->parent, stmt->condition);
			if (stmt->handler != NULL && stmt->elog_level >= ERROR )
			{
				/* handler must be UNDO */
				if (stmt->handler->handler_type != PLPGPSM_HANDLER_UNDO)
					ereport(ERROR,
							(errcode(ERRCODE_SYNTAX_ERROR),
							 errmsg("condition is not handled by UNDO handler"),
							 errhint("Signals with ERROR level should be handled by UNDO handlers only"),
								parser_errposition1(stmt->location)));
			}
		}
	}
	else if (astnode->type == PLPGPSM_DECLARE_CONDITION)
	{
		PLpgPSM_declare_condition *declcond = (PLpgPSM_declare_condition *) astnode;

		/* condition must be unique in scope and all nested scopes */
		if (search_condition(pstate, astnode->parent, declcond->condition->name, NULL, declcond) != NULL)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg("condition is not uniqie in scope"),
					 errdetail("condition \"%s\" is already defined",
								    declcond->condition->name),
						parser_errposition1(declcond->condition->location)));

		if (declcond->condition->sqlstate != 0)
		{
			int	sqlstate = declcond->condition->sqlstate;

			if (search_condition(pstate, astnode->parent, NULL, &sqlstate, declcond) != NULL)
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("condition is not uniqie in scope"),
						 errdetail("condition with SQLSTATE \"%s\" is already defined",
									    unpack_sql_state(sqlstate)),
							parser_errposition1(declcond->location)));
		}
	}
	else if (astnode->type == PLPGPSM_STMT_DIAGNOSTICS)
	{
		PLpgPSM_stmt_diagnostics *stmt = (PLpgPSM_stmt_diagnostics *) astnode;
		ListCell	*l;

		foreach(l, stmt->diag_info_list)
		{
			PLpgPSM_diag_info *dinfo = (PLpgPSM_diag_info *) lfirst(l);

			Assert(dinfo->astnode.type == PLPGPSM_DIAG_INFO);
			second_stage(pstate, (PLpgPSM_astnode *) dinfo->target);
		}
	}
}

/*
 * create necessary declarations for function's parameters
 *
 */
static List *
make_params(HeapTuple procTup)
{
	char		**argnames;
	Oid		*argtypes;
	char		*argmodes;
	int	nargs;
	List		*params = NIL;

	nargs = get_func_arg_info(procTup,
							&argtypes, &argnames, &argmodes);

	if (nargs > 0)
	{
		int	i;

		for (i = 0; i < nargs; i++)
		{
			PLpgPSM_variable *var = NULL;
			PLpgPSM_declare_param *dpar;
			char		buf[32];

			if (argnames[i] != NULL)
				var = plpgpsm_new_variable(argnames[i], -1);
			else
			{
				snprintf(buf, sizeof(buf), "$%d", i + 1);
				var = plpgpsm_new_variable(pstrdup(buf), -1);
			}

			var->astnode.is_visible = false;

			dpar = new_declparam(var, argtypes[i], argmodes ? argmodes[i] : PROARGMODE_IN, i);
			if (dpar != NULL)
				params = lappend(params, dpar);
		}
	}

	return params;
}

/*
 * returns a block statement with label, returns NULL when
 * label doesn't exists.
 *
 */
static PLpgPSM_astnode *
_search_statement(PLpgPSM_astnode *astnode, char *label)
{
	Assert(astnode != NULL && label != NULL);

	/*
	 * labels scope is limited - inside handlers outer labels
	 * are invisible
	 */
	if (astnode->type == PLPGPSM_DECLARE_HANDLER)
		return NULL;

	if (plpgpsm_allow_cast_to_eb(astnode))
	{
		PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) astnode;

		if (stmt->label != NULL && strcmp(stmt->label, label) == 0)
			return astnode;
	}

	/* recursive search to outer statement */
	if (astnode->parent != NULL)
		return _search_statement(astnode->parent, label);

	return NULL;
}

static PLpgPSM_astnode *
get_labeled_statement(PLpgPSM_astnode *astnode, char *label)
{
	PLpgPSM_astnode *lnode;

	/* look to parent */
	if (astnode->parent != NULL)
	{
		lnode = _search_statement(astnode->parent, label);
		if (lnode != NULL)
			return lnode;
	}

	return NULL;
}

PLpgPSM_variable *
plpgpsm_new_variable(char *name, int location)
{
	PLpgPSM_variable *var = (PLpgPSM_variable *) palloc(sizeof(PLpgPSM_variable));

	var->astnode.type = PLPGPSM_VARIABLE;
	var->astnode.is_visible = true;
	var->name = name;
	var->location = location;
	var->offset = parser.nvars++;

	return var;
}

static PLpgPSM_declare_param *
new_declparam(PLpgPSM_variable *variable, Oid typeid, char param_mode, int param_offset)
{
	PLpgPSM_declare_param   *dpar = (PLpgPSM_declare_param *) palloc(sizeof(PLpgPSM_declare_param));

	if (param_mode == PROARGMODE_IN || param_mode == PROARGMODE_OUT || param_mode == PROARGMODE_INOUT)
	{
		dpar->astnode.type = PLPGPSM_DECLARE_PARAM;
		dpar->astnode.is_visible = false;
		dpar->variable = variable;
		dpar->typeid = typeid;
		dpar->param_offset = param_offset;
		dpar->variable->astnode.parent = (PLpgPSM_astnode *) dpar;
		dpar->param_mode = param_mode;

		parser.nargs++;

		if (param_mode == PROARGMODE_OUT || param_mode == PROARGMODE_INOUT)
			parser.noutargs++;

		return dpar;
	}
	else
		return NULL;
}

/*
 * returns true when two conditions are distinct
 *
 */
static bool
is_distinct_conditions(PLpgPSM_condition *c1, PLpgPSM_condition *c2)
{
	if (c1->type == PLPGPSM_CONDITION_SQLSTATE)
	{
		if (c2->type == PLPGPSM_CONDITION_SQLSTATE && c1->sqlstate == c2->sqlstate)
			return false;
		if (c1->sqlstate == MAKE_SQLSTATE('0','2','0','0','0') && 
				c2->type == PLPGPSM_CONDITION_NOTFOUND)
			return false;
		if (c2->type == PLPGPSM_CONDITION_NAMED && c2->sqlstate != 0 && c1->sqlstate == c2->sqlstate)
			return false;
	}
	else if (c1->type == PLPGPSM_CONDITION_NAMED)
	{
		if (c2->type == PLPGPSM_CONDITION_NAMED)
		{
			if (strcmp(c1->name, c2->name) == 0)
				return false;
			if (c1->sqlstate != 0 && c1->sqlstate == c2->sqlstate)
				return false;
		}
		if (c1->sqlstate == MAKE_SQLSTATE('0','2','0','0','0') &&
				c2->type == PLPGPSM_CONDITION_NOTFOUND)
			return false;
	}
	else if (c1->type == PLPGPSM_CONDITION_NOTFOUND)
	{
		if (c2->type == PLPGPSM_CONDITION_SQLSTATE && c2->sqlstate == MAKE_SQLSTATE('0','2','0','0','0'))
			return false;
		if (c2->type == PLPGPSM_CONDITION_NAMED && c2->sqlstate == MAKE_SQLSTATE('0','2','0','0','0'))
			return false;
		if (c2->type == PLPGPSM_CONDITION_NOTFOUND)
			return false;
	}
	else if (c1->type == PLPGPSM_CONDITION_SQLWARNING || c1->type == PLPGPSM_CONDITION_SQLEXCEPTION)
		return c1->type != c2->type;

	return true;

}

/*
 * Ensure so handled conditions are not ambigonuous
 *
 */
static void
check_condition_overlap(PLpgPSM_eb_stmt *stmt)
{
	ListCell	*l1, *l2;
	ListCell	*lc1, *lc2;

	foreach(l1, stmt->stmts)
	{
		PLpgPSM_declare_handler *dh1 = (PLpgPSM_declare_handler *) lfirst(l1);

		if (dh1->astnode.type == PLPGPSM_DECLARE_VARIABLE || dh1->astnode.type == PLPGPSM_DECLARE_CONDITION ||
				dh1->astnode.type == PLPGPSM_DECLARE_CURSOR)
			continue;
		else if (dh1->astnode.type == PLPGPSM_DECLARE_HANDLER)
		{
			foreach(lc1, dh1->condition_list)
			{
				PLpgPSM_condition *c1 = lfirst(lc1);

				foreach(l2, stmt->stmts)
				{
					PLpgPSM_declare_handler *dh2 = (PLpgPSM_declare_handler *) lfirst(l2);

					if (dh2->astnode.type == PLPGPSM_DECLARE_VARIABLE || dh2->astnode.type == PLPGPSM_DECLARE_CONDITION ||
							dh2->astnode.type == PLPGPSM_DECLARE_CURSOR)
						continue;
					else if (dh2->astnode.type == PLPGPSM_DECLARE_HANDLER)
					{
						foreach(lc2, dh2->condition_list)
						{
							PLpgPSM_condition *c2 = lfirst(lc2);

							if (c1 != c2)
								if (!is_distinct_conditions(c1, c2))
									ereport(ERROR,
											(errcode(ERRCODE_SYNTAX_ERROR),
											 errmsg("handled condition is not uniqie in current compound statement"),
									parser_errposition1(c1->location > c2->location ? c1->location : c2->location)));

						}
					}
					else
						break;
				}
			}
		}
		else
			break;
	}
}
