%{
/*-------------------------------------------------------------------------
 *
 * gram.y				- Parser for the PL/pgPSM procedural language
 *
 * Portions Copyright (c) 1996-2012, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  src/pl/plpgpsm/src/gram.y
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"

#include "catalog/pg_type.h"
#include "parser/parse_type.h"
#include "parser/scanner.h"
#include "parser/scansup.h"

/* Location tracking support --- simpler than bison's default */
#define YYLLOC_DEFAULT(Current, Rhs, N) \
	do { \
		if (N) \
			(Current) = (Rhs)[1]; \
		else \
			(Current) = (Rhs)[0]; \
	} while (0)

/*
 * Bison doesn't allocate anything that needs to live across parser calls,
 * so we can easily have it use palloc instead of malloc.  This prevents
 * memory leaks if we error out during parsing.  Note this only works with
 * bison >= 2.0.  However, in bison 1.875 the default is to use alloca()
 * if possible, so there's not really much problem anyhow, at least if
 * you're building with gcc.
 */
#define YYMALLOC palloc
#define YYFREE   pfree


typedef struct
{
	int			location;
	int			leaderlen;
} sql_error_callback_arg;

#define parser_errposition(pos)  plpgpsm_scanner_errposition(pos)

union YYSTYPE;					/* need forward reference for tok_is_keyword */

static PLpgPSM_ESQL *read_esql(int until1, int until2, int until3, const char *expected,
						 bool is_expression, int *endtoken, int startlocation,
						 bool is_optional);

static void check_labels(const char *label1, const char *label2);
static void check_labels_in_current_scope(List *stmts, PLpgPSM_astnode *stmt);
static PLpgPSM_ident *read_typename(void);
static void set_parent(List *nodes, PLpgPSM_astnode *parent);
static PLpgPSM_declare_variable *new_declvar(List *variable_list, PLpgPSM_ident *typename,
										PLpgPSM_ESQL *expr);
static PLpgPSM_referer *new_referer(PLpgPSM_ident *ident, PLpgPSM_astnode *ref);
static PLpgPSM_ident *new_ident_var(char *scope, char *name, int location);
static PLpgPSM_stmt_assign *new_stmt_assign(PLpgPSM_referer *referer,
					    List *idents,
							    PLpgPSM_ESQL *expr, int location);
static PLpgPSM_eb_stmt *new_search_case(PLpgPSM_ESQL *expr,
							List *case_when_list,
							PLpgPSM_eb_stmt *else_stmts,
										    int location);
static void cast_to_boolean(PLpgPSM_ESQL *expr);
static PLpgPSM_astnode *declare_prefetch(void);
static bool is_unreserved_keyword(int tok);
static PLpgPSM_astnode *for_prefetch(void);
static PLpgPSM_astnode *newsignal(PLpgPSM_astnode_enum type, PLpgPSM_condition *condition,
								    List *signal_info_list,
								    int location);
static PLpgPSM_condition *condition(PLpgPSM_condition_enum type, int sqlstate, char *name,
									    int location);


%}

%expect 0
%name-prefix="plpgpsm_yy"
%locations

%union {
		core_YYSTYPE			core_yystype;
		/* these fields must match core_YYSTYPE: */
		int						ival;
		char					*str;
		const char				*keyword;

		PLword					word;
		PLcword					cword;
		bool					boolean;
		Oid						oid;
		PLpgPSM_astnode			*astnode;
		PLpgPSM_ESQL			*esql;
		List				*list;
		PLpgPSM_eb_stmt			*eb;
		PLpgPSM_condition		*condition;
		PLpgPSM_ident			*ident;
		PLpgPSM_referer		*referer;
		PLpgPSM_variable	*variable;
		PLpgPSM_signal_info_enum		signal_info_enum;
		PLpgPSM_diagarea_type		diagarea;
		PLpgPSM_diag_info_type		diag_info_type;
}


/*
 * Basic non-keyword token types.  These are hard-wired into the core lexer.
 * They must be listed first so that their numeric codes do not depend on
 * the set of keywords.  Keep this list in sync with backend/parser/gram.y!
 *
 * Some of these are not directly referenced in this file, but they must be
 * here anyway.
 */
%token <str>	IDENT FCONST SCONST BCONST XCONST Op
%token <ival>	ICONST PARAM
%token			TYPECAST DOT_DOT COLON_EQUALS

/*
 * Other tokens recognized by plpgpsm's lexer interface layer (scanner.c).
 */
%token <word>		T_WORD		/* unrecognized simple identifier */
%token <cword>		T_CWORD		/* unrecognized composite identifier */

/*
 * Keyword tokens.  Some of these are reserved and some are not;
 * see scanner.c for info.  Be sure unreserved keywords are listed
 * in the "unreserved_keyword" production below.
 */
%token <keyword>	AS
%token <keyword>	ATOMIC
%token <keyword>	BEGIN
%token <keyword>	CASE
%token <keyword>	CLOSE
%token <keyword>	CONDITION
%token <keyword>	CONDITION_IDENTIFIER
%token <keyword>	CONTINUE
%token <keyword>	CURRENT
%token <keyword>	CURSOR
%token <keyword>	DECLARE
%token <keyword>	DEFAULT
%token <keyword>	DETAIL_TEXT
%token <keyword>	DIAGNOSTICS
%token <keyword>	DO
%token <keyword>	ELSE
%token <keyword>	ELSEIF
%token <keyword>	END
%token <keyword>	EXIT
%token <keyword>	FETCH
%token <keyword>	FOR
%token <keyword>	FROM
%token <keyword>	FOUND
%token <keyword>	GET
%token <keyword>	HANDLER
%token <keyword>	HINT_TEXT
%token <keyword>	IF
%token <keyword>	INTO
%token <keyword>	ITERATE
%token <keyword>	LEAVE
%token <keyword>	LOOP
%token <keyword>	MESSAGE_TEXT
%token <keyword>	NO
%token <keyword>	NOT
%token <keyword>	OPEN
%token <keyword>	PRINT
%token <keyword>	REPEAT
%token <keyword>	RESIGNAL
%token <keyword>	RETURN
%token <keyword>	RETURNED_SQLCODE
%token <keyword>	RETURNED_SQLSTATE
%token <keyword>	ROW_COUNT
%token <keyword>	SCROLL
%token <keyword>	SELECT
%token <keyword>	SET
%token <keyword>	SIGNAL
%token <keyword>	SQLCODE
%token <keyword>	SQLEXCEPTION
%token <keyword>	SQLSTATE
%token <keyword>	SQLWARNING
%token <keyword>	STACKED
%token <keyword>	THEN
%token <keyword>	UNTIL
%token <keyword>	VALUE
%token <keyword>	WHEN
%token <keyword>	WHILE
%token <keyword>	UNDO

%type <list>     stmts stmts_opt
%type <astnode>  stmt
%type <astnode>  stmt_return stmt_print stmt_assign simple_assign stmt_leave stmt_iterate
%type <astnode>  stmt_if stmt_case
%type <astnode>  stmt_loop stmt_while stmt_repeat_until stmt_compound
%type <astnode>  declaration declare_prefetch
%type <astnode>  stmt_open stmt_close stmt_fetch stmt_for for_prefetch
%type <astnode>  stmt_signal stmt_resignal signal_info stmt_diagnostics diag_info
%type <esql>     opt_expr expr_until_do expr_until_end opt_expr_until_when expr_until_then
                    expr_until_comma_semi
%type <str>      opt_label opt_end_label label
%type <eb>       case_when opt_else_stmts elseif_then
%type <list>     case_when_list elseif_then_list simple_assign_list
%type <list>     opt_elseif_then_list
%type <list>     expr_list target_list condition_list declaration_list
%type <list>     signal_info_list diag_info_list
%type <ival>        sqlstate
%type <condition>   condition opt_sqlstate signaled_condition
%type <boolean>     opt_atomic
%type <diagarea>  diag_area_opt
%type <keyword>   unreserved_keyword
%type <referer>   target
%type <signal_info_enum>  signal_info_type
%type <diag_info_type>    diag_info_type

%%

sqlpsm_statement:
			stmt			{ parser.asttree = $1; }
			| stmt ';'		{ parser.asttree = $1; }
		;

stmt:
			stmt_assign		{ $$ = $1; }
			| stmt_if		{ $$ = $1; }
			| stmt_case		{ $$ = $1; }
			| stmt_leave		{ $$ = $1; }
			| stmt_iterate		{ $$ = $1; }
			| stmt_loop		{ $$ = $1; }
			| stmt_while		{ $$ = $1; }
			| stmt_repeat_until	{ $$ = $1; }
			| stmt_return		{ $$ = $1; }
			| stmt_print		{ $$ = $1; }
			| stmt_compound		{ $$ = $1; }
			| stmt_open		{ $$ = $1; }
			| stmt_close		{ $$ = $1; }
			| stmt_fetch		{ $$ = $1; }
			| stmt_for		{ $$ = $1; }
			| stmt_signal		{ $$ = $1; }
			| stmt_resignal		{ $$ = $1; }
			| stmt_diagnostics	{ $$ = $1; }
		;

stmts_opt:
			stmts ';'		{ $$ = $1; }
			|			{ $$ = NIL; }
		;

stmts:
			stmts ';' stmt
				{
					check_labels_in_current_scope($1, $3);
					$$ = lappend($1, $3);
				}
			| stmt
				{
					$$ = list_make1($1);
				}
		;

/*
 * compound statement
 *
 * [ label: ]
 * BEGIN
 * [ variable's or condition's declarations; ]
 * [ cursor's declarations; ]
 * [ handler's declarations; ]
 * [ statements; ]
 * END [ label ]
 *
 */
stmt_compound:
			opt_label BEGIN opt_atomic stmts_opt END opt_end_label
				{
					PLpgPSM_eb_stmt *stmt;

					check_labels($1, $6);
					if ($1 != NULL || $3)
					{
						stmt = plpgpsm_eb_stmt(PLPGPSM_STMT_COMPOUND, NULL, @2,
								    $1, NULL, $4);
						stmt->option = $3;
					}
					else
						stmt = plpgpsm_eb_stmt(PLPGPSM_SIMPLE_BLOCK, NULL, @2,
								    $1, NULL, $4);

					$$ = (PLpgPSM_astnode *) stmt;
				}
			| opt_label BEGIN opt_atomic declaration_list END opt_end_label
				{
					PLpgPSM_eb_stmt *stmt;

					check_labels($1, $6);
					stmt = plpgpsm_eb_stmt(PLPGPSM_STMT_COMPOUND, NULL, @2,
								    $1, NULL, $4);
					stmt->option = $3;

					$$ = (PLpgPSM_astnode *) stmt;
				}
			| opt_label BEGIN opt_atomic declaration_list ';' stmts_opt END opt_end_label
				{
					PLpgPSM_eb_stmt *stmt;

					check_labels($1, $8);
					stmt = plpgpsm_eb_stmt(PLPGPSM_STMT_COMPOUND, NULL, @2,
								    $1, NULL, list_concat($4, $6));
					stmt->option = $3;

					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

opt_atomic:
			ATOMIC			{ $$ = true; }
			| 			{ $$ = false; }
		;

declaration_list:
			declaration_list ';' declaration
				{
					PLpgPSM_astnode *last_decl = (PLpgPSM_astnode *) llast($1);
					PLpgPSM_astnode_enum last = last_decl->type;

					/* ensure enabled order of declarations */
					if ($3->type == PLPGPSM_DECLARE_VARIABLE)
					{
						if (last == PLPGPSM_DECLARE_CURSOR || last == PLPGPSM_DECLARE_HANDLER)
							yyerror("syntax error, variable declaration after cursor or handler");
					}
					else if ($3->type == PLPGPSM_DECLARE_CONDITION)
					{
						if (last == PLPGPSM_DECLARE_CURSOR || last == PLPGPSM_DECLARE_HANDLER)
							yyerror("syntax error, condition declaration after cursor or handler");
					}
					else if ($3->type == PLPGPSM_DECLARE_CURSOR)
					{
						if (last == PLPGPSM_DECLARE_HANDLER)
							yyerror("syntax error, cursor declaration after handler");
					}
					else if ($3->type != PLPGPSM_DECLARE_HANDLER)
					{
						elog(ERROR, "internal error, unexpected value");
					}

					/* declared variables must be unique in DECLARE part */
					if ($3->type == PLPGPSM_DECLARE_VARIABLE)
					{
						ListCell	*lc;
						PLpgPSM_declare_variable	*dvar;

						dvar = (PLpgPSM_declare_variable *) $3;
						foreach (lc, dvar->variables)
						{
							ListCell *dlc;
							PLpgPSM_variable   *var = (PLpgPSM_variable *) lfirst(lc);

							foreach (dlc, $1)
							{
								PLpgPSM_astnode *iter = (PLpgPSM_astnode *) lfirst(dlc);

								if (iter->type == PLPGPSM_DECLARE_VARIABLE)
								{
									PLpgPSM_declare_variable *dv = (PLpgPSM_declare_variable *) iter;
									ListCell	*vlc;

									foreach (vlc, dv->variables)
									{
										PLpgPSM_variable *var2 = (PLpgPSM_variable *) lfirst(vlc);

										Assert (((PLpgPSM_astnode *) lfirst(vlc))->type == PLPGPSM_VARIABLE);

										if (strcmp(var2->name, var->name) == 0)
											ereport(ERROR,
													(errcode(ERRCODE_SYNTAX_ERROR),
										 errmsg("identifier \"%s\" is not unique in current scope",
														var->name),
											    parser_errposition(var->location)));
									}
								}
							}
						}
					}

					$$ = lappend($1, $3);
				}
			| declaration
				{
					$$ = list_make1($1);
				}
		;

declaration:
			DECLARE declare_prefetch
				{
					PLpgPSM_declare_variable *declvar = (PLpgPSM_declare_variable *) $2;

					Assert(declvar->astnode.type == PLPGPSM_DECLARE_VARIABLE);
					declvar->lineno = plpgpsm_location_to_lineno(@1);
					declvar->location = @1;
					$$ = (PLpgPSM_astnode *) declvar;
				}
			| DECLARE declare_prefetch CONDITION opt_sqlstate
				{
					PLpgPSM_declare_condition *declcond = (PLpgPSM_declare_condition *) $2;
					PLpgPSM_condition *condition = $4;

					Assert(declcond->astnode.type == PLPGPSM_DECLARE_CONDITION);
					declcond->lineno = plpgpsm_location_to_lineno(@1);
					declcond->location = @1;

					if (condition != NULL)
					{
						Assert(condition->type == PLPGPSM_CONDITION_SQLSTATE);
						if (ERRCODE_TO_CATEGORY(condition->sqlstate) == ERRCODE_TO_CATEGORY(MAKE_SQLSTATE('0','0','0','0','0')))
							ereport(ERROR,
									(errcode(ERRCODE_SYNTAX_ERROR),
									 errmsg("cannot attach error category '00'"),
											parser_errposition(condition->location)));
						declcond->condition->sqlstate = condition->sqlstate;
						pfree(condition);
					}
					$$ = (PLpgPSM_astnode *) declcond;
				}
			| DECLARE declare_prefetch HANDLER FOR condition_list stmt
				{
					PLpgPSM_declare_handler *declhandler = (PLpgPSM_declare_handler *) $2;

					Assert(declhandler->astnode.type == PLPGPSM_DECLARE_HANDLER);
					declhandler->lineno = plpgpsm_location_to_lineno(@1);
					declhandler->location = @1;
					declhandler->condition_list = $5;
					declhandler->stmt = $6;
					declhandler->stmt->parent = (PLpgPSM_astnode *) declhandler;
					$$ = (PLpgPSM_astnode *) declhandler;
				}
			| DECLARE declare_prefetch CURSOR FOR
				{
					int	tok;
					PLpgPSM_declare_cursor *declcursor = (PLpgPSM_declare_cursor *) $2;

					Assert(declcursor->astnode.type == PLPGPSM_DECLARE_CURSOR);
					declcursor->lineno = plpgpsm_location_to_lineno(@1);
					declcursor->location = @1;

					tok = yylex();
					if (tok != T_WORD && !is_unreserved_keyword(tok))
					{
						plpgpsm_push_back_token(tok);
						declcursor->esql = read_esql(';', 0, 0, ";", false, NULL, -1, false);
						declcursor->esql->astnode.parent = (PLpgPSM_astnode *) declcursor;
					}
					else
					{
						declcursor->esql = NULL;
					}
					$$ = (PLpgPSM_astnode *) declcursor;
				}
		;

declare_prefetch:
			{ $$ = declare_prefetch(); }
		;

condition_list:
			condition_list ',' condition		{ $$ = lappend($1, $3); }
			| condition				{ $$ = list_make1($1); }
		;

condition:
			sqlstate	{ $$ = condition(PLPGPSM_CONDITION_SQLSTATE, $1, NULL, @1); }
			| NOT FOUND	{ $$ = condition(PLPGPSM_CONDITION_NOTFOUND, MAKE_SQLSTATE('0','2','0','0','0'), NULL, @1); }
			| SQLWARNING	{ $$ = condition(PLPGPSM_CONDITION_SQLWARNING, 0, NULL, @1); }
			| SQLEXCEPTION	{ $$ = condition(PLPGPSM_CONDITION_SQLEXCEPTION, 0, NULL, @1); }
			| T_WORD	{ $$ = condition(PLPGPSM_CONDITION_NAMED, 0, $1.ident, @1); }
		;

opt_sqlstate:
			FOR sqlstate		{ $$ = condition(PLPGPSM_CONDITION_SQLSTATE, $2, NULL, @2); }
			|			{ $$ = NULL; }
		;

sqlstate:
			SQLSTATE opt_value SCONST
				{
					const char *sqlstatestr;

					sqlstatestr = $3;

					if (strlen(sqlstatestr) != 5)
						yyerror("invalid SQLSTATE code");
					if (strspn(sqlstatestr, "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ") != 5)
						yyerror("invalid SQLSTATE code");

					$$ = MAKE_SQLSTATE(sqlstatestr[0],
									    sqlstatestr[1],
									    sqlstatestr[2],
									    sqlstatestr[3],
									    sqlstatestr[4]);
					if ($$ == 0)
						yyerror("invalid SQLSTATE code");
				}
		;

opt_value:
			VALUE			{ }
			|			{ }
		;

/*
 * Assign statement
 *
 *    SET target1 = expr1, target2 = expr2, ..
 *    SET (target1, target2, ..) = ( ... )
 *
 */
stmt_assign:
			SET simple_assign_list
				{
					if (list_length($2) > 1)
					{
						PLpgPSM_eb_stmt *sb;

						sb = plpgpsm_eb_stmt(PLPGPSM_SIMPLE_BLOCK, NULL, @1,
									    NULL, NULL, $2);
						sb->astnode.is_visible = false;
						$$ = (PLpgPSM_astnode *) sb;

					}
					else
					{
						$$ = (PLpgPSM_astnode *) linitial($2);
					}
				}
			| SET '(' target_list ')' '=' '('
				{
					plpgpsm_push_back_token('(');

					$$ = (PLpgPSM_astnode *) new_stmt_assign(NULL, $3,
						    read_esql(';', 0, 0, ";", true, NULL, -1, false),
												    @1);
				}
		;

simple_assign_list:
			simple_assign_list ',' simple_assign		{ $$ = lappend($1, $3); }
			| simple_assign					{ $$ = list_make1($1); }
		;

simple_assign:
			target '=' expr_until_comma_semi
				{
					$$ = (PLpgPSM_astnode *) new_stmt_assign($1, NIL, $3, @1);
				}
		;

target_list:
			target_list ',' target		{ $$ = lappend($1, $3); }
			| target			{ $$ = list_make1($1); }
		;

target:
			T_WORD
				{
					$$ = new_referer(new_ident_var(NULL, $1.ident, @1), NULL);
				}
			| T_CWORD
				{
					$$ = new_referer(new_ident_var(strVal(linitial($1.idents)),
								       strVal(lsecond($1.idents)),
								       @1), NULL);
				}
			| unreserved_keyword
				{
					$$ = new_referer(new_ident_var(NULL, pstrdup($1), @1), NULL);
				}
		;

/*
 * Conditional statement
 *
 * IF a1 THEN s1; ELSEIF a2 THEN s2; ELSEIF a3 THEN s3; ELSE s4; END IF will be stored to
 *
 * PLPGPSM_STMT_IF
 *   PLPGPSM_CONDITIONAL_BLOCK
 *      condition:  a1
 *      statements: s1
 *   PLPGPSM_CONDITIONAL_BLOCK
 *      condition:  a2
 *      statements: s2
 *   PLPGPSM_CONDITIONAL_BLOCK
 *      condition:  a3
 *      statements: s3
 *   PLPGPSM_CONDITIONAL_BLOCK
 *      condition:  ""
 *      statements: s4
 *
 */
stmt_if:		IF  expr_until_then THEN stmts_opt opt_elseif_then_list opt_else_stmts END IF
				{
					PLpgPSM_eb_stmt *eb;
					PLpgPSM_eb_stmt *cb;
					List			*cblist;

					cast_to_boolean($2);
					cb = plpgpsm_eb_stmt(PLPGPSM_CONDITIONAL_BLOCK, NULL, @1,
									    NULL, $2, $4);
					cblist = list_make1(cb);
					if ($5 != NIL)
						cblist = list_concat(cblist, $5);
					if ($6 != NULL)
						cblist = lappend(cblist, $6);

					eb = plpgpsm_eb_stmt(PLPGPSM_STMT_IF, NULL, @1,
									    NULL, NULL, cblist);
					$$ = (PLpgPSM_astnode *) eb;
				}
		;

opt_else_stmts:
			ELSE stmts_opt
				{
					PLpgPSM_eb_stmt *eb;

					eb = plpgpsm_eb_stmt(PLPGPSM_CONDITIONAL_BLOCK, NULL, @1,
									    NULL, NULL, $2);
					$$ = eb;
				}
			|
				{
					$$ = NULL;
				}
		;

opt_elseif_then_list:
			elseif_then_list	{ $$ = $1; }
			|			{ $$ = NIL; }
		;

elseif_then_list:
			elseif_then_list elseif_then	{ $$ = lappend($1, $2); }
			| elseif_then			{ $$ = list_make1($1); }
		;

elseif_then:
			ELSEIF expr_until_then THEN stmts_opt
				{
					PLpgPSM_eb_stmt *eb;

					cast_to_boolean($2);
					eb = plpgpsm_eb_stmt(PLPGPSM_CONDITIONAL_BLOCK, NULL, @1,
									    NULL, $2, $4);
					$$ = eb;
				}
		;

expr_until_then:
				{ $$ = read_esql(THEN, 0, 0, "THEN", true, NULL, -1, true); }
		;

/*
 * Conditional statement
 *
 * CASE expr WHEN expr, expr, .. THEN stmts ; ELSE stmts END CASE
 * CASE WHEN expr THEN stmts ; WHEN expr THEN stmts ; ELSE  stmts END CASE
 *
 *      will be stored to
 *
 * PLPGPSM_STMT_CASE
 *   ASSIGN hidenvar = expr
 *   PLPGPSM_CONDITIONAL_BLOCK
 *      condition:  a1
 *      statements: s1
 *   PLPGPSM_CONDITIONAL_BLOCK
 *      condition:  a2
 *      statements: s2
 *   PLPGPSM_CONDITIONAL_BLOCK
 *      condition:  a3
 *      statements: s3
 *   PLPGPSM_CONDITIONAL_BLOCK
 *      condition:  ""
 *      statements: s4
 *
 */
stmt_case:		CASE opt_expr_until_when case_when_list opt_else_stmts END CASE
				{
					/*
					 * When opt_expr is not used, then a output is just sequences
					 * of conditional blocks. But when expr is entered, then we need
					 * allocate a variable, that will hold a result of expression
					 * and we have to join a variable and searching expressions.
					 *
					 */
					if ($2 != NULL)
					{
						$$ = (PLpgPSM_astnode *) new_search_case($2, $3, $4, @1);
					}
					else
					{
						List	*cblist = $3;

						if ($4 != NULL)
							cblist = lappend(cblist, $4);

						$$ = (PLpgPSM_astnode *) plpgpsm_eb_stmt(PLPGPSM_STMT_CASE,
											NULL, @1,
												NULL, NULL,
													    cblist);
					}
				}
		;

opt_expr_until_when:
				{ $$ = read_esql(WHEN, 0, 0, "WHEN", true, NULL, -1, true); }
		;

case_when_list:
			case_when_list case_when	{ $$ = lappend($1, $2); }
			| case_when			{ $$ = list_make1($1); }
		;

case_when:
			WHEN expr_until_then THEN stmts_opt
				{
					PLpgPSM_eb_stmt *eb;

					eb = plpgpsm_eb_stmt(PLPGPSM_CONDITIONAL_BLOCK, NULL, @1,
								    NULL, $2, $4);
					$$ = eb;
				}
		;

/*
 * LOOP cycle
 *
 * [ label: ] LOOP statements ; END LOOP [ label ]
 *
 */
stmt_loop:
			opt_label LOOP stmts_opt END LOOP opt_end_label
				{
					PLpgPSM_eb_stmt *stmt;

					check_labels($1, $6);
					stmt = plpgpsm_eb_stmt(PLPGPSM_STMT_LOOP, NULL, @2,
								    $1, NULL, $3);
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

/*
 * WHILE cycle
 *
 * [ label: ] WHILE expr DO statements ; END WHILE [ label ]
 *
 */
stmt_while:
			opt_label WHILE expr_until_do DO stmts_opt END WHILE opt_end_label
				{
					PLpgPSM_eb_stmt *stmt;

					check_labels($1, $8);
					cast_to_boolean($3);
					stmt = plpgpsm_eb_stmt(PLPGPSM_STMT_WHILE, NULL, @2,
								    $1, $3, $5);
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

/*
 * used in stmt_while
 *
 */
expr_until_do:
				{ $$ = read_esql(DO, 0, 0, "DO", true, NULL, -1, false); }
		;

/*
 * REPEAT cycle
 *
 * [ label: ] REPEAT statements ; UNTIL expr END REPEAT [ label ]
 *
 */
stmt_repeat_until:
			opt_label REPEAT stmts_opt UNTIL expr_until_end END REPEAT opt_end_label
				{
					PLpgPSM_eb_stmt *stmt;

					check_labels($1, $8);
					cast_to_boolean($5);
					stmt = plpgpsm_eb_stmt(PLPGPSM_STMT_REPEAT_UNTIL, NULL, @2,
								    $1, $5, $3);
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

/*
 * used in stmt_repeat_until
 *
 */
expr_until_end:
				{ $$ = read_esql(END, 0, 0, "END", true, NULL, -1, false); }
		;

/*
 * When missing a colon, then move a position back, because we want
 * to mark a WORD, not next token.
 *
 */
opt_label:		T_WORD
				{
					$$ = $1.ident;
					if (yylex() != ':')
						ereport(ERROR,
								(errcode(ERRCODE_SYNTAX_ERROR),
								 errmsg("syntax error, bizzare label \"%s\"", $1.ident),
										parser_errposition(@1)));
				}
			|
				{
					$$ = NULL; 
				}
		;

opt_end_label:
			T_WORD			{ $$ = $1.ident; }
			|			{ $$ = NULL; }
		;

/*
 * loop controls - LEAVE and ITERATE
 *
 */
stmt_leave:
			LEAVE label
				{
					PLpgPSM_stmt_leave_iterate *stmt = (PLpgPSM_stmt_leave_iterate *) palloc(sizeof(PLpgPSM_stmt_leave_iterate));

					stmt->astnode.type = PLPGPSM_STMT_LEAVE;
					stmt->astnode.is_visible = true;
					stmt->lineno = plpgpsm_location_to_lineno(@1);
					stmt->location = @2;
					stmt->label = $2;
					stmt->stop_node = NULL;
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

stmt_iterate:
			ITERATE label
				{
					PLpgPSM_stmt_leave_iterate *stmt = (PLpgPSM_stmt_leave_iterate *) palloc(sizeof(PLpgPSM_stmt_leave_iterate));

					stmt->astnode.type = PLPGPSM_STMT_ITERATE;
					stmt->astnode.is_visible = true;
					stmt->lineno = plpgpsm_location_to_lineno(@1);
					stmt->location = @2;
					stmt->label = $2;
					stmt->stop_node = NULL;
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

label:
			T_WORD			{ $$ = $1.ident; }
		;

unreserved_keyword:
			ATOMIC
			| FOUND
			| NOT
			| PRINT
			| SQLEXCEPTION
			| SQLSTATE
			| SQLWARNING
			| MESSAGE_TEXT
			| HINT_TEXT
			| DETAIL_TEXT
			| STACKED
			| CURRENT
			| DIAGNOSTICS
			| RETURNED_SQLCODE
			| RETURNED_SQLSTATE
			| ROW_COUNT
			| CONDITION_IDENTIFIER
		;

/*
 * RETURN stmt
 *
 */
stmt_return:
			RETURN opt_expr
				{
					PLpgPSM_stmt_return *stmt = (PLpgPSM_stmt_return *) palloc(sizeof(PLpgPSM_stmt_return));

					stmt->astnode.type = PLPGPSM_STMT_RETURN;
					stmt->astnode.is_visible = true;
					stmt->lineno = plpgpsm_location_to_lineno(@1);
					stmt->location = @1;
					stmt->expr = $2;

					if (stmt->expr != NULL)
					{
						stmt->expr->astnode.parent = (PLpgPSM_astnode *) stmt;
						if (parser.noutargs > 0)
							ereport(ERROR,
									(errcode(ERRCODE_SYNTAX_ERROR),
									 errmsg("RETURN should not have a expression when OUT parameters are used"),
											parser_errposition(@2)));
					}
					else if (parser.noutargs == 0 && !parser.is_void)
						ereport(ERROR,
								(errcode(ERRCODE_SYNTAX_ERROR),
								 errmsg("RETURN should have a expression when OUT parameters are not used"),
										parser_errposition(@1)));

					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

/*
 * used in stmt_return
 *
 */
opt_expr:
				{ $$ = read_esql(';', 0, 0, ";", true, NULL, -1, true); }
		;

/*
 * debug output
 *
 * PRINT expr, expr, ..
 *
 */
stmt_print:
			PRINT expr_list
				{
					PLpgPSM_stmt_print *stmt = (PLpgPSM_stmt_print *) palloc(sizeof(PLpgPSM_stmt_print));

					stmt->astnode.type = PLPGPSM_STMT_PRINT;
					stmt->astnode.is_visible = true;
					stmt->lineno = plpgpsm_location_to_lineno(@1);
					stmt->location = @1;
					stmt->expr_list = $2;
					set_parent(stmt->expr_list, (PLpgPSM_astnode *) stmt);
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

expr_list:
			expr_list ',' expr_until_comma_semi	{ $$ = lappend($1, $3); }
			| expr_until_comma_semi			{ $$ = list_make1($1); }
		;

expr_until_comma_semi:
				{ $$ = read_esql(';', ',', 0, "; or ','", true, NULL, -1, false); }
		;

/*
 * OPEN cursor
 *
 */
stmt_open:
			OPEN target
				{
					PLpgPSM_stmt_open *stmt = (PLpgPSM_stmt_open *) palloc(sizeof(PLpgPSM_stmt_open));

					stmt->astnode.type = PLPGPSM_STMT_OPEN;
					stmt->astnode.is_visible = true;
					stmt->lineno = plpgpsm_location_to_lineno(@1);
					stmt->location = @1;

					stmt->cursor = $2;
					stmt->cursor->ident->astnode.type = PLPGPSM_IDENT_CURSOR;
					stmt->cursor->astnode.parent = (PLpgPSM_astnode *) stmt;
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

/*
 * CLOSE cursor
 *
 */
stmt_close:
			CLOSE target
				{
					PLpgPSM_stmt_close *stmt = (PLpgPSM_stmt_close *) palloc(sizeof(PLpgPSM_stmt_close));

					stmt->astnode.type = PLPGPSM_STMT_CLOSE;
					stmt->astnode.is_visible = true;
					stmt->lineno = plpgpsm_location_to_lineno(@1);
					stmt->location = @1;

					stmt->cursor = $2;
					stmt->cursor->ident->astnode.type = PLPGPSM_IDENT_CURSOR;
					stmt->cursor->astnode.parent = (PLpgPSM_astnode *) stmt;
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

/*
 * FETCH cursor INTO targets
 *
 */
stmt_fetch:
			FETCH target INTO target_list
				{
					PLpgPSM_stmt_fetch *stmt = (PLpgPSM_stmt_fetch *) palloc(sizeof(PLpgPSM_stmt_fetch));

					stmt->astnode.type = PLPGPSM_STMT_FETCH;
					stmt->astnode.is_visible = true;
					stmt->lineno = plpgpsm_location_to_lineno(@1);
					stmt->location = @1;

					stmt->cursor = $2;
					stmt->cursor->ident->astnode.type = PLPGPSM_IDENT_CURSOR;
					stmt->cursor->astnode.parent = (PLpgPSM_astnode *) stmt;
					stmt->target_list = $4;
					set_parent(stmt->target_list, (PLpgPSM_astnode *) stmt);
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

/*
 * iterate over query
 *
 * [ label: ] FOR [ namespace AS ] [ cursor_name CURSOR FOR ] Sqlquery DO statements; END FOR [ label ]
 *
 *  this is transalated
 *
 *    FOR
 *        | cursor
 *      stmts
 *    END FOR
 *
 *
 */
stmt_for:
			opt_label FOR for_prefetch DO stmts_opt END FOR opt_end_label
				{
					PLpgPSM_eb_stmt *stmt = (PLpgPSM_eb_stmt *) $3;

					check_labels($1, $8);

					stmt->location = @2;
					stmt->lineno = plpgpsm_location_to_lineno(@2);
					stmt->label = $1;
					stmt->stmts = $5;
					set_parent(stmt->stmts, (PLpgPSM_astnode *) stmt);
					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

/*
 * SIGNAL [ SQLSTATE [ VALUE ] sqlstate ] [ SET name = 'value' [, ... ]]
 *
 * note: reraise a exception
 */
stmt_signal:
			SIGNAL signaled_condition
				{
					$$ = newsignal(PLPGPSM_STMT_SIGNAL, $2, NULL, @1);
				}
			| SIGNAL signaled_condition SET signal_info_list
				{
					$$ = newsignal(PLPGPSM_STMT_SIGNAL, $2, $4, @1);
				}
		;

/*
 * RESIGNAL [ SQLSTATE [ VALUE ] sqlstate ] [ SET name = 'value' [, ... ]]
 *
 * note: reraise a exception
 */
stmt_resignal:
			RESIGNAL
				{
					$$ = newsignal(PLPGPSM_STMT_RESIGNAL, NULL, NIL, @1);
				}
			| RESIGNAL SET signal_info_list
				{
					$$ = newsignal(PLPGPSM_STMT_RESIGNAL, NULL, $3, @1);
				}
			| RESIGNAL signaled_condition
				{
					$$ = newsignal(PLPGPSM_STMT_RESIGNAL, $2, NULL, @1);
				}
			| RESIGNAL signaled_condition SET signal_info_list
				{
					$$ = newsignal(PLPGPSM_STMT_RESIGNAL, $2, $4, @1);
				}
		;

signaled_condition:
			T_WORD			{ $$ = condition(PLPGPSM_CONDITION_NAMED, 0, $1.ident, @1); }
			| sqlstate		{ $$ = condition(PLPGPSM_CONDITION_SQLSTATE, $1, NULL, @1); }
		;

signal_info_list:
			signal_info_list ',' signal_info
				{
					ListCell	*l;
					PLpgPSM_signal_info *s1 = (PLpgPSM_signal_info *) $3;

					foreach(l, $1)
					{
						PLpgPSM_signal_info *s2 = (PLpgPSM_signal_info *) lfirst(l);

						if (s1->type == s2->type)
							yyerror("signal info already specified");
					}
					$$ = lappend($1, $3);
				}
			| signal_info
				{
					$$ = list_make1($1);
				}
		;

signal_info:
			signal_info_type '=' expr_until_comma_semi
				{
					PLpgPSM_signal_info *sinfo;

					sinfo = (PLpgPSM_signal_info *) palloc(sizeof(PLpgPSM_signal_info));

					sinfo->astnode.type = PLPGPSM_SIGNAL_INFO;
					sinfo->astnode.is_visible = true;
					sinfo->location = @1;
					sinfo->lineno = plpgpsm_location_to_lineno(@1);
					sinfo->type = $1;
					sinfo->expr = $3;
					sinfo->expr->astnode.parent = (PLpgPSM_astnode *) sinfo;

					$$ = (PLpgPSM_astnode *) sinfo;
				}
		;

signal_info_type:
			MESSAGE_TEXT			{ $$ = PLPGPSM_SIGNAL_INFO_MESSAGE; }
			| DETAIL_TEXT			{ $$ = PLPGPSM_SIGNAL_INFO_DETAIL; }
			| HINT_TEXT			{ $$ = PLPGPSM_SIGNAL_INFO_HINT; }
		;

/*
 * diagnostics statement
 *
 *   GET [CURRENT|STACKED] DIAGNOSTICS var = key, ...
 *
 */
stmt_diagnostics:
			GET diag_area_opt DIAGNOSTICS diag_info_list
				{
					PLpgPSM_stmt_diagnostics *stmt;

					stmt = (PLpgPSM_stmt_diagnostics *) palloc(sizeof(PLpgPSM_stmt_diagnostics));
					stmt->astnode.type = PLPGPSM_STMT_DIAGNOSTICS;
					stmt->astnode.is_visible = true;
					stmt->location = @1;
					stmt->lineno = plpgpsm_location_to_lineno(@1);
					stmt->diag_area = $2;
					stmt->diag_info_list = $4;
					set_parent(stmt->diag_info_list, (PLpgPSM_astnode *) stmt);

					$$ = (PLpgPSM_astnode *) stmt;
				}
		;

diag_area_opt:
			CURRENT				{ $$ = PLPGPSM_DIAGAREA_CURRENT; }
			| STACKED			{ $$ = PLPGPSM_DIAGAREA_STACKED; }
			|				{ $$ = PLPGPSM_DIAGAREA_CURRENT; }
		;

diag_info_list:
			diag_info_list ',' diag_info	{ $$ = lappend($1, $3); }
			| diag_info			{ $$ = list_make1($1); }
		;

diag_info:
			target '=' diag_info_type
				{
					PLpgPSM_diag_info *dinfo;

					dinfo = (PLpgPSM_diag_info *) palloc(sizeof(PLpgPSM_diag_info));
					dinfo->astnode.type = PLPGPSM_DIAG_INFO;
					dinfo->astnode.is_visible = true;
					dinfo->location = @1;
					dinfo->lineno = plpgpsm_location_to_lineno(@1);
					dinfo->target = $1;
					dinfo->target->astnode.parent = (PLpgPSM_astnode *) dinfo;
					dinfo->type = $3;

					$$ = (PLpgPSM_astnode *) dinfo;
				}

diag_info_type:
			RETURNED_SQLSTATE		{ $$ = PLPGPSM_DIAGINFO_RETURNED_SQLSTATE; }
			| RETURNED_SQLCODE		{ $$ = PLPGPSM_DIAGINFO_RETURNED_SQLCODE; }
			| MESSAGE_TEXT			{ $$ = PLPGPSM_DIAGINFO_MESSAGE_TEXT; }
			| DETAIL_TEXT			{ $$ = PLPGPSM_DIAGINFO_DETAIL_TEXT; }
			| HINT_TEXT			{ $$ = PLPGPSM_DIAGINFO_HINT_TEXT; }
			| ROW_COUNT			{ $$ = PLPGPSM_DIAGINFO_ROW_COUNT; }
			| CONDITION_IDENTIFIER		{ $$ = PLPGPSM_DIAGINFO_CONDITION_IDENTIFIER; }
		;

/*
 * syntax of FOR statement uses a "postfix form", so it
 * isn't possible to implement with gramatic rules.
 */
for_prefetch:					{ $$ = for_prefetch(); }
		;

%%

static PLpgPSM_condition *
condition(PLpgPSM_condition_enum type, int sqlstate, char *name, int location)
{
	PLpgPSM_condition *c = palloc0(sizeof(PLpgPSM_condition));

	c->astnode.type = PLPGPSM_CONDITION;
	c->astnode.is_visible = true;
	c->type = type;
	c->location = location;
	c->sqlstate = sqlstate;
	c->name = name;

	return c;
}

static PLpgPSM_astnode *
newsignal(PLpgPSM_astnode_enum type, PLpgPSM_condition *condition, List *signal_info_list, int location)
{
	PLpgPSM_stmt_signal *stmt = (PLpgPSM_stmt_signal *) palloc(sizeof(PLpgPSM_stmt_signal));

	stmt->astnode.type = type;
	stmt->astnode.is_visible = true;
	stmt->location = location;
	stmt->lineno = plpgpsm_location_to_lineno(location);
	stmt->condition = condition;
	stmt->signal_info_list = signal_info_list;
	stmt->handler = NULL;
	set_parent(stmt->signal_info_list, (PLpgPSM_astnode *) stmt);

	return (PLpgPSM_astnode *) stmt;
}

static PLpgPSM_astnode *
for_prefetch(void)
{
	char *namespace = NULL;
	char *cursor_name = NULL;
	int tok;
	PLpgPSM_eb_stmt *stmt_for;
	PLpgPSM_ESQL   *esql;
	PLpgPSM_cursor *cursor;

	/* read a possible loopvar_name */
	tok = yylex();
	if (tok == T_WORD)
	{
		int tok1;

		namespace = yylval.word.ident;
		tok1 = yylex();
		if (tok1 == AS)
		{
			/* try to read cursor name */
			tok = yylex();
			if (tok == T_WORD)
			{
				cursor_name = yylval.word.ident;
				tok1 = yylex();
				if (tok1 == CURSOR)
				{
					if (yylex() != FOR)
						yyerror("expected FOR");
				}
				else
				{
					cursor_name = NULL;
					plpgpsm_push_back_token(tok1);
					plpgpsm_push_back_token(tok);
				}
			}
			else
			plpgpsm_push_back_token(tok);
		}
		else if (tok1 == CURSOR)
		{
			cursor_name = namespace;
			namespace = NULL;
			if (yylex() != FOR)
				yyerror("expected FOR");
		}
		else
		{
			plpgpsm_push_back_token(tok1);
			plpgpsm_push_back_token(tok);
		}
	}
	else
		plpgpsm_push_back_token(tok);

	esql = read_esql(DO, 0, 0, "DO", false, NULL, -1, false);
	stmt_for = plpgpsm_eb_stmt(PLPGPSM_STMT_FOR, NULL, -1,
					    NULL, esql, NULL);

	stmt_for->namespace = namespace;
	cursor = (PLpgPSM_cursor *) palloc0(sizeof(PLpgPSM_cursor));
	cursor = (PLpgPSM_cursor *) palloc0(sizeof(PLpgPSM_cursor));
	cursor->astnode.type = PLPGPSM_CURSOR;
	cursor->astnode.is_visible = false;
	cursor->lineno = -1;
	cursor->location = -1;
	cursor->offset = parser.nvars++;

	if (cursor_name == NULL)
	{
		char	cursorname[20];

		snprintf(cursorname, sizeof(cursorname), "____%d", parser.nvars);
		cursor->name = pstrdup(cursorname);
		cursor->force_name = false;
	}
	else
	{
		cursor->name = cursor_name;
		cursor->force_name = true;
	}

	cursor->astnode.parent = (PLpgPSM_astnode *) stmt_for;
	stmt_for->cursor = cursor;

	return (PLpgPSM_astnode *) stmt_for;
}

/*
 * ensure so end label is same as begin label
*/
static void
check_labels(const char *label1, const char *label2)
{
	if (label2 != NULL && label1 == NULL)
		yyerror("syntax error, missing begin label");
	if (label2 == NULL || label1 == NULL)
		return;
	if (strcmp(label1, label2) != 0)
		yyerror("end label is defferent than begin label");
}

/*
 * list of tokens that are prohibited in expression
 *
 */
static bool
is_disallowed(int tok)
{
	switch (tok)
	{
		case BEGIN:
		case IF:
		case LOOP:
		case WHILE:
		case REPEAT:
		case RETURN:
		case ELSEIF:
		case DO:
		case SET:
		case UNTIL:
		case DEFAULT:
		case OPEN:
		case CLOSE:
		case FETCH:
		case SIGNAL:
		case RESIGNAL:
		case INTO:
		case GET:
			return true;
		default:
			return false;
	}
}

static bool
is_disallowed_in_typename(int tok)
{
	switch (tok)
	{
		case ATOMIC:
		case BEGIN:
		case CASE:
		case CONDITION:
		case DECLARE:
		case DEFAULT:
		case DO:
		case ELSE:
		case ELSEIF:
		case END:
		case FOR:
		case HANDLER:
		case IF:
		case ITERATE:
		case LEAVE:
		case LOOP:
		case NOT:
		case REPEAT:
		case RETURN:
		case SET:
		case THEN:
		case UNTIL:
		case VALUE:
		case WHEN:
		case WHILE:
		case SCROLL:
		case PRINT:
		case OPEN:
		case CLOSE:
		case FETCH:
		case INTO:
		case SIGNAL:
		case RESIGNAL:
		case GET:
		case '=':
			return true;
		default:
			return false;
	}
}

static bool
is_unreserved_keyword(int tok)
{
	switch (tok)
	{
		case AS:
		case ATOMIC:
		case CONTINUE:
		case CURSOR:
		case CURRENT:
		case DIAGNOSTICS:
		case EXIT:
		case FOUND:
		case NO:
		case NOT:
		case PRINT:
		case SCROLL:
		case SQLCODE:
		case SQLEXCEPTION:
		case SQLSTATE:
		case SQLWARNING:
		case STACKED:
		case UNDO:
		case MESSAGE_TEXT:
		case DETAIL_TEXT:
		case HINT_TEXT:
		case RETURNED_SQLCODE:
		case RETURNED_SQLSTATE:
		case CONDITION_IDENTIFIER:
		case ROW_COUNT:
			return true;
		default:
			return false;
	}
}

/*
 * Returns true, when token is end token of some statement
 *
 */
static bool
is_statement_final_token(int tok)
{
	int tok1, tok2;

	if (tok == END)
	{
		plpgpsm_peek2(&tok1, &tok2, NULL, NULL);

		switch (tok1)
		{
			case FOR:
			case CASE:
			case IF:
			case LOOP:
			case WHILE:
			case REPEAT:
				return true;
			default:
				return false;
		}
	}

	return false;
}

/*
 * One difference between PL/pgSQL and PL/pgPSM is optionality finishing semicolon
 * in PL/pgPSM. SQL/PSM allow single statements function. Just "RETURN expr" is
 * correct. So when one from "until" token is semicolon, then until token is 
 * optional. Semicolon or other delimiter is returned back.
 *
 */
static PLpgPSM_ESQL *
read_esql(int until1, int until2, int until3,
						 const char *expected,
						 bool is_expression,
						 int *endtoken,
						 int startlocation,
						 bool		is_optional)
{
	int parenlevel = 0;
	int tok;
	StringInfoData ds;
	PLpgPSM_ESQL *esql;

	bool		optional_limiter;		/* limiter is optional */
	bool		weak_limiter;			/* limiter is ignored between brackets - THEN, ',' */

	initStringInfo(&ds);

	optional_limiter = (until1 == ';' || until2 == ';' || until3 == ';');

	/*
	 * limiter should be ignored between parenthesis
	 */
	weak_limiter = (until1 == THEN || until2 == THEN || until3 == THEN
			|| until1 == WHEN || until2 == WHEN || until3 == WHEN
			|| until1 == END || until2 == END || until3 == END
			|| until1 == ',' || until2 == ',' || until3 == ',');

	for (;;)
	{
		/* read a current location before you read a next tag */
		tok = yylex();

		if (startlocation < 0)
			startlocation = yylloc;

		if (tok == 0 && optional_limiter)
		{
			if (optional_limiter)
			{
				/*
				 * in this case, there is nothing to return back, so
				 * just verify closed parenthesis.
				 *
				 */
				if (parenlevel > 0)
					yyerror("mismatched parentheses");

				if (endtoken)
					*endtoken = '\0';
				break;
			}
			else
				ereport(ERROR,
						(errcode(ERRCODE_SYNTAX_ERROR),
						 errmsg("missing \"%s\" at end of SQL expression",
												expected),
						 parser_errposition(yylloc)));
		}

		/*
		 * When limiter is comma or THEN, then it can be ignored under parenthesis
		 *
		 *    SET a = (10 "," 20) , b =
		 *    IF (CASE WHEN x THEN b END) THEN
		 *
		 */
		if ((tok == ',' || tok == THEN || tok == END) && weak_limiter && parenlevel > 0)
		{
			/* this is not expected limiter */
			continue;
		}
		/*
		 * Usually limiter has higher priority than parenthesis. This is identification
		 * of missing parenthesis.
		 *
		 */
		if (tok == until1 || tok == until2 || tok == until3)
		{
			if (parenlevel > 0)
				yyerror("mismatched parenthesis");

			if (endtoken)
				*endtoken = tok;
			plpgpsm_push_back_token(tok);
			break;
		}

		/*
		 * often error should be missing semicolon on end of expression. We can do
		 * some smarter and early checking - don't allow a END (IF|CASE|LOOP|REPEAT|WHILE)
		 * inside expression.
		 *
		 */
		if (is_statement_final_token(tok) || (is_disallowed(tok) && is_expression))
				yyerror("syntax error");

		if (tok == '(' || tok == '[')
			parenlevel++;
		else if (tok == ')' || tok == ']')
		{
			parenlevel--;
			if (parenlevel < 0)
				yyerror("mismatched parentheses");
		}
	}

	plpgpsm_append_source_text(&ds, startlocation, yylloc);

	if (ds.len == 0)
	{
		if (!is_optional)
			yyerror("missing expression");
		return NULL;
	}

	esql = palloc(sizeof(PLpgPSM_ESQL));

	/* try to truncate from right a returned string */
	while (isspace(ds.data[ds.len - 1]))
		ds.data[--ds.len] = '\0';

	esql->astnode.type = PLPGPSM_ESQL;
	esql->astnode.is_visible = true;

	esql->is_expression = is_expression;
	esql->location = startlocation;
	esql->lineno = plpgpsm_location_to_lineno(startlocation);
	esql->sqlstr = ds.data;
	esql->var_refs = NIL;
	esql->plan = NULL;

	return esql;
}

static PLpgPSM_ident *
read_typename()
{
	int parenlevel = 0;
	int tok;
	StringInfoData ds;
	int startlocation = -1;
	PLpgPSM_ident		*result;

	initStringInfo(&ds);

	for (;;)
	{
		/* read a current location before you read a next tag */
		tok = yylex();

		if (startlocation < 0)
			startlocation = yylloc;

		if (tok == ';' || tok == DEFAULT)
			break;

		if (tok == 0)
			yyerror("syntax error");

		if (is_disallowed_in_typename(tok))
			yyerror("syntax error");

		if (tok == '(' || tok == '[')
			parenlevel++;
		else if (tok == ')' || tok == ']')
		{
			parenlevel--;
			if (parenlevel < 0)
				yyerror("mismatched parentheses");
		}
	}

	plpgpsm_append_source_text(&ds, startlocation, yylloc);

	if (ds.len == 0)
	{
		yyerror("missing data type");
		return NULL;
	}

	/* try to truncate from right a returned string */
	while (isspace(ds.data[ds.len - 1]))
		ds.data[--ds.len] = '\0';

	plpgpsm_push_back_token(tok);

	result = (PLpgPSM_ident *) palloc(sizeof(PLpgPSM_ident));
	result->astnode.type = PLPGPSM_IDENT_TYPE;
	result->astnode.is_visible = true;
	result->name = ds.data;
	result->location = startlocation;

	return result;
}

/*
 * These states are used for controling state machine when DECLARE statement
 * is processed.
 */
typedef enum
{
	Initial = 0,
	Unknown = 1,
	EXPECTED_VARIABLE = 2,
	COMPLETE_LIST_OF_VARIABLES = 3,
	EXPECTED_DATATYPE = 4,
	EXPECTED_EXPRESSION = 5,
	EXPECTED_CURSOR = 6
} DeclareParsingState;

/*
 * Because I would to see a parser by the most simply, then
 * a parsing of declaratin part is handy written. This section needs
 * a relative large prefetch lookup.
 */
static PLpgPSM_astnode *
declare_prefetch(void)
{
	DeclareParsingState state = Initial;
	int tok;
	PLpgPSM_ident *ident = NULL;
	List	*variables = NIL;
	PLpgPSM_cursor_option cursor_option = 0;

	for (;;)
	{
		if (state == EXPECTED_DATATYPE)
		{
			PLpgPSM_ident *typename;
			PLpgPSM_ESQL *defexpr;

			typename = read_typename();

			if ((tok = yylex()) == DEFAULT)
				defexpr = read_esql(';', 0, 0, ";", true, NULL, -1, false);
			else
			{
				defexpr = NULL;
				plpgpsm_push_back_token(tok);
			}

			if (variables == NIL)
				variables = list_make1(plpgpsm_new_variable(ident->name, ident->location));

			return (PLpgPSM_astnode *) new_declvar(variables, typename, defexpr);
		}

		if (state == EXPECTED_CURSOR)
		{
			PLpgPSM_declare_cursor *declcur;
			PLpgPSM_cursor *cursor;

			tok = yylex();
			Assert(tok == CURSOR);

			cursor = (PLpgPSM_cursor *) palloc0(sizeof(PLpgPSM_cursor));
			cursor->astnode.type = PLPGPSM_CURSOR;
			cursor->lineno = plpgpsm_location_to_lineno(ident->location);
			cursor->location = ident->location;
			cursor->offset = parser.nvars++;
			cursor->name = ident->name;
			cursor->force_name = false;

			declcur = (PLpgPSM_declare_cursor *) palloc0(sizeof(PLpgPSM_declare_cursor));
			declcur->astnode.type = PLPGPSM_DECLARE_CURSOR;
			declcur->astnode.is_visible = true;
			declcur->cursor = cursor;
			declcur->cursor->astnode.parent = (PLpgPSM_astnode *) declcur;
			declcur->option = cursor_option;

			plpgpsm_push_back_token(tok);

			return (PLpgPSM_astnode *) declcur;
		}

		if ((tok = yylex()) == 0)
			yyerror("unpexpected end of function definition");

		if (state == Initial)
		{
			if (tok == T_WORD)
			{
				/* store identifier */
				ident = new_ident_var(NULL, yylval.word.ident, yylloc);
				state = Unknown;
				continue;
			}
			else if (tok == CONTINUE || tok == EXIT || tok == UNDO)
			{
				int tok1;
				const char *varname = yylval.keyword; /* store pointer for possible later usage */

				/*
				 * recheck next symbol, when next token is HANDLER,
				 * then we found a handler's declaration, otherwise
				 * it's variable declaration.
				 */
				if ((tok1 = yylex()) == HANDLER)
				{
					PLpgPSM_declare_handler *declhandler;

					declhandler = (PLpgPSM_declare_handler *) palloc(sizeof(PLpgPSM_declare_handler));
					declhandler->astnode.type = PLPGPSM_DECLARE_HANDLER;
					switch (tok)
					{
						case CONTINUE:
							declhandler->handler_type = PLPGPSM_HANDLER_CONTINUE;
							break;
						case EXIT:
							declhandler->handler_type = PLPGPSM_HANDLER_EXIT;
							break;
						case UNDO:
							declhandler->handler_type = PLPGPSM_HANDLER_UNDO;
							break;
					}

					plpgpsm_push_back_token(tok1);

					return (PLpgPSM_astnode *) declhandler;
				}
				else
				{
					ident = new_ident_var(NULL, pstrdup(varname), yylloc);
					state = Unknown;
					continue;
				}
			}
			else if (is_unreserved_keyword(tok))
			{
				ident = new_ident_var(NULL, yylval.word.ident, yylloc);
				state = Unknown;
				continue;
			}
		}

		if (state == Unknown)
		{
			if (tok == ',')
			{
				/* store identificator */
				state = EXPECTED_VARIABLE;
				continue;
			}
			else if (tok == CONDITION)
			{
				PLpgPSM_declare_condition *declcond;

				declcond = (PLpgPSM_declare_condition *) palloc(sizeof(PLpgPSM_declare_condition));
				declcond->astnode.type = PLPGPSM_DECLARE_CONDITION;
				declcond->astnode.is_visible = true;

				declcond->condition = condition(PLPGPSM_CONDITION_NAMED, 0, ident->name, ident->location);
				declcond->astnode.parent = (PLpgPSM_astnode *) declcond;
				pfree(ident);

				plpgpsm_push_back_token(tok);

				return (PLpgPSM_astnode *) declcond;
			}
			else if (tok == CURSOR)
			{
				cursor_option = PLPGPSM_CURSOR_NOSCROLL;
				state = EXPECTED_CURSOR;

				plpgpsm_push_back_token(tok);

				continue;
			}
			else if (tok == SCROLL)
			{
				int tok1;

				tok1 = yylex();
				if (tok1 == CURSOR)
				{
					cursor_option = PLPGPSM_CURSOR_SCROLL;
					state = EXPECTED_CURSOR;

					plpgpsm_push_back_token(tok);

					continue;
				}
				else
				{
					state = EXPECTED_DATATYPE;

					plpgpsm_push_back_token(tok1);
					plpgpsm_push_back_token(tok);

					continue;
				}
			}
			else if (tok == NO)
			{
				int tok1;

				tok1 = yylex();
				if (tok1 != SCROLL)
				{
					state = EXPECTED_DATATYPE;

					plpgpsm_push_back_token(tok1);
					plpgpsm_push_back_token(tok);

					continue;
				}
				else
				{
					int tok2 = yylex();

					if (tok2 != CURSOR)
					{
						state = EXPECTED_DATATYPE;

						plpgpsm_push_back_token(tok2);
						plpgpsm_push_back_token(tok1);
						plpgpsm_push_back_token(tok);

						continue;
					}
					else
					{
						state = EXPECTED_CURSOR;
						cursor_option = PLPGPSM_CURSOR_NOSCROLL;

						plpgpsm_push_back_token(tok2);

						continue;
					}
				}
			}
			else if (tok == T_WORD || is_unreserved_keyword(tok))
			{
				state = EXPECTED_DATATYPE;

				plpgpsm_push_back_token(tok);

				continue;
			}
		}

		if (state == EXPECTED_VARIABLE)
		{
			ListCell	*lc;

			if (tok != T_WORD && !is_unreserved_keyword(tok))
				yyerror("missing a variable identifier");

			foreach (lc, variables)
			{
				PLpgPSM_variable *variable = (PLpgPSM_variable *) lfirst(lc);

				if (strcmp(variable->name, yylval.word.ident) == 0)
					ereport(ERROR,
							(errcode(ERRCODE_SYNTAX_ERROR),
						errmsg("variable \"%s\" is not unique", yylval.word.ident),
								     parser_errposition(yylloc)));
			}

			if (variables != NIL)
				variables = lappend(variables, plpgpsm_new_variable(yylval.word.ident, yylloc));
			else
				variables = list_make2(plpgpsm_new_variable(ident->name, ident->location),
							plpgpsm_new_variable(yylval.word.ident, yylloc));

			state = COMPLETE_LIST_OF_VARIABLES;

			continue;
		}

		if (state == COMPLETE_LIST_OF_VARIABLES)
		{
			if (tok == ',')
			{
				/* do nothing */
				state = EXPECTED_VARIABLE;

				continue;
			}
			else
			{
				state = EXPECTED_DATATYPE;

				plpgpsm_push_back_token(tok);

				continue;
			}
		}

		yyerror("syntax error");
	}

	elog(ERROR, "unexpeced error");

	return NULL;
}

/*
 * Ensure unique label in statement scope
 *
 */
static void
check_labels_in_current_scope(List *stmts, PLpgPSM_astnode *stmt)
{
	if (plpgpsm_allow_cast_to_eb(stmt))
	{
		char *label = ((PLpgPSM_eb_stmt *) stmt)->label;

		if (label != NULL)
		{
			ListCell	*lc;

			foreach (lc, stmts)
			{
				PLpgPSM_astnode *astnode = (PLpgPSM_astnode *) lfirst(lc);

				if (plpgpsm_allow_cast_to_eb(astnode))
				{
					char *label2 = ((PLpgPSM_eb_stmt *) astnode)->label;

					if (label2 != NULL && strcmp(label, label2) == 0)
						ereport(ERROR,
								(errcode(ERRCODE_SYNTAX_ERROR),
					errmsg("label \"%s\" is not unique in current scope", label),
					     parser_errposition(((PLpgPSM_eb_stmt *) stmt)->location)));
				}
			}
		}
	}
}

/*
 * ensure setting of correct parent for all nodes in list
 *
 */
static void 
set_parent(List *nodes, PLpgPSM_astnode *parent)
{
	ListCell	*lc;

	foreach (lc, nodes)
	{
		((PLpgPSM_astnode *) lfirst(lc))->parent = parent;
	}
}

/*
 * Simple aux. routines
 *
 */

static PLpgPSM_referer *
new_referer(PLpgPSM_ident *ident, PLpgPSM_astnode *ref)
{
	PLpgPSM_referer *referer = (PLpgPSM_referer *) palloc(sizeof(PLpgPSM_referer));

	referer->astnode.type = PLPGPSM_REFERER;
	referer->astnode.is_visible = false;
	referer->ident = ident;
	referer->ref = ref;
	ident->astnode.parent = (PLpgPSM_astnode *) referer;

	return referer;
}

static PLpgPSM_ident *
new_ident_var(char *scope, char *name, int location)
{
	PLpgPSM_ident *ident = (PLpgPSM_ident *) palloc(sizeof(PLpgPSM_ident));

	ident->astnode.type = PLPGPSM_IDENT_VARIABLE;
	ident->astnode.is_visible = true;
	ident->scope = scope;
	ident->name = name;
	ident->location = location;

	return ident;
}

static PLpgPSM_stmt_assign *
new_stmt_assign(PLpgPSM_referer *referer, List *idents, PLpgPSM_ESQL *expr, int location)
{
	PLpgPSM_stmt_assign *stmt = (PLpgPSM_stmt_assign *) palloc(sizeof(PLpgPSM_stmt_assign));

	stmt->astnode.type = PLPGPSM_STMT_ASSIGN;
	stmt->astnode.is_visible = true;
	stmt->location = location;
	stmt->lineno = location != -1 ? plpgpsm_location_to_lineno(location) : -1;
	stmt->expr = expr;
	stmt->expr->astnode.parent = (PLpgPSM_astnode *) stmt;
	stmt->target = referer;
	stmt->target_list = idents;

	if (stmt->target != NULL)
	{
		stmt->target->astnode.parent = (PLpgPSM_astnode *) stmt;
	}
	else if (list_length(stmt->target_list) == 1)
	{
		stmt->target = linitial(stmt->target_list);
		stmt->target->astnode.parent = (PLpgPSM_astnode *) stmt;
		stmt->target_list = NIL;
	}
	else
	{
		stmt->target = NULL;
		set_parent(stmt->target_list, (PLpgPSM_astnode *) stmt);
	}

	return stmt;
}

/*
 * result of this rule will be hiearchy:
 *  compound statement
 *    -> declare variable
 *    -> case
 *       -> assign statement
 *       -> conditional block
 *          -> stmts
 *
 * This technique is mix of AST and bytecode, I would not increase
 * complexity of CASE statement execution. Assign statement is inside
 * CASE statement, because it is invisible, and when evaluation of
 * expression raises expression, then we would to find CASE as a nearest
 * outer visible statement.
 *
 */
static PLpgPSM_eb_stmt *
new_search_case(PLpgPSM_ESQL *expr, List *case_when_list, PLpgPSM_eb_stmt *else_stmts, int location)
{
	PLpgPSM_variable *case_var;
	PLpgPSM_declare_variable *dvar;
	PLpgPSM_eb_stmt *case_stmt;
	ListCell		*lc;
	List		*cblist;
	char	varname[20];
	PLpgPSM_stmt_assign *assign_stmt;
	PLpgPSM_eb_stmt *result;
	PLpgPSM_ident *fake_ident;

	snprintf(varname, sizeof(varname), "____%d", parser.nvars);
	case_var = plpgpsm_new_variable(pstrdup(varname), location);
	case_var->astnode.is_visible = false;

	dvar = new_declvar(list_make1(case_var), NULL, NULL);
	dvar->astnode.is_visible = false;

	fake_ident = new_ident_var(NULL, pstrdup(varname), -1);
	fake_ident->astnode.is_visible = false;

	assign_stmt = new_stmt_assign(new_referer(fake_ident, NULL),
									NIL, expr, -1);
	assign_stmt->astnode.is_visible = false;

	cblist = list_make1(assign_stmt);
	cblist = list_concat(cblist, case_when_list);
	if (else_stmts != NULL)
		cblist = lappend(cblist, else_stmts);

	case_stmt =  plpgpsm_eb_stmt(PLPGPSM_STMT_CASE,
							NULL, location,
								NULL, NULL,
								    cblist);

	result = plpgpsm_eb_stmt(PLPGPSM_STMT_COMPOUND,
							NULL, location,
								NULL, NULL,
							list_make2(dvar, case_stmt));

	result->astnode.is_visible = false;

	/* modify a conditional blocks */
	foreach (lc, case_when_list)
	{
		PLpgPSM_eb_stmt *cb = (PLpgPSM_eb_stmt *) lfirst(lc);

		Assert(((PLpgPSM_astnode *) lfirst(lc))->type == PLPGPSM_CONDITIONAL_BLOCK);

		if (cb->expr != NULL)
		{
			StringInfoData		ds;

			if (expr != NULL)
			{
				initStringInfo(&ds);
				appendStringInfo(&ds, "(%s IN(%s))", varname, cb->expr->sqlstr);

				/* release previous expression */
				pfree(cb->expr->sqlstr);
				cb->expr->sqlstr = ds.data;
				cb->expr->location -= (strlen(varname) + 4);

				/* remove location, when is wrong */
				if (cb->expr->location < 0)
					cb->expr->location = -1;
			}
			else if (cb->expr != NULL)
				cast_to_boolean(cb->expr);
		}
	}

	return result;
}

/*
 * Enhance expression with cast to boolean
 *
 * This fragment should be removed if we can to specify target
 * type for expressions.
 *
 */
static void
cast_to_boolean(PLpgPSM_ESQL *expr)
{
	StringInfoData		ds;

	if (!expr->is_expression)
		elog(ERROR, "cannot to cast query to boolean");

	initStringInfo(&ds);
	appendStringInfo(&ds, "(%s)::boolean", expr->sqlstr);
	pfree(expr->sqlstr);
	expr->sqlstr = ds.data;
	(expr->location)--;
}

static PLpgPSM_declare_variable *
new_declvar(List *variable_list, PLpgPSM_ident *typename, PLpgPSM_ESQL *expr)
{
	PLpgPSM_declare_variable *declvar;

	declvar = (PLpgPSM_declare_variable *) palloc(sizeof(PLpgPSM_declare_variable));

	declvar->astnode.type = PLPGPSM_DECLARE_VARIABLE;
	declvar->astnode.is_visible = true;
	declvar->location = -1;
	declvar->variables = variable_list;
	declvar->typename = typename;
	declvar->expr = expr;
	if (declvar->expr != NULL)
		declvar->expr->astnode.parent = (PLpgPSM_astnode *) declvar;

	set_parent(declvar->variables, (PLpgPSM_astnode *) declvar);

	return declvar;
}
