/* Module:          SQLSpecialColumns.c
 *
 * Description:     Retrieves the following information about columns within a specified table: 
 *					The optimal set of columns that uniquely identifies a row in the table. 
 *					Columns that are automatically updated when any value in the row is 
 *					updated by a transaction. 
 *
 * Classes:         
 *
 * API functions:   SQLSpecialColumns
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLSpecialColumns(
									SQLHSTMT		hDrvStmt,
									SQLUSMALLINT	nColumnType,
									SQLCHAR			*szCatalogName,
									SQLSMALLINT		nCatalogNameLength,
									SQLCHAR			*szSchemaName,
									SQLSMALLINT		nSchemaNameLength,
									SQLCHAR			*szTableName,
									SQLSMALLINT		nTableNameLength,
									SQLUSMALLINT	nScope,
									SQLUSMALLINT	fNullable
									)
{
	static char *func = "SQLSpecialColumns";
	TupleNode *row;
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	ConnInfo *ci;
	SQLHSTMT hcol_stmt;
	StatementClass *col_stmt;
	char columns_query[STD_STATEMENT_LEN];
	SQLRETURN result;
	char relhasrules[MAX_INFO_STRING];

	mylog("%s: entering...stmt=%u\n", func, stmt);

    if( ! stmt)
	{
		SC_log_error(func, "", NULL);
        return SQL_INVALID_HANDLE;
    }
	ci = &stmt->hdbc->connInfo;

	stmt->manual_result = TRUE;


	/* ********************************************************************** */
	/*	Create the query to find out if this is a view or not... */
	/* ********************************************************************** */
	sprintf(columns_query, "select c.relhasrules "
		"from pg_user u, pg_class c where "
		"u.usesysid = c.relowner");

	my_strcat(columns_query, " and c.relname like '%.*s'", szTableName, nTableNameLength);
	my_strcat(columns_query, " and u.usename like '%.*s'", szSchemaName, nSchemaNameLength);


    result = SQLAllocStmt( stmt->hdbc, &hcol_stmt);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO))
	{
		stmt->errornumber = STMT_NO_MEMORY_ERROR;
		stmt->errormsg = "Couldn't allocate statement for SQLSpecialColumns result.";
		SC_log_error(func, "", stmt);
        return SQL_ERROR;
    }
	col_stmt = (StatementClass *) hcol_stmt;

	mylog("SQLSpecialColumns: hcol_stmt = %u, col_stmt = %u\n", hcol_stmt, col_stmt);

    result = SQLExecDirect(hcol_stmt, columns_query,
                           strlen(columns_query));
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) 
	{
		stmt->errormsg = SC_create_errormsg(hcol_stmt);
		stmt->errornumber = col_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(hcol_stmt, SQL_DROP);
        return SQL_ERROR;
    }

    result = SQLBindCol(hcol_stmt, 1, SQL_C_CHAR,
                        relhasrules, MAX_INFO_STRING, NULL);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO))
	{
		stmt->errormsg = col_stmt->errormsg;
		stmt->errornumber = col_stmt->errornumber;
		SC_log_error(func, "", stmt);
		SQLFreeStmt(hcol_stmt, SQL_DROP);
        return SQL_ERROR;
    }

	result = SQLFetch(hcol_stmt);
	SQLFreeStmt(hcol_stmt, SQL_DROP);

    stmt->result = QR_Constructor();
    extend_bindings(stmt, 8);

    QR_set_num_fields(stmt->result, 8);
    QR_set_field_info(stmt->result, 0, "SCOPE", PG_TYPE_INT2, 2);
    QR_set_field_info(stmt->result, 1, "COLUMN_NAME", PG_TYPE_TEXT, MAX_INFO_STRING);
    QR_set_field_info(stmt->result, 2, "DATA_TYPE", PG_TYPE_INT2, 2);
    QR_set_field_info(stmt->result, 3, "TYPE_NAME", PG_TYPE_TEXT, MAX_INFO_STRING);
    QR_set_field_info(stmt->result, 4, "PRECISION", PG_TYPE_INT4, 4);
    QR_set_field_info(stmt->result, 5, "LENGTH", PG_TYPE_INT4, 4);
    QR_set_field_info(stmt->result, 6, "SCALE", PG_TYPE_INT2, 2);
    QR_set_field_info(stmt->result, 7, "PSEUDO_COLUMN", PG_TYPE_INT2, 2);

    if ( relhasrules[0] != '1' )
	{
		/* use the oid value for the rowid */
		if(nColumnType == SQL_BEST_ROWID) 
		{
			row = (TupleNode *)malloc(sizeof(TupleNode) + (8 - 1) * sizeof(TupleField));

			set_tuplefield_int2(&row->tuple[0], SQL_SCOPE_SESSION);
			set_tuplefield_string(&row->tuple[1], "oid");
			set_tuplefield_int2(&row->tuple[2], pgtype_to_sqltype(stmt, PG_TYPE_OID));
			set_tuplefield_string(&row->tuple[3], "OID");
			set_tuplefield_int4(&row->tuple[4], pgtype_precision(stmt, PG_TYPE_OID, PG_STATIC, PG_STATIC));
			set_tuplefield_int4(&row->tuple[5], pgtype_length(stmt, PG_TYPE_OID, PG_STATIC, PG_STATIC));
			set_tuplefield_int2(&row->tuple[6], pgtype_scale(stmt, PG_TYPE_OID, PG_STATIC));
			set_tuplefield_int2(&row->tuple[7], SQL_PC_PSEUDO);

			QR_add_tuple(stmt->result, row);

		} 
		else if(nColumnType == SQL_ROWVER) 
		{

			Int2 the_type = PG_TYPE_INT4;

			if (atoi(ci->row_versioning)) 
			{
				row = (TupleNode *)malloc(sizeof(TupleNode) + (8 - 1) * sizeof(TupleField));

				set_tuplefield_null(&row->tuple[0]);
				set_tuplefield_string(&row->tuple[1], "xmin");
				set_tuplefield_int2(&row->tuple[2], pgtype_to_sqltype(stmt, the_type));
				set_tuplefield_string(&row->tuple[3], pgtype_to_name(stmt, the_type));
				set_tuplefield_int4(&row->tuple[4], pgtype_precision(stmt, the_type, PG_STATIC, PG_STATIC));
				set_tuplefield_int4(&row->tuple[5], pgtype_length(stmt, the_type, PG_STATIC, PG_STATIC));
				set_tuplefield_int2(&row->tuple[6], pgtype_scale(stmt, the_type, PG_STATIC));
				set_tuplefield_int2(&row->tuple[7], SQL_PC_PSEUDO);

				QR_add_tuple(stmt->result, row);
			}
		}
	}



    stmt->status = STMT_FINISHED;
    stmt->currTuple = -1;
	stmt->rowset_start = -1;
	stmt->current_col = -1;

	mylog("SQLSpecialColumns(): EXIT,  stmt=%u\n", stmt);
    return SQL_SUCCESS;
}
