/* Module:          SQLGetInfo.c
 *
 * Description:     Returns general information about the driver and data source 
 *					associated with a connection.
 *
 * Classes:         
 *
 * API functions:   SQLGetInfo
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLGetInfo(
							 SQLHDBC		hDrvDbc,
							 SQLUSMALLINT	nInfoType,
							 SQLPOINTER		pInfoValue,
							 SQLSMALLINT	nInfoValueMax,
							 SQLSMALLINT	*pnLength
							 )
{
	static char *func = "SQLGetInfo";
	ConnectionClass *conn = (ConnectionClass *) hDrvDbc;
	ConnInfo *ci;
	char *p = NULL, tmp[MAX_INFO_STRING];
	int len = 0, value = 0;
	SQLRETURN result;

	mylog( "%s: entering...nInfoType=%d\n", func, nInfoType);

	if ( ! conn) 
	{
		CC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	ci = &conn->connInfo;

    switch (nInfoType)
	{
		case SQL_ACCESSIBLE_PROCEDURES: /* ODBC 1.0 */
			p = "N";
			break;

		case SQL_ACCESSIBLE_TABLES: /* ODBC 1.0 */
			p = "N";
			break;

		case SQL_ACTIVE_CONNECTIONS: /* ODBC 1.0 */
			len = 2;
			value = MAX_CONNECTIONS;
			break;

		case SQL_ACTIVE_STATEMENTS: /* ODBC 1.0 */
			len = 2;
			value = 0;
			break;

		case SQL_ALTER_TABLE: /* ODBC 2.0 */
			len = 4;
			value = SQL_AT_ADD_COLUMN;
			break;

		case SQL_BOOKMARK_PERSISTENCE: /* ODBC 2.0 */
			/* very simple bookmark support */ 
			len = 4;
			value = globals.use_declarefetch ? 0 : (SQL_BP_SCROLL);	
			break;

		case SQL_COLUMN_ALIAS: /* ODBC 2.0 */
			p = "N";
			break;

		case SQL_CONCAT_NULL_BEHAVIOR: /* ODBC 1.0 */
			len = 2;
			value = SQL_CB_NON_NULL;
			break;

		case SQL_CONVERT_BIGINT:
		case SQL_CONVERT_BINARY:
		case SQL_CONVERT_BIT:
		case SQL_CONVERT_CHAR:
		case SQL_CONVERT_DATE:
		case SQL_CONVERT_DECIMAL:
		case SQL_CONVERT_DOUBLE:
		case SQL_CONVERT_FLOAT:
		case SQL_CONVERT_INTEGER:
		case SQL_CONVERT_LONGVARBINARY:
		case SQL_CONVERT_LONGVARCHAR:
		case SQL_CONVERT_NUMERIC:
		case SQL_CONVERT_REAL:
		case SQL_CONVERT_SMALLINT:
		case SQL_CONVERT_TIME:
		case SQL_CONVERT_TIMESTAMP:
		case SQL_CONVERT_TINYINT:
		case SQL_CONVERT_VARBINARY:
		case SQL_CONVERT_VARCHAR: /* ODBC 1.0 */
			len = 4;
			value = nInfoType;
			break;

		case SQL_CONVERT_FUNCTIONS: /* ODBC 1.0 */
			len = 4;
			value = 0;
			break;

		case SQL_CORRELATION_NAME: /* ODBC 1.0 */
			/*	Saying no correlation name makes Query not work right.
			value = SQL_CN_NONE;
			*/
			len = 2;
			value = SQL_CN_ANY;
			break;

		case SQL_CURSOR_COMMIT_BEHAVIOR: /* ODBC 1.0 */
			len = 2;
			value = SQL_CB_CLOSE;
			break;

		case SQL_CURSOR_ROLLBACK_BEHAVIOR: /* ODBC 1.0 */
			len = 2;
			value = SQL_CB_CLOSE;
			break;

		case SQL_DATA_SOURCE_NAME: /* ODBC 1.0 */
			p = CC_get_DSN(conn);
			break;

		case SQL_DATA_SOURCE_READ_ONLY: /* ODBC 1.0 */
			p = CC_is_onlyread(conn) ? "Y" : "N";
			break;

		case SQL_DATABASE_NAME: /* Support for old ODBC 1.0 Apps */
			/*	Returning the database name causes problems in MS Query.
			It generates query like: "SELECT DISTINCT a FROM byronncrap3 crap3"
			p = CC_get_database(conn);
			*/
			p = "";
			break;

		case SQL_DBMS_NAME: /* ODBC 1.0 */
			p = DBMS_NAME;
			break;

		case SQL_DBMS_VER: /* ODBC 1.0 */
			/* The ODBC spec wants ##.##.#### ...whatever... so prepend the driver */
			/* version number to the dbms version string */
			sprintf(tmp, "%s %s", POSTGRESDRIVERVERSION, conn->pg_version);
			p = tmp;
			break;

		case SQL_DEFAULT_TXN_ISOLATION: /* ODBC 1.0 */
			len = 4;
			value = SQL_TXN_READ_COMMITTED; /*SQL_TXN_SERIALIZABLE; */
			break;

		case SQL_DRIVER_NAME: /* ODBC 1.0 */
			p = DRIVER_FILE_NAME;
			break;

		case SQL_DRIVER_ODBC_VER:
			p = DRIVER_ODBC_VER;
			break;

		case SQL_DRIVER_VER: /* ODBC 1.0 */
			p = POSTGRESDRIVERVERSION;
			break;

		case SQL_EXPRESSIONS_IN_ORDERBY: /* ODBC 1.0 */
			p = "N";
			break;

		case SQL_FETCH_DIRECTION: /* ODBC 1.0 */
			len = 4;
			value = globals.use_declarefetch ? (SQL_FD_FETCH_NEXT) : (SQL_FD_FETCH_NEXT |
                                   SQL_FD_FETCH_FIRST |
                                   SQL_FD_FETCH_LAST |
                                   SQL_FD_FETCH_PRIOR |
                                   SQL_FD_FETCH_ABSOLUTE |
								   SQL_FD_FETCH_RELATIVE | 
								   SQL_FD_FETCH_BOOKMARK);
			break;

		case SQL_FILE_USAGE: /* ODBC 2.0 */
			len = 2;
			value = SQL_FILE_NOT_SUPPORTED;
			break;

		case SQL_GETDATA_EXTENSIONS: /* ODBC 2.0 */
			len = 4;
			value = (SQL_GD_ANY_COLUMN | SQL_GD_ANY_ORDER | SQL_GD_BOUND | SQL_GD_BLOCK);
			break;

		case SQL_GROUP_BY: /* ODBC 2.0 */
			len = 2;
			value = SQL_GB_GROUP_BY_EQUALS_SELECT;
			break;

		case SQL_IDENTIFIER_CASE: /* ODBC 1.0 */
			/*	are identifiers case-sensitive (yes, but only when quoted.  If not quoted, they
			default to lowercase)
			*/
			len = 2;
			value = SQL_IC_LOWER;
			break;

		case SQL_IDENTIFIER_QUOTE_CHAR: /* ODBC 1.0 */
			/* the character used to quote "identifiers" */
			p = PG_VERSION_LE(conn, 6.2) ? " " : "\"";
			break;

		case SQL_KEYWORDS: /* ODBC 2.0 */
			p = "";
			break;

		case SQL_LIKE_ESCAPE_CLAUSE: /* ODBC 2.0 */
			/*	is there a character that escapes '%' and '_' in a LIKE clause?
			not as far as I can tell
			*/
			p = "N";
			break;

		case SQL_LOCK_TYPES: /* ODBC 2.0 */
			len = 4;
			value = globals.lie ? (SQL_LCK_NO_CHANGE | SQL_LCK_EXCLUSIVE | SQL_LCK_UNLOCK) : SQL_LCK_NO_CHANGE;
			break;

		case SQL_MAX_BINARY_LITERAL_LEN: /* ODBC 2.0 */
			len = 4;
			value = 0;
			break;

		case SQL_MAX_CHAR_LITERAL_LEN: /* ODBC 2.0 */
			len = 4;
			value = 0;
			break;

		case SQL_MAX_COLUMN_NAME_LEN: /* ODBC 1.0 */
			len = 2;
			value = MAX_COLUMN_LEN;
			break;

		case SQL_MAX_COLUMNS_IN_GROUP_BY: /* ODBC 2.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_COLUMNS_IN_INDEX: /* ODBC 2.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_COLUMNS_IN_ORDER_BY: /* ODBC 2.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_COLUMNS_IN_SELECT: /* ODBC 2.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_COLUMNS_IN_TABLE: /* ODBC 2.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_CURSOR_NAME_LEN: /* ODBC 1.0 */
			len = 2;
			value = MAX_CURSOR_LEN;
			break;

		case SQL_MAX_INDEX_SIZE: /* ODBC 2.0 */
			len = 4;
			value = 0;
			break;

		case SQL_MAX_OWNER_NAME_LEN: /* ODBC 1.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_PROCEDURE_NAME_LEN: /* ODBC 1.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_QUALIFIER_NAME_LEN: /* ODBC 1.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_ROW_SIZE: /* ODBC 2.0 */
			len = 4;
			if (PG_VERSION_GE(conn, 7.1))
			{
				/* Large Rowa in 7.1+ */
				value = MAX_ROW_SIZE;
			}
			else 
			{
				/* Without the Toaster we're limited to the blocksize */
				value = BLCKSZ;
			}
			break;

		case SQL_MAX_ROW_SIZE_INCLUDES_LONG: /* ODBC 2.0 */
			/*	does the preceding value include LONGVARCHAR and LONGVARBINARY
			fields?   Well, it does include longvarchar, but not longvarbinary.
			*/
			p = "Y";
			break;

		case SQL_MAX_STATEMENT_LEN: /* ODBC 2.0 */
			/* maybe this should be 0? */
			len = 4;
			if (PG_VERSION_GE(conn, 7.0))
			{
				/* Long Queries in 7.0+ */
				value = MAX_STATEMENT_LEN;
			}
			else if (PG_VERSION_GE(conn, 6.5)) /* Prior to 7.0 we used 2*BLCKSZ */
			{
				value = (2*BLCKSZ);
			}
			else /* Prior to 6.5 we used BLCKSZ */
			{
				value = BLCKSZ;
			}
			break;

		case SQL_MAX_TABLE_NAME_LEN: /* ODBC 1.0 */
			len = 2;
			value = MAX_TABLE_LEN;
			break;

		case SQL_MAX_TABLES_IN_SELECT: /* ODBC 2.0 */
			len = 2;
			value = 0;
			break;

		case SQL_MAX_USER_NAME_LEN:
			len = 2;
			value = 0;
			break;

		case SQL_MULT_RESULT_SETS: /* ODBC 1.0 */
			/* Don't support multiple result sets but say yes anyway? */
			p = "Y";
			break;

		case SQL_MULTIPLE_ACTIVE_TXN: /* ODBC 1.0 */
			p = "Y";
			break;

		case SQL_NEED_LONG_DATA_LEN: /* ODBC 2.0 */
			/*	Dont need the length, SQLPutData can handle any size and multiple calls */
			p = "N";
			break;

		case SQL_NON_NULLABLE_COLUMNS: /* ODBC 1.0 */
			len = 2;
			value = SQL_NNC_NON_NULL;
			break;

		case SQL_NULL_COLLATION: /* ODBC 2.0 */
			/* where are nulls sorted? */
			len = 2;
			value = SQL_NC_END;
			break;

		case SQL_NUMERIC_FUNCTIONS: /* ODBC 1.0 */
			len = 4;
			value = 0;
			break;

		case SQL_ODBC_API_CONFORMANCE: /* ODBC 1.0 */
			len = 2;
			value = SQL_OAC_LEVEL1;
			break;

		case SQL_ODBC_SAG_CLI_CONFORMANCE: /* ODBC 1.0 */
			len = 2;
			value = SQL_OSCC_NOT_COMPLIANT;
			break;

		case SQL_ODBC_SQL_CONFORMANCE: /* ODBC 1.0 */
			len = 2;
			value = SQL_OSC_CORE;
			break;

		case SQL_ODBC_SQL_OPT_IEF: /* ODBC 1.0 */
			p = "N";
			break;

		case SQL_OJ_CAPABILITIES: /* ODBC 2.01 */
			len = 4;
			if (PG_VERSION_GE(conn, 7.1)) 
			{ 
				/* OJs in 7.1+ */
				value = (SQL_OJ_LEFT |
					SQL_OJ_RIGHT |
					SQL_OJ_FULL |
					SQL_OJ_NESTED |
					SQL_OJ_NOT_ORDERED |
					SQL_OJ_INNER |
					SQL_OJ_ALL_COMPARISON_OPS);
			}
			else 
			{
				/* OJs not in <7.1 */
				value = 0;
			}
			break;

		case SQL_ORDER_BY_COLUMNS_IN_SELECT: /* ODBC 2.0 */
			p = (PG_VERSION_LE(conn, 6.3)) ? "Y" : "N";
			break;

		case SQL_OUTER_JOINS: /* ODBC 1.0 */
			if (PG_VERSION_GE(conn, 7.1))
			{ 
				/* OJs in 7.1+ */
				p = "Y";
			}
			else 
			{ 
				/* OJs not in <7.1 */
				p = "N";
			}
			break;

		case SQL_OWNER_TERM: /* ODBC 1.0 */
			p = "owner";
			break;

		case SQL_OWNER_USAGE: /* ODBC 2.0 */
			len = 4;
			value = 0;
			break;

		case SQL_POS_OPERATIONS: /* ODBC 2.0 */
			len = 4;
			value = globals.lie ? (SQL_POS_POSITION | SQL_POS_REFRESH | SQL_POS_UPDATE | SQL_POS_DELETE | SQL_POS_ADD) : (SQL_POS_POSITION | SQL_POS_REFRESH);
			break;

		case SQL_POSITIONED_STATEMENTS: /* ODBC 2.0 */
			len = 4;
			value = globals.lie ? (SQL_PS_POSITIONED_DELETE | 
								SQL_PS_POSITIONED_UPDATE | 
								SQL_PS_SELECT_FOR_UPDATE) : 0;
			break;

		case SQL_PROCEDURE_TERM: /* ODBC 1.0 */
			p = "procedure";
			break;

		case SQL_PROCEDURES: /* ODBC 1.0 */
			p = "Y";
			break;

		case SQL_QUALIFIER_LOCATION: /* ODBC 2.0 */
			len = 2;
			value = SQL_QL_START;
			break;

		case SQL_QUALIFIER_NAME_SEPARATOR: /* ODBC 1.0 */
			p = "";
			break;

		case SQL_QUALIFIER_TERM: /* ODBC 1.0 */
			p = "";
			break;

		case SQL_QUALIFIER_USAGE: /* ODBC 2.0 */
			len = 4;
			value = 0;
			break;

		case SQL_QUOTED_IDENTIFIER_CASE: /* ODBC 2.0 */
			/* are "quoted" identifiers case-sensitive?  YES! */
			len = 2;
			value = SQL_IC_SENSITIVE;
			break;

		case SQL_ROW_UPDATES: /* ODBC 1.0 */
			/*  Driver doesn't support keyset-driven or mixed cursors, so
			not much point in saying row updates are supported
			*/
			p = globals.lie ? "Y" : "N";
			break;

		case SQL_SCROLL_CONCURRENCY: /* ODBC 1.0 */
			len = 4;
			value = globals.lie ? (SQL_SCCO_READ_ONLY | 
								SQL_SCCO_LOCK | 
								SQL_SCCO_OPT_ROWVER | 
								SQL_SCCO_OPT_VALUES) : (SQL_SCCO_READ_ONLY);
			break;

		case SQL_SCROLL_OPTIONS: /* ODBC 1.0 */
			len = 4;
			value = globals.lie ? (SQL_SO_FORWARD_ONLY | 
								SQL_SO_STATIC | 
								SQL_SO_KEYSET_DRIVEN | 
								SQL_SO_DYNAMIC | 
								SQL_SO_MIXED) : (globals.use_declarefetch ? SQL_SO_FORWARD_ONLY : (SQL_SO_FORWARD_ONLY | SQL_SO_STATIC));
			break;

		case SQL_SEARCH_PATTERN_ESCAPE: /* ODBC 1.0 */
			p = "";
			break;

		case SQL_SERVER_NAME: /* ODBC 1.0 */
			p = CC_get_server(conn);
			break;

		case SQL_SPECIAL_CHARACTERS: /* ODBC 2.0 */
			p = "_";
			break;

		case SQL_STATIC_SENSITIVITY: /* ODBC 2.0 */
			len = 4;
			value = globals.lie ? (SQL_SS_ADDITIONS | SQL_SS_DELETIONS | SQL_SS_UPDATES) : 0;
			break;

		case SQL_STRING_FUNCTIONS: /* ODBC 1.0 */
			len = 4;
			value = (SQL_FN_STR_CONCAT |
				SQL_FN_STR_LCASE | 
				SQL_FN_STR_LENGTH | 
				SQL_FN_STR_LOCATE | 
				SQL_FN_STR_LTRIM | 
				SQL_FN_STR_RTRIM |
				SQL_FN_STR_SUBSTRING |
				SQL_FN_STR_UCASE);
			break;

		case SQL_SUBQUERIES: /* ODBC 2.0 */
			/* postgres 6.3 supports subqueries */
			len = 4;
			value = (SQL_SQ_QUANTIFIED |
				SQL_SQ_IN |
				SQL_SQ_EXISTS |
				SQL_SQ_COMPARISON);
			break;

		case SQL_SYSTEM_FUNCTIONS: /* ODBC 1.0 */
			len = 4;
			value = 0;
			break;

		case SQL_TABLE_TERM: /* ODBC 1.0 */
			p = "table";
			break;

		case SQL_TIMEDATE_ADD_INTERVALS: /* ODBC 2.0 */
			len = 4;
			value = 0;
			break;

		case SQL_TIMEDATE_DIFF_INTERVALS: /* ODBC 2.0 */
			len = 4;
			value = 0;
			break;

		case SQL_TIMEDATE_FUNCTIONS: /* ODBC 1.0 */
			len = 4;
			value = (SQL_FN_TD_NOW);
			break;

		case SQL_TXN_CAPABLE: /* ODBC 1.0 */
			/* Postgres can deal with create or drop table statements in a transaction */
			len = 2;
			value = SQL_TC_ALL;
			break;

		case SQL_TXN_ISOLATION_OPTION: /* ODBC 1.0 */
			len = 4;
			value = SQL_TXN_READ_COMMITTED; /* SQL_TXN_SERIALIZABLE; */
			break;

		case SQL_UNION: /* ODBC 2.0 */
			/*  unions with all supported in postgres 6.3 */
			len = 4;
			value = (SQL_U_UNION | SQL_U_UNION_ALL);
			break;

		case SQL_USER_NAME: /* ODBC 1.0 */
			p = CC_get_username(conn);
			break;

		default:
			/* unrecognized key */
			conn->errormsg = "Unrecognized key passed to SQLGetInfo.";
			conn->errornumber = CONN_NOT_IMPLEMENTED_ERROR;
			CC_log_error(func, "", conn);
			return SQL_ERROR;
	
    }

	result = SQL_SUCCESS;

	mylog("SQLGetInfo: p='%s', len=%d, value=%d, cbMax=%d\n", p?p:"<NULL>", len, value, nInfoValueMax);

	/*	NOTE, that if pInfoValue is NULL, then no warnings or errors should
		result and just pnLength is returned, which indicates what length 
		would be required if a real buffer had been passed in.
	*/
	if (p)
	{  /* char/binary data */
		len = strlen(p);

		if (pInfoValue)
		{
			strncpy_null((char *)pInfoValue, p, (size_t)nInfoValueMax);

			if (len >= nInfoValueMax)
			{
				result = SQL_SUCCESS_WITH_INFO;
				conn->errornumber = STMT_TRUNCATED;
				conn->errormsg = "The buffer was too small for the result.";
			}
		}
	}
	else
	{	/* numeric data */
		
		if (pInfoValue) 
		{
		
			if (len == 2 ) 
			{
				*((WORD *)pInfoValue) = (WORD) value;
			}
			else if (len == 4)
			{
				*((DWORD *)pInfoValue) = (DWORD) value;
			}
		}
	}

	if (pnLength)
	{
		*pnLength = len;
	}

	return result;
}

