/* Module:          SQLExecDirect.c
 *
 * Description:     Executes a preparable statement, using the current values 
 *					of the parameter marker variables if any parameters exist 
 *					in the statement. 
 *
 * Classes:         
 *
 * API functions:   SQLExecDirect
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLExecDirect(
								SQLHSTMT		hDrvStmt,
								SQLCHAR			*szSqlStr,
								SQLINTEGER		nSqlStr
								)
{
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	SQLRETURN result;
	static char *func = "SQLExecDirect";

	mylog( "%s: entering...\n", func);
    
	if ( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	if (stmt->statement)
	{
		free(stmt->statement);
	}

	/* keep a copy of the un-parametrized statement, in case */
	/* they try to execute this statement again */
	stmt->statement = make_string(szSqlStr, nSqlStr, NULL);
	if ( ! stmt->statement)
	{
		stmt->errornumber = STMT_NO_MEMORY_ERROR;
		stmt->errormsg = "No memory available to store statement";
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	mylog("**** %s: hDrvStmt=%u, statement='%s'\n", func, hDrvStmt, stmt->statement);

	stmt->prepare = FALSE;

	/* If an SQLPrepare was performed prior to this, but was left in  */
	/* the premature state because an error occurred prior to SQLExecute */
	/* then set the statement to finished so it can be recycled. */
	if ( stmt->status == STMT_PREMATURE )
	{
		stmt->status = STMT_FINISHED;
	}

	stmt->statement_type = statement_type(stmt->statement);

	/*	Check if connection is onlyread (only selects are allowed) */
	if ( CC_is_onlyread(stmt->hdbc) && STMT_UPDATE(stmt))
	{
		stmt->errornumber = STMT_EXEC_ERROR;
		stmt->errormsg = "Connection is readonly, only select statements are allowed.";
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}
	
	mylog("%s: calling SQLExecute...\n", func);

	result = SQLExecute(hDrvStmt);

	mylog("%s: returned %hd from SQLExecute\n", func, result);
	return result;
}

