/*
 * citext.c
 * functions to create case insensitive data type over type text
 */

#include "postgres.h"
#include <string.h>

#include <ctype.h>

#include "mb/pg_wchar.h"
#include "miscadmin.h"
#include "access/tuptoaster.h"
#include "lib/stringinfo.h"
#include "utils/builtins.h"
#include "utils/pg_locale.h"


///////////////////////////////////////////////////////////////////////////////
// Exposed functions
extern Datum citext_eq(PG_FUNCTION_ARGS);
extern Datum citext_ne(PG_FUNCTION_ARGS);
extern Datum citext_gt(PG_FUNCTION_ARGS);
extern Datum citext_ge(PG_FUNCTION_ARGS);
extern Datum citext_lt(PG_FUNCTION_ARGS);
extern Datum citext_le(PG_FUNCTION_ARGS);
extern Datum citextcmp(PG_FUNCTION_ARGS);
extern Datum citext_smaller(PG_FUNCTION_ARGS);
extern Datum citext_larger(PG_FUNCTION_ARGS);

PG_FUNCTION_INFO_V1(citext_eq);
PG_FUNCTION_INFO_V1(citext_ne);
PG_FUNCTION_INFO_V1(citext_gt);
PG_FUNCTION_INFO_V1(citext_ge);
PG_FUNCTION_INFO_V1(citext_lt);
PG_FUNCTION_INFO_V1(citext_le);
PG_FUNCTION_INFO_V1(citextcmp);
PG_FUNCTION_INFO_V1(citext_smaller);
PG_FUNCTION_INFO_V1(citext_larger);

///////////////////////////////////////////////////////////////////////////////

/////////////////////// FORWARD DECLARATIONS////////////////////////////////////
static void cilower(char *str, int len1);
static int citext_cmp(text *arg1, text *arg2);
///////////////////////////////////////////////////////////////////////////////



///////////////////////////////////////////////////////////////////////////////

static void cilower(char *str, int len1)
{
    // The char * MUST BE A COPY!
    while (len1-- > 0)
	{
        *str = tolower((unsigned char) *str);
	    str++;
	}
}

/*
 * Comparison functions for citext strings.
 *
 * Note: btree indexes need these routines not to leak memory; therefore,
 * be careful to free working copies of toasted datums.  Most places don't
 * need to be so careful.
 */

/* citext_cmp()
 * Internal comparison function for text strings.
 * Returns -1, 0 or 1
 */
static int citext_cmp(text *arg1, text *arg2)
{
    char       *a1p, *a2p;
	int        len1, len2;

    a1p = VARDATA(arg1);
	a2p = VARDATA(arg2);

    len1 = VARSIZE(arg1) - VARHDRSZ;
	len2 = VARSIZE(arg2) - VARHDRSZ;

    cilower(a1p, len1);
    cilower(a2p, len2);

    return varstr_cmp(a1p, len1, a2p, len2);
}


Datum
citext_eq(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P(0);
	text       *arg2 = PG_GETARG_TEXT_P(1);
	bool        result;

    // fast path for different-length inputs
	if (VARSIZE(arg1) != VARSIZE(arg2))
	    result = false;
	else
    {
        if ( (Pointer)arg1 == PG_GETARG_POINTER(0) )
            arg1 = PG_GETARG_TEXT_P_COPY(0);
        if ( (Pointer)arg2 == PG_GETARG_POINTER(1) )
	        arg2 = PG_GETARG_TEXT_P_COPY(1);
        result = (citext_cmp(arg1, arg2) == 0);
    }

    PG_FREE_IF_COPY(arg1, 0);
	PG_FREE_IF_COPY(arg2, 1);

    PG_RETURN_BOOL(result);
}

Datum
citext_ne(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P(0);
	text       *arg2 = PG_GETARG_TEXT_P(1);
	bool        result;

    // fast path for different-length inputs
	if (VARSIZE(arg1) != VARSIZE(arg2))
	    result = true;
	else
    {
        if ( (Pointer)arg1 == PG_GETARG_POINTER(0) )
            arg1 = PG_GETARG_TEXT_P_COPY(0);
        if ( (Pointer)arg2 == PG_GETARG_POINTER(1) )
	        arg2 = PG_GETARG_TEXT_P_COPY(1);
        result = (citext_cmp(arg1, arg2) != 0);
    }

    PG_FREE_IF_COPY(arg1, 0);
	PG_FREE_IF_COPY(arg2, 1);

    PG_RETURN_BOOL(result);
}

Datum
citext_lt(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P_COPY(0);
	text       *arg2 = PG_GETARG_TEXT_P_COPY(1);
	bool        result;

    result = (citext_cmp(arg1, arg2) < 0);

    PG_FREE_IF_COPY(arg1, 0);
	PG_FREE_IF_COPY(arg2, 1);

    PG_RETURN_BOOL(result);
}

Datum
citext_le(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P_COPY(0);
	text       *arg2 = PG_GETARG_TEXT_P_COPY(1);
	bool        result;

    result = (citext_cmp(arg1, arg2) <= 0);

    PG_FREE_IF_COPY(arg1, 0);
	PG_FREE_IF_COPY(arg2, 1);

    PG_RETURN_BOOL(result);
}

Datum
citext_gt(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P_COPY(0);
	text       *arg2 = PG_GETARG_TEXT_P_COPY(1);
	bool        result;

    result = (citext_cmp(arg1, arg2) > 0);

    PG_FREE_IF_COPY(arg1, 0);
	PG_FREE_IF_COPY(arg2, 1);

    PG_RETURN_BOOL(result);
}

Datum
citext_ge(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P_COPY(0);
	text       *arg2 = PG_GETARG_TEXT_P_COPY(1);
	bool        result;

    result = (citext_cmp(arg1, arg2) >= 0);

    PG_FREE_IF_COPY(arg1, 0);
	PG_FREE_IF_COPY(arg2, 1);

    PG_RETURN_BOOL(result);
}

Datum
citextcmp(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P_COPY(0);
	text       *arg2 = PG_GETARG_TEXT_P_COPY(1);
	int32      result;

    result = citext_cmp(arg1, arg2);

    PG_FREE_IF_COPY(arg1, 0);
	PG_FREE_IF_COPY(arg2, 1);

    PG_RETURN_INT32(result);
}

Datum
citext_larger(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P(0);
	text       *arg2 = PG_GETARG_TEXT_P(1);
	text       *result;

    result = ((citext_cmp(arg1, arg2) > 0) ? arg1 : arg2);

    PG_RETURN_TEXT_P(result);
}

Datum
citext_smaller(PG_FUNCTION_ARGS)
{
    text       *arg1 = PG_GETARG_TEXT_P(0);
	text       *arg2 = PG_GETARG_TEXT_P(1);
	text       *result;

    result = ((citext_cmp(arg1, arg2) < 0) ? arg1 : arg2);

    PG_RETURN_TEXT_P(result);
}
