/*
 * audit_trail (table_name, backup_delete) -- log changes to another table
 *
 * written by Mark Warren (mwarren@genuity.net)
 *
 */

/*
 * includes 
 */
#include "postgres.h"
#include "fmgr.h"
#include "executor/spi.h"	/* this is what you need to work with SPI */
#include "commands/trigger.h"	/* -"- and triggers */
#include <string.h>		/* strlena () */

/*
 * declarations 
 */

typedef struct
{
  Datum user_name;		/* current_user */
  char *user_name_char;		/* in CString format */
  Datum time_stamp;		/* current_timestamp */
  char *time_stamp_char;	/* in CString format */

  int ncolumns_source;		/* no of columns in table that
				 * pulled the trigger */
  char *source_table;		/* the name of the table that
				 * pulled the trigger */
  char *audit_table;		/* the name of the audit table */
  char *deleted_table;		/* the name of the _deleted table */
  bool backup_deletes;		/* backup_deletes or no */

  char *extra_info_column;	/* name of column that holds value 
				 * to be put into extra_info */
  char *extra_info_value;

  HeapTuple old_tuple;		/* original values */
  HeapTuple new_tuple;		/* new values */
  TupleDesc tupdesc;		/* tuple descriptor */

  char *record_id;		/* the id of the record affected */
}
AuditData;

extern Datum audit_trail(PG_FUNCTION_ARGS);
static char *do_quote_ident(char *iptr);
static char *do_quote_literal(char *iptr);
static int get_no_columns(char *table_name);
static char *get_column_type(AuditData * data, char *column_name);
static char *get_column_text(AuditData * data, HeapTuple tup,
			     char *column_name);
static Datum __audit_insert(TriggerData * trigdata, AuditData * data);
static Datum __audit_delete(TriggerData * trigdata, AuditData * data);
static Datum __audit_update(TriggerData * trigdata, AuditData * data);

/*
 * this is a V1 (new) function 
 */
PG_FUNCTION_INFO_V1(audit_trail);

Datum
audit_trail(PG_FUNCTION_ARGS)
{
  TriggerData *trigdata = (TriggerData *) fcinfo->context;
  AuditData *data = NULL;

  int tmp = 0;
  int ret = -1;
  int ncolumns_audit = 0;	/* number of columns in audit table */
  int ncolumns_deleted = 0;	/* number of columns _deleted table */
  Datum update_datum = (Datum) NULL;

  /*
   * allocate memory for our audit data structure 
   */
  data = (AuditData *) palloc(sizeof(AuditData));
  if (data == (AuditData *) NULL)
  {
    elog(ERROR,
	 "audit_data (%s): failed to allocate memory for data structure",
	 SPI_getrelname(trigdata->tg_relation));
  }
  bzero(data, sizeof(AuditData));

  /*
   * initialize audit data 
   */
  data->ncolumns_source = 0;
  data->source_table = NULL;
  data->audit_table = NULL;
  data->deleted_table = NULL;
  data->backup_deletes = false;

  /***********************************************************
   * do some checks to make sure that we were called correctly
   ***********************************************************/

  /*
   * Called by trigger manager ? 
   */
  if (!CALLED_AS_TRIGGER(fcinfo))
  {
    elog(ERROR, "audit_trail (%s): not fired by trigger manager",
	 SPI_getrelname(trigdata->tg_relation));
  }

  /*
   * Should be called for ROW trigger 
   */
  if (TRIGGER_FIRED_FOR_STATEMENT(trigdata->tg_event))
  {
    elog(ERROR,
	 "audit_trail (%s): can't process STATEMENT events",
	 SPI_getrelname(trigdata->tg_relation));
  }

  /*
   * Connect to SPI manager 
   */
  if ((ret = SPI_connect()) < 0)
  {
    elog(ERROR, "audit_trail (%s): SPI_connect returned %d",
	 SPI_getrelname(trigdata->tg_relation), ret);
  }

  /****************************************************
   * find the name of the table that pulled the trigger 
   ****************************************************/
  data->source_table = SPI_getrelname(trigdata->tg_relation);

  /*******************************************************************
   * check to see if we got all our arguments, set audit struct values
   * for arguments 
   *******************************************************************/
  if (trigdata->tg_trigger->tgnargs > 1 && trigdata->tg_trigger->tgnargs < 4)
  {
    /*
     * make sure that we were given the name of the audit_trail table
     * as an argument
     */

    data->audit_table = (char *)
      palloc((strlen(trigdata->tg_trigger->tgargs[0]) + 1) * sizeof(char));
    sprintf(data->audit_table, "%s", trigdata->tg_trigger->tgargs[0]);

    /*
     * enable backup_deletes or no
     */

    if (strncmp(trigdata->tg_trigger->tgargs[1], "true", 1) == 0 ||
	strcmp(trigdata->tg_trigger->tgargs[1], "1") ||
	strncmp(trigdata->tg_trigger->tgargs[1], "yes", 1) == 0)
    {
      data->backup_deletes = true;
      tmp = strlen(data->source_table) + strlen("_deleted") + 1;
      data->deleted_table = (char *) palloc(sizeof(char) * tmp);
      snprintf(data->deleted_table, tmp, "%s_deleted", data->source_table);
    }

    /*
     * if we were given a third argument, it's the name of the column
     * in the source table to copy into the extra_info column of the audit
     * table 
     */

    if (trigdata->tg_trigger->tgnargs == 3)
    {
      data->extra_info_column = (char *)
	palloc((strlen(trigdata->tg_trigger->tgargs[2]) + 1) * sizeof(char));
      sprintf(data->extra_info_column, "%s", trigdata->tg_trigger->tgargs[2]);
    }

  }
  else
  {
    elog(ERROR,
	 "audit_trail (%s): incorrect number of arguments",
	 data->source_table);
  }

  /************************************************
   * collect some information that we'll need later
   ************************************************/

  /*
   * find the number of columns in the table that pulled the trigger 
   */
  data->ncolumns_source = get_no_columns(data->source_table);

  /*
   * find the number of columns in the deleted table 
   */
  if (data->backup_deletes)
  {
    ncolumns_deleted = get_no_columns(data->deleted_table);
    /*
     * guess as to whether or not the deleted table is formed
     * correctly, based on the number of columns
     */
    if (ncolumns_deleted != data->ncolumns_source)
    {
      elog(ERROR,
	   "audit_trail (%s): %s does not appear to be formed as %s without constraints",
	   data->source_table, data->deleted_table, data->source_table);
    }
  }

  /*
   * find the number of columns in the audit table, make a guess as to
   * whether it's formed well based on that number
   */
  ncolumns_audit = get_no_columns(data->audit_table);
  if (ncolumns_audit < 12)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s doesn't appear to be a well formed audit_trail",
	 data->source_table, data->audit_table);
  }

  /*
   * get the current user name 
   */
  data->user_name =
    DirectFunctionCall1(textin, current_user((FunctionCallInfo) NULL));
  data->user_name_char =
    DatumGetCString(DirectFunctionCall1(textout, data->user_name));

  /*
   * get the current timestamp 
   */
  data->time_stamp =
    DirectFunctionCall1(timestamptz_in, CStringGetDatum("now"));
  data->time_stamp_char =
    DatumGetCString(DirectFunctionCall1(timestamptz_out, data->time_stamp));

  /*
   * make tuples easy to access in the AuditData struct 
   */
  data->old_tuple = trigdata->tg_trigtuple;
  data->new_tuple = trigdata->tg_newtuple;
  data->tupdesc = (trigdata->tg_relation)->rd_att;

  /*
   * get the id of the record being changed 
   */
  tmp = SPI_fnumber(data->tupdesc, "id");
  if (tmp < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s has no 'id' field",
	 data->source_table, data->source_table);
  }

  data->record_id = SPI_getvalue(data->old_tuple, data->tupdesc, tmp);
  if (data->record_id == (char *) NULL)
  {
    elog(ERROR,
	 "audit_trail (%s): id field of relation %s is NULL",
	 data->source_table, data->source_table);
  }

  /*
   * get the value of the extra_info_column from the source
   * table
   */
  if (data->extra_info_column != (char *) NULL)
  {
    data->extra_info_value =
      do_quote_literal(get_column_text
		       (data, data->old_tuple, data->extra_info_column));
  }
  else
  {
    data->extra_info_value =
      (char *) palloc(sizeof(char) * (strlen("NULL") + 1));
    sprintf(data->extra_info_value, "NULL");
  }

  /***********************************************************
   * call the appropriate real function for the type of change
   ***********************************************************/

  if (TRIGGER_FIRED_BY_INSERT(trigdata->tg_event))
  {
    /*
     * trigger called from INSERT 
     */
    update_datum = __audit_insert(trigdata, data);
  }
  else if (TRIGGER_FIRED_BY_DELETE(trigdata->tg_event))
  {
    /*
     * trigger called from DELETE 
     */
    update_datum = __audit_delete(trigdata, data);
  }
  else if (TRIGGER_FIRED_BY_UPDATE(trigdata->tg_event))
  {
    /*
     * trigger called from UPDATE 
     */
    update_datum = __audit_update(trigdata, data);
  }
  else
  {
    elog(ERROR,
	 "audit_trail (%s): trigger fired by unknown event",
	 data->source_table);
  }

  SPI_finish();

  PG_RETURN_DATUM(update_datum);
}


static Datum
__audit_insert(TriggerData * trigdata, AuditData * data)
{
  int query_len = 0;
  char *query = NULL;
  int ret = -1;
  HeapTuple rettuple = NULL;
  int attnum = 0;
  Datum version;
  /*
   * ensure that we were called for an insert 
   */
  if (!TRIGGER_FIRED_BY_INSERT(trigdata->tg_event))
  {
    elog(ERROR,
	 "audit_trail (%s): __audit_insert not called for insert event",
	 data->source_table);
  }

  /*
   * Add an entry to the audit trail table logging this new record
   */

  query_len = 109;
  query_len += strlen(data->audit_table);
  query_len += strlen(data->source_table) + 2;
  query_len += strlen(data->user_name_char) + 2;
  query_len += strlen(data->time_stamp_char) + 2;
  query_len += strlen(data->record_id);
  query_len += strlen(data->extra_info_value);

  query = (char *) palloc(sizeof(char) * query_len);
  if (query == (char *) NULL)
  {
    elog(ERROR, "audit_trail (%s): failed to allocate memory",
	 data->source_table);
  }

  snprintf(query, query_len,
	   "INSERT INTO %s (change_type,table_name,object_id,modified_by,date_modified,extra_info) VALUES ('CREATE',%s,%s,%s,%s,%s)",
	   data->audit_table, do_quote_literal(data->source_table),
	   data->record_id, do_quote_literal(data->user_name_char),
	   do_quote_literal(data->time_stamp_char), data->extra_info_value);

  if ((ret = SPI_exec(query, 0)) < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): could not update audit table %s on insert into %s",
	 data->source_table, data->audit_table, data->source_table);
  }

  /*
   * free the memory allocated for the query 
   */
  pfree(query);
  /*
   * Fill in created_by and date_created of record being inserted
   */
  rettuple = data->old_tuple;
  attnum = SPI_fnumber(data->tupdesc, "created_by");
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing 'created_by' attribute",
	 data->source_table, data->source_table);
  }

  if (SPI_gettypeid(data->tupdesc, attnum) != VARCHAROID)
  {
    elog(ERROR,
	 "audit_trail (%s): attribute 'created_by' of relation %s must be of VARCHAR type",
	 data->source_table, data->source_table);
  }

  rettuple =
    SPI_modifytuple(trigdata->tg_relation, rettuple, 1, &attnum,
		    &data->user_name, NULL);
  if (rettuple == NULL)
  {
    elog(ERROR,
	 "audit_trail (%s): %d returned by SPI_modifytuple",
	 data->source_table, SPI_result);
  }

  attnum = SPI_fnumber(data->tupdesc, "date_created");
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing 'date_created' attribute",
	 data->source_table, data->source_table);
  }

  if (SPI_gettypeid(data->tupdesc, attnum) != TIMESTAMPTZOID)
  {
    elog(ERROR,
	 "audit_trail (%s): attribute 'date_created' of relation %s must of TIMESTAMPTZ type",
	 data->source_table, data->source_table);
  }

  rettuple =
    SPI_modifytuple(trigdata->tg_relation, rettuple, 1, &attnum,
		    &data->time_stamp, NULL);
  if (rettuple == NULL)
  {
    elog(ERROR,
	 "audit_trail (%s): %d returned by SPI_modifytuple",
	 data->source_table, SPI_result);
  }

  attnum = SPI_fnumber(data->tupdesc, "version");
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing 'version' attribute",
	 data->source_table, data->source_table);
  }

  if (SPI_gettypeid(data->tupdesc, attnum) != INT4OID)
  {
    elog(ERROR,
	 "audit_trail (%s): attribute 'version' of relation %s must be of INT4 type",
	 data->source_table, data->source_table);
  }

  version = Int32GetDatum(0);
  rettuple =
    SPI_modifytuple(trigdata->tg_relation, rettuple, 1, &attnum,
		    &version, NULL);
  if (rettuple == NULL)
  {
    elog(ERROR,
	 "audit_trail (%s): %d returned by SPI_modifytuple",
	 data->source_table, SPI_result);
  }

  return (PointerGetDatum(rettuple));
}


static Datum
__audit_delete(TriggerData * trigdata, AuditData * data)
{
  int query_len = 0;
  char *query = NULL;
  int ret = -1;
  char *ctmp = NULL;
  char *before_char = NULL;
  int i = 0;
  /*
   * ensure that we were called for an insert 
   */
  if (!TRIGGER_FIRED_BY_DELETE(trigdata->tg_event))
  {
    elog(ERROR,
	 "audit_trail (%s): __audit_delete not called for delete event",
	 data->source_table);
  }

  /*
   * Add an entry to the audit trail table logging this deletion
   */

  /*
   * add the length of the audit trail table name to our query size 
   */
  query_len = 109;
  query_len += strlen(data->audit_table);
  query_len += strlen(data->source_table) + 2;
  query_len += strlen(data->user_name_char) + 2;
  query_len += strlen(data->time_stamp_char) + 2;
  query_len += strlen(data->record_id);
  query_len += strlen(data->extra_info_value);
  query = (char *) palloc(sizeof(char) * query_len);
  if (query == (char *) NULL)
  {
    elog(ERROR, "audit_trail (%s): failed to allocate memory",
	 data->source_table);
  }

  snprintf(query, query_len,
	   "INSERT INTO %s (change_type,table_name,object_id,modified_by,date_modified,extra_info) VALUES ('DELETE',%s,%s,%s,%s,%s)",
	   data->audit_table, do_quote_literal(data->source_table),
	   data->record_id, do_quote_literal(data->user_name_char),
	   do_quote_literal(data->time_stamp_char), data->extra_info_value);
  if ((ret = SPI_exec(query, 0)) < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): could not update audit table %s on delete from %s",
	 data->source_table, data->audit_table, data->source_table);
  }

  /*
   * free the memory allocated for the query 
   */
  pfree(query);
  /*
   * Copy the record to the _deleted table if backup_deletes is enabled
   */
  /*
   * restart query_len at a base of 35 characters ('INSERT INTO _deleted 
   * () VALUES ()') 
   */
  query_len = 35;
  /*
   * add the size of the _deleted table base name 
   */
  query_len += strlen(data->source_table);
  /*
   * calculate the sizes for the columns 
   */
  for (i = 1; i <= data->ncolumns_source; i++)
  {
    /*
     * the columns name 
     */
    query_len += strlen(do_quote_ident(SPI_fname(data->tupdesc, i)));
    /*
     * the column value 
     */
    before_char = SPI_getvalue(data->old_tuple, data->tupdesc, i);
    /*
     * size + 1 byte for ',' 
     */
    if (before_char == NULL)
    {
      query_len += 5;
    }
    else
    {
      query_len += strlen(do_quote_literal(before_char));
    }
  }

  /*
   * allocate memory 
   */
  query = (char *) palloc(sizeof(char) * query_len);
  ctmp = query;
  /*
   * build the query 
   */
  sprintf(query, "INSERT INTO %s_deleted (", data->source_table);
  /*
   * move our progress pointer to the end of our addition 
   */
  query = ctmp + strlen(ctmp);
  /*
   * add the column names 
   */
  for (i = 1; i <= data->ncolumns_source; i++)
  {
    sprintf(query, "%s,", do_quote_ident(SPI_fname(data->tupdesc, i)));
    query = ctmp + strlen(ctmp);
  }

  query = ctmp + strlen(ctmp) - 1;
  sprintf(query, ") VALUES (");
  query = ctmp + strlen(ctmp);
  /*
   * add values 
   */
  for (i = 1; i <= data->ncolumns_source; i++)
  {
    before_char = SPI_getvalue(data->old_tuple, data->tupdesc, i);
    if (before_char == NULL)
    {
      sprintf(query, "NULL,");
    }
    else
    {
      sprintf(query, "%s,", do_quote_literal(before_char));
    }
    query = ctmp + strlen(ctmp);
  }

  query = ctmp + strlen(ctmp) - 1;
  sprintf(query, ")");
  /*
   * make query point at the beginning of the char array 
   */
  query = ctmp;
  /*
   * execute the query 
   */
  if ((ret = SPI_exec(query, 0)) < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): could not backup record to %s_deleted table",
	 data->source_table, data->source_table);
  }

  pfree(query);
  return (PointerGetDatum(trigdata->tg_trigtuple));
}


static Datum
__audit_update(TriggerData * trigdata, AuditData * data)
{
  int attnum = 0;
  bool isnull;
  char *tmp = NULL;
  int i = 0;
  char *old_value = NULL;
  char *new_value = NULL;
  char *column_type = NULL;
  char *column_name = NULL;
  Datum old_version;
  Datum new_version;
  int query_len = 0;
  char *query = NULL;
  int ret = -1;
  HeapTuple rettuple = NULL;

  attnum = SPI_fnumber(data->tupdesc, "version");
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing 'version' attribute",
	 data->source_table, data->source_table);
  }

  old_version =
    SPI_getbinval(data->old_tuple, data->tupdesc, attnum, &isnull);
  if (isnull)
  {
    elog(ERROR,
	 "audit_trail (%s): version of record id %s is NULL, this should not be",
	 data->source_table, data->record_id);
  }

  for (i = 1; i <= data->ncolumns_source; i++)
  {
    /*
     * get the old and new values for the column
     */
    tmp = SPI_getvalue(data->old_tuple, data->tupdesc, i);
    if (tmp == (char *) NULL)
    {
      old_value = (char *) palloc(sizeof(char) * (strlen("NULL") + 1));
      sprintf(old_value, "NULL");
    }
    else
    {
      old_value = do_quote_literal(tmp);
    }

    tmp = SPI_getvalue(data->new_tuple, data->tupdesc, i);
    if (tmp == (char *) NULL)
    {
      new_value = (char *) palloc(sizeof(char) * (strlen("NULL") + 1));
      sprintf(new_value, "NULL");
    }
    else
    {
      new_value = do_quote_literal(tmp);
    }

    /*
     * if they differ, update the audit trail
     */
    if (strcmp(old_value, new_value) != 0)
    {
      /*
       * get the column name for this column and it's type
       */

      column_name = SPI_fname(data->tupdesc, i);
      if (column_name == (char *) NULL)
      {
	elog(ERROR,
	     "audit_trail (%s): attribute name for attribute number %d is NULL",
	     data->source_table, i);
      }

      column_type = get_column_type(data, column_name);
      if (column_type == (char *) NULL)
      {
	elog(ERROR,
	     "audit_trail (%s): attribute type for attribute %s is NULL",
	     data->source_table, column_name);
      }

      /***************************************************
       * put an entry into the audit table for this change
       ***************************************************/
      query_len = 174;
      query_len += strlen(data->audit_table);
      query_len += strlen(data->source_table) + 2;
      query_len += strlen(data->record_id);
      query_len += strlen(data->user_name_char) + 2;
      query_len += strlen(data->time_stamp_char) + 2;
      query_len += strlen(data->extra_info_value);
      query_len += strlen(column_name) + 2;

      query_len +=
	strlen(DatumGetCString(DirectFunctionCall1(int4out, old_version)));

      query_len += strlen(column_type) + 2;
      query_len += strlen(old_value);
      query_len += strlen(new_value);

      query = (char *) palloc(sizeof(char) * query_len);
      if (query == (char *) NULL)
      {
	elog(ERROR, "audit_trail (%s): failed to allocate memory",
	     data->source_table);
      }

      snprintf(query, query_len,
	       "INSERT INTO %s (change_type,table_name,object_id,modified_by,date_modified,extra_info,column_name,was_from_version,value_type,old_value,new_value) VALUES ('UPDATE',%s,%s,%s,%s,%s,%s,%d,%s,%s,%s)",
	       data->audit_table, do_quote_literal(data->source_table),
	       data->record_id, do_quote_literal(data->user_name_char),
	       do_quote_literal(data->time_stamp_char),
	       data->extra_info_value, do_quote_literal(column_name),
	       DatumGetInt32(old_version), do_quote_literal(column_type),
	       old_value, new_value);

      if ((ret = SPI_exec(query, 0)) < 0)
      {
	elog(ERROR,
	     "audit_trail (%s): could not update audit table %s on delete from %s",
	     data->source_table, data->audit_table, data->source_table);
      }

      /*
       * free the memory allocated for the query
       */
      pfree(query);
      query = (char *) NULL;
    }
  }

  /*
   * Fill in created_by and date_created of record being inserted
   */

  rettuple = data->new_tuple;
  attnum = SPI_fnumber(data->tupdesc, "modified_by");
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing 'modified_by' attribute",
	 data->source_table, data->source_table);
  }

  if (SPI_gettypeid(data->tupdesc, attnum) != VARCHAROID)
  {
    elog(ERROR,
	 "audit_trail (%s): attribute 'modified_by' of relation %s must be of VARCHAR type",
	 data->source_table, data->source_table);
  }

  rettuple =
    SPI_modifytuple(trigdata->tg_relation, rettuple, 1, &attnum,
		    &data->user_name, NULL);
  if (rettuple == NULL)
  {
    elog(ERROR,
	 "audit_trail (%s): %d returned by SPI_modifytuple",
	 data->source_table, SPI_result);
  }

  attnum = SPI_fnumber(data->tupdesc, "date_modified");
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing 'date_modified' attribute",
	 data->source_table, data->source_table);
  }

  if (SPI_gettypeid(data->tupdesc, attnum) != TIMESTAMPTZOID)
  {
    elog(ERROR,
	 "audit_trail (%s): attribute 'date_modified' of relation %s must of TIMESTAMPTZ type",
	 data->source_table, data->source_table);
  }

  rettuple =
    SPI_modifytuple(trigdata->tg_relation, rettuple, 1, &attnum,
		    &data->time_stamp, NULL);
  if (rettuple == NULL)
  {
    elog(ERROR,
	 "audit_trail (%s): %d returned by SPI_modifytuple",
	 data->source_table, SPI_result);
  }

  attnum = SPI_fnumber(data->tupdesc, "version");
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing 'version' attribute",
	 data->source_table, data->source_table);
  }

  if (SPI_gettypeid(data->tupdesc, attnum) != INT4OID)
  {
    elog(ERROR,
	 "audit_trail (%s): attribute 'version' of relation %s must be of INT4 type",
	 data->source_table, data->source_table);
  }

  new_version = Int32GetDatum(DatumGetInt32(old_version) + 1);
  rettuple =
    SPI_modifytuple(trigdata->tg_relation, rettuple, 1, &attnum,
		    &new_version, NULL);
  if (rettuple == NULL)
  {
    elog(ERROR,
	 "audit_trail (%s): %d returned by SPI_modifytuple",
	 data->source_table, SPI_result);
  }

  return (PointerGetDatum(rettuple));
}


static int
get_no_columns(char *table_name)
{
  char query[250];
  int ret;
  int ncolumns = -1;
  /*
   * build the query string 
   */
  snprintf(query, 249,
	   "SELECT COUNT(pg_attribute.attname) AS a FROM pg_class, pg_attribute WHERE pg_class.relname=%s AND pg_attribute.attnum > 0 AND pg_attribute.attrelid=pg_class.oid",
	   do_quote_literal(table_name));
  /*
   * execute the query 
   */
  if ((ret = SPI_exec(query, 0)) < 0)
  {
    elog(ERROR,
	 "audit_trail: could not get number of columns from relation %s",
	 table_name);
  }

  /*
   * get the query result 
   */
  if (SPI_processed > 0)
  {
    /*
     * ->vals array contains rows of data, the third argument to
     * SPI_getvalue is the column number 
     */
    /*
     * columns numbering starts at 1 
     */
    ncolumns =
      DatumGetInt32(DirectFunctionCall1
		    (int4in,
		     CStringGetDatum(SPI_getvalue
				     (SPI_tuptable->
				      vals[0], SPI_tuptable->tupdesc, 1))));
    if (ncolumns < 1)
    {
      elog(ERROR, "audit_trail: relation %s does not exist", table_name);
    }
  }
  else
  {
    elog(ERROR,
	 "audit_trail: could not get number columns in relation %s",
	 table_name);
  }

  return (ncolumns);
}

static char *
get_column_type(AuditData * data, char *column_name)
{
  int attnum = 0;
  char *str = NULL;
  attnum = SPI_fnumber(data->tupdesc, column_name);
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing '%s' attribute",
	 data->source_table, data->source_table, column_name);
  }

  str = SPI_gettype(data->tupdesc, attnum);
  return (str);
}

static char *
get_column_text(AuditData * data, HeapTuple tup, char *column_name)
{
  int attnum = 0;
  char *str = NULL;
  attnum = SPI_fnumber(data->tupdesc, column_name);
  if (attnum < 0)
  {
    elog(ERROR,
	 "audit_trail (%s): relation %s is missing '%s' attribute",
	 data->source_table, data->source_table, column_name);
  }

  str = SPI_getvalue(tup, data->tupdesc, attnum);
  return (str);
}

/*
 * MULTIBYTE dependant internal functions follow
 *
 */
/*
 * from src/backend/utils/adt/quote.c and slightly modified 
 */

#ifndef MULTIBYTE

/*
 * Return a properly quoted identifier 
 */
static char *
do_quote_ident(char *iptr)
{
  char *result;
  char *result_return;
  char *cp1;
  char *cp2;
  int len;
  len = strlen(iptr);
  result = (char *) palloc(len * 2 + 3);
  result_return = result;
  cp1 = VARDATA(iptr);
  cp2 = VARDATA(result);
  *result++ = '"';
  while (len-- > 0)
  {
    if (*iptr == '"')
    {
      *result++ = '"';
    }
    if (*iptr == '\\')
    {
      /*
       * just add a backslash, the ' will be follow 
       */
      *result++ = '\\';
    }
    *result++ = *iptr++;
  }
  *result++ = '"';
  *result++ = '\0';
  return result_return;
}

/*
 * Return a properly quoted literal value 
 */
static char *
do_quote_literal(char *lptr)
{
  char *result;
  char *result_return;
  int len;
  len = strlen(lptr);
  result = (char *) palloc(len * 2 + 3);
  result_return = result;
  *result++ = '\'';
  while (len-- > 0)
  {
    if (*lptr == '\'')
    {
      *result++ = '\\';
    }
    if (*lptr == '\\')
    {
      /*
       * just add a backslash, the ' will be follow 
       */
      *result++ = '\\';
    }
    *result++ = *lptr++;
  }
  *result++ = '\'';
  *result++ = '\0';
  return result_return;
}

#else

/*
 * Return a properly quoted identifier (MULTIBYTE version) 
 */
static char *
do_quote_ident(char *iptr)
{
  char *result;
  char *result_return;
  int len;
  int wl;
  len = strlen(iptr);
  result = (char *) palloc(len * 2 + 3);
  result_return = result;
  *result++ = '"';
  while (len > 0)
  {
    if ((wl = pg_mblen(iptr)) != 1)
    {
      len -= wl;
      while (wl-- > 0)
      {
	*result++ = *iptr++;
      }
      continue;
    }

    if (*iptr == '"')
    {
      *result++ = '"';
    }
    if (*iptr == '\\')
    {
      /*
       * just add a backslash, the ' will be follow 
       */
      *result++ = '\\';
    }
    *result++ = *iptr++;
    len--;
  }
  *result++ = '"';
  *result++ = '\0';
  return result_return;
}

/*
 * Return a properly quoted literal value (MULTIBYTE version) 
 */
static char *
do_quote_literal(char *lptr)
{
  char *result;
  char *result_return;
  int len;
  int wl;
  len = strlen(lptr);
  result = (char *) palloc(len * 2 + 3);
  result_return = result;
  *result++ = '\'';
  while (len > 0)
  {
    if ((wl = pg_mblen(lptr)) != 1)
    {
      len -= wl;
      while (wl-- > 0)
      {
	*result++ = *lptr++;
      }
      continue;
    }

    if (*lptr == '\'')
    {
      *result++ = '\\';
    }
    if (*lptr == '\\')
    {
      /*
       * just add a backslash, the ' will be follow 
       */
      *result++ = '\\';
    }
    *result++ = *lptr++;
    len--;
  }
  *result++ = '\'';
  *result++ = '\0';
  return result_return;
}

#endif
