/*----------------------------------------------------------------------------
//
//   Class/module:     xmubCustUtil
//
//   Project:          xmub
//   System:           <>
//    Subsystem:       <>
//    Function block:  <>
//
//   Creation date:    05.11.1994 

// = LIBRARY
//    xmub
//
// = FILENAME
//    xmubCustUtil.h
//
// = AUTHORS
//    Ulrika Bornetun <bre@ubszh.ch.net>
//
// = COPYRIGHT
//    (C) Copyright Union Bank of Switzerland 1994
//        All rights reserved

//  Permission to use, copy, modify, and distribute this software and its
//  documentation for any purpose and without fee is hereby granted,
//  provided that the above copyright notice appear in all copies.
//  Union Bank of Switzerland makes no representations about the usability
//  of this software for any purpose. It is provided "as is" without express
//  or implied warranty.
//
//   Modifications:
//
//   Version / When / Who
//   What
//   --------------------------------------------
//   %StartLog%
//   %EndLog%
//
//---------------------------------------------------------------------------*/

/* SCCS module identifier. */
/* SCCSID = @(#) Module: xmubCustUtil.h, Version: 1.1, Date: 95/02/18 15:10:54 */

#ifndef _INC_XMUB_CUST_UTIL_H
#  define _INC_XMUB_CUST_UTIL_H


/*----------------------------------------------------------------------------
//  Include files
----------------------------------------------------------------------------*/

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>

/*----------------------------------------------------------------------------
//  Macro definitions
----------------------------------------------------------------------------*/

#define XmUbNsubstringResources   "substringResources"
#define XmUbNsubstringDelimiters  "substringDelimiters"

#define XmUbCSubstringResources   "SubstringResources"
#define XmUbCSubstringDelimiters  "SubstringDelimiters"


#define XMUB_RES_SET_CHANGED   1

/* Names and classes used in the default definitions. */

#define FONTMAP_RESOURCE_NAME  "fmap"
#define FONTMAP_CLASS_NAME     "Fmap"
#define PALETTE_RESOURCE_NAME  "palette"
#define PALETTE_CLASS_NAME     "Palette"

/*----------------------------------------------------------------------------
//  Type declarations
----------------------------------------------------------------------------*/

typedef struct 
{
  char*  res_name;
  char*  res_class;
} xmubAppResDesc, *xmubAppResRef;


typedef struct _xmubResourceMappingInfo  *xmubResourceMappingInfoRef;

/*
  A pointer to a  struct of the type xmubResSetChangeCallbackStruct is sent
  as call_data to the resource set change callback. Its fields are:

  reason           XMUB_RES_SET_CHANGED
  res_name         The name of the resource that was changed.
                   May be NULL.
  res_class        The class of the resource that was changed.
                   May be NULL.
  new_value        The new value of the resource.
                   May be NULL.
  mapping_changed  True if the changed lead to a real resource replacement,
                   False otherwise.

  Note that the strings point to internal parameters and may not be changed
  or deallocated.
*/

typedef struct 
{
  int      reason;

  char*    res_name;
  char*    res_class;
  char*    new_value;

  Boolean  mapping_changed;
  
} xmubResSetChangeCallbackStruct;


typedef void (*xmubResSetChangeCallback)
  ( xmubResourceMappingInfoRef, XtPointer, xmubResSetChangeCallbackStruct* );

typedef struct 
{
  xmubResSetChangeCallback  callback;
  XtPointer                 client_data;

} xmubResSetChangeCallbackRec, *xmubResSetChangeCallbackList;


typedef struct _xmubResourceMappingInfo
{
  XrmDatabase  orig_db;
  XrmDatabase  mapping_db;
  XrmDatabase  translation_db;
  XrmDatabase  merge_db;
  
  char*        app_name;
  char*        app_class;

  Display*     display;

  /* Internal information. */
  XrmQuark     string_quark;

  /* Callbacks. */
  xmubResSetChangeCallbackList  set_change_callbacks;
  Cardinal                      num_set_change_callbacks;
  
  /* Temporary fields. */
  Boolean      specific;

  XrmQuark     *selection;
  int          num_selections;
  Boolean      inclusive_selection;

  Boolean      replace_substrings;
  char*        substring_delimiters;
  char*        value;
  Boolean      replaced;
  Boolean      selected;
  
  Boolean      save_in_orig;
  
} xmubResourceMappingInfo;

/*----------------------------------------------------------------------------
//  Global definitions
----------------------------------------------------------------------------*/

/*
  The xmub_res_options array can be used in XtOpenDisplay to assign the
  default options to the default resources.
  The mappings that are done are "-palette" to "*palette" and
  "-fmap" to "*fmap".
*/

extern XrmOptionDescRec  xmub_options[];
extern Cardinal          xmub_num_options;

/*
  The xmub_resource_sets array is a default for the resources that define
  resource sets. The supported resource sets are
  "palette", "Palette" and "fmap", "Fmap".
  This array may be used in the function xmubResAddMapFile.
*/

extern xmubAppResDesc  xmub_resource_sets[];
extern Cardinal        xmub_num_resource_sets;


/*----------------------------------------------------------------------------
//  Function prototypes
----------------------------------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#endif

/*  For man-page generation: class xmubCustUtil  */
/*
// = TITLE
//    A library for resource mapping.
//	   
// = DESCRIPTION
//    This module contains functions for manipulating the mapping of
//    resources.
//
// = SEE ALSO
//	  xmubCustUi
*/

/*  For man-page generation: public: */
/*
// = INITIALIZATION
*/

Widget
xmubAppInitialize( XtAppContext                *app_context_return,
                   String                      application_class,
                   XrmOptionDescList           options,
                   Cardinal                    num_options,
                   Cardinal                    *argc_in_out,
                   String                      *argv_in_out,
                   String                      *fallback_resources,
                   ArgList                     args, 
                   Cardinal                    num_args,
                   xmubAppResDesc              *resource_sets,
                   Cardinal                    num_resource_sets,
                   xmubResourceMappingInfoRef  *ref_return );
/*
// Performs the function of XtAppInitialize, plus initalizes
// the resource mapping facility, reads the default mapping
// file and performs the replacements in it.
//
// The resource_sets parameter defines the resource sets to be used.
// See function xmubResAddMapFile for an explanation.
// An info record is allocated and the pointer to it is returned in
// ref_return. It must be freed by the caller with the function
// xmubResFreeInfo.
//
// The return value is the created shell widget.
//
// The resources argc and argv are added to the resources for the created
// shell.
*/

void
xmubInitResourceMapping( Display*                    display,
                         xmubAppResDesc              resource_sets[],
                         Cardinal                    num_resource_sets,
                         xmubResourceMappingInfoRef  *res_info );
/*
// Initializes the resource mapping facility, reads the default mapping
// file and performs the replacements in it.
// The resource_sets parameter defines the resource sets to be used.
// See function xmubResAddMapFile for an explanation.
//
// If res_info is NULL, an info record is returned. It must
// be freed by the caller with the function xmubResFreeInfo.
*/

xmubResourceMappingInfoRef
xmubResInitialize( Display*  display );
/*
// Sets up an info record that will be used in further calls and returns
// a reference to it. The database entries are set to NULL.
//
// The info record is allocated in the function and must be freed by
// the caller. Note that the info record should be freed with the function
// xmubResFreeInfo, and not until all calls to this module have been done.
*/

char*
xmubResGetMappingFileName( Display*  display );
/*
// Returns the name of the file that contains the mapping database.
// The returned string must be deallocated with XtFree when no longer
// needed. If no file is found, NULL is returned.
//
// If the environment variable XRESOURCE_MAP_FILE is defined and points
// to a readable file, its translation is used as the file name.
// Otherwise, a map file with the name 'app_class'.map is looked for
// first in XUSERFILESEARCHPATH with the type set to app-defaults,
// then in the path
//
//   $XAPPLRESDIR/%L/%N/%S:$XAPPLRESDIR/%l/%N%S:$XAPPLRESDIR/%N%S:
//   $HOME/%N%S:.
//
// if XAPPLRESDIR is defined, and in
//
//   $HOME/%L/%N%S:$HOME/%l%N%S:$HOME/%N%S:.
//
// if it is not defined.
//
// Lastly, the system default path is searched.
// If no file with the name 'app_class'.map is found, the same search is
// performed for the file "xresource.map".
// If $HOME is not defined, the parts including it are left out.
//
// For VMS, there is a special version of this function. It uses the
// logical name XRESOURCE_MAP_FILE as the file name if defined.
// Otherwise, it searches first for the file 'app_class'.map and if
// not found, for the file "xresource.map" in the path
//
//   DECW$USER_DEFAULTS, DECW$SYSTEM_DEFAULTS, []
*/

/*
// = ADDING MAPPINGS
*/
Boolean
xmubResAddMapFile( xmubResourceMappingInfoRef  info,
                   char*                       file_name,
                   Boolean                     add_defaults,
                   xmubAppResDesc              resource_sets[],
                   Cardinal                    num_resource_sets );
/*
// Adds mappings from the resource file indicated with file_name to
// the mapping database in info.
// If file_name is NULL, the default file as returned by
// xmubResGetMappingFileName is used.
//
// resource_sets is an array of records that define the resources
// that are used to store valid set names. The records define the
// name and class of the respective resources. If NULL is defined,
// a default array with the following definition is taken:
//   { { "palette", "Palette" }, { "fmap", "Fmap" } }
//
// If resource sets are defined, the function will extract the values
// for these resource sets and use these as the selection criteria
// when reading from the file. If no resource sets are defined, only
// mappings defined with '*', i.e. without a resource set name, will
// be used from the mapping file. Note that these general resources will
// only be added if add_defaults is True, and will be overridden by the
// selected sets.
//
// add_defaults should be True when initializing the resource mapping,
// and should only be set False when an update is made.
//
// False is returned if the specified file was not readable, or if there
// were no resources selected in the file.
// True is returned if everything was OK.
//
// The file database is merged into the mapping database.
*/

Boolean
xmubResAddMappings( xmubResourceMappingInfoRef  info,
                    XrmDatabase                 db,
                    Boolean                     add_defaults,
                    xmubAppResDesc              resource_sets[],
                    Cardinal                    num_resource_sets );
/*
// Adds mappings from the database to the translation database.
//
// resource_sets is an array of records that define the resources
// that are used to store valid set names. The records define the
// name and class of the respective resources. If NULL is defined,
// a default array with the following definition is taken:
//   { { "palette", "Palette" }, { "fmap", "Fmap" } }
//
// If resource sets are defined, the function will extract the values
// for these resource sets and use these as the selection criteria
// when reading from the database. If no resource sets are defined, only
// mappings defined with '*', i.e. without a resource set name, will
// be used from the database. Note that these general resources will
// only be added if add_defaults is True, and will be overridden by the
// selected sets.
//
// add_defaults should be True when initializing the resource mapping,
// and should only be set False when an update is made.
//
// False is returned if no resources were selected.
// True is returned if everything was OK.
//
// The mapping database is not modified.
*/

void
xmubResAddStringMapping( xmubResourceMappingInfoRef  info,
                         char*                       keyword,
                         char*                       translation );
/*
// Adds the translation *'keyword': 'translation' to both the mapping
// and the translation databases.
*/

/*
// = REPLACING KEYWORDS
*/
void
xmubResReplace( xmubResourceMappingInfoRef  info,
                XrmDatabase                 start_db );
/*
// Takes start_db and replaces keywords defined info -\> translation_db with
// their respective values. The result is placed in info -\> merge_db.
// Note that only the modified resources are placed in merge_db.
// If start_db is NULL, the display database is taken.
//   
// Two passes are run. The first pass replaces only whole definitions,
// and no substrings. If the application resource substringResources is
// defined, a second pass is run. This pass takes the resources defined
// in substringResources and runs a substring replacement for them.
// The resource substringDelimiters is used for determining the character
// set used to delimit the substrings. The default for this resource is
// " ".
//
// Unless there are very special requirements, this function should be
// used instead of xmubResReplaceKeywords, since it makes it possible
// to define the substring set from the outside. This function calls
// xmubResReplaceKeywords internally.
*/

void
xmubResReplaceKeywords( xmubResourceMappingInfoRef  info,
                        XrmDatabase                 start_db,
                        Boolean                     replace_substrings,
                        char*                       substring_delimiters,
                        char**                      resource_selection,
                        Cardinal                    num_resource_selection,
                        Boolean                     inclusive_selection );
/*
// Takes start_db and replaces keywords defined info -\> translation_db with
// their respective values. The result is placed in info -\> merge_db.
// Note that only the modified resources are placed in merge_db.
// If start_db is NULL, the display database is taken.
//
// If replace_substrings is True, parts of the resource definitions may be
// replaced. substring_delimiters may be used to define a set of characters
// that are recognized as valid delimiters. For example, if
// replace_substrings is True and substring_delimiters is "(),", and
// the keyword UserFont has been defined, the occurrence in resource value
// "Ergo15,UserFont,Rom14" would be replaced, but not the one in the value
// "Select a new UserFont".
//
// The parameter resource_selection can be used to define an array of
// strings with resource names that should be replaced or ignored.
//
// If inclusive_selection is True, the resources in the set are the ones
// that are included, and all others are ignored. If inclusive_selection
// is False, all resources EXCEPT the ones in resource_selection are
// processed. resource_selection should contain resource names or classes
// as entries, e.g. "background","layout","labelString". The size of the
// array is defined in num_resource_selection.
//
// The string definitions that are replaced are stored in the internal
// database info -\> orig_db. This is only done if the start_db is not
// already orig_db.
//
// Note:
//
// Since replacing keywords in substrings is very inefficient, slow and
// rarely needed, it should be used with care.
// The most efficient way to code the program is to perform the
// general replacements first, then call the function again for the
// substring replacement, but this time with a resource_selection that
// matches the resources that will be processed.
//
// Example:
//
// First call to xmubResReplaceKeywords specifies no resource_selection
// and sets replace_substrings False.
// Second call to xmubResReplaceKeywords sets replace_substrings True,
// and specifies the resource set "fontList","layout".
//
// Note 2:
//
// The function xmubResReplace offers a simpler interface and more
// customization possibilities for the user and should be used
// instead of this function whenever possible.
*/


/*
// = MERGING RESOURCES
*/
void
xmubResMergeMappedResources( xmubResourceMappingInfoRef  info,
                             XrmDatabase                 target_db );
/*
// Merges the resources in info -\> merge_db into target_db.
// If target_db is NULL, the display database is used.
*/

/*
// = UPDATING RESOURCES
*/
Boolean
xmubResUpdateResourceSet( xmubResourceMappingInfoRef  info,
                          char*                       res_name,
                          char*                       res_class,
                          char*                       new_value,
                          xmubResSetChangeCallback    apply_callback,
                          XtPointer                   client_data );
/*
// Set the resource *'res_name' in the display database to new_value
// and updates the translation database using the default functions.
// The mapping database is used for selecting the values.
// If more advanced replacements than what is done in xmubResReplace
// must be made, this function is not sufficient.
// It is, however, enough for most applications.
//
// True is returned if a replacement was actually done, False if no
// replacement was done.
//
// The resource set changed callbacks are called just before the
// function returns. The mapping_changed field in the callback
// structure is set to the return value of the function.
//
// If an apply_callback is set, it is called just before the resource
// set change callbacks are called. It is called with the same callback
// structure as the resource change callbacks.
//
// Note that the default values are not selected. This leads to that
// nothing is done if new_value is empty. Also, parts of the resource
// sets that fall back to the default will not be changed.
//
// For a full update, set the affected resource values and then call
// xmubUpdateAllResources.
//
// Example:
//
// To change to the color set "Arizona", using the default values, call
// this function with the parameters
//   ( info, "palette", "Palette", "Arizona" )
//
// After this call, you may loop through your widgets and update the color,
// if desired.
*/

Boolean
xmubResUpdateAllResources( xmubResourceMappingInfoRef  info,
                           xmubAppResDesc              resource_sets[],
                           Cardinal                    num_resource_sets,
                           xmubResSetChangeCallback    apply_callback,
                           XtPointer                   client_data );
/*
// Clears the translation database and refreshes all resources according
// to xmubResReplace from the mapping_db.
// The resources are then merged into the translation db.
// The original strings are taken from orig_db.
//
// If an apply_callback is set, it is called just before the resource
// set change callbacks are called. It is called with the same callback
// structure as the resource change callbacks.
// Note that it is called with res_name, res_class and new_value set
// to NULL.
//
// The resource change callbacks are called for each resource, even
// though they may not have been changed.
*/

/*
// = LISTING RESOURCE CLASSES
*/
char**
xmubResListClasses( xmubResourceMappingInfoRef  info,
                    char*                       keyword,
                    Boolean                     sorted,
                    XrmDatabase                 db );
/*
// Runs through the database and returns a list of all resource sets
// that have the keyword defined. The list is guaranteed not to contain
// duplicates. If sorted is set, the strings are sorted in ascending order
// with strcoll(). If db is NULL, info -\> mapping_db is taken.
//
// The returned array is null-terminated. The strings and the array
// have been allocated in the function and must be freed by the caller
// using XtFree(). If no resource set containing the keyword is found,
// NULL is returned.
*/

/*
// = CALLBACKS
*/
void
xmubResAddSetChangedCallback( xmubResourceMappingInfoRef  info,
                              xmubResSetChangeCallback    callback,
                              XtPointer                   client_data );
/*
// Adds the callback and its client_data to the internal list of
// callbacks for the resource set change.
*/

void
xmubResRemoveSetChangedCallback( xmubResourceMappingInfoRef  info,
                                 xmubResSetChangeCallback    callback,
                                 XtPointer                   client_data );
/*
// Removes the callback.
*/


/*
// = FREEING STORAGE
*/
void
xmubResFreeInfo( xmubResourceMappingInfoRef  info );
/*
// Frees the information pointed to by info.
*/

void
xmubResClearInternalDatabases(
  xmubResourceMappingInfoRef  info,
  Boolean                     clear_orig_db,
  Boolean                     clear_mapping_db,
  Boolean                     clear_translation_db,
  Boolean                     clear_merge_db );
/*
// Clears the internal databases selectively.
*/

#ifdef __cplusplus
}
#endif

#endif
