/*----------------------------------------------------------------------------
--
--  Module:           xitTune
--
--  Project:          xit  - X Internal Toolkit
--  System:           xit  - X Internal Toolkit
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    xitTune can play beeps, chirps or musical tunes through the speaker. 
--    xitTune usues a facility of X Windows to make sounds.
--
--  Filename:         xitTestTune.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-04-17
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "&Z% Module: xitTestTune.c, Version: 1.1, Date: 95/02/18 15:10:50";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include "System.h"

/*
#include "xitTune.h"
*/


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define LONGEST_TUNE  10000
#define MAX_NOTES     4000

/* Sleep overhead in milliseconds -- ignore sleeps shorter than this. */
#define SLEEPY   31


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Music generation parameters */
typedef struct {
  Display         *display;
  XKeyboardState  kbd_settings;
  XtAppContext    context;
  int             curr_tune;
  int             notes;
  int             note[ MAX_NOTES ][ 3 ];
} TUNE_REC, *TUNE_REC_REF;
 

/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

static float factor[ 13 ] = {
  0.59460355750136, /*  3  C  */
  0.62996052494744, /*  4  C# */
  0.66741992708502, /*  5  D  */
  0.70710678118655, /*  6  D# */
  0.74915353843834, /*  7  E  */
  0.79370052598410, /*  8  F  */
  0.84089641525371, /*  9  F# */
  0.89089871814034, /* 10  G  */
  0.94387431268169, /* 11  G# */
  1.00000000000000, /*  0  A  */
  1.05946309435930, /*  1  A# */
  1.12246204830940, /*  2  B  */
  0.00000000000000  /* pause  */
};
 
/* Tune parameters */
static char  *scale = "ccddeffggaab_=";

/* Milliseconds pause between base notes. */
static int  speed[ 23 ] = {
  4096, /*  0 */
  2896, /*  1 */
  2048, /*  2 */
  1448, /*  3 */
  1024, /*  4 */
   724, /*  5 */
   512, /*  6 */
   362, /*  7 */
   256, /*  8 */
   181, /*  9 */
   128, /* 10 */
    90, /* 11 */
    64, /* 12 */
    45, /* 13 */
    32, /* 14 */
    23, /* 15 */
    16, /* 16 */
    11, /* 17 */
     8, /* 18 */
     6, /* 19 */
     4, /* 20 */
     3, /* 21 */
     2  /* 22 */
};
 
 

/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static int 
  decodeTune( TUNE_REC_REF  tune_list,
              char          *text,
              int           soctave,
              int           stempo );

static void 
  sonifyTunes( TUNE_REC_REF  tune_list );

static int 
  toInteger( char *text,
             int  *offset );

static void 
  tone( Display  *display,
        int      frequency,
        int      duration,
        int      volume );

static void 
  sonifyCB( TUNE_REC_REF  tune_list );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  xitTuPlayTuneFromFile( Display       *display,
                         XtAppContext  context,
                         char          *filename )
{

  /* Variables. */
  int           soctave;
  int           stempo;
  char          buffer[ 500 ];
  char          *char_ref;
  char          *music;
  FILE          *file_ref;
  TUNE_REC_REF  tune_list;


  /* Code. */

  file_ref = fopen( filename, "r" );
  if( file_ref == NULL ) {
    fprintf( stderr, "xitTune: Cannot find file %s\n", filename );
    return;
  }

  /* Allocate memory. */
  music     = (char *) SysMalloc( LONGEST_TUNE + 256 );
  tune_list = (TUNE_REC_REF) SysMalloc( sizeof( TUNE_REC ) );

  *music  = '\0';

  /* Fetch the music from the file. */
  while( 1 ) {
    char_ref = fgets( buffer, sizeof( buffer ), file_ref );
    if( char_ref == NULL )
      break;

    if( strlen( music ) + strlen( buffer ) > LONGEST_TUNE ) {
      fprintf( stderr, "xitTune: The tune is too long, truncated\n" );

      music[ LONGEST_TUNE ] = '\0';
      break;
    }

    strcpy( music + strlen( music ), buffer );
  } /* while */

  fclose( file_ref );

  /* Decode the tune. */
  (void) decodeTune( tune_list, music, soctave, stempo );

  tune_list -> context = context;
  tune_list -> display = display;

  /* Let the music start! */
  sonifyTunes( tune_list );

  /* Release allocated memory. */
  SysFree( music );


  return;

} /* xitTuPlayTuneFromFile */


/*----------------------------------------------------------------------*/

static int 
  decodeTune( TUNE_REC_REF  tune_list,
              char          *text,
              int           soctave,
              int           stempo )
{

  /* Variables. */
  int  duration;
  int  duty;
  int  frequency;
  int  i;
  int  lastnote;
  int  n;
  int  note;
  int  octave1 = 0;
  int  octave;
  int  op;
  int  pause;
  int  shortness;
  int  tempo;
  int  volume;


  /* Code. */
 
  duty      = 7;
  frequency = 0;
  octave    = 3;
  shortness = 1;
  soctave   = 0;
  stempo    = 0;
  tempo     = 8;
  volume    = 25;


  octave += soctave;
  if( octave1 < 0 ) 
    octave1 = 0;
  else if( octave1 > 11 ) 
    octave1 = 11;
 
  tempo = tempo + stempo;
  if( tempo < 0 ) 
    tempo = 0;
  else if( tempo > 18 ) 
    tempo=18;
 
  tune_list -> notes = 0;
  if( text[ 0 ] == '-' || text[ 0 ]== '\0' ) 
    return( 0 );
 
  note = 0;

  /* Go through the sequence of operation substring codes. */
  for( i = 0; text[ i ]; ) {

    if( tune_list -> notes >= MAX_NOTES - 1 ) {
      fprintf( stderr, "The tune has too many notes, and is truncated\n" );
      return( 2 );
    }
 
    op = text[ i++ ];

    /* Ignore punctuation. */
    if( op == ',' || op == ';' || op == ':' || op == '.' || isspace( op ) ) 
      continue;


    switch( op ) {

      /* Regular note. */
      case 'a':
      case 'b':
      case 'c':
      case 'd':
      case 'e':
      case 'f':
      case 'g':
      case '_':
      case '=':
        lastnote = note;
        note = strchr( scale, op ) - scale;

        if( op == '=' ) 
          note = lastnote;
 
        /* Sharp? */
        if( text[ i ] == '#' ) {
          note++;
          i++;

        /* Flat? */
        } else if( text[ i ] =='*' ) {
          note--;
          i++;
        }
 
        /* Possible local octave step. */
        octave1 = octave + soctave;

        /* Up? */
        if( text[ i ] == '+' ) {
          octave1++;
          i++;

        /* Down? */
        } else if( text[ i ] == '-' ) {
          octave1--;
          i++;
        }

        if( octave1 < 0 ) 
          octave1 = 0;
        else if( octave1 > 11) 
          octave1 = 11;
 
        if( op != '+' ) 
          frequency =0.5 + (1<<octave1) * 55.0 * factor[ note ];
 
        /* Possible local note length specification. */
        if( isdigit( text[ i ] ) ) {
          n = text[ i ] - '0';
          duration= 1 + speed[ tempo ] / (1<<n);
          i++;

        } else 
          duration = 1 + speed[ tempo ] / shortness;

         /* 50% extension? */
        if( text[ i ] == '.' ) {
          duration = 1 + (3 * duration) / 2;
          i++;
        }
 
        pause = ((8 - duty) * duration + 4) / 8;
        duration = duration - pause;
 
        tune_list -> note[ tune_list -> notes ][ 0 ] = frequency;
        tune_list -> note[ tune_list -> notes ][ 1 ] = duration;
        tune_list -> note[ tune_list -> notes ][ 2 ] = volume;

        tune_list -> notes++;

 
        /* Allow for some sleeping overhead. */
        pause = pause - SLEEPY;
        if( pause > 0 ) {
          tune_list -> note[ tune_list -> notes ][ 0 ] = 0;
          tune_list -> note[ tune_list -> notes ][ 1 ] = pause;
          tune_list -> note[ tune_list -> notes ][ 2 ] = 0;

          tune_list -> notes++;
        }
        break;
 
 
      /* Specific frequency with optional duration. */
      case 'F':
        frequency = toInteger( text, &i );
        if( text[ i ] == 'D' ) {
          i++;
          duration = toInteger( text, &i );
        } else 
          duration = 1 + speed[ tempo ] / shortness;
 
        tune_list -> note[ tune_list -> notes ][ 0 ] = frequency;
        tune_list -> note[ tune_list -> notes ][ 1 ] = duration;
        tune_list -> note[ tune_list -> notes ][ 2 ] = volume;

        tune_list -> notes++;
        break;
 
      /* Note length. */
      case 'L':
        n = toInteger( text, &i );
        if( n < 0 )
          n = 0;
        if( n > 6 ) 
          n = 6;
        shortness = 1<<n;
        break;
 
      /* Note duty mode. */
      case 'D':
        duty = toInteger( text, &i );
        if( duty < 1 ) 
          duty = 1;
        else if( duty > 8 ) 
          duty = 8;
        break;
 
      /* Fake load a named voice into an instrument. */
      case 'N':
        for( n = 0; n < 7 && text[ i + n ] != '@'; n++ );
          i = i + n + 2;
        break;
 
      /* Octave. */
      case 'O':
        /* Relative increase. */
        if( text[ i ] == '+' ) {
          i++;
          octave = octave + toInteger( text, &i );

        /* Relative decrease. */
        } else if( text[ i ]== '-' ) {
          i++;
          octave = octave - toInteger( text, &i );

        /* Absolute set. */
        } else 
          octave = toInteger( text, &i ) + soctave;

        if( octave < 0 ) 
          octave = 0;
        else if( octave > 11 ) 
          octave = 11;
        break;
 
      /* Tempo. */
      case 'T':
        /* Relative increase. */
        if( text[ i ] == '+' ) {
          i++;
          tempo = tempo + toInteger( text, &i );

        /* Relative decrease. */
        } else if( text[ i ] == '-' ) {
          i++;
          tempo = tempo - toInteger( text, &i );
        }

        tempo = toInteger( text, &i ) + stempo;
        if( tempo < 0 ) 
          tempo = 0;
        else if( tempo > 18 ) 
          tempo = 18;
        break;
 
      /* Volume. */
      case 'V':
        /* Relative increase. */
        if( text[ i ] == '+' ) {
          i++;
          volume = volume + toInteger( text, &i );

        /* Relative decrease. */
        } else if( text[ i ] == '-' ) {
          i++;
          volume = volume - toInteger( text, &i );
        } else 
          volume = toInteger( text, &i );

        if( volume < 0 )
          volume = 0;
        else if( volume > 100 )
          volume = 100;
        break;
 
      /* Fake chord for compatibility. */
      case 'C':
        for( n = 0; n < 256 && text[ i + n ] != ';'; ++n )
          ;
        i = i + n;
        break;
 
      /* Fake modulation control for compatibility. */
      /* Fake balance for compatibility. */
      /* Fake instrument for compatibility. */
      /* Fake portamento for compatibility. */
      case 'M':
        i++;
      case 'B':
      case 'I':
      case 'P':
        if( text[ i ] == '+' || text[ i ] == '-' ) 
          i++;
        toInteger( text, &i );
        break;
 
      /* Fake all notes off for compatibility. */
      case '0':
        break;

      /* Fake set all defaults for compatibility. */
      case '#':
        break;

      /* No-op (possible interrupt stop point). */
      case '$':
        break;
 
      /* Major pause. */
      case '*':
        pause = toInteger( text, &i );
        tune_list -> note[ tune_list -> notes ][ 0 ] = 0;
        tune_list -> note[ tune_list -> notes ][ 1 ] = pause;
        tune_list -> note[ tune_list -> notes ][ 2 ] = volume;

        tune_list -> notes++;
        break;
 
      /* Comment. */
      case '<':
        for( ; text[ i ]; i++ ) 
          if( text[ i ] == '>' ) 
            break;
        i++;
        break;
 
      default:
        fprintf( stderr,
                 "Invalid operation code %s at position %d ignored in tune %s",
                 text + i - 1, i, text );
        return( 1 );

    } /* switch */
 
  } /* loop */

 
  return( 0 );

} /* decodeTune */


/*----------------------------------------------------------------------*/

static void 
  sonifyTunes( TUNE_REC_REF  tune_list )
{

  /* Code. */ 

  /* Get current keyboard settings */
  XGetKeyboardControl( tune_list -> display, &tune_list -> kbd_settings );

  /* Start! */
  tune_list -> curr_tune = 0;

  (void) XtAppAddTimeOut( tune_list -> context, 0L,
                          (XtTimerCallbackProc) sonifyCB,
                          (XtPointer) tune_list );

  return;

} /* sonifyTunes */


/*----------------------------------------------------------------------*/

static int 
  toInteger( char *text,
             int  *offset )
{

  /* Variables. */
  int  n;
  char *next;


  /* Code. */
 
  n = strtol( text + *offset, &next, 10 );
  *offset = next - text;


  return( n );
 
} /* toInteger */


/*----------------------------------------------------------------------*/

static void 
  tone( Display  *display,
        int      frequency,
        int      duration,
        int      volume )
{

  /* Variables. */
  XKeyboardControl  state;


  /* Code. */ 
 
  if( ! display ) 
    return;

  if( duration <= 0 ) 
    return;

  if( duration > 10000 ) 
    duration = 10000;
 
  if( frequency <= 0 || volume <= 0 ) {
    frequency = 0;
    volume = 0;
  }

 
  state.bell_pitch    = frequency;
  state.bell_duration = duration;
  XChangeKeyboardControl( display, (KBBellPitch | KBBellDuration), &state );
 
  XBell( display, volume );
  XSync( display, False );
 
  /* Reset defaults */
  state.bell_pitch    = -1;
  state.bell_duration = -1;
  XChangeKeyboardControl( display, (KBBellPitch | KBBellDuration), &state );
 

  return;

} /* tone */


/*----------------------------------------------------------------------*/

static void 
  sonifyCB( TUNE_REC_REF  tune_list )
{

  /* Variables. */
  int  index;


  /* Code. */

  index = tune_list -> curr_tune;

  tone( tune_list -> display,
        tune_list -> note[ index ][ 0 ], 
        tune_list -> note[ index ][ 1 ], 
        tune_list -> note[ index ][ 2 ] );

  /* Next note. */
  index++;

  /* Are we done? */
  if( index >= tune_list -> notes ) {

    XKeyboardControl  control;

    control.key_click_percent = tune_list -> kbd_settings.key_click_percent;
    control.bell_percent      = tune_list -> kbd_settings.bell_percent;
    control.bell_pitch        = tune_list -> kbd_settings.bell_pitch;
    control.bell_duration     = tune_list -> kbd_settings.bell_duration;

    XChangeKeyboardControl(
      tune_list -> display,
      (KBKeyClickPercent | KBBellPercent | KBBellPitch | KBBellDuration),
      &control );

    SysFree( tune_list );

    return;
  }

  /* Setup for next note. */
  tune_list -> curr_tune = index;

  (void) XtAppAddTimeOut( tune_list -> context, 
                          (unsigned long) tune_list -> note[ index ][ 2 ], 
                          (XtTimerCallbackProc) sonifyCB,
                          (XtPointer) tune_list );


  return;

} /* sonifyCB */
