/*----------------------------------------------------------------------------
--
--  Module:           xitMsgWin
--
--  Project:          xit   - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Window for general messages. The messages displayed are numbered and
--    they can be viewed and a file log can be created.
--
--  Filename:         xitMsgWin.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-06-16
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitMsgWin.c, Version: 1.1, Date: 95/02/18 15:10:42";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

#include "System.h"
#include "TimDate.h"

#include "xitTools.h"
#include "xitMsgWin.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the show window. */
#define msgWinLa         dataLocalW[  0 ]
#define msgWinRc         dataLocalW[  1 ]
#define msgWinTx         dataLocalW[  2 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for internal data. */
typedef struct {

  /* Max messages to display in the message window. */
  int  max_msg;

  /* Number of messages in the window. */
  int  no_msg;

  /* Selected message. */
  int  selected_msg;

  /* File name for error log file. */
  char  *log_file;

  /* Parent window. */
  Widget  parentW;

  /* Message window. */
  Widget  msgWinW;

} MSG_REC, *MSG_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static Widget
  createMsgWindow( MSG_REC_REF        msg_win_ref,
                   XIT_MSW_LABEL_REC  *labels );

static void 
  clearCB( Widget       widget,
           MSG_REC_REF  msg_win_ref,
           XtPointer    call_data );

static void 
  closeCB( Widget       widget,
           MSG_REC_REF  msg_win_ref,
           XtPointer    call_data );

static void 
  destroyCB( Widget       widget,
             MSG_REC_REF  msg_win_ref,
             XtPointer    call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XIT_MSW_HANDLE
  xitCreateMsgWindow( Widget             parent,
                      int                max_messages,
                      char               *log_file,
                      XIT_MSW_LABEL_REC  *labels )
{

  /* Variables. */
  MSG_REC_REF  msg_win_ref;


  /* Code. */

  /* Create and initialize our private data. */
  msg_win_ref = SysNew( MSG_REC );
  if( msg_win_ref == NULL )
    return( NULL );

  msg_win_ref -> parentW      = parent;
  msg_win_ref -> log_file     = NULL;
  msg_win_ref -> max_msg      = max_messages;
  msg_win_ref -> no_msg       = 0;
  msg_win_ref -> selected_msg = 0;

  /* Any log file? */
  if( log_file != NULL && *log_file != '\0' ) {
    msg_win_ref -> log_file = SysNewString( log_file );
  }

  /* Create the message window. */
  msg_win_ref -> msgWinW = createMsgWindow( msg_win_ref, labels );


  return( (XIT_MSW_HANDLE) msg_win_ref );

} /* xitCreateMsgWindow */


/*----------------------------------------------------------------------*/

void
  xitMsgWinClearWindow( XIT_MSW_HANDLE  msg_win_handle )
{

  /* Variables. */
  MSG_REC_REF  msg_win_ref;


  /* Code. */

  if( msg_win_handle == NULL )
    return;

  /* Our private data. */
  msg_win_ref = (MSG_REC_REF) msg_win_handle;


  /* Clear the window. */
  clearCB( NULL, msg_win_ref, NULL );


  return;

} /* xitMsgWinClearWindow */


/*----------------------------------------------------------------------*/

void
  xitMsgWinDestroy( XIT_MSW_HANDLE  msg_win_handle )
{

  /* Variables. */
  MSG_REC_REF  msg_win_ref;


  /* Code. */

  if( msg_win_handle == NULL )
    return;

  /* Our private data. */
  msg_win_ref = (MSG_REC_REF) msg_win_handle;


  /* Destroy the window. */
  XtDestroyWidget( msg_win_ref -> msgWinW );


  return;

} /* xitMsgWinDestroy */


/*----------------------------------------------------------------------*/

void
  xitMsgWinDisplayMessage( XIT_MSW_HANDLE  msg_win_handle,
                           char            *msg,
                           int             msg_id )
{

  /* Variables. */
  char            *char_ref;
  char            *text_ref;
  char            buffer[ 50 ];
  char            msg_buf[ 1000 ];
  Widget          mainW;
  Widget          tempW;
  XmTextPosition  position;
  MSG_REC_REF     msg_win_ref;
  TIM_TIME_REF    now;


  /* Code. */

  if( msg_win_handle == NULL )
    return;

  /* Our private data. */
  msg_win_ref = (MSG_REC_REF) msg_win_handle;


  now   = TimLocalTime( TimMakeTimeNow() );
  mainW = XtNameToWidget( msg_win_ref -> msgWinW, 
                          "SysMsgWinTlBase.SysMsgWinTlFo" );


  /* Build the message. */
  msg_buf[ 0 ] = '\0';

  if( msg_id > 0 ) {
    sprintf( buffer, "%2.2d  ", msg_id );
    strcat( msg_buf, buffer );
  }

  TimFormatDate( now, buffer, sizeof( buffer ) );
  strcat( msg_buf, buffer );
  strcat( msg_buf, " " );

  TimFormatTime( now, buffer, sizeof( buffer ) );
  strcat( msg_buf, buffer );
  strcat( msg_buf, "\n" );

  strcat( msg_buf, msg );

  if( msg_buf[ strlen( msg_buf ) - 1 ] != '\n' )
    strcat( msg_buf, "\n" );
  strcat( msg_buf, ">>\n" );


  /* Write the message in the message window. */
  tempW = XtNameToWidget( mainW, "MsgWinTxSW.MsgWinTx" );

  /* Remove the top message? */
  if( msg_win_ref -> no_msg >= msg_win_ref -> max_msg ) {
    text_ref = xitStringGetText( tempW );

    char_ref = strstr( text_ref, "\n>>\n" );
    if( char_ref != NULL ) {
      position = (XmTextPosition) (char_ref - text_ref + 3);
      XmTextReplace( tempW, 0, position, "" );

      SysFree( text_ref );
    }
  }

  /* Add the new message. */
  msg_win_ref -> no_msg++;

  text_ref = xitStringGetText( tempW );
  position = (XmTextPosition) strlen( text_ref );
  SysFree( text_ref );

  XmTextReplace( tempW, position, position, msg_buf );

  position = position + (XmTextPosition) strlen( msg_buf );
  XmTextShowPosition( tempW, position );


  /* Make sure the window is displayed. */
  XtPopup( msg_win_ref -> msgWinW, XtGrabNone );

  XRaiseWindow( XtDisplay( msg_win_ref -> msgWinW ), 
                XtWindow(  msg_win_ref -> msgWinW ) );

  XtMapWidget( msg_win_ref -> msgWinW );


  return;

} /* xitMsgWinDisplayMessage */


/*----------------------------------------------------------------------*/

static Widget
  createMsgWindow( MSG_REC_REF        msg_win_ref,
                   XIT_MSW_LABEL_REC  *labels )
{

  /* Variables. */
  Boolean   default_geometry = False;
  int       index;
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    controlBu[ 1 ];
  Widget    dataLocalW[ 3 ];
  Widget    sysMsgWinTl;
  Widget    toplevel;
  Widget    workFo;

  static XIT_PUSH_STRUCT control_def[] = {
    { "ClosePb", "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "MsgWinTx", NULL, 2, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { NULL, NULL,    NULL },
    { "",   closeCB, NULL },
    { NULL, NULL,    NULL },
  };


  /* Code. */

  /* Set message strings. */
  action_buttons[ 1 ].label = labels -> close_button;
  action_buttons[ 1 ].data  = msg_win_ref;

  control_def[ 0 ].title = labels -> clear_button;


  /* Create a form dialog with buttons. */
  toplevel = xitGetToplevelWidget( msg_win_ref -> parentW );

  sysMsgWinTl = xitCreateToplevelDialog( toplevel, "SysMsgWinTl",
                                         2, 0,
                                         action_buttons,
                                         XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle,    labels -> title ); n++;
  XtSetArg( args[ n ], XmNiconName, labels -> title ); n++;
  XtSetValues( sysMsgWinTl, args, n );


  /* If no geometry defined, position in the middle of the screen. */
  n = 0;
  XtSetArg( args[ n ], XmNgeometry, &char_ref ); n++;
  XtGetValues( sysMsgWinTl, args, n );

  if( char_ref != NULL && strpbrk( char_ref, "+-" ) != NULL )
    default_geometry = False;

  if( default_geometry ) {
    n = 0;
    XtSetArg( args[ n ], XmNx,
              WidthOfScreen( XtScreen( sysMsgWinTl ) ) / 2 ); n++;
    XtSetArg( args[ n ], XmNy, 
              HeightOfScreen( XtScreen( sysMsgWinTl ) ) / 2 ); n++;
    XtSetValues( sysMsgWinTl, args, n );
  }


  /* Pop down if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( sysMsgWinTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( sysMsgWinTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( sysMsgWinTl, wm_delete_window, 
                             (XtCallbackProc) closeCB,
                             (XtPointer) msg_win_ref );
  } /* block */

  XtAddCallback( sysMsgWinTl,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) msg_win_ref );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( sysMsgWinTl, "SysMsgWinTlBase.SysMsgWinTlFo" );


  /* Label for the message list. */
  msgWinLa = xitCreateLabel( workFo, "MsgWinLa", 
                             labels -> list_label, XmALIGNMENT_BEGINNING );

  /* Text window with messages.. */
  msgWinTx = xitCreateTextScrolled( workFo, &text_buffer_def[ 0 ] );

  n = 0;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( msgWinTx, args, n );


  /* Control buttons. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  msgWinRc = XmCreateRowColumn( workFo, "MsgWinRc", args, n );

  for( index = 0; index < XtNumber( controlBu ); index++ )
    controlBu[ index ] = xitCreatePushButton( msgWinRc, 
                                              &control_def[ index ] );

  XtAddCallback( controlBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) clearCB, (XtPointer) msg_win_ref );

  /* Put the elements together. */
  xitAttachWidget( msgWinLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( msgWinTx ),
                   XmATTACH_WIDGET, msgWinLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( msgWinRc,
                   XmATTACH_WIDGET, XtParent( msgWinTx ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( msgWinLa,             args, n );
  XtSetValues( XtParent( msgWinTx ), args, n );
  XtSetValues( msgWinRc,             args, n );

  /* Manage the widgets. */
  XtManageChildren( controlBu, XtNumber( controlBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the size of the window. */
  xitSetSizeToplevelDialog( sysMsgWinTl, True );

  /* Make the final attachments. */
  xitAttachWidget( msgWinRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( XtParent( msgWinTx ),
                   XmATTACH_WIDGET, msgWinLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,     XmATTACH_WIDGET, msgWinRc );


  return( sysMsgWinTl );

} /* createMsgWindow */


/*----------------------------------------------------------------------*/

static void 
  clearCB( Widget       widget,
           MSG_REC_REF  msg_win_ref,
           XtPointer    call_data )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( msg_win_ref -> msgWinW, 
                          "SysMsgWinTlBase.SysMsgWinTlFo" );

  /* Clear the message window (but leave the log file). */
  tempW = XtNameToWidget( mainW, "MsgWinTxSW.MsgWinTx" );

  XmTextSetString( tempW, "" );

  msg_win_ref -> no_msg = 0;


  return;

} /* clearCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget       widget,
           MSG_REC_REF  msg_win_ref,
           XtPointer    call_data )
{

  /* Code. */

  XtPopdown( msg_win_ref -> msgWinW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget       widget,
             MSG_REC_REF  msg_win_ref,
             XtPointer    call_data )
{

  /* Code. */

  if( msg_win_ref -> log_file != NULL )
    SysFree( msg_win_ref -> log_file );

  SysFree( msg_win_ref );


  return;

} /* destroyCB */
