/*----------------------------------------------------------------------------
--
--  Module:           xtmDiary
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Main program for XDiary.
--
--  Filename:         xtmDiary.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmDiary.c, Version: 1.1, Date: 95/02/18 16:05:44";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

#include <Xm/Xm.h>

#include "System.h"
#include "Message.h"
#include "SigHandler.h"
#include "TimDate.h"

#include "msgTopic.h"
#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmAlarmCtrl.h"
#include "xtmCalDb.h"
#include "xtmCalendar.h"
#include "xtmCustBase.h"
#include "xtmDbMisc.h"
#include "xtmHelp.h"
#include "xtmHoliday.h"
#include "xtmResource.h"
#include "xtmSetup.h"
#include "xtmSchedMain.h"
#include "xtmShowToDo.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitMsgWin.h"
#include "xitStickyMsg.h"
#include "xitTools.h"
#include "xitInfo.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdiary"

/* Program class (also name of application resource file). */
#define PROGRAM_CLASS  "XDiary"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;

/* Name of text domain. */
static char  *text_domain      = "XDiary";
static char  *info_text_domain = "Topic";

/* Name of module. */
static char  *module_name = "xtmDiary";

/* Keyboard translations for newline in multi-line editor. */
XtTranslations  xtm_di_newline_trans;

/* Handle to the 'today' summary window. */
static XTM_SD_HANDLE  today_show_handle = NULL;

/* Application data. */
static  XTM_GL_BASE_DATA  appl_data;


/* Key translations. */
static char newlineTrans[] = 
  "<Key>Return:    newline()";


/* Program options. */
static XrmOptionDescRec options[] = {
  { "-alarm",        "*.giveAlarm",         XrmoptionNoArg,  "True" },
  { "-noalarm",      "*.giveAlarm",         XrmoptionNoArg,  "False" },
  { "-noAlarm",      "*.giveAlarm",         XrmoptionNoArg,  "False" },
  { "-debug",        "*.debug",             XrmoptionNoArg,  "True" },
  { "-delta",        "*.entryDelta",        XrmoptionSepArg, NULL },
  { "-geometry",     "CalTl.geometry",      XrmoptionSepArg, NULL },
  { "-h",            "*.StartupHelp",       XrmoptionNoArg,  "True" },
  { "-help",         "*.StartupHelp",       XrmoptionNoArg,  "True" },
  { "-iconic",       "CalTl.iconic",        XrmoptionNoArg,  "True" },
  { "-lan",          "*.msgLanguage",       XrmoptionSepArg, NULL },
  { "-language",     "*.msgLanguage",       XrmoptionSepArg, NULL },
  { "-noFileLock",   "*.useFileLock",       XrmoptionNoArg,  "False" },
  { "-nofilelock",   "*.useFileLock",       XrmoptionNoArg,  "False" },
  { "-readOnly",     "*.readOnlyMode",      XrmoptionNoArg,  "True" },
  { "-readonly",     "*.readOnlyMode",      XrmoptionNoArg,  "True" },
  { "-ro",           "*.readOnlyMode",      XrmoptionNoArg,  "True" },
  { "-title",        "*.title",             XrmoptionSepArg, NULL },
  { "-usage",        "*.StartupHelp",       XrmoptionNoArg,  "True" },
  { "-version",      "*.VersionHelp",       XrmoptionNoArg,  "True" },
};


/* Base program resources. */
static XtResource  base_resources[] = {

  { "defaultDbDir", "DefaultDbDir", XtRString, sizeof( String ),
    XtOffset( XTM_GL_BASE_DATA_REF, default_db_dir ), 
    XtRString, "." },

  { "startupHelp", "StartupHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_GL_BASE_DATA_REF, startup_help ), 
    XtRString, "False" },

  { "title", "Title", XtRString, sizeof( String ),
    XtOffset( XTM_GL_BASE_DATA_REF, title ), 
    XtRString, "XDiary" },

  { "versionHelp", "VersionHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_GL_BASE_DATA_REF, version_help ), 
    XtRString, "False" },

};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  calActionCB( XTM_CA_REASON reason,
               void          *call_data,
               void          *user_data );

static void
  displayStartup( XTM_GL_BASE_DATA_REF  appl_data_ref );

static void
  displayUsage();

static void
  mainRedisplayCB( int   this_signal,
                   void  *user_data );

static void
  mainRedisplayDo( XTM_GL_BASE_DATA_REF appl_data_ref );

static void
  refreshCB( UINT32  flags,
             void    *user_data,
             void    *call_data );

static void
  summaryToday( XTM_GL_BASE_DATA_REF appl_data_ref );

static void 
  summaryActionCB( XTM_SD_REASON reason,
                   void          *user_data );

static void
  terminateCB( int   this_signal,
               void  *user_data );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  main( int argc, char **argv )
{

  /* Variables. */
  int                   index;
  int                   temp;
  char                  *char_ref;
  Arg                   args[ 10 ];
  Cardinal              n;
  Display               *display;
  TIM_TIME_REF          now;
  TIM_STATUS_TYPE       time_status;
  XTM_GL_CUSTOM_DATA    custom_data;
  XTM_CB_STATUS         custom_status;
  XTM_CD_CAL_INFO       default_db;
  XTM_GL_PROC_INFO      process_info;


  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;
  xitErSetApplicationName( PROGRAM_CLASS );

  now = TimMakeTimeNow();


  /* Save the original command parameters. */
  custom_data.orig_argc = argc;
  custom_data.orig_argv = (char**) XtMalloc( argc * sizeof( char* ) );

  for( index = 0; index < argc; index++ )
    custom_data.orig_argv[ index ] = XtNewString( argv[ index ] );
  
  /* NLS enabled. */
  xtmToSetLocale( program_name );


  /* Initialization. */
  SysInitializeEnvironment();
  xtmDbInitializeProcessId();


  /* Initialize toolkit and open display. */
  XtToolkitInitialize();

  appl_data.context = XtCreateApplicationContext();
  display = XtOpenDisplay( appl_data.context, NULL,
                           NULL, PROGRAM_CLASS,
                           options, XtNumber( options ),
#if XtSpecificationRelease < 5
                           (Cardinal *) &argc,
#else
                           (int *) &argc,
#endif
                           argv );

  if( display == NULL )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "Cannot open display, check your DISPLAY variable." );

  /* Resource mapping.*/
  xtmToInitializeResourceMapping( argc, argv, display );
  
  /* Default translations */
  xtm_di_newline_trans   = XtParseTranslationTable( newlineTrans );


  /* Create application shell. */
  n = 0;
  appl_data.toplevel = XtAppCreateShell( NULL, PROGRAM_CLASS,
                                         applicationShellWidgetClass,
                                         display,
                                         args, n );

  /* Get base application resources. */
  XtGetApplicationResources( appl_data.toplevel, (XtPointer) &appl_data, 
                             base_resources, 
                             XtNumber( base_resources ), 
                             NULL, 0 );

  /* Get customize resources. */
  xtmRsFetchCustomResources( &custom_data, appl_data.toplevel );


  /* A valid resource file? */
  if( ! custom_data.valid_resource_file )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "XDiary will only run with a correct X-Windows resource "
                  "file.\nPlease check the XDiary installation." );


  /* Initialize the global data records. */
  custom_data.cal_db_handle   = NULL;
  custom_data.group_db_handle = NULL;
  custom_data.archive_files   = NULL;
  custom_data.include_files   = NULL;

  process_info.alarm_pid     = 0;
  process_info.mainFo        = NULL;

  appl_data.flags            = 0;
  appl_data.alarm_fifo_ref   = NULL;
  appl_data.alarm_fifo_name  = NULL;
  appl_data.msg_send_handle  = NULL;
  appl_data.info_handle      = NULL;
  appl_data.msg_win_ref      = NULL;
  appl_data.custom_data      = &custom_data;
  appl_data.process_info     = &process_info;

  for( index = 0; index < XTM_GL_MAX_ALARM_PRC; index++ )
    appl_data.process_info -> processes[ index ].db_name[ 0 ] = '\0';


  /* Display current version? */
  if( appl_data.version_help ) {
    printf( "%s: Version: %s\n", program_name, VERSION_ID );
    exit( 0 );
  }

  /* Help requested? */
  if( appl_data.startup_help ) {
    displayUsage();
    exit( 0 );
  }


  /* Get customized data from file. */
  custom_status = xtmCbGetDataFromFile( appl_data.custom_data );

  /* Initialize necessary text domains. */
  msgInitialize();
  msgInitCatalogue( text_domain, NULL, custom_data.msg_language,
                    msgXdiaryTexts );
  msgInitCatalogue( info_text_domain, NULL, custom_data.msg_language,
                    msgTopicTexts );

  /* Default catalogue Xdiary. */
  msgTextDomain( text_domain );

  /* What version are we using. */
  if( custom_status == XTM_CB_WRONG_VERSION ) {
    char_ref = (char *) 
      SysMalloc( strlen( msgGetText( MXDI_CUST_WRONG_VERSION ) ) + 50 );

    sprintf( char_ref, msgGetText( MXDI_CUST_WRONG_VERSION ),
             xtmCbWhatVersion() );

    xitStDisplaySticky( appl_data.toplevel,
                        char_ref, XmUNSPECIFIED_PIXMAP,
                        msgGetText( MXDI_OK_BUTTON ), NULL,
                        NULL, NULL, NULL,
                        NULL );
    SysFree( char_ref );
    exit( 1 );
  }


  /* Initialize the time formats. */
  time_status = TimInitializeFormat( custom_data.date_format,
                                     custom_data.time_format );
  if( time_status != TIM_OK )
    xitErMessage( appl_data.toplevel, XIT_ER_ERROR, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  /* Set colors and fonts in the resource database. */
  xtmRsFetchColors( &custom_data, appl_data.toplevel );


  /* Initialize holidays. */
  xtmHoInitialize( custom_data.holidays_db_dir, custom_data.workdays );


  /* Use file locking? */
  xtmDbUseFileLock( custom_data.use_file_lock );

  /* Run in debug mode? */
  if( custom_data.debug_mode ) {
    SysDebugOn( True );
  }


  /* Check the delta time. */
  if( custom_data.default_entry_delta < 5 ||
      custom_data.default_entry_delta > 360 )
    custom_data.default_entry_delta = 60;
  else
    custom_data.default_entry_delta = custom_data.default_entry_delta;


  /* Check list start and stop time. */
  if( custom_data.start_hour < 0 || custom_data.start_hour > 23 )
    custom_data.start_hour = 8;

  if( custom_data.stop_hour < 0 || custom_data.stop_hour > 23 )
    custom_data.stop_hour = 18;

  if( custom_data.start_hour > custom_data.stop_hour ) {
    temp = custom_data.start_hour;
    custom_data.start_hour = custom_data.stop_hour;
    custom_data.stop_hour  = temp;
  }


  /* Make sure the XDiary setup is coreect. */
  xtmStCheckSetup( appl_data.custom_data, 
                   appl_data.toplevel,
                   appl_data.context );


  /* Initialize the signal handler module. */
  SigInitialize();

  /* Initialize the update module. */
  if( custom_data.do_refresh )
    xtmUpInitialize( appl_data.context, custom_data.refresh_rate );
  else
    xtmUpInitialize( appl_data.context, 0 );


  /* Do we have a default database? */
  (void) xtmCdFetchDefaultDb( custom_data.cal_db_handle, &default_db, NULL );

  /* We do require read access to your default database. */
  if( ! flagIsSet( default_db.operations, XTM_DB_FLAG_MODE_READ ) )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );


  /* Initialize the main calendar. */
  appl_data.cal_handle = xtmCaInitialize( &appl_data,
                                          appl_data.toplevel,
                                          True, True,
                                          default_db.short_name,
                                          calActionCB, (void *) &appl_data );
  if( appl_data.cal_handle == NULL )
    exit( 1 );

  /* Main calendar window. */
  appl_data.calW = xtmCaMainWindow( appl_data.cal_handle );


  /* Initialize the error message window. */
  {
    XIT_MSW_LABEL_REC  msg_win_labels;

    msg_win_labels.title        = msgGetText( MXDI_MSG_WIN_TITLE );
    msg_win_labels.list_label   = msgGetText( MXDI_MSG_WIN_LIST_LABEL );
    msg_win_labels.close_button = msgGetText( MXDI_CLOSE_BUTTON );
    msg_win_labels.clear_button = msgGetText( MXDI_CLEAR_BUTTON );

    appl_data.msg_win_ref = xitCreateMsgWindow( appl_data.calW,
                                                50, custom_data.error_log_file,
                                                &msg_win_labels );
  } /* block */

  /* Initialize the help system. */
  appl_data.info_handle = xitInInitialize( appl_data.toplevel,
                                           NULL, NULL );


  /* Diaplay the calendar. */
  xtmCaView( appl_data.cal_handle, TimLocalTime( now ) );

  /* Session management. */
  xtmToSessionMgmt( xtmCaToplevelWindow( appl_data.cal_handle ),
                    custom_data.orig_argc, custom_data.orig_argv );

  /* Display the startup schedule. */
  displayStartup( &appl_data );


  /* Signal handlers. */
  (void) SigRegister( SIGHUP,  0, terminateCB,     &appl_data );
  (void) SigRegister( SIGTERM, 0, terminateCB,     &appl_data );
  (void) SigRegister( SIGINT,  0, terminateCB,     &appl_data );
  (void) SigRegister( SIGILL,  0, terminateCB,     &appl_data );
  (void) SigRegister( SIGQUIT, 0, terminateCB,     &appl_data );
  (void) SigRegister( SIGABRT, 0, terminateCB,     &appl_data );
  (void) SigRegister( SIGUSR1, 0, mainRedisplayCB, &appl_data );

  /* Tell the user what's up today? */
  if( appl_data.custom_data -> show_day_to_do )
    summaryToday( &appl_data );


  /* Display everything. */
  XmUpdateDisplay( appl_data.calW );


  /* Start the alarm process. */
  xtmAcStartDefaultAlarmPrc( appl_data.calW, &appl_data );


  /* Register for refresh updates. */
  (void) xtmUpRegister( XTM_UP_REFRESH_TICK, refreshCB, (void *) &appl_data );


  /* Update calendar time. */
  xtmUpDoUpdate( XTM_UP_MINUTE_TICK, (void *) now );


  /* Are we running a new release? */
  if( custom_data.new_release )
    xtmHlDisplayReleaseNotes( appl_data.info_handle, 
                              xtmCaToplevelWindow( appl_data.cal_handle ) );


  /* Do not realize the toplevel shell but enter the event loop. */
  do {

    XEvent  event;

    XtAppNextEvent( appl_data.context, &event );

    /* The calendar is interested in events. */
    xtmCaProcessEvents( appl_data.cal_handle, &event );

    XtDispatchEvent( &event );

  } while( True );


} /* main */


/*----------------------------------------------------------------------*/

static void
  displayStartup( XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  char                    *char_ref;
  char                    *db_names;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  XTM_SM_HANDLE           sched_handle;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;


  /* Start day view for all databases if flag set. */
  (void) xtmCdFetchDbNames( custom_data -> cal_db_handle, &db_names );
  char_ref = db_names;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( custom_data -> cal_db_handle, db_name,
                              &db_info, NULL );

    if( flagIsSet( db_info.flags, 
                   (XTM_CD_FLAG_DEFAULT_START_ICON | 
                    XTM_CD_FLAG_DEFAULT_START_WIN) ) ) {

      /* Create the schedule window. */
      sched_handle = xtmSmInitialize( appl_data_ref,
                                      appl_data.toplevel,
                                      db_info.short_name, NULL,
                                      NULL, NULL );

      /* Give an error message if the schedule could not be initialized. */
      if( sched_handle == NULL ) {
        xitErMessage( NULL, XIT_ER_ERROR, 
                      module_name, "displayStartup",
                      msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );

        if( db_names != NULL )
          SysFree( db_names );

        return;
      }

      /* Start as icon? */
      if( flagIsSet( db_info.flags, XTM_CD_FLAG_DEFAULT_START_ICON ) )
        xtmSmViewSchedule( sched_handle, 0, True );
      else
        xtmSmViewSchedule( sched_handle, 0, False );

    } /* if */

  } while( True );

  if( db_names != NULL )
    SysFree( db_names );


  return;

} /* displayStartup */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Your personal calendar.\n" 
    "\n"
    "XDiary is your personal organizer that combines the functions of a\n"
    "desktop calendar, an appointment book and an alarm clock. XDiary will\n"
    "help you keep track of your meetings, appointments and plan your time.\n"
    "\n"
    "Usage:\n"
    "  %s [flags]\n"
    "\n"
    "Flags:\n"
    "  -fmap large      : Use a large font.\n"
    "  -fmap medium     : Use a medium font.\n"
    "  -fmap small      : Use a small font.\n"
    "  -help            : Display this help.\n"
    "  -h               : See above.\n"
    "  -language <lan>  : Use the language <lan>.\n"
    "  -lan <lan>       : See above.\n"
    "  -noFileLock      : Don't use any file locking.\n"
    "  -palette gray    : Use color palette Gray.\n"
    "  -palette lila    : Use color palette Lila.\n"
    "  -palette motif   : Use color palette Motif.\n"
    "  -palette neon    : Use color palette Neon.\n"
    "  -palette nordic  : Use color palette Nordic.\n"
    "  -palette red     : Use color palette Red.\n"
    "  -palette sea     : Use color palette Sea.\n"
    "  -palette sky     : Use color palette Sky.\n"
    "  -palette wood    : Use color palette Wood.\n"
    "  -readOnly        : Run in read-only mode.\n"
    "  -ro              : See above.\n"
    "  -usage           : Display this help.\n"
    "  -version         : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );


  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static void
  mainRedisplayDo( XTM_GL_BASE_DATA_REF appl_data_ref )
{

  /* Code. */

  /* Update the calendars. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;

} /* mainRedisplayDo */


/*----------------------------------------------------------------------*/

static void
  summaryToday( XTM_GL_BASE_DATA_REF appl_data_ref )
{

  /* Variables. */
  Boolean                 entry_defined[ 1 ];
  TIM_TIME_REF            now;
  TIM_TIME_REF            today;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         default_db;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  now = TimLocalTime( TimMakeTimeNow() );

  today = TimMakeTime( TimIndexOfYear(  now ),
                       TimIndexOfMonth( now ),
                       TimIndexOfDay(   now ),
                       0, 0, 0 );

  (void) xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle, 
                              &default_db, NULL );

  /* Check current month. */
  xtmDmEntriesDefined( appl_data_ref,
                       default_db.short_name,
                       today, 1, True, True, 
                       entry_defined );

  /* Any entries defined today? */
  if( ! entry_defined[ 0 ] )
    return;


  /* Create summary window? */
  today_show_handle = xtmSdInitialize( appl_data_ref, 
                                       appl_data_ref -> toplevel,
                                       default_db.short_name,
                                       summaryActionCB, NULL );

  if( today_show_handle == NULL )
    return;


  /* Display the summary window. */
  xtmSdDisplaySummary( today_show_handle,
                       today, today,
                       custom_data_ref -> filter_day_view_tags );


  return;

} /* summaryToday */


/*----------------------------------------------------------------------*/

static void 
  calActionCB( XTM_CA_REASON reason,
               void          *call_data,
               void          *user_data )
{

  /* Variables. */
  Boolean                 ok;
  char                    *db_name;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = (XTM_GL_BASE_DATA_REF) user_data;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Exit XDiary? */
  if( reason == XTM_CA_REASON_EXIT ) {
    terminateCB( SIGTERM, (void *) appl_data_ref );

    return;
  }


  /* New date for the last used day view? */
  if( reason == XTM_CA_REASON_NEW_DATE ) {

    ok = xtmSmViewLastUsed( (TIM_TIME_REF) call_data );
    if( ! ok )
      reason = XTM_CA_REASON_DATE_SELECT;

  } /* if */


  /* Date selected? */
  if( reason == XTM_CA_REASON_DATE_SELECT  || 
      reason == XTM_CA_REASON_OPEN_VIEW ) {

    TIM_TIME_REF     display_date;
    TIM_TIME_REF     now;
    XTM_CD_CAL_INFO  db_info;
    XTM_SM_HANDLE    sched_handle;

    now = TimLocalTime( TimMakeTimeNow() );

    if( reason == XTM_CA_REASON_OPEN_VIEW )
      display_date = TimMakeTime( TimIndexOfYear(  now ),
                                  TimIndexOfMonth( now ),
                                  TimIndexOfDay(   now ), 0, 0, 0 );
    else
      display_date = (TIM_TIME_REF) call_data;


    /* Fetch the database to use. */
    if( reason == XTM_CA_REASON_OPEN_VIEW ) {
      db_name = (char *) call_data;

      (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                                db_name,
                                &db_info, NULL );
    } else {
      (void) xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle,
                                  &db_info, NULL );
    }

    /* Initialize and create the schedule view. */
    sched_handle = xtmSmInitialize( appl_data_ref,
                                    appl_data_ref -> toplevel,
                                    db_info.short_name, NULL,
                                    NULL, NULL );

    /* Give an error message if the database could not be opened. */
    if( sched_handle == NULL ) {
      xitErMessage( NULL, XIT_ER_ERROR,
                    module_name, "calActionCB",
                    msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );
      return;
    }

    /* Display the schedule. */
    xtmSmViewSchedule( sched_handle, display_date, False );

  } /* if */


  return;

} /* calActionCB */


/*----------------------------------------------------------------------*/

static void
  mainRedisplayCB( int   this_signal,
                   void  *user_data )
{

  /* Variables. */
  XTM_GL_BASE_DATA_REF appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_GL_BASE_DATA_REF) user_data;

  /* Start a 'NULL' timer to do the update. */
  XtAppAddTimeOut( appl_data_ref -> context, 0, 
                   (XtTimerCallbackProc) mainRedisplayDo,
                   (XtPointer) appl_data_ref );


  return;

} /* mainRedisplayCB */


/*----------------------------------------------------------------------*/

static void
  refreshCB( UINT32  flags,
             void    *user_data,
             void    *call_data )
{

  /* Variables. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_GL_BASE_DATA_REF) user_data;

  /* Update the calendar and all day views. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;

} /* refreshCB */


/*----------------------------------------------------------------------*/

static void 
  summaryActionCB( XTM_SD_REASON reason,
                   void          *user_data )
{

  /* Code. */

  if( today_show_handle == NULL )
    return;

  if( reason == XTM_SD_REASON_DESTROY )
    today_show_handle = NULL;


  return;

} /* summaryActionCB */


/*----------------------------------------------------------------------*/

static void
  terminateCB( int   this_signal,
               void  *user_data )
{

  /* Variables. */
  int                  status;
  XTM_GL_BASE_DATA_REF appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_GL_BASE_DATA_REF) user_data;

  /* Kill all alarm processes. */
  if( appl_data_ref -> process_info -> alarm_pid != 0 )
    kill( appl_data_ref -> process_info -> alarm_pid, SIGTERM );

  if( appl_data_ref -> alarm_fifo_ref != NULL ) {
    fclose( appl_data_ref -> alarm_fifo_ref );

    status = unlink( appl_data_ref -> alarm_fifo_name );
  }

  exit( 0 );

} /* terminateCB */


