/*----------------------------------------------------------------------------
--
--  Module:           xtmDump
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Dumps the contents of an XDiary database. The entries are saved
--    in the file $HOME/XDiary.backup by default.
--
--  Filename:         xtmDump.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-09-21
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmDump.c, Version: 1.1, Date: 95/02/18 16:04:29";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#include <X11/Intrinsic.h>

#include "System.h"
#include "LstLinked.h"
#include "TimDate.h"

#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdiary"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char *program_name;

/* Name of module. */
static char  *module_name = "xtmDumpDiary";

/* Display profile. */
static int      lines = 10000;
static char     output_file[ 150 ];

static Boolean  backup          = False;
static Boolean  comments        = False;
static Boolean  display_output  = False;
static Boolean  last_month      = False;
static Boolean  last_week       = False;
static Boolean  last_year       = False;
static Boolean  next_month      = False;
static Boolean  next_week       = False;
static Boolean  next_year       = False;
static Boolean  repeated        = True;
static Boolean  repeated_inline = False;
static Boolean  this_month      = False;
static Boolean  this_week       = False;
static Boolean  this_year       = False;
static Boolean  today           = False;
static Boolean  tomorrow        = False;
static Boolean  us_date         = False;
static Boolean  yesterday       = False;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  displayUsage();

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 );

static void
  printDisplayOutput( XTM_CD_CAL_INFO       *db_info,
                      FILE                  *file_ref,
                      TIM_TIME_REF          entry_date,
                      XTM_DB_ALL_ENTRY_REF  entry_ref,
                      char                  *entry_text );

static void
  printTaggedOutput( XTM_CD_CAL_INFO       *db_info,
                     FILE                  *file_ref,
                     Boolean               backup,
                     TIM_TIME_REF          entry_date,
                     XTM_DB_ALL_ENTRY_REF  entry_ref,
                     char                  *entry_text );

static void
  processAppointments( XTM_CD_CAL_INFO  *db_info,
                       FILE             *file_ref,
                       UINT32           flags,
                       Boolean          backup,
                       TIM_TIME_REF     from_date,
                       TIM_TIME_REF     to_date );

static void
  processRepeated( XTM_CD_CAL_INFO  *db_info,
                   FILE             *file_ref,
                   Boolean          backup,
                   TIM_TIME_REF     from_date,
                   TIM_TIME_REF     to_date );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

int
main( int argc, char *argv[] )
{

  /* Variables. */
  Boolean          do_file_lock = True;
  int              arg_ref;
  int              status;
  char             buffer[ 100 ];
  char             filter[ 50 ];
  char             from_date_buffer[ 20 ];  
  char             to_date_buffer[ 20 ];  
  char             *char_ref;
  FILE             *file_ref;
  TIM_STATUS_TYPE  tstatus;
  TIM_TIME_REF     from_date;
  TIM_TIME_REF     from_date_now;
  TIM_TIME_REF     to_date;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  /* Defaults. */
  filter[ 0 ]           = '\0';
  from_date_buffer[ 0 ] = '\0';
  output_file[ 0 ]      = '\0';
  to_date_buffer[ 0 ]   = '\0';

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;

  /* Initialization. */
  SysInitializeEnvironment();

  
  /* Fetch qualifiers. */
  arg_ref = 1;
  while( arg_ref < argc && argv[ arg_ref ][ 0 ] == '-' ) {

    if( noCaseStrcmp( argv[ arg_ref ], "-from" ) == 0 ||
        noCaseStrcmp( argv[ arg_ref ], "-f" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        strcpy( from_date_buffer, argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -from requires parameter.\n", program_name );
        displayUsage();
        exit( 0 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-to" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-t" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        strcpy( to_date_buffer, argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -to requires parameter.\n", program_name );
        displayUsage();
        exit( 0 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-filter" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-fil" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        strcpy( filter, argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -filter requires parameter.\n", program_name );
        displayUsage();
        exit( 0 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-lines" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-lin" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        lines = atoi( argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -lines requires parameter.\n", program_name );
        displayUsage();
        exit( 0 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-output" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-out" ) == 0 ) {
      arg_ref++;
      if( arg_ref < argc )
        strcpy( output_file, argv[ arg_ref ] );
      else {
        fprintf( stderr, "%s: -output requires parameter.\n", program_name );
        displayUsage();
        exit( 0 );
      }

    } else if( noCaseStrcmp( argv[ arg_ref ], "-noFilelock" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-nof" ) == 0 ) {
      do_file_lock = False;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-yesterday" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-yes" ) == 0 ) {
      yesterday = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-today" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-tod" ) == 0 ) {
      today = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-tomorrow" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-tom" ) == 0 ) {
      tomorrow = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-lastWeek" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-lw" ) == 0 ) {
      last_week = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-thisWeek" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-tw" ) == 0 ) {
      this_week = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-nextWeek" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-nw" ) == 0 ) {
      next_week = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-lastMonth" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-lm" ) == 0 ) {
      last_month = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-thisMonth" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-tm" ) == 0 ) {
      this_month = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-nextMonth" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-nm" ) == 0 ) {
      next_month = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-lastYear" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-ly" ) == 0 ) {
      last_year = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-thisYear" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-ty" ) == 0 ) {
      this_year = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-nextYear" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-ny" ) == 0 ) {
      next_year = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-noRepeated" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-nor" ) == 0 ) {
      repeated = False;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-noStanding" ) == 0 ) {
      repeated = False;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-repInline" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-repi" ) == 0 ) {
      repeated_inline = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-backup" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-bac" ) == 0 ) {
      backup = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-comments" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-com" ) == 0 ) {
      comments = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-us" ) == 0 ) {
      us_date = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-help" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-help" ) == 0 ||
               noCaseStrcmp( argv[ arg_ref ], "-usage" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-version" ) == 0 ) {
      printf( "%s: Version: %s\n", program_name, VERSION_ID );
      exit( 0 );

    } /* if */

    arg_ref++;

  } /* while */

  /* Check arguments. */
  if( (argc - arg_ref) < 1 ) {
    displayUsage();
    exit( 1 );
  }


  /* Force US date and time format? */
  if( us_date )
    (void) TimInitializeFormat( "MDY/0100", "12ampm:0" );

  /* Fetch the required parameters. */
  strcpy( db_info.directory, argv[ arg_ref++ ] );

  if( ! backup && strlen( filter ) == 0 )
    display_output = True;

  if( lines <= 0 )
    lines = 10000;


  from_date_now = TimLocalTime( TimMakeTimeNow() );
  from_date_now = TimMakeTime(  TimIndexOfYear(  from_date_now ),
                                TimIndexOfMonth( from_date_now ),
                                TimIndexOfDay(   from_date_now ),
                                0, 0, 0 );
  from_date = from_date_now;
  to_date   = from_date;

  if( yesterday ) {
    from_date = from_date_now;
    TimAddDays( &from_date, -1 );
    to_date = from_date;

  } else if( today ) {
    from_date = from_date_now;
    to_date = from_date;

  } else if( tomorrow ) {
    from_date = from_date_now;
    TimAddDays( &from_date, 1 );
    to_date = from_date;

  } else if( last_week ) {
    from_date = from_date_now;
    TimAddDays( &from_date, (TimIndexOfDayInWeek( from_date ) - 1) * (-1) );
    TimAddDays( &from_date, (-1) * 7 );
    to_date = from_date;
    TimAddDays( &to_date, 6 );

  } else if( this_week ) {
    from_date = from_date_now;
    TimAddDays( &from_date, (TimIndexOfDayInWeek( from_date ) - 1) * (-1) );
    to_date = from_date;
    TimAddDays( &to_date, 6 );

  } else if( next_week ) {
    from_date = from_date_now;
    TimAddDays( &from_date, (TimIndexOfDayInWeek( from_date ) - 1) * (-1) );
    TimAddDays( &from_date, 7 );
    to_date = from_date;
    TimAddDays( &to_date, 6 );

  } else if( last_month ) {
    from_date = from_date_now;
    from_date = TimMakeTime( TimIndexOfYear(  from_date ),
                             TimIndexOfMonth( from_date ),
                             1, 0, 0, 0 );
    TimAddMonths( &from_date, -1 );
    to_date = from_date;
    TimAddDays( &to_date, TimDaysInMonth( to_date ) - 1 );

  } else if( this_month ) {
    from_date = from_date_now;
    from_date = TimMakeTime( TimIndexOfYear(  from_date ),
                             TimIndexOfMonth( from_date ),
                             1, 0, 0, 0 );
    to_date = from_date;
    TimAddDays( &to_date, TimDaysInMonth( to_date ) - 1 );

  } else if( next_month ) {
    from_date = from_date_now;
    from_date = TimMakeTime( TimIndexOfYear(  from_date ),
                             TimIndexOfMonth( from_date ),
                             1, 0, 0, 0 );
    TimAddMonths( &from_date, 1 );
    to_date = from_date;
    TimAddDays( &to_date, TimDaysInMonth( to_date ) - 1 );

  } else if( last_year ) {
    from_date = from_date_now;
    from_date = TimMakeTime( TimIndexOfYear(  from_date ),
                             1, 1, 0, 0, 0 );
    TimAddYears( &from_date, -1 );
    to_date = from_date;
    TimAddDays( &to_date, TimDaysInYear( to_date ) - 1 );

  } else if( this_year ) {
    from_date = from_date_now;
    from_date = TimMakeTime( TimIndexOfYear(  from_date ),
                             1, 1, 0, 0, 0 );
    to_date = from_date;
    TimAddDays( &to_date, TimDaysInYear( to_date ) - 1 );

  } else if( next_year ) {
    from_date = from_date_now;
    from_date = TimMakeTime( TimIndexOfYear(  from_date ),
                             1, 1, 0, 0, 0 );
    TimAddYears( &from_date, 1 );
    to_date = from_date;
    TimAddDays( &to_date, TimDaysInYear( to_date ) - 1 );

  } else {
    if( strlen( from_date_buffer ) > 0 ) {
      tstatus = TimMakeDateFromString( &from_date, from_date_buffer );

      if( tstatus != TIM_OK ) {
        fprintf( stderr, "%s: Invalid date %s.\n", 
                 program_name, from_date_buffer );
        exit( 1 );
      }
    } else {
      from_date = TimLocalTime( TimMakeTimeNow() );
      from_date = TimMakeTime( TimIndexOfYear(  from_date ),
                               TimIndexOfMonth( from_date ),
                               TimIndexOfDay(   from_date ),
                               0, 0, 0 );
    } /* if */

    if( strlen( to_date_buffer ) > 0 ) {
      tstatus = TimMakeDateFromString( &to_date, to_date_buffer );

      if( tstatus != TIM_OK ) {
        fprintf( stderr, "%s: Invalid date %s.\n", 
                 program_name, to_date_buffer );
        exit( 1 );
      }

    } else {
      to_date = from_date;
    } /* if */

  } /* if */

  /* Check the dates. */
  if( from_date > to_date ) {
    TIM_TIME_REF  temp_date;

    temp_date = from_date;
    from_date = to_date;
    to_date   = temp_date;
  }

  /* Initialize. */
  xtmDbInitializeProcessId();

  /* Use file locking? */
  xtmDbUseFileLock( do_file_lock );

  /* What operations can we do? */
  xtmDbCheckDbOperations( db_info.directory, False, &db_info.operations );

  /* We need at least read. */
  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {
    fprintf( stderr, "%s: You cannot read the requested database.\n",
             program_name );
    exit( 1 );
  }


  /* The file where the tagged data is written. */
  if( ! backup && strlen( output_file ) == 0 )
    strcpy( output_file, "-" );

  if( strlen( output_file ) == 0 ) {
    char_ref = getenv( "HOME" );

    if( char_ref != NULL )
      sprintf( output_file, "%s/XDiary.backup", char_ref );
    else
      sprintf( output_file, "./XDiary.backup" );
  }

  if( strcmp( output_file, "-" ) == 0 )
    file_ref = stdout;
  else
    file_ref = fopen( output_file, "w" );

  if( file_ref == NULL ) {
    fprintf( stderr, "%s: Cannot open file %s for write.\n" ,
             program_name, output_file );
    exit( 1 );
  }


  /* Backup of diary database? */
  if( backup ) {

    UINT32  flags;

    /* Header comments? */
    if( comments ) {
      fprintf( file_ref, "#%%DateRange: " );
      TimFormatIsoDate( from_date, buffer, sizeof( buffer ) );
      fprintf( file_ref, "%s ", buffer );
      TimFormatIsoDate( to_date, buffer, sizeof( buffer ) );
      fprintf( file_ref, "%s\n", buffer );

      fprintf( file_ref, "#%%DateFormat: %s\n", TimWhatDateFormat() );
      fprintf( file_ref, "#%%TimeFormat: %s\n", TimWhatTimeFormat() );

      fprintf( file_ref, "\n" );
    }

    if( repeated && ! repeated_inline ) {
      processRepeated( &db_info, file_ref, 
                       backup,
                       from_date, to_date );
    }

    flags = 0;
    if( repeated_inline )
      flagSet( flags, (XTM_DB_FETCH_STANDING | XTM_DB_FETCH_STICKY) );

    processAppointments( &db_info, file_ref, flags,
                         backup,
                         from_date, to_date );
  }

  /* Display diary database? */
  if( ! backup ) {
    if( repeated )
      processAppointments( &db_info, file_ref, 
                           (XTM_DB_FETCH_STANDING | XTM_DB_FETCH_STICKY),
                           backup,
                           from_date, to_date );
    else
      processAppointments( &db_info, file_ref, 
                           XTM_DB_FETCH_STICKY,
                           backup,
                           from_date, to_date );
  }

  if( strcmp( output_file, "-" ) != 0 )
    fclose( file_ref );


  /* Call display filter? */
  if( ! backup && strlen( filter ) > 0 ) {
      sprintf( buffer, "%s %s", filter, output_file );
      status = system( buffer );
  }

  if( strcmp( output_file, "-" ) != 0 )
    printf( "%s: Output saved in the file %s\n",
            program_name, output_file );


  exit( 0 );

} /* main */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Dumps the contents of an XDiary database.\n"
    "\n"
    "Dumps the selected entries to an ASCII file (default is\n"
    "$HOME/XDiary.backup).\n"
    "\n"
    "Usage:\n"
    "  %s [flags] dbDir\n"
    "\n"
    "where:\n"
    "  dbDir  : Directory containg XDiary database.\n\n"
    "\n"
    "Flags:\n"
    "  -bac[kup]      : Backup the defined diary database.\n"
    "  -com[ments]    : Write comments in the file.\n"
    "  -fil[ter] file : Filter entries through file before they are \n"
    "                   displayed.\n"
    "  -lin[es] n     : Display n lines of each entry.\n"
    "  -noF[ileLock]  : Don't use any file locking.\n"
    "  -noR[epeated]  : Repeated entries are not included.\n"
    "  -out[put] file : Save output in the file file. Default is\n"
    "                   XDiary.backup in your home directory. If you define\n"
    "                   - as the output file, output is written to standard\n"
    "                   output.\n"
    "  -rep[Inline]   : Display repeated inline when doing a backup dump.\n"
    "  -us            : Use US date for the -form and -to date.\n"
    "  -usage         : Print this message.\n"
    "  -version       : Display the current version.\n"
    "\n"
    "  -f[rom] date   : Diaplay entries from this date.\n"
    "  -t[o] date     : Display entries to this date.\n"
    "\n"
    "  -yes[terday]      : Display yesterday's entries.\n"
    "  -tod[ay]          : Display today's entries.\n"
    "  -tom[orrow]       : Display tomorrow's entries.\n"
    "  -lastWeek  [-lw]  : Display last week's entries.\n"
    "  -thisWeek  [-tw]  : Display this week's entries.\n"
    "  -nextWeek  [-nw]  : Display next week's entries.\n"
    "  -lastMonth [-lm]  : Display last month's entries.\n"
    "  -thisMonth [-tm]  : Display this month's entries.\n"
    "  -nextMonth [-nm]  : Display next month's entries.\n"
    "  -lastYear  [-ly]  : Display last year's entries.\n"
    "  -thisYear  [-ty]  : Display this year's entries.\n"
    "  -nextYear  [-ny]  : Display next year's entries.\n"
    "\n",
    program_name, VERSION_ID, program_name );

  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 )
{

  /* Variables. */
  char  *char_ref1;
  char  *char_ref2;


  /* Code. */

  if( strlen( buffer1 ) != strlen( buffer2 ) )
    return( strcmp( buffer1, buffer2 ) );

  char_ref1 = buffer1;
  char_ref2 = buffer2;

  while( *char_ref1 != '\0' ) {
    if( tolower( *char_ref1 ) < tolower( *char_ref2 ) )
      return( -1 );
    else if( tolower( *char_ref1 ) > tolower( *char_ref2 ) )
      return( 1 );

    char_ref1++;
    char_ref2++;
  }

  return( 0 );

} /* noCaseStrcmp */


/*----------------------------------------------------------------------*/

static void
  processAppointments( XTM_CD_CAL_INFO  *db_info,
                       FILE             *file_ref,
                       UINT32           flags,
                       Boolean          backup,
                       TIM_TIME_REF     from_date,
                       TIM_TIME_REF     to_date )
{

  /* Variables. */
  int                     index;
  char                    buffer[ 100 ];
  char                    *char_ref;
  char                    *entry_text;
  TIM_TIME_REF            current_date;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;
  LST_DESC_TYPE           list_ref[ 2 ];
  LST_STATUS              lst_status;


  /* Code. */

  /* Open the entry database. */
  open_request.name       = db_info -> short_name;
  open_request.directory  = db_info -> directory;
  open_request.operations = XTM_DB_FLAG_MODE_READ;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  db_status = xtmDbOpenEntryDb( &open_request, &database );
  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot open database.\n", program_name );
    exit( 1 );
  }


  /* Process all days. */
  current_date = from_date;

  while( current_date <= to_date ) {

    /* Fetch entries this day. */
    db_status = xtmDbFetchEntriesInDay( &database, current_date, 
                                        flags, 
                                        &list_ref[ 1 ], &list_ref[ 0 ] );

    if( db_status != XTM_DB_OK ) {
      fprintf( stderr, "%s: Cannot fetch entries.\n", program_name );
      exit( 1 );
    }

    /* New day mark? */
    if( comments &&
        (LstLinkElements( list_ref[ 0 ] ) > 0 ||
         LstLinkElements( list_ref[ 1 ] ) > 0) ) {
      TimFormatIsoDate( current_date, buffer, sizeof( buffer ) );
      fprintf( file_ref, "#%%NewDay: %s\n", buffer );
    }

    /* Process the notes and entries. */
    for( index = 0; index < 2; index++ ) {

      /* Any elements in the list? */
      if( LstLinkElements( list_ref[ index ] ) > 0 ) {

        /* Process all the elements. */
        lst_status = LstLinkCurrentFirst( list_ref[ index ] );
        while( lst_status == LST_OK ) {

          lst_status = LstLinkGetCurrent( list_ref[ index ], &entry_record );

          /* Fetch the complete entry. */
          db_status = xtmDbFetchEntry( &database, entry_record.entry.id,
                                       &entry_record, &entry_text );

          if( db_status != XTM_DB_OK ) {
            fprintf( stderr, "%s: Cannot fetch entry %d. Not saved!\n",
                     program_name, entry_record.entry.id );

            lst_status = LstLinkCurrentNext( list_ref[ index ] );
            continue;
          }

          if( entry_text != NULL )
            char_ref = entry_text;
          else
            char_ref = entry_record.entry.text;

          /* Repeated entries are not marked. */
          entry_record.entry.entry_category = XTM_DB_ENTRY_LIST;

          /* Produce tagged output */
          printTaggedOutput( db_info, file_ref, 
                             backup,
                             current_date, &entry_record, char_ref );

          /* Produce printed output. */
          if( display_output )
            printDisplayOutput( db_info, file_ref, 
                                current_date, &entry_record, char_ref );

          /* Free text. */
          if( entry_text != NULL )
            SysFree( entry_text );

          /* Next record. */
          lst_status = LstLinkCurrentNext( list_ref[ index ] );

        } /* while */
    
        /* Delete the list. */
        LstLinkClear( list_ref[ index ] );

      } /* if */

    } /* loop */

    /* The next day. */
    TimAddDays( &current_date, 1 );

  } /* while */

  /* Close the database again. */
  xtmDbCloseEntryDb( &database );

  return;

} /* processAppointments */


/*----------------------------------------------------------------------*/

static void
  processRepeated( XTM_CD_CAL_INFO  *db_info,
                   FILE             *file_ref,
                   Boolean          backup,
                   TIM_TIME_REF     from_date,
                   TIM_TIME_REF     to_date )
{

  /* Variables. */
  int                     index;
  char                    *char_ref;
  char                    *entry_text;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;
  LST_DESC_TYPE           list_ref[ 2 ];
  LST_STATUS              lst_status;


  /* Code. */

  /* Open the entry database. */
  open_request.name       = db_info -> short_name;
  open_request.directory  = db_info -> directory;
  open_request.operations = XTM_DB_FLAG_MODE_READ;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  db_status = xtmDbOpenEntryDb( &open_request, &database );
  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot open database.\n", program_name );
    exit( 1 );
  }


  /* Fetch the repeated entries. */
  db_status = xtmDbFetchStandEntries( &database,
                                      &list_ref[ 1 ], &list_ref[ 0 ] );
  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot fetch repeated entries.\n", program_name );
    exit( 1 );
  }

  /* Process the notes and entries. */
  for( index = 0; index < 2; index++ ) {

    /* Any elements in the list? */
    if( LstLinkElements( list_ref[ index ] ) > 0 ) {

      /* Process all the elements. */
      lst_status = LstLinkCurrentFirst( list_ref[ index ] );
      while( lst_status == LST_OK ) {

        lst_status = LstLinkGetCurrent( list_ref[ index ], &entry_record );

        /* Fetch the complete entry. */
        db_status = xtmDbFetchEntry( &database, entry_record.entry.id,
                                     &entry_record, &entry_text );

        if( db_status != XTM_DB_OK ) {
          fprintf( stderr, "%s: Cannot fetch entry entry %d.\n",
                   program_name, entry_record.entry.id );
          exit( 1 );
        }

        if( entry_text != NULL )
          char_ref = entry_text;
        else
          char_ref = entry_record.entry.text;

        /* Repeated entries are not marked when printing. */
        if( ! backup )
          entry_record.entry.entry_category = XTM_DB_ENTRY_LIST;


        /* Produce tagged output */
        printTaggedOutput( db_info, file_ref, 
                           backup,
                           entry_record.entry.date_stamp, 
                           &entry_record, char_ref );

        /* Produce printed output. */
        if( display_output )
          printDisplayOutput( db_info, file_ref, 
                              entry_record.entry.date_stamp,
                              &entry_record, char_ref );

        /* Free text. */
        if( entry_text != NULL )
          SysFree( entry_text );

        /* Next record. */
        lst_status = LstLinkCurrentNext( list_ref[ index ] );

      } /* while */
    
      /* Delete the list. */
      LstLinkClear( list_ref[ index ] );

    } /* if */

  } /* loop */


  /* Close the database again. */
  xtmDbCloseEntryDb( &database );


  return;

} /* processRepeated */


/*----------------------------------------------------------------------*/

static void
  printTaggedOutput( XTM_CD_CAL_INFO       *db_info,
                     FILE                  *file_ref,
                     Boolean               backup,
                     TIM_TIME_REF          entry_date,
                     XTM_DB_ALL_ENTRY_REF  entry_ref,
                     char                  *entry_text )
{

  /* Variables. */
  int   line_size;
  char  buffer[ 50 ];
  char  flags[ 20 ];
  char  alarm_flags[ 20 ];
  char  *char_ref;


  /* Code. */

  if( ! backup )
    return;

  /* Get the flags. */
  flags[ 0 ] = '\0';

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ALARM ) )
    strcat( flags, "A" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_HIDE_IN_ALARM ) )
    strcat( flags, "a" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_KEEP_IN_BG ) )
    strcat( flags, "b" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_HIDE_IN_CALENDAR ) )
    strcat( flags, "c" );

  if( entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE )
    if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_NOTE_DONE ) )
      strcat( flags, "D" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_HIDE_IN_DAY_VIEW ) )
    strcat( flags, "d" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_KEEP_IN_FG ) )
    strcat( flags, "f" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ONLY_OWNER_DELETE ) )
    strcat( flags, "G" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ONLY_OWNER_CHANGE ) )
    strcat( flags, "g" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_IMPORTANT ) )
    strcat( flags, "I" );

  if( entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE )
    strcat( flags, "N" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_PRIVATE ) )
    strcat( flags, "P" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_HIDE_IN_PRINT ) )
    strcat( flags, "p" );

  if( entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST )
    strcat( flags, "S" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_HIDE_IN_SUMMARY ) )
    strcat( flags, "s" );

  if( entry_ref -> entry.entry_category == XTM_DB_STICKY_LIST )
    strcat( flags, "T" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_HIDE_IN_DUMP ) )
    strcat( flags, "u" );


  /* Get the alarm flags. */
  alarm_flags[ 0 ] = '\0';

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ACTION_SCRIPT ) )
    strcat( alarm_flags, "a" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ACTION_TEXT ) )
    strcat( alarm_flags, "t" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ACTION_ALARM1 ) )
    strcat( alarm_flags, "1" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ACTION_ALARM2 ) )
    strcat( alarm_flags, "2" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ACTION_ALARM3 ) )
    strcat( alarm_flags, "3" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ACTION_ALARM4 ) )
    strcat( alarm_flags, "4" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ACTION_ALARM5 ) )
    strcat( alarm_flags, "5" );


  /* Entry start. */
  fprintf( file_ref, "@!XDiaryDump-V1.1\n" );

  TimFormatIsoDate( entry_date, buffer, sizeof( buffer ) );
  fprintf( file_ref, "Ti %s ", buffer );

  TimFormatIsoTime( entry_ref -> entry.time_stamp, buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s ",  buffer );
  fprintf( file_ref, "%d\n", (int) entry_ref -> entry.duration );


  /* Flags (if any). */
  if( flags[ 0 ] != '\0' )
    fprintf( file_ref, "Fl %s\n", flags );

  /* Alarm flags (if any). */
  if( alarm_flags[ 0 ] != '\0' )
    fprintf( file_ref, "Af %s\n", alarm_flags );

  /* Date and time information. */
  TimFormatStrTime( entry_date, "%Y %m %d %w", buffer, sizeof( buffer ) );
  fprintf( file_ref, "Dt %s ", buffer );

  TimFormatStrTime( entry_ref -> entry.time_stamp, "%H %M ", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s ", buffer );

  TimFormatStrTime( entry_ref -> entry.time_stamp + 
                    entry_ref -> entry.duration * 60, "%H %M ", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s ", buffer );

  TimFormatStrTime( entry_ref -> entry.time_stamp, "%I %M %p ", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s ", buffer );

  TimFormatStrTime( entry_ref -> entry.time_stamp + 
                    entry_ref -> entry.duration * 60, "%I %M %p ", 
                    buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s\n", buffer );


  /* Detailed date and time information. */
  fprintf( file_ref, "De %d %d %d\n",
           (int) entry_date, 
           (int) entry_ref -> entry.time_stamp,
           (int) entry_ref -> entry.time_stamp + 
                 entry_ref -> entry.duration * 60 );

  /* Information for repeated entries. */
  if( entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST ||
      entry_ref -> entry.entry_category == XTM_DB_STICKY_LIST )
  {

    UINT32  stand_flags;

    stand_flags = entry_ref -> stand_entry.flags;

    /* From date. */
    if( entry_ref -> stand_entry.from > 0 ) {
      TimFormatIsoDate( entry_ref -> stand_entry.from,
                        buffer, sizeof( buffer ) );
      fprintf( file_ref, "Sf %s\n", buffer );
    }

    /* To date. */
    if( entry_ref -> stand_entry.to > 0 ) {
      TimFormatIsoDate( entry_ref -> stand_entry.to, 
                        buffer, sizeof( buffer ) );
      fprintf( file_ref, "St %s\n", buffer );
    }

    /* Valid a fixed day in the week? */
    if( entry_ref -> stand_entry.every_n == 0 ) {
      fprintf( file_ref, "Sd %d %d %d %d %d %d %d\n",
               (int) entry_ref -> stand_entry.valid_days[ 0 ],
               (int) entry_ref -> stand_entry.valid_days[ 1 ],
               (int) entry_ref -> stand_entry.valid_days[ 2 ],
               (int) entry_ref -> stand_entry.valid_days[ 3 ],
               (int) entry_ref -> stand_entry.valid_days[ 4 ],
               (int) entry_ref -> stand_entry.valid_days[ 5 ],
               (int) entry_ref -> stand_entry.valid_days[ 6 ] );

      if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_DAY_IN_MONTH ) ) {
        fprintf( file_ref, "Sr " );

        if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_2ND ) )
          fprintf( file_ref, "b\n" );
        else if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_3RD ) )
          fprintf( file_ref, "c\n" );
        else if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_1ST_LAST ) )
          fprintf( file_ref, "d\n" );
        else if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_2ND_LAST ) )
          fprintf( file_ref, "e\n" );
        else if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_3RD_LAST ) )
          fprintf( file_ref, "f\n" );
        else
          fprintf( file_ref, "a\n" );
      }
    }

    /* Valid every n? */
    if( entry_ref -> stand_entry.every_n > 0 ) {
      fprintf( file_ref, "Sn %d ", (int) entry_ref -> stand_entry.every_n );

      switch( entry_ref -> stand_entry.valid_every ) {
        case XTM_DB_VALID_DAY:
          char_ref = "D";
          break;
        case XTM_DB_VALID_WEEK:
          char_ref = "W";
          break;
        case XTM_DB_VALID_MONTH:
          char_ref = "M";
          break;
        case XTM_DB_VALID_MONTH_LAST:
          char_ref = "L";
          break;
        case XTM_DB_VALID_YEAR:
          char_ref = "Y";
          break;
        default:
          char_ref = "-";
          break;
      } /* switch */

      fprintf( file_ref, "%s\n", char_ref );
    }

    /* Skip weeks? */
    if( entry_ref -> stand_entry.skip_week[ 0 ] != 0 ||
        entry_ref -> stand_entry.skip_week[ 1 ] != 0 ) {

      int     index;
      UINT32  flags;
      UINT32  skip_flag;

      fprintf( file_ref, "Sw " );

      for( index = 1; index <= 53; index++ ) {
        skip_flag = (1 << (index % 30));
        if( index > 30 )
          flags = entry_ref -> stand_entry.skip_week[ 0 ];
        else
          flags = entry_ref -> stand_entry.skip_week[ 1 ];

        if( flagIsSet( flags, skip_flag ) )
          fprintf( file_ref, "%d ", index );

      } /* lopp */

      fprintf( file_ref, "\n" );

    } /* if */

    /* Action on non-workdays? */
    if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_NWDAY_SKIP ) )
      fprintf( file_ref, "Sy s\n" );
    else if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_NWDAY_NEXT ) )
      fprintf( file_ref, "Sy n\n" );
    else if( flagIsSet( stand_flags, XTM_DB_FLAG_SE_NWDAY_PREV ) )
      fprintf( file_ref, "Sy p\n" );

  } /* if */


  /* The alarm offsets. */
  fprintf( file_ref, "Al %d %d %d %d %d %d %d %d %d %d\n",
           (int) entry_ref -> entry.alarm_valid[  0 ], 
           (int) entry_ref -> entry.alarm_offset[ 0 ],
           (int) entry_ref -> entry.alarm_valid[  1 ], 
           (int) entry_ref -> entry.alarm_offset[ 1 ],
           (int) entry_ref -> entry.alarm_valid[  2 ], 
           (int) entry_ref -> entry.alarm_offset[ 2 ],
           (int) entry_ref -> entry.alarm_valid[  3 ], 
           (int) entry_ref -> entry.alarm_offset[ 3 ],
           (int) entry_ref -> entry.alarm_valid[  4 ], 
           (int) entry_ref -> entry.alarm_offset[ 4 ] );


  /* The alarm tunes. */
  fprintf( file_ref, "Tu %d\n", entry_ref -> entry.alarm_melody );


  /* Lines to show. */
  fprintf( file_ref, "Li %d %d\n", 
           entry_ref -> entry.day_list_lines, 
           entry_ref -> entry.alarm_lines );


  /* Background and foreground colors. */
  fprintf( file_ref, "Cl %d %d\n", 
           (int) entry_ref -> entry.bg_color_index, 
           (int) entry_ref -> entry.fg_color_index );


  /* Entry tag. */
  if( strlen( entry_ref -> entry.tag ) > 0 )
    fprintf( file_ref, "Tg %s\n", entry_ref -> entry.tag );


  /* Write each line in the entry. */
  char_ref = entry_text;

  while( *char_ref != '\0' ) {
    line_size = strcspn( char_ref, "\n" );

    fprintf( file_ref, "<<" );
    fwrite(  char_ref, 1, line_size, file_ref );
    fprintf( file_ref, "\n" );

    char_ref = char_ref + line_size;
    if( *char_ref == '\0' )
      break;

    /* Skip the \n character. */
    char_ref ++;
  } /* while */

  fprintf( file_ref, "$$\n\n" );

  return;

} /* printTaggedOutput */


/*----------------------------------------------------------------------*/

static void
  printDisplayOutput( XTM_CD_CAL_INFO       *db_info,
                      FILE                  *file_ref,
                      TIM_TIME_REF          entry_date,
                      XTM_DB_ALL_ENTRY_REF  entry_ref,
                      char                  *entry_text )
{

  /* Variables. */
  int   line_count = 0;
  int   line_size;
  char  buffer[ 50 ];
  char  *char_ref;


  /* Code. */

  /* Skip this entry? */
  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_HIDE_IN_DUMP ) )
    return;


  TimFormatStrTime( entry_date, "%B %d %Y, %A", buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s  ", buffer );

  if( entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE )
    strcpy( buffer, "Note" );
  else
    TimFormatIsoTime( entry_ref -> entry.time_stamp,
                      buffer, sizeof( buffer ) );
  fprintf( file_ref, "%s   ", buffer );

  /* Entry tag. */
  if( strlen( entry_ref -> entry.tag ) > 0 )
    fprintf( file_ref, "(%s)", entry_ref -> entry.tag );

  fprintf( file_ref, "\n" );

  /* Is this a private entry? */
  if(   flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_PRIVATE ) &&
      ! flagIsSet( db_info -> operations,    XTM_DB_FLAG_MODE_PRIV ) ) {
    strcpy( buffer, "(Private)" );
    char_ref = buffer;
  } else {
    char_ref = entry_text;
  }

  /* Write each line in the entry. */
  while( *char_ref != '\0' && line_count < lines ) {
    line_size = strcspn( char_ref, "\n" );

    fprintf( file_ref, "  " );
    fwrite(  char_ref, 1, line_size, file_ref );
    fprintf( file_ref, "\n" );

    char_ref = char_ref + line_size;
    if( *char_ref == '\0' )
      break;

    /* Skip the \n character. */
    char_ref ++;
    line_count++;
  } /* while */

  fprintf( file_ref, "\n" );

  return;

} /* printDisplayOutput */
