/*----------------------------------------------------------------------------
--
--  Module:           xtmSchedAct
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--   Actions to copy, move and delete in the schedule window.
--
--  Filename:         xtmSchedAct.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmSchedAct.c, Version: 1.1, Date: 95/02/18 15:52:43";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>

#include "LstLinked.h"
#include "Message.h"
#include "System.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCpMvEntry.h"
#include "xtmDbMisc.h"
#include "xtmEditEntry.h"
#include "xtmFormat.h"
#include "xtmFreeze.h"
#include "xtmIcons.h"
#include "xtmSchedTool.h"
#include "xtmSchedWin.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmEntryClip.h"
#include "xtmDelEntry.h"
#include "xtmDuplEntry.h"
#include "xtmArchive.h"


#include "xtmSchedPriv.h"
#include "xtmSchedAct.h"

#include "xtmStipple.bm"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Copy/move operations. */
typedef enum { CM_MOVE, CM_COPY } CM_OPERATION;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmSchdAction";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  animateActionMoveCopyCB( SCHED_REC_REF  sched_ref,
                           XEvent         *event );

static void
  animateActionNewDurationCB( SCHED_REC_REF  sched_ref,
                              XEvent         *event );

static void
  animateActionNewEntryCB( SCHED_REC_REF  sched_ref,
                           XEvent         *event );

static void 
  appointmentEditorCB( XTM_ED_REASON reason,
                       void          *user_data );

static void
  changeDurationCB( Widget         widget,
                    SCHED_REC_REF  sched_ref,
                    XtPointer      call_data );

static Widget
  createEntryEditPopup( SCHED_REC_REF  sched_ref,
                        Widget         parentW );

static Widget
  createEntryTextPopup( SCHED_REC_REF  sched_ref,
                        Widget         parentW );

static void
  displayEntryEditPopup( SCHED_REC_REF  sched_ref,
                         XEvent         *event );

static void
  displayEntryTextPopup( SCHED_REC_REF  sched_ref,
                         XEvent         *event );

static void
  drawAnimateTrace( Widget  widget,
                    int     x1,
                    int     y1,
                    int     x2,
                    int     y2 );

static void 
  entryPuMenuCB( Widget                     widget,
                 SCHED_REC_REF              sched_ref,
                 XmRowColumnCallbackStruct  *call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmSaActionAppHook( Widget         widget,
                      SCHED_REC_REF  sched_ref,
                      XEvent         *event )
{

  /* Code. */

  if( event -> xany.type != ButtonPress )
    return;


  /* Button 1 is New Entry if pointer on entryBB. */
  if( event -> xbutton.button == 1 ) {

    int       day_number;
    Position  pos_x;
    Widget    mainW;
    Widget    bgW;

    mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

    /* React only on events on the background. */
    bgW = XtNameToWidget( mainW, 
      "SchedPa.Pane2Fo.EntrySw.ClipWindow.EntryBb" );

    if( bgW != widget )
      return;

    /* Define new entry? */
    if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
      return;

    if( sched_ref -> selected_widget != NULL )
      return;

    /* Data for the animation. */
    day_number = (int) event -> xbutton.x / (int) sched_ref -> day_width;
    pos_x      = (Position) (sched_ref -> day_width * day_number + 6);

    /* Start the animation. */
    xtmSaStartAnimation( sched_ref, event, (ANI_ENTRY | ANI_NEW_ENTRY ),
                         bgW, pos_x, 0,
                         sched_ref -> day_width - 14, 0 );

  } /* if */


  /* Button 3 is popup menu. */
  if( event -> xbutton.button == 3 ) {

    /* Entry edit popup? */
    if( flagIsClear( event -> xbutton.state, ShiftMask ) )
      displayEntryEditPopup( sched_ref, event );

    /* Entry text popup? */
    if( flagIsSet( event -> xbutton.state, ShiftMask ) )
      displayEntryTextPopup( sched_ref, event );

  } /* if */


  return;

} /* xtmSaActionAppHook */


/*----------------------------------------------------------------------*/

void
  xtmSaActionNoteHook( Widget         widget,
                       SCHED_REC_REF  sched_ref,
                       XEvent         *event )
{

  /* Code. */

  if( event -> xany.type != ButtonPress )
    return;


  /* Button 3 is popup menu. */
  if( event -> xbutton.button == 3 ) {

    /* Entry edit popup? */
    if( flagIsClear( event -> xbutton.state, ShiftMask ) )
      displayEntryEditPopup( sched_ref, event );

    /* Entry text popup? */
    if( flagIsSet( event -> xbutton.state, ShiftMask ) )
      displayEntryTextPopup( sched_ref, event );

  } /* if */


  return;

} /* xtmSaActionNoteHook */


/*----------------------------------------------------------------------*/

void 
  xtmSaAppointmentEditorShow( SCHED_REC_REF  sched_ref )
{


  /* Variables. */
  Widget                  mainW;
  ENTRY_INFO              *entry_info_ref;
  TIM_TIME_REF            default_time;
  TIM_TIME_REF            use_date;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  default_time   = TimMakeTime( 1970, 1, 1, 12, 0, 0 );
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );


  /* The database we are editing. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            sched_ref -> db_name,
                            &db_info, NULL );


  if( sched_ref -> editor_handle == NULL )
    sched_ref -> editor_handle = 
      xtmEdInitialize( appl_data_ref,
                       appl_data_ref   -> toplevel,
                       sched_ref       -> entry_delta,
                       custom_data_ref -> start_hour,
                       custom_data_ref -> stop_hour,
                       custom_data_ref -> default_entry_delta,
                       appointmentEditorCB, 
                       (void *) sched_ref );

  if( entry_info_ref == NULL ) {

    use_date = sched_ref -> schedule_start;
    xtmEdEditEntry( sched_ref -> editor_handle,
                    db_info.short_name,
                    0,
                    use_date, default_time );

  } else {
    xtmEdEditEntry( sched_ref      -> editor_handle,
                    entry_info_ref -> db_name,
                    entry_info_ref -> id,
                    entry_info_ref -> date_stamp, 
                    entry_info_ref -> time_stamp );

    xtmSwUnselectEntry( sched_ref );
  } /* if */


  return;

} /* xtmSaAppointmentEditorShow */


/*----------------------------------------------------------------------*/

void 
  xtmSaAppointmentEditorShowFor( SCHED_REC_REF  sched_ref,
                                 UINT32         entry_id,
                                 char           *calendar,
                                 TIM_TIME_REF   as_date,
                                 TIM_TIME_REF   as_time )
{


  /* Variables. */
  Widget                  mainW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  if( sched_ref -> editor_handle == NULL )
    sched_ref -> editor_handle = 
      xtmEdInitialize( appl_data_ref,
                       appl_data_ref   -> toplevel,
                       sched_ref       -> entry_delta,
                       custom_data_ref -> start_hour,
                       custom_data_ref -> stop_hour,
                       custom_data_ref -> default_entry_delta,
                       appointmentEditorCB, 
                       (void *) sched_ref );

  /* Edit the entry. */
  xtmEdEditEntry( sched_ref -> editor_handle,
                  calendar, entry_id, as_date, as_time );


  return;

} /* xtmSaAppointmentEditorShowFor */


/*----------------------------------------------------------------------*/

void
  xtmSaDoChangeFlags( SCHED_REC_REF  sched_ref,
                      char           *db_name,
                      int            entry_id,
                      TIM_TIME_REF   date_stamp,
                      UINT32         flags,
                      Boolean        set_flags )
{

  /* Variables. */
  Boolean                 ok;
  char                    *entry_text = NULL;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;


  /* Code. */

  /* Open the database for write access. */
  ok = xtmDmOpenDatabase( sched_ref -> appl_data_ref,
                          db_name, XTM_DB_FLAG_MODE_WRITE,
                          &database );
  if( ! ok )
    return;


  /* Fetch the entry to change. */
  ok = xtmDmFetchEntry( sched_ref -> scheduleW,
                        &database,     entry_id,
                        &entry_record, &entry_text );
  if( ! ok )
    raise exception;


  /* Make sure the date is correct for notes marked as done. */
  if( set_flags && flagIsSet( flags, XTM_DB_FLAG_NOTE_DONE ) )
    entry_record.entry.date_stamp = date_stamp;


  /* Set or clear the flags. */
  if( set_flags )
    flagSet( entry_record.entry.flags, flags );
  else
    flagClear( entry_record.entry.flags, flags );


  /* Save the modified entry. */
  (void) xtmDbDeleteEntry( &database, entry_id );

  ok = xtmDmInsertEntry( sched_ref -> scheduleW,
                         &database,     entry_id,
                         &entry_record, entry_text );
  if( ! ok )
    raise exception;

  if( entry_text != NULL )
    SysFree( entry_text );


  /* That's it. */
  xtmDbCloseEntryDb( &database );


  /* Update the schedule/calendar. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE), NULL );


  return;


  /* Exception handler. */
  exception:
    if( entry_text != NULL )
      SysFree( entry_text );

    xtmDbCloseEntryDb( &database );

    return;

} /* xtmSaDoChangeFlags */


/*----------------------------------------------------------------------*/

void
  xtmSaDoCutBuffer( SCHED_REC_REF         sched_ref,
                    XTM_SA_CUT_OPERATION  operation,
                    char                  *db_name,
                    UINT32                entry_id )
{

  /* Variables. */
  Boolean                 ok;
  UINT32                  access_mode;
  char                    *entry_text;
  TIM_TIME_REF            time_tmp;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;


  /* Code. */

  /* What kind of access do we need? */
  if( operation == XTM_SA_DO_CUT || operation == XTM_SA_DO_PASTE )
    access_mode = XTM_DB_FLAG_MODE_WRITE;
  else
    access_mode = XTM_DB_FLAG_MODE_READ;


  /* Open the database (we need this for all actions). */
  ok = xtmDmOpenDatabase( sched_ref -> appl_data_ref,
                          db_name, access_mode,
                          &database );
  if( ! ok )
    return;


  /* For Copy/Cut, fetch the selected entry and paste into the clipboard. */
  if( operation == XTM_SA_DO_COPY || operation == XTM_SA_DO_CUT ) {

    ok = xtmDmFetchEntry( sched_ref -> scheduleW,
                          &database,     entry_id,
                          &entry_record, &entry_text );
    if( ! ok )
      raise exception;

    /* Not archived. */
    flagClear( entry_record.entry.flags, XTM_DB_FLAG_ARCHIVED );

    xtmEcSaveEntry( &entry_record, entry_text );
    if( entry_text != NULL )
      SysFree( entry_text );

  } /* if */


  /* For CUT, remove the selected entry. */
  if( operation == XTM_SA_DO_CUT ) {

    ok = xtmDmDeleteEntry( sched_ref -> scheduleW,
                           &database, entry_id );
    if( ! ok )
      raise exception;

  } /* if */


  /* For Paste, fetch entry from clipboard and paste into the daylist. */
  if( operation == XTM_SA_DO_PASTE ) {

    ok = xtmEcRetrieveEntry( &entry_record, &entry_text );
    if( ! ok )
      raise exception;

    /* Pick a date for paste. */
    if( entry_record.entry.entry_type == XTM_DB_DAY_NOTE )
      ok = xtmStPickDate( sched_ref, XTM_ST_PICK_NOTE, &time_tmp );
    else
      ok = xtmStPickDate( sched_ref, XTM_ST_PICK_APP, &time_tmp );

    if( ! ok )
      return;

    /* Clear start dates for repeated entries. */
    entry_record.entry.date_stamp = (UINT32) time_tmp;
    entry_record.stand_entry.from = (UINT32) time_tmp;
    entry_record.stand_entry.to   = 0;

    /* Modify starting day for repeated entries. */
    if( entry_record.entry.entry_category == XTM_DB_REP_ENTRY_LIST ||
        entry_record.entry.entry_category == XTM_DB_STICKY_LIST ) {
      entry_record.stand_entry.from = (UINT32) time_tmp;

      if( entry_record.stand_entry.to > 0 &&
          entry_record.stand_entry.to < entry_record.stand_entry.from )
        entry_record.stand_entry.to == 0;
    }

    /* Save the entry. */
    ok = xtmDmInsertEntry( sched_ref -> scheduleW,
                           &database,     entry_id,
                           &entry_record, entry_text );
    if( ! ok )
      raise exception;
  
  } /* if */


  /* That's it. */
  xtmDbCloseEntryDb( &database );


  /* Update the schedule/calendar. */
  if( operation != XTM_SA_DO_COPY )
    xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER),
                   NULL );


  return;


  /* Exception handler. */
  exception:
    xtmDbCloseEntryDb( &database );

    return;

} /* xtmSaDoCutBuffer */


/*----------------------------------------------------------------------*/

void
  xtmSaDoNotThisWeek( SCHED_REC_REF  sched_ref,
                      char           *db_name,
                      int            entry_id,
                      TIM_TIME_REF   date_stamp )
{

  /* Variables. */
  Boolean                 ok;
  int                     week_no;
  UINT32                  *flags;
  UINT32                  skip_flag;
  char                    *entry_text = NULL;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;


  /* Code. */

  /* Open the database for write access. */
  ok = xtmDmOpenDatabase( sched_ref -> appl_data_ref,
                          db_name, XTM_DB_FLAG_MODE_WRITE,
                          &database );
  if( ! ok )
    return;


  /* Fetch the entry to change. */
  ok = xtmDmFetchEntry( sched_ref -> scheduleW,
                        &database,     entry_id,
                        &entry_record, &entry_text );
  if( ! ok )
    raise exception;

  /* Only for repeated entries. */
  if( entry_record.entry.entry_category != XTM_DB_REP_ENTRY_LIST )
    raise exception;


  /* Mark the week as 'not done'. */
  week_no = TimIndexOfWeek( date_stamp );

  skip_flag = (1 << (week_no % 30));
  if( week_no  > 30 )
    flags = &entry_record.stand_entry.skip_week[ 0 ];
  else
    flags = &entry_record.stand_entry.skip_week[ 1 ];

  flagSet( *flags, skip_flag );


  /* Save the modified entry. */
  (void) xtmDbDeleteEntry( &database, entry_id );

  ok = xtmDmInsertEntry( sched_ref -> scheduleW,
                         &database,     entry_id,
                         &entry_record, entry_text );
  if( ! ok )
    raise exception;

  if( entry_text != NULL )
    SysFree( entry_text );


  /* That's it. */
  xtmDbCloseEntryDb( &database );


  /* Update the schedule/calendar. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;


  /* Exception handler. */
  exception:
    if( entry_text != NULL )
      SysFree( entry_text );

    xtmDbCloseEntryDb( &database );

    return;

} /* xtmSaDoNotThisWeek */


/*----------------------------------------------------------------------*/

void
  xtmSaMapCtrlMenuCB( Widget         widget,
                      SCHED_REC_REF  sched_ref,
                      XtPointer      call_data )
{

  /* Variables. */
  Widget  tempW;


  /* Code. */

  /* Confirm actions? */
  tempW = XtNameToWidget( widget, "ConfirmBu" );
  if( flagIsSet( sched_ref -> flags, XTM_SM_CONFIRM ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  /* Entry handles? */
  tempW = XtNameToWidget( widget, "HandleBu" );
  if( flagIsSet( sched_ref -> flags, XTM_SM_ENTRY_HANDLES ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  /* List layout? */
  tempW = XtNameToWidget( widget, "ListLayBu" );
  if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  /* Use grid? */
  tempW = XtNameToWidget( widget, "UseGridBu" );
  if( flagIsSet( sched_ref -> flags, XTM_SM_USE_GRID ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  /* Display entry flags? */
  tempW = XtNameToWidget( widget, "EntryFlagsBu" );
  if( flagIsSet( sched_ref -> flags, XTM_SM_ENTRY_FLAGS ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );

  /* True colors for included entries? */
  tempW = XtNameToWidget( widget, "TrueColorIncBu" );
  if( flagIsSet( sched_ref -> flags, XTM_SM_TRUE_COLOR_INC ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  return;

} /* xtmSaMapCtrlMenu */


/*----------------------------------------------------------------------*/

void
  xtmSaMapEditMenuCB( Widget         widget,
                      SCHED_REC_REF  sched_ref,
                      XtPointer      call_data )
{

  /* Variables. */
  int                     index;
  UINT32                  can_do;
  Boolean                 db_write;
  Boolean                 ok;
  Cardinal                num_children;
  Widget                  tempW;
  WidgetList              children;
  ENTRY_INFO              *entry_info_ref;
  XTM_CD_CAL_INFO         db_info;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Assume we cannot do anything. */
  (void) xitGetChildren( widget, &num_children, &children );

  for( index = 0; index < num_children; index++ ) {
    if( XmIsPushButton( *(children + index) ) )
      XtSetSensitive( *(children + index), False );
  }


  /* Read only mode? */
  if( flagIsSet( sched_ref -> flags, XTM_SM_READ_ONLY ) )
    return;


  /* Information about the current DB. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                          sched_ref -> db_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  db_write = flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE );

  /* We can always edit. */
  tempW = XtNameToWidget( widget, "EditBu" );
  XtSetSensitive( tempW, True );


  /* If we han an cut buffer entry, can we paste it? */
  if( db_write && ! xtmEcIsClipboardEmpty() ) {
    tempW = XtNameToWidget( widget, "PasteBufBu" );
    XtSetSensitive( tempW, True );
  }

  /* Fetch reference to selected widget. */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
  if( entry_info_ref == NULL )
    return;

  /* Information about the entry DB. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          entry_info_ref -> db_name,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* What can we do with the entry. */
  xtmDbGetEntryPermissions( db_info.operations,
                            entry_info_ref -> uid,
                            entry_info_ref -> flags,
                            &can_do );


  if( flagIsClear( can_do, (XTM_DB_PROT_READ | XTM_DB_PROT_WRITE) ) )
    return;


  /* If we can read, what can we do? */
  if( flagIsSet( can_do, XTM_DB_PROT_READ ) ) {

    tempW = XtNameToWidget( widget, "CopyBu" );
    XtSetSensitive( tempW, True );

    tempW = XtNameToWidget( widget, "DuplBu" );
    XtSetSensitive( tempW, True );

    if( flagIsClear( entry_info_ref -> flags, XTM_DB_FLAG_ARCHIVED ) ) {
      tempW = XtNameToWidget( widget, "ArchBu" );
      XtSetSensitive( tempW, True );
    }

    tempW = XtNameToWidget( widget, "InfoBu" );
    XtSetSensitive( tempW, True );

    tempW = XtNameToWidget( widget, "CopyBufBu" );
    XtSetSensitive( tempW, True );
  }


  /* If we can write, what can we do? */
  if( flagIsSet( can_do, XTM_DB_PROT_WRITE ) ) {

    if( flagIsSet( can_do, XTM_DB_PROT_DELETE ) ) {
      tempW = XtNameToWidget( widget, "DeleteBu" );
      XtSetSensitive( tempW, True );
    }

    if( flagIsSet( can_do, XTM_DB_PROT_CHANGE ) ) {
      tempW = XtNameToWidget( widget, "MoveBu" );
      XtSetSensitive( tempW, True );

      if( entry_info_ref -> category == XTM_DB_REP_ENTRY_LIST ) {
        tempW = XtNameToWidget( widget, "FreezeBu" );
        XtSetSensitive( tempW, True );
      }

      tempW = XtNameToWidget( widget, "CutBufBu" );
      XtSetSensitive( tempW, True );
    }
  }


  return;

} /* xtmSaMapEditMenu */


/*----------------------------------------------------------------------*/

void
  xtmSaMapEntryPuMenuCB( Widget         widget,
                         SCHED_REC_REF  sched_ref,
                         XtPointer      call_data )
{

  /* Variables. */
  int                     index;
  UINT32                  can_do;
  Boolean                 db_write;
  Boolean                 ok;
  Cardinal                num_children;
  Widget                  tempW;
  WidgetList              children;
  ENTRY_INFO              *entry_info_ref;
  XTM_CD_CAL_INFO         db_info;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Assume we cannot do anything. */
  (void) xitGetChildren( widget, &num_children, &children );

  for( index = 0; index < num_children; index++ ) {
    if( XmIsPushButton( *(children + index) ) )
      XtSetSensitive( *(children + index), False );
  }


  /* Information about the current DB. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                          sched_ref -> db_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  db_write = flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE );

  /* We can always edit. */
  tempW = XtNameToWidget( widget, "EditBu" );
  XtSetSensitive( tempW, True );


  /* Read only mode? */
  if( flagIsSet( sched_ref -> flags, XTM_SM_READ_ONLY ) )
    return;


  /* If we han an cut buffer entry, can we paste it? */
  if( db_write && ! xtmEcIsClipboardEmpty() ) {
    tempW = XtNameToWidget( widget, "PasteBufBu" );
    XtSetSensitive( tempW, True );
  }


  /* Fetch reference to selected widget. */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
  if( entry_info_ref == NULL )
    return;

  /* Information about the entry DB. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          entry_info_ref -> db_name,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* What can we do with the entry. */
  xtmDbGetEntryPermissions( db_info.operations,
                            entry_info_ref -> uid,
                            entry_info_ref -> flags,
                            &can_do );


  if( flagIsClear( can_do, (XTM_DB_PROT_READ | XTM_DB_PROT_WRITE) ) )
    return;


  /* If we can read, what can we do? */
  if( flagIsSet( can_do, XTM_DB_PROT_READ ) ) {

    tempW = XtNameToWidget( widget, "CopyBu" );
    XtSetSensitive( tempW, True );

    tempW = XtNameToWidget( widget, "DuplBu" );
    XtSetSensitive( tempW, True );

    tempW = XtNameToWidget( widget, "CopyBufBu" );
    XtSetSensitive( tempW, True );

    if( flagIsClear( entry_info_ref -> flags, XTM_DB_FLAG_ARCHIVED ) ) {
      tempW = XtNameToWidget( widget, "ArchBu" );
      XtSetSensitive( tempW, True );
    }

  } /* if */


  /* If we can write, what can we do? */
  if( flagIsSet( can_do, XTM_DB_PROT_WRITE ) ) {

    if( flagIsSet( can_do, XTM_DB_PROT_DELETE ) ) {
      tempW = XtNameToWidget( widget, "DeleteBu" );
      XtSetSensitive( tempW, True );
    }

    if( flagIsSet( can_do, XTM_DB_PROT_CHANGE ) ) {
      tempW = XtNameToWidget( widget, "MoveBu" );
      XtSetSensitive( tempW, True );

      if( entry_info_ref -> category == XTM_DB_REP_ENTRY_LIST ) {
        tempW = XtNameToWidget( widget, "NotWeekBu" );
        XtSetSensitive( tempW, True );

        tempW = XtNameToWidget( widget, "FreezeBu" );
        XtSetSensitive( tempW, True );
      }

      if( entry_info_ref -> type == XTM_DB_DAY_NOTE &&
          flagIsClear( entry_info_ref -> flags, XTM_DB_FLAG_NOTE_DONE ) ) {
        tempW = XtNameToWidget( widget, "NoteDoneBu" );
        XtSetSensitive( tempW, True );
      }

      tempW = XtNameToWidget( widget, "CutBufBu" );
      XtSetSensitive( tempW, True );
    }

  } /* if */


  return;

} /* xtmSaMapEntryPuMenuCB */


/*----------------------------------------------------------------------*/

void
  xtmSaMapFileMenuCB( Widget         widget,
                      SCHED_REC_REF  sched_ref,
                      XtPointer      call_data )
{

  /* Variables. */
  Boolean                 ok;
  Widget                  tempW;
  XTM_CD_CAL_INFO         db_info;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Assume we cannot do anything. */
  tempW = XtNameToWidget( widget, "UploadBu" );
  XtSetSensitive( tempW, False );

  tempW = XtNameToWidget( widget, "DownloadBu" );
  XtSetSensitive( tempW, False );


  /* Information about the current DB. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                          sched_ref -> db_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  if( flagIsClear( db_info.flags, XTM_CD_FLAG_IS_EXTERNAL ) )
    return;


  /* Can upload? */
  tempW = XtNameToWidget( widget, "UploadBu" );
  XtSetSensitive( tempW, True );


  /* Can download? */
  tempW = XtNameToWidget( widget, "DownloadBu" );
  if( flagIsClear( sched_ref -> flags, XTM_SM_READ_ONLY ) &&
      flagIsSet(   db_info.operations, XTM_DB_FLAG_MODE_WRITE ) )
    XtSetSensitive( tempW, True );


  return;

} /* xtmSaMapFileMenuCB */


/*----------------------------------------------------------------------*/

void
  xtmSaStartAnimation( SCHED_REC_REF  sched_ref,
                       XEvent         *event,
                       UINT32         flags,
                       Widget         animateBaseW,
                       Position       start_x,
                       Position       start_y,
                       Dimension      marker_width,
                       Dimension      marker_height )
{

  /* Variables. */
  Boolean                 ok;
  char                    buffer[ 200 ];
  int                     new_delta_y;
  Widget                  mainW;
  ANIMATE_REC_REF         animate_ref;
  ENTRY_INFO              *entry_info_ref = NULL;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;

  static XIT_PUSH_STRUCT button_def[] = {
    { "AnimatePb", " ", "", False, NULL }
  };


  /* Code. */

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  animate_ref     = sched_ref -> animate_ref;
  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;

  /* If we are already doing animation, don't start again. */
  if( animate_ref -> animate_on )
    return;

  /* Display the time or not. */
  if( flagIsClear( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
    animate_ref -> display_time = True;
  else
    animate_ref -> display_time = False;


  /* Move or copy an existing entry? */
  if( flagIsSet( flags, (ANI_MOVE_COPY | ANI_NEW_DURATION) ) ) {

    entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
    if( entry_info_ref == NULL )
      return;

    /* We can only move normal entries. */
    if( flagIsSet( flags, ANI_MOVE_COPY ) ) {
      if( entry_info_ref -> category != XTM_DB_ENTRY_LIST ) {
        xitErMessage( sched_ref -> scheduleW, XIT_ER_ERROR,
                      module_name, "xtmSaStartAnimation",
                      msgGetText( MXDI_CANNOT_COPY_MOVE_STAND ) );
        return;
      } 
    }

    /* We can only change duration for appointments. */
    if( flagIsSet( flags, ANI_NEW_DURATION ) ) {
      if( entry_info_ref -> type != XTM_DB_DAY_ENTRY )
        return;
    }

    /* Do you have write permission? */
    ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            entry_info_ref -> db_name,
                            &db_info, NULL );
    if( ! ok )
      return;

    if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE ) ) {
      sprintf( buffer, 
               msgGetText( MXDI_ERRMSG_NO_ACCESS_NAMED_DB ), 
               db_info.short_name );

      xitErMessage( sched_ref -> scheduleW, XIT_ER_ERROR, 
                    module_name, "xtmSaStartAnimation",
                    buffer );
      return;
    }    

    /* Do not display the time when moving notes. */
    if( entry_info_ref -> type == XTM_DB_DAY_NOTE )
      animate_ref -> display_time = False;

  } /* if */


  /* Do we have the animation widgets we need? */
  if( flagIsSet( flags, ANI_TIME ) ) {
    if( animate_ref -> aniTimeW == NULL )
      animate_ref -> aniTimeW = xitCreatePushButton( animateBaseW,
                                                     &button_def[ 0 ] );
    animate_ref -> animateW = animate_ref -> aniTimeW;
  }

  if( flagIsSet( flags, ANI_NOTE ) ) {
    if( animate_ref -> aniNoteW == NULL )
      animate_ref -> aniNoteW = xitCreatePushButton( animateBaseW,
                                                     &button_def[ 0 ] );
    animate_ref -> animateW = animate_ref -> aniNoteW;
  }

  if( flagIsSet( flags, ANI_ENTRY ) ) {
    if( animate_ref -> aniEntryW == NULL )
      animate_ref -> aniEntryW = xitCreatePushButton( animateBaseW,
                                                      &button_def[ 0 ] );
    animate_ref -> animateW = animate_ref -> aniEntryW;
  }


  /* Initialize the animation button. */
  {
    Arg       args[ 10 ];
    Cardinal  n;

    /* We tell Motif how big the button is. */
    n = 0;
    XtSetArg( args[ n ], XmNhighlightThickness, 0 ); n++;
    XtSetArg( args[ n ], XmNrecomputeSize, False ); n++;
    XtSetValues( animate_ref -> animateW, args, n );
  }


  /* "Grid" to use (always use at least a 5 minutes grid). */
  if( flagIsSet( sched_ref -> flags, XTM_SM_USE_GRID ) )
    animate_ref -> grid_minutes = sched_ref -> entry_delta / 2.0;
  else
    animate_ref -> grid_minutes = 5.0;


  /* Default start and size to use. */
  animate_ref -> start_x = event -> xbutton.x;
  animate_ref -> start_y = event -> xbutton.y;

  animate_ref -> width  = (int) marker_width;
  animate_ref -> height = (int) marker_height;


  /* Start day and time. */
  xtmStPosXToDate( sched_ref,
                   animate_ref  -> start_x,
                   &animate_ref -> start_date,
                   &animate_ref -> start_x );

  xtmStPosYToTime( sched_ref,
                   animate_ref  -> start_y,
                   animate_ref  -> grid_minutes,
                   &animate_ref -> start_time,
                   &animate_ref -> start_y );

  animate_ref -> end_x = animate_ref -> start_x + (int) marker_width;
  animate_ref -> end_y = animate_ref -> start_y + (int) marker_height;


  /* Move/Copy entry? */
  if( flagIsSet( flags, ANI_MOVE_COPY ) ) {

    animate_ref -> start_x = -1;
    animate_ref -> start_y = -1;
    animate_ref -> end_x   = -1;
    animate_ref -> end_y   = -1;

    animate_ref -> duration = 0;

    animate_ref -> actionCB = animateActionMoveCopyCB;

  } /* if */


  /* Define a new entry? */
  if( flagIsSet( flags, ANI_NEW_ENTRY ) ) {

    xtmStPosXToDate( sched_ref,
                     start_x,
                     &animate_ref -> start_date,
                     &animate_ref -> start_x );

    animate_ref -> end_x    = animate_ref -> start_x + (int) marker_width;
    animate_ref -> end_y    = animate_ref -> start_y;
    animate_ref -> duration = 0;

    animate_ref -> actionCB = animateActionNewEntryCB;

  } /* if */


  /* Change duration? */
  if( flagIsSet( flags, ANI_NEW_DURATION ) ) {

    animate_ref -> start_x = start_x;
    animate_ref -> start_y = start_y;

    animate_ref -> end_x = animate_ref -> start_x + (int) marker_width;
    animate_ref -> end_y = animate_ref -> start_y + (int) marker_height;

    xtmStPosYToDuration( sched_ref,
                         animate_ref -> start_y, animate_ref -> end_y,
                         animate_ref -> grid_minutes,
                         &animate_ref -> duration, &new_delta_y );

    animate_ref -> end_y = animate_ref -> start_y + new_delta_y;

    animate_ref -> start_date = entry_info_ref -> date_stamp;
    animate_ref -> start_time = entry_info_ref -> time_stamp;

    if( animate_ref -> end_y < animate_ref -> start_y ) {
      animate_ref -> end_y    = animate_ref -> start_y;
      animate_ref -> duration = 0;
    }

    animate_ref -> actionCB = animateActionNewDurationCB;

  } /* if */


  /* We are now doing animation. */
  animate_ref -> flags      = flags;
  animate_ref -> animate_on = True;

  animate_ref -> old_start_date = animate_ref -> start_date;
  animate_ref -> old_start_time = animate_ref -> start_time;
  animate_ref -> duration       = animate_ref -> duration;

  flagClear( animate_ref -> flags, ANI_STARTED );

  if( flagIsSet( flags, (ANI_NEW_ENTRY | ANI_NEW_DURATION) ) )
    drawAnimateTrace( animate_ref -> animateW,
                      animate_ref -> start_x, animate_ref -> start_y,
                      animate_ref -> end_x,   animate_ref -> end_y );


  return;

} /* xtmSaStartAnimation */


/*----------------------------------------------------------------------*/

void
  xtmSaStopAnimation( Widget         widget,
                      SCHED_REC_REF  sched_ref,
                      XEvent         *event )
{

  /* Variables. */
  Widget                mainW;
  ANIMATE_REC_REF       animate_ref;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = sched_ref -> appl_data_ref;
  animate_ref   = sched_ref -> animate_ref;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Are we doing animation? */
  if( ! animate_ref -> animate_on )
    return;

  /* Clean up. */
  if( animate_ref -> start_x != -1 )
    XtUnmapWidget( animate_ref -> animateW );

  if( flagIsSet( animate_ref -> flags, ANI_STARTED ) )
    XUngrabPointer( XtDisplay( widget ), CurrentTime );


  animate_ref -> animate_on = False;
  flagClear( animate_ref -> flags, ANI_STARTED );


  /* Call action procedure? */
  if( animate_ref -> actionCB != NULL )
    (* animate_ref -> actionCB)( sched_ref, event );


  /* Clear the track pixmap and label. */    
  xtmStClearEntryTime( sched_ref );


  return;

} /* xtmSaStopAnimation */


/*----------------------------------------------------------------------*/

void
  xtmSaTrackAnimation( Widget         widget,
                       SCHED_REC_REF  sched_ref,
                       XEvent         *event )
{

  /* Variables. */
  int                     new_end_x = 0;
  int                     new_end_y = 0;
  int                     new_start_x = 0;
  int                     new_start_y = 0;
  int                     old_end_x;
  int                     old_end_y;
  int                     old_start_x;
  int                     old_start_y;
  Widget                  mainW;
  ANIMATE_REC_REF         animate_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  animate_ref     = sched_ref -> animate_ref;
  custom_data_ref = sched_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );


  /* Are we doing animation? */
  if( ! animate_ref -> animate_on )
    return;


  /* Have we started the animation? */
  if( flagIsClear( animate_ref -> flags, ANI_STARTED ) ) {

    Cursor  cursor;

    /* We should have a nice cursor. */
    if( flagIsSet( animate_ref -> flags, ANI_NEW_DURATION ) )
      cursor = XCreateFontCursor( XtDisplay( widget ), XC_bottom_side );
    else if( flagIsSet( animate_ref -> flags, ANI_MOVE_COPY ) )
      cursor = XCreateFontCursor( XtDisplay( widget ), XC_fleur );
    else
      cursor = XCreateFontCursor( XtDisplay( widget ), XC_hand2 );


    /* Grab the pointer to get all events we want. */
    XGrabPointer( XtDisplay( widget ), XtWindow( widget ),
                  False, 
                  (ButtonPressMask | ButtonMotionMask | ButtonReleaseMask),
                  GrabModeAsync, GrabModeAsync,
                  None, cursor, CurrentTime );

    /* Yes, we are doing animation.... */
    flagSet( animate_ref -> flags, ANI_STARTED );

  } /* if */


  /* Wee need these later. */
  old_start_x = animate_ref -> start_x;
  old_start_y = animate_ref -> start_y;
  old_end_x   = animate_ref -> end_x;
  old_end_y   = animate_ref -> end_y;


  /* Copy/Move entry? */
  if( flagIsSet( animate_ref -> flags, ANI_MOVE_COPY ) ) {

    xtmStPosXToDate( sched_ref,
                     (int) event -> xbutton.x,
                     &animate_ref -> start_date,
                     &new_start_x );

    xtmStPosYToTime( sched_ref,
                     (int) event -> xbutton.y,
                     animate_ref  -> grid_minutes,
                     &animate_ref -> start_time,
                     &new_start_y );

    new_end_x = new_start_x + animate_ref -> width;
    new_end_y = new_start_y + animate_ref -> height;

  } /* if */


  /* Mark new entry or change duration? */
  if( flagIsSet( animate_ref -> flags, (ANI_NEW_ENTRY | ANI_NEW_DURATION) ) ) {

    new_start_x = old_start_x;
    new_start_y = old_start_y;

    new_end_x = old_end_x;
    new_end_y = event -> xbutton.y;

    /* We can only change duration, not the start time. */
    if( new_end_y < new_start_y ) {
      new_end_y = new_start_y;
      animate_ref -> duration = 0;

    } else {
      int  minutes;
      int  new_delta_y;

      xtmStPosYToDuration( sched_ref,
                           new_start_y, new_end_y,
                           animate_ref -> grid_minutes,
                           &minutes, &new_delta_y );

      if( minutes == animate_ref -> duration ) {
        new_end_y = old_end_y;
      } else {
        new_end_y = new_delta_y + new_start_y;
        animate_ref -> duration = minutes;
      }

    } /* if */

  } /* if */


  /* No change? */
  if( old_start_x == new_start_x &&
      old_start_y == new_start_y &&
      old_end_x   == new_end_x   &&
      old_end_y   == new_end_y )
    return;


  /* Display the animation button. */
  drawAnimateTrace( animate_ref -> animateW,
                    new_start_x, new_start_y,
                    new_end_x,   new_end_y );

  
  /* Save the positions. */
  animate_ref -> start_x = new_start_x;
  animate_ref -> start_y = new_start_y;
  animate_ref -> end_x   = new_end_x;
  animate_ref -> end_y   = new_end_y;


  /* Track what we are doing. */    
  if( animate_ref -> display_time ) {

    Boolean               only_start_time = False;
    XTM_ST_ANI_DIRECTION  direction = XTM_ST_ANI_STILL;

    if( flagIsSet( animate_ref -> flags, ANI_MOVE_COPY ) )
      only_start_time = True;

    /* Where are we moving? */
    if( animate_ref -> start_date < animate_ref -> old_start_date )
      direction = XTM_ST_ANI_BACKWARD;
    else if( animate_ref -> start_date > animate_ref -> old_start_date )
      direction = XTM_ST_ANI_FORWARD;
    else if( animate_ref -> start_time < animate_ref -> old_start_time )
      direction = XTM_ST_ANI_BACKWARD;
    else if( animate_ref -> start_time > animate_ref -> old_start_time )
      direction = XTM_ST_ANI_FORWARD;
    else if( animate_ref -> duration < animate_ref -> old_duration )
      direction = XTM_ST_ANI_BACKWARD;
    else if( animate_ref -> duration > animate_ref -> old_duration )
      direction = XTM_ST_ANI_FORWARD;

    xtmStPresentEntryTime( sched_ref, only_start_time, True, direction,
                           animate_ref -> start_time,
                           animate_ref -> duration );
  } /* if */

  /* Remember where we were. */
  animate_ref -> old_start_date = animate_ref -> start_date;
  animate_ref -> old_start_time = animate_ref -> start_time;
  animate_ref -> old_duration   = animate_ref -> duration;


  return;

} /* xtmSaTrackAnimation */


/*----------------------------------------------------------------------*/

static void
  drawAnimateTrace( Widget  widget,
                    int     x1,
                    int     y1,
                    int     x2,
                    int     y2 )
{

  /* Variables. */
  int       temp;
  Arg       args[ 10 ];
  Cardinal  n;


  /* Code. */

  /* This must be visible. */
  if( ! XtIsManaged( widget ) )
    XtManageChild( widget );
  else
    XtMapWidget( widget );

  XRaiseWindow( XtDisplay( widget ), XtWindow( widget ) );


  /* Make sure we have a nice rectangle. */
  if( x2 < x1 ) {
    temp = x1;
    x1   = x2;
    x2   = temp;
  }

  if( y2 < y1 ) {
    temp = y1;
    y1   = y2;
    y2   = temp;
  }

  /* Correct size and position. */
  if( x2 - x1 < 3 )
    x2 = x1 + 3;

  if( y2 - y1 < 3 )
    y2 = y1 + 3;

  n = 0;
  XtSetArg( args[ n ], XmNx, (Position) x1 ); n++;
  XtSetArg( args[ n ], XmNy, (Position) y1 ); n++;
  XtSetArg( args[ n ], XmNheight, (Dimension) y2 - y1 ); n++;
  XtSetArg( args[ n ], XmNwidth, (Dimension) x2 - x1 ); n++;
  XtSetValues( widget, args, n );


  return;

} /* drawAnimateTrace */


/*----------------------------------------------------------------------*/

static void
  changeDurationCB( Widget         widget,
                    SCHED_REC_REF  sched_ref,
                    XtPointer      call_data )
{

  /* Variables. */
  Boolean                 ok;
  char                    *entry_text = NULL;
  ANIMATE_REC_REF         animate_ref;
  ENTRY_INFO              *entry_info_ref;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;


  /* Code. */

  animate_ref = sched_ref -> animate_ref;


  /* Selected entry. */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
  if( entry_info_ref == NULL )
    return;

  xtmSwUnselectEntry( sched_ref );

  /* Open the database and fetch the entry. */
  ok = xtmDmOpenDatabase( sched_ref -> appl_data_ref,
                          entry_info_ref -> db_name, XTM_DB_FLAG_MODE_WRITE,
                          &database );
  if( ! ok )
    return;

  ok = xtmDmFetchEntry( sched_ref -> scheduleW,
                        &database, entry_info_ref -> id,
                        &entry_record, &entry_text );
  if( ! ok )
    raise exception;

  /* Save the modified appointment. */
  entry_record.entry.duration = (UINT16) animate_ref -> duration;

  ok = xtmDmInsertEntry( sched_ref -> scheduleW,
                         &database, entry_info_ref -> id,
                         &entry_record, entry_text );
  if( ! ok )
    raise exception;

  if( entry_text != NULL )
    SysFree( entry_text );
  

  /* Remove the old entry. */
  (void) xtmDbDeleteEntry( &database, entry_info_ref -> id );


  /* That's it. */
  xtmDbCloseEntryDb( &database );


  /* Update the schedule/calendar. */
  xtmUpDoUpdate( (XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;


  /* Exception handler. */
  exception:
    if( entry_text != NULL )
      SysFree( entry_text );

    xtmDbCloseEntryDb( &database );

    return;

} /* changeDurationCB */


/*----------------------------------------------------------------------*/

static void
  animateActionMoveCopyCB( SCHED_REC_REF  sched_ref,
                           XEvent         *event )
{

  /* Variables. */
  ANIMATE_REC_REF   animate_ref;
  ENTRY_INFO        *entry_info_ref;
  TIM_TIME_REF      to_time;
  XTM_CM_OPERATION  operation;


  /* Code. */

  animate_ref = sched_ref -> animate_ref;

  /* Selected entry. */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
  if( entry_info_ref == NULL )
    return;

  /* Copy or move entry? */
  if( (event -> xbutton.state & ShiftMask) != 0 )
    operation = XTM_CM_COPY;
  else
    operation = XTM_CM_MOVE;

  xtmSwUnselectEntry( sched_ref );


  if( flagIsSet( sched_ref -> flags, XTM_SM_LIST_LAYOUT ) )
    to_time = entry_info_ref -> time_stamp;
  else
    to_time = animate_ref -> start_time;

  xtmCmCopyMoveEntryTo( sched_ref -> appl_data_ref,
                        sched_ref -> scheduleW,
                        operation,
                        entry_info_ref -> db_name,
                        entry_info_ref -> id,
                        entry_info_ref -> db_name,
                        animate_ref -> start_date, 
                        to_time );


  return;

} /* animateActionMoveCopyCB */


/*----------------------------------------------------------------------*/

static void
  animateActionNewDurationCB( SCHED_REC_REF  sched_ref,
                              XEvent         *event )
{

  /* Variables. */
  char             buffer[ 100 ];
  Widget           tempW;
  ANIMATE_REC_REF  animate_ref;
  ENTRY_INFO       *entry_info_ref;


  /* Code. */

  animate_ref = sched_ref -> animate_ref;

  /* If no duration is defined, do nothing. */
  if( animate_ref -> duration <= 0 ) {
    xtmSwUnselectEntry( sched_ref );

    return;
  }

  /* Fetch reference to selected widget. */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
  if( entry_info_ref == NULL )
    return;


  /* User confirmation? */
  if( sched_ref -> appl_data_ref -> custom_data -> confirm_actions ) {
    sprintf( buffer, msgGetText( MXDI_CHANGE_DURATION_TO ), 
             animate_ref -> duration / 60, 
             animate_ref -> duration % 60 );

    tempW = xitCreateQuestionDialog( 
              sched_ref -> scheduleW, "QuestionDialog",
              msgGetText( MXDI_QUESTION_MESSAGE_LABEL ),
              buffer,
              changeDurationCB, sched_ref, 
              NULL, NULL );
    return;
  }

  /* Change duration. */
  changeDurationCB( NULL, sched_ref, NULL );


  return;

} /* animateActionNewDurationCB */


/*----------------------------------------------------------------------*/

static void
  animateActionNewEntryCB( SCHED_REC_REF  sched_ref,
                           XEvent         *event )
{

  /* Variables. */
  Boolean                 ok;
  ANIMATE_REC_REF         animate_ref;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  animate_ref     = sched_ref -> animate_ref;
  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* If no duration is defined, do nothing. */
  if( animate_ref -> start_y == animate_ref -> end_y )
    return;


  /* Should we pick a day? */
  if( flagIsSet( animate_ref -> flags, ANI_PICK_DAY ) ) {
    ok = xtmStPickDate( sched_ref, XTM_ST_PICK_APP, 
                        &animate_ref -> start_date );
    if( ! ok )
      return;
  }


  /* Fetch the current database. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            sched_ref -> db_name,
                            &db_info, NULL );


  /* Call the appointment editor. */
  if( sched_ref -> editor_handle == NULL )
    sched_ref -> editor_handle = 
      xtmEdInitialize( appl_data_ref,
                       appl_data_ref   -> toplevel,
                       sched_ref       -> entry_delta,
                       custom_data_ref -> start_hour,
                       custom_data_ref -> stop_hour,
                       custom_data_ref -> default_entry_delta,
                       appointmentEditorCB, 
                       (void *) sched_ref );

  /* Display the editor. */
  xtmEdEditEntry( sched_ref -> editor_handle,
                  db_info.short_name,
                  0,
                  animate_ref -> start_date,
                  animate_ref -> start_time );

  /* Our duration. */
  xtmEdSetValues( sched_ref -> editor_handle,
                  XTM_ED_SET_DURATION,
                  0, 0, (UINT32) animate_ref -> duration, NULL );


  return;

} /* animateActionNewEntryCB */


/*----------------------------------------------------------------------*/

static void 
  appointmentEditorCB( XTM_ED_REASON reason,
                       void          *user_data )
{

  /* Variables. */
  SCHED_REC_REF  sched_ref;


  /* Code. */

  sched_ref = (SCHED_REC_REF) user_data;

  if( reason == XTM_ED_REASON_DESTROY )
    sched_ref -> editor_handle = NULL;


  return;

} /* appointmentEditorCB */


/*----------------------------------------------------------------------*/

static Widget
  createEntryEditPopup( SCHED_REC_REF  sched_ref,
                        Widget         parentW )
{

  /* Variables. */
  int     index;
  Widget  menuPopupBu[ 14 ];
  Widget  popupW;

  static XIT_MENU_BUTTON_STRUCT popup_casc[] = {
    { "",      "",  NULL, "EditBu",     True,  False, False },
    { "",      "",  NULL, "DeleteBu",   True,  False, False },
    { "",      "",  NULL, "MoveBu",     True,  False, False },
    { "",      "",  NULL, "CopyBu",     True,  False, False },
    { "",      "",  NULL, "DuplBu",     True,  False, False },
    { "",      "",  NULL, "ArchBu",     True,  False, False },
    { "",      "",  NULL, "FreezeBu",   True,  False, False },
    { XIT_SEP, " ", NULL, "",           False, False, False },
    { "",      "",  NULL, "NotWeekBu",  True,  False, False },
    { "",      "",  NULL, "NoteDoneBu", True,  False, False },
    { XIT_SEP, " ", NULL, "",           False, False, False },
    { "",      "",  NULL, "CopyBufBu",  True,  False, False },
    { "",      "",  NULL, "CutBufBu",   True,  False, False },
    { "",      "",  NULL, "PasteBufBu", True,  False, False },
  };


  /* Code. */

  /* The menu text. */
  if( flagIsSet( sched_ref -> flags, XTM_SM_READ_ONLY ) )
    popup_casc[  0 ].title = msgGetText( MXDI_SHOW_MENU );
  else
    popup_casc[  0 ].title = msgGetText( MXDI_EDIT_MENU );

  popup_casc[  1 ].title = msgGetText( MXDI_DELETE_MENU );
  popup_casc[  2 ].title = msgGetText( MXDI_MOVE_MENU );
  popup_casc[  3 ].title = msgGetText( MXDI_COPY_MENU );
  popup_casc[  4 ].title = msgGetText( MXDI_DUPL_MENU );
  popup_casc[  5 ].title = msgGetText( MXDI_ARCH_MENU );
  popup_casc[  6 ].title = msgGetText( MXDI_FREEZE_MENU );
  popup_casc[  8 ].title = msgGetText( MXDI_NOT_THIS_WEEK_MENU );
  popup_casc[  9 ].title = msgGetText( MXDI_NOTE_DONE_MENU );
  popup_casc[ 11 ].title = msgGetText( MXDI_COPY_BUF_MENU );
  popup_casc[ 12 ].title = msgGetText( MXDI_CUT_BUF_MENU );
  popup_casc[ 13 ].title = msgGetText( MXDI_PASTE_BUF_MENU );


  /* Create the popup menu. */
  popupW = XmCreatePopupMenu( parentW, "EntryPu", NULL, 0 );

  XtAddCallback( popupW, XmNentryCallback,
                 (XtCallbackProc) entryPuMenuCB, (XtPointer) sched_ref );
  XtAddCallback( popupW, XmNmapCallback,
                 (XtCallbackProc) xtmSaMapEntryPuMenuCB,
                 (XtPointer) sched_ref );

  for( index = 0; index < XtNumber( menuPopupBu ); index++ ) {
    menuPopupBu[ index ] = xitCreateMenuPushButton( popupW, 
                                                    &popup_casc[ index ] );

    if( XmIsPushButton( menuPopupBu[ index ] ) )
      XtAddCallback( menuPopupBu[ index ], XmNactivateCallback, 
                     (XtCallbackProc) entryPuMenuCB, (XtPointer) index );
  }

  XtManageChildren( menuPopupBu, XtNumber( menuPopupBu ) );


  return( popupW );

} /* createEntryEditPopup */


/*----------------------------------------------------------------------*/

static Widget
  createEntryTextPopup( SCHED_REC_REF  sched_ref,
                        Widget         parentW )
{

  /* Variables. */
  Arg       args[ 10 ];
  Widget    entryTextLa;
  Widget    popupW;


  /* Code. */

  /* Create the popup menu. */
  popupW = XmCreatePopupMenu( parentW, "EntryPu", args, 0 );


  /* The entry text. */
  entryTextLa = xitCreateLabel( popupW, "EntryTextLa", " ", -1 );

  XtManageChild( entryTextLa );


  return( popupW );

} /* createEntryTextPopup */


/*----------------------------------------------------------------------*/

static void
  displayEntryEditPopup( SCHED_REC_REF  sched_ref,
                         XEvent         *event )
{

  /* Code. */

  /* Create the edit popup menu? */
  if( sched_ref -> entry_popupW == NULL )
    sched_ref -> entry_popupW =
      createEntryEditPopup( sched_ref,
                            sched_ref -> appl_data_ref -> toplevel );

  /* Post the menu. */
  XmMenuPosition( sched_ref -> entry_popupW, (XButtonPressedEvent *)event );
  XtManageChild(  sched_ref -> entry_popupW );


  return;

} /* displayEntryEditPopup */


/*----------------------------------------------------------------------*/

static void
  displayEntryTextPopup( SCHED_REC_REF  sched_ref,
                         XEvent         *event )
{

  /* Variables. */
  Widget      tempW;
  ENTRY_INFO  *entry_info_ref;


  /* Code. */

  /* There must be an entry selected. */
  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );
  if( entry_info_ref == NULL )
    return;

  if( entry_info_ref -> entry_text == NULL )
    return;

  /* Create the text popup menu? */
  if( sched_ref -> entry_text_popupW == NULL )
    sched_ref -> entry_text_popupW = 
      createEntryTextPopup( sched_ref,
                            sched_ref -> appl_data_ref -> toplevel );


  /* Set the text of the entry. */
  tempW = XtNameToWidget( sched_ref -> entry_text_popupW, "EntryTextLa" );

  xitStringSetLabel( tempW, entry_info_ref -> entry_text );


  /* Post the menu. */
  XmMenuPosition( sched_ref -> entry_text_popupW,
                  (XButtonPressedEvent *)event );

  XtManageChild( sched_ref -> entry_text_popupW );


  return;

} /* displayEntryTextPopup */


/*----------------------------------------------------------------------*/

static void 
  entryPuMenuCB( Widget                     widget,
                 SCHED_REC_REF              sched_ref,
                 XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Widget                  mainW;
  ENTRY_INFO              *entry_info_ref;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = sched_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sched_ref -> scheduleW, "SchedTlBase.SchedTlFo" );

  entry_info_ref = xtmStFetchSelectedInfo( sched_ref );


  /* Select what to do. */
  switch( (int) call_data -> data ) {

    /* Edit the entry. */
    case 0:
      xtmSaAppointmentEditorShow( sched_ref );
      break;


    /* Delete the entry. */
    case 1:
      xtmDlDeleteEntry( appl_data_ref, sched_ref -> scheduleW,
                        entry_info_ref -> db_name, 
                        entry_info_ref -> id );
      break;


    /* Move the entry. */
    case 2:
      xtmCmCopyMoveEntry( appl_data_ref, sched_ref -> scheduleW,
                          XTM_CM_MOVE,
                          entry_info_ref -> db_name, 
                          entry_info_ref -> id );
      break;


    /* Copy the entry. */
    case 3:
      xtmCmCopyMoveEntry( appl_data_ref, sched_ref -> scheduleW,
                          XTM_CM_COPY,
                          entry_info_ref -> db_name, 
                          entry_info_ref -> id );
      break;


    /* Duplicate the entry. */
    case 4:
      xtmDpDuplicateEntry( appl_data_ref, sched_ref -> scheduleW,
                           entry_info_ref -> db_name, 
                           entry_info_ref -> id );
      break;


    /* Archive the entry. */
    case 5:
      xtmArArchiveEntry( appl_data_ref, sched_ref -> scheduleW,
                         entry_info_ref -> db_name, 
                         entry_info_ref -> id );
      break;


    /* Freeze a repeated entry. */
    case 6:
      xtmFzFreezeEntry( appl_data_ref, sched_ref -> scheduleW,
                        entry_info_ref -> db_name, 
                        entry_info_ref -> id,
                        entry_info_ref -> date_stamp );
      break;


    /* Skip week for standing entry. */
    case 8:
      xtmSaDoNotThisWeek( sched_ref,
                          entry_info_ref -> db_name, 
                          entry_info_ref -> id,
                          entry_info_ref -> date_stamp );
      break;


    /* Mark the note as done. */
    case 9:
      xtmSaDoChangeFlags( sched_ref,
                          entry_info_ref -> db_name, 
                          entry_info_ref -> id,
                          entry_info_ref -> date_stamp,
                          XTM_DB_FLAG_NOTE_DONE, True );
      break;


    /* Copy entry to buffer. */
    case 11:
      xtmSaDoCutBuffer( sched_ref, XTM_SA_DO_COPY,
                        entry_info_ref -> db_name, 
                        entry_info_ref -> id );
      break;


    /* Cut entry to buffer. */
    case 12:
      xtmSaDoCutBuffer( sched_ref, XTM_SA_DO_CUT,
                        entry_info_ref -> db_name, 
                        entry_info_ref -> id );
      break;


    /* Paste entry from buffer. */
    case 13:
      xtmSaDoCutBuffer( sched_ref, XTM_SA_DO_PASTE,
                        sched_ref -> db_name,
                        0 );
      break;

  } /* switch */

  /* The entry is not selected. */
  xtmSwUnselectEntry( sched_ref );


  return;

} /* entryPuMenuCB */
