/*----------------------------------------------------------------------------
--
--  Module:           xtmDbFilter
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Selection window with filter to use for database entries.
--
--  Filename:         xtmDbFilter.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-01-26
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmDbFilter.c, Version: 1.1, Date: 95/02/18 15:52:07";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include <Xm/ToggleB.h>
#include <Xm/Text.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmDbMisc.h"
#include "xtmFields.h"
#include "xtmHelp.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "xtmDbFilter.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the include window. */
#define ho1Sp              dataLocalW[  0 ]
#define ho2Sp              dataLocalW[  1 ]
#define introLa            dataLocalW[  2 ]
#define pickSpecLa         dataLocalW[  3 ]
#define pickSpecRc         dataLocalW[  4 ]
#define pickTypeLa         dataLocalW[  5 ]
#define pickTypeRc         dataLocalW[  6 ]
#define searchStrCaseTb    dataLocalW[  7 ]
#define searchStrLa        dataLocalW[  8 ]
#define searchStrTx        dataLocalW[  9 ]
#define tagStrCaseTb       dataLocalW[ 10 ]
#define tagStrFs           dataLocalW[ 11 ]
#define tagStrLa           dataLocalW[ 12 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Filter record. */
typedef struct {

  /* The filter window. */
  Widget  filterW;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_FI_ACTION_CB  actionCB;

} FILTER_REC, *FILTER_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmDbFilter";

/* IDs for the help windows. */
static char  *filter_window_id = "CalFilter";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  applyCB( Widget          widget,
           FILTER_REC_REF  filter_ref,
           XtPointer       call_data );

static void
  cancelCB( Widget          widget,
            FILTER_REC_REF  filter_ref,
            XtPointer       call_data );

static Widget
  createFilterWindow( FILTER_REC_REF  filter_ref,
                      Widget          parent,
                      char            *tags_menu,
                      Boolean         app_modal );

static void 
  destroyCB( Widget          widget,
             FILTER_REC_REF  filter_ref,
             XtPointer       call_data );

static void
  fetchWindowData( FILTER_REC_REF     filter_ref,
                   XTM_DM_FILTER_REC  *use_filter_ref );

static void 
  helpCB( Widget          widget,
          FILTER_REC_REF  filter_ref,
          XtPointer       call_data );

static void
  okCB( Widget          widget,
        FILTER_REC_REF  filter_ref,
        XtPointer       call_data );

static void
  setFilterValues( FILTER_REC_REF     filter_ref,
                   XTM_DM_FILTER_REC  *use_filter_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_FI_HANDLE
  xtmFiInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   char                  *tags_menu,
                   Boolean               app_modal,
                   XTM_FI_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  FILTER_REC_REF     filter_ref;
  XTM_DM_FILTER_REC  filter_rec;


  /* Code. */

  /* Create and initialize our private data. */
  filter_ref = SysNew( FILTER_REC );
  if( filter_ref == NULL )
    return( NULL );

  filter_ref -> appl_data_ref = appl_data_ref;
  filter_ref -> actionCB      = actionCB;
  filter_ref -> user_data     = user_data;


  /* Create the filter window. */
  filter_ref -> filterW = createFilterWindow( filter_ref, parent,
                                              tags_menu,  app_modal );
  if( filter_ref -> filterW == NULL ) {
    SysFree( filter_ref );

    return( NULL );
  }


  /* Set default values. */
  filter_rec.flags = 0;
  flagSet( filter_rec.flags, XTM_DM_FLAG_ALL );

  filter_rec.text_string[ 0 ] = '\0';
  filter_rec.tag_string[  0 ] = '\0';

  setFilterValues( filter_ref, &filter_rec );


  return( (XTM_FI_HANDLE) filter_ref );

} /* xtmFiInitialize */


/*----------------------------------------------------------------------*/

void
  xtmFiDestroy( XTM_FI_HANDLE  filter_handle )
{

  /* Variables. */
  FILTER_REC_REF  filter_ref;


  /* Code. */

  if( filter_handle == NULL )
    return;

  /* Our private data. */
  filter_ref = (FILTER_REC_REF) filter_handle;


  /* Destroy the window. */
  XtDestroyWidget( filter_ref -> filterW );


  return;

} /* xtmFiDestroy */


/*----------------------------------------------------------------------*/

void
  xtmFiDisplayFilterWindow( XTM_FI_HANDLE      filter_handle,
                            XTM_DM_FILTER_REC  *use_filter_ref )
{

  /* Variables. */
  FILTER_REC_REF  filter_ref;


  /* Code. */

  if( filter_handle == NULL )
    return;

  /* Our private data. */
  filter_ref = (FILTER_REC_REF) filter_handle;


  /* Set filter data? */
  if( use_filter_ref != NULL )
    setFilterValues( filter_ref, use_filter_ref );


  /* Make sure the window is displayed. */
  XtManageChild( filter_ref -> filterW );


  return;

} /* xtmFiDisplayFilterWindow */


/*----------------------------------------------------------------------*/

static Widget
  createFilterWindow( FILTER_REC_REF  filter_ref,
                      Widget          parent,
                      char            *tags_menu,
                      Boolean         app_modal )
{

  /* Variables. */
  int            index;
  char           buffer[ 100 ];
  char           *pick_labels[ 9 ];
  Arg            args[ 10 ];
  Cardinal       n;
  Widget         dataLocalW[ 13 ];
  Widget         filtFd;
  Widget         pickTb[ 9 ];
  Widget         tempW;
  Widget         workFo;

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "SearchStrTx", NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,     NULL },
    { "", applyCB,  NULL },
    { "", cancelCB, NULL },
    { "", helpCB,   NULL },
  };


  /* Code. */

  /* Set message strings. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = filter_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 1 ].data  = filter_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 2 ].data  = filter_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 3 ].data  = filter_ref;

  pick_labels[ 0 ] = msgGetText( MXDI_FILTER_APPOINTMENTS_LABEL );
  pick_labels[ 1 ] = msgGetText( MXDI_FILTER_NOTES_LABEL );
  pick_labels[ 2 ] = msgGetText( MXDI_FILTER_ALARMS_LABEL );
  pick_labels[ 3 ] = msgGetText( MXDI_FILTER_IMPORTANT_LABEL );
  pick_labels[ 4 ] = msgGetText( MXDI_FILTER_PRIVATE_LABEL );
  pick_labels[ 5 ] = msgGetText( MXDI_FILTER_DONE_LABEL );
  pick_labels[ 6 ] = msgGetText( MXDI_FILTER_NOT_DONE_LABEL );
  pick_labels[ 7 ] = msgGetText( MXDI_FILTER_REPEATED_LABEL );
  pick_labels[ 8 ] = msgGetText( MXDI_FILTER_STICKY_LABEL );


  /* Create a form dialog with buttons. */
  filtFd = xitCreateFormDialog( parent, "FiltFd",
                                1, 0,
                                action_buttons,
                                XtNumber( action_buttons ) );

  XtAddCallback( filtFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) filter_ref );


  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_FILTER_DB_TITLE ) ); n++;
  XtSetValues( XtParent( filtFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmUNMAP ); n++;
  XtSetValues( filtFd, args, n );

  if( app_modal ) {
    n = 0;
    XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
    XtSetValues( filtFd, args, n );
  }


  /* Create a container for the contents of the window. */
  workFo = XtNameToWidget( filtFd, "FiltFdFo" );

  introLa = xitCreateLabel( workFo, "IntroLa", 
                            msgGetText( MXDI_FILTER_INTRO_LABEL ), -1 );


  /* The type of entry to select. */
  pickTypeLa = xitCreateLabel( workFo, "PickTypeLa", 
                               msgGetText( MXDI_FILTER_PICK_TYPE_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns, 2 ); n++;
  pickTypeRc = XmCreateRowColumn( workFo, "PickTypeRc", args, n );

  for( index = 0; index < 2; index++ ) {
    sprintf( buffer, "Pick%dTb", index + 1 );

    pickTb[ index ] = xitCreateToggleButton( pickTypeRc, buffer,
                                             pick_labels[ index ], False );
  }


  /* The specific entries to select. */
  pickSpecLa = xitCreateLabel( workFo, "PickSpecLa", 
                               msgGetText( MXDI_FILTER_PICK_SPEC_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns, 2 ); n++;
  pickSpecRc = XmCreateRowColumn( workFo, "PickSpecRc", args, n );

  for( index = 2; index < 9; index++ ) {
    sprintf( buffer, "Pick%dTb", index + 1 );

    pickTb[ index ] = xitCreateToggleButton( pickSpecRc, buffer,
                                             pick_labels[ index ], False );
  }


  /* Separator. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
  ho1Sp = XmCreateSeparatorGadget( workFo, "Ho1Sp", args, n );


  /* Search text. */
  searchStrLa = xitCreateLabel( workFo, "SearchStrLa", 
                                msgGetText( MXDI_SEARCH_STRING_LABEL ), -1 );

  searchStrTx = xitCreateTextCols( workFo, &text_buffer[ 0 ], 40 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, XTM_DM_MAX_TEXT_SEARCH ); n++;
  XtSetValues( searchStrTx, args, n );

  searchStrCaseTb = xitCreateToggleButton( 
                        workFo, "SearchStrCaseTb", 
                        msgGetText( MXDI_CASE_SENSITIVE_LABEL ),
                        False );

  /* Separator. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL );  n++;
  ho2Sp = XmCreateSeparatorGadget( workFo, "Ho2Sp", args, n );


  /* Search tag. */
  tagStrLa = xitCreateLabel( workFo, "TagStrLa", 
                             msgGetText( MXDI_SEARCH_TAG_LABEL ), -1 );

  tagStrFs = xtmFlCreateSelField( workFo, "TagStrFs", 
                                  XTM_DM_MAX_TAG_SEARCH, tags_menu,
                                  ' ', True,
                                  NULL, NULL );

  tempW = xitFieldSelectGetChild( tagStrFs, xitFIELD_SELECT_TEXT_FIELD );

  xitFieldSelectSetAppend( tagStrFs, True );

  tagStrCaseTb = xitCreateToggleButton( 
                   workFo, "TagStrCaseTb", 
                   msgGetText( MXDI_CASE_SENSITIVE_LABEL ),
                   False );


  /* Put the Parts together. */
  xitAttachWidget( introLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( pickTypeLa,
                   XmATTACH_WIDGET, introLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( pickTypeRc,
                   XmATTACH_WIDGET, pickTypeLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,       XmATTACH_NONE, NULL );
  xitAttachWidget( pickSpecLa,
                   XmATTACH_WIDGET, pickTypeRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,       XmATTACH_NONE, NULL );
  xitAttachWidget( pickSpecRc,
                   XmATTACH_WIDGET, pickSpecLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,       XmATTACH_NONE, NULL );
  xitAttachWidget( ho1Sp,
                   XmATTACH_WIDGET, pickSpecRc, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,       XmATTACH_NONE, NULL );
  xitAttachWidget( searchStrLa,
                   XmATTACH_WIDGET, ho1Sp, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( searchStrTx,
                   XmATTACH_WIDGET, searchStrLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,        XmATTACH_NONE, NULL );
  xitAttachWidget( searchStrCaseTb,
                   XmATTACH_WIDGET, searchStrTx, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,        XmATTACH_NONE, NULL );
  xitAttachWidget( ho2Sp,
                   XmATTACH_WIDGET, searchStrCaseTb, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,            XmATTACH_NONE, NULL );
  xitAttachWidget( tagStrLa,
                   XmATTACH_WIDGET, ho2Sp, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( tagStrFs,
                   XmATTACH_WIDGET, tagStrLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( tagStrCaseTb,
                   XmATTACH_WIDGET, tagStrFs, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( introLa,         args, n );
  XtSetValues( pickTypeLa,      args, n );
  XtSetValues( pickTypeRc,      args, n );
  XtSetValues( pickSpecLa,      args, n );
  XtSetValues( pickSpecRc,      args, n );
  XtSetValues( searchStrLa,     args, n );
  XtSetValues( searchStrTx,     args, n );
  XtSetValues( searchStrCaseTb, args, n );
  XtSetValues( tagStrLa,        args, n );
  XtSetValues( tagStrFs,        args, n );
  XtSetValues( tagStrCaseTb,    args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( ho1Sp, args, n );
  XtSetValues( ho2Sp, args, n );


  /* Manage the children. */
  xitManageChildren( pickTb, XtNumber( pickTb ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the initial sizes. */
  xitSetSizeFormDialog( filtFd, True );


  return( filtFd );

} /* createFilterWindow */


/*----------------------------------------------------------------------*/

static void
  fetchWindowData( FILTER_REC_REF     filter_ref,
                   XTM_DM_FILTER_REC  *use_filter_ref )
{

  /* Variables. */
  char    *char_ref;
  char    *text_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( filter_ref -> filterW, "FiltFdFo" );

  use_filter_ref -> flags            = 0;
  use_filter_ref -> text_string[ 0 ] = '\0';
  use_filter_ref -> tag_string[ 0 ]  = '\0';


  /* Type of entries. */
  tempW = XtNameToWidget( mainW, "PickTypeRc.Pick1Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_APPOINTMENTS );


  tempW = XtNameToWidget( mainW, "PickTypeRc.Pick2Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_NOTES );

  if( flagIsSet( use_filter_ref -> flags, XTM_DM_FLAG_APPOINTMENTS ) &&
      flagIsSet( use_filter_ref -> flags, XTM_DM_FLAG_NOTES ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_ALL );


  /* Specific entries. */
  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick3Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_ALARMS );

  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick4Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_IMPORTANT );

  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick5Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_PRIVATE );

  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick6Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_DONE );

  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick7Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_NOT_DONE );

  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick8Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_REPEATED );

  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick9Tb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_STICKY );


  /* Case sensitive search. */
  tempW = XtNameToWidget( mainW, "SearchStrCaseTb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_TEXT_CASE_SENS );


  tempW = XtNameToWidget( mainW, "TagStrCaseTb" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_TAG_CASE_SENS );


  /* Text search string. */
  tempW    = XtNameToWidget( mainW, "SearchStrTx" );
  text_ref = xitStringGetText( tempW );

  char_ref = text_ref;
  while( isspace( *char_ref ) )
    char_ref++;

  if( strlen( char_ref ) > 0 ) {
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_SEARCH_TEXT );
    strcpy( use_filter_ref -> text_string, char_ref );
  }

  SysFree( text_ref );


  /* Tag search string. */
  tempW = XtNameToWidget( mainW, "TagStrFs" );
  xitFieldSelectGetCurrent( tempW, &text_ref );

  char_ref = text_ref;
  while( isspace( *char_ref ) )
    char_ref++;

  if( strlen( char_ref ) > 0 ) {
    flagSet( use_filter_ref -> flags, XTM_DM_FLAG_SEARCH_TAG );
    strcpy( use_filter_ref -> tag_string, char_ref );
  }

  SysFree( text_ref );


  return;

} /* fetchWindowData */


/*----------------------------------------------------------------------*/

static void
  setFilterValues( FILTER_REC_REF     filter_ref,
                   XTM_DM_FILTER_REC  *use_filter_ref )
{

  /* Variables. */
  Boolean  set;
  UINT32   filter;
  Widget   mainW;
  Widget   tempW;


  /* Code. */

  mainW = XtNameToWidget( filter_ref -> filterW, "FiltFdFo" );


  /* Set filter buttons. */
  filter = use_filter_ref -> flags;


  /* Type of filter. */
  set = (flagIsSet( filter, (XTM_DM_FLAG_ALL | XTM_DM_FLAG_APPOINTMENTS) )) ?
           True : False;
  tempW = XtNameToWidget( mainW, "PickTypeRc.Pick1Tb" );
  XmToggleButtonSetState( tempW, set, False );

  set = (flagIsSet( filter, (XTM_DM_FLAG_ALL | XTM_DM_FLAG_NOTES) )) ?
           True : False;
  tempW = XtNameToWidget( mainW, "PickTypeRc.Pick2Tb" );
  XmToggleButtonSetState( tempW, set, False );


  /* Specific filter. */
  set   = (flagIsSet( filter, XTM_DM_FLAG_ALARMS )) ? True : False;
  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick3Tb" );
  XmToggleButtonSetState( tempW, set, False );

  set   = (flagIsSet( filter, XTM_DM_FLAG_IMPORTANT )) ? True : False;
  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick4Tb" );
  XmToggleButtonSetState( tempW, set, False );

  set   = (flagIsSet( filter, XTM_DM_FLAG_PRIVATE )) ? True : False;
  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick5Tb" );
  XmToggleButtonSetState( tempW, set, False );

  set   = (flagIsSet( filter, XTM_DM_FLAG_DONE )) ? True : False;
  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick6Tb" );
  XmToggleButtonSetState( tempW, set, False );

  set   = (flagIsSet( filter, XTM_DM_FLAG_NOT_DONE )) ? True : False;
  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick7Tb" );
  XmToggleButtonSetState( tempW, set, False );

  set   = (flagIsSet( filter, XTM_DM_FLAG_REPEATED )) ? True : False;
  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick8Tb" );
  XmToggleButtonSetState( tempW, set, False );

  set   = (flagIsSet( filter, XTM_DM_FLAG_STICKY )) ? True : False;
  tempW = XtNameToWidget( mainW, "PickSpecRc.Pick9Tb" );
  XmToggleButtonSetState( tempW, set, False );


  /* Set case sensitive buttons. */
  filter = use_filter_ref -> flags;

  set   = (flagIsSet( filter, XTM_DM_FLAG_TEXT_CASE_SENS )) ? True : False;
  tempW = XtNameToWidget( mainW, "SearchStrCaseTb" );
  XmToggleButtonSetState( tempW, set, False );

  set   = (flagIsSet( filter, XTM_DM_FLAG_TAG_CASE_SENS )) ? True : False;
  tempW = XtNameToWidget( mainW, "TagStrCaseTb" );
  XmToggleButtonSetState( tempW, set, False );


  /* Set search string. */
  tempW = XtNameToWidget( mainW, "SearchStrTx" );
  XmTextSetString( tempW, use_filter_ref -> text_string );


  /* Tag seach string. */
  tempW = XtNameToWidget( mainW, "TagStrFs" );
  xitFieldSelectSetCurrent( tempW, use_filter_ref -> tag_string, False );


  return;

} /* setFilterValues */


/*----------------------------------------------------------------------*/

static void
  applyCB( Widget          widget,
           FILTER_REC_REF  filter_ref,
           XtPointer       call_data )
{

  /* Variables. */
  XTM_DM_FILTER_REC  filter_rec;


  /* Code. */

  /* Fetch the window data. */
  fetchWindowData( filter_ref, &filter_rec );


  /* Call callback function? */
  if( filter_ref -> actionCB != NULL )
    (* filter_ref -> actionCB)( XTM_FI_REASON_APPLY,
                                &filter_rec,
                                filter_ref -> user_data );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void
  cancelCB( Widget          widget,
            FILTER_REC_REF  filter_ref,
            XtPointer       call_data )
{

  /* Code. */

  /* Call callback function? */
  if( filter_ref -> actionCB != NULL )
    (* filter_ref -> actionCB)( XTM_FI_REASON_CANCEL,
                                NULL, filter_ref -> user_data );

  /* Hide the window. */
  XtUnmanageChild( filter_ref -> filterW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget          widget,
             FILTER_REC_REF  filter_ref,
             XtPointer       call_data )
{

  /* Code. */

  SysFree( filter_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget          widget,
          FILTER_REC_REF  filter_ref,
          XtPointer       call_data )
{

  /* Code. */

  xtmHlDisplayHelp( filter_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    filter_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  okCB( Widget          widget,
        FILTER_REC_REF  filter_ref,
        XtPointer       call_data )
{

  /* Variables. */
  XTM_DM_FILTER_REC  filter_rec;


  /* Code. */

  /* Fetch the window data. */
  fetchWindowData( filter_ref, &filter_rec );


  /* Call callback function? */
  if( filter_ref -> actionCB != NULL )
    (* filter_ref -> actionCB)( XTM_FI_REASON_OK,
                                &filter_rec,
                                filter_ref -> user_data );

  /* Hide the window. */
  XtUnmanageChild( filter_ref -> filterW );


  return;

} /* okCB */
