/*----------------------------------------------------------------------------
--
--  Module:           xtmCusCals
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Manage the calendars in XDiary. You can add, change and delete
--    calendars.
--
--  Filename:         xtmCustCals.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-07-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmCustCals.c, Version: 1.1, Date: 95/02/18 15:52:02";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustom.h"
#include "xtmCustBase.h"
#include "xtmCustIncl.h"
#include "xtmCustImp.h"
#include "xtmDbTools.h"
#include "xtmHelp.h"
#include "xtmLocate.h"
#include "xtmPickDiary.h"
#include "xitError.h"
#include "xitParSet.h"
#include "xitTools.h"
#include "xtmCustCals.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the group window. */
#define calActionRc        dataLocalW[  0 ]
#define calDataRc          dataLocalW[  1 ]
#define calLa              dataLocalW[  2 ]
#define calLi              dataLocalW[  3 ]
#define calLocLa           dataLocalW[  4 ]
#define calLocTx           dataLocalW[  5 ]
#define calMoreRc          dataLocalW[  6 ]
#define calNameLa          dataLocalW[  7 ]
#define calNameTx          dataLocalW[  8 ]

/* What fields do we set in the customize window? */
#define SET_CAL_LIST      (1<<0)
#define RESET_CAL_FIELD   (1<<1)


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for calendar customization. */
typedef struct {

  /* Any changes to the group fields. */
  Boolean  field_changed;

  /* Name of selected calendar. */
  char  cal_selected[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* The group window. */
  Widget  calW;

  /* Parameter window. */
  Widget  paramW;

  /* Calendar database. */
  XTM_CD_HANDLE  cal_db_handle;

  /* Imported calendars. */
  XTM_CP_HANDLE  import_handle;

  /* Included calendars. */
  XTM_CI_HANDLE  included_handle;

  /* Pick calendars. */
  XTM_PD_HANDLE  pick_handle;

  /* Application data. */
  XTM_CU_BASE_DATA_REF  appl_data_ref;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  /* Parameter placeholder. */
  XTM_CD_CAL_INFO  param_db_info;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_CC_ACTION_CB  actionCB;

} CAL_REC, *CAL_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmCustCals";

/* IDs for the help windows. */
static char  *cust_cals_window_id = "CustCals";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  addCB( Widget       widget,
         CAL_REC_REF  cal_ref,
         XtPointer    call_data );

static void 
  calImportCB( Widget       widget,
               CAL_REC_REF  cal_ref,
               XtPointer    call_data );

static void 
  calImportApplyCB( XTM_CP_REASON     reason,
                    char              *db_name,
                    void              *user_data );

static void 
  calIncludeCB( Widget       widget,
                CAL_REC_REF  cal_ref,
                XtPointer    call_data );

static void 
  calIncludeApplyCB( XTM_CI_REASON     reason,
                     char              *db_name,
                     XTM_CD_INCL_CALS  *new_db_incl_ref,
                     void              *user_data );

static void 
  calIsSelected( CAL_REC_REF  cal_ref,
                 char         *db_name );

static void 
  calListCB( Widget                widget, 
             CAL_REC_REF           cal_ref,
             XmListCallbackStruct  *call_data );

static void 
  calMoreCB( Widget       widget,
             CAL_REC_REF  cal_ref,
             XtPointer    call_data );

static void
  calParamChangedCB( XIT_PS_REASON  reason,
                     void           *client_data );

static void 
  calSelectCB( Widget       widget,
               CAL_REC_REF  cal_ref,
               XtPointer    call_data );

static void
  calSelectApplyCB( XTM_PD_REASON    reason,
                    XTM_CD_CAL_INFO  *db_ref,
                    void             *client_data );

static void 
  cancelCB( Widget       widget,
            CAL_REC_REF  cal_ref,
            XtPointer    call_data );

static Widget
  createCalendarWindow( CAL_REC_REF  cal_ref,
                        Widget       parent );

static void
  deleteCB( Widget       widget,
            CAL_REC_REF  cal_ref,
            XtPointer    call_data );

static void 
  destroyCB( Widget       widget,
             CAL_REC_REF  cal_ref,
             XtPointer    call_data );

static void 
  fieldChangedCB( Widget       widget,
                  CAL_REC_REF  cal_ref,
                  XtPointer    call_data );

static void
  guessLocationCB( Widget       widget,
                   CAL_REC_REF  cal_ref,
                   XtPointer    call_data );

static void 
  helpCB( Widget       widget,
          CAL_REC_REF  cal_ref,
          XtPointer    call_data );

static void
  makeDbLine( XTM_CD_CAL_INFO  *db_info,
              char             *buffer );

static void 
  okCB( Widget       widget,
        CAL_REC_REF  cal_ref,
        XtPointer    call_data );

static void
  setAction( CAL_REC_REF  cal_ref,
             Boolean      add_button,
             Boolean      delete_button );

static void
  setCalendarData( CAL_REC_REF  cal_ref,
                   UINT32       action );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_CC_HANDLE
  xtmCcInitialize( XTM_CU_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   XTM_CC_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  /* Create and initialize our private data. */
  cal_ref = SysNew( CAL_REC );
  if( cal_ref == NULL )
    return( NULL );

  cal_ref -> appl_data_ref     = appl_data_ref;
  cal_ref -> custom_data_ref   = appl_data_ref -> custom_data;
  cal_ref -> cal_db_handle     = NULL;
  cal_ref -> import_handle     = NULL;
  cal_ref -> included_handle   = NULL;
  cal_ref -> pick_handle       = NULL;
  cal_ref -> field_changed     = False;
  cal_ref -> paramW            = NULL;
  cal_ref -> cal_selected[ 0 ] = '\0';
  cal_ref -> actionCB          = actionCB;
  cal_ref -> user_data         = user_data;


  /* Create the calendars window. */
  cal_ref -> calW = createCalendarWindow( cal_ref, parent );

  if( cal_ref -> calW == NULL ) {
    SysFree( cal_ref );

    return( NULL );
  }


  return( (XTM_CC_HANDLE) cal_ref );

} /* xtmCcInitialize */


/*----------------------------------------------------------------------*/

void
  xtmCcDestroy( XTM_CC_HANDLE  cal_handle )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return;

  /* Our private data. */
  cal_ref = (CAL_REC_REF) cal_handle;

  /* Destroy the window. */
  XtDestroyWidget( cal_ref -> calW );


  return;

} /* xtmCcDestroy */


/*----------------------------------------------------------------------*/

void 
  xtmCcCustomCalendars( XTM_CC_HANDLE  cal_handle,
                        XTM_CD_HANDLE  cal_db_handle )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  cal_ref = (CAL_REC_REF) cal_handle;


  /* Free any old calendar DB and make a copy of the DB. */
  if( cal_ref -> cal_db_handle != NULL )
    xtmCdFree( cal_ref -> cal_db_handle );

  cal_ref -> cal_db_handle = xtmCdCopy( cal_db_handle );
  if( cal_ref -> cal_db_handle == NULL )
    return;


  /* Fill in all the fields and set action buttons. */
  setCalendarData( cal_ref, (SET_CAL_LIST | RESET_CAL_FIELD) );

  setAction( cal_ref, True, False );


  /* Make sure the message window is visable. */
  XtManageChild( cal_ref -> calW );


  return;

} /* xtmCcCustomCalendars */


/*----------------------------------------------------------------------*/

static void 
  calIsSelected( CAL_REC_REF  cal_ref,
                 char         *db_name )
{

  /* Variables. */
  Boolean          ok;
  Widget           mainW;
  Widget           tempW;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  mainW = XtNameToWidget( cal_ref -> calW, "CalFdFo" );


  /* Fetch the selected database. */
  ok = xtmCdFetchNamedDb( cal_ref -> cal_db_handle, db_name,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* Set short name and location. */
  tempW = XtNameToWidget( mainW, "CalDataRc.CalNameTx" );
  XmTextSetString( tempW, db_info.short_name );

  tempW = XtNameToWidget( mainW, "CalDataRc.CalLocTx" );
  XmTextSetString( tempW, db_info.directory );


  /* Action buttons. */
  setAction( cal_ref, True, True );


  /* The More/Include buttons are sensitive. */
  tempW = XtNameToWidget( mainW, "CalMoreRc.MorePb" );
  XtSetSensitive( tempW, True );

  tempW = XtNameToWidget( mainW, "CalMoreRc.IncludePb" );
  XtSetSensitive( tempW, True );


  /* Save the selected database. */
  cal_ref -> field_changed = False;

  strcpy( cal_ref -> cal_selected, db_name );


  return;

} /* calIsSelected */


/*----------------------------------------------------------------------*/

static Widget
  createCalendarWindow( CAL_REC_REF  cal_ref,
                        Widget       parent )
{

  /* Variables. */
  int       index;
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 9 ];
  Widget    calFd;
  Widget    calActionBu[ 2 ];
  Widget    calMoreBu[ 6 ];
  Widget    workFo;

  static XIT_PUSH_STRUCT cal_action_def[] = {
    { "AddPb",    "", "", True, NULL },
    { "DeletePb", "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "CalNameTx", NULL, 1, True },
    { "CalLocTx",  NULL, 1, True },
  };

  static XIT_PUSH_STRUCT cal_more_def[] = {
    { "MorePb",    "", "", True, NULL },
    { "IncludePb", "", "", True, NULL },
    { "Hidden1Pb", "", "", True, NULL },
    { "PickPb",    "", "", True, NULL },
    { "GuessPb",   "", "", True, NULL },
    { "ImportPb",  "", "", True, NULL },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   okCB,     NULL },
    { NULL, NULL,     NULL },
    { "",   cancelCB, NULL },
    { NULL, NULL,     NULL },
    { "",   helpCB,   NULL },
  };

  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = cal_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 2 ].data  = cal_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 4 ].data  = cal_ref;

  cal_more_def[ 0 ].title = msgGetText( MXDI_MORE_PARAMETERS );
  cal_more_def[ 1 ].title = msgGetText( MXDI_MORE_INCLUDE );
  cal_more_def[ 2 ].title = "";
  cal_more_def[ 3 ].title = msgGetText( MXDI_PICK_CALENDAR );
  cal_more_def[ 4 ].title = msgGetText( MXDI_GUESS_LOCATION_BUTTON );
  cal_more_def[ 5 ].title = msgGetText( MXDI_MORE_IMPORT );

  cal_action_def[ 0 ].title = msgGetText( MXDI_ADD_CHANGE_ITEM );
  cal_action_def[ 1 ].title = msgGetText( MXDI_DELETE_ITEM );


  /* Create a form dialog with buttons. */
  calFd = xitCreateFormDialog( parent, "CalFd",
                               1, 0,
                               action_buttons, 
                               XtNumber( action_buttons ) );

  XtAddCallback( calFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) cal_ref );

  char_ref = msgGetText( MXDI_CUST_CAL_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, char_ref ); n++;
  XtSetValues( XtParent( calFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( calFd, args, n );


  /* Use this form as a working form. */
  workFo = XtNameToWidget( calFd, "CalFdFo" );

  /* What is this? */  
  calLa = xitCreateLabel( workFo, "CalLa", 
                          msgGetText( MXDI_DB_LOCATION_LABEL ), -1 );

  /* List with calendars. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  calLi = XmCreateScrolledList( workFo, "CalLi", args, n );

  XtAddCallback( calLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) calListCB, (XtPointer) cal_ref );
  XtAddCallback( calLi, XmNdefaultActionCallback,
                 (XtCallbackProc) calListCB, (XtPointer) cal_ref );


  /* Further customization for databases. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNnumColumns,   2 ); n++;
  XtSetArg( args[ n ], XmNpacking,      XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  calMoreRc = XmCreateRowColumn( workFo, "CalMoreRc", args, n );

  for( index = 0; index < XtNumber( cal_more_def ); index++ )
    calMoreBu[ index ] = xitCreatePushButton( calMoreRc,
                                              &cal_more_def[ index ] );

  n = 0;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  XtSetValues( calMoreBu[ 2 ], args, n );

  XtAddCallback( calMoreBu[ 0 ],  XmNactivateCallback, 
                 (XtCallbackProc) calMoreCB, (XtPointer) cal_ref );
  XtAddCallback( calMoreBu[ 1 ],  XmNactivateCallback, 
                 (XtCallbackProc) calIncludeCB, (XtPointer) cal_ref );
  XtAddCallback( calMoreBu[ 3 ],  XmNactivateCallback, 
                 (XtCallbackProc) calSelectCB, (XtPointer) cal_ref );
  XtAddCallback( calMoreBu[ 4 ],  XmNactivateCallback, 
                 (XtCallbackProc) guessLocationCB, (XtPointer) cal_ref );
  XtAddCallback( calMoreBu[ 5 ],  XmNactivateCallback, 
                 (XtCallbackProc) calImportCB, (XtPointer) cal_ref );


  /* Data for the calendar. */
  calDataRc = XmCreateRowColumn( workFo, "CalDataRc", args, 0 );

  /* Name of calendar. */
  calNameLa = xitCreateLabel( calDataRc, "CalNameLa", 
                              msgGetText( MXDI_DB_NAME_LABEL ), -1 );

  calNameTx = xitCreateTextCols( calDataRc, &text_buffer[ 0 ], 35 );
  XtAddCallback( calNameTx, XmNvalueChangedCallback, 
                 (XtCallbackProc) fieldChangedCB, (XtPointer) cal_ref );


  /* Location for the calendar. */
  calLocLa = xitCreateLabel( calDataRc, "CalLocLa", 
                             msgGetText( MXDI_DB_DIR_LABEL ), -1 );

  calLocTx = xitCreateTextCols( calDataRc, &text_buffer[ 1 ], 35 );
  XtAddCallback( calLocTx, XmNvalueChangedCallback, 
                 (XtCallbackProc) fieldChangedCB, (XtPointer) cal_ref );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 15 ); n++;
  XtSetValues( calNameTx, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 50 ); n++;
  XtSetValues( calLocTx, args, n );


  /* Action (add and delete). */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  calActionRc = XmCreateRowColumn( workFo, "CalActionRc", args, n );

  for( index = 0; index < XtNumber( cal_action_def ); index++ )
    calActionBu[ index ] = xitCreatePushButton( calActionRc,
                                                &cal_action_def[ index ] );

  XtAddCallback( calActionBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) addCB, (XtPointer) cal_ref );
  XtAddCallback( calActionBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) deleteCB, (XtPointer) cal_ref );


  /* Place the main elements together. */
  xitAttachWidget( calLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( calLi ),
                   XmATTACH_WIDGET, calLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( calMoreRc,
                   XmATTACH_WIDGET, XtParent( calLi ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,              XmATTACH_NONE, NULL );
  xitAttachWidget( calDataRc,
                   XmATTACH_WIDGET, calMoreRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( calActionRc,
                   XmATTACH_WIDGET, calDataRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( calLa,             args, n );
  XtSetValues( XtParent( calLi ), args, n );
  XtSetValues( calMoreRc,         args, n );
  XtSetValues( calDataRc,         args, n );
  XtSetValues( calActionRc,       args, n );

  /* Manage all the children. */
  XtManageChildren( calActionBu, XtNumber( calActionBu ) );
  XtManageChildren( calMoreBu,   XtNumber( calMoreBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the size of the window. */
  xitSetSizeFormDialog( calFd, True );


  /* Make the final attachments. */
  xitAttachWidget( calActionRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( calDataRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, calActionRc );
  xitAttachWidget( calMoreRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, calDataRc );
  xitAttachWidget( XtParent( calLi ),
                   XmATTACH_WIDGET, calLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,  XmATTACH_WIDGET, calMoreRc );

  return( calFd );

} /* createCalendarWindow */


/*----------------------------------------------------------------------*/

static void
  makeDbLine( XTM_CD_CAL_INFO  *db_info,
              char             *buffer )
{

  /* Variables. */
  char  flags_buffer[ 20 ];


  /* Code. */

  /* Fetch the flags values. */
  flags_buffer[ 0 ] = '\0';

  if( flagIsSet( db_info -> operations, XTM_DB_FLAG_MODE_READ ) )
    strcat( flags_buffer, "r" );
  else
    strcat( flags_buffer, "-" );

  if( flagIsSet( db_info -> operations, XTM_DB_FLAG_MODE_WRITE ) )
    strcat( flags_buffer, "w" );
  else
    strcat( flags_buffer, "-" );

  strcat( flags_buffer, " " );

  if( flagIsSet( db_info -> flags, XTM_CD_FLAG_ALARM ) )
    strcat( flags_buffer, "A" );

  if( flagIsSet( db_info -> flags, XTM_CD_FLAG_DEFAULT_DB ) )
    strcat( flags_buffer, "D" );

  sprintf( buffer, "%-15.15s %-2.2s %s", 
           db_info -> short_name,
           flags_buffer,
           db_info -> directory );


  return;

} /* makeDbLine */


/*----------------------------------------------------------------------*/

static void
  setAction( CAL_REC_REF  cal_ref,
             Boolean      add_button,
             Boolean      delete_button )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( cal_ref -> calW, "CalFdFo" );

  tempW = XtNameToWidget( mainW, "CalActionRc.AddPb" );
  XtSetSensitive( tempW, add_button );

  tempW = XtNameToWidget( mainW, "CalActionRc.DeletePb" );
  XtSetSensitive( tempW, delete_button );


  return;

} /* setAction */


/*----------------------------------------------------------------------*/

static void
  setCalendarData( CAL_REC_REF  cal_ref,
                   UINT32       action )
{

  /* Variables. */
  int       index;
  int       index1;
  char      buffer[ 100 ];
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;
  XmString  list_items[ 500 ];


  /* Code. */

  mainW = XtNameToWidget( cal_ref -> calW, "CalFdFo" );


  /* Set values for the cals. */
  if( flagIsSet( action, SET_CAL_LIST ) ) {

    char  *char_ref;
    char  *db_names;

    (void) xtmCdFetchDbNames( cal_ref -> cal_db_handle, &db_names );
    char_ref = db_names;
    index    = 0;

    do {

      int              char_read;
      char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
      XTM_CD_CAL_INFO  db_info;

      while( isspace( *char_ref ) )
        char_ref++;

      if( *char_ref == '\0' )
        break;

      char_read = strlen( char_ref );
      sscanf( char_ref, "%s%n", db_name, &char_read );
      char_ref = char_ref + char_read;


      /* Fetch information about the database. */
      (void) xtmCdFetchNamedDb( cal_ref -> cal_db_handle, db_name,
                                &db_info, NULL );

      /* We cannot change included calendars. */
      if( flagIsSet( db_info.flags, XTM_CD_FLAG_IS_INCLUDED ) )
        continue;

      makeDbLine( &db_info, buffer );

      list_items[ index ] = XmStringCreate( buffer, CS );
      index++;

    } while( True && index < XtNumber( list_items ) );

    if( db_names != NULL )
      SysFree( db_names );


    /* The list is always sorted. */
    xitSortStringList( list_items, index );


    tempW = XtNameToWidget( mainW, "CalLiSW.CalLi" );

    /* Assign the calendars to the list. */
    n = 0;
    XtSetArg( args[ n ], XmNitems, list_items ); n++;
    XtSetArg( args[ n ], XmNitemCount, index ); n++;
    XtSetValues( tempW, args, n );

    XmListDeselectAllItems( tempW );

    /* Free allocated memory. */
    for( index1 = 0; index1 < index; index1++ )
      XmStringFree( list_items[ index1 ] );


    /* The More/Include buttons are not sensitive. */
    tempW = XtNameToWidget( mainW, "CalMoreRc.MorePb" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( mainW, "CalMoreRc.IncludePb" );
    XtSetSensitive( tempW, False );

  } /* if */


  /* Empty the calendar name and location field. */
  if( flagIsSet( action, RESET_CAL_FIELD ) ) {

    tempW = XtNameToWidget( mainW, "CalDataRc.CalNameTx" );
    XmTextSetString( tempW, "" );

    tempW = XtNameToWidget( mainW, "CalDataRc.CalLocTx" );
    XmTextSetString( tempW, "" );

    cal_ref -> field_changed = False;

  } /* if */


  return;

} /* setCalendarData */


/*----------------------------------------------------------------------*/

static void
  addCB( Widget       widget,
         CAL_REC_REF  cal_ref,
         XtPointer    call_data )
{

  /* Variables. */
  Boolean           exists;
  int               items;
  char              directory[ 51 ];
  char              short_name[ 21 ];
  char              *char_ref;
  Widget            mainW;
  Widget            tempW;
  XTM_CD_CAL_INFO   db_info;
  XTM_CD_INCL_CALS  db_incl;


  /* Code. */

  mainW = XtNameToWidget( cal_ref -> calW, "CalFdFo" );


  /* Fetch the short name. */
  tempW = XtNameToWidget( mainW, "CalDataRc.CalNameTx" );

  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%15s", short_name );
  SysFree( char_ref );

  if( items != 1 )
    return;


  /* Fetch the location. */
  tempW = XtNameToWidget( mainW, "CalDataRc.CalLocTx" );

  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%50s", directory );
  SysFree( char_ref );

  if( items != 1 )
    return;


  /* Does the database already exist? */
  exists = xtmCdFetchNamedDb( cal_ref -> cal_db_handle, 
                              short_name,
                              &db_info, &db_incl );


  /* The database is already defined, change the directory. */
  if( exists ) {

    if( flagIsSet( db_info.flags, XTM_CD_FLAG_IS_INCLUDED ) ) {
      xitErMessage( cal_ref -> calW, XIT_ER_ERROR, 
                    module_name, "addCB",
                    msgGetText( MXDI_NO_MOD_INCL_CAL_ERR ) );

      return;
    }

    strcpy( db_info.directory, directory );

    (void) xtmCdChangeEntry( cal_ref -> cal_db_handle,
                             short_name,
                             &db_info, &db_incl );

  } /* if */


  /* The database does not exist, make a new one. */
  if( ! exists ) {

    /* Add the database. */
    xtmCdAddDatabase( cal_ref -> cal_db_handle,
                      short_name, directory,
                      XTM_CD_FLAG_ALARM );

    /* Fetch the added database. */
    (void) xtmCdFetchNamedDb( cal_ref -> cal_db_handle, 
                              short_name,
                              &db_info, &db_incl );

    /* If we cannot read the database, tell the user. */
    if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {
      xitErMessage( mainW, XIT_ER_ERROR,
                    module_name, "addCB",
                    msgGetText( MXDI_CUSTOM_DIR_NOT_EXIST ) );

    /* Display the customize parameters. */
    } else {
      strcpy( cal_ref -> cal_selected, short_name );

      calMoreCB( NULL, cal_ref, NULL );

    } /* if */

  } /* if */


  /* Set database in the list and clear fields. */
  setCalendarData( cal_ref, (SET_CAL_LIST | RESET_CAL_FIELD) );

  setAction( cal_ref, True, False );


  return;

} /* addCB */


/*----------------------------------------------------------------------*/

static void 
  calImportCB( Widget       widget,
               CAL_REC_REF  cal_ref,
               XtPointer    call_data )
{

  /* Code. */

  /* Is the import window initialized? */
  if( cal_ref -> import_handle == NULL )
    cal_ref -> import_handle = xtmCpInitialize( cal_ref -> custom_data_ref,
                                                cal_ref -> calW,
                                                calImportApplyCB,
                                                (void *) cal_ref );

  /* Display the include window. */
  xtmCpImportCalendar( cal_ref -> import_handle,
                       cal_ref -> cal_db_handle );


  return;

} /* calImportCB */


/*----------------------------------------------------------------------*/

static void 
  calImportApplyCB( XTM_CP_REASON     reason,
                    char              *db_name,
                    void              *user_data )
{

  /* Variables. */
  CAL_REC_REF       cal_ref;


  /* Code. */

  if( reason != XTM_CP_REASON_IMPORT )
    return;

  cal_ref = (CAL_REC_REF) user_data;


  /* Fill in all the fields and set action buttons. */
  setCalendarData( cal_ref, (SET_CAL_LIST | RESET_CAL_FIELD) );
  setAction( cal_ref, True, False );


  return;

} /* calImportApplyCB */


/*----------------------------------------------------------------------*/

static void 
  calIncludeCB( Widget       widget,
                CAL_REC_REF  cal_ref,
                XtPointer    call_data )
{

  /* Code. */

  /* Is the include window initialized? */
  if( cal_ref -> included_handle == NULL )
    cal_ref -> included_handle = xtmCiInitialize( cal_ref -> custom_data_ref,
                                                  cal_ref -> calW,
                                                  calIncludeApplyCB,
                                                  (void *) cal_ref );

  /* Display the include window. */
  xtmCiEditCustInclude( cal_ref -> included_handle,
                        cal_ref -> cal_db_handle,
                        cal_ref -> cal_selected );


  return;

} /* calIncludeCB */


/*----------------------------------------------------------------------*/

static void 
  calIncludeApplyCB( XTM_CI_REASON     reason,
                     char              *db_name,
                     XTM_CD_INCL_CALS  *new_db_incl_ref,
                     void              *user_data )
{

  /* Variables. */
  Boolean           ok;
  CAL_REC_REF       cal_ref;
  XTM_CD_CAL_INFO   db_info;
  XTM_CD_INCL_CALS  db_incl;


  /* Code. */

  if( reason != XTM_CI_REASON_OK )
    return;

  cal_ref = (CAL_REC_REF) user_data;


  /* Fetch the database we are changing. */
  ok = xtmCdFetchNamedDb( cal_ref -> cal_db_handle, db_name,
                          &db_info, &db_incl );
  if( ! ok )
    return;


  /* Change the contents. */
  (void) xtmCdChangeEntry( cal_ref -> cal_db_handle, db_name,
                           &db_info, new_db_incl_ref );


  return;

} /* calIncludeApplyCB */


/*----------------------------------------------------------------------*/

static void 
  calListCB( Widget                widget, 
             CAL_REC_REF           cal_ref,
             XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char  db_name[ 50 ];
  char  *char_ref;


  /* Code. */

  /* Fetch the selected calendar line. */
  char_ref = xitStringGetString( call_data -> item, CS );

  sscanf( char_ref, "%s", db_name );
  SysFree( char_ref );

  calIsSelected( cal_ref, db_name );


  return;

} /* calListCB */


/*----------------------------------------------------------------------*/

static void 
  calMoreCB( Widget       widget,
             CAL_REC_REF  cal_ref,
             XtPointer    call_data )
{

  /* Variables. */
  Boolean                 ok;
  Cardinal                n;
  XIT_PS_LABELS           param_labels;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         *db_ref;

  static XIT_PAR_REC  par_rec[] = {

    { "", "", XIT_PS_TYPE_YES_NO_FLAG,          /* Alarm process */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO_FLAG,          /* Hide in calendar */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_COLOR_INDEX_FIELD,    /* Include color */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO_FLAG,          /* Notify if changed */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SELECT_INDEX_FIELD,   /* Stacking order for entr. */
      0, 0, 20, 20, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_YES_NO_FLAG,          /* Default DB */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_TEXT_FIELD,           /* Mail address */
      0, 0, 20, 25, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO_FLAG,          /* Start as icon */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_YES_NO_FLAG,          /* Start as window */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_YES_NO_FLAG,          /* Is remote calendar? */
      0, 0, 0, 0, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_TEXT_FIELD,           /* Remote host. */
      0, 0, 20, 40, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_HIDDEN_TEXT_FIELD,    /* Remote password. */
      0, 0, 20, 20, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_TEXT_FIELD,           /* Remote type. */
      0, 0, 10, 10, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_TEXT_FIELD,           /* Remote user. */
      0, 0, 20, 20, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SEPARATOR,
      0, 0, 0, 0, "", 0, NULL, NULL, 1 },
    { "", "", XIT_PS_TYPE_SELECT_FIELD,         /* Alarm tags */
      0, 0, 20, 50, "", 0, NULL, NULL, 0 },
    { "", "", XIT_PS_TYPE_SELECT_FIELD,         /* View tags */
      0, 0, 20, 50, "", 0, NULL, NULL, 0 },
  };

  static int     default_db_color = 0;
  static int     win_stack_order = 0;
  static UINT32  default_flags = 0;


  /* Code. */

  /* Labels in the parameter window. */
  param_labels.window_title        = msgGetText( MXDI_CUSTOM_TITLE );
  param_labels.parameter_list      = msgGetText( MXDI_PARAMETERS_LABEL );
  param_labels.description         = msgGetText( MXDI_DESCRIPTION_LABEL );
  param_labels.parameter_value     = msgGetText( MXDI_VALUE_LABEL );
  param_labels.yes_label           = msgGetText( MXDI_YES_LABEL );
  param_labels.no_label            = msgGetText( MXDI_NO_LABEL );
  param_labels.on_label            = msgGetText( MXDI_OFF_LABEL );
  param_labels.off_label           = msgGetText( MXDI_ON_LABEL );
  param_labels.cancel_button       = msgGetText( MXDI_CANCEL_BUTTON );
  param_labels.defaults_button     = msgGetText( MXDI_DEFAULTS_LABEL );
  param_labels.ok_button           = msgGetText( MXDI_OK_BUTTON );
  param_labels.reset_button        = msgGetText( MXDI_RESET_BUTTON );
  param_labels.date_order_label    = msgGetText( MXDI_DATE_ORDER_LABEL );
  param_labels.mdy_label           = msgGetText( MXDI_MDY_LABEL );
  param_labels.dmy_label           = msgGetText( MXDI_DMY_LABEL );
  param_labels.ymd_label           = msgGetText( MXDI_YMD_LABEL );
  param_labels.date_sep_label      = msgGetText( MXDI_DATE_SEPARATOR_LABEL );
  param_labels.day_zero_label      = msgGetText( MXDI_DAY_LEAD_ZERO_LABEL );
  param_labels.month_zero_label    = msgGetText( MXDI_MONTH_LEAD_ZERO_LABEL );
  param_labels.century_label       = msgGetText( MXDI_CENTURY_LABEL );
  param_labels.hour12_label        = msgGetText( MXDI_DATE_HOUR12_LABEL );
  param_labels.hour24_label        = msgGetText( MXDI_DATE_HOUR24_LABEL );
  param_labels.time_sep_label      = msgGetText( MXDI_TIME_SEPARATOR_LABEL );
  param_labels.hour_zero_label     = msgGetText( MXDI_HOUR_LEAD_ZERO_LABEL );
  param_labels.week_1st_day_label  = msgGetText( MXDI_WEEK_1ST_DAY_LABEL );
  param_labels.week_1st_days       = msgGetText( MXDI_WEEK_1ST_DAYS );
  param_labels.change_button       = msgGetText( MXDI_CHANGE_BUTTON );
  param_labels.change_hidden_label = msgGetText( MXDI_ENTER_HIDDEN_LABEL );
  param_labels.weekdays_short      = msgGetText( MXDI_WEEKDAYS_MO_FIRST );

  param_labels.msg_confirm_default = msgGetText( MXDI_OVERWRITE_PARAM_CONF );
  param_labels.msg_confirm_reset   = msgGetText( MXDI_OVERWRITE_PARAM_CONF );
  param_labels.msg_invalid_value   = msgGetText( MXDI_NOT_VALID_PARAM );


  custom_data_ref = cal_ref -> custom_data_ref;


  /* Fetch the calendar data and save in placeholder. */
  ok = xtmCdFetchNamedDb( cal_ref -> cal_db_handle, 
                          cal_ref -> cal_selected,
                          &cal_ref -> param_db_info, NULL );
  if( ! ok )
    return;

  db_ref = &cal_ref -> param_db_info;

  /* Pretend we have an external password. */
  cal_ref -> param_db_info.has_extern_pwd = True;


  /* Parameter values. */
  n = 0;

  /* Alarm process. */
  par_rec[ n ].parameter_label = msgGetText( MXDI_DB_ALARM_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DB_ALARM_HELP );
  par_rec[ n ].value           = &db_ref -> flags;
  par_rec[ n ].default_value   = &default_flags;
  par_rec[ n ].size            = sizeof( db_ref -> flags );
  par_rec[ n ].flag_offset     = XTM_CD_FLAG_ALARM;

  /* Hide in calendar. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_HIDE_CAL_IN_CAL_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_HIDE_CAL_IN_CAL_HELP );
  par_rec[ n ].value           = &db_ref -> flags;
  par_rec[ n ].default_value   = &default_flags;
  par_rec[ n ].size            = sizeof( db_ref -> flags );
  par_rec[ n ].flag_offset     = XTM_CD_FLAG_HIDE_IN_CAL;

  /* Include color. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_INCL_COLOR_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_INCL_COLOR_HELP );
  par_rec[ n ].value           = &db_ref -> incl_bg_index;
  par_rec[ n ].default_value   = &default_db_color;
  par_rec[ n ].size            = sizeof( db_ref -> incl_bg_index );
  par_rec[ n ].select_values   = custom_data_ref -> db_color_bg_list;

  /* Notify if cahnegd. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_NOTIFY_CHANGED_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_NOTIFY_CHANGED_HELP );
  par_rec[ n ].value           = &db_ref -> flags;
  par_rec[ n ].default_value   = &default_flags;
  par_rec[ n ].size            = sizeof( db_ref -> flags );
  par_rec[ n ].flag_offset     = XTM_CD_FLAG_NOTIFY;

  /* Stacking order for entr. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_CAL_STACKING_ORDER_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_CAL_STACKING_ORDER_HELP );
  par_rec[ n ].value           = &db_ref -> win_stack_order;
  par_rec[ n ].default_value   = &win_stack_order;
  par_rec[ n ].size            = sizeof( db_ref -> win_stack_order );
  par_rec[ n ].select_values   = msgGetText( MXDI_CAL_STACKING_ORDER_LIST );

  /* Separator. */
  n++;

  /* Default DB. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_DEFAULT_DB_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_DEFAULT_DB_HELP );
  par_rec[ n ].value           = &db_ref -> flags;
  par_rec[ n ].default_value   = &default_flags;
  par_rec[ n ].size            = sizeof( db_ref -> flags );
  par_rec[ n ].flag_offset     = XTM_CD_FLAG_DEFAULT_DB;

  /* Mail address. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_MAIL_ADDR_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_MAIL_ADDR_HELP );
  par_rec[ n ].value           = db_ref -> mail_address;
  par_rec[ n ].default_value   = cal_ref -> cal_selected;
  par_rec[ n ].size            = sizeof( db_ref -> mail_address );

  /* Start as icon. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_START_ICON_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_START_ICON_HELP );
  par_rec[ n ].value           = &db_ref -> flags;
  par_rec[ n ].default_value   = &default_flags;
  par_rec[ n ].size            = sizeof( db_ref -> flags );
  par_rec[ n ].flag_offset     = XTM_CD_FLAG_DEFAULT_START_ICON;

  /* Start as window. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_START_WINDOW_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_START_WINDOW_HELP );
  par_rec[ n ].value           = &db_ref -> flags;
  par_rec[ n ].default_value   = &default_flags;
  par_rec[ n ].size            = sizeof( db_ref -> flags );
  par_rec[ n ].flag_offset     = XTM_CD_FLAG_DEFAULT_START_WIN;

  /* Separator. */
  n++;

  /* Is remote calendar? */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_IS_REMOTE_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_IS_REMOTE_HELP );
  par_rec[ n ].value           = &db_ref -> flags;
  par_rec[ n ].default_value   = &default_flags;
  par_rec[ n ].size            = sizeof( db_ref -> flags );
  par_rec[ n ].flag_offset     = XTM_CD_FLAG_IS_EXTERNAL;

  /* Remote host. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_REMOTE_HOST_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_REMOTE_HOST_HELP );
  par_rec[ n ].value           = db_ref -> extern_host;
  par_rec[ n ].default_value   = "";
  par_rec[ n ].size            = sizeof( db_ref -> extern_host );

  /* Remote password. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_REMOTE_PWD_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_REMOTE_PWD_HELP );
  par_rec[ n ].value           = db_ref -> extern_pwd;
  par_rec[ n ].default_value   = "";
  par_rec[ n ].size            = sizeof( db_ref -> extern_pwd );

  /* Remote type. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_REMOTE_TYPE_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_REMOTE_TYPE_HELP );
  par_rec[ n ].value           = db_ref -> extern_type;
  par_rec[ n ].default_value   = "";
  par_rec[ n ].size            = sizeof( db_ref -> extern_type );

  /* Remote user. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_REMOTE_USER_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_REMOTE_USER_HELP );
  par_rec[ n ].value           = db_ref -> extern_uid;
  par_rec[ n ].default_value   = "";
  par_rec[ n ].size            = sizeof( db_ref -> extern_uid );

  /* Separator. */
  n++;

  /* Alarm tags. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_ALARM_TAGS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_ALARM_TAGS_HELP );
  par_rec[ n ].value           = db_ref -> alarm_tags;
  par_rec[ n ].default_value   = "";
  par_rec[ n ].size            = sizeof( db_ref -> alarm_tags );
  par_rec[ n ].select_values   = custom_data_ref -> entry_tags_menu;

  /* View tags. */
  n++;
  par_rec[ n ].parameter_label = msgGetText( MXDI_VIEW_TAGS_PARAM );
  par_rec[ n ].parameter_help  = msgGetText( MXDI_VIEW_TAGS_HELP );
  par_rec[ n ].value           = db_ref -> view_tags;
  par_rec[ n ].default_value   = "";
  par_rec[ n ].size            = sizeof( db_ref -> view_tags );
  par_rec[ n ].select_values   = custom_data_ref -> entry_tags_menu;


  /* Create the parameter window? */
  if( cal_ref -> paramW == NULL )
    cal_ref -> paramW = xitCreateParameterSet( cal_ref -> calW,
                                               0, &param_labels,
                                               calParamChangedCB,
                                               (void *) cal_ref );

  /* Display the window and set the parameters. */
  XtManageChild( cal_ref -> paramW );

  xitParameterSetDisplay( cal_ref -> paramW, 
                          &par_rec[ 0 ], XtNumber( par_rec ) );


  return;

} /* calMoreCB */


/*----------------------------------------------------------------------*/

static void
  calParamChangedCB( XIT_PS_REASON  reason,
                     void           *client_data )
{

  /* Variables. */
  CAL_REC_REF       cal_ref;
  XTM_CD_CAL_INFO   db_info;
  XTM_CD_INCL_CALS  db_incl;


  /* Code. */

  if( reason != XIT_PS_REASON_OK )
    return;

  cal_ref = (CAL_REC_REF) client_data;

  /* Fetch the current database. */
  (void) xtmCdFetchNamedDb( cal_ref -> cal_db_handle, 
                            cal_ref -> cal_selected,
                            &db_info, &db_incl );

  /* Change the contents. */
  (void) xtmCdChangeEntry( cal_ref -> cal_db_handle,
                           cal_ref -> cal_selected,
                           &cal_ref -> param_db_info, &db_incl );
    
  /* Update the database fields. */
  setCalendarData( cal_ref, SET_CAL_LIST );


  return;

} /* calParamChangedCB */


/*----------------------------------------------------------------------*/

static void 
  calSelectCB( Widget       widget,
               CAL_REC_REF  cal_ref,
               XtPointer    call_data )
{

  /* Code. */

  /* Initialize? */
  if( cal_ref -> pick_handle == NULL )
    cal_ref -> pick_handle = xtmPdInitialize(
                               cal_ref -> calW,
                               True,
                               cal_ref -> appl_data_ref -> info_handle,
                               calSelectApplyCB,
                               (void *)cal_ref );

  /* Diaplay the window. */
  xtmPdDisplayPickWindow( cal_ref -> pick_handle,
                          cal_ref -> custom_data_ref -> db_selection_dir,
                          "*_cal" );

  return;

} /* calSelectCB */


/*----------------------------------------------------------------------*/

static void
  calSelectApplyCB( XTM_PD_REASON    reason,
                    XTM_CD_CAL_INFO  *db_ref,
                    void             *client_data )
{

  /* Variables. */
  Widget       mainW;
  Widget       tempW;
  CAL_REC_REF  cal_ref;


  /* Code. */

  if( reason != XTM_PD_REASON_OK )
    return;

  cal_ref = (CAL_REC_REF) client_data;
  mainW   = XtNameToWidget( cal_ref -> calW, "CalFdFo" );

  tempW = XtNameToWidget( mainW, "CalDataRc.CalNameTx" );
  XmTextSetString( tempW, db_ref -> short_name );

  tempW = XtNameToWidget( mainW, "CalDataRc.CalLocTx" );
  XmTextSetString( tempW, db_ref -> directory );
  

  return;

} /* calSelectApplyCB */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget       widget,
            CAL_REC_REF  cal_ref,
            XtPointer    call_data )
{

  /* Variables. */

  /* Code. */

  XtUnmanageChild( cal_ref -> calW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void
  deleteCB( Widget       widget,
            CAL_REC_REF  cal_ref,
            XtPointer    call_data )
{

  /* Variables. */
  Boolean          ok;
  int              selected_item_count;
  char             short_name[ 21 ];
  char             *char_ref;
  Arg              args[ 3 ];
  Cardinal         n;
  Widget           mainW;
  Widget           tempW;
  XmString         *selected_items;
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  mainW = XtNameToWidget( cal_ref -> calW, "CalFdFo" );


  /* Fetch selected items from the list. */
  tempW = XtNameToWidget( mainW, "CalLiSW.CalLi" );

  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems, &selected_items ); n++;
  XtGetValues( tempW, args, n );

  if( selected_item_count != 1 )
    return;  


  /* Remove the database from the database list. */
  char_ref = xitStringGetString( *selected_items, CS );

  sscanf( char_ref, "%s", short_name );
  SysFree( char_ref );


  /* Fetch the database to delete. */
  ok = xtmCdFetchNamedDb( cal_ref -> cal_db_handle, short_name,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* Delete the entry. */
  (void) xtmCdDeleteEntry( cal_ref -> cal_db_handle, short_name );


  /* Remove the database from the list and clear fields. */
  setCalendarData( cal_ref, (SET_CAL_LIST | RESET_CAL_FIELD) );

  setAction( cal_ref, True, False );


  return;

} /* deleteCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget       widget,
             CAL_REC_REF  cal_ref,
             XtPointer    call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_CC_REASON_DESTROY,
                             NULL,
                             cal_ref -> user_data );


  /* Release the user data. */
  if( cal_ref -> cal_db_handle != NULL )
    xtmCdFree( cal_ref -> cal_db_handle );

  if( cal_ref -> included_handle != NULL )
    xtmCiDestroy( cal_ref -> included_handle );

  if( cal_ref -> pick_handle != NULL )
    xtmPdDestroy( cal_ref -> pick_handle );

  SysFree( cal_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  fieldChangedCB( Widget       widget,
                  CAL_REC_REF  cal_ref,
                  XtPointer    call_data )
{

  /* Code. */

  cal_ref -> field_changed = True;


  return;

} /* fieldChangedCB */


/*----------------------------------------------------------------------*/

static void
  guessLocationCB( Widget       widget,
                   CAL_REC_REF  cal_ref,
                   XtPointer    call_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     items;
  char                    cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  char                    location[ 51 ];
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = cal_ref -> custom_data_ref;
  mainW           = XtNameToWidget( cal_ref -> calW, "CalFdFo" );


  /* Fetch the calendar name. */
  tempW    = XtNameToWidget( mainW, "CalDataRc.CalNameTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%s", cal_name );
  SysFree( char_ref );

  if( items != 1 ) {
    xitErMessage( cal_ref -> calW, XIT_ER_INFO,
                  module_name, "guessLocationCB",
                  msgGetText( MXDI_MISSING_CALENDAR_NAME ) );

    return;
  }


  /* Do we know of the calendar already? */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, cal_name,
                          &db_info, NULL );

  if( ok && flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {
    tempW = XtNameToWidget( mainW, "CalDataRc.CalLocTx" );
    XmTextSetString( tempW, db_info.directory );

    return;
  }


  /* Try to locate the calendar. */
  ok = xtmLcLocateCalendar( cal_ref -> calW, cal_name,
                            custom_data_ref -> find_location_script,
                            location, sizeof( location ) );

  if( ! ok || strlen( location ) == 0 ) {
    xitErMessage( cal_ref -> calW, XIT_ER_INFO,
                  module_name, "guessLocationCB",
                  msgGetText( MXDI_CANNOT_LOCATE_CALENDAR ) );

    return;
  }

  tempW = XtNameToWidget( mainW, "CalDataRc.CalLocTx" );
  XmTextSetString( tempW, location );


  return;

} /* guessLocationCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget       widget,
          CAL_REC_REF  cal_ref,
          XtPointer    call_data )
{

  /* Code. */

  xtmHlDisplayHelp( cal_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    cust_cals_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget       widget,
        CAL_REC_REF  cal_ref,
        XtPointer    call_data )
{

  /* Code. */

  /* Any changes that are not saved? */
  if( cal_ref -> field_changed ) {
    (void) xitCreateQuestionDialog( 
             cal_ref -> calW, "QuestionDialog", 
	     msgGetText( MXDI_QUESTION_MESSAGE_LABEL ),
             msgGetText( MXDI_INFORMATION_NOT_SAVED ),
             okCB, cal_ref,
             NULL, NULL );

    cal_ref -> field_changed = False;
    return;
  }


  /* Do we have a user action callback registered? */
  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_CC_REASON_OK,
                             cal_ref -> cal_db_handle,
                             cal_ref -> user_data );


  XtUnmanageChild( cal_ref -> calW );


  return;

} /* okCB */
