/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/

#include "nsMdpAgent.h" 
#include "ip.h"  	// for hdr_ip def
#include "flags.h"  // for hdr_flags def


extern bool use_rtt_rate_bias;              // true by default
extern bool use_asymmetric_rtt_smoothing;   // false by default
extern bool use_ewma_loss_estimate;         // false by default

static class NsMdpAgentClass : public TclClass
{
	public:
		NsMdpAgentClass() : TclClass("Agent/MDP") {}
	 	TclObject *create(int argc, const char*const* argv) 
			{return (new NsMdpAgent());}
} class_mdp_agent;	

nsaddr_t NsMdpAgent::MulticastAddrMask()
{
    return 0x80000000;  // ns version > 2.1b5 uses 32-bit addressing always
                        // with most significant bit indicating mcast
/* Old code for old-style ns addresses
    Tcl& tcl = Tcl::instance();	
	tcl.evalf("AddrParams set McastShift_");
	const char *result = tcl.result();
    unsigned int shift = atoi(result);
    return ((nsaddr_t)(0x01 << shift)); */
}  // end NsMdpAgent::MulticastMask()



NsMdpAgent::NsMdpAgent()
	: Agent(PT_UDP), mcast_ttl(32), 
      mcast_loopback(false), mdp_socket(NULL),
      color_packets(false)
{
	// Init base_object_id to random value by default
    struct timeval theTime;
    struct timezone tz;
    gettimeofday(&theTime, &tz);
    srand(theTime.tv_usec);  // seed random number generator 
}  // end NsMdpAgent::NsMdpAgent()

bool NsMdpAgent::useUdp = false;


int NsMdpAgent::command(int argc, const char*const* argv) 
{
	if (2 == argc) 
	{
		if (strcmp(argv[1], "stop") == 0) 
		{
		  Stop();
		  return TCL_OK;
		}
		else if (0 == strcmp(argv[1], "txRate"))
		{
			Tcl& tcl = Tcl::instance();
			tcl.resultf("%lu", TxRate());
			return TCL_OK;
		}
		else if (0 == strcmp(argv[1], "addr"))
		{
			Tcl& tcl = Tcl::instance();
			tcl.resultf("%lu", GetAgentId());
			return TCL_OK;
		}
        else if (0 == strcmp(argv[1], "useUdp"))
		{
			useUdp = true;
			return TCL_OK;
		}
        else if (0 == strcmp(argv[1], "goodput"))
		{
			Tcl& tcl = Tcl::instance();
			tcl.resultf("%f", ClientGoodput());
			return TCL_OK;
		}
        else if (0 == strcmp(argv[1], "resetGoodput"))
		{
			ResetClientGoodput();
			return TCL_OK;
		}
    } 	
	else if (3 == argc)
	{
		if (0 == strcmp(argv[1], "ttl"))
		{
			unsigned char theTTL = atoi(argv[2]);
			SetSessionTTL(theTTL);
			return TCL_OK;
		}
        else if (0 == strcmp(argv[1], "congestionControl"))
        {
            bool state;
            if (0 == strcmp(argv[2], "on"))  
                state = true;
            else if (0 == strcmp(argv[2], "off"))   
                state = false;
            else
                return TCL_ERROR;
            SetSessionCongestionControl(state);
            return TCL_OK;
        }
        else if (0 == strcmp(argv[1], "fastStart"))
        {
            bool state;
            if (0 == strcmp(argv[2], "on"))  
                state = true;
            else if (0 == strcmp(argv[2], "off"))   
                state = false;
            else
                return TCL_ERROR;
            SetSessionFastStart(state);
            return TCL_OK;
        }
        else if (0 == strcmp(argv[1], "ecnEnable"))
        {
            bool state;
            if (0 == strcmp(argv[2], "on"))  
                state = true;
            else if (0 == strcmp(argv[2], "off"))   
                state = false;
            else
                return TCL_ERROR;
            SetSessionEcnEnable(state);
            return TCL_OK;  
        }
        else if (0 == strcmp(argv[1], "rttRateBias"))
        {
            if (0 == strcmp(argv[2], "on"))  
                use_rtt_rate_bias = true;
            else if (0 == strcmp(argv[2], "off"))   
                use_rtt_rate_bias = false;
            else
                return TCL_ERROR;
            return TCL_OK;  
        }
        else if (0 == strcmp(argv[1], "asymRttSmoothing"))
        {
            if (0 == strcmp(argv[2], "on"))  
                use_asymmetric_rtt_smoothing = true;
            else if (0 == strcmp(argv[2], "off"))   
                use_asymmetric_rtt_smoothing = false;
            else
                return TCL_ERROR;
            return TCL_OK;  
        }
        else if (0 == strcmp(argv[1], "ewmaLossEstimate"))
        {
            if (0 == strcmp(argv[2], "on"))  
                use_ewma_loss_estimate = true;
            else if (0 == strcmp(argv[2], "off"))   
                use_ewma_loss_estimate = false;
            else
                return TCL_ERROR;
            return TCL_OK;  
        }
        else if (0 == strcmp(argv[1], "colorPackets"))
        {
            if (0 == strcmp(argv[2], "on"))  
                color_packets = true;
            else if (0 == strcmp(argv[2], "off"))   
                color_packets = false;
            else
                return TCL_ERROR;
            return TCL_OK;  
        }
		else if (0 == strcmp(argv[1], "txRate"))
		{
            unsigned long theRate = 0;
			if (1 != sscanf(argv[2], "%lu", &theRate))
            {
                return TCL_ERROR;
            }
            else
            {
			    SetSessionTxRate(theRate);
                return TCL_OK;
            }
		}
		else if (0 == strcmp(argv[1], "segmentSize"))
		{
			if (SetSessionSegmentSize(atoi(argv[2])))
	            return TCL_OK;
            else
                return TCL_ERROR;
		}
		else if (0 == strcmp(argv[1], "blockSize"))
		{
			if (SetSessionBlockSize(atoi(argv[2])))
	            return TCL_OK;
            else
                return TCL_ERROR;
		}
		else if (0 == strcmp(argv[1], "numParity"))
		{
			if (SetSessionNumParity(atoi(argv[2])))
	            return TCL_OK;
            else
                return TCL_ERROR;
		}
		else if (0 == strcmp(argv[1], "autoParity"))
		{
			SetSessionAutoParity(atoi(argv[2]));
	        return TCL_OK;
		}
		else if (0 == strcmp(argv[1], "numRepeats"))
		{
			SetNumRepeats(atoi(argv[2]));
	        return TCL_OK;
		}
		else if (0 == strcmp(argv[1], "unicastNacks"))
		{
            if (0 == strcmp(argv[2], "on"))  
                SetSessionUnicastNacks(true);
            else if (0 == strcmp(argv[2], "off"))   
                SetSessionUnicastNacks(false);
            else
                return TCL_ERROR;
            return TCL_OK;
		}
        else if (0 == strcmp(argv[1], "emcon"))
		{
            bool state;
            if (0 == strcmp(argv[2], "on"))  
                state = true;
            else if (0 == strcmp(argv[2], "off"))   
                state = false;
            else
                return TCL_ERROR;
			if (SetSessionEmcon(state))
	            return TCL_OK;
            else
                return TCL_ERROR;
		}
		else if (0 == strcmp(argv[1], "txBufferSize"))
		{
            unsigned long theSize;
			sscanf(argv[2], "%lu", &theSize);
            if (SetSessionTxBufferSize(theSize))
			    return TCL_OK;
            else
                return TCL_ERROR;
		}
		else if (0 == strcmp(argv[1], "rxBufferSize"))
		{
			unsigned long theSize;
			sscanf(argv[2], "%lu", &theSize);
            if (SetSessionRxBufferSize(theSize))
			    return TCL_OK;
            else
                return TCL_ERROR;
		}
        else if (0 == strcmp(argv[1], "baseObject"))
        {
            if(SetSessionBaseObjectId(atoi(argv[2])))
                return TCL_OK;
            else
                return TCL_ERROR;   
        }
        else if (0 == strcmp(argv[1], "backoffWindow"))
        {
            float theWindow;
            if (1 == sscanf(argv[2], "%f", &theWindow))
            {
                SetMdpBackoffWindow(theWindow);   
                return TCL_OK;
            }
            else
            {
                return TCL_ERROR;
            }
        }
        
        else if (0 == strcmp(argv[1], "groupSize"))
        {
            unsigned int theSize;
            if (1 == sscanf(argv[2], "%u", &theSize))
            {
                SetMdpGroupSize(theSize);   
                return TCL_OK;
            }
            else
            {
                return TCL_ERROR;
            }
        }
		else if (0 == strcmp(argv[1], "debugLevel"))
		{
			SetDebugLevel(atoi(argv[2]));
			return TCL_OK;
		}
		else if (0 == strcmp(argv[1], "messageTrace"))
		{
			if (!strcmp("on", argv[2]))
				SetMdpMessageTrace(true);
			else if (!strcmp("off", argv[2]))
				SetMdpMessageTrace(false);
            else
                return TCL_ERROR;
			return TCL_OK;
		}
		else if (0 == strcmp(argv[1], "logFile"))
		{
			StopLogging();
			if (StartLogging(argv[2]))
				return (TCL_OK);
			else
				return (TCL_ERROR);
		} 
        else if (0 == strcmp(argv[1], "send"))
        {
            if (QueueTxObject(atoi(argv[2])))
                return TCL_OK;
            else
                return TCL_ERROR;
        }
        else if (0 == strcmp(argv[1], "sendLoss"))
        {
            float lossRate;
            if (1 == sscanf(argv[2], "%f", &lossRate))
            {
                SetSendDropRate(lossRate);
                return TCL_OK;
            }
            else
            {
                return TCL_ERROR;
            }
        }
        else if (0 == strcmp(argv[1], "recvLoss"))
        {
            float lossRate;
            sscanf(argv[2], "%f", &lossRate);
            SetRecvDropRate(lossRate);
            return TCL_OK;
        }
        else if (0 == strcmp(argv[1], "initialGrtt"))
        {
            float grttValue;
            sscanf(argv[2], "%f", &grttValue);
            SetGrttEstimate(grttValue);
            return TCL_OK;
        }
	}
	else if (4 == argc)
	{
		if (strcmp(argv[1], "txRateBounds") == 0) 
        {
            float min, max;
            if (1 != sscanf(argv[2], "%f", &min)) return TCL_ERROR;
            if (1 != sscanf(argv[3], "%f", &max)) return TCL_ERROR;
            SetTxRateBounds(min, max);
            return TCL_OK;
        }
	}
	else if (5 == argc)
    {
        if (strcmp(argv[1], "start") == 0) 
		{
			nsaddr_t theGroup;
			sscanf(argv[3], "%i", &theGroup);
            unsigned short thePort = atoi(argv[4]);
			if (strcmp(argv[2], "client") == 0)
			{
				if (StartClient(theGroup, thePort))
                {
                    client_start_time = Scheduler::instance().clock();
                    return TCL_OK;
                }
                else
                {
                    return TCL_ERROR;
                }
			}
            else if (strcmp(argv[2], "server") == 0)
            {
                if (StartServer(theGroup, thePort))
                    return TCL_OK;
                else
                    return TCL_ERROR;
                    
            }
		}	
    }
    return Agent::command(argc, argv);
}  // end NsMdpAgent::command()


// (TBD) When multiple sessions/sockets per MDP Agent are supported
// group joins/leaves will need to track which socket.
bool NsMdpAgent::JoinGroup(UdpSocket* /*theSocket*/, SIMADDR theGroup)
{
	Tcl& tcl = Tcl::instance();	
	//tcl.evalf("%s info vars node_", this->name());
	tcl.evalf("%s set node_", this->name());
	const char *result = tcl.result();
    Tcl::instance().evalf("%s join-group %s 0x%x",
			                result, this->name(), 
							theGroup);	
	return true;
}  // end NsMdpAgent::JoinGroup() 

void NsMdpAgent::LeaveGroup(UdpSocket* /*theSocket*/, SIMADDR theGroup)
{
	Tcl& tcl = Tcl::instance();	
	tcl.evalf("%s info vars node_", this->name());
	const char *result = tcl.result();
    
	Tcl::instance().evalf("%s leave-group %s 0x%x",
			                result, this->name(), 
							theGroup);	
}  // end NsMdpAgent::LeaveGroup() 

/*void NsMdpAgent::SetReceiptStatus(bool state)
{
    Tcl& tcl = Tcl::instance();	
	tcl.evalf("%s set node_", this->name());
	const char *result = tcl.result();
    Tcl::instance().evalf("%s color %s",
            result, (state ? "green" : "black"));
}  // end NsMdpAgent::SetReceiptStatus() */


bool NsMdpAgent::SendTo(UdpSocket* /*theSocket*/, 
                        SIMADDR theAddr, unsigned short thePort,
                        char* buffer, unsigned int len)
{	
    // Allocate _and_ init packet
	Packet* p = allocpkt(len);
	// Set packet header fields as needed
	hdr_ip* iph = hdr_ip::access(p);
	hdr_flags::access(p)->ect() = 1;
    
    
	iph->daddr() = theAddr;
	iph->dport() = thePort;
    
    if (theAddr & MulticastAddrMask())
    {
        iph->ttl() = mcast_ttl;
    }
	else
    {
		iph->ttl() = 255;
    }
	
	hdr_cmn* ch = hdr_cmn::access(p);
    
    // Assign packet types & flow id's based on MDP packet types
    // MDP_REPORT is the lowest ordinal packet type
    int packetType = (buffer[0] - MDP_REPORT) + PT_MDP_REPORT;
    
    if (useUdp)
        ch->ptype() = PT_UDP;
    else
        ch->ptype() = (packet_t) packetType;
    
    
    // Flow id's are only overidden when MDP packet coloring is wanted
    if (color_packets) iph->flowid() = packetType;    
  					
	// Copy data to be sent into packet
	// (TBD) In a simulation environment, actually ending data content is 
	// only required for certain MDPv2 control messages, NACKs, etc ... we 
	// could save some simulation host resources by being smarter here
	char* data = (char*) p->accessdata();
	memcpy(data, buffer, len);
	// Set packet length field to MDP payload size for now
    // (TBD) Should we add UDP/IP header overhead?
	ch->size() = len;
	send(p, 0);
    return true;
}  // end NsMdpAgent::SendTo()

// If multiple sockets/agent are to be supported, this function
// will need to be changed to demux recv packets to the 
// correct "socket(s)" based on packet destination address/port

void NsMdpAgent::recv(Packet *p, Handler */*h*/)
{
    // Give the packet to the socket given mcast/loopback/packet source
	nsaddr_t srcAddr = hdr_ip::access(p)->src().addr_;
    unsigned short srcPort = hdr_ip::access(p)->src().port_;
	nsaddr_t dstAddr = hdr_ip::access(p)->dst().addr_;
    
    bool isUnicast = (0 == (dstAddr & MulticastAddrMask()));
    
    if ((srcAddr != here_.addr_) || (!isUnicast && mcast_loopback))
	{
		
		// if NULL == mdp_socket, mdp_socket was closed before packet was received
		
	
		// Hack to check for ECN flag
		if(hdr_flags::access(p)->ecn_to_echo_)
		{
            SetSessionEcnStatus(true);
			//fprintf(stdout, "Time:%f Node:%d received packet with ECN flag!\n", 
			//			     Scheduler::instance().clock(), addr_);
		}
        else
        {
            SetSessionEcnStatus(false);
        }
    
        if (mdp_socket)
        {
            char* data = (char*)p->accessdata();
            unsigned int len = ((PacketData*)(p->userdata()))->size();
            HandleRecvPacket(data, len, srcAddr, srcPort, isUnicast);
	
        }
    }  
	Packet::free(p);
    
}  // end NsMdpAgent::recv()



double NsMdpAgent::ClientGoodput()
{
    if (IsClient())
    {
        double clientDuration = Scheduler::instance().clock() - client_start_time;
        double goodBytes = (4.295e+09 * ((double)goodput_recvd_high)) + goodput_recvd_low;
        return ((8.0 * goodBytes) / (1000.0 * clientDuration));
    }
    else
    {
        return -1.0;
    }
}  // end NsMdpAgent::ClientGoodput()
