/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#ifndef _MDP_SESSION
#define _MDP_SESSION

#include "udpSocket.h"
#include "mdpMessage.h"
#include "mdpObject.h"
#include "mdpNode.h"
#include "sysdefs.h"   // for PATH_MAX

// Message tracing option for debugging
// Currently global
#ifdef PROTO_DEBUG
extern bool mdp_trace;
void SetMdpMessageTrace(bool state);
#endif

/********************************************************************
 * The MdpSession class maintains the primary state for a single instance
 * of the MDPv2 Protocol for a particular destination address/port
 */

// Some session constants (TBD) Move to appropriate places
const int SERVER_PACK_REQ_REPEAT = 20;
const unsigned int MDP_RECV_START_THRESHOLD = 2;


// These types are used for the opaque MDP API
typedef const void* MdpInstanceHandle;
typedef const void* MdpSessionHandle;
typedef const void* MdpNodeHandle;
typedef const void* MdpObjectHandle;
const MdpObjectHandle MDP_NULL_OBJECT = ((MdpObjectHandle)0);
typedef unsigned long MdpNodeId;
const MdpNodeId MDP_NULL_NODE = ((MdpNodeId)0);


typedef bool (MdpNotifyCallback)(MdpNotifyCode      notifyCode,
                                 MdpInstanceHandle  instanceHandle,
                                 MdpSessionHandle   sessionHandle,
                                 MdpNodeHandle      nodeHandle,
                                 MdpObjectHandle    objectHandle,
                                 MdpError           errorCode);
class MdpSessionMgr
{
    friend class MdpSession;
        
    // Methods
    public:
	    MdpSessionMgr();
        ~MdpSessionMgr();
        unsigned long LocalNodeId() {return node_id;}
        void SetLocalNodeId(unsigned long theId) {node_id = theId;}
        void SetNodeName(const char *theName) 
            {strncpy(node_name, theName, MDP_NODE_NAME_MAX);}
		void Close();  // shutdown all sessions
	    MdpSession* FindSessionByAddress(const NetworkAddress *theAddr);
        MdpSession* FindSession(const MdpSession *theSession);
        MdpSession* NewSession();
	    void DeleteSession(MdpSession *theSession);
        
        void SetNotifyCallback(MdpNotifyCallback* func)
            {notify_func = func;}
        void SetTimerInstaller(ProtocolTimerInstallFunc* func)
            {timer_mgr.SetInstaller(func, (void*)this);}
		void SetSocketInstaller(UdpSocketInstallFunc* func)
            {socket_installer = func;}     
        
        void SetUserData(const void* userData)
            {user_data = userData;}
        const void* GetUserData() 
            {return user_data;}
                
        void DoTimers() {timer_mgr.DoTimers();}
        
    private:
	    void Add(class MdpSession* theSession);
	    void Remove(class MdpSession* theSession);
        // Call the installed application notifier if there is one
        void NotifyApplication(MdpNotifyCode notifyCode,
                               MdpSession*   theSession,
                               MdpNode*      theNode,
                               MdpObject*    theObject, 
                               MdpError      errorCode);
        
        ProtocolTimerMgr        timer_mgr;
	    char                    node_name[MDP_NODE_NAME_MAX];
        unsigned long           node_id;
        class MdpSession*       list_head;
        class MdpSession*       list_tail; 
        
        const void*             user_data;
        MdpNotifyCallback*      notify_func;
        UdpSocketInstallFunc*   socket_installer;
        
};  // end class MdpSessionMgr


// Session status flags (low session status byte has MdpReport'ing status flags)
enum MdpServerFlag
{
    MDP_SERVER_EMCON    = 0x01,
    MDP_SERVER_CLOSING  = 0x02
};
    
enum MdpClientFlag
{
    MDP_CLIENT_UNICAST_NACKS    = 0x01,
    MDP_CLIENT_EMCON            = 0x02,
    MDP_CLIENT_ACKING           = 0x04,
    MDP_CLIENT_MULTICAST_ACKS   = 0x08
};
            
const unsigned char MDP_DEFAULT_TTL = 8;
const unsigned long MDP_DEFAULT_TX_RATE = 64000;
const unsigned int  MDP_DEFAULT_SEGMENT_SIZE = 1024;
const unsigned int  MDP_DEFAULT_BLOCK_SIZE = 64;
const unsigned int  MDP_DEFAULT_NPARITY = 32;

const unsigned int  MDP_DEFAULT_TX_HOLD_COUNT_MIN = 8;
const unsigned int  MDP_DEFAULT_TX_HOLD_COUNT_MAX = 64;
const unsigned int  MDP_DEFAULT_TX_HOLD_SIZE_MAX = (8*1024*1024);

const double MDP_REPORT_INTERVAL = 90.0;
const double MDP_DEFAULT_GRTT_REQ_INTERVAL_MIN = 1.0;
const double MDP_DEFAULT_GRTT_REQ_INTERVAL_MAX = 40.0;

class MdpSession 
#ifdef USE_INHERITANCE
    : public ProtocolTimerOwner, public UdpSocketOwner
#endif        
{
    friend class MdpSessionMgr;  
    friend class MdpServerNode; 
    friend class MdpObject;
    
    // Methods
    public:
	    MdpSession(MdpSessionMgr* theMgr);
	    ~MdpSession();
       // Inits session as a server
        MdpError OpenServer(unsigned short  segmentSize, 
                            unsigned char   numData, 
                            unsigned char   numParity,
                            unsigned long   bufferSize);
        void CloseServer(bool hardShutdown);
        unsigned short SegmentSize() {return segment_size;}
        MdpError OpenClient(unsigned long bufferSize);
        void CloseClient();
        void Close(bool hardShutdown);  // shutdown session
        
        void SetUserData(const void* userData)
            {user_data = userData;}
        const void* GetUserData() 
            {return user_data;}
        
        bool IsServer() {return (0 != (status & MDP_SERVER));}
        bool IsClient() {return (0 != (status & MDP_CLIENT));}
        bool IsOpen() {return (status != 0);}
        // Set/get session parameters
        void SetName(const char* theName) {strncpy(name, theName, MDP_SESSION_NAME_MAX);}
        const NetworkAddress* Address() {return &addr;}
        bool SetAddress(const char* theAddress, unsigned short rxPort,
                        unsigned short txPort = 0);
#ifdef SIMULATOR_ADDRESS
		bool SetSimAddress(SIMADDR theAddress, unsigned short thePort)
		{
			addr.SimSetAddress(theAddress);
            addr.SetPort(thePort);
            tx_port = thePort;  // Sims use single socket for now
			return true;
		}
#endif // SIMULATOR_ADDRESS		
		bool IsUnicast() {return (!addr.IsMulticast());}	
        bool SetGrttProbeInterval(double intervalMin, double intervalMax);
        void SetGrttProbing(bool state);
        void SetTTL(unsigned char theTTL);
        bool SetTOS(unsigned char theTOS);
        bool SetMulticastInterfaceAddress(const char* theAddr);
        void SetMulticastLoopback(bool state);
        void SetPortReuse(bool state);
        unsigned long TxRate() {return (tx_rate << 3);}
        void SetTxRate(unsigned long value) 
        {
            tx_rate = value >> 3;
            goal_tx_rate = tx_rate;
            double pktInterval = (double)segment_size / (double)tx_rate;
            grtt_advertise = MAX(pktInterval, grtt_measured);
            grtt_quantized = QuantizeRtt(grtt_advertise);
            grtt_advertise = UnquantizeRtt(grtt_quantized);
        }
        void SetTxRateBounds(unsigned long min, unsigned long max)
        {
            min >>= 3;
            max >>= 3;
            tx_rate_min = MIN(min, max);
            tx_rate_max = MAX(min, max);  
            if (min || max)
            {
                tx_rate = MIN(tx_rate, tx_rate_max);
                tx_rate = MAX(tx_rate, tx_rate_min);
                double pktInterval = (double)segment_size / (double)tx_rate;
                grtt_advertise = MAX(pktInterval, grtt_measured);
                grtt_quantized = QuantizeRtt(grtt_advertise);
                grtt_advertise = UnquantizeRtt(grtt_quantized); 
            }
        }
        double GrttEstimate() {return grtt_advertise;}
        void SetGrttEstimate(double value)
        {
            value = MAX(value, MDP_GRTT_MIN);
            value = MIN(value, grtt_max);
            grtt_measured = value;
            double pktInterval = (double)segment_size/ (double)tx_rate;
            value = MAX(pktInterval, value);
            grtt_quantized = QuantizeRtt(value);
            grtt_advertise = UnquantizeRtt(grtt_quantized);
        }
        double GrttMax() {return grtt_max;}
        void SetGrttMax(double value)
        {
            value = MIN(value, 120.0);
            grtt_max = MAX(1.0, value);
            grtt_measured = MIN(grtt_max, grtt_measured);
            double pktInterval = (double)segment_size/ (double)tx_rate;
            grtt_advertise = MAX(pktInterval, grtt_measured);
            grtt_quantized = QuantizeRtt(grtt_advertise);
            grtt_advertise = UnquantizeRtt(grtt_quantized);
        }
        
        void SetCongestionControl(bool state)
            {congestion_control = state;}
        void SetFastStart(bool state)
            {fast_start = state;}
        void SetTxCacheDepth(unsigned long minCount, unsigned long maxCount, unsigned long maxSize)
        {
            // Trust no one
            pend_count_min = hold_count_min = MIN(minCount, maxCount);
            pend_count_max = hold_count_max = MAX(minCount, maxCount);
            hold_size_max = maxSize;   
        }
        MdpError SetArchiveDirectory(const char *thePath);
        const char *ArchivePath() {return archive_path;}
        void SetArchiveMode(bool value) {archive_mode = value;}
        bool ArchiveMode() {return archive_mode;}
        void SetAutoParity(unsigned char value) {auto_parity = value;}
        unsigned char AutoParity() {return auto_parity;}
        void SetUnicastNacks(bool state)
        {
            state ? SetClientFlag(MDP_CLIENT_UNICAST_NACKS) :
                    UnsetClientFlag(MDP_CLIENT_UNICAST_NACKS);
        }
        bool UnicastNacks() {return (client_status & MDP_CLIENT_UNICAST_NACKS);}
        void SetMulticastAcks(bool state)
        {
            state ? SetClientFlag(MDP_CLIENT_MULTICAST_ACKS) :
                    UnsetClientFlag(MDP_CLIENT_MULTICAST_ACKS);
        }
        bool MulticastAcks() 
            {return (0 != (client_status & MDP_CLIENT_MULTICAST_ACKS));}      
        void SetEmconClient(bool state)
        {
            // (TBD) Update existing recv obj nacking_mode on EMCON change
            if (state)
            {
                SetClientFlag(MDP_CLIENT_EMCON);
                status_reporting = false;
            }
            else
            {
                UnsetClientFlag(MDP_CLIENT_EMCON);
            }
        }
        unsigned int RobustFactor() {return robust_factor;}
        void SetRobustFactor(unsigned int theValue)
            {robust_factor = theValue;}
        bool StreamIntegrity() {return stream_integrity;}
        void SetStreamIntegrity(bool state)
            {stream_integrity = state;}
        MdpNackingMode DefaultNackingMode() {return default_nacking_mode;}
        void SetDefaultNackingMode(MdpNackingMode theMode)
            {default_nacking_mode = theMode;}
        bool EmconClient() 
            {return (0 != (client_status & MDP_CLIENT_EMCON));}
        void SetEmconServer(bool state)
        {
            state ? SetServerFlag(MDP_SERVER_EMCON) :
                    UnsetServerFlag(MDP_SERVER_EMCON);
        }
        bool EmconServer() {return (0 != (server_status & MDP_SERVER_EMCON));}
        bool ServerIsClosing()
        {
            return (0 != (server_status & MDP_SERVER_CLOSING));    
        }        
        
        void SetStatusReporting(bool state);
        void SetClientAcking(bool state)
        {
            state ? SetStatusFlag(MDP_ACKING) :
                    UnsetStatusFlag(MDP_ACKING);
        }
        bool ClientAcking() {return (0 != (status & MDP_ACKING));}
        
        MdpError AddAckingHost(const char* hostAddress);
        MdpError RemoveAckingHost(const char* hostAddress);
        
        void Notify(MdpNotifyCode   notifyCode, 
                    MdpNode*        theNode,
                    MdpObject*      theObject,
                    MdpError        errorCode)
        {
            notify_pending++;
            mgr->NotifyApplication(notifyCode, this, theNode, theObject, errorCode); 
            notify_pending--;  
        }
        bool NotifyPending() {return (0 != notify_pending);}
        void SetNotifyAbort() {notify_abort = true;}
        
        MdpNode* FindServerById(unsigned long nodeId)
        {
            return server_list.FindNodeById(nodeId);
        }      
        void DeleteRemoteServer(MdpServerNode* theServer);
        
        // Public server routines
		MdpDataObject* NewTxDataObject(const char*      infoPtr,
                                       unsigned short   infoSize, 
									   char*            dataPtr, 
									   unsigned long    dataSize,
									   MdpError*        error);
        
        MdpFileObject* NewTxFileObject(const char* path, 
                                       const char* name,
                                       MdpError* error);
        
        MdpSimObject* NewTxSimObject(unsigned short infoSize,
                                     unsigned long  dataSize,
                                     MdpError*      error);
        
        
        MdpError QueueTxObject(MdpObject* theObject);
        MdpObject* FindTxObject(MdpObject* theObject);
        MdpObject* FindTxObjectById(unsigned long theId);
        MdpError RemoveTxObject(MdpObject* theObject);
        void SetBaseObjectTransportId(unsigned long baseId)
            {next_transport_id = baseId;}
        
        // Protocol state information
        unsigned long PendingRecvObjectCount();
        unsigned long ClientBufferPeakUsage();
        unsigned long ClientBufferOverruns();
        
        // Congestion control
        double CalculateGoalRate(double l, double tRTT, double t0);
        // Some debug routines
#ifdef PROTO_DEBUG
        void SetRecvDropRate(double rate) {recv_drop_rate = rate;}
        void SetSendDropRate(double rate) {send_drop_rate = rate;}
#endif // PROTO_DEBUG
        
        void IncrementResyncs() {client_stats.resync++;}
        void IncrementFailures() {client_stats.fail++;}
    
        // Session socket recv packet handler
        bool RxSocketRecvHandler(UdpSocket* theSocket);
        bool TxSocketRecvHandler(UdpSocket* theSocket);
        bool HandleRecvPacket(char *buffer, unsigned int len, NetworkAddress *src, bool unicast);
        void SetEcnEnable(bool state) {ecn_enable = state;}
        void SetEcnStatus(bool state) {ecn_status = state;}
        
        
    private:
    // Members
        char		    name[MDP_SESSION_NAME_MAX];
	    NetworkAddress	addr;
        unsigned short  tx_port;
	    unsigned char	ttl;
        unsigned long   interface_address;
        bool            loopback;
        bool            port_reuse;
        unsigned char   tos;

	    int		        status;      // Node reporting status
        int             server_status;
        int             client_status;
            
        MdpSessionMgr    *mgr;
        UdpSocket	     rx_socket;
        UdpSocket        tx_socket;
        bool             single_socket; // if tx & rx port are the same
        bool             ecn_enable;    // ECN hack for simulations only
        bool             ecn_status;    // ECN hack for simulations only
        unsigned long    tx_rate;    // max bytes per second (UDP payload)
        unsigned long    tx_rate_min;  // lower bound on congestion control rate
        unsigned long    tx_rate_max;  // upper bound on congestion control rate
        MdpMessageQueue  tx_queue;
        ProtocolTimer    tx_timer;
        
        ProtocolTimer    report_timer;

        MdpNodeTree     server_list; // Note: Servers may also be clients
									 // but will appear only in server_list
        
        const void*     user_data;
               
        // Server-specific state
        // Blocking parameters for file transmissions
        unsigned int    ndata, nparity;
        unsigned short  segment_size;
        unsigned int    auto_parity;
        unsigned short  server_sequence;
        unsigned long   next_transport_id;
        
        bool            status_reporting;
        bool            congestion_control;  // on or off
        bool            fast_start;
        double          rate_increase_factor;
        double          rate_decrease_factor;
        MdpNodeTree     representative_list;
        unsigned int    representative_count;
        unsigned long   goal_tx_rate;
        double          fictional_rate;  // (TBD) lose this
        double          nominal_packet_size;
        MdpNodeTree     pos_ack_list;  // list of positive acking clients
        MdpMessagePool  msg_pool;
        MdpVectorPool   server_vector_pool;
        MdpBlockPool    server_block_pool;
        
        ProtocolTimer    grtt_req_timer;
        double           grtt_req_interval;
        double           grtt_req_interval_min;
        double           grtt_req_interval_max;
        bool             grtt_req_probing;
        double           grtt_wildcard_period;
        unsigned char    grtt_req_sequence;
        double           grtt_max;
        MdpNodeId        bottleneck_id;
        unsigned char    bottleneck_sequence;
        double           bottleneck_rtt;
        double           bottleneck_loss;
        unsigned char    grtt_quantized;
        double           grtt_advertise;
        double           grtt_measured;
        // State for collecting grtt estimates (TBD - make a GrttFilter class???)
        bool             grtt_response;
        double           grtt_current_peak;   
        int              grtt_decrease_delay_count; 
        
        bool            tx_pend_queue_active;
        unsigned long   current_tx_object_id;
        ProtocolTimer   tx_interval_timer;  
        unsigned int    flush_count;
        bool            pos_ack_pending;
        
        MdpObjectList   tx_pend_queue;      // Tx objects pending transmission
        unsigned long   pend_count_min;
        unsigned long   pend_count_max;
        unsigned long   pend_size_max;
        MdpObjectList   tx_repair_queue;    // Active tx objects 
        MdpObjectList   tx_hold_queue;      // Tx objects held for potential repair
        unsigned long   hold_count_min;
        unsigned long   hold_count_max;
        unsigned long   hold_size_max;
        MdpEncoder      encoder;
                
        // Client state
        // (TBD) Move file-specific stuff out of MDP protocol core
        char            archive_path[PATH_MAX];
        bool            archive_mode;
        unsigned long   client_window_size; // per server buffering (in bytes)
        bool            stream_integrity;
        MdpNackingMode  default_nacking_mode;
        
        // For client statistics reporting
        MdpClientStats  client_stats;
        struct timeval  last_report_time;
        
#ifdef PROTO_DEBUG
        double           recv_drop_rate;
        double           send_drop_rate;
#endif  // PROTO_DEBUG
        
        int             notify_pending;
        bool            notify_abort;   // true when session is deleted
                                        // during notify callback
        unsigned int    robust_factor;  // flush repeats, ack attempts, etc
        
        MdpSession      *prev, *next;            
    
    // Methods    
        // MdpSession timer handlers
        bool OnReportTimeout();
        bool OnTxTimeout();
        bool OnTxIntervalTimeout();
        bool OnGrttReqTimeout();
        
        MdpError Open();  // Generic initialization called by OpenServer() or OpenClient();        
        void SetStatusFlag(MdpStatusFlag flag) {status |= (int) flag;}
        void UnsetStatusFlag(MdpStatusFlag flag) {status &= ~((int) flag);}
        void SetServerFlag(MdpServerFlag flag) {server_status |= (int) flag;}
        void UnsetServerFlag(MdpServerFlag flag) {server_status &= ~((int) flag);}
        void SetClientFlag(MdpClientFlag flag) {client_status |= (int) flag;}
        void UnsetClientFlag(MdpClientFlag flag) {client_status &= ~((int) flag);}       
        void ActivateTimer(ProtocolTimer *theTimer) {mgr->timer_mgr.InstallTimer(theTimer);}
       
        bool AddAckingNode(unsigned long nodeId);
        void RemoveAckingNode(unsigned long nodeId);
        
        bool TxQueueIsEmpty() {return (tx_queue.IsEmpty());}
        
        bool IsActiveServing() {return (!(tx_pend_queue.IsEmpty() &&
                                          tx_repair_queue.IsEmpty()));}
        
        unsigned long LocalNodeId() {return mgr->node_id;}
        const char *LocalNodeName() {return mgr->node_name;}
        
        void PurgeClientMessages(MdpServerNode* theServer);
        void PurgeServerMessages();
        
	    bool Serve();
        void QueueMessage(MdpMessage *theMsg);
        void DeactivateTxObject(MdpObject* theObject);
        void ReactivateTxObject(MdpObject* theObject);
        void TransmitObjectInfo(MdpObject* theObject);
        void TransmitFlushCmd(unsigned long objectId);
        bool TransmitObjectAckRequest(unsigned long objectId);
        bool ServerReclaimResources(MdpObject* theObject, unsigned long blockId);
        
        MdpServerNode* NewRemoteServer(unsigned long nodeId);
        
        // MdpSession message handlers
        void HandleMdpReport(MdpMessage *theMsg, NetworkAddress *src);
        void HandleObjectMessage(MdpMessage *theMsg, NetworkAddress *src);
        void HandleMdpServerCmd(MdpMessage *theMsg, NetworkAddress *src);
        void ClientHandleMdpNack(MdpMessage *theMsg);
        void ServerHandleMdpNack(MdpMessage *theMsg, bool unicast);
        void ServerHandleMdpAck(MdpMessage *theMsg);
        void ServerProcessClientResponse(unsigned long   clientId,
                                         struct timeval* grttTimestamp,
                                         unsigned short  lossQuantized,
                                         unsigned char   clientSequence);
        void AdjustRate(bool activeResponse);
};  // end class MdpSession 

#ifdef PROTO_DEBUG
// This routine produces a timestamped trace of MDP messages
void MessageTrace(bool send, unsigned long node_id, MdpMessage *msg, int len, const NetworkAddress *src);
#endif // PROTO_DEBUG

#endif // _MDP_SESSION
