/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include "mdpSession.h"

#include <time.h>   // for gmtime()
#include <errno.h>  // for errno

#define ADJUST_RATE 1

#if defined(NS2) || defined(OPNET)
bool use_rtt_rate_bias = true;
#endif // SIMULATOR

#ifdef PROTO_DEBUG
bool mdp_trace = false;
void SetMdpMessageTrace(bool state) {mdp_trace = state;}
#endif  // PROTO_DEBUG

/**************************************************************************
 * MdpSessionMgr implementation
 */
        
MdpSessionMgr::MdpSessionMgr()
    : list_head(NULL), list_tail(NULL),
      notify_func(NULL)
{
    
}

MdpSessionMgr::~MdpSessionMgr()
{
    Close();
}

void MdpSessionMgr::Close()
{
	// Close open sessions ... 
    // Note: will delete cached archive files
    //  session->SetArchiveMode(true) to prevent cache file deletion
    MdpSession *session;
    while((session = list_head))
    {
        session->Close(true);
        list_head = list_head->next;
        delete session;
    } 
    list_tail = (MdpSession *) NULL;
}  // end MdpSessionMgr::Shutdown()

MdpSession *MdpSessionMgr::NewSession()
{
	MdpSession *theSession = new MdpSession(this);
	if (theSession) Add(theSession);
	return theSession;
}  // end MdpSessionMgr::NewSession()

void MdpSessionMgr::DeleteSession(MdpSession *theSession)
{
    if (theSession->NotifyPending())
    {
        theSession->SetNotifyAbort();
    }
    else
    {
        Remove(theSession);    
        delete theSession;
    }
}  // end MdpSessionMgr::DeleteSession()
        
        
void MdpSessionMgr::Add(MdpSession *theSession)
{
    theSession->next = (MdpSession *) NULL;
    if ((theSession->prev = list_tail))
	    list_tail->next = theSession;
    else
	    list_head = theSession;
    list_tail = theSession;  
    theSession->mgr = this;
}  // end MdpSessionMgr::AddSession()


// This routine assumes the session is in the list 
void MdpSessionMgr::Remove(MdpSession *theSession)
{
    if (theSession->prev)
	    theSession->prev->next = theSession->next;
    else
	    list_head = theSession->next;	
    if (theSession->next)
	    theSession->next->prev = theSession->prev;
    else
	    list_tail = theSession->prev;
    theSession->mgr = (MdpSessionMgr *) NULL;
}  // end MdpSessionMgr::RemoveSession()

MdpSession *MdpSessionMgr::FindSessionByAddress(const NetworkAddress *theAddr)
{
    MdpSession *nextSession = list_head;
    while(nextSession)
    {
	    if (theAddr->IsEqual(&nextSession->addr))
	        return nextSession;
	    else
	        nextSession = nextSession->next;
    }
    return (MdpSession *) NULL;
}  // end MdpSessionMgr::FindSessionByAddress()


MdpSession *MdpSessionMgr::FindSession(const MdpSession *theSession)
{
    MdpSession *nextSession = list_head;
    while(nextSession)
    {
	    if (theSession == nextSession)
	        return nextSession;
	    else
	        nextSession = nextSession->next;
    }
    return (MdpSession *) NULL;
}  // end MdpSessionMgr::FindSessionById()

void MdpSessionMgr::NotifyApplication(MdpNotifyCode  notifyCode,
                                      MdpSession*    theSession,
                                      MdpNode*       theNode,
                                      MdpObject*     theObject,
                                      MdpError       errorCode)
{
	if (notify_func)
        notify_func(notifyCode, 
                    (MdpInstanceHandle)this, 
                    (MdpSessionHandle) theSession, 
                    (MdpNodeHandle) theNode, 
                    (MdpObjectHandle) theObject, 
                    errorCode);
}  // end MdpSessionMgr::Notify()


MdpSession::MdpSession(MdpSessionMgr *theMgr)
    : tx_port(0), ttl(MDP_DEFAULT_TTL), interface_address(INADDR_ANY), 
      loopback(false), port_reuse(false), tos(0),
      status(0), server_status(0), client_status(0),
      mgr(theMgr), single_socket(false), ecn_status(false), 
      tx_rate(MDP_DEFAULT_TX_RATE/8 ), tx_rate_min(0), tx_rate_max(0),
      user_data(NULL),
      ndata(MDP_DEFAULT_BLOCK_SIZE), nparity(MDP_DEFAULT_NPARITY), 
      segment_size(MDP_DEFAULT_SEGMENT_SIZE), auto_parity(0), 
      next_transport_id(1), status_reporting(false), 
      congestion_control(false), fast_start(false),
      rate_increase_factor(1.0), rate_decrease_factor(0.75),
      representative_count(0), goal_tx_rate(MDP_DEFAULT_TX_RATE/8), nominal_packet_size(0.0),
      grtt_req_interval_min(MDP_DEFAULT_GRTT_REQ_INTERVAL_MIN),
      grtt_req_interval_max(MDP_DEFAULT_GRTT_REQ_INTERVAL_MAX),
      grtt_req_probing(true),
      grtt_req_sequence(0), grtt_max(MDP_GRTT_MAX), 
      bottleneck_id(MDP_NULL_NODE), bottleneck_sequence(128), 
      bottleneck_rtt(MDP_DEFAULT_GRTT_ESTIMATE), bottleneck_loss(0.0),
      grtt_response(false), grtt_current_peak(0), grtt_decrease_delay_count(0), 
      archive_mode(false), client_window_size(0), 
      stream_integrity(true), default_nacking_mode(MDP_NACKING_NORMAL),
      notify_pending(0), notify_abort(false), robust_factor(MDP_DEFAULT_ROBUSTNESS)
{
	name[0] = '\0';
    
    fictional_rate = (double)goal_tx_rate; 
    
    // Init session socket objects
    rx_socket.Init((UdpSocketOwner*) this, 
                   (UdpSocketRecvHandler)&MdpSession::RxSocketRecvHandler,
                   mgr->socket_installer, mgr);
    tx_socket.Init((UdpSocketOwner*) this, 
                   (UdpSocketRecvHandler)&MdpSession::TxSocketRecvHandler,
                   mgr->socket_installer, mgr);
    
    // Calculate quantized version of initial grtt
    grtt_measured = MDP_DEFAULT_GRTT_ESTIMATE;
    double pktInterval = (double)segment_size / (double)tx_rate;
    grtt_advertise = MAX(pktInterval, grtt_measured);
    grtt_quantized = QuantizeRtt(grtt_advertise);
    grtt_advertise = UnquantizeRtt(grtt_quantized);
    
    // Init some object queueing defaults
    pend_count_min = hold_count_min = MDP_DEFAULT_TX_HOLD_COUNT_MIN;
    pend_count_max = hold_count_max = MDP_DEFAULT_TX_HOLD_COUNT_MAX;
    hold_size_max = MDP_DEFAULT_TX_HOLD_SIZE_MAX;;  // will be set when opened as a server (unless first overridden)
#ifdef PROTO_DEBUG
    recv_drop_rate = send_drop_rate = (double) 0.0;
#endif // PROTO_DEBUG
    
    // Init session timers 
    report_timer.Init(MDP_REPORT_INTERVAL, -1, 
                      (ProtocolTimerOwner *)this, 
		              (ProtocolTimeoutFunc)&MdpSession::OnReportTimeout);    
    tx_timer.Init(0.0, -1, 
                  (ProtocolTimerOwner *)this, 
                  (ProtocolTimeoutFunc)&MdpSession::OnTxTimeout);
    tx_interval_timer.Init(0.0, 1, 
                           (ProtocolTimerOwner *)this, 
                           (ProtocolTimeoutFunc)&MdpSession::OnTxIntervalTimeout);    
    grtt_req_timer.Init(0.0, -1, 
                        (ProtocolTimerOwner *)this, 
                        (ProtocolTimeoutFunc)&MdpSession::OnGrttReqTimeout);

}  // end MdpSession::MdpSession()

MdpSession::~MdpSession()
{
    if (IsOpen()) Close(true);
}  // end MdpSession::~MdpSession()


MdpSimObject* MdpSession::NewTxSimObject(unsigned short   infoSize, 
										 unsigned long    dataSize,
                                         MdpError*        error)
{
	MdpSimObject* theObject 
        = new MdpSimObject(this, (MdpServerNode*)  NULL, next_transport_id);    
	if (theObject)
	{
        if (infoSize)
        {
            // (TBD) get rid of need for info memory alloc
            char* simInfo = new char[infoSize];
            if (!simInfo || !theObject->SetInfo(simInfo, infoSize))
            {
                delete simInfo;
                delete theObject;
                *error = MDP_ERROR_MEMORY;
                return NULL;   
            }
            delete simInfo;
        }
        theObject->SetSizeInfo(dataSize);       
		// Queue it for transmission
    	*error = QueueTxObject(theObject);
    	if (MDP_ERROR_NONE != *error)
    	{
        	delete theObject;
        	return NULL;
    	}
    	else
    	{
        	next_transport_id++;
        	return theObject;
    	}
	}
	else
	{
    	*error = MDP_ERROR_MEMORY;
    	return NULL;
	}
}  // end MdpSession::NewTxSimObject()



MdpDataObject *MdpSession::NewTxDataObject(const char*      infoPtr,
                                           unsigned short   infoSize, 
										   char*            dataPtr, 
										   unsigned long    dataSize,
										   MdpError*        error)
{
	MdpDataObject *theObject 
        = new MdpDataObject(this, (MdpServerNode*)NULL, next_transport_id);
    
	if (theObject)
	{
        if (infoPtr)
        {
            if (!theObject->SetInfo(infoPtr, infoSize))
            {
                delete theObject;
                if (error) *error = MDP_ERROR_MEMORY;
                return NULL;   
            }
        }        
        theObject->SetData(dataPtr, dataSize);
        
		// Queue it for transmission
    	MdpError err = QueueTxObject(theObject);
    	if (MDP_ERROR_NONE != err)
    	{
        	if (error) *error = err;
            delete theObject;
        	return NULL;
    	}
    	else
    	{
        	next_transport_id++;
        	return theObject;
    	}
	}
	else
	{
    	if (error) *error = MDP_ERROR_MEMORY;
    	return NULL;
	}
	
}  // end MdpSession::NewTxDataObject()

MdpFileObject *MdpSession::NewTxFileObject(const char* thePath, 
                                           const char* theName,
                                           MdpError*    error)
{
    ASSERT(thePath);
    ASSERT(theName);
    // Form file name
    char full_name[PATH_MAX];
    strncpy(full_name, thePath, PATH_MAX);
    int len = MIN(strlen(thePath), PATH_MAX);
    if ((DIR_DELIMITER != thePath[len-1]) &&
        (DIR_DELIMITER != name[0]) && (len < PATH_MAX ))
    {
        full_name[len] = DIR_DELIMITER;
        len++;
    }    
    strncat(full_name, theName, PATH_MAX - len);
    
    // Make sure it's a valid file
    if (MDP_FILE_NORMAL == MdpFileGetType(full_name))
    {
        MdpFileObject *theObject = 
            new MdpFileObject(this, (MdpServerNode*) NULL, next_transport_id);
                
        if (theObject)
        {   
            // Note: If the file name is too long (> segment_size),
            // this will fail and NULL will be returned.
            MdpError err = theObject->SetFile(thePath, theName);
            if (MDP_ERROR_NONE != err)
            {
                if (error) *error = err;
                delete theObject;
                return NULL;
            }
            
            
            // Queue it for transmission
            err = QueueTxObject(theObject);
            if (MDP_ERROR_NONE != err)
            {
                if (error) *error = err;
                delete theObject;
                return NULL;
            }
            else
            {
                next_transport_id++;
                return theObject;
            }
        }
        else
        {
            if (error) *error = MDP_ERROR_MEMORY;
            return NULL;
        }
    }
    else
    {
        if (error) *error = MDP_ERROR_FILE_OPEN;
        return NULL;
    }
}  // end MdpSession::NewTxFileObject()

MdpError MdpSession::QueueTxObject(MdpObject *theObject)
{
    ASSERT(theObject);
    // Make sure session is open as a server
    if (!IsServer()) return MDP_ERROR_SESSION_INVALID;        
    unsigned long pmin = pend_count_min;
    if (!pmin) pmin = 1;
    unsigned long pmax = pend_count_max;
    if (!pmin) pmax = 1;
    if ((tx_pend_queue.ObjectCount() >= pmax) ||
        ((tx_pend_queue.ObjectCount() >= pmin) &&
         ((tx_pend_queue.ByteCount()+theObject->Size()) > pend_size_max)))
    {
        return MDP_ERROR_TX_QUEUE_FULL;   
    }
    else
    {
        if (congestion_control && !IsActiveServing())
        {
            double interval = MIN(grtt_advertise, bottleneck_rtt);            
            // Probes should occupy no more than our transmit rate
            double nominal = ((double)segment_size)/((double)tx_rate);
            interval = MAX(interval, nominal);
            if (grtt_req_timer.TimeRemaining() > interval)
            {
                grtt_req_timer.SetInterval(interval);   
                grtt_req_timer.Reset();   
            }
        }
        tx_pend_queue.Insert(theObject);
        if (tx_interval_timer.IsActive()) tx_interval_timer.Deactivate();
		if (tx_queue.IsEmpty()) Serve();
        return MDP_ERROR_NONE;
    }
}  // end MdpSession::QueueTxObject()

MdpObject* MdpSession::FindTxObject(MdpObject *theObject)
{
    // Search transmit object queues for object
    MdpObject *obj = tx_hold_queue.Find(theObject);
    if (!obj)
        if (!(obj = tx_repair_queue.Find(theObject)))
            obj = tx_pend_queue.Find(theObject);
    return obj;
}  // end MdpSession::FindTxObject()

MdpObject* MdpSession::FindTxObjectById(unsigned long theId)
{
    // Search transmit object queues for object
    MdpObject* obj = tx_hold_queue.FindFromTail(theId);
    if (!obj)
        if (!(obj = tx_repair_queue.FindFromTail(theId)))
            obj = tx_pend_queue.FindFromTail(theId);
    return obj;
}  // end MdpSession::FindTxObjectById()

MdpError MdpSession::RemoveTxObject(MdpObject* theObject)
{   
    ASSERT(theObject);
    bool current = ((theObject->TransportId() == current_tx_object_id) && 
                   !theObject->NotifyPending());
    MdpError result = theObject->TxAbort();
    if (current)
    {
        if (tx_pend_queue.IsEmpty())
        {
            // NOTE: If the user queue's something,
            // "Serve()" is re-entrantly called
            Notify(MDP_NOTIFY_TX_QUEUE_EMPTY, NULL, NULL, MDP_ERROR_NONE);
        }
    }
    return result;
}  // end MdpSession::RemoveTxObject()

// The following routine decides what the 
// server should send next ... This is being called
// when a new server message is needed
// (If "false" is returned, session may no longer be valid!
bool MdpSession::Serve()
{
    // Find lowest ordinal TxPending object in the repair_queue
    MdpObject *theObject = tx_repair_queue.Head();    
    while (theObject && !theObject->TxPending())
    {
        ASSERT(theObject->RepairPending());
        theObject = theObject->next; 
    }      
    
    if (!theObject)  // Nothing currently pending transmission
    {             
        // First, get any positive acknowledgements for 
        // current object before sending another 
        if (pos_ack_pending)
        {
            if (tx_interval_timer.IsActive()) return true;
            if (TransmitObjectAckRequest(current_tx_object_id))
            {
                DMSG(6, "mdp: Server requesting postive acknowledgement for obj:%lu\n", 
                        current_tx_object_id);
                flush_count++;  // pack_req's count as flushes
                tx_interval_timer.SetInterval(2.0 * GrttEstimate());
                ActivateTimer(&tx_interval_timer);
                return true;   
            }
            else
            {
                // Everyone has acknowledged
                pos_ack_pending = false;
                MdpObject* currentTxObject = FindTxObjectById(current_tx_object_id);
                if (currentTxObject)
                {
                    currentTxObject->Notify(MDP_NOTIFY_TX_OBJECT_ACK_COMPLETE,
                                            MDP_ERROR_NONE);  
                    if (currentTxObject->WasAborted()) currentTxObject->TxAbort();
                }
            }
        }  // end if (pos_ack_pending)
        
        // Get next object in the tx_pend_queue
        if((theObject = tx_pend_queue.RemoveHead()))
        {
            tx_pend_queue_active = true;
            MdpError err = theObject->Open();
            if (err != MDP_ERROR_NONE)
            {
                theObject->SetError(err);
                delete theObject;
                return Serve();  // try again (TBD) have user restart server ???
            }
            // theObject's size was filled on theObject->Open()
            if (!theObject->TxInit(ndata, nparity, segment_size))
            {
                theObject->Close();
                theObject->SetError(MDP_ERROR_MEMORY);
                delete theObject;
                return Serve();  // try again (TBD) have user restart server ???
            }
            tx_repair_queue.Insert(theObject);
            theObject->Notify(MDP_NOTIFY_TX_OBJECT_START, MDP_ERROR_NONE);
            if (theObject->WasAborted())
            {
                theObject->TxAbort();
                theObject = NULL;
                return Serve();
            }
            current_tx_object_id = theObject->TransportId();
            theObject->SetFirstPass();
            pos_ack_pending = pos_ack_list.Head() ? true : false;
        }
        else
        {
            // Notify application of empty queue if we haven't already
            if (tx_pend_queue_active)
            {
                tx_pend_queue_active = false;
                // Serve() will be called re-entrantly if 
                // something is queued by application
                Notify(MDP_NOTIFY_TX_QUEUE_EMPTY, NULL, NULL, MDP_ERROR_NONE);
            }
            // Flush if we're done transmitting for the moment
            if (tx_queue.IsEmpty() &&  !tx_interval_timer.IsActive())
            {                
                if (flush_count < robust_factor)
                {
                    TransmitFlushCmd(current_tx_object_id);
                    flush_count++;
                    tx_interval_timer.SetInterval(2.0 * GrttEstimate());
                    ActivateTimer(&tx_interval_timer);
                }       
                else if (ServerIsClosing())
                {
                    // Complete graceful close & notify app
                    CloseServer(true); 
                    Notify(MDP_NOTIFY_SERVER_CLOSED, NULL, NULL, MDP_ERROR_NONE);
                }   
            }  
            return true;
        }  // end if/else(tx_pend_queue.RemoveHead())
    }  // end if(!theObject)
    
    ASSERT(theObject);
    
    // Deactivate the tx_interval_timer if it's active
    if (tx_interval_timer.IsActive()) tx_interval_timer.Deactivate();
        
    // Build MDP_DATA message & queue it for transmission
    MdpMessage *theMsg = 
        theObject->BuildNextServerMsg(&msg_pool, &server_vector_pool,
                                      &server_block_pool, &encoder);

    if (theMsg) 
    {
        // Fill in common message fields
        theMsg->version = MDP_PROTOCOL_VERSION;
        theMsg->sender = mgr->node_id;
        DMSG(6, "mdp: Server Serve() Queuing message ...\n");
        QueueMessage(theMsg);
                
        // (TBD) Make Pos ack process part of object state
        // Reset positive ack polling/ object flushing with new activity
        if (!theObject->TxPending())// && !theObject->RepairPending())  
        {
            // This object transmission complete for now
            if (!theObject->RepairPending())
            {
                DeactivateTxObject(theObject);
                DMSG(6, "mdp: Server sent last pending vector for obj: %.64s\n", 
                        theObject->Info());
            }
            if (theObject->FirstPass())
            {
                theObject->ClearFirstPass();
                theObject->Notify(MDP_NOTIFY_TX_OBJECT_FIRST_PASS,
                                  MDP_ERROR_NONE);
                if (theObject->WasAborted())
                {
                    RemoveTxObject(theObject);
                    theObject = NULL;   
                }
            }        
		} 
        // Reset flushing on any object transmit activity
        flush_count = 0;
    }
    else
    {
        DMSG(0, "mdp: Server failed to get a message from object: %.64s\n", 
                theObject->Info());
        // Probably out of resources because of pending tx messages
        // This will allow us to recover but we need a better approach here
        if (tx_queue.IsEmpty())
        {
            tx_interval_timer.SetInterval(GrttEstimate());
            ActivateTimer(&tx_interval_timer);
        }
    }  
    return true;
}  // end MdpSession::Serve()


bool MdpSession::OnReportTimeout()
{
    struct timeval currentTime;   
    ::GetSystemTime(&currentTime);	
	unsigned long period = 
		currentTime.tv_sec - last_report_time.tv_sec;
    if (period)
    {
        client_stats.duration += period;
        client_stats.tx_rate /= period;
        client_stats.rx_rate /= period;
        client_stats.goodput /= period;
    }
    else
    {
        client_stats.tx_rate = 0;
        client_stats.rx_rate = 0;
        client_stats.goodput = 0;
    }
    // Update client buffer usage reporting
    client_stats.buf_stat.peak = ClientBufferPeakUsage();
    client_stats.buf_stat.overflow = ClientBufferOverruns(); 

    // Get latest urrent recv objects pending;
    client_stats.active = PendingRecvObjectCount();

#ifdef PROTO_DEBUG
        // Print out the report we would (or will) send
        // Only if we've heard from a server
        if (server_list.Head())
        {
            MdpClientStats *stat = &client_stats;
            DMSG(2, "*******************************************************\n");
            DMSG(2, "MDP_REPORT for node:%lu \n", LocalNodeId());
            struct tm *rx_time = gmtime((time_t *)&currentTime.tv_sec);
            DMSG(2, "   Date & time (GMT): %02d/%02d %02d:%02d:%02d.%06lu\n",
                            rx_time->tm_mon,
                            rx_time->tm_mday, 
                            rx_time->tm_hour, 
                            rx_time->tm_min,
                            rx_time->tm_sec,
                            currentTime.tv_usec);
            DMSG(2, "   Session duration : %lu sec.\n", stat->duration);
            DMSG(2, "   Objects completed: %lu\n", stat->success);
            DMSG(2, "   Objects pending  : %lu\n", stat->active);
            DMSG(2, "   Objects failed   : %lu\n", stat->fail);
            DMSG(2, "   Server resyncs   : %lu\n", stat->resync);
		    DMSG(2, "   Current goodput  : %.3f kbps\n", 
                            ((double)stat->goodput * 8.0)/1000.0);
            DMSG(2, "   Current rx rate  : %.3f kbps\n", 
                            ((double)stat->rx_rate * 8.0)/1000.0);
            DMSG(2, "   Current tx rate  : %.3f kbps\n", 
                            ((double)stat->tx_rate * 8.0)/1000.0);
            DMSG(2, "   NACKs transmitted: %lu\n", stat->nack_cnt);
            DMSG(2, "   NACKs suppressed : %lu\n", stat->supp_cnt);
            DMSG(2, "   Block loss stats : %lu, %lu, %lu, %lu, %lu, %lu\n",
                            stat->blk_stat.lost_00, stat->blk_stat.lost_05,
                            stat->blk_stat.lost_10, stat->blk_stat.lost_20,
                            stat->blk_stat.lost_40, stat->blk_stat.lost_50);
            DMSG(2, "   Buffer usage     : peak>%luK overruns>%lu\n",
                            (stat->buf_stat.peak+500)/1000, 
                            stat->buf_stat.overflow);
            DMSG(2, "*******************************************************\n"); 
        }
#endif // PROTO_DEBUG      
    
    // If we're not supposed to send reports, bail
    if (status_reporting)
    {    
        MdpServerNode* nextServer;   
        if (UnicastNacks() && !IsServer())
            nextServer = (MdpServerNode*) server_list.Head();
        else
            nextServer = (MdpServerNode*) true;  // dummy server for broadcast
        while (nextServer)
        {
            MdpMessage *theMsg = msg_pool.Get();
            if (theMsg)
            {
                theMsg->type = MDP_REPORT;
                theMsg->version = MDP_PROTOCOL_VERSION;
                theMsg->sender = LocalNodeId();
                theMsg->report.status = (status & 0x00ff);
                theMsg->report.flavor = MDP_REPORT_HELLO;
                strncpy(theMsg->report.node_name, LocalNodeName(), MDP_NODE_NAME_MAX);

                // if MDP_CLIENT, update client_stats into MDP_REPORT
                if (IsClient())
            	    memcpy(&theMsg->report.client_stats, &client_stats,
                           sizeof(MdpClientStats));

                if (UnicastNacks() && !IsServer())
                {
                    theMsg->report.dest = nextServer->Address();
                    QueueMessage(theMsg);
                    nextServer = (MdpServerNode*) nextServer->NextInTree();
                }
                else
                {
                    theMsg->report.dest = &addr;  // broadcast to session addr
                    QueueMessage(theMsg);
                    break;  // send only one report for broadcast
                }
            }
            else
            {
                DMSG(0, "Session msg_pool is empty, can't transmit MDP_REPORT\n");
                break;
            }
        }  // end while(nextServer);  
    }  // end if(status_reporting)
    // Reset counters
	client_stats.tx_rate = 0;
    client_stats.rx_rate = 0;
    client_stats.goodput = 0;
    last_report_time = currentTime;
    // Application is never notified on this timeout
    return true;
}  // end MdpSession::OnReportTimeout()

void MdpSession::PurgeClientMessages(MdpServerNode *theServer)
{
    // Remove client messages destined to specific remote server from tx_queue
    MdpMessage *next = tx_queue.Head();
    while (next)
    {
        switch (next->type)
        {
            case MDP_NACK:
                if (theServer == next->nack.server)
                {
                    theServer->VectorReturnToPool(next->nack.data);
                }
                else
                {
                    next = next->Next();
                    continue;
                }    
                break;
                
            case MDP_ACK:
                if (theServer != next->ack.server) 
                {
                    next = next->Next();
                    continue;
                }
                break;
                
            default:
                // Ignore other message types
                next = next->Next();
                continue;
        }
        MdpMessage *msg = next;
        next = next->Next();
        tx_queue.Remove(msg);
        theServer->MessageReturnToPool(msg);
    }        
}  // end MdpSession::PurgeClientMessages()

void MdpSession::PurgeServerMessages()
{
    // Remove messages specific to local server from tx_queue
    MdpMessage *next = tx_queue.Head();
    while (next)
    {
        switch (next->type)
        {
            case MDP_INFO:
                server_vector_pool.Put(next->object.info.data);
                break;
                
            case MDP_DATA:
                server_vector_pool.Put(next->object.data.data);
                break;
                
            case MDP_PARITY:
                server_vector_pool.Put(next->object.parity.data);
                break;
                
            case MDP_CMD:
                if (MDP_CMD_SQUELCH == next->cmd.flavor)
                {
                    server_vector_pool.Put(next->cmd.squelch.data);      
                }
                else if (MDP_CMD_ACK_REQ == next->cmd.flavor)
                {
                    server_vector_pool.Put(next->cmd.ack_req.data);
                }
                else if (MDP_CMD_GRTT_REQ == next->cmd.flavor)
                {
                    if (next->cmd.grtt_req.data)
                        server_vector_pool.Put(next->cmd.grtt_req.data);
                }
                else if (MDP_CMD_NACK_ADV == next->cmd.flavor)
                {
                    server_vector_pool.Put(next->cmd.nack_adv.data);
                }
				break;
                
            default:
                // Ignore other message types
                next = next->Next();
                continue;
        }
        MdpMessage *msg = next;
        next = next->Next();
        tx_queue.Remove(msg);
        msg_pool.Put(msg);
    }        
}  // end MdpSession::PurgeServerMessages()

bool MdpSession::OnTxTimeout()
{
    bool result = true;
    // Get next server message
    MdpMessage *theMsg = tx_queue.GetNextMessage();   
    if (theMsg)
    {
        // "Pre-Pack()" message specific processing
        // (TBD) Fill common fields here ???
        switch(theMsg->type)
        {
            case MDP_INFO:
            case MDP_DATA:
            case MDP_PARITY:
                theMsg->object.sequence = server_sequence++;
                break;
                
            case MDP_CMD:
                theMsg->cmd.sequence = server_sequence++;
                theMsg->cmd.grtt = grtt_quantized;
                if (MDP_CMD_GRTT_REQ == theMsg->cmd.flavor)
                    ::GetSystemTime(&theMsg->cmd.grtt_req.send_time);
                break;
                
            case MDP_NACK:
                theMsg->nack.server->CalculateGrttResponse(&theMsg->nack.grtt_response);
                theMsg->nack.loss_estimate = QuantizeLossFraction(theMsg->nack.server->LossEstimate());
                theMsg->nack.grtt_req_sequence = theMsg->nack.server->GrttReqSequence();
                break;
                
            case MDP_ACK:
                theMsg->ack.server->CalculateGrttResponse(&theMsg->ack.grtt_response);
                theMsg->ack.loss_estimate = QuantizeLossFraction(theMsg->ack.server->LossEstimate());
                theMsg->ack.grtt_req_sequence = theMsg->ack.server->GrttReqSequence();
                break;
        }        
        // Pack the message into a buffer and send
        char buffer[MSG_SIZE_MAX];
        unsigned int len = theMsg->Pack(buffer);
        ASSERT((len > 0) && (len <= MSG_SIZE_MAX));
        // MDP_REPORTS are not currently counted in the data we collect
        if (IsClient() && (theMsg->type != MDP_REPORT)) 
            client_stats.tx_rate += len;        
        // Determine destination address and return _some_ vectors to vector pool
        const NetworkAddress *destAddr = NULL;
        bool clientMsg = false;
        switch (theMsg->type)
        {   
            case MDP_REPORT:
                destAddr = theMsg->report.dest;
                msg_pool.Put(theMsg);
                clientMsg = true;
                break;
                
            case MDP_CMD:
                destAddr = &addr;
                if (MDP_CMD_GRTT_REQ == theMsg->cmd.flavor)
                {
                    if (theMsg->cmd.grtt_req.data)
                    {
                        server_vector_pool.Put(theMsg->cmd.grtt_req.data);
                        theMsg->cmd.grtt_req.data = NULL; // (TBD) remove after debug
                    }
                }
                else if (MDP_CMD_ACK_REQ == theMsg->cmd.flavor)
                {
                    server_vector_pool.Put(theMsg->cmd.ack_req.data);
                    theMsg->cmd.ack_req.data = NULL; // (TBD) remove after debug
                }
                else if (MDP_CMD_SQUELCH == theMsg->cmd.flavor)
                {
                    server_vector_pool.Put(theMsg->cmd.squelch.data);
                    theMsg->cmd.squelch.data = NULL; // (TBD) remove after debug
                }
                else if (MDP_CMD_NACK_ADV == theMsg->cmd.flavor)
                {
                    server_vector_pool.Put(theMsg->cmd.nack_adv.data);
                    theMsg->cmd.nack_adv.data = NULL; // (TBD) remove after debug
                }
                msg_pool.Put(theMsg);
                break;
                   
            case MDP_INFO:
                destAddr = &addr;
                server_vector_pool.Put(theMsg->object.info.data);
                theMsg->object.info.data = NULL;       // (TBD) remove after DEBUG
                msg_pool.Put(theMsg);
                break;
                
            case MDP_DATA:
                destAddr = &addr;
                server_vector_pool.Put(theMsg->object.data.data);
                theMsg->object.data.data = NULL;   // (TBD) remove after DEBUG
                msg_pool.Put(theMsg);
                break;
                
            case MDP_PARITY:
                destAddr = &addr;
                server_vector_pool.Put(theMsg->object.parity.data);
                theMsg->object.parity.data = NULL;   // (TBD) remove after DEBUG
                msg_pool.Put(theMsg);
                break;
                
            case MDP_NACK:
                destAddr = theMsg->nack.dest;
                theMsg->nack.server->VectorReturnToPool(theMsg->nack.data);
                theMsg->nack.data = NULL;   // (TBD) remove after DEBUG
                theMsg->nack.server->MessageReturnToPool(theMsg);
                clientMsg = true;
                break;
                
             case MDP_ACK:
                destAddr = theMsg->ack.dest;
                theMsg->ack.server->MessageReturnToPool(theMsg);
                clientMsg = true;
                break;
        }        
        ASSERT(destAddr);  
        if (!clientMsg || !EmconClient())
        {    
        // Send the message
        // (TBD) Error check the SendTo() call and "Notify" user on error
#ifdef PROTO_DEBUG
            if ((send_drop_rate != 0.0) && 
                (UniformRand(100.0) < send_drop_rate))
            {
                    if (mdp_trace) MessageTrace(true, LocalNodeId(), theMsg, len, destAddr);
            }
            else
            {
                if (single_socket)
                    rx_socket.SendTo(destAddr, buffer, len);
                else
                    tx_socket.SendTo(destAddr, buffer, len);
                if (mdp_trace) MessageTrace(true, LocalNodeId(), theMsg, len, destAddr);
            }
#else 
            if (single_socket)
                rx_socket.SendTo(destAddr, buffer, len);
            else
                tx_socket.SendTo(destAddr, buffer, len);
#endif // PROTO_DEBUG        
            // Put next server message into queue
            if (tx_queue.IsEmpty() && IsServer()) Serve();
            // Calculate next timeout for the "tx_timer"       
		    tx_timer.SetInterval((double)len / (double)tx_rate);
            
            nominal_packet_size += 0.05 * (((double)len) - nominal_packet_size);
            
            result = true;
        }
    }
    else
	{
        tx_timer.Deactivate();
		result = false;
    }
    if (notify_abort)
    {
        mgr->DeleteSession(this);
        return false;
    }
    else
    {
        return result;
    }
}  // end MdpSession::OnTxTimeout() 

bool MdpSession::OnTxIntervalTimeout()
{
  tx_interval_timer.Deactivate();
  if (tx_queue.IsEmpty() && IsServer()) Serve();
  if (notify_abort) mgr->DeleteSession(this);
  return false;   
}  // end MdpSession::OnTxIntervalTimeout()

const int MDP_GRTT_DECREASE_DELAY = 2;
bool MdpSession::OnGrttReqTimeout()
{
    MdpMessage* theMsg = msg_pool.Get();
    if (theMsg)
    {
        if (grtt_response && (grtt_wildcard_period >= grtt_req_interval))
        {
#ifdef PROTO_DEBUG
            double old_advertise = grtt_advertise;
#endif // PROTO_DEBUG
            // Introduce new peak with linear smoothing filter ...
            // Increases grtt quickly, decreases it slowly
            // only after MDP_GRTT_DECREASE_DELAY consecutive smaller 
            // peak responses ...
            if (grtt_current_peak < grtt_measured)
            {
                if (grtt_decrease_delay_count-- == 0)
                {
                    grtt_measured = 0.5 * grtt_measured + 
                                    0.5 * grtt_current_peak;
                    grtt_decrease_delay_count = MDP_GRTT_DECREASE_DELAY;
                    grtt_current_peak = 0.0;
                }
            }
            else
            {
                // Increase has already been incorporated into estimate
                grtt_current_peak = 0.0;
                grtt_decrease_delay_count = MDP_GRTT_DECREASE_DELAY;
            }
            if (grtt_measured < MDP_GRTT_MIN)
                grtt_measured = MDP_GRTT_MIN;
            else if (grtt_measured > grtt_max)
                grtt_measured = grtt_max;
            
            double pktInterval = 2.0 * (double)segment_size/(double)tx_rate;
            grtt_advertise = MAX(pktInterval, grtt_measured);
            grtt_quantized = QuantizeRtt(grtt_advertise);
            // Recalculate grtt_advertise since quantization rounds upward
            grtt_advertise = UnquantizeRtt(grtt_quantized);
            grtt_response = false;
            
#ifdef PROTO_DEBUG
            if (grtt_advertise != old_advertise)
            {
                struct timeval rxTime;
                ::GetSystemTime(&rxTime);
                struct tm *rx_time = gmtime((time_t *)&rxTime.tv_sec);
                DMSG(2, "mdp: Server new grtt: time>%02d:%02d:%02d.%06lu grtt>%f sec.\n",
                                rx_time->tm_hour, 
                                rx_time->tm_min,
                                rx_time->tm_sec,
                                rxTime.tv_usec,
                                grtt_advertise);
            }
#endif // PROTO_DEBUG
        }  // end if (grtt_response ...)
    
        // Queue next MDP_GRTT_REQ
        theMsg->type = MDP_CMD;
        theMsg->version = MDP_PROTOCOL_VERSION;
        theMsg->sender = mgr->node_id;
        theMsg->cmd.grtt = grtt_quantized;
        theMsg->cmd.flavor = MDP_CMD_GRTT_REQ;
        // Message transmission routine will fill in grtt.send_time
        
        double interval, holdTime;
        if (congestion_control && IsActiveServing())
        {   
            AdjustRate(false);
                                  
            // Prepare next request for congestion control representatives
            // Get a vector
            char* vector = server_vector_pool.Get();
            if (vector)
            {
                theMsg->cmd.grtt_req.data = vector;
                theMsg->cmd.grtt_req.len = 0;
                // Pack vector with reps and their rtt's (quantized)
                MdpRepresentativeNode* nextRep = (MdpRepresentativeNode*) representative_list.Head();   
                while (nextRep)
                {
                    if (nextRep->DecrementTimeToLive())
                    {
                        if (!theMsg->cmd.grtt_req.AppendRepresentative(nextRep->Id(),
                                                                       nextRep->RttEstimate(),
                                                                       segment_size))
                        {
                            DMSG(0, "MdpSession::OnGrttReqTimeout() Probe rep list overflow!\n");
                        }
                        nextRep = (MdpRepresentativeNode*) nextRep->NextInTree();
                    }
                    else
                    {
                        //fprintf(stderr, "IMPEACHING rep: %u\n", nextRep->Id());
                        MdpRepresentativeNode* theRep = nextRep;
                        nextRep = (MdpRepresentativeNode*) nextRep->NextInTree();
                        representative_list.DetachNode(theRep);
                        delete theRep;
                        // If all representatives go away, reset probing
                        // (TBD) go back to "slow/no start"
                        if (!(--representative_count)) 
                        {
                            grtt_req_interval = grtt_req_interval_min;
                            grtt_wildcard_period = grtt_req_interval + 1.0;   
                        }
                    }
                    
                }  // end while(nextRep)
            }
            else
            {
                DMSG(0, "Session server_vector_pool is empty, can't transmit MDP_GRTT_REQ\n");
                msg_pool.Put(theMsg);
                return true;
            }
                       
            interval = MIN(grtt_measured, bottleneck_rtt);            
            // Probes should occupy less than our total transmit rate
            double nominal = ((double)segment_size)/((double)tx_rate);
            interval = MAX(interval, nominal);
            //if (fast_start) fprintf(stderr, "Probe interval: %f (grtt:%f brtt:%f nom:%f)\n", 
            //                                 interval, grtt_measured, bottleneck_rtt, nominal);
            
            // Periodically do "wildcard" probe that everyone responds to
            if (grtt_wildcard_period >= grtt_req_interval)
            {
                theMsg->cmd.grtt_req.flags = MDP_CMD_GRTT_FLAG_WILDCARD;
                grtt_wildcard_period = interval;
                holdTime = grtt_req_interval;
                if (grtt_req_interval < grtt_req_interval_max)
                {
                    grtt_req_interval *= 2.0;
                    if (grtt_req_interval > grtt_req_interval_max) 
                        grtt_req_interval = grtt_req_interval_max;
                }    
                if (grtt_req_interval < grtt_req_interval_min)
                    grtt_req_interval = grtt_req_interval_min;
            }
            else
            {
                // We evaluate grtt response once per grtt_req_interval
                // "grtt_wildcard_period" keeps track of how long
                // it has been.
                grtt_wildcard_period += interval;
                theMsg->cmd.grtt_req.flags = 0;   
                holdTime = 0.0; //grtt_measured/2.0; 
            }
        }
        else  // !(congestion_control && IsActiveServing())
        {
            interval = holdTime = grtt_req_interval;
            if (congestion_control)
                theMsg->cmd.grtt_req.flags = MDP_CMD_GRTT_FLAG_WILDCARD;
            else
                 theMsg->cmd.grtt_req.flags = 0;  // Don't ACK when no congestion control
            theMsg->cmd.grtt_req.data = NULL;
            theMsg->cmd.grtt_req.len = 0;  
            if (grtt_req_interval < grtt_req_interval_max)
            {
                grtt_req_interval *= 2.0;
                if (grtt_req_interval > grtt_req_interval_max) 
                    grtt_req_interval = grtt_req_interval_max;
            } 
            if (grtt_req_interval < grtt_req_interval_min)
                grtt_req_interval = grtt_req_interval_min;      
            // We evaluate grtt response once per grtt_req_interval
            // "grtt_wildcard_period" keeps track of how long
            // it has been.
            grtt_wildcard_period = interval;
        }  // end if/else(congestion_control)
        
        
        grtt_req_sequence++;  // incr before send
        theMsg->cmd.grtt_req.sequence = grtt_req_sequence;
                  
        // "hold_time" field is used to tell clients how to set their grtt_ack back-off timers
        theMsg->cmd.grtt_req.hold_time.tv_sec = (unsigned long) holdTime;
        theMsg->cmd.grtt_req.hold_time.tv_usec = 
            (unsigned long)(((holdTime -  (double)theMsg->cmd.grtt_req.hold_time.tv_sec) * 1.0e06) + 0.5);
        theMsg->cmd.grtt_req.segment_size = segment_size;
        theMsg->cmd.grtt_req.rate = tx_rate;
        theMsg->cmd.grtt_req.rtt = QuantizeRtt(bottleneck_rtt);
        theMsg->cmd.grtt_req.loss = QuantizeLossFraction(bottleneck_loss);
           
        grtt_req_timer.SetInterval(interval); 
        
        QueueMessage(theMsg);
    }
    else
    {
        DMSG(0, "Session msg_pool is empty, can't transmit MDP_GRTT_REQ\n");
    }
    // Application not currently notified on this timeout
    return true;
}  // end MdpSession::OnGrttReqTimeout()


void MdpSession::AdjustRate(bool activeResponse)
{
#ifdef ADJUST_RATE // Temporarily turn on/off rate adjustment
    // Adjust transmit rate within bounds
    if (goal_tx_rate >= tx_rate)
    {
        unsigned int probe_advance;
        int diff = grtt_req_sequence - bottleneck_sequence;
        if (diff < 0) diff += 256;        
        if (diff < 2)
            probe_advance = 3;  // normal increase towards goal
        else if (diff < 3)
            probe_advance = 2;  // hold steady regardless of goal
        else if (diff < 4)
            probe_advance = 1;  // slow decrease regardless of goal
        else
            probe_advance = 0;  // fast decrease regardless of goal
        if (activeResponse)
            probe_advance = MAX(probe_advance, 2);  // increase on responses only
        //else if (fast_start)
        //    probe_advance = 2;
        else
            probe_advance = MIN(probe_advance, 2);  // decrease on timeout only
        
        // Only increase if recent ACK
        switch (probe_advance)
        {
            case 3:  // timely response, increase
            {
                // Exp increase
                unsigned long exp_rate = 
                    (unsigned long)(((double)tx_rate) / rate_decrease_factor);
                // Lin increase
                unsigned long lin_rate = tx_rate + 
                    (unsigned long)(rate_increase_factor * ((double)nominal_packet_size));      
                //if (fast_start)
                    tx_rate = MAX(lin_rate, exp_rate);  // fast increase
                //else
                //    tx_rate = MIN(lin_rate, exp_rate);  // slow increase
                if (tx_rate > goal_tx_rate) 
                    tx_rate = goal_tx_rate;                   
            }
            break;

            case 2:  // less than timely response, no change
                break;

            case 1:  // late response
            case 0:  // very late response
            {
                //fprintf(stderr, "Late response(%u) diff:%u\n", probe_advance, diff);
                // Lin decrease
                unsigned long lin_rate = 
                    (unsigned long)(rate_increase_factor * ((double)nominal_packet_size));
                if (tx_rate > lin_rate)
                    lin_rate = tx_rate - lin_rate;
                else
                    lin_rate = 0;
                // Exp decrease
                unsigned long exp_rate = 
                    (unsigned long) (((double)tx_rate) * rate_decrease_factor);

                if (probe_advance)
                    tx_rate = MAX(lin_rate, exp_rate);  // slow decrease
                else
                    tx_rate = MIN(lin_rate, exp_rate);  // fast decrease

                // Don't go below one packet per GRTT or one packet per second for now
                unsigned long minRate = 
                    (unsigned long) MIN((double)segment_size, ((double)segment_size)/grtt_measured);
                minRate = MAX(minRate, 1);
                if (tx_rate < minRate) tx_rate = minRate;
            }
            break;

            default:
                // should never happen
                ASSERT(0);
                break;
        }  // end switch(probe_advance)
    }
    else //if (goal_tx_rate < tx_rate)
    {
        // Decrease rate quickly towards goal
        // Lin decrease
        unsigned long lin_rate = 
            (unsigned long)(rate_increase_factor * ((double) segment_size));
        if (tx_rate > lin_rate)
            lin_rate = tx_rate - lin_rate;
        else
            lin_rate = 0;
        // Exp decrease
        unsigned long exp_rate = 
            (unsigned long) (((double)tx_rate) * rate_decrease_factor);

        tx_rate = MIN(lin_rate, exp_rate);  // fast decrease

        // Minumum rate bound MIN(1 segment per GRTT, 1 segment per second)
        unsigned long minRate = (unsigned long)(((double)segment_size)/grtt_measured);
        minRate = (unsigned long) MIN(segment_size, minRate);
        minRate = MAX(minRate, 1);
        minRate = MAX(minRate, goal_tx_rate);
        if (tx_rate < minRate) tx_rate = minRate;
    }

    // Stay within rate bounds if they have been set 
    // (ZERO min/max indicates no bound)
    if (tx_rate_max && (tx_rate > tx_rate_max)) tx_rate = tx_rate_max;
    if (tx_rate_min && (tx_rate < tx_rate_min)) tx_rate = tx_rate_min;
    
    double pktInterval = (double)segment_size / (double)tx_rate;
    grtt_advertise = MAX(pktInterval, grtt_measured);
    grtt_quantized = QuantizeRtt(grtt_advertise);
    grtt_advertise = UnquantizeRtt(grtt_quantized);
    
#ifdef PROTO_DEBUG  
    if (congestion_control)
    {          
        // For debugging congestion control, log current bottlneck info and tx rate 
        struct timeval currentTime;	
        ::GetSystemTime(&currentTime);
        double theTime = (double) currentTime.tv_sec + 
                 ((double) currentTime.tv_usec)/1.0e06;
        MdpRepresentativeNode* rep = (MdpRepresentativeNode*) representative_list.Head(); 
        if (rep) 
        {
            // Find bottleneck representative
            MdpRepresentativeNode* next_rep = rep;
            while (next_rep)
            {
                if (rep->RateEstimate() < rep->RateEstimate()) 
                    rep = next_rep;
                next_rep = (MdpRepresentativeNode*)  next_rep->NextInTree();
            } 
            DMSG(1, "RateControlStatus> time>%f loss>%f rtt>%f tzero>%f goal_rate>%f " 
                    "tx_rate>%f frate>%f grtt>%f server>%lu\n", 
                theTime, rep->LossEstimate(), rep->RttEstimate(), 
                rep->RttEstimate() + 4.0 * rep->RttVariance(),
                ((double)(goal_tx_rate*8))/1000.0, ((double)(tx_rate * 8))/1000.0, 
                ((fictional_rate*8.0)/1000.0), grtt_measured, LocalNodeId());
            
        } // end if (rep)
    }  // end if (congestion_control)
#endif // PROTO_DEBUG
#endif  // ADJUST_RATE
}  // end MdpSession::AdjustRate()

void MdpSession::ServerProcessClientResponse(unsigned long   clientId,
                                             struct timeval* grttTimestamp,
                                             unsigned short  lossQuantized,
                                             unsigned char   clientSequence)
{   
    struct timeval currentTime;
    ::GetSystemTime(&currentTime);
    double clientRtt; 
    // Only process ack if it's "valid" (non-zero grttTimestamp field)
    if (grttTimestamp->tv_sec || grttTimestamp->tv_usec)
    {
        // Calculate rtt_estimate for this client and process the response
        if (currentTime.tv_usec < grttTimestamp->tv_usec)
        {
            clientRtt = 
                (double)(currentTime.tv_sec - grttTimestamp->tv_sec - 1);
            clientRtt += 
                ((double)(1000000 - (grttTimestamp->tv_usec - currentTime.tv_usec))) / 1.0e06;
        }
        else
        {
            clientRtt = 
                (double)(currentTime.tv_sec - grttTimestamp->tv_sec);
            clientRtt += 
                ((double) (currentTime.tv_usec - grttTimestamp->tv_usec)) / 1.0e06;
        }   
        
        // Lower limit on RTT (because of coarse timer resolution on some systems,
        // this can sometimes actually end up a negative value!)
        if (clientRtt < 1.0e-06) clientRtt = 1.0e-06;
        
#ifdef PROTO_DEBUG
        struct tm *rx_time = gmtime((time_t *)&currentTime.tv_sec);
        DMSG(6, "mdp: %02d/%02d %02d:%02d:%02d.%06lu node>%lu "
                "Server received GRTT response from client>%lu "
                " (%lf sec)\n", rx_time->tm_mon, rx_time->tm_mday, 
                rx_time->tm_hour, rx_time->tm_min, rx_time->tm_sec,
                currentTime.tv_usec, LocalNodeId(), clientId, clientRtt);
#endif // PROTO_DEBUG
                        
        // Process client's grtt response
        grtt_response = true;
        if (clientRtt > grtt_current_peak)
        {
            // Immediately incorporate bigger RTT's
            grtt_current_peak = clientRtt;
            if (clientRtt > grtt_measured)
            {
                grtt_decrease_delay_count = MDP_GRTT_DECREASE_DELAY;
                grtt_measured = 0.25 * grtt_measured + 0.75 * clientRtt; 
                if (grtt_measured > grtt_max) grtt_measured = grtt_max; 
                
                double pktInterval = (double)segment_size/(double)tx_rate;
                grtt_advertise = MAX(pktInterval, grtt_measured);
                grtt_quantized = QuantizeRtt(grtt_advertise);
                // Recalculate grtt_advertise since quantization rounds upward
                grtt_advertise = UnquantizeRtt(grtt_quantized);
            }
        }  
        //if (fast_start) fprintf(stderr, "New clientRtt: %f\n", clientRtt);
    }
    else 
    {
        clientRtt = grtt_measured;
    }
   
    double clientLoss = (double) UnquantizeLossFraction(lossQuantized);   
    if (clientLoss > 0.0001) 
    {
        if (fast_start) 
        {
            //fprintf(stderr, "FastStart terminated.\n");
            fast_start = false;
        }
    }
    else
    {
        clientLoss = 0.0001;
    }
    
    MdpRepresentativeNode* theRep = 
        (MdpRepresentativeNode*) representative_list.FindNodeById(clientId);
    if (theRep)
    {
        theRep->UpdateRttEstimate(clientRtt);
        theRep->SetLossEstimate(clientLoss);
        
        double clientRate = CalculateGoalRate(clientLoss,
                                              theRep->RttEstimate(),
                                              theRep->RttVariance());
                                              //0.75 * theRep->RttEstimate());
        // Adjustment to prevent oscillation due to queue/RTT cycling
#if defined(NS2) || defined(OPNET)
        if (use_rtt_rate_bias)  
#endif  // SIMULATOR
            clientRate *= theRep->RttSqrt() / sqrt(clientRtt);
        
        theRep->SetRateEstimate(clientRate);
        theRep->SetGrttReqSequence(clientSequence);
        theRep->ResetTimeToLive();
    }
    else
    {
        // Potential new representative
        double clientRate = CalculateGoalRate(clientLoss, clientRtt, 0.75*clientRtt);
        // No scalling of the rate required
        if (representative_count < 1)
        {
            if (!(theRep = new MdpRepresentativeNode(clientId)))
            {
                DMSG(0, "mdp: node:%lu Server error allocating MdpRepresentativeNode: %s\n",
                            LocalNodeId(), strerror(errno));
                return;
            }
            representative_list.AttachNode(theRep);
            representative_count++;
        }
        else
        {
            double maxRate = 0.0;
            MdpRepresentativeNode* nextRep = (MdpRepresentativeNode*)representative_list.Head();
            while (nextRep)
            {
                if (nextRep->RateEstimate() > maxRate)
                {
                    theRep = nextRep; 
                    maxRate = nextRep->RateEstimate();  
                }
                nextRep = (MdpRepresentativeNode*)nextRep->NextInTree();
            }
            if (clientRate < theRep->RateEstimate())
            {
                // Bump old guy off the list (nodes sorted by id)
                representative_list.DetachNode(theRep);
                theRep->SetId(clientId);
                representative_list.AttachNode(theRep);
                theRep->ResetTimeToLive();
            }
            else
            {
                return; // This guy didn't make the list;
            }
        }
        ASSERT(theRep);
        theRep->SetRttEstimate(clientRtt);
        theRep->SetRttVariance(0.75*clientRtt);
        theRep->SetRttSqrt(sqrt(clientRtt));
        theRep->SetLossEstimate(clientLoss);
        theRep->SetRateEstimate(clientRate);           
        theRep->SetGrttReqSequence(clientSequence);
    }  // end if/else(theRep)
    
    // Find minimum rate among representatives 
    MdpRepresentativeNode* nextRep = 
        (MdpRepresentativeNode*)representative_list.Head();
    ASSERT(nextRep);
    MdpNodeId prevBottleneck = bottleneck_id;
    bottleneck_id = nextRep->Id();
    double minRate = nextRep->RateEstimate();
    bottleneck_sequence = nextRep->GrttReqSequence();
    
    bottleneck_rtt = nextRep->RttEstimate();
    bottleneck_loss = nextRep->LossEstimate();
    while (nextRep)
    {
        if (nextRep->RateEstimate() < minRate) 
        {
            bottleneck_id = nextRep->Id();
            minRate = nextRep->RateEstimate();
            bottleneck_sequence = nextRep->GrttReqSequence();
            bottleneck_rtt = nextRep->RttEstimate();
            bottleneck_loss = nextRep->LossEstimate();
        }
        nextRep = (MdpRepresentativeNode*)nextRep->NextInTree();
    }
    
    fictional_rate = 
        CalculateGoalRate(bottleneck_loss, bottleneck_rtt, 0.75*bottleneck_rtt);
    goal_tx_rate = (unsigned long) (minRate + 0.5);
    
    // ??? Adjust only if IsActiveServing() ???
    if (congestion_control && (bottleneck_id == clientId)) AdjustRate(true);

#ifdef PROTO_DEBUG    
    if (prevBottleneck != bottleneck_id)
    {
        struct timeval currentTime;	
        ::GetSystemTime(&currentTime);
        double theTime = (double) currentTime.tv_sec + 
                 ((double) currentTime.tv_usec)/1.0e06;
        DMSG(3, "BottleneckStatus> time>%f bottleneck>%lu\n",
                  theTime, prevBottleneck);
        DMSG(3, "BottleneckStatus> time>%f bottleneck>%lu\n",
                  theTime, bottleneck_id);
    } 
#endif // PROTO_DEBUG
      
}  // end MdpSession::ServerProcessClientResponse()

// tRTT = round trip time estimate (sec)
// l = packet loss fraction
double MdpSession::CalculateGoalRate(double l, double tRTT, double tRTTvar)
{
    if (tRTT < 1.0e-08) tRTT = 1.0e-08;
    double t0 = tRTT + (4.0 * tRTTvar);
    const double b = 1.0;
    if (l < 1.0e-08) l = 1.0e-08;
    double result =  (nominal_packet_size) /
        ( 
         (tRTT * sqrt((2.0/3.0) * b * l)) + 
         (t0 * MIN(1.0, (3.0*sqrt((3.0/8.0)*b*l))) * l * (1 + 32.0 * l * l))
        );
    
    return result;
}  // end MdpSession::CalculateGoalRate()


// Moves object from session (active) tx_repair_queue 
// to (idle) tx_hold_queue
void MdpSession::DeactivateTxObject(MdpObject* theObject)
{
    tx_repair_queue.Remove(theObject); 
    tx_hold_queue.Insert(theObject);      
    while ((tx_hold_queue.ObjectCount() > hold_count_max) ||
           ((tx_hold_queue.ObjectCount() > hold_count_min) &&
            ((tx_hold_queue.ByteCount()+theObject->Size()) > hold_size_max)))
    {
        MdpObject *obj = tx_hold_queue.RemoveHead();
        ASSERT(obj);
        if (obj->IsOpen()) obj->Close();
        obj->Notify(MDP_NOTIFY_TX_OBJECT_FINISHED, MDP_ERROR_NONE);
        delete obj;        
    } 
}  // end MdpSession::DeactivateTxObject()

// Moves object from session (inactive) tx_hold_queue  
// to (idle) tx_repair_queue
void MdpSession::ReactivateTxObject(MdpObject* theObject)
{
    if (congestion_control && !! IsActiveServing())
    {
        double interval = MIN(grtt_measured, bottleneck_rtt);            
        // Probes should occupy no more than our transmit rate
        double nominal = ((double)segment_size)/((double)tx_rate);
        interval = MAX(interval, nominal);
        
        if (grtt_req_timer.TimeRemaining() > interval)
        {
            grtt_req_timer.SetInterval(interval);   
            grtt_req_timer.Reset();   
        }
    }
    tx_hold_queue.Remove(theObject);
    tx_repair_queue.Insert(theObject);
}  // end MdpSession::ActivateTxObject()

bool MdpSession::TransmitObjectAckRequest(unsigned long object_id)
{
    MdpAckingNode* nextNode = (MdpAckingNode*) pos_ack_list.Head();
    MdpMessage* msg = NULL;
    char* vector = NULL;
    while (nextNode)
    {   
        if (((nextNode->LastAckObject() != object_id) || !nextNode->HasAcked()) 
            && (nextNode->AckReqCount() > 0))
        {
            if (!msg)
            {
                if ((msg = msg_pool.Get()))
                {
                    if(!(vector = server_vector_pool.Get()))
                    {
                        DMSG(0, "mdp: Server error getting vector for AckReqCmd!");
                        msg_pool.Put(msg);
                        msg = NULL;
                        return true;
                    }
                }
                else
                {
                    DMSG(0, "mdp: Server msg_pool empty! can't send AckReqCmd!");
                    return true;
                }
                msg->type = MDP_CMD;
                msg->version = MDP_PROTOCOL_VERSION;
                msg->sender = mgr->node_id;
                msg->cmd.grtt = grtt_quantized;
                msg->cmd.flavor = MDP_CMD_ACK_REQ;
                msg->cmd.ack_req.object_id = object_id;
                msg->cmd.ack_req.data = vector;
                msg->cmd.ack_req.len = 0;
                // (TBD) Add grtt est to ACK_REQ
            }
            if (msg->cmd.ack_req.AppendAckingNode(nextNode->Id(), segment_size))
                nextNode->DecrementAckReqCount();
            else
                break;  // message vector is full
        }
        else
        {
            // Give up on this node for this object
            // (TBD) report (notify) failure to ACK ???
            nextNode->SetLastAckObject(object_id);   
        }
        nextNode = (MdpAckingNode*) nextNode->NextInTree();
    }  // end while(nextNode)
    if (msg) 
    {
        QueueMessage(msg);
        return true;
    }    
    else
    {
        return false; // evidently ACKing is complete
    }
}  // end MdpSession::TransmitObjectAckRequest() 

void MdpSession::TransmitFlushCmd(unsigned long object_id)
{
    DMSG(6, "mdp: Server flushing (object_id = %lu) ...\n", object_id);
                    
    MdpMessage *msg = msg_pool.Get();   
    if (!msg)
    {
        DMSG(0, "mdp: Server msg_pool empty! can't send FlushCmd!");
        return;
    }
    msg->type = MDP_CMD;
    msg->version = MDP_PROTOCOL_VERSION;
    msg->sender = mgr->node_id;
    msg->cmd.grtt = grtt_quantized;
    msg->cmd.flavor = MDP_CMD_FLUSH;
    if (ServerIsClosing())
        msg->cmd.flush.flags = MDP_CMD_FLAG_EOT;
    else
        msg->cmd.flush.flags = 0;
    msg->cmd.flush.object_id = object_id;
    QueueMessage(msg);
}  // end MdpSession::TransmitFlushCmd()

 

// This method is called when the server finds
// itself "resource-constrained"
// Note that "theObject" should be in the tx_repair_queue when this is called
bool MdpSession::ServerReclaimResources(MdpObject*    theObject, 
                                        unsigned long blockId)
{
    // 1) Find oldest tx_hold_queue (inactive) object with resources
    MdpObject *obj = tx_hold_queue.Head();
    while (obj)
    {
        if (obj->HaveBuffers()) 
            break;
        else
            obj = obj->next;
    }   
    
    if (obj)
    {
        ASSERT(obj != theObject);
        // Free all of the old object's resources
        obj->FreeBuffers(); 
        DMSG(6, "mdp: Server reclaimed resources from object: \"%.64s\"\n",
                obj->Info());
        return true;
    }
    else // 2) Reverse search tx_repair_queue for (active) object with resources
    {
        obj = tx_repair_queue.Tail();
        while (obj)
        {
            if (obj->HaveBuffers() || obj == theObject)
            {
                MdpBlock* blk = obj->ServerStealBlock((obj != theObject), blockId);
                if (blk)
                {
                   blk->Empty(&server_vector_pool);
                   server_block_pool.Put(blk);
                    
                    DMSG(6, "mdp: Server reclaimed resources from object: %lu block:%lu\n",
                        obj->TransportId(), blk->Id());
                   return true; 
                }
            }
            obj = obj->prev;
        }
        return false;
    }
}  // end MdpSession::ServerReclaimResources()


unsigned long MdpSession::PendingRecvObjectCount()
{
    MdpServerNode *nextServer = (MdpServerNode*) server_list.Head();
    unsigned long total = 0;
    while (nextServer)
    {
        total += nextServer->PendingCount();
        nextServer = (MdpServerNode*) nextServer->NextInTree();
    }   
    return total;
}  // end MdpSession::PendingRecvObjectCount()


MdpServerNode* MdpSession::NewRemoteServer(unsigned long nodeId)
{
    MdpServerNode* theServer = new MdpServerNode(nodeId, this);
    if (theServer->Open())
    {
        if (theServer) server_list.AttachNode(theServer);
        return theServer;
    }
    else
    {
        delete theServer;
        return NULL;
    }
}  // end MdpSession::NewRemoteServer()

void MdpSession::DeleteRemoteServer(MdpServerNode* theServer)
{
    server_list.DetachNode(theServer);
    theServer->Delete();
}  // end // end MdpSession::NewRemoteServer()


bool MdpSession::AddAckingNode(unsigned long nodeId)
{
    if (!pos_ack_list.FindNodeById(nodeId))
    {
        MdpAckingNode* theAcker;
        if ((theAcker = new MdpAckingNode(nodeId, this)))
        {
            pos_ack_list.AttachNode(theAcker);
            return true;
        }
        else
        {
            return false;
        }
    }
    else
    {
        return true;  // already in list
    }
}  // end MdpSession::AddAckingNode()

void MdpSession::RemoveAckingNode(unsigned long nodeId)
{
    MdpAckingNode* theAcker;
    if ((theAcker = (MdpAckingNode*) pos_ack_list.FindNodeById(nodeId)))
        pos_ack_list.DeleteNode(theAcker);
}  // end MdpSession::RemoveAckingNode()

// This must be called _before_ opening a session for it to have effect
bool MdpSession::SetMulticastInterfaceAddress(const char* theAddress)
{
    NetworkAddress interfaceAddr;
    if (interfaceAddr.LookupHostAddress(theAddress))
    {
         interface_address = interfaceAddr.IPv4HostAddr();
         return true;  
    }
    else
    {
        interface_address = INADDR_NONE;
        return false;
    }
}  // end MdpSession::SetMulticastInterfaceAddress()

void MdpSession::SetMulticastLoopback(bool state)
{
    loopback = state;
    if (rx_socket.IsOpen()) rx_socket.SetLoopback(state);
}

void MdpSession::SetPortReuse(bool state)
{
    port_reuse = state;
    if (rx_socket.IsOpen()) rx_socket.SetReuse(state);
}

void MdpSession::SetTTL(unsigned char theTTL)
{
    ttl = theTTL;
    if (rx_socket.IsOpen()) rx_socket.SetTTL(theTTL);
}  // end MdpSession ::SetTTL()

bool MdpSession::SetTOS(unsigned char theTOS)
{
    UdpSocket* theSocket;
    if (single_socket)
        theSocket = &rx_socket;
    else
        theSocket = &tx_socket;
    tos = theTOS;
    if (theSocket->IsOpen()) 
    {
        return ((UDP_SOCKET_ERROR_NONE == theSocket->SetTOS(theTOS)));
    }
    else
    {
        return true;
    }
}  // end MdpSession::SetTOS()

bool MdpSession::SetAddress(const char* theAddress, unsigned short rxPort,
                            unsigned short txPort)
{
    if (IsOpen()) return false;
    bool result = addr.LookupHostAddress(theAddress);
    addr.SetPort(rxPort);
    tx_port  = txPort;
    if (rxPort == txPort) 
        single_socket = true;
    else
        single_socket = false;
    return result;
}

bool MdpSession::SetGrttProbeInterval(double intervalMin, double intervalMax)
{
    if ((intervalMax < intervalMin) || (intervalMin < 0.1) || 
        (intervalMax < 5.0))
    {
        return false;
    }
    else
    {
        grtt_req_interval_min = intervalMin;
        grtt_req_interval_max = intervalMax;
        return true;
    }
}  // end MdpSession::SetGrttProbeInterval()

void MdpSession::SetGrttProbing(bool state)
{
    if (grtt_req_timer.IsActive())
    {
        if (!state) grtt_req_timer.Deactivate();
    }
    else
    {
        if (state)
        {
            grtt_response = false;
            grtt_current_peak = 0;
            grtt_decrease_delay_count = 0;
            bottleneck_sequence = 128;
            grtt_req_sequence = 0;
            grtt_req_timer.SetInterval(0.0);
            grtt_req_interval = grtt_req_interval_min;
            grtt_wildcard_period = grtt_req_interval_min * 2.0;
            ActivateTimer(&grtt_req_timer);
        }   
    }
    grtt_req_probing = state;
}  // end MdpSession::SetGrttProbing()

MdpError MdpSession::AddAckingHost(const char* host_address)
{
    NetworkAddress theAddr;
    if (!theAddr.LookupHostAddress(host_address))
    {
        DMSG(0, "mdp: Error looking up new acking host address!\n");
        return MDP_ERROR_DNS_FAIL;
    }
    unsigned long theId = theAddr.IPv4HostAddr();
    if (AddAckingNode(theId))
        return MDP_ERROR_NONE;
    else
        return MDP_ERROR_MEMORY;
}  // end MdpSession::AddAckingHost()   

MdpError MdpSession::RemoveAckingHost(const char* host_address)
{
    NetworkAddress theAddr;
    if (!theAddr.LookupHostAddress(host_address))
    {
        DMSG(0, "mdp: Error looking up new acking host address!\n");
        return MDP_ERROR_DNS_FAIL;
    }
    unsigned long theId = theAddr.IPv4HostAddr();
    RemoveAckingNode(theId);
    return MDP_ERROR_NONE;
}  // end MdpSession::RemoveAckingHost()   

// Queue a message for transmission
void MdpSession::QueueMessage(MdpMessage *theMsg)
{
    ASSERT(theMsg);
    tx_queue.QueueMessage(theMsg);
    // Install tx_timer if it's not already active
    if(!tx_timer.IsActive())
    {
		tx_timer.SetInterval(0.0);
        ActivateTimer(&tx_timer);
    }
}  // end MdpSession::QueueMessage()


void MdpSession::SetStatusReporting(bool state)
{
    // report_timer always runs to collect statistics
    // for DEBUG level 2 printouts
#ifdef PROTO_DEBUG
    status_reporting = state;
#else   
    if (state != status_reporting)
    {
        if (state)
        {
           if ((IsServer() || IsClient()) && (report_timer.IsActive())) 
               ActivateTimer(&report_timer); 
        }
        else
        {
            if (report_timer.IsActive()) 
                report_timer.Deactivate();
        }
        status_reporting = state;
    } 
#endif // end if/else PROTO_DEBUG
}  // end MdpSession::SetStatusReporting()

MdpError MdpSession::Open()
{
    // Masure the session sockets are open
	if (!rx_socket.IsOpen())
    {
		// Use destination address type so we can determine
		// whether IPv6 or IPv4 socket for Win32

		if (UDP_SOCKET_ERROR_NONE != rx_socket.Open(addr.Port(), addr.Type())) 
            return MDP_ERROR_SOCKET_OPEN;     
		if (addr.IsMulticast())
        {
            if(UDP_SOCKET_ERROR_NONE != rx_socket.JoinGroup(&addr, ttl))
	        {
                rx_socket.Close();
	            return MDP_ERROR_SOCKET_MCAST;
	        }
        }    
        if (!single_socket)
        {
            if (UDP_SOCKET_ERROR_NONE != tx_socket.Open(tx_port, addr.Type())) 
            {
                rx_socket.Close();
                return MDP_ERROR_SOCKET_OPEN;   
            }
            if (addr.IsMulticast())
            {
                if(UDP_SOCKET_ERROR_NONE != 
                    tx_socket.SetMulticastInterface(interface_address))
	            {
                    tx_socket.Close();
                    rx_socket.Close();
	                return MDP_ERROR_SOCKET_MCAST;
	            }
                if(UDP_SOCKET_ERROR_NONE != 
                    tx_socket.SetTTL(ttl))
	            {
                    tx_socket.Close();
                    rx_socket.Close();
	                return MDP_ERROR_SOCKET_MCAST;
	            }
            }
        }  // end if(!single_socket)
                
        // if non-default IP TOS
        if (tos)
        {
            UdpSocketError error;
            if (single_socket)
                error = rx_socket.SetTOS(tos);
            else
                error = tx_socket.SetTOS(tos);
            if(UDP_SOCKET_ERROR_NONE != error)
            {
                rx_socket.Close();
                tx_socket.Close();
                return MDP_ERROR_SOCKET_TOS;   
            }  
        }    
        if (loopback) rx_socket.SetLoopback(true);
        if (port_reuse) rx_socket.SetReuse(true);
    }   
    // For now, clients & servers share a common message pool
    if (0 == msg_pool.Depth())
    {
        if(MDP_ERROR_NONE != msg_pool.Init(MDP_MSG_POOL_DEPTH))
        {
            CloseServer(true);  // quick, hard shutdown
            return MDP_ERROR_MEMORY;
        }
    }    
    return MDP_ERROR_NONE;
}  // end MdpSession::Open()

void MdpSession::Close(bool hardShutdown)
{
    if (IsServer()) 
    {
        CloseServer(hardShutdown);
        if (!IsClient()) return;
    }
    if (IsClient()) 
    {
        CloseClient(); 
        return;
    }
     
    // Purge general (MDP_REPORT) messages from tx_queue
    MdpMessage *next = tx_queue.Head();
    while (next)
    {
        MdpMessage *msg;
        switch(next->type)
        {
            case MDP_REPORT:
                msg = next;
                next = next->Next();
                tx_queue.Remove(msg);
                msg_pool.Put(msg);
                break;

            default:
                 // This should never happen
                DMSG(0, "mdp: Illegal messages remain in tx_queue?!\n"); 
                break;
        }
    }  
    // Delete any node & rx_object information
    pos_ack_list.Destroy();
    server_list.Destroy();
    
    // Shutdown any outstanding timers
    if (report_timer.IsActive()) 
		report_timer.Deactivate();
    if (tx_timer.IsActive()) 
		tx_timer.Deactivate();   
	msg_pool.Destroy();
    // Generic
    rx_socket.Close(); 
    tx_socket.Close();
}  // end MdpSession::Close();


/*******************************************************************
 * Mdp Server control routines
 */

// Server-specific initialization
MdpError MdpSession::OpenServer(unsigned short segmentSize, 
                                unsigned char  numData, 
                                unsigned char  numParity,
                                unsigned long  bufferSize)
{
    if ((segmentSize < MDP_SEGMENT_SIZE_MIN) ||
        (segmentSize > MDP_SEGMENT_SIZE_MAX))
        return MDP_ERROR_PARAMETER_INVALID;
    segment_size = segmentSize;	
    nominal_packet_size = (double)segmentSize + 24;
    int blockSize = numData + numParity;
    if (blockSize > 256) return MDP_ERROR_PARAMETER_INVALID;
    ndata = numData;
    nparity = numParity;
    
    // Check to see if we need to do a general init first
    if (!IsClient())
    {
        // Since client side hasn't already initialized
        // the session, we need to do it now
        MdpError err = Open();
        if (err != MDP_ERROR_NONE) return err;   
    }
 
    // Allocate server block buffer memory                   
    unsigned long bit_mask_len = blockSize/8;
    if (blockSize%8) bit_mask_len += 1;    
    unsigned long block_pool_depth =
        (bufferSize - (MDP_MSG_POOL_DEPTH * segmentSize)) /
        (sizeof(MdpBlock) + (blockSize *sizeof(char *)) + 
         (2 * bit_mask_len) + (numParity*segmentSize));   
          
    if (block_pool_depth == 0)
    {
        CloseServer(true);
        return MDP_ERROR_PARAMETER_INVALID;
    }    
    unsigned long vector_pool_depth = MDP_MSG_POOL_DEPTH +
            block_pool_depth * numParity;
    DMSG(2, "mdp: Allocating %lu blocks for server buffering ...\n", block_pool_depth);

    if(block_pool_depth != server_block_pool.Init(block_pool_depth, blockSize))
    {
        CloseServer(true);
        return MDP_ERROR_MEMORY;
    } 
    if (vector_pool_depth!= server_vector_pool.Init(vector_pool_depth, segmentSize))
    {
        CloseServer(true);
        return MDP_ERROR_MEMORY;
    }
    
    // Default tx_pend & rx_hold queue depths
    pend_size_max = (block_pool_depth * numData * segmentSize);
    
    if(!encoder.Init(nparity, segmentSize))
    {
        CloseServer(true);
        return MDP_ERROR_MEMORY;
    }
    
    // Server inits
    SetStatusFlag(MDP_SERVER);
    pos_ack_pending = false;
    
    // Set GRTT request timer to fire right away with minimum subsequent interval
    grtt_response = false;
    grtt_current_peak = 0;
    grtt_decrease_delay_count = 0;
    bottleneck_sequence = 128;
    grtt_req_sequence = 0;
    grtt_req_timer.SetInterval(0.0);
    grtt_req_interval = grtt_req_interval_min;
    grtt_wildcard_period = grtt_req_interval_min * 2.0;
    
    // Kick start probing on startup
    if (grtt_req_probing) 
    {
        OnGrttReqTimeout();
        ActivateTimer(&grtt_req_timer);
    }

#ifdef ADJUST_RATE    
    if (congestion_control) 
    {
        goal_tx_rate = tx_rate = segmentSize;
        if (tx_rate_min || tx_rate_max)
            goal_tx_rate = tx_rate = MAX(tx_rate_min, 1);
    }
    else
#endif // ADJUST_RATE
    {
        goal_tx_rate = tx_rate;
    }
    
    double pktInterval = (double)segment_size / (double)tx_rate;
    grtt_advertise = MAX(grtt_measured, pktInterval);    
    grtt_quantized = QuantizeRtt(grtt_advertise);
    grtt_advertise = UnquantizeRtt(grtt_quantized);
            
    if (!report_timer.IsActive()
#ifndef PROTO_DEBUG
        && status_reporting  // always start timer for DEBUG
#endif // !PROTO_DEBUG
       )
    {
        ActivateTimer(&report_timer);
        // (TBD) send initial report immediately ?!
    }
    
	flush_count = robust_factor;
    return MDP_ERROR_NONE;
}  // end MdpSession::OpenServer()


void MdpSession::CloseServer(bool hardShutdown)
{
    // Stop probing
    if (grtt_req_timer.IsActive()) grtt_req_timer.Deactivate();
    // Dequeue all pending & held objects
    tx_pend_queue.Destroy();
    tx_hold_queue.Destroy();
    tx_repair_queue.Destroy();
    
    encoder.Destroy();
    
    // Remove server-specific messages from tx_queue
    PurgeServerMessages();
    
    representative_list.Destroy();
    representative_count = 0;
    
    // Free memory allocated for server buffering
    server_block_pool.Destroy();
    server_vector_pool.Destroy(); 
    if (IsClient())
    {
        if (report_timer.IsActive()) 
            report_timer.Deactivate();
    }
    
    if (hardShutdown)
    {
        UnsetStatusFlag(MDP_SERVER);
        // If we were server-only, do a complete session close
        if (!IsClient()) Close(true);
    }
    else
    {
        // Start graceful end-of-transmission flushing
        SetServerFlag(MDP_SERVER_CLOSING); 
        Serve();
    }
}  // end MdpSession::CloseServer()


MdpError MdpSession::OpenClient(unsigned long bufferSize)
{
    // Check to see if we need to do a general init first
    if (!IsServer())
    {
        // Since client side hasn't already initialized
        // the session, we need to do it now
        MdpError err = Open();
        if (err != MDP_ERROR_NONE) return err;   
    }
    client_window_size = bufferSize;
    SetStatusFlag(MDP_CLIENT);
    // Client inits
    ::GetSystemTime(&last_report_time);
    client_stats.Init();
    if (!report_timer.IsActive()
#ifndef PROTO_DEBUG
        && status_reporting 
#endif // !PROTO_DEBUG
       )
    {
        ActivateTimer(&report_timer);
        // (TBD) send initial report immediately ?!
    }
    return MDP_ERROR_NONE;
}  // end MdpSession::OpenClient()


void MdpSession::CloseClient()
{
    UnsetStatusFlag(MDP_CLIENT);
     
    // Remove client-specific messages from tx_queue
    MdpMessage *next = tx_queue.Head();
    while (next)
    {
        MdpMessage *msg = next;
        next = next->Next();
        switch (msg->type)
        {
            case MDP_NACK:
                msg->nack.server->VectorReturnToPool(msg->nack.data);
                tx_queue.Remove(msg);
                msg->nack.server->MessageReturnToPool(msg);
                break;
                
            case MDP_ACK:
                tx_queue.Remove(msg);
                msg->ack.server->MessageReturnToPool(msg);
                break;
                
            default:
                break;
        }
    }     
    // If we were client-only, do a complete session close
    if (!IsServer()) Close(true);
}  // end MdpSession::CloseClient()


unsigned long MdpSession::ClientBufferPeakUsage()
{
    unsigned long usage = 0;
    MdpServerNode *next_server = (MdpServerNode*) server_list.Head();
    while (next_server)
    {
        usage += next_server->PeakUsage();
        next_server = (MdpServerNode*) next_server->NextInTree();
    }
    return usage;
}  // end MdpSession::ClientPeakBufferUsage()

unsigned long MdpSession::ClientBufferOverruns()
{
    unsigned long overruns = 0;
    MdpServerNode *next_server = (MdpServerNode*) server_list.Head();
    while (next_server)
    {
        overruns += next_server->Overruns();
        next_server = (MdpServerNode*) next_server->NextInTree(); 
    }     
    return overruns; 
}  // end MdpSession::ClientBufferOverruns()


MdpError MdpSession::SetArchiveDirectory(const char *path)
{
    strncpy(archive_path, path, PATH_MAX);
    int len = MIN(PATH_MAX, strlen(archive_path));
    if (archive_path[len-1] != DIR_DELIMITER)
    {
        if (len < PATH_MAX)
        {
            archive_path[len++] = DIR_DELIMITER;
            if (len < PATH_MAX) archive_path[len] = '\0';
        }   
        else
        {
            archive_path[len-1] = DIR_DELIMITER;
        }     
    }
    if (MdpFileIsWritable(archive_path))
        return MDP_ERROR_NONE;
    else
        return MDP_ERROR_FILE_OPEN;
}  // end MdpSession::SetArchive()
