/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include <math.h>

#include "mdpNode.h"
#include "mdpSession.h"
#include <time.h>  // for gmtime()

#if defined(NS2) || defined(OPNET)
bool use_asymmetric_rtt_smoothing = false;
bool use_ewma_loss_estimate = false;
#endif // SIMULATOR


static const double MAXRATE  = 25000000.0;
static const double SAMLLFLOAT  = 0.0000001;

static double p_to_b(double p, double rtt, double tzero, int psize, int bval) 
{
    if (p < 0 || rtt < 0) return MAXRATE; 
    double res=rtt*sqrt(2*bval*p/3);
    double tmp1=3*sqrt(3*bval*p/8);
    if (tmp1>1.0) tmp1=1.0;
    double tmp2=tzero*p*(1+32*p*p);
    res+=tmp1*tmp2;
    if (res < SAMLLFLOAT)
        res=MAXRATE;
    else
        res=psize/res;
    if (res > MAXRATE) 
        res = MAXRATE; 
    return res;
}  // end 

static double b_to_p(double b, double rtt, double tzero, int psize, int bval) 
{
    int ctr=0;
    double p=0.5;
    double pi=0.25;
    while(1) 
    {
        double bres = p_to_b(p,rtt,tzero,psize, bval);
        /*
         * if we're within 5% of the correct value from below, this is OK
         * for this purpose.
         */
        if ((bres > 0.95*b) && (bres < 1.05*b)) 
            return p;
        if (bres>b)
            p+=pi;
        else
            p-=pi;
        pi/=2.0;
        ctr++;
        if (ctr>30) return p;
    }
}  // end b_to_p()




LossEventEstimator::LossEventEstimator()
    : lag_mask(0xffffffff), lag_depth(0), lag_test_bit(0x01),
      event_window(0), event_index(0), 
      event_window_time(0.0), event_index_time(0.0), 
      seeking_loss_event(true),
      no_loss(true), initial_loss(0.0), loss_interval(0.0),
      current_discount(1.0)
{
    memset(history, 0, 9*sizeof(unsigned long));
    discount[0] = 1.0;
}

const double LossEventEstimator::weight[8] = 
{
    1.0, 1.0, 1.0, 1.0,
    0.8, 0.6, 0.4, 0.2  
};
        
int LossEventEstimator::SequenceDelta(unsigned short a, unsigned short b)
{
    int delta = a - b;
    if (delta < -0x8000)
        return (delta + 0x10000);
    else if (delta < 0x8000)
        return delta;
    else 
        return delta - 0x10000; 
}  // end LossEventEstimator::SequenceDelta()


bool LossEventEstimator::ProcessRecvPacket(unsigned short theSequence, 
                                           bool           ecnStatus)
{
    if (!init) 
    {
        Init(theSequence); 
        return false;
    } 
    unsigned int outageDepth = 0;
    // Process packet through lag filter and check for loss
    int delta = SequenceDelta(theSequence, lag_index);  
    if (delta > 100)        // Very new packet arrived
    {
        Sync(theSequence);  // resync
        return false;   
    }
    else if (delta > 0)     // New packet arrived
    {
        if (lag_depth)
        {
            unsigned int outage = 0;
            for (int i = 0; i < delta; i++)
            {
                if (i <= (int)lag_depth)
                {
                    outage++;
                    if (lag_mask & lag_test_bit)
                    {
                        if (outage > 1) 
                            outageDepth = MAX(outage, outageDepth);
                        outage = 0;  
                    }
                    else
                    {
                        lag_mask |= lag_test_bit;
                    }
                    lag_mask <<= 1;
                } 
                else
                {
                    outage += delta - lag_depth - 1; 
                    break;
                }
            }
            outageDepth = MAX(outage, outageDepth);
            lag_mask |= 0x01;
        }
        else
        {
            if (delta > 1) outageDepth = delta - 1;
        }
        lag_index = theSequence;
    }
    else if (delta < -100)              // Very old packet arrived
    {
         Sync(theSequence); // resync
         return false; 
    }
    else if (delta < -((int)lag_depth)) // Old packet arrived
    {
        ChangeLagDepth(-delta);
    }
    else if (delta < 0)                 // Lagging packet arrived
    {                                   // (duplicates have no effect)
        lag_mask |= (0x01 << (-delta));
        return false;
    }
    else // (delta == 0)
    {
        return false;                    // Duplicate packet arrived, ignore
    }        
    
    if (ecnStatus) outageDepth += 1;
    
    bool newLossEvent = false;
    
    if (!seeking_loss_event)
    {
        struct timeval currentTime;	 
        ::GetSystemTime(&currentTime);
        double theTime = (((double)currentTime.tv_sec) + 
                      (((double)currentTime.tv_usec)/1.0e06));
        if (theTime > event_index_time)
        //if (SequenceDelta(theSequence, event_index) > event_window) 
        {
            seeking_loss_event = true;
        }
        
        // (TBD) Should we reset our history on
        //  outages within the event_window???
    }
    
    if (seeking_loss_event)
    {
        double scale;
        if (history[0] > loss_interval)
            scale = 0.125 / (1.0 + log((double)(event_window ? event_window : 1)));
        else
            scale = 0.125;
        if (outageDepth)  // non-zero outageDepth means pkt loss(es)
        {
            if (no_loss)  // first loss
            {
                //fprintf(stderr, "First Loss: seq:%u init:%f history:%lu adjusted:",
                //                 theSequence, initial_loss, history[0]);
                if (initial_loss != 0.0)
                {
                    unsigned long initialHistory = (unsigned long) ((1.0 / initial_loss) + 0.5);
                    history[0] = MAX(initialHistory, history[0]/2);
                }
                //fprintf(stderr, "%lu\n", history[0]);
                no_loss = false;
            }
            
            // Old method
            if (loss_interval > 0.0)
                loss_interval += scale*(((double)history[0]) - loss_interval);
            else
                loss_interval = (double) history[0]; 
            
            // New method
            // New loss event, shift loss interval history & discounts
            memmove(&history[1], &history[0], 8*sizeof(unsigned long));
            history[0] = 0;
            memmove(&discount[1], &discount[0], 8*sizeof(double));
            discount[0] = 1.0;
            current_discount = 1.0;
            
            event_index = theSequence;
            //if (event_window) 
                seeking_loss_event = false;
            newLossEvent = true;
            no_loss = false;
            struct timeval currentTime;	 
            ::GetSystemTime(&currentTime);
            event_index_time = (((double)currentTime.tv_sec) + 
                                (((double)currentTime.tv_usec)/1.0e06));
            event_index_time += event_window_time;
        }
        else 
        {
            //if (no_loss) fprintf(stderr, "No loss (seq:%u) ...\n", theSequence);
            if (loss_interval > 0.0)
            {
                double diff = ((double)history[0]) - loss_interval;
                if (diff >= 1.0)
                {
                    //scale *= (diff * diff) / (loss_interval * loss_interval);
                    loss_interval += scale*log(diff);
                }
            }
        }    
    }  
    else
    {
        if (outageDepth) history[0] = 0;
    }  // end if/else (seeking_loss_event)
    
    if (history[0] < 100000) history[0]++;
    
    return newLossEvent;
}  // end LossEventEstimator::ProcessRecvPacket()

double LossEventEstimator::LossFraction()
{
#if defined(NS2) || defined(OPNET)
    if (use_ewma_loss_estimate)
        return MdpLossFraction();   // MDP EWMA approach
    else
#endif // SIMULATOR
    return (TfrcLossFraction());    // ACIRI TFRC approach
}  // end LossEventEstimator::LossFraction()



// TFRC Loss interval averaging with discounted, weighted averaging
double LossEventEstimator::TfrcLossFraction()
{
    if (!history[1]) return 0.0;   
    // Compute older weighted average s1->s8 for discount determination  
    double average = 0.0;
    double scaling = 0.0;
	unsigned int i;
    for (i = 1; i < 9; i++)
    {
        if (history[i])
        {
            average += history[i] * weight[i-1] * discount[i];
            scaling += discount[i] * weight[i-1];
        }
        else
        {
            break;
        }
    }
    double s1 = average / scaling;

    // Compute discount if applicable  
     if (history[0] > (2.0*s1))
    {
        current_discount = (2.0*s1) / (double) history[0];
        current_discount = MAX (current_discount, 0.5);
    }
    
    // Re-compute older weighted average s1->s8 with discounting
	if (current_discount < 1.0)
    { 
        average = 0.0;
        scaling = 0.0;
        for (i = 1; i < 9; i++)
        {
            if (history[i])
            {
                average += current_discount * history[i] * weight[i-1] * discount[i];
                scaling += current_discount * discount[i] * weight[i-1];
            }
            else
            {
                break;
            }
        }
        s1 = average / scaling;
    }
    
    // Compute newer weighted average s0->s7 with discounting
    average = 0.0;
    scaling = 0.0;
	for (i = 0; i < 8; i++)
    {
        if (history[i])
        {
            double d = (i > 0) ? current_discount : 1.0;
            average += d * history[i] * weight[i] * discount[i];
            scaling += d * discount[i] * weight[i];
        }
        else
        {
            break;
        }
    }
    double s0 = average / scaling;   
    // Use max of old/new averages
    return (1.0 /  MAX(s0, s1));
}  // end LossEventEstimator::LossFraction()


MdpNode::MdpNode(unsigned long theId)
    : id(theId), left(NULL), right(NULL), parent(NULL), user_data(NULL)
{
    
}

MdpNode::~MdpNode()
{
}



MdpServerNode::MdpServerNode(unsigned long nodeId, MdpSession* theSession)
    : MdpNode(nodeId), session(theSession), server_synchronized(false), 
      segment_size(1024), ndata(0), nparity(0),
      active(false), local_representative(false), 
      notify_pending(0), notify_delete(false)
{
    memset(&last_grtt_req_send_time, 0, sizeof(struct timeval));
    grtt_req_sequence = 128;
    
    strcpy(name, "Unknown");
    grtt_advertise = MDP_DEFAULT_GRTT_ESTIMATE;
    grtt_quantized = QuantizeRtt(grtt_advertise);
    
    // Init our ProtocolTimers
    unsigned int activeRepeats = session->RobustFactor();
    activeRepeats = MAX(1, activeRepeats);
    double activeTimeout = 2.0 * session->GrttEstimate() * activeRepeats;
    activeTimeout = MAX(activeTimeout, MDP_NODE_ACTIVITY_TIMEOUT_MIN);
    activity_timer.Init(activeTimeout, activeRepeats,
                        (ProtocolTimerOwner *)this, 
                        (ProtocolTimeoutFunc) &MdpServerNode::OnActivityTimeout);
    // The intervals for these timer are dynamically set 
    repair_timer.Init(0, 1, (ProtocolTimerOwner *)this, 
                      (ProtocolTimeoutFunc) &MdpServerNode::OnRepairTimeout);
    ack_timer.Init(0, 0, (ProtocolTimerOwner *)this, 
                   (ProtocolTimeoutFunc) &MdpServerNode::OnAckTimeout);

}

MdpServerNode::~MdpServerNode()
{
    Close();   
}

// Server resource allocation
bool MdpServerNode::Open()
{
    if (IsOpen()) Close();
    // (TBD) Make object stream sync window user-tunable
    if (!object_transmit_mask.Init(256))
    {
        Close();
        return false;
    }    
    if (!object_repair_mask.Init(256))
    {
        Close();
        return false;
    }    
    return true;
}  // end MdpServerNode::Open()

void MdpServerNode::Close()
{
    if (IsActive()) Deactivate();
    object_repair_mask.Destroy();
    object_transmit_mask.Destroy();
    server_synchronized = false;
}  // end MdpServerNode::Close()

bool MdpServerNode::Activate(unsigned long bufferSize)
{
    ASSERT(!IsActive() && IsOpen());
    // Init decoder
    if (!decoder.Init(nparity, segment_size))
    {
        DMSG(0, "MdpNode::Activate(): Error initing decoder!\n");
        return false;   
    }
    // Allocate server buffer memory 
    unsigned int blockSize = ndata + nparity;                  
    unsigned long bit_mask_len = blockSize/8;
    if (blockSize%8) bit_mask_len += 1;    
    unsigned long block_pool_depth =
        (bufferSize - (MDP_MSG_POOL_DEPTH * segment_size) - (nparity*segment_size)) /
        (sizeof(MdpBlock) + (blockSize *sizeof(char*)) + 
         (2 * bit_mask_len) + (ndata * segment_size));          
    if (block_pool_depth < 1) 
    {
        DMSG(0, "MdpNode::Activate(): Bad parameters! (need at least 1 block of buffering)\n");
        return false;
    }    
    unsigned long vector_pool_depth = 
        MDP_MSG_POOL_DEPTH + nparity + (block_pool_depth * ndata);
    
    DMSG(2, "MdpNode::Activate(): Allocating %lu blocks for recv buffering ...\n", 
            block_pool_depth);
    if(block_pool_depth != block_pool.Init(block_pool_depth, blockSize))
    {
        DMSG(0, "MdpNode::Activate(): Error allocating recv buffer block resources!\n");
        block_pool.Destroy();
        decoder.Destroy();
        return false;
    } 
	DMSG(2, "MdpNode::Activate(): Allocating %lu vectors(%d) for recv buffering ...\n", 
			vector_pool_depth, segment_size);    
    if (vector_pool_depth != vector_pool.Init(vector_pool_depth, segment_size))
    {
        DMSG(0, "MdpNode::Activate(): Error allocating recv buffer vector resources!\n");
        vector_pool.Destroy();
        block_pool.Destroy();
        decoder.Destroy();
        return false;
    }   
    if(MDP_ERROR_NONE != msg_pool.Init(MDP_MSG_POOL_DEPTH))
    {
        DMSG(0, "MdpNode::Activate(): Error allocating recv buffer message resources!\n");
        vector_pool.Destroy();
        block_pool.Destroy();
        decoder.Destroy();
        return false;
    }    
    active = true;
    return true;
}  // end MdpServerNode::Activate()

void MdpServerNode::Deactivate()
{
    DMSG(2, "MdpNode::Deactivate(): server node:%lu gone inactive ...\n",
            Id());
    if (activity_timer.IsActive()) activity_timer.Deactivate();
    // Drop state on any pending receive objects
    live_objects.Destroy();
    // Purge any outstanding messages from the session's tx_queue
    session->PurgeClientMessages(this);
    msg_pool.Destroy();
    block_pool.Destroy();
    vector_pool.Destroy();
    decoder.Destroy();
    active = false;
}  // end MdpServerNode::Deactivate()

void MdpServerNode::Delete()
{
    Close();
    if (NotifyPending())
        notify_delete = true; 
    else
        delete this;
}  // end MdpServerNode::Delete()





void MdpServerNode::Sync(unsigned long objectId)
{
    while (object_transmit_mask.IsSet() &&
          (MDP_RECV_OBJECT_INVALID == ObjectSequenceCheck(objectId)))
    {
        MdpObject* theObject = FindRecvObjectById(recv_pending_low);
        DeactivateRecvObject(recv_pending_low, theObject);
        // (TBD) Increment session client stats failure count?
        // (TBD) Set theObject error value?
        if (theObject) delete theObject;
    }
    if (!object_transmit_mask.IsSet() &&
        (MDP_RECV_OBJECT_INVALID == ObjectSequenceCheck(objectId)))
    {
        server_synchronized = false;
    }
}  // end MdpServerNode::Sync()

void MdpServerNode::HardSync(unsigned long objectId)
{
    // while (object_transmit_mask.IsSet() && (INVALID _or_ recv_pending_low < syncId))
    unsigned long diff = recv_pending_low - objectId;
    while (object_transmit_mask.IsSet() &&
           ((MDP_RECV_OBJECT_INVALID == ObjectSequenceCheck(objectId)) ||
            ((diff > 0x80000000) || ((diff == 0x80000000) && (recv_pending_low > objectId)))))
    
    {
        MdpObject* theObject = FindRecvObjectById(recv_pending_low);
        DeactivateRecvObject(recv_pending_low, theObject);
        // (TBD) Increment session client stats failure count?
        // (TBD) Set theObject error value?
        if (theObject) delete theObject;
        diff = recv_pending_low - objectId;
    }
    
    if (!object_transmit_mask.IsSet() &&
        (MDP_RECV_OBJECT_INVALID == ObjectSequenceCheck(objectId)))
    {
        server_synchronized = false;
    }
    
}  // end MdpServerNode::HardSync()

// Steal resources so another object/block can be buffered
// Rules for stealing depend upon "nacking mode" (reliability)
// of objects in question
bool MdpServerNode::ReclaimResources(MdpObject*     theObject,
                                     unsigned long  blockId)
{
    ASSERT(theObject);
    // First, try to get resources from any incomplete "unreliable" 
    //(non-NACKED) objects starting with oldest objects
    MdpObject *obj = live_objects.Head();
    while (obj)
    {
        if (MDP_NACKING_NONE == obj->NackingMode())
        {
            if ((obj == theObject) || obj->HaveBuffers())
                break;
        }
        obj = obj->next;
    }
                
    // If no "unreliable" objects are using resources,
    // try to free resources from newest "reliable" (NACKED)
    // objects
    if (!obj && (MDP_NACKING_NONE != theObject->NackingMode()))
    {
        obj = live_objects.Tail();
        while (obj)
        {
            if (obj->HaveBuffers() || obj == theObject)
                break;
            else
                obj = obj->prev;
        }
    }
    
    if (obj && obj->HaveBuffers())
    {        
        MdpBlock* blk = obj->ClientStealBlock((obj != theObject), blockId);
        if (blk)
        {
            DMSG(0, "mdp: Client stealing resources from object:%u block:%u\n", 
                    obj->TransportId(), blk->Id());
            blk->Empty(&vector_pool);
            block_pool.Put(blk);
            return true;
        }
    }        
    DMSG(0, "mdp: Client has no idle resources to reclaim!\n");
    return false;
}  // end MdpServerNode::ReclaimResources()


void MdpServerNode::UpdateLossEstimate(unsigned short theSequence, bool ecnStatus)
{
    if (loss_estimator.ProcessRecvPacket(theSequence, ecnStatus))
    {
        // This makes reps immediately send a congestion control
        // response when new congestion events occur
        //if (ack_timer.IsActive()) ack_timer.Deactivate();
        //if (local_representative) OnAckTimeout();
    }
    
#ifdef PROTO_DEBUG   
    struct timeval currentTime;	 
    ::GetSystemTime(&currentTime);
    double theTime = (double) currentTime.tv_sec + 
             ((double) currentTime.tv_usec)/1.0e06;
    double mdpInterval = loss_estimator.MdpLossFraction();
    if (mdpInterval > 0.0) 
        mdpInterval = 1.0/ mdpInterval;
    else 
        mdpInterval = 0.0;
    double tfrcInterval = loss_estimator.TfrcLossFraction();
    if (tfrcInterval > 0.0) 
        tfrcInterval = 1.0/ tfrcInterval;
    else 
        tfrcInterval = 0.0;
    DMSG(10, "LossIntervalTracking> time>%f current>%lu "
            "mdp>%f tfrc>%f, client>%lu\n",
            theTime, loss_estimator.CurrentLossInterval(), 
            mdpInterval, tfrcInterval, session->LocalNodeId());
#endif // PROTO_DEBUG    
    
    
}  // end MdpServerNode::UpdateLossEstimate()

// Server round trip time estimation
void MdpServerNode::UpdateGrtt(unsigned char new_grtt_quantized)
{
    if (new_grtt_quantized != grtt_quantized)
    {
        grtt_quantized = new_grtt_quantized;
        grtt_advertise = UnquantizeRtt(new_grtt_quantized);
        
#ifdef PROTO_DEBUG        
        DMSG(2, "mdp: Client recv'd new grtt_estimate from server: %f sec ",
                grtt_advertise);
        struct timeval  rxTime;
        ::GetSystemTime(&rxTime);
        struct tm *rx_time = gmtime((time_t *)&rxTime.tv_sec);
        DMSG(2, "at time (GMT): %02d/%02d %02d:%02d:%02d.%06lu\n",
                        rx_time->tm_mon,
                        rx_time->tm_mday, 
                        rx_time->tm_hour, 
                        rx_time->tm_min,
                        rx_time->tm_sec,
                        rxTime.tv_usec);
#endif // PROTO_DEBUG
    }
}  // end MdpServerNode::UpdateGrtt()

void MdpServerNode::CalculateGrttResponse(struct timeval* response)
{
    ASSERT(response);
    if (last_grtt_req_send_time.tv_sec || last_grtt_req_send_time.tv_usec)
    {
        // 1st - Get current time
        ::GetSystemTime(response);    
        // 2nd - Calculate hold_time (current_time - recv_time)
        if (response->tv_usec < last_grtt_req_recv_time.tv_usec)
        {
            response->tv_sec = response->tv_sec - last_grtt_req_recv_time.tv_sec - 1;
            response->tv_usec = 1000000 - (last_grtt_req_recv_time.tv_usec - 
                                           response->tv_usec);
        }
        else
        {
            response->tv_sec = response->tv_sec - last_grtt_req_recv_time.tv_sec;
            response->tv_usec = response->tv_usec - last_grtt_req_recv_time.tv_usec;
        }
        // 3rd - Calculate adjusted grtt_send_time (send_time + hold_time)
        response->tv_sec += last_grtt_req_send_time.tv_sec;
        response->tv_usec += last_grtt_req_send_time.tv_usec;
        if (response->tv_usec > 1000000)
        {
            response->tv_usec -= 1000000;
            response->tv_sec += 1;
        }    
    }
    else  // we haven't had a grtt_request from this server yet
    {
        response->tv_sec = 0;
        response->tv_usec = 0;
    }
}  // end MdpServerNode::CalculateGrttResponse()


void MdpServerNode::HandleGrttRequest(MdpMessage* theMsg)
{
    // Record receive time and send time of GRTT_REQUEST
    ::GetSystemTime(&last_grtt_req_recv_time);
    memcpy(&last_grtt_req_send_time, &theMsg->cmd.grtt_req.send_time, sizeof(struct timeval));        
    grtt_req_sequence = theMsg->cmd.grtt_req.sequence;  
    double rttEstimate = 0.0;
    bool rep = theMsg->cmd.grtt_req.FindRepresentative(session->LocalNodeId(),
                                                       &rttEstimate); 
    bool wildcard = (theMsg->cmd.grtt_req.flags & MDP_CMD_GRTT_FLAG_WILDCARD) ? true : false;   
    
    if (rep)
    {
        if (!local_representative)
        {
           local_representative = true; 
#if defined(NS2) || defined(OPNET) 
           session->Notify(MDP_NOTIFY_REPRESENTATIVE_ELECT, NULL, NULL, MDP_ERROR_NONE);  
#endif // (SIM)
           DMSG(2, "RepresentativeStatus> time>%lu.%06lu server>%lu node>%lu elected.\n",
                 last_grtt_req_recv_time.tv_sec, last_grtt_req_recv_time.tv_usec,
                 id, session->LocalNodeId()); 
        }
        // Local rttEstimate was provided in representative list.
    }
    else
    {
        if (local_representative)
        {
            if (!wildcard)
            {
                local_representative = false;   
#if defined(NS2) || defined(OPNET) 
                session->Notify(MDP_NOTIFY_REPRESENTATIVE_IMPEACH, NULL, NULL, MDP_ERROR_NONE);
#endif
                DMSG(2, "RepresentativeStatus> time>%lu.%06lu server>%lu node>%lu impeached.\n",
                     last_grtt_req_recv_time.tv_sec, last_grtt_req_recv_time.tv_usec,
                     id, session->LocalNodeId()); 
            }    
        }
        else
        {
            rttEstimate = UnquantizeRtt(theMsg->cmd.grtt_req.rtt);
        }
    }  // end if/else rep
    
    double pktsPerRtt = ((double)theMsg->cmd.grtt_req.rate * rttEstimate) /
                         (double)theMsg->cmd.grtt_req.segment_size;
    loss_estimator.SetEventWindow((unsigned short)(pktsPerRtt + 0.5)); 
    loss_estimator.SetEventWindowTime(rttEstimate);
    
    // Track initializer for loss estimator
    if (loss_estimator.NoLoss())
    {
        double lossInit = b_to_p(((double)theMsg->cmd.grtt_req.rate)/2.0,
                                  rttEstimate, 4.0*rttEstimate, // Use 4*Rtt for Tzero
                                  theMsg->cmd.grtt_req.segment_size, 1);
        //fprintf(stderr, "Setting initial loss: %f\n", lossInit);
        loss_estimator.SetInitialLoss(lossInit);
    }

    if (rep || wildcard)
    {
        // Explicitly respond to request  
        double holdTime = 0.0;
        if (!rep)
            holdTime = ((double) theMsg->cmd.grtt_req.hold_time.tv_sec) + 
                        ((double) theMsg->cmd.grtt_req.hold_time.tv_usec / 1.0e06);
        if (ack_timer.IsActive()) 
        {
            if (ack_timer.Interval() > holdTime)
                ack_timer.Deactivate(); 
            else
                return;
        }   
        ack_timer.SetInterval(UniformRand(holdTime));
        session->ActivateTimer(&ack_timer); 
    }
}  // end MdpServerNode::HandleGrttRequest()


MdpRecvObjectStatus MdpServerNode::ObjectSequenceCheck(unsigned long object_id)
{
    // Have we already established receive sync with this server?
    if (server_synchronized) 
    {       
        // if (object_id < recv_pending_low)
        unsigned long diff = object_id - recv_pending_low;
        if ((diff > 0x80000000) || ((diff == 0x80000000) && (object_id > recv_pending_low)))
        {
            if (object_transmit_mask.NumBits() < (recv_pending_low - object_id))
            	return MDP_RECV_OBJECT_INVALID;
			else
				return MDP_RECV_OBJECT_COMPLETE;
        }
        else 
        {
            // if recv_pending_high < object_id
            diff = recv_pending_high - object_id;
            if ((diff > 0x80000000) || ((diff == 0x80000000) && (recv_pending_high > object_id)))
            {
                diff = object_id - (recv_pending_low & (~((unsigned long)0x07)));
                if (diff < object_transmit_mask.NumBits())
                    return MDP_RECV_OBJECT_NEW;
                else // It's out of range (we're way out of sync with server)
                    return MDP_RECV_OBJECT_INVALID;
            }
            else
            {
                if (object_transmit_mask.Test(object_id))
                    return MDP_RECV_OBJECT_PENDING;
                else
                    return MDP_RECV_OBJECT_COMPLETE; 
            }  
        } 
    }
    else
    {
        // We can sync up anywhere in the object stream
        return MDP_RECV_OBJECT_NEW;
    }
}  // end MdpServerNode::ObjectSequenceCheck()


void MdpServerNode::ObjectRepairCheck(unsigned long objectId, unsigned long blockId)
{
    ASSERT(server_synchronized);
    
    if (session->EmconClient()) return;
    
    if (!repair_timer.IsActive())
    { 
        // 1) Are there any older objects pending repair?
        // if (recv_pending_low <= objectId)
        unsigned long diff = recv_pending_low - objectId;
        if ((0 == diff) || (diff > 0x80000000) || ((diff == 0x80000000) && (recv_pending_low > objectId)))
        {
            if (object_repair_mask.IsSet()) object_repair_mask.Clear();      
            unsigned long index;
            if (object_transmit_mask.FirstSet(&index))
            {
                do
                {
                    if (object_repair_mask.CanFreeSet(index))
                        object_repair_mask.Set(index);            
                } while (objectId != index++);
            }           
        }            
        else
        {
            return;
        }
        
        bool start_timer = false;      
        MdpObject *obj = live_objects.Head(); 
        while (obj)
        {
            unsigned long transport_id = obj->TransportId();
            // if transport_id < objectId
            unsigned long diff = transport_id - objectId;
            if ((diff > 0x80000000) || ((diff == 0x80000000) && (transport_id > objectId)))
            {
                start_timer |= obj->ClientFlushObject(false);
                object_repair_mask.Unset(transport_id);
                obj = obj->next;
            }
            else if (0 == diff)
            {
                start_timer |= obj->ClientRepairCheck(blockId, false);
                object_repair_mask.Unset(transport_id);
                break;
            }
            else
            {
                break;
            }
        }
        start_timer |= object_repair_mask.IsSet();   
            
        if (start_timer)
        {
            object_repair_mask.Clear(); 
            // BACKOFF related
            //double backoff = UniformRand(MDP_BACKOFF_WINDOW*grtt_advertise);
            double backoff;
            if (session->IsUnicast())
                backoff = 0.0;
            else
                backoff = erand(MDP_BACKOFF_WINDOW*grtt_advertise, 
                                   (double)MDP_GROUP_SIZE);
            repair_timer.SetInterval(backoff);			   
            session->ActivateTimer(&repair_timer);
            DMSG(6, "mdp: Client installing ObjectRepairTimer for server:%.32s (timeout = %f)...\n", 
                    name, repair_timer.Interval());
        }
        current_object_id = objectId;       
    }
    else if (repair_timer.RepeatCount())
    {
        // This set current object backwards if server has rewound
        // if (objectId < current_object_id)
        unsigned long diff = objectId - current_object_id;
        if((diff > 0x80000000) || ((diff == 0x80000000)) && (objectId > current_object_id))
        {
            current_object_id = objectId;
        }
#ifdef NEVER
        // This updates current with server forward progress
        // if (current_object_id < objectId)
        unsigned long diff = current_object_id - objectId;
        if((diff > 0x80000000) || ((diff == 0x80000000)) && (current_object_id > objectId))
        {
            MdpObject *obj = live_objects.FindFromTail(current_object_id);
            while (obj)
            {
                unsigned long transport_id = obj->TransportId();
                diff = transport_id - objectId;
                if((diff > 0x80000000) || ((diff == 0x80000000) && (obj->TransportId() > objectId)))
                {
                    obj->ClientFlushObject(true);
                    obj = obj->next;
                }
                else if (0 == diff)
                {
                    obj->ClientRepairCheck(blockId, true);
                    break;
                }
                else
                {
                    break;
                }
            }
            current_object_id = objectId;
        }
#endif // OLD_CODE
    }
}  // end MdpServerNode::ObjectRepairCheck()


bool MdpServerNode::FlushServer(unsigned long objectId)
{
    MdpRecvObjectStatus objectStatus = ObjectSequenceCheck(objectId);
    switch(objectStatus)
    {
        case MDP_RECV_OBJECT_NEW:
            ActivateRecvObject(objectId, NULL);
            break;
            
        case MDP_RECV_OBJECT_INVALID:
            // ???
            return (object_transmit_mask.IsSet());
        
        default:
            break;
    }
    ObjectRepairCheck(objectId+1, 0);
    // Indicate if anything is receive pending
    return (object_transmit_mask.IsSet());  
}  // end MdpServerNode::FlushServer()

void MdpServerNode::HandleSquelchCmd(unsigned long  syncId, 
                                     char*          squelchPtr,
                                     unsigned short squelchLen)
{
    if (server_synchronized)
    {
        // if INVALID _or_ recv_pending_low < syncId
        unsigned long diff = recv_pending_low - syncId;
        if ((MDP_RECV_OBJECT_INVALID == ObjectSequenceCheck(syncId)) ||
            ((diff > 0x80000000) || ((diff == 0x80000000) && (recv_pending_low > syncId))))
        {
            HardSync(syncId);
            session->IncrementResyncs();
        }

        char *squelchEnd = squelchPtr + squelchLen;
        while (squelchPtr < squelchEnd)
        {
            unsigned long objectId;
            memcpy(&objectId, squelchPtr, sizeof(long));
            objectId = ntohl(objectId);
            squelchPtr += sizeof(long);      
            MdpObject* theObject = NULL;
            MdpRecvObjectStatus objectStatus = ObjectSequenceCheck(objectId);
            switch (objectStatus)
            {
                case MDP_RECV_OBJECT_PENDING:
                    theObject = FindRecvObjectById(objectId);
                case MDP_RECV_OBJECT_NEW:
                    DeactivateRecvObject(objectId, theObject);
                    if (theObject) 
                    {
                        delete theObject;
                        session->IncrementFailures();
                    }
                    break;

                default:
                    break;
            }
        }
    }
}  // end MdpServerNode::HandleSquelchCmd()

// Only call this after getting a MDP_RECV_OBJECT_NEW or 
// MDP_RECV_OBJECT_PENDING from "ObjectSequenceCheck()"
MdpObject* MdpServerNode::NewRecvObject(unsigned long   objectId,
                                        MdpObjectType   theType,
                                        unsigned long   objectSize,
                                        const char*     theInfo,
                                        unsigned short  infoSize)
{
	ActivateRecvObject(objectId, NULL);
    MdpObject* theObject = NULL;
    switch(theType)
    {
        case MDP_OBJECT_DATA:
            theObject = new MdpDataObject(session, this, objectId);
            if (theObject)
            {
                if (theInfo)
                {
                    if (!theObject->SetInfo(theInfo, infoSize))
                    {
                        delete theObject;
						DeactivateRecvObject(objectId, NULL);
                        return NULL;   
                    }
                }
            }
            break;
            
        case MDP_OBJECT_FILE:
            theObject = new MdpFileObject(session, this, objectId);
            if (theObject)
            {
                if (!((MdpFileObject*)theObject)->SetPath(session->ArchivePath()))
                {
                    delete theObject;
					DeactivateRecvObject(objectId, NULL);
                    return NULL;
                }
                if (!theObject->SetInfo(theInfo, infoSize))
                {
                    delete theObject;
					DeactivateRecvObject(objectId, NULL);
                    return NULL;   
                }
            } 
            break;
            
        case MDP_OBJECT_SIM:
            theObject = new MdpSimObject(session, this, objectId);
            if (theObject)
            {
                if (theInfo)
                {
                    if (!theObject->SetInfo(theInfo, infoSize))
                    {
                        delete theObject;
						DeactivateRecvObject(objectId, NULL);
                        return NULL;   
                    }
                }
            }
            break;
            
        default:
            // (TBD) Notify unsupported object type
            break;   
    }    
    if (theObject)
    {
        theObject->SetSizeInfo(objectSize);
        theObject->Notify(MDP_NOTIFY_RX_OBJECT_START, MDP_ERROR_NONE);
        if (theObject->WasAborted())
        {
            theObject->RxAbort();
            theObject = NULL; 
        }
        else if (MDP_ERROR_NONE != theObject->Open())
        {
            DMSG(0, "mdp: node:%lu Error opening the object for receive!\n", 
                    session->LocalNodeId());
            DeactivateRecvObject(objectId, NULL);
            theObject->SetError(MDP_ERROR_FILE_OPEN);
            delete theObject;
            theObject = NULL;
        }
        else
        {
            if (theObject->RxInit(ndata, nparity, segment_size))
            {
                ActivateRecvObject(objectId, theObject);
            }
            else
            {
                DMSG(0, "mdp: node:%lu Error initing the object for receive!\n",
					    session->LocalNodeId());
                DeactivateRecvObject(theObject->TransportId(), NULL);
                theObject->SetError(MDP_ERROR_MEMORY);
                delete theObject;
                theObject = NULL;            
            }
        }    
    }
    else
    {
        DMSG(0, "mdp: node:%lu Error allocating memory for new MdpObject!\n",
				session->LocalNodeId());
        DeactivateRecvObject(objectId, NULL);   
    }
    return theObject;
}  // end MdpServerNode::NewRecvObject()

// Only call this after getting a MDP_RECV_OBJECT_NEW or 
// MDP_RECV_OBJECT_PENDING from "ObjectSequenceCheck()"
void MdpServerNode::ActivateRecvObject(unsigned long object_id, class MdpObject *theObject)
{
    if (theObject) live_objects.Insert(theObject);   
    if (server_synchronized)  
    {
        // if recv_pending_high < object_id
        unsigned long diff = recv_pending_high - object_id;
        if ((diff > 0x80000000) || ((diff == 0x80000000) && (recv_pending_high > object_id)))
        {
            unsigned long index = recv_pending_high + 1;
            while (index != object_id) 
            {
                ASSERT(object_transmit_mask.CanFreeSet(index));
                object_transmit_mask.Set(index++);  
            }
            ASSERT(object_transmit_mask.CanFreeSet(index));
            object_transmit_mask.Set(object_id);
            recv_pending_high = object_id; 
            object_transmit_mask.FirstSet(&recv_pending_low);
        }
        else
        {
            ASSERT(object_transmit_mask.Test(object_id));
        }
    }
    else
    {
        object_transmit_mask.Set(object_id);
        recv_pending_low = recv_pending_high = object_id;
        server_synchronized = true;
    }
}  // end MdpServerNode::ActivateRecvObject()

void MdpServerNode::DeactivateRecvObject(unsigned long object_id, class MdpObject *theObject)
{
    ASSERT(server_synchronized);
    if (theObject) 
    {
        live_objects.Remove(theObject);
        theObject->Close();
    }
    object_transmit_mask.Unset(object_id);
    if (object_repair_mask.Test(object_id))
        object_repair_mask.Unset(object_id);
    if (object_transmit_mask.IsSet())
        object_transmit_mask.FirstSet(&recv_pending_low);
    else
        recv_pending_low = recv_pending_high+1;
    // If server has timed out and no active recv objects,
    // free up buffering resources used by this server node
    if (!live_objects.Head() && !activity_timer.IsActive())
    {
         DMSG(6, "mdp: All pending objects deactivate _and_ server has timed out!\n");
         Deactivate();  
         Close();
         session->DeleteRemoteServer(this);
    }
}  // end MdpServerNode::DeactivateRecvObject()

void MdpServerNode::AcknowledgeRecvObject(unsigned long object_id)
{
    server_pos_ack_object_id = object_id;
    if (ack_timer.IsActive())
    {
        if (ack_timer.Interval() > grtt_advertise)
            ack_timer.Deactivate();
        else
            return;
    }
    ack_timer.SetInterval(UniformRand(grtt_advertise));
    session->ActivateTimer(&ack_timer);   
}  // end MdpServerNode::AcknowledgeRecvObject()

void MdpServerNode::ResetActivityTimer() 
{	
    double theInterval = grtt_advertise;
    unsigned int activeRepeats = session->RobustFactor();
    activeRepeats = MAX(1, activeRepeats);
    activity_timer.SetRepeats(activeRepeats);
    theInterval *= (2*activeRepeats);	
    if (theInterval < MDP_NODE_ACTIVITY_TIMEOUT_MIN)
        theInterval = MDP_NODE_ACTIVITY_TIMEOUT_MIN;       
    if (activity_timer.IsActive())
    {
        if (theInterval != activity_timer.Interval())
        {
            activity_timer.Deactivate();
			activity_timer.SetInterval(theInterval);
            session->ActivateTimer(&activity_timer);
        }
        else
        {
            activity_timer.ResetRepeats();
        }
    }
    else
    {        
		activity_timer.SetInterval(theInterval);
        session->ActivateTimer(&activity_timer); 
    }   
}  // end MdpServerNode::ResetActivityTimer()
 

bool MdpServerNode::OnAckTimeout()
{
    MdpMessage *theMsg = msg_pool.Get();
    if (theMsg)
    {
        theMsg->type = MDP_ACK;
        theMsg->version = MDP_PROTOCOL_VERSION;
        theMsg->sender = session->LocalNodeId();       
        theMsg->ack.server_id = id;
        // TxTimeout will fill in grtt_send_time field
        theMsg->ack.type = MDP_ACK_OBJECT;
        theMsg->ack.object_id = server_pos_ack_object_id;
        // For now, MDP_ACKs are always unicast to the server
        if (session->MulticastAcks())
            theMsg->ack.dest = session->Address();
        else
            theMsg->ack.dest = &addr;
        theMsg->ack.server = this;
        session->QueueMessage(theMsg);
    }
    else
    {
        DMSG(0, "Server msg_pool is empty, can't transmit MDP_ACK\n");
    }
    return true;  // one shots never need re-installed
}  // end MdpServerNode::OnAckTimeout()


bool MdpServerNode::OnRepairTimeout()
{
    if (!repair_timer.RepeatCount())
    {
        // Timer was in "repeat nack hold-off" phase
        DMSG(6, "mdp: node:%lu Client NodeRepairTimer NACK hold-off released ...\n",
                session->LocalNodeId());
        return true;
    }
    
    // Ask for repairs up through current object           
    // First grab an MdpMessage and data vector to use for potential NACK
    MdpMessage *theMsg;
    char *vector;
    if(!(theMsg = msg_pool.Get()))
    {
        DMSG(0, "mdp: Sender msg_pool is empty, can't queue NACK!\n");
        repair_timer.Deactivate();
        // Reset repair cycle for pending objects
        MdpObject *obj = live_objects.Head();
        while (obj)
        {
            obj->current_block_id = 0;
            obj = obj->next;
        }
        return false;
    }
    if(!(vector = vector_pool.Get()))
    {
        DMSG(0, "mdp: Sender vector_pool is empty, can't queue NACK!\n");
        msg_pool.Put(theMsg);
        repair_timer.Deactivate();
        // Reset repair cycle for pending objects
        MdpObject *obj = live_objects.Head();
        while (obj)
        {
            obj->current_block_id = 0;
            obj = obj->next;
        }
        return false;
    }  
    theMsg->type = MDP_NACK;
    theMsg->version = MDP_PROTOCOL_VERSION;
    theMsg->sender = session->LocalNodeId();
    theMsg->nack.server_id = id;
    // Note: MdpTransmitTimer timeout will calculate grtt_send_time
    theMsg->nack.server = this;
    theMsg->nack.data = vector;
    if (session->UnicastNacks())
        theMsg->nack.dest = &addr;
    else
        theMsg->nack.dest = &session->addr;

    unsigned int content = 0;
    
    // new object_repair_mask = object_transmit_mask - object_repair_mask 
    unsigned long index;
    unsigned long current_id = current_object_id;
    
    if (object_transmit_mask.FirstSet(&index))
    {
        // while (index <= current_object_id)
        unsigned long diff = index - current_id;
        while ((diff > 0x80000000) || (0 == diff) || ((0x80000000 == diff) && (index < current_id)))
        {
            if (object_transmit_mask.Test(index))
            {
                if (object_repair_mask.Test(index))
                {
                    object_repair_mask.Unset(index);
                }
                else
                {
                    ASSERT(object_repair_mask.CanFreeSet(index));
                    object_repair_mask.Set(index);
                }
            }
            else 
            {
                object_repair_mask.Unset(index);
            }
            diff = ++index - current_id;
        }
    }
    else
    {
        object_repair_mask.Clear();
    }
    
    // At this point the "object_repair_mask" contains repairs for objects which
    // we haven't been completely repair suppressed by someone else
    
    // Pack repair nacks for objects for which we have state
    MdpObject* obj = live_objects.Head();
    while (obj)
    {
        unsigned long transport_id = obj->TransportId();
        if (object_repair_mask.Test(transport_id))  // no-one else has asked for full repair
        {
            object_repair_mask.Unset(transport_id);
            // if (transport_id < current_object_id)
            unsigned long diff = transport_id - current_id;
            if ((diff > 0x80000000) || ((0x80000000 == diff) && (transport_id < current_id)))
            {
                MdpObjectNack onack;
                if (onack.Init(transport_id, &vector[content], 
                               segment_size - content))
                {
                    obj->BuildNack(&onack, false);
                    content += onack.Pack();
                }
                else
                {
                    obj->current_block_id = 0;
                    obj = obj->next;
                    break;
                }
            }
            else if (0 == diff)  // BuildNack for current_object_id
            {
                MdpObjectNack onack;
                if (onack.Init(transport_id, &vector[content], 
                               segment_size - content))
                {
                    obj->BuildNack(&onack, true);
                    content += onack.Pack();
                }
                else
                {
                    obj->current_block_id = 0;
                }
                obj = obj->next;
                break;
            }
            else
            {
                break;
            }  
        }
        else
        {
            obj->current_block_id = 0;
        }
        obj = obj->next;
    }  // end while(obj)
    
    while (obj)
    {
        obj->current_block_id = 0;
        obj = obj->next;
    }
    
    // If there's anything left in the "object_repair_mask"
    // While there's room, pack nacks for objects we're missing state
    // and no-one else has nack'd
    if (object_repair_mask.FirstSet(&index))
    {
        MdpNackingMode nackingMode = session->DefaultNackingMode();
        if ((MDP_NACKING_NONE != nackingMode) && session->StreamIntegrity())
        {
            MdpObjectNack onack;
            MdpRepairNack rnack;
            // default nack for INFOONLY or NORMAL
            if (MDP_NACKING_INFOONLY == nackingMode)
                rnack.type = MDP_REPAIR_INFO;
            else
                rnack.type = MDP_REPAIR_OBJECT;
            rnack.type = MDP_REPAIR_OBJECT;
            while (object_repair_mask.NextSet(&index))
            {
                // if index <= current_id
                unsigned long diff = index - current_id;
                if ((diff > 0x80000000) || (diff == 0) || ((0x80000000 == diff) && (index < current_id)))
                {
                    if (onack.Init(index, &vector[content], segment_size - content))
                    {
                        if (onack.AppendRepairNack(&rnack))
                        {
                            content += onack.Pack();
                            DMSG(6, "mdp: Client requesting OBJECT_REPAIR for object:%lu\n", 
                                    index);
                        }
                        else
                        {
                            DMSG(0, "MdpNodeRepairTimer::DoTimeout() full nack msg ...\n");
                            break;
                        }
                    }
                    else
                    {
                        DMSG(0, "MdpNodeRepairTimer::DoTimeout() full nack msg ...\n");
                        break;
                    }
                    index++;
                }
                else
                {
                    break;
                }
            }  // end while(object_repair_mask.NextSet())
        }  // end if (session->StreamIntegrity() && (MDP_NACKING_NONE != nackingMode))
    }  // end if (object_repair_mask.FirstSet(&index))
    
    // If we've accumulated any NACK content, send it; else NACK was suppressed
    if (content)
    {
        session->client_stats.nack_cnt++;
        theMsg->nack.nack_len = content;
        session->QueueMessage(theMsg);
    }
    else
    {
        DMSG(6, "mdp: Client MDP_NACK message was SUPPRESSED.\n");
        session->client_stats.supp_cnt++;
        // Return unused resources to pools
        vector_pool.Put(vector); 
        msg_pool.Put(theMsg);
    }
    // Set NACK hold-off interval (BACKOFF related)
    double holdOff;
    if (session->IsUnicast())
        holdOff = 1.5 * grtt_advertise;
    else
        holdOff = (MDP_BACKOFF_WINDOW+2.0) * grtt_advertise;
    
    DMSG(6, "mdp: node:%lu Client holding off NACKs for %f sec\n",
            session->LocalNodeId(), 
            (MDP_BACKOFF_WINDOW+2.0)*grtt_advertise);
    repair_timer.SetInterval(holdOff);
    return true;  
}  // end MdpServerNode::OnRepairTimeout()


bool MdpServerNode::OnActivityTimeout()
{  
    DMSG(6, "mdp: node:%lu ActivityTimeout for server:%lu (repeats:%d)\n",
            session->LocalNodeId(), Id(), activity_timer.RepeatCount());
    // We did a little trick here by normally just resetting
    // the repeat count so we wouldn't reinstall the timer
    // on every packet arrival ... so don't count first timeout
    int repeatCount = activity_timer.RepeatCount();
    if (repeatCount < activity_timer.Repeats())
    {
        MdpObject* latest_obj = live_objects.Tail();           
        if (repeatCount)
        {
            if (latest_obj) FlushServer(latest_obj->TransportId());
        }
        else
        {
            Notify(MDP_NOTIFY_REMOTE_SERVER_INACTIVE, NULL, MDP_ERROR_NONE);
            if (notify_delete)
            {
                Delete();
                return false;   
            }
            if (!latest_obj)
            {
                // No pending objects, so free server resources
                Deactivate();
                // (TBD) Let the application control whether we drop
                // _all_ state on the remote server, ... for now
                // we'll drop all state on inactive remote servers
                Close();
                session->DeleteRemoteServer(this);
                return false;
            }
        }
    }
    return true;
}  // end MdpServerNode::OnActivityTimeout()

void MdpServerNode::Notify(MdpNotifyCode    notifyCode, 
                           MdpObject*       theObject, 
                           MdpError         errorCode)
{
    notify_pending++;
    session->Notify(notifyCode, (MdpNode*)this, theObject, errorCode);   
    notify_pending--;
}  // end MdpServerNode::Notify()



MdpAckingNode::MdpAckingNode(unsigned long nodeId, class MdpSession* theSession)
    : MdpNode(nodeId), session(theSession), ack_status(false)
{
    ack_req_count = session->RobustFactor();
    // (TBD) Make minimum ack_req_count a parameter instead of 1?
    if (ack_req_count < 1) ack_req_count = 1;
}

void MdpAckingNode::SetLastAckObject(unsigned long objectId) 
{
    if (ack_object_id != objectId)
    {
        ack_object_id = objectId;
        ack_req_count = session->RobustFactor();        
        // (TBD) Make minimum ack_req_count a parameter instead of 1?
        if (ack_req_count < 1) ack_req_count = 1;
    }
    ack_status = true;
}  // end MdpAckingNode::SetLastAckObject()
        
MdpRepresentativeNode::MdpRepresentativeNode(unsigned long nodeId)
    : MdpNode(nodeId), ttl(5), loss_estimate(0.0), rtt_estimate(0.5), rtt_rough(0.5),
      rtt_variance(1.0), rate_estimate(0.0), grtt_req_sequence(0)
{  
     
}

void MdpRepresentativeNode::UpdateRttEstimate(double value)
{
#if defined(NS2) || defined(OPNET)
    if (use_asymmetric_rtt_smoothing)
		UpdateRttEstimateAsymmetric(value);
#endif // SIMULATOR
		UpdateRttEstimateNormal(value);
}  // end MdpRepresentativeNode::UpdateRttEstimate()

// Asymmetric RTT smoothing
void MdpRepresentativeNode::UpdateRttEstimateAsymmetric(double value)
{
    double err = value - rtt_rough;
    rtt_rough += (1.0/8.0) * err;
    rtt_variance += (1.0/4.0) * (fabs(err) - rtt_variance);
    
    // Smoothed asymmetric update of RTT (better for multicast?)
    err = value - rtt_estimate;
    if (err < 0.0)
        rtt_estimate += (1.0/64.0) * err;
    else
        rtt_estimate += (1.0/8.0) * err;
    
    // Update rtt_sqrt average
    err = sqrt(value) - rtt_sqrt;
    rtt_sqrt += (0.05) * err;

}  // end MdpRepresentativeNode::UpdateRttEstimate()
                
// Normal RTT smoothing

void MdpRepresentativeNode::UpdateRttEstimateNormal(double value)
{
    // For rtt variance tracking (not so smooth)
    double err = value - rtt_rough;
    rtt_rough += (1.0/8.0) * err;
    rtt_variance += (1.0/4.0) * (fabs(err) - rtt_variance);
    
    // Highly smoothed rtt_estimate
    err = value - rtt_estimate;
    rtt_estimate += (0.05) * err;
    // Update rtt_sqrt average
    err = sqrt(value) - rtt_sqrt;
    rtt_sqrt += (0.05) * err;
}  // end MdpRepresentativeNode::UpdateRttEstimateNormal()



// Below is the implementation of binary tree and link list
// manipulations for our MdpNode base class

MdpNode* MdpNode::NextInTree()
{
    if (right)
    {
        MdpNode* x = right;
        while (x->left) x = x->left;
        return x;
    }
    else  
    {
        MdpNode* x = this;
        MdpNode* y = parent;
        while(y && (y->right == x))
        {
            x = y;
            y = y->parent;
        }
        return y;
    }
}  // end MdpNode::NextInTree()


MdpNode* MdpNode::PrevInTree()
{
    if (left)
    {
        MdpNode* x = left;
        while (x->right) x = x->right;
        return x;
    }
    MdpNode* x = this;
    MdpNode* y = parent;
    while(y && (y->left == x))
    {
        x = y;
        y = y->parent;
    }
    return y;
}  // end MdpNode::PrevInTree()


MdpNodeTree::MdpNodeTree()
    : root(NULL)
{

}

MdpNodeTree::~MdpNodeTree()
{
    Destroy();
}


MdpNode *MdpNodeTree::FindNodeById(unsigned long node_id)
{
    MdpNode* x = root;
    while(x && (x->id != node_id))
    {
		if (node_id < x->id)
            x = x->left;
        else
            x = x->right;
    }
    return x;   
}  // end MdpNodeTree::FindNodeById() 



void MdpNodeTree::AttachNode(MdpNode *node)
{
    ASSERT(node);
    node->left = NULL;
    node->right = NULL;
    MdpNode *x = root;
    while (x)
    {
        if (node->id < x->id)
        {
            if (!x->left)
            {
                x->left = node;
                node->parent = x;
                return;
            }
            else
            {
                x = x->left;
            }
        }
        else
        {
           if (!x->right)
           {
               x->right = node;
               node->parent = x;
               return;
           }
           else
           {
               x = x->right;
           }   
        }
    }
    root = node;  // root _was_ NULL
}  // end MdpNodeTree::AddNode()


void MdpNodeTree::DetachNode(MdpNode* node)
{
    ASSERT(node);
    MdpNode *y, *x;
    if (!node->left || !node->right)
        y = node;
    else
        y = node->NextInTree();
    if (y->left)
        x = y->left;
    else
        x = y->right;
    if (x) x->parent = y->parent;
    if (!y->parent)
        root = x;
    else if (y == y->parent->left)
        y->parent->left = x;
    else
        y->parent->right = x;
    
    if (node != y)
    {
        if ((y->parent = node->parent))
        {
            if (y->id < y->parent->id)
                y->parent->left = y;
            else
                y->parent->right = y;
        }
        else
        {
            root = y;
        }
        if ((y->left = node->left)) y->left->parent = y;
        if ((y->right = node->right)) y->right->parent = y;
    }         
}  // end MdpNodeTree::DetachNode()


MdpNode *MdpNodeTree::Head()
{
    MdpNode *x = root;
    if (!x) return NULL;
    while (x->left) x = x->left;
    return x;   
}  // end MdpNodeTree::Head()


void MdpNodeTree::Destroy()
{
    MdpNode* node;
    while ((node = Head())) 
    {
        DetachNode(node);
        delete node;
    }
}  // end MdpNodeTree::Destroy()

MdpNodeList::MdpNodeList()
    : head(NULL), tail(NULL)
{
}

MdpNode* MdpNodeList::FindNodeById(unsigned long nodeId)
{
    MdpNode *next = head;
    while (next)
    {
        if (nodeId == next->id)
            return next;
        else
            next = next->right;
    }
    return NULL;
}  // MdpNodeList::Find()

void MdpNodeList::Append(MdpNode *theNode)
{
    ASSERT(theNode);
    theNode->left = tail;
    if (tail)
        tail->right = theNode;
    else
        head = theNode;
    tail = theNode;
    theNode->right = NULL;
}  // end MdpNodeList::Append()

void MdpNodeList::Remove(MdpNode *theNode)
{
    ASSERT(theNode);
    if (theNode->right)
        theNode->right->left = theNode->left;
    else
        tail = theNode->left;
    if (theNode->left)
        theNode->left->right = theNode->right;
    else
        head = theNode->right;
}  // end MdpNodeList::Remove()
