/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#ifndef _MDP_MESSAGE
#define _MDP_MESSAGE
        
#include "sysdefs.h"   // for bool type definition
#include "mdpError.h"
#include "networkAddress.h"

#include "debug.h"

#include <math.h>

#ifdef NS2
extern double MDP_BACKOFF_WINDOW;  // Number of GRTT periods for NACK backoff
extern unsigned int MDP_GROUP_SIZE; // Group size estimate (TBD - automate this)
#else
const double MDP_BACKOFF_WINDOW = 2;
const unsigned int MDP_GROUP_SIZE = 1000;
#endif // NS2

const unsigned char MDP_PROTOCOL_VERSION    = 9;  
const unsigned int  MSG_SIZE_MAX            = 8192;
const unsigned int  MDP_SESSION_NAME_MAX    = 64; 
const unsigned int  MDP_NODE_NAME_MAX       = 64;

const int MDP_SEGMENT_SIZE_MIN = 64;
const int MDP_SEGMENT_SIZE_MAX = 8128;  

enum MdpMessageType
{
    MDP_MSG_INVALID, 
    MDP_REPORT, 
    MDP_INFO, 
    MDP_DATA, 
    MDP_PARITY,     // MDP_PARITY could be made a type of MDP_DATA
    MDP_CMD,
    MDP_NACK,
    MDP_ACK
};
    
// MDP Node report "status" flags
enum MdpStatusFlag
{
    MDP_CLIENT =    0x01,
    MDP_SERVER =    0x02,
    MDP_ACKING =    0x04    // Client will provide positive ack of receipt
};
    
// MdpReport types
enum MdpReportType
{
    MDP_REPORT_INVALID,   
    MDP_REPORT_HELLO
};
    
/*********************************************************************
 * These "Stats" classes are used for clients reporting statistics of protocol
 * performance via MDP_REPORT messages
 */

class MdpBlockStats
{
    public:
        unsigned long   count;    // total number of blocks received
        unsigned long   lost_00;  // blocks recv'd error free first pass
        unsigned long   lost_05;  // 0-5% loss first pass
        unsigned long   lost_10;  // 5-10% loss first pass
        unsigned long   lost_20;  // 10-20% loss first pass
        unsigned long   lost_40;  // 20-40% loss first pass
        unsigned long   lost_50;  // more than 40% loss first pass
        
    public:
        int Pack(char *buffer);
        int Unpack(char *buffer);
};

class MdpBufferStats
{
    public:
        unsigned long   buf_total;  // buffer memory allocated (bytes)
        unsigned long   peak;       // peak buffer usage
        unsigned long   overflow;   // buffer overflow count
        
    public:
        int Pack(char *buffer);
        int Unpack(char *buffer);
};

class MdpClientStats
{
    public:
        unsigned long   duration;   // Period (secs) of client's participation
        unsigned long   success;    // No. of successful object transfers
        unsigned long   active;     // No. of current pending transfers
        unsigned long   fail;       // No. of failed object transfers
        unsigned long   resync;     // No. of server re-sync's
        MdpBlockStats   blk_stat;   // Recv block loss statistics
        unsigned long   tx_rate;    // Node tx_rate (bytes/sec)
        unsigned long   nack_cnt;   // Transmitted NACK count
        unsigned long   supp_cnt;   // Suppressed NACK count
        MdpBufferStats  buf_stat;   // Buffer usage stats
        unsigned long   goodput;    // ave. recv'd bytes/sec
        unsigned long   rx_rate;
        
    public:
        int Pack(char *buffer);
        int Unpack(char *buffer);
        void Init()
            {memset(this, 0, sizeof(MdpClientStats));}
};

    
class MdpReportMsg
{
    // Members
    public:
        unsigned char           status;
        unsigned char           flavor;
        char                    node_name[MDP_NODE_NAME_MAX];
        MdpClientStats          client_stats;
        const NetworkAddress*   dest;
};
    

// MDP Object flags
const int MDP_DATA_FLAG_REPAIR      = 0x01; // Marks packets which are repair transmissions
const int MDP_DATA_FLAG_BLOCK_END   = 0x02; // Marks end of transmission for a block
const int MDP_DATA_FLAG_RUNT        = 0x04; // Marks the last data packet for an object
const int MDP_DATA_FLAG_INFO        = 0x10; // Indicates availability of MDP_INFO
const int MDP_DATA_FLAG_UNRELIABLE  = 0x20;
const int MDP_DATA_FLAG_FILE        = 0x80; // Indicates object data should be stored in a file

class MdpInfoMsg  // (TBD) add other information (MIME-type, etc))
{
    public:
        unsigned short	segment_size; // Size (in bytes) of a full data segment
        unsigned short  len;
        char*           data;
};

class MdpDataMsg
{
    // Members
    public:
        unsigned long	offset;    // Offset (in bytes) of attached data
        unsigned short  segment_size;  // opt. segment size (only for LAST_FLAG msgs)
        unsigned short  len;       // Length of data vector (not a packet field)
	    char*		    data;     // Pointer to data vector
};

class MdpParityMsg
{
    // Members
    public:
        unsigned long	offset; // Offset of start of encoding block
        unsigned char   id;     // Parity vector id
        unsigned short	len;    // Length of parity vector (not a packet field)
	    char*		    data;   // Points to parity vector
};

class MdpObjectMsg
{
    public:
        unsigned short  sequence;
	    unsigned long	object_id;    // MDP server transfer object identifier
        unsigned long	object_size;  // Size (in bytes) of the object
	    unsigned char	ndata;        // Data segments per block
        unsigned char   nparity;      // Max parity segments per block
        unsigned char   flags;        // See MDP object flags defined above
        unsigned char   grtt;         // Server's current grtt estimate (quantized)
        union
        {
            MdpInfoMsg      info;
            MdpDataMsg      data;
            MdpParityMsg    parity;   
        };
};


// MdpCmdMsg flavors
enum
{
    MDP_CMD_NULL,
    MDP_CMD_FLUSH,      // commands clients to flush pending NACKs for this server
    MDP_CMD_SQUELCH,    // commands clients to quit NACKing a particular object
    MDP_CMD_ACK_REQ,    // requests clients to ACK receipt of a specific object
    MDP_CMD_GRTT_REQ,   // requests clients to include GRTT_ACK in MDP_NACKs
    MDP_CMD_NACK_ADV    // Advertises repair state from unicast NACKs
};

enum 
{
    MDP_CMD_FLAG_EOT = 0x01  // end-of-transmission flag    
};

class MdpFlushCmd
{
    public:
        char          flags;
        unsigned long object_id; // last object transmitted
};

class MdpSquelchCmd
{
    public:
        unsigned long   sync_id;    // "lowest" objectId valid for this server 
        unsigned short  len;        // len of attached data (not sent)
        char*           data;       // list of objectId's to squelch
};

class MdpAckReqCmd
{
    public:
        unsigned long   object_id;      // which object to ack
        unsigned short  len;            // len of attached data (not sent)
        char*           data;           // list of nodes needed to ACK
        
        bool FindAckingNode(unsigned long nodeId);
        bool AppendAckingNode(unsigned long nodeId, unsigned short maxLen);
};


enum
{
    MDP_CMD_GRTT_FLAG_WILDCARD = 0x01   // Wildcard response (everyone)
};
    
class MdpGrttReqCmd
{
    public:
        char           flags;
        unsigned char  sequence;
        struct timeval send_time;    // server request time stamp
        struct timeval hold_time;    // client response window (should quantize)
        unsigned short segment_size; // server tx segment_size 
        unsigned long  rate;         // current server tx_rate (bytes/second)
        unsigned char  rtt;          // quantized current bottleneck rtt estimate
        unsigned short loss;         // current bottleneck loss estimate           
        unsigned short len;          // length of data content (not sent)
        char*          data;         // list of representative node id's
                                     // and their rtt's
        
        bool AppendRepresentative(unsigned long repId, double repRtt,
                                  unsigned short maxLen);
        bool FindRepresentative(unsigned long repId, double* repRtt);
        
};

class MdpNackAdvCmd
{
    public:
        unsigned short len; 
        char*          data;  
};
    
class MdpCmdMsg
{
    public:
        unsigned short  sequence;
        unsigned char   grtt;
        unsigned char   flavor;     // command flavor (see above)
        union
        {
            MdpFlushCmd     flush;
            MdpSquelchCmd   squelch;
            MdpAckReqCmd    ack_req;
            MdpGrttReqCmd   grtt_req;
            MdpNackAdvCmd   nack_adv;
        };
};

    
// An "MdpNackMsg" contains a concatenation of one or more
// "MdpObjectNacks".  Each "MdpObjectNack" contains a
// concatenation of one or more "MdpRepairNacks" of varying
// types.
            
class MdpNackMsg
{
    public:
    // Members
        unsigned long           server_id;
        struct timeval          grtt_response;
        unsigned short          loss_estimate;
        unsigned char           grtt_req_sequence;
        char*                   data;
        unsigned short          nack_len;  // nack_len field _not_ explicitly sent 
                                           // receiver derives from packet size          
        const NetworkAddress*   dest;
        class MdpServerNode*    server;
        
};

enum MdpRepairType
{
    MDP_REPAIR_INVALID = 0,
    MDP_REPAIR_SEGMENTS,    // (1) Repair needed only for some segments
    MDP_REPAIR_BLOCKS,      // (2) Entire block retransmissions needed
    MDP_REPAIR_INFO,        // (3) Object info needed
    MDP_REPAIR_OBJECT       // (4) Entire object needs retransmitted
};
    
class MdpRepairNack
{
    public:
    // Members 
        MdpRepairType   type;
        unsigned char   nerasure;  // MDP_REPAIR_SEGMENT only
        unsigned long   offset;    // MDP_REPAIR_BLOCK, MDP_REPAIR_SEGMENT
        unsigned short  mask_len;  // length of attached BLOCK/SEGMENT repair mask
        char*           mask;      // pointer to mask data
        
        int Pack(char* buffer, unsigned int buflen);
        int Unpack(char* buffer);
};    

class MdpObjectNack
{
    public:
    // Members   
        unsigned long   object_id;
        unsigned short  nack_len;     // nack_len field _is_ explicitly sent
        unsigned short  max_len;
        char*           data;
        
        bool Init(unsigned long id, char* buffer, unsigned int buflen);
        bool AppendRepairNack(MdpRepairNack* nack);
        int Pack();
        int Unpack(char* buffer); 
};

        
enum MdpAckType
{
    MDP_ACK_INVALID = 0,  
    MDP_ACK_OBJECT,
    MDP_ACK_GRTT
};
    
class MdpAckMsg
{
    // Members
    public:
        unsigned long           server_id;    // ID of server to receive ACK
        struct timeval          grtt_response;
        unsigned short          loss_estimate;
        unsigned char           grtt_req_sequence;
        MdpAckType              type;
        unsigned long           object_id;    // ID of object we are ACKing
        
        const NetworkAddress*   dest;
        class MdpServerNode*    server;
};

class MdpMessage
{  
    friend class MdpMessageQueue;
    friend class MdpMessagePool;
    
    // Members
    public:
        // (TBD) Compress type/version fields?
	    unsigned char		type;    // Type of MDP message
	    unsigned char		version; // MDP protocol version number
	    unsigned long		sender;  // Node id of message source
	    union
	    {
	        MdpReportMsg	    report;
            MdpObjectMsg        object;  // includes MDP_INFO, MDP_DATA, & MDP_PARITY
            MdpCmdMsg           cmd;
            MdpNackMsg          nack;
            MdpAckMsg           ack;
	    };
        
    private:
        MdpMessage *prev;
        MdpMessage *next;   // for message queues, pools, etc
	
    // Methods
    public:
	    // Returns length of message packed
	    int Pack(char *buffer);        
        // Returns FALSE if message is bad
	    bool Unpack(char *buffer, int packet_length); 
        MdpMessage* Next() {return next;}
};

// (TBD) make these static members of the MdpMessage class for name space protection
// Some utility routines
int AppendNackData(char *buffer, unsigned char type, unsigned long id,
                   unsigned long mask_len, 
                   unsigned char *mask);
// Returns number of bytes parsed
int ParseNackData(char *buffer, unsigned char *type, unsigned long *id,
                  unsigned long *mask_len, 
                  unsigned char **mask);


// These are the GRTT estimation bounds set for the current
// MDPv2 protocol.  (Note that our Grtt quantization routines
// are good for the range of 1.0e-06 <= 1000.0)
const double MDP_GRTT_MIN = 0.001;  // 1 msec
const double MDP_GRTT_MAX = 15.0;   // 15 sec

const double RTT_MIN = 1.0e-06;
const double RTT_MAX = 1000.0;        
inline double UnquantizeRtt(unsigned char rtt_q)
{
	return ((rtt_q < 31) ? 
			(((double)(rtt_q+1))/(double)RTT_MIN) :
		    (RTT_MAX/exp(((double)(255-rtt_q))/(double)13.0)));
}

unsigned char QuantizeRtt(double rtt);

inline unsigned short QuantizeLossFraction(double loss)
{
    return (unsigned short) (65535.0 * loss);
}  // end QuantizeLoss()

inline double UnquantizeLossFraction(unsigned short loss_q)
{
    return (((double)loss_q) / 65535.0);
}  // end UnquantizeLoss

// Prioritized FIFO queue of MDP messages
class MdpMessageQueue
{
    // Members
    private:
        MdpMessage *head, *tail;
    
    // Methods
    public:
        MdpMessageQueue();
        bool IsEmpty() {return (head == NULL);}
        void QueueMessage(MdpMessage *theMsg);
        MdpMessage *GetNextMessage();
        MdpMessage *Head() {return head;}
        void Remove(MdpMessage *theMsg);
        MdpMessage* FindNackAdv();
};    

// Right now, the pool maintains a fixed amount of messages as
// dictated on "Init()" ... eventually some hooks for dynamic
// resource management will be added.
class MdpMessagePool
{
    // Members
    private:
        unsigned long   message_count;
        unsigned long   message_total;
        MdpMessage      *message_list;
            
    // Methods
    public:
        MdpMessagePool();
        ~MdpMessagePool();
        MdpError Init(unsigned long count);
        void Destroy();
        MdpMessage *Get();
        void Put(MdpMessage *);
        unsigned long Count() {return message_count;}
        unsigned long Depth() {return message_total;}
};
    


#endif // _MDP_MESSAGE
