/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
 
#ifndef _MDP_API
#define _MDP_API

#ifdef WIN32
#include "winsock2.h"  // for SOCKET definition under WIN32
#endif
              
    
/* Mdp type definitions */
typedef const void* MdpInstanceHandle;
typedef const void* MdpSessionHandle;
typedef const void* MdpNodeHandle;
typedef const void* MdpObjectHandle;

typedef unsigned long MdpNodeId;
typedef unsigned long MdpObjectTransportId;

// Currently supported Mdp object types
enum MdpObjectType 
{
    MDP_OBJECT_INVALID, 
    MDP_OBJECT_FILE, 
    MDP_OBJECT_DATA, 
    MDP_OBJECT_SIM
};

/* Mdp constants */
#ifndef NULL
#define NULL 0
#endif /* !NULL */
const MdpInstanceHandle MDP_NULL_INSTANCE = ((MdpInstanceHandle)0);
const MdpSessionHandle MDP_NULL_SESSION = ((MdpSessionHandle)0);
const MdpObjectHandle MDP_NULL_OBJECT = ((MdpObjectHandle)0);

const MdpNodeId MDP_NULL_NODE = ((MdpNodeId)0);

const unsigned int MDP_SESSION_NAME_MAX = 64; 
const unsigned int MDP_NODE_NAME_MAX = 64;

/* These defaults are listed for user information only.  Changing them
   here in the header file will have no affect on protocol operation.
   API functions are provided to properly override these default
   settings as needed */
const unsigned char MDP_DEFAULT_TTL = 8;
const unsigned long MDP_DEFAULT_TX_RATE = 64000;
const unsigned int MDP_DEFAULT_SEGMENT_SIZE = 1024;
const unsigned int MDP_DEFAULT_BLOCK_SIZE = 64;
const unsigned int MDP_DEFAULT_NPARITY = 32;
const unsigned long MDP_DEFAULT_BUFFER_SIZE = (1024*1024);  // 1 Mbyte
const double MDP_DEFAULT_GRTT_ESTIMATE = 0.5;

const unsigned long MDP_DEFAULT_TX_CACHE_COUNT_MIN = 8;
const unsigned long MDP_DEFAULT_TX_CACHE_COUNT_MAX = 64;
const unsigned long MDP_DEFAULT_TX_CACHE_SIZE_MAX = (8*1024*1024);
                
/* Mdp Error Codes */
enum MdpError 
{
    MDP_ERROR_NONE = 0,
	MDP_ERROR,
    MDP_ERROR_INSTANCE_INVALID,
    MDP_ERROR_SESSION_INVALID,
    MDP_ERROR_OBJECT_INVALID,
    MDP_ERROR_NODE_INVALID,
    MDP_ERROR_PARAMETER_INVALID,    
    MDP_ERROR_TX_QUEUE_FULL,
    MDP_ERROR_FILE_OPEN,
    MDP_ERROR_FILE_EMPTY,
    MDP_ERROR_FILE_LOCKED,
    MDP_ERROR_MEMORY,
    MDP_ERROR_SOCKET_OPEN,
    MDP_ERROR_DNS_FAIL,
    MDP_ERROR_SOCKET_MCAST,
    MDP_ERROR_SOCKET_TOS
};   

enum MdpNackingMode 
{
    MDP_NACKING_NONE, 
    MDP_NACKING_INFOONLY, 
    MDP_NACKING_NORMAL
};

/* Function prototypes */
const char* MdpVersion();

/* This function must be called before anything else */
/* The default "NULL" arguments cause Mdp to automatically pick
   default node id (local IP address) and node name ("user@host") */
MdpInstanceHandle MdpInit(const char* localName = (const char*)NULL, 
                          MdpNodeId   localId = MDP_NULL_NODE,
                          MdpError*   error = NULL);

MdpNodeId MdpGetLocalNodeId(MdpInstanceHandle instance);
MdpError MdpSetLocalNodeId(MdpInstanceHandle instance, MdpNodeId nodeId);

void MdpInstanceSetUserData(MdpInstanceHandle instance, const void* userData);
const void* MdpInstanceGetUserData(MdpInstanceHandle instance);

void MdpDestroy(MdpInstanceHandle instance);

/* MdpSession creation/destruction */
MdpSessionHandle MdpNewSession(MdpInstanceHandle    instance,
                               const char*          address, 
                               unsigned short       rxPort, 
                               unsigned short       txPort = 0,
                               unsigned char        ttl = MDP_DEFAULT_TTL, 
                               MdpError*            error = NULL);

void MdpDeleteSession(MdpInstanceHandle instance, 
                      MdpSessionHandle  sessionHandle);

void MdpSessionSetUserData(MdpInstanceHandle instance, const void* userData);
void* MdpSessionGetUserData(MdpInstanceHandle instance);

/* Mdp Session parameters */

/****************************************************************************/
/* General parameters ******************************************************/

/* These can be called at _any_ time after session is created */
MdpError MdpSessionSetTxRate(MdpSessionHandle sessionHandle, double txRate);
MdpError MdpSessionSetTxRateBounds(MdpSessionHandle sessionHandle, double min, double max);
MdpError MdpSessionSetGrttProbeInterval(MdpSessionHandle sessionHandle, double intervalMin, double intervalMax);
MdpError MdpSessionSetGrttProbing(MdpSessionHandle sessionHandle, bool state);
MdpError MdpSessionSetStatusReporting(MdpSessionHandle sessionHandle, bool state);
MdpError MdpSessionSetTOS(MdpSessionHandle sessionHandle, unsigned char theTOS);


MdpError MdpSessionSetMulticastLoopback(MdpSessionHandle    sessionHandle,
                                        bool                state);
MdpError MdpSessionSetPortReuse(MdpSessionHandle    sessionHandle,
                                bool                state);
MdpError MdpSessionSetRobustFactor(MdpSessionHandle sessionHandle, unsigned int value);


#ifdef PROTO_DEBUG
MdpError MdpSessionSetRecvDropRate(MdpSessionHandle sessionHandle, double rate);
MdpError MdpSessionSetSendDropRate(MdpSessionHandle sessionHandle, double rate);
void MdpSetMessageTrace(bool state);
#endif // PROTO_DEBUG

/* These can be set only _before_ the session is opened as client or server */
MdpError MdpSessionSetMulticastInterfaceAddress(MdpSessionHandle sessionHandle, 
                                                const char*      interfaceAddr);

/****************************************************************************/
/* Server-specific functions ************************************************/

/* These should called set only _before_ the session is opened as a server */
MdpError MdpSessionSetBaseObjectTransportId(MdpSessionHandle     sessionHandle, 
                                            MdpObjectTransportId baseId);

/* Start server operation */
MdpError MdpSessionOpenServer(MdpSessionHandle sessionHandle, 
                              int segmentSize = MDP_DEFAULT_SEGMENT_SIZE, 
                              int blockSize = MDP_DEFAULT_BLOCK_SIZE, 
                              int nparity = MDP_DEFAULT_NPARITY,
                              unsigned long buffer_size = MDP_DEFAULT_BUFFER_SIZE);

/* These can be set at _any_ time */
MdpError MdpSessionSetCongestionControl(MdpSessionHandle sessionHandle, bool state);
MdpError MdpSessionSetServerEmcon(MdpSessionHandle sessionHandle, bool state);
MdpError MdpSessionSetAutoParity(MdpSessionHandle   sessionHandle, 
                                 unsigned char      autoParity);
// (TBD) Change these acking funcs to use MdpNodeId as second argument!!!
MdpError MdpSessionAddAckingClient(MdpSessionHandle sessionHandle, const char* host);
MdpError MdpSessionRemoveAckingClient(MdpSessionHandle sessionHandle, const char* host);
MdpError MdpSessionSetTxCacheDepth(MdpSessionHandle  sessionHandle, unsigned long minCount,
                                   unsigned long maxCount,  unsigned long maxSize);
MdpError MdpSessionSetServerGrttEstimate(MdpSessionHandle sessionHandle, double initalGrtt);
MdpError MdpSessionGetServerGrttEstimate(MdpSessionHandle sessionHandle, double* grttEstimate);
MdpError MdpSessionSetServerGrttMax(MdpSessionHandle sessionHandle, double grttMax);

/* These should be called only _after_ server start */


/* Queue a file object for transmission  */
MdpObjectHandle MdpSessionQueueTxFile(MdpSessionHandle  sessionHandle, 
                                      const char*       path, 
                                      const char*       name, 
                                      MdpError*         error);

/* Queue a static data object for transmission */
MdpObjectHandle MdpSessionQueueTxData(MdpSessionHandle  sessionHandle, 
								  const char*           infoPtr,
                                  unsigned short        infoSize,
                                  char*                 dataPtr,
								  unsigned long         dataSize,
                                  MdpError*             error);

/* Remove object from the server transmit queue, 
   stops transmission and repair of the object */
MdpError MdpSessionRemoveTxObject(MdpSessionHandle sessionHandle, 
                                  MdpObjectHandle  objectHandle);

/* This routine has potential application for EMCON transmission modes */
/* Note: The specific transmit object _must_ be de-queued before re-queuing it! */
MdpObjectHandle MdpSessionRequeueTxFile(MdpSessionHandle sessionHandle, const char* path, 
                                        const char* name, MdpError* error,
                                        MdpObjectTransportId transportId);

/* Stop server operation */
MdpError MdpSessionCloseServer(MdpSessionHandle sessionHandle,
                               bool             hardShutdown = false);

/****************************************************************************/
/* Client-specific functions *********************************************/

/* These should be called only _before_ the session is opened as a client */
MdpError MdpSessionSetClientEmcon(MdpSessionHandle sessionHandle, bool state);

/* Start client operation */
MdpError MdpSessionOpenClient(MdpSessionHandle sessionHandle, 
                              unsigned long bufferSize = MDP_DEFAULT_BUFFER_SIZE);

/* These can be called at _any_ time */
MdpError MdpSessionSetClientAcking(MdpSessionHandle sessionHandle, bool state);
MdpError MdpSessionSetClientUnicastNacks(MdpSessionHandle sessionHandle, bool state);
MdpError MdpSessionSetClientMulticastAcks(MdpSessionHandle sessionHandle, bool state);
MdpError MdpSessionSetArchiveDirectory(MdpSessionHandle sessionHandle, const char *path);
MdpError MdpSessionSetArchiveMode(MdpSessionHandle sessionHandle, bool state);
MdpError MdpSessionSetDefaultNackingMode(MdpSessionHandle sessionHandle, MdpNackingMode nackingMode);
MdpError MdpSessionSetClientStreamIntegrity(MdpSessionHandle sessionHandle, bool state);

/* These can be called only _after_ client start */
MdpError MdpSessionAbortRxObject(MdpSessionHandle   sessionHandle, 
                                 MdpObjectHandle    objectHandle);
MdpError MdpSessionGetNodeAddress(MdpSessionHandle  sessionHandle,
                                  MdpNodeId         nodeId, 
                                  unsigned long*    address, 
                                  unsigned short*   port);
MdpError MdpSessionGetNodeGrttEstimate(MdpSessionHandle sessionHandle,
                                       MdpNodeId        nodeId, 
                                       double*          grttEstimate);

MdpObjectHandle MdpSessionGetNodeNextRxObject(MdpSessionHandle sessionHandle,
                                              MdpNodeId        nodeId,
                                              MdpObjectHandle  previousObject,
                                              MdpError*        error = NULL);

MdpError MdpSessionDeactivateNode(MdpSessionHandle sessionHandle,
                                  MdpNodeId        nodeId);

MdpError MdpSessionDeleteNode(MdpSessionHandle sessionHandle,
                              MdpNodeId        nodeId);

/* Stop client operation */
MdpError MdpSessionCloseClient(MdpSessionHandle sessionHandle);


/* MdpObject information and management */

/* You should only use the following routines for objectHandle's passed to your 
   MdpNotifyCallback routine or immediately after queuing an object for transmission ... 
   The objectHandle's may not be valid at other times and use of bad object id's can make 
   the current MDPv2 API crash.  This will be addressed when a more robust API is 
   created. */


MdpObjectTransportId MdpObjectGetTransportId(MdpObjectHandle objectHandle);

/* This will return MDP_NULL_NODE if local node is the source */
MdpNodeId MdpObjectGetSourceNodeId(MdpObjectHandle objectHandle);

MdpObjectType MdpObjectGetType(MdpObjectHandle objectHandle);
unsigned short MdpObjectGetInfoSize(MdpObjectHandle objectHandle);
unsigned short MdpObjectGetInfo(MdpObjectHandle objectHandle, char* buffer, unsigned short *buflen);
unsigned long MdpObjectGetSize(MdpObjectHandle objectHandle);
unsigned long MdpObjectGetRecvBytes(MdpObjectHandle objectHandle);

MdpError MdpObjectSetNackingMode(MdpObjectHandle objectHandle, MdpNackingMode nackingMode);

/* These should only be called for objects of type "MDP_DATA_OBJECT" */
MdpError MdpObjectSetData(MdpObjectHandle objectHandle, char* dataPtr, unsigned long dataSize);
char* MdpObjectGetData(MdpObjectHandle objectHandle, unsigned long* dataSize, MdpError* error);

/* This will return MDP_ERROR_OBJECT_INVALID for non MDP_FILE_OBJECTs */
MdpError MdpObjectGetFileName(MdpObjectHandle objectHandle, char* name, unsigned int maxlen);


/* Callback function prototypes */

/* Why does MDPv2 use this messy callback approach in its API?
   - A better long term approach will be to embed MDPv2 protocol
   - operation in its own thread.  But meanwhile, to simplify cross-
   - platform portability (and until we think of a better approach),
   - we'll allow developers using MDPv2 for applications (or simulations)
   - to do that however they like.  These callback mechanisms can
   - be buried by developers however they see fit for the platforms
   - for which they are concerned.  Example code will be provided on
   - using the callbacks for timer and socket installation
 */


/* Mdp "Notify" codes */
enum MdpNotifyCode
{
    MDP_NOTIFY_ERROR,
    MDP_NOTIFY_TX_OBJECT_START,
    MDP_NOTIFY_TX_OBJECT_FIRST_PASS,
    MDP_NOTIFY_TX_OBJECT_ACK_COMPLETE,
    MDP_NOTIFY_TX_OBJECT_FINISHED,
    MDP_NOTIFY_TX_QUEUE_EMPTY,
    MDP_NOTIFY_RX_OBJECT_START, 
    MDP_NOTIFY_RX_OBJECT_INFO,
    MDP_NOTIFY_RX_OBJECT_UPDATE,
    MDP_NOTIFY_RX_OBJECT_COMPLETE,
	MDP_NOTIFY_OBJECT_DELETE,
    MDP_NOTIFY_LOCAL_SERVER_CLOSED,
    MDP_NOTIFY_REMOTE_SERVER_NEW,
    MDP_NOTIFY_REMOTE_SERVER_ACTIVE,
    MDP_NOTIFY_REMOTE_SERVER_INACTIVE,
    MDP_NOTIFY_REMOTE_SERVER_DELETE,
};

/* Note - The user's MDPv2 application should not spend much time
        - in the "notify callback" since it blocks the thread of
        - execution of the MDPv2 protocol.  The user of this code
        - should consider burying the MDPv2 protocol engine in a
        - separate thread or process if needed. (See above note).
*/

/* The "MdpNotifyCallback" is used by MDPv2 to "notify" the host
   application of updated session/object status.  This allows the
   host application to update user interface displays as needed,
   kick off post-processing of received data, and be prompted
   for additional data to transmit 
*/
typedef bool (MdpNotifyCallback)(MdpNotifyCode      notifyCode,
                                 MdpInstanceHandle  instanceHandle,
                                 MdpSessionHandle   sessionHandle,
                                 MdpNodeHandle      nodeHandle,
                                 MdpObjectHandle    objectHandle,
                                 MdpError           errorCode);

void MdpSetNotifyCallback(MdpInstanceHandle     instanceHandle, 
                          MdpNotifyCallback*    notifyFunc);


/* The "MdpTimerInstallCallback" is used by MDPv2 to request the host
   application to install, modify, or remove a system timer which
   MDPv2 uses for protocol timing.  (The "MdpOnTimerEvent" function should
   be called when the system timer fires) (Note: MDPv2 assumes the system
   timer is a "one-shot" timer which needs re-installation after it fires)
*/                            

typedef const void* MdpTimerHandle;

// Possible TimerInstaller "Actions"
enum MdpTimerInstallCmd
{
    MDP_TIMER_INSTALL, 
    MDP_TIMER_MODIFY, 
    MDP_TIMER_REMOVE
};
    

typedef bool (MdpTimerInstallCallback)(MdpTimerInstallCmd cmd, 
                                       double             delay,
                                       MdpTimerHandle     timerHandle,
                                       MdpInstanceHandle  instanceHandle);

void MdpSetTimerInstallCallback(MdpInstanceHandle           instanceHandle, 
                                MdpTimerInstallCallback*    timerInstaller);
                                    
/* These can be called while a MdpTimerHandle is valid.  This is the interval
   between the when INSTALL and REMOVE commands are issued for a given
   MdpTimerHandle */
void MdpTimerSetUserData(MdpTimerHandle timerHandle, const void* userData);
const void* MdpTimerGetUserData(MdpTimerHandle timerHandle);

/* The user's code should call this routine (with the corresponding 
   "timerHandle" supplied when the "TimerInstallCallback" was called) 
   when the installed timer fires.  This prompts MDP to service its 
   pending timeouts.
*/
double MdpTimerOnTimeout(MdpTimerHandle timerHandle);


                                    
/* The "MdpSocketInstallCallback" is used by MDPv2 to request the host 
   application to asynchronously "monitor" a socket file descriptor for
   data to be read.  The host application must call the "MdpOnSocketDataReady" 
   function with the corresponding "socketHandle" supplied when there is data 
   ready to be read on the socket file descriptor installed with the 
   "SocketInstallCallback".
*/

typedef const void* MdpSocketHandle;
#ifdef WIN32
SOCKET MdpSocketGetDescriptor(MdpSocketHandle socketHandle);
const SOCKET MDP_INVALID_SOCKET = INVALID_SOCKET;
#else 
int MdpSocketGetDescriptor(MdpSocketHandle socketHandle);
const int MDP_INVALID_SOCKET = -1;
#endif

enum MdpSocketInstallCmd 
{
    MDP_SOCKET_INSTALL, 
    MDP_SOCKET_REMOVE
};
        
    
typedef bool (MdpSocketInstallCallback)(MdpSocketInstallCmd cmd,
                                        MdpSocketHandle     socketHandle,
                                        MdpInstanceHandle   instanceHandle);

void MdpSetSocketInstallCallback(MdpInstanceHandle         instance, 
                                 MdpSocketInstallCallback* socketInstaller);

/* These can be called while a MdpSocketHandle is valid.  This is the interval
   between the when INSTALL and REMOVE commands are issued for a given
   MdpSocketHandle */
void MdpSocketSetUserData(MdpSocketHandle socketHandle, const void* userData);
const void* MdpSocketGetUserData(MdpSocketHandle socketHandle);

/* The user's code should call this routine (with the "socketHandle" supplied
   by the corresponding "MdpSocketInstallCallback") when there is data available
   to be read on an installed socket. */

void MdpSocketOnDataReady(MdpSocketHandle socketHandle);

     
#endif /* _MDP_API */
