/*
   Copyright (C) 2004 Manuel Bouyer
This file is part of the pty UART module for gpsim.
the pty UART module for gpsim is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

*/

#include <errno.h>
#include <stdlib.h>
#include <string>
#include <util.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <termios.h>
#ifdef __FreeBSD__
#include <libutil.h>
#else
#include <util.h>
#endif

#include "usart.h"


ptyUSART_external::ptyUSART_external(ptyUSART_CORE *_usart)
{
    char name[20];
    int on;
    usart = _usart;
    if (openpty(&master, &slave, name, NULL, NULL) < 0) {
      perror("openpty");
      exit(-1);
    }
    cout << "opened tty: " <<  name << endl;
    on = 1;
    if (ioctl(master, TIOCEXT, &on) < 0)
      perror("TIOCEXT");
    on = 1;
    if (ioctl(master, TIOCPKT, &on) < 0)
      perror("TIOCPKT");
    bufsize = 0;
}

ptyUSART_external::~ptyUSART_external(void)
{
    close(slave);
    close(master);
}

void ptyUSART_external::put_char(const unsigned char c)
{
    fd_set wr;
    struct timeval timeout;

    FD_ZERO(&wr);      
    FD_SET(master, &wr);
    memset(&timeout, 0, sizeof(struct timeval));
    if (select(master+1, NULL, &wr, NULL, &timeout) < 0) {
      perror("select");
      return;
    }
    if (FD_ISSET(master, &wr))
      write(master, &c, 1);
}

int ptyUSART_external::poll(void)
{
    fd_set rd;
    struct timeval timeout;
    int rsize;
    struct termios ts;
    bool use_parity, parity;
    int bits, stop;

    while (1) {
      if (bufsize > 0) { // do we have still data in the buffer ?
	unsigned char value = *curchar;
	curchar++; bufsize--;
	return value;
      }
      // see if we can read some more data from the pty
      FD_ZERO(&rd);      
      FD_SET(master, &rd);
      memset(&timeout, 0, sizeof(struct timeval));
      if (select(master+1, &rd, NULL, NULL, &timeout) < 0) {
        perror("select");
        return -1;
      } 
      if (!FD_ISSET(master, &rd))
        return -1;
      rsize = read(master, rbuf, BUFSIZ);
      if (rsize == 0)
        return -1;
      if (rsize < 0) {
        perror("read");
        return -1;
      }
      if (rbuf[0] & TIOCPKT_IOCTL) {
        memcpy(&ts, &rbuf[1], MIN(rsize - 1, sizeof(struct termios)));
        //printf("c_iflag 0x%x c_oflag 0x%x c_cflag 0x%x ispeed %d ospeed %d\n", ts.c_iflag, ts.c_oflag, ts.c_cflag, ts.c_ispeed, ts.c_ospeed);
	switch (ts.c_cflag & CSIZE) {
	case CS5:
	  bits = 5;
	  break;
	case CS6:
	  bits = 6;
	  break;
	case CS7:
	  bits = 7;
	  break;
	case CS8:
	  bits = 8;
	  break;
	}
	if (ts.c_cflag & CSTOPB)
	  stop = 2;
	else
	  stop = 1;
        if (ts.c_cflag & PARENB) {
	  use_parity = 1;
          if (ts.c_cflag & PARODD)
            parity = 1;
          else
           parity = 0;
        } else
          use_parity = 0;
        usart->new_serial_parameters(ts.c_ispeed, ts.c_ospeed, bits, use_parity, parity, stop);
        continue;
      }
      curchar = &rbuf[1];
      bufsize = rsize - 1;
   }
}
