/*
   Copyright (C) Manuel Bouyer
   Copyright (C) 1998,1999,2000,2001 T. Scott Dattalo

This file is part of gpsim.

gpsim is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

gpsim is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with gpsim; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*

  usart.cc

  This is gpsim's universal synchronous/asynchronous receiver/transceiver.

  Features:

    8 or 9 bit receiver and transmitter
    0 or 1 start bits
    0 or 1 stop bits
    0 or 1 parity bits and even/odd selectable
    variable sized transmit and receive buffers

*/


#include <errno.h>
#include <stdlib.h>
#include <string>

#include "usart.h"

#define DEFAULT_BAUD    9600

#define RX_MAX_EVENTS 1024

// #define DEBUG


/**********************************************************************************

             gpsim's USART module


  The  USART module is a general purpose universal synchronous/asynchronous
serial receiver and transmitter. In other words, it's a serial port. It's 
purpose is to provide a tool to assist in the debugging of serial interfaces.
Users can load this module and tie it to their receive and transmit pins
of their simulated PIC's. Then experiments can be conducted on things like
baud rate variation, transmit inundation, protocol development, etc.
This module is attached to a pty, so that the usual Unix serial tools can
be used against the virtual serial interface.

**********************************************************************************/


//--------------------------------------------------------------
//
//

class ptyUSART_RXPIN : public IOPIN
{
public:

  ptyUSART_CORE *usart;

  ptyUSART_RXPIN(void) {
    //cout << "ptyUSART_RXPIN constructor - do nothing\n";
  }
  ptyUSART_RXPIN (ptyUSART_CORE *_usart,
	       IOPORT *i, 
	       unsigned int b, 
	       char *opt_name=NULL) : IOPIN(i,b,opt_name, (Register **)0) { 

    usart = _usart;

    // Let the pin think it's in the high state. If this is wrong,
    // then the I/O pin driving it will correct it. (Starting off
    // low prevents the start bit from being captured.) 
    // Note, may want to add a flag that indicates if the pin
    // has ever been driven at all. This way, we can capture the
    // first edge. Or we could add another parameter to the constructor.

    bDrivenState = true;

  };

  virtual
  void setDrivenState(bool new_dstate) { 
    bool diff = new_dstate ^ bDrivenState;

#ifdef DEBUG
    cout << "usart rx setDrivenState " << new_dstate << " at " <<
	cycles.value << '\n';
#endif
    if( usart && diff ) {
      bDrivenState = new_dstate;
      IOPIN::setDrivenState(new_dstate);
      usart->new_rx_edge(new_dstate);

    }

  }

};




//--------------------------------------------------------------
//
//

class ptyUSART_TXPIN : public IO_bi_directional
{
public:

  ptyUSART_CORE *usart;

  ptyUSART_TXPIN(void) {
    //cout << "ptyUSART_TXPIN constructor - do nothing\n";
  }
  ptyUSART_TXPIN (ptyUSART_CORE *_usart,
	       IOPORT *i, 
	       unsigned int b, 
	       char *opt_name=NULL) : IO_bi_directional(i,b,opt_name) { 

    usart = _usart;

    bDrivingState = true;
    update_direction(1,true);   // Make the TX pin an output.

  };
  // Always return the value of bDrivingState, not the value or the port
  bool getDrivingState(void) {return bDrivingState;}

};





//=================================================================
//
//
//  ptyTXREG
//  Create a transmit register based upon the transmit register
// defined in the main gpsim code.
//

class ptyTXREG : public TriggerObject
{
 public:
  ptyUSART_TXPIN *txpin;


  guint64 time_per_bit;
  guint64 time_per_packet;

  double baud;
  int bits_per_byte;
  int  stop_bits;
  bool use_parity;
  bool parity;         // 0 = even, 1 = odd

  unsigned int txr;    // Transmit register
  int bit_count;       // used while transmitting.

  ptyTXREG(void) {
    txpin = NULL;

    baud = DEFAULT_BAUD;
    bits_per_byte = 8;
    stop_bits = 1;
    use_parity = 0;

    txr = 0;
    bit_count = 0;

    update_packet_time();

    cycles.set_break(cycles.value + time_per_bit, this);

  }

  void update_packet_time(void) {
    Processor *cpu = get_active_cpu();
    // for now the stop bit time is included in the total packet time

    if(baud <= 0.0)
      baud = DEFAULT_BAUD;  //arbitrary
    if (cpu == NULL) {
      time_per_packet = time_per_bit = 0;
      return;
    }
    /*
      Calculate the total time to send a "packet", i.e. start bit, data, parity, and stop
    */

    time_per_packet = cpu->get_frequency() * ( (1.0 +             // start bit
					    (double)bits_per_byte + // data bits
					    (double)stop_bits  +  // stop bit(s)
					    (double)use_parity)   //
					    /baud);
    time_per_bit = cpu->get_frequency() * ( 1.0/baud );
    //cout << "update_packet_time ==> 0x" << hex<< time_per_packet << "\n";
    //cout << "update_bit_time ==> 0x" << hex<< time_per_bit << "\n";
  }

  void new_serial_parameters(int new_baud, int data_bits, bool new_use_parity, bool new_parity, int new_stop_bits)
  {
    baud = new_baud;
    bits_per_byte = data_bits;
    use_parity = new_use_parity;
    parity = new_parity;
    stop_bits = new_stop_bits;
    update_packet_time();
  }

  virtual void callback(void) {

    guint64 cur_time = cycles.value;

    if (bit_count) {
      //cout << "UART TX: callback() bit_count" << bit_count << "\n";
      if(txpin) {
        txpin->putState((txr & 1) ? true : false);
        //cout << "usart tx module sent a " << (txr&1) <<  " bit count " << bit_count << '\n';
      }
      txr >>= 1;
      bit_count--;

    } else {
      build_tx_packet();
    }
    cycles.set_break(cur_time + time_per_bit, this);
  }

  void build_tx_packet(void) {
    int tb;

    bit_count = 0;
  
    // cout << "USART TX poll()\n";
    tb = txpin->usart->external_io->poll();
    if (tb < 0)
      return;

    unsigned int tx_byte = ((int)tb) &  ((1<<bits_per_byte) - 1);

    txr =  ((3 << bits_per_byte) | tx_byte) << 1;

    // total bits = byte + start and stop bits
    bit_count = bits_per_byte + 1 + 1 + 1;

    if (verbose)
      cout << hex << "ptyTXREG::" << __FUNCTION__ << " byte to send 0x" << (int)tb <<" txr 0x" << txr << "  bits " << bit_count << '\n';

  }

};

//=================================================================
//
//  ptyRCREG 
//
// Create a receive register
//

class ptyRCREG : public TriggerObject
{
 public:

  ptyUSART_RXPIN *rxpin;

#define MAX_PW  0xfffffff

  enum RX_STATES {
    RS_WAITING_FOR_START,
    RS_RECEIVING,
    RS_STOPPED,
    RS_OVERRUN
  } receive_state;

  BoolEventLogger *rx_event;

  double baud;
  int bits_per_byte;
  double  stop_bits;
  bool use_parity;
  bool parity;         // 0 = even, 1 = odd

  guint64 time_per_bit;
  guint64 time_per_packet;

  guint64 start_time;

  unsigned int start_bit_time;
  unsigned int start_bit_index;
  bool last_bit;

  struct _pw {
    guint64 width;
    guint64 sumofwidths;
    guint32 occurs;
  } pulses[64];
  guint32 start_index;

  /**************************/
  // ptyRCREG constructor
  /**************************/
  ptyRCREG(void) {

    start_bit_time = 0;
    start_bit_index = 0;
    start_index = 0;
    last_bit = 1;
    rx_event = new BoolEventLogger(RX_MAX_EVENTS);

    for(int i = 0; i<64; i++) {
      pulses[i].width = MAX_PW;
      pulses[i].sumofwidths = 0;
      pulses[i].occurs = 0;
    }

    receive_state = RS_WAITING_FOR_START;

    new_serial_parameters(DEFAULT_BAUD, 8, 0, 0, 1);

  }

  void new_serial_parameters(int new_baud, int data_bits, bool new_use_parity, bool new_parity, int new_stop_bits)
  {
    baud = new_baud;
    bits_per_byte = data_bits;
    use_parity = new_use_parity;
    parity = new_parity;
    stop_bits = new_stop_bits;
    update_packet_time();
  }

  void update_packet_time(void) {
    Processor *cpu = get_active_cpu();
    // for now the stop bit time is included in the total packet time

    // cout << "update_packet_time baud " << baud << endl;
    if(baud <= 0.0)
      baud = DEFAULT_BAUD;  //arbitrary
    if (cpu == NULL) {
      time_per_packet = time_per_bit = 0;
      return;
    }
    // cout << "update_packet_time baud now " << baud << endl;

    /*
      Calculate the total time to send a "packet", i.e. start bit, data, parity, and stop
    */

    time_per_packet = cpu->get_frequency() * ( (1.0 +             // start bit
					    bits_per_byte +       // data bits
					    stop_bits  +          // stop bit(s)
					    use_parity)           //
					    /baud);
    time_per_bit = cpu->get_frequency() * ( 1.0/baud );
    if (verbose) {
      cout << "update_packet_time ==> 0x" << hex<< time_per_packet << "\n";
      cout << "time_per_bit ==> 0x" << hex<< time_per_bit << "\n";
    }
  }

  void se(guint64 t) {
    cout << "Search for event at t = 0x"<<t<<'\n';

    if(rx_event->get_state(t))
      cout << "high";
    else
      cout << "low";

    cout <<" at t = 0x"<<t << '\n';
  }
    

  virtual void callback(void) {
    if(0) {
      cout << "\n\n";
      cout << "ptyRCREG::" << __FUNCTION__ << "\n";
      cout << "\n\n";
    }

    //// process the data.....

    //rx_event->dump(-1);  // dump all events
    if (verbose)
      rx_event->dump_ASCII_art( time_per_bit/4, start_bit_index );  // time_per_packet/10,-1);

    guint64 current_time =  cycles.value;
    int edges = rx_event->get_edges(start_time, current_time);
#ifdef DEBUG
    cout << " gpsim time is " << current_time << " start time " << start_time << "\n";
    cout << " # of edges for one byte time " << edges << '\n';
#endif

    if(edges > (1+bits_per_byte))
      cout << "noisy\n";

    // Decipher the byte:

    if(!rx_event->get_state(current_time))
      cout << "no stop bit\n";

    switch(receive_state) {
    case RS_WAITING_FOR_START:
      if (verbose)
        cout << "waiting for start\n";
      break;
    case RS_RECEIVING:
      if(last_bit) {
	// The last edge was a rising one and if the baud
	// is correct it was also the last rising edge before
	// the stop bit. So process the event queue and 
	// decipher the byte from it...
	if (verbose)
	  cout << "Looks like we've definitely received a stop bit\n";
	receive_state = RS_WAITING_FOR_START;

	unsigned int b = decode_byte(start_bit_index, time_per_bit);
	if (verbose)
	  cout << "ptyRCREG: decoded to 0x" << b << "\n";
	if (b < 0x100) {
	  rxpin->usart->external_io->put_char(b);
	}
      } else {
	receive_state = RS_OVERRUN;
	cout << "Looks like we've overrun\n";
      }

      break;
    case RS_STOPPED:
      receive_state = RS_WAITING_FOR_START;
      cout << "received a stop bit\n";
      break;

    default:
      cout << "RX callback unknown receive_state " << receive_state << endl;
      break;
    }


  };


  void start(void) {

    start_time = cycles.value;

    receive_state = RS_RECEIVING;
    start_bit_time = start_time;
    start_bit_index = rx_event->get_index();
    //rx_event->log_time(start_bit_time);

    cycles.set_break(start_bit_time + time_per_packet, this);
    if (verbose) {
      cout << "ptyRCREG::start Setting Break\n";
      cout << "ptyRCREG::start   last_cycle = " << 
        hex << start_bit_time << " future_cycle = " <<
        start_bit_time + time_per_packet << '\n';
    }

  }

  // Insert a pulse into the array. small pulses are at the beginning.
  void add_pulse(guint64 pw) {

    guint64 scaled_pw = pw;
#ifdef DEBUG
    cout << " add_pulse\n";
#endif
    for(int i=0; i<64; i++) {

      if(pulses[i].width == scaled_pw) {
	pulses[i].sumofwidths += pw;
	pulses[i].occurs++;
#ifdef DEBUG
	cout << "incrementing pulse pw=" << pw << "  occurs="<< pulses[i].occurs<<"at index " << i <<'\n';
	cout << "average pw in bin " << (pulses[i].sumofwidths/pulses[i].occurs) << '\n';
#endif
	return;
      }

      if(pw < pulses[i].width) {
	// Insert into the array
	for(int j=63; j>i; j--) {
	  pulses[j].width = pulses[j-1].width;
	  pulses[j].occurs = pulses[j-1].occurs;
	  pulses[j].sumofwidths = pulses[j-1].sumofwidths;
	}
	pulses[i].width = scaled_pw;
	pulses[i].sumofwidths = pw;
	pulses[i].occurs = 1;

#ifdef DEBUG
	cout << "inserting pulse " << pw << "  at index " << i <<'\n';
#endif
	return;
      }
    }
    cout << " add_pulse did not add\n";

  }

  // remove a pulse from the array
  void del_pulse(guint64 pw) {

    guint64 scaled_pw = pw;

    for(int i=0; i<64; i++) {

      if(pulses[i].width == scaled_pw) {

#ifdef DEBUG
	cout << "deleting pulse " << pw << "  at index " << i << '\n';
#endif
	if(pulses[i].occurs > 1) {
	  pulses[i].occurs--;
	  pulses[i].sumofwidths -= pw;
	  return;
	}

	// Delete from the array
	for(int j=i; j<63; j++) {
	  pulses[j].width = pulses[j+1].width;
	  pulses[j].occurs = pulses[j+1].occurs;
	  pulses[j].sumofwidths = pulses[j+1].sumofwidths;
	}
	pulses[63].width = MAX_PW;
	pulses[63].occurs = 0;
	pulses[63].sumofwidths =0;

	return;
      }
    }
  }

  void dump_pulses(void) {
    for(int i=0; i<64; i++) {
      if(pulses[i].occurs)
	cout << "width 0x"<<hex<< pulses[i].width << "  avg width 0x"<< (pulses[i].sumofwidths/pulses[i].occurs)
	     << "  occurs 0x"<<pulses[i].occurs <<'\n';
      else
	return;
    }

  }

  unsigned int decode_byte(guint32 sindex, guint64 bit_time) {
  
    guint32 cur_index = rx_event->get_index();
    if(sindex & 1) {
      sindex = (sindex + 1) & (RX_MAX_EVENTS - 1);
      if(sindex == cur_index)
	return 0x400;
    }

    guint64 cur_time = cycles.value;
    guint64 t1 = rx_event->get_time(sindex) + bit_time + bit_time/2;

    guint32 index1 = rx_event->get_index(t1);
    guint32 index2 = (index1 + 1);
    bool decoding = 1;
    guint32 b = 0;

    if (verbose)
      cout << "decode_byte current time 0x"<<hex <<cur_time << " start bit time 0x"<< rx_event->get_time(sindex)<< " (sindex 0x" << sindex << ")\n";
    if(t1 >= cur_time) 
      return 0x800;

    int i = 0;

    while(i<8  && decoding) {
      b = (b>>1) | ((index1 & 0x0001) << 7);
#ifdef DEBUG
      cout << " time: 0x" << hex << t1 << " (0x" << rx_event->get_time(index1)
        << ") evt index: 0x" << index1 <<" b "<<b << '\n';
#endif
      t1 += bit_time;
      if(t1>=cur_time)
	decoding = 0;
      if((index1 != cur_index) && (t1 > rx_event->get_time(index2))) {
	index1 = index2;
	index2 = index2 + 1;
      }
      i++;
    }

    return b;



#if 0
    if(index2 & 1  && index2 != cur_index) {
      index2 = index2 + 1;
    }

    t1 = rx_event->get_time(index2) + bit_time + bit_time/2;
    if(t1>cur_time)
      decoding = 0;
    index1 = rx_event->get_index(t1);
    index2 = index1 + 1;
#endif

  }

  /*
      new_rx_edge(bool bit)

      This routine get's called when there's a change on the
      RX line. The time the edge occurred is stored into an
      event buffer.

  */

  void new_rx_edge(bool bit) {
    // If this bit is different from the last one we got
    // then save it in the event buffer.
    if (time_per_bit == 0)
      update_packet_time();

    if(bit ^ last_bit) {
      guint32 cur_index = rx_event->get_index();

      rx_event->event(bit);            // log the event
      if(cur_index != start_index)     // true after first edge
	add_pulse(rx_event->get_time(cur_index) - 
		  rx_event->get_time(cur_index - 1));

      // If we have received more than 64 edges, then start removing the
      // old pulses.

      if( ((cur_index - start_index) & (RX_MAX_EVENTS - 1)) > 63) {

	guint32 old_index = (start_index + 1) & (RX_MAX_EVENTS - 1);

	del_pulse(rx_event->get_time(old_index) - 
		  rx_event->get_time(start_index));

	start_index = old_index;
      }

      //dump_pulses();


      last_bit ^= 1;                 // change current state


	cur_index = rx_event->get_index();
	guint32 edges = (cur_index - start_bit_index) & (RX_MAX_EVENTS - 1);

	switch(receive_state) {
	case RS_WAITING_FOR_START:
	  if(!bit) {
	    // Looks like we just got a start bit.
	    start_bit_time = cycles.value;

            if (verbose)
	      cout  << "Start bit at t = 0x" << start_bit_time << '\n';

	    start();
	  }

	  break;
	case RS_RECEIVING:
	  if ((cycles.value - start_time) < (time_per_bit / 2)) {
            //too short for a start bit, probably a glitch. Go back waiting
	    //for start
            if (verbose)
              cout << "ignoring short start byte (" <<
                cycles.value - start_time << ")\n";
            receive_state = RS_WAITING_FOR_START;
          } else if(edges > 9) {
	    receive_state = RS_OVERRUN;
	    cout << "Looks like we've overrun (edges: " << edges << ")\n";
	  }

	  break;
	case RS_OVERRUN:
	  if(bit) {
	    cout << "Clearing overrun condition\n";
	    receive_state = RS_WAITING_FOR_START;
	  }
	  break;
	}


    }

  }


};

class ptyUSART_IOPORT : public IOPORT
{
public:
  ptyUSARTModule *usart;

  void trace_register_write(void) {
    if (verbose)
    	cout << "ptyUSART_IOPORT::trace_register_write\n";
  }

  ptyUSART_IOPORT (ptyUSARTModule *new_usart, unsigned int _num_iopins=4): IOPORT(_num_iopins) {
    usart = new_usart;
  }
};


//--------------------------------------------------------------
ptyUSART_CORE::ptyUSART_CORE(ptyUSART_IOPORT *new_iop)
{

  if (verbose)
    cout << "new ptyUSART_CORE\n";

  port = new_iop;

  external_io = new ptyUSART_external(this);

  rcreg=new ptyRCREG;
  txreg=new ptyTXREG;

}

//--------------------------------------------------------------
void ptyUSART_CORE::new_rx_edge(bool bit)
{
  if(rcreg)
    rcreg->new_rx_edge(bit);

}

//--------------------------------------------------------------
// callback to change the serial parameters from the pty
void ptyUSART_CORE::new_serial_parameters(int rx_baud, int tx_baud, int data_bits, bool use_parity, bool parity, int stop)
{
   cout << dec << "serial parameters: " << rx_baud << "/" << tx_baud <<
     " " << data_bits;
   if (use_parity)
     if (parity)
       cout << "o";
     else
       cout << "e";
   else
     cout << "n";
   cout << stop << endl;
   rcreg->new_serial_parameters(rx_baud, data_bits, use_parity, parity, stop);
   txreg->new_serial_parameters(tx_baud, data_bits, use_parity, parity, stop);
}

//--------------------------------------------------------------
// create_iopin_map 
//
//  This is where the information for the Module's package is defined.
// Specifically, the I/O pins of the module are created.

#define ptyUSART_PKG_TXPIN        1
#define ptyUSART_PKG_RXPIN        2
#define ptyUSART_PKG_CTSPIN       3
#define ptyUSART_PKG_RTXPIN       4

void ptyUSARTModule::create_iopin_map(void)
{


  // Create an I/O port to which the I/O pins can interface
  //   The module I/O pins are treated in a similar manner to
  //   the pic I/O pins. Each pin has a unique pin number that
  //   describes it's position on the physical package. This
  //   pin can then be logically grouped with other pins to define
  //   an I/O port. 

  port = new ptyUSART_IOPORT(this, 4);


  // Here, we name the port `pins'. So in gpsim, we will reference
  //   the bit positions as U1.pin0, U1.pin1, ..., where U1 is the
  //   name of the usart (which is assigned by the user and
  //   obtained with the name() member function call).

  char *pin_name = (char *)name().c_str();   // Get the name of this usart
  if(pin_name) 
    port->new_name(pin_name);
  else
    port->new_name("usart_port");

  // Create the usart core from the usart class in the main
  // gpsim source code.

  usart = new ptyUSART_CORE(port);

  // Define the physical package.
  //   The Package class, which is a parent of all of the modules,
  //   is responsible for allocating memory for the I/O pins.
  //
  //   ptyUSART I/O pins:
  //
  //    1 - Tx - Transmit
  //    2 - Rx - Receive
  //    3 - CTS - Clear To Send - NOTYET
  //    4 - RTS - Request To Send - NOTYET


  create_pkg(2);


  // Define the I/O pins and assign them to the package.
  //   There are two things happening here. First, there is
  //   a new I/O pin that is being created.The second thing is
  //   that the pins are "assigned" to the package. If we
  //   need to reference these newly created I/O pins (like
  //   below) then we can call the member function 'get_pin'.

  ptyUSART_TXPIN *txpin = new ptyUSART_TXPIN(usart,port, 0,"TX");
  ptyUSART_RXPIN *rxpin = new ptyUSART_RXPIN(usart,port, 1,"RX");
  assign_pin(1, txpin);
  assign_pin(2, rxpin);


  // Create an entry in the symbol table for the new I/O pins.
  // This is how the pins are accessed at the higher levels (like
  // in the CLI).

  symbol_table.add_stimulus(get_pin(1));
  symbol_table.add_stimulus(get_pin(2));


  // Complete the usart initialization

  if(usart->txreg) {
    usart->txreg->txpin = txpin; 
  }

  if(usart->rcreg) {
    usart->rcreg->rxpin = rxpin;
  }

}


//--------------------------------------------------------------

Module * ptyUSARTModule::ptyUSART_construct(const char *new_name)
{

  if (verbose)
    cout << "ptyUSART construct \n";

  ptyUSARTModule *um = new ptyUSARTModule(new_name);

  if(new_name) {
    um->new_name((char *)new_name);
    //um->res->put_name(new_name);
  }

  um->create_iopin_map();

  return um;

}

ptyUSARTModule::ptyUSARTModule(const char *_name)
{

  port = NULL;
  usart = NULL;

}

ptyUSARTModule::~ptyUSARTModule()
{
    // FIXME
}
