/*
 * Copyright (C) 2004 Rene Hexel. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice(s), this list of conditions and the following disclaimer as
 *    the first lines of this file unmodified other than the possible
 *    addition of one or more copyright notices.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice(s), this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. The name of the copyright holder may not be used to endorse or
 *    promote products derived from this software without specific
 *    prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*	$Id: semtest.c,v 1.1 2004/01/15 22:06:30 rh Exp $	*/

#include <sys/types.h>

#include <errno.h>
#include <pthread.h>
#include <semaphore.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>

/*
 *
 * simple POSIX semaphore testing program
 *
 */

#define P(s) sem_wait(&(s))
#define V(s) sem_post(&(s))

#define	MINFILL	0	/* minimum fill level */
#define	MAXFILL	2	/* maximum capacity */
#define	PRODDELAY 1	/* producer delay */
#define	CONSDELAY 2	/* consumer delay */

sem_t p, c, q;		/* three semaphores for testing */
pthread_t tid, ptid;	/* child an parent tid */


/*
 * cleanup()
 *	release allocated resources
 */
void cleanup(void)
{
	void *status;

	if (pthread_self() == tid)		/* child */
	{
		pthread_cancel(ptid);		/* cancel parent */
		pthread_join(ptid, &status);	/* wait for parent */
	}
	else
	{
		pthread_cancel(tid);		/* cancel child */
		pthread_join(tid, &status);	/* wait for child */
	}

	if (p) sem_destroy(&p);			/* release semaphores */
	if (c) sem_destroy(&c);
	if (q) sem_destroy(&q);

	p = c = q = NULL;
}


/*
 * prodcons()
 * 	simple producer/consumer function called by parent and child
 *
 * parameters:
 *	msg	the message to print within critical region
 *	grab	the semaphore to grab before entering the critical region
 *	release	the semaphore to release after leaving the critical region
 *	delay	the number of seconds to wait
 */
void prodcons(const char *msg, sem_t grab, sem_t release, int delay)
{
	int i;					/* current counter */

	for (i = 0; i < 10 ; i++)
	{
		P(grab);			/* grab our semaphore */
		P(q);				/* protect prinf() */
		printf("%s %d\n", msg, i);	/* print out message */
		V(q);				/* leave critical region */
		V(release);			/* release other semaphore */
		if (delay > 0)
			sleep(delay);		/* wait a while */
	}
}

/*
 * child()
 *	child process run by pthread_create().  This simulates the producer
 */
void *child(void *arg)
{
	/*
	 * call prodcons() as the producer
	 */
	prodcons("produced element", p, c, PRODDELAY);

	return arg;				/* never reached */
}


/*
 * main()
 *	set up a 3 semaphores to protect a virtual resource with
 *	minimum fill level MINFILL and maximum fill level MAXFILL
 */
int main(int argc, char *argv[])
{
	int error = EXIT_FAILURE;		/* exit code */

	atexit(cleanup);			/* release semaphores on exit */
	signal(SIGTERM, exit);			/* make sure we clean up */
	signal(SIGINT, exit);

	if (sem_init(&p, 0, MAXFILL)  ||
	    sem_init(&c, 0, -MINFILL) ||
	    sem_init(&q, 0, 1))			/* error getting semaphores? */
	{
		perror("getting semaphores failed");
		exit(error);
	}

	ptid = pthread_self();			/* parent thread ID */
	if (pthread_create (&tid, NULL, child, NULL) != 0)
	{					/* creating child failed */
		perror("pthread_create() failed");
		exit(error);
	}

	/*
	 * call prodcons() as the consumer
	 */
	prodcons("consumed element", c, p, CONSDELAY);

	cleanup();

	return EXIT_SUCCESS;
}
