/* plugin.c
 * Copyright (C) 2001 QT4Linux and OpenQuicktime Teams
 *
 * This file is part of OpenQuicktime, a free QuickTime library.
 *
 * Based on QT4Linux by Adam Williams.
 *
 * OpenQuicktime is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * OpenQuicktime is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef WIN32
#include <dirent.h>
#include <dlfcn.h>
#else
#include <windows.h>
static char* dlerror() { return "DLL Error";};
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "funcprotos.h"
#include "openquicktime.h"

static int total_vcodecs = 0;
static int total_acodecs = 0;
static quicktime_extern_video_t *vcodecs = NULL; 
static quicktime_extern_audio_t *acodecs = NULL; 


#define CODEC_PREFIX "quicktime_codec_"


typedef struct {
  int chunk_number;
  unsigned int chunk_size;
  char *chunk_buffer;
  int nimportenawaq;
} ChunkList;

static ChunkList* chunkList;
static unsigned int chunkListSize = 0 ;

static void initChunkList(int size) {
  int i;

  //printf("Chunk List INIT\n");
  chunkListSize = size;
  chunkList = (ChunkList*)malloc(chunkListSize * sizeof(ChunkList));
  for(i=0; i<chunkListSize; i++) {
    chunkList[i].chunk_number = -1;
    chunkList[i].chunk_size = 0;
    chunkList[i].chunk_buffer = NULL;
  }
}



static int set_audio_param_external(quicktime_t *file, 
                                    int track, 
	                            const char* param, 
                                    const void* data)
{
  int index = 0;
  char *compressor;

  compressor = quicktime_audio_compressor(file, track);
  index=quicktime_find_acodec(compressor);

  if(index>=0) {
     return  acodecs[index].set_param(file,track,param,data);
  }

  fprintf(stderr, 
          "set_audio_param_external: Can't find the corresponding codec: %s\n",
          compressor);
  return -1;
}

static int get_audio_param_external(quicktime_t *file, 
                                    int track, 
	                            const char* param, 
                                    void* data)
{
  int index = 0;
  char *compressor;

  compressor = quicktime_audio_compressor(file, track);
  index=quicktime_find_acodec(compressor);

  if(index>=0) {
     return  acodecs[index].get_param(file,track,param,data);
  }

  fprintf(stderr, 
          "get_audio_param_external: Can't find the corresponding codec: %s\n",
          compressor);
  return -1;
}



static int decode_chunk_external(quicktime_t *file, int track, long chunk, int channel, quicktime_extern_audio_t *codec)
{
  int result = 0;
  int i, j, ret, size = 0;
  long chunk_samples, chunk_bytes, chunk_bytes1, chunk_bytes2;
  unsigned char *chunk_ptr, *block_ptr;
  quicktime_trak_t *trak = file->atracks[track].track;
  longest offset, chunk_offset;
  ChunkList *selectedChunk;
  
  //  printf("decode_chunk_external chunk %d\n", chunk);
 
  // Get the byte count to read. 
  chunk_samples = quicktime_chunk_samples(trak, chunk);

  selectedChunk = &(chunkList[chunk%chunkListSize]);

  if(chunk == selectedChunk->chunk_number) {
    //printf("Using old chunk %u instead\n", chunkList[chunk%chunkListSize].chunk_number);
    codec->chunk = chunk;
    codec->work_size = selectedChunk->chunk_size;
    codec->work_buffer = selectedChunk->chunk_buffer;
    return 0;
  } else {
    selectedChunk->chunk_number = chunk;
  }

  // Really decompress the chunk now

  // Difference between the 2 sound chunk offsets 
  chunk_bytes1 =  quicktime_chunk_to_offset(file->atracks[track].track, chunk+1) -  quicktime_chunk_to_offset(file->atracks[track].track, chunk);

  // Calculating the offset of the next video chunk
  if(file->total_vtracks) {
      chunk_bytes2 = quicktime_offset_to_chunk(&offset, file->vtracks[0].track, quicktime_chunk_to_offset(file->atracks[track].track, chunk));

      if(chunk_bytes2 != 1)
	chunk_bytes2++;

      chunk_bytes2 = quicktime_chunk_to_offset(file->vtracks[0].track, chunk_bytes2) - quicktime_chunk_to_offset(file->atracks[track].track, chunk);

  } else
    chunk_bytes2 = 1000000000;
  //printf(" chunk_bytes1 %d,  chunk_bytes2 %d\n",  chunk_bytes1,  chunk_bytes2);

  // We choose the smallest positive one
  if(chunk_bytes2 > chunk_bytes1 || chunk_bytes2<=0)
    chunk_bytes = chunk_bytes1;
  else
    chunk_bytes = chunk_bytes2;

  //  chunk_bytes = codec->sample_to_bytes(chunk_samples, file->atracks[track].channels);

  // Get the buffer to read into. 
  if(selectedChunk->chunk_buffer && selectedChunk->chunk_size < chunk_samples*file->atracks[track].channels*2)
    {
      //printf("freeing selectedChunk->chunk_buffer\n");
      free(selectedChunk->chunk_buffer);
      selectedChunk->chunk_buffer = NULL;
      selectedChunk->chunk_size = 0;
    }
  
  if(!selectedChunk->chunk_buffer)
    {
      //printf("mallocing selectedChunk->chunk_buffer\n");
      selectedChunk->chunk_size = chunk_samples*file->atracks[track].channels*2;
      selectedChunk->chunk_buffer = (char *)malloc(selectedChunk->chunk_size);
    }

  if(codec->read_buffer && codec->read_size < chunk_bytes)
    {
      //printf("freeing read_buffer\n");
      free(codec->read_buffer);
      codec->read_buffer = NULL;
    }

  if(!codec->read_buffer)
    {
      //printf("mallocinging read_buffer chunk_bytes %d\n", chunk_bytes);
      codec->read_size = chunk_bytes;
      codec->read_buffer = (char *)malloc(codec->read_size);
    }

  //  printf("before quicktime_read_chunk\n");
 
  result = quicktime_read_chunk(file, (char *)codec->read_buffer, track, chunk, 0, chunk_bytes);
	
  // Now decode the chunk, one block at a time, until the total samples in the chunk 
  // is reached.

  if(!result)
    { 

      //printf("chunk %d ", chunk);
      codec->decode(file, track, chunk_bytes, (unsigned char*)codec->read_buffer, selectedChunk->chunk_size, (unsigned char*)selectedChunk->chunk_buffer);
      //printf("OK for %d\n", chunk);
    }
  codec->work_buffer = selectedChunk->chunk_buffer;
  codec->work_size = selectedChunk->chunk_size;
  codec->chunk = selectedChunk->chunk_number;
  codec->buffer_channel = channel;
  
  return result;
}



static int decode_audio_external(quicktime_t *file, 
				int16_t *output_i, 
				float *output_f, 
				long samples, 
				int track,
				int channel)
{
  int			index = 0;
  quicktime_audio_map_t *atrack = &(file->atracks[track]);
  char                  *compressor;
  quicktime_extern_audio_t *codec;
  int result = 0;
  longest chunk_sample, chunk;
  unsigned int chunk_bytes, chunk_samples;
  unsigned long int i, chunk_start, chunk_end;
  quicktime_trak_t *trak = file->atracks[track].track;

  //printf("\nStarting to decode %d samples\n", samples);

  if(chunkListSize == 0)
    // Lets keep the 64 last decoded chunks
    initChunkList(64); 
  
  compressor = quicktime_audio_compressor(file, track);
  index=quicktime_find_acodec(compressor);
  codec = &(acodecs[index]); 
 
  /* Get the first chunk with this routine and then increase the chunk number. */
  quicktime_chunk_of_sample(&chunk_sample, 
			    &chunk, trak, 
			    file->atracks[track].current_position);

  //  printf("DECODE AUDIO EXTERNAL %u CHANNEL %u CHUNK %u\n", samples, channel, chunk);

  /* Read chunks and extract ranges of samples until the output is full. */
  for(i = 0; i < samples && !result; )
    {
      /* Get chunk we're on. */
      chunk_samples = quicktime_chunk_samples(trak, chunk);
      
      /*      if(!codec->work_buffer /*||	 /*codec->chunk != chunk ||
			       codec->buffer_channel != channel)
			       {*/
	  /* read a new chunk if necessary */
	  result = decode_chunk_external(file, track, chunk, channel, codec);
	  //	}
     
      /* Get boundaries from the chunk */
      chunk_start = 0;
      if(chunk_sample < file->atracks[track].current_position)
	chunk_start = file->atracks[track].current_position - chunk_sample;
      
      chunk_end = chunk_samples;
      if(chunk_sample + chunk_end > file->atracks[track].current_position + samples)
	chunk_end = file->atracks[track].current_position + samples - chunk_sample;      
      /* Read from the chunk */
      if(output_i)
	{
	  while(chunk_start < chunk_end)
	    {
	      output_i[i++] = ((int16_t*)codec->work_buffer)[chunk_start*file->atracks[track].channels+channel];
	      chunk_start++;
	    }
	}
      else
	if(output_f)
	  {
	    while(chunk_start < chunk_end)
	      {
		output_f[i++] = (float)((int16_t*)codec->work_buffer)[chunk_start*file->atracks[track].channels+channel] / 32767;
		chunk_start++;
	      }
	  }      
      chunk++;
      chunk_sample += chunk_samples;
    }
  
  //printf("OKOK \n");

  return result;




}



/*  } */

static int encode_audio_external(quicktime_t *file, 
				int16_t **input_i, 
				float **input_f, 
				int track, 
				long samples)
{
  int index = 0;
  quicktime_audio_map_t *atrack = &(file->atracks[track]);
  char *compressor;
  quicktime_extern_audio_t *codec;
  long step, i, j, result;
  int16_t *input_ptr; 
  longest offset, chunk_bytes;
  static longest samplecounter=0; 
  char *SoundPtr;

  compressor = quicktime_audio_compressor(file, track);
  index=quicktime_find_acodec(compressor);
  codec=&(acodecs[index]);

  /* Get buffer sizes */
  if(codec->work_buffer && codec->work_size < samples * atrack->channels * 2)
    {
      /* Create new buffer */
      free(codec->work_buffer);
      codec->work_size = samples * atrack->channels * 2;
      codec->work_buffer = (char *)malloc(codec->work_size);
    }
  else
    if(!codec->work_buffer)
      {
	codec->work_size = samples * atrack->channels * 2;
	codec->work_buffer = (char *)malloc(codec->work_size);
      }

  /* Get output size */
  /* lame_encode_buffer says the outbut buffer for n input samples should
     be 1.25*n+7200, and here codec->read_buffer will always contain 
     one encoded frame = 1152 bytes, so this should be enough */
  chunk_bytes = 2*1152+7200; 
  if(codec->read_buffer && codec->read_size < chunk_bytes)
    {
      free(codec->read_buffer);
      codec->read_buffer = 0;
    }
  if(!codec->read_buffer)
    {
      codec->read_size = chunk_bytes;
      //printf("Allocating %i bytes for the read_buffer\n", codec->read_size);
      codec->read_buffer = (char *)malloc(codec->read_size);
    }

  /* Arm the input buffer after the last overflow */
  step = atrack->channels;

  for(j = 0; j < atrack->channels; j++)
    {
      input_ptr = ((int16_t *)codec->work_buffer) + j;
      
      if(input_i)
	{
	  for(i = 0; i < samples; i++)
	    {
	      *input_ptr = input_i[j][i];
	      input_ptr += step;
	    }
	}
      else
	if(input_f)
	  {
	    for(i = 0; i < samples; i++)
	      {
		*input_ptr = (int16_t)(input_f[j][i] * 32767);
		input_ptr += step;
	      }
	  }
    }
  
  SoundPtr = codec->work_buffer;

  for(i = 0; i< samples; i+=1152) {
    /* Call the external encoding function */
    chunk_bytes = acodecs[index].encode
      (file, 
       track,
       ((i+1152>=samples)?samples-i:1152)
       * atrack->channels * 2, 
       (unsigned char*)SoundPtr+i*atrack->channels * 2, 
       (unsigned char*)codec->read_buffer);

    //printf("encoding %li-%li from %li => chunk_bytes = %lli\n", 
    //   i, i+((i+1152>=samples)?samples-i:1152), samples, chunk_bytes);

    /* Writing complex sample */
    if (chunk_bytes==0) break;

    offset = quicktime_position(file);
    result = file->quicktime_write_data(file,
				  (char*)codec->read_buffer,
				  chunk_bytes);
    
    if(result) result = 0; else result = 1; /* defeat fwrite's return */
    
    quicktime_update_tables(file,
			    atrack->track, 
			    offset, 
			    atrack->current_chunk++, 
			    samplecounter++,
			    1,
			    chunk_bytes);
    
  }

  return chunk_bytes;
}





static int set_video_param_external(quicktime_t *file, 
                                    int track, 
	                            const char* param, 
                                    const void* data)
{
  int index = 0;
  char *compressor;

  compressor = quicktime_video_compressor(file, track);
  index=quicktime_find_vcodec(compressor);

  if(index>=0) {
     return  vcodecs[index].set_param(file,track,param,data);
  }

  fprintf(stderr, 
          "set_video_param_external: Can't find the corresponding codec: %s\n",
          compressor);
  return -1;
}

static int get_video_param_external(quicktime_t *file, 
                                    int track, 
	                            const char* param, 
                                    void* data)
{
  int index = 0;
  char *compressor;

  compressor = quicktime_video_compressor(file, track);
  index=quicktime_find_vcodec(compressor);

  if(index>=0) {
     return  vcodecs[index].get_param(file,track,param,data);
  }

  fprintf(stderr, 
          "get_video_param_external: Can't find the corresponding codec: %s\n",
          compressor);
  return -1;
}


static int decode_video_external(quicktime_t *file, 
				unsigned char **row_pointers, 
				int track)
{
  unsigned char		*input;
  //  unsigned char		*output=row_pointers[0]; 
  int			index = 0;
  int                   error = -1;
  quicktime_video_map_t *vtrack = &(file->vtracks[track]);
  unsigned int          bytes;
  char                  *compressor;
  int i=0;

  compressor = quicktime_video_compressor(file, track);
  index=quicktime_find_vcodec(compressor);

  if(index>=0){

    if(vtrack->current_position == -1)
      bytes = quicktime_frame_size(file, 0, track);
    else
      bytes = quicktime_frame_size(file, vtrack->current_position, track);

    if(bytes<=0) {
      fprintf(stderr, "Decode_video: frame size equal %u\n", bytes);
      return -1;
    }

    quicktime_set_video_position(file, vtrack->current_position, track);

    input=(unsigned char*)malloc(bytes);
    if(input) {

      //      printf("Position in the file: %lli Frame Number %li\n", file->file_position, vtrack->current_position);

      if(file->quicktime_read_data(file,(char*)input, bytes))
	{ 
	  error = vcodecs[index].decode(file, track, bytes, input, row_pointers/*output*/); 	  
	}
      else 
	fprintf(stderr, "Decode_video : can't read data from file\n");

    } else 
	fprintf(stderr, "Decode_video : Can't allocate decoding buffer");

    free(input);

  } else {
    fprintf(stderr, 
	    "Decode_video : Can't find the corresponding codec: ",
	    quicktime_video_compressor(file, track) );
  }

  //    file->vtracks[track].current_position++;

    return error;
}




static int encode_video_external(quicktime_t *file, 
				 unsigned char **row_pointers, 
				 int track)
{
  //  unsigned char *input = row_pointers[0];
  unsigned char *output; 
  int index = 0;
  int error = -1;
  quicktime_video_map_t *vtrack = &(file->vtracks[track]);
  unsigned int bytes;
  long offset = quicktime_position(file);
  char *compressor;
  short width, height, depth;
  int IsAKeyFrame;
  
  compressor = quicktime_video_compressor(file, track);
  index = quicktime_find_vcodec(compressor);
  if(index >= 0)
    {
      /* Initializing some local variables */
      width  = vtrack->track->tkhd.track_width;
      height = vtrack->track->tkhd.track_height;
      depth  = file->vtracks[track].track->mdia.minf.stbl.stsd.table[0].depth;
      bytes  = width * height * depth / 8;
	  
      output = (unsigned char*)malloc(bytes);
      if(output) 
	{
	  bytes = vcodecs[index].encode(file, track, row_pointers, output, &IsAKeyFrame); 
	  
	  if(bytes > 0)
	    {
	      //	      printf("Writing %u bytes\n", bytes);
	      
	      error = !file->quicktime_write_data(file, (char*)output, bytes);
	      
	      quicktime_update_tables(file,
				      file->vtracks[track].track,
				      offset,
				      file->vtracks[track].current_chunk,
				      file->vtracks[track].current_position,
				      1,
				      bytes);
	      if( IsAKeyFrame ) {
		//		printf("Seems that frame %li is a Keyframe ...\n", file->vtracks[track].current_chunk);
		quicktime_insert_keyframe(file, file->vtracks[track].current_chunk, track);
	      }

	      file->vtracks[track].current_chunk++;
	    } 
	  else 
	    fprintf(stderr, "encode_video_external: Error in external encoding function\n");
	  
	  free(output);
	} 
      else 
	fprintf(stderr, "encode_video_external: Can't allocate encoding buffer");
    } 
  else 
    fprintf(stderr, 
	    "encode_video_external: Can't find the corresponding codec: ",
	    quicktime_video_compressor(file, track) );
  
  return error;    
}



int quicktime_vcodec_size()
{
	return total_vcodecs;
}

int quicktime_acodec_size()
{
	return total_acodecs;
}

int quicktime_find_vcodec(char *fourcc)
//int quicktime_vcodec_index(char *fourcc)
{
  	int i;

  	for(i = 0; i < total_vcodecs; i++)
  	{
    	if(quicktime_match_32(fourcc, vcodecs[i].fourcc))
      		return i;
  	}
  	return -1;
}

int quicktime_find_acodec(char *fourcc)
//int quicktime_acodec_index(char *fourcc)
{
	int i;

	for(i = 0; i < total_acodecs; i++)
	{
    	if(quicktime_match_32(fourcc, acodecs[i].fourcc))
    		return i;
	}
	return -1;
}


int quicktime_register_vcodec(char *fourcc, 
	void *(*init_vcodec)(quicktime_video_map_t *))
{
	int index = quicktime_find_vcodec(fourcc);

	if(index == -1)
	{
    	total_vcodecs++;
    	vcodecs = (quicktime_extern_video_t *)realloc(vcodecs,
	    	total_vcodecs * sizeof(quicktime_extern_video_t));

    	vcodecs[total_vcodecs - 1].init = init_vcodec;
    	quicktime_copy_char32(vcodecs[total_vcodecs - 1].fourcc, fourcc);
    	return total_vcodecs - 1;
	}
	return index;
}

int quicktime_register_acodec(char *fourcc, 
	void *(*init_acodec)(quicktime_audio_map_t *))
{
	int index = quicktime_find_acodec(fourcc);

	if(index == -1)
	{
    	total_acodecs++;
    	acodecs = (quicktime_extern_audio_t *)realloc(acodecs,
			total_acodecs * sizeof(quicktime_extern_audio_t));
    	acodecs[total_acodecs - 1].init = init_acodec;
    	quicktime_copy_char32(acodecs[total_acodecs - 1].fourcc, fourcc);
    	return total_acodecs - 1;
	}
	return index;
}


#ifndef WIN32

static int select_codec(const struct dirent *ptr)
{
	if(strncmp(ptr->d_name, CODEC_PREFIX, 15) == 0)
    	return 1;
	else 
    	return 0;
}

#endif


int quicktime_delete_external_vcodec(quicktime_video_map_t *vtrack)
{
  char *compressor = vtrack->track->mdia.minf.stbl.stsd.table[0].format;
  int i, newtotal_vcodecs=0;
  int index = quicktime_find_vcodec(compressor);

  quicktime_extern_video_t *newvcodecs = NULL;
      
  // printf("quicktime_delete_external_vcodec\n");

  if( index > -1 )
    {

      int usecounter = vcodecs[index].delete_codec(vtrack);

      printf("Compressor %s, usecounter %d\n", compressor, usecounter);
          
      // Now really dettaching the librarie
      if(usecounter == 0) {
	newtotal_vcodecs = total_vcodecs-1;
	newvcodecs = (quicktime_extern_video_t *) 
	  realloc(
		  newvcodecs,
		  newtotal_vcodecs*sizeof(quicktime_extern_video_t));
	if(newvcodecs==NULL)
	  printf("Pas bon du gros NULL\n");
	
	for(i=0; i < total_vcodecs; i++)
	  {
	    if( i < index )
	      {
		newvcodecs[i].codec.delete_vcodec = vcodecs[i].codec.delete_vcodec ;
		newvcodecs[i].codec.decode_video = vcodecs[i].codec.decode_video;
		newvcodecs[i].codec.encode_video = vcodecs[i].codec.encode_video;
		newvcodecs[i].codec.delete_acodec = vcodecs[i].codec.delete_acodec ;
		newvcodecs[i].codec.decode_audio = vcodecs[i].codec.decode_audio;
		newvcodecs[i].codec.encode_audio = vcodecs[i].codec.encode_audio;
		newvcodecs[i].codec.reads_colormodel = vcodecs[i].codec.reads_colormodel;
		newvcodecs[i].codec.writes_colormodel = vcodecs[i].codec.writes_colormodel;
		newvcodecs[i].codec.priv = vcodecs[i].codec.priv ;
		
		newvcodecs[i].init = vcodecs[i].init;
		newvcodecs[i].decode = vcodecs[i].decode;
		newvcodecs[i].encode = vcodecs[i].encode;
		newvcodecs[i].delete_codec = vcodecs[i].delete_codec;
		newvcodecs[i].handle = vcodecs[i].handle;	      
		newvcodecs[i].codec.set_param = vcodecs[i].codec.set_param;
		newvcodecs[i].codec.get_param = vcodecs[i].codec.get_param;
		strncpy(newvcodecs[i].fourcc, vcodecs[i].fourcc, 5);
	      }
	    if( i > index )
	      {
		newvcodecs[i-1].codec.delete_vcodec = vcodecs[i].codec.delete_vcodec ;
		newvcodecs[i-1].codec.decode_video = vcodecs[i].codec.decode_video;
		newvcodecs[i-1].codec.encode_video = vcodecs[i].codec.encode_video;
		newvcodecs[i-1].codec.delete_acodec = vcodecs[i].codec.delete_acodec ;
		newvcodecs[i-1].codec.decode_audio = vcodecs[i].codec.decode_audio;
		newvcodecs[i-1].codec.encode_audio = vcodecs[i].codec.encode_audio;
		newvcodecs[i-1].codec.reads_colormodel = vcodecs[i].codec.reads_colormodel;
		newvcodecs[i-1].codec.writes_colormodel = vcodecs[i].codec.writes_colormodel;
		newvcodecs[i-1].codec.priv =vcodecs[i].codec.priv ;
		
		newvcodecs[i-1].init = vcodecs[i].init;
		newvcodecs[i-1].decode = vcodecs[i].decode;
		newvcodecs[i-1].encode = vcodecs[i].encode;
		newvcodecs[i-1].delete_codec = vcodecs[i].delete_codec;
		newvcodecs[i-1].handle = vcodecs[i].handle;	      
		newvcodecs[i-1].set_param = vcodecs[i].set_param;
		newvcodecs[i-1].get_param = vcodecs[i].get_param;      
		strncpy(newvcodecs[i-1].fourcc, vcodecs[i].fourcc, 5);
		
	      }
	  }
	
	//      printf("delete realloc %p size %\n", vcodecs, total_vcodecs);
	
#ifndef WIN32
	dlclose(vcodecs[index].handle);
#else
    FreeLibrary(vcodecs[index].handle);
#endif
	free(vcodecs);
	vcodecs = newvcodecs;
	total_vcodecs = newtotal_vcodecs;
	//printf("A LA FIN DE DELETE vcodecs %u total_vcodecd %i\n", vcodecs, total_vcodecs);
	
      }
    }
}


int quicktime_delete_external_acodec(quicktime_audio_map_t *atrack)
{
  char *compressor = atrack->track->mdia.minf.stbl.stsd.table[0].format;
  int i, newtotal_acodecs=0;
  int index = quicktime_find_acodec(compressor);

  quicktime_extern_audio_t *newacodecs = NULL;

  if( index > -1 )
    {
      int usecounter = acodecs[index].delete_codec(atrack);

      printf("Compressor %s, usecounter %d\n", compressor, usecounter);
          
      // Now really dettaching the librarie
      if(usecounter == 0) {
	if(acodecs[index].work_buffer)
	  free(acodecs[index].work_buffer);
	
	if(acodecs[index].read_buffer)
	  free(acodecs[index].read_buffer);
	
	printf("Compressor %s, index %i, acodecs %p atrack %p\n", compressor, index, acodecs, atrack);
	
	newtotal_acodecs = total_acodecs-1;
	newacodecs = (quicktime_extern_audio_t *)
	  realloc(
		  newacodecs,
		  newtotal_acodecs*sizeof(quicktime_extern_audio_t));
	if(newacodecs==NULL)
	  printf("Pas bon du gros NULL\n");
	
	for(i=0; i < total_acodecs; i++)
	  {
	    if( i < index )
	      {
		newacodecs[i].codec.delete_vcodec = acodecs[i].codec.delete_vcodec ;
		newacodecs[i].codec.decode_video = acodecs[i].codec.decode_video;
		newacodecs[i].codec.encode_video = acodecs[i].codec.encode_video;
		newacodecs[i].codec.delete_acodec = acodecs[i].codec.delete_acodec ;
		newacodecs[i].codec.decode_audio = acodecs[i].codec.decode_audio;
		newacodecs[i].codec.encode_audio = acodecs[i].codec.encode_audio;
		newacodecs[i].codec.reads_colormodel = acodecs[i].codec.reads_colormodel;
		newacodecs[i].codec.writes_colormodel = acodecs[i].codec.writes_colormodel;
		newacodecs[i].codec.priv =acodecs[i].codec.priv ;
		
		newacodecs[i].init = acodecs[i].init;
		newacodecs[i].decode = acodecs[i].decode;
		newacodecs[i].encode = acodecs[i].encode;
		newacodecs[i].delete_codec = acodecs[i].delete_codec;
		newacodecs[i].handle = acodecs[i].handle;	      
		newacodecs[i].codec.set_param = acodecs[i].codec.set_param;
		newacodecs[i].codec.get_param = acodecs[i].codec.get_param;
		strncpy(newacodecs[i].fourcc, acodecs[i].fourcc, 5);
	      }
	    if( i > index )
	      {
		newacodecs[i-1].codec.delete_vcodec = acodecs[i].codec.delete_vcodec ;
		newacodecs[i-1].codec.decode_video = acodecs[i].codec.decode_video;
		newacodecs[i-1].codec.encode_video = acodecs[i].codec.encode_video;
		newacodecs[i-1].codec.delete_acodec = acodecs[i].codec.delete_acodec ;
		newacodecs[i-1].codec.decode_audio = acodecs[i].codec.decode_audio;
		newacodecs[i-1].codec.encode_audio = acodecs[i].codec.encode_audio;
		newacodecs[i-1].codec.reads_colormodel = acodecs[i].codec.reads_colormodel;
		newacodecs[i-1].codec.writes_colormodel = acodecs[i].codec.writes_colormodel;
		newacodecs[i-1].codec.priv =acodecs[i].codec.priv ;
		
		newacodecs[i-1].init = acodecs[i].init;
		newacodecs[i-1].decode = acodecs[i].decode;
		newacodecs[i-1].encode = acodecs[i].encode;
		newacodecs[i-1].delete_codec = acodecs[i].delete_codec;
		newacodecs[i-1].handle = acodecs[i].handle;	      
		newacodecs[i-1].set_param = acodecs[i].set_param;
		newacodecs[i-1].get_param = acodecs[i].get_param;      
		strncpy(newacodecs[i-1].fourcc, acodecs[i].fourcc, 5);
		
	      }
	  }
#ifndef WIN32
	dlclose(acodecs[index].handle);
#else
    FreeLibrary(acodecs[index].handle);
#endif
	free(acodecs);
	acodecs = newacodecs;
	total_acodecs = newtotal_acodecs;
      }
    }
}


static int writes_codec_colormodel(quicktime_t *file, 
				 int colormodel, 
				 int track)
{
  quicktime_video_map_t *vtrack = &(file->vtracks[track]);
  unsigned int          bytes;
  char                  *compressor;
  int index;

  compressor = quicktime_video_compressor(file, track);
  index=quicktime_find_vcodec(compressor);

  return vcodecs[index].writes_colormodel(file, 
				   colormodel, 
				   track);

}



int quicktime_register_external_vcodec(const char *codec_name)
{
  	void *handle;
	int (*quicktime_codec_register)(quicktime_extern_video_t*);
	char path[1024];
	char *error;

	sprintf(path, "%s%s.so", CODEC_PREFIX, codec_name);
	//	fprintf(stderr, "Trying to load external codec %s\n", path);

#ifndef WIN32
	handle = dlopen(path, RTLD_NOW);
#else
    handle = LoadLibrary(path);
#endif
	//	fprintf(stderr, "After dlopen %s\n", path);
	if(!handle)
	{
    	fprintf(stderr, "Can't load the codec\n");
    	fprintf(stderr, "%s\n", dlerror());
    	return -1;
	}
	fprintf(stderr, "External codec %s loaded\n", path);

#ifndef WIN32
    quicktime_codec_register = (int(*)(quicktime_extern_video_t*))dlsym(handle, "quicktime_codec_register");
#else
    quicktime_codec_register = (int(*)(quicktime_extern_video_t*))GetProcAddress(handle, "quicktime_codec_register");
#endif
    if((error = dlerror()) != NULL)  
	{
	  fprintf(stderr, "%s\n",error);
	  return -1;
	}

	total_vcodecs++;
	vcodecs = (quicktime_extern_video_t *)realloc(vcodecs,
		total_vcodecs*sizeof(quicktime_extern_video_t));

	if((*quicktime_codec_register)(&(vcodecs[total_vcodecs - 1]))) 
	{
	  //	printf("register external realloc %p size %i\n", vcodecs, total_vcodecs);

    	vcodecs[total_vcodecs - 1].codec.delete_vcodec = quicktime_delete_external_vcodec;

/*vcodecs[total_vcodecs - 1].delete_codec;*/
    	vcodecs[total_vcodecs - 1].codec.decode_video = decode_video_external;
    	vcodecs[total_vcodecs - 1].codec.encode_video = encode_video_external;
    	vcodecs[total_vcodecs - 1].codec.set_param = set_video_param_external;
    	vcodecs[total_vcodecs - 1].codec.get_param = get_video_param_external;
	vcodecs[total_vcodecs - 1].handle = handle;
	vcodecs[total_vcodecs - 1].codec.reads_colormodel = 
	  vcodecs[total_vcodecs - 1].reads_colormodel;
	vcodecs[total_vcodecs - 1].codec.writes_colormodel = writes_codec_colormodel;

    	return total_vcodecs - 1;
	} 
	else
        return -1;
}



int quicktime_register_external_acodec(const char *codec_name)
{
  void *handle;
  int (*quicktime_codec_register)(quicktime_extern_audio_t*);
  char path[1024];
  char *error;
  
  sprintf(path, "%s%s.so", CODEC_PREFIX, codec_name);
  fprintf(stderr, "Trying to load external codec %s\n", path);

#ifndef WIN32
  handle = dlopen(path, RTLD_NOW);
#else
  handle = LoadLibrary(path);
#endif
  fprintf(stderr, "After dlopen %s\n", path);
  if(!handle)
    {
      fprintf(stderr, "Can't load the codec\n");
      fprintf(stderr, "%s\n", dlerror());
      return -1;
    }
  fprintf(stderr, "External codec %s loaded\n", path);
  
#ifndef WIN32
  quicktime_codec_register = (int(*)(quicktime_extern_audio_t*))dlsym(handle, "quicktime_codec_register");
#else
  quicktime_codec_register = (int(*)(quicktime_extern_audio_t*))GetProcAddress(handle, "quicktime_codec_register");
#endif
  if((error = dlerror()) != NULL)  
    {
      fprintf(stderr, "%s\n",error);
      return -1;
    }
  
  total_acodecs++;
  acodecs = (quicktime_extern_audio_t *)realloc(acodecs,
						total_acodecs*sizeof(quicktime_extern_audio_t));
  
  if((*quicktime_codec_register)(&(acodecs[total_acodecs-1])))
    {
      //            printf("adding intermediate functions: %i\n", total_acodecs);
      
      acodecs[total_acodecs - 1].codec.delete_acodec = quicktime_delete_external_acodec;
      acodecs[total_acodecs - 1].codec.decode_audio = decode_audio_external;
      acodecs[total_acodecs - 1].codec.encode_audio = encode_audio_external;
      acodecs[total_acodecs - 1].codec.set_param = set_audio_param_external;
      acodecs[total_acodecs - 1].codec.get_param = get_audio_param_external;
      acodecs[total_acodecs - 1].handle = handle;
 
      acodecs[total_acodecs - 1].work_buffer=NULL;
      acodecs[total_acodecs - 1].work_size=0;
      acodecs[total_acodecs - 1].read_buffer=NULL;
      acodecs[total_acodecs - 1].read_size=0;
      acodecs[total_acodecs - 1].chunk=0;
      acodecs[total_acodecs - 1].buffer_channel=0;
     
      return total_acodecs - 1;
    } 
  else
    return -1;
}


int quicktime_init_vcodec_core(int index, quicktime_video_map_t *vtrack)
{
	((quicktime_codec_t*)vtrack->codec)->delete_vcodec = vcodecs[index].codec.delete_vcodec;
	((quicktime_codec_t*)vtrack->codec)->decode_video = vcodecs[index].codec.decode_video;
	((quicktime_codec_t*)vtrack->codec)->encode_video = vcodecs[index].codec.encode_video;
	((quicktime_codec_t*)vtrack->codec)->writes_colormodel = vcodecs[index].codec.writes_colormodel;
	((quicktime_codec_t*)vtrack->codec)->reads_colormodel = vcodecs[index].codec.reads_colormodel;
	((quicktime_codec_t*)vtrack->codec)->set_param = vcodecs[index].codec.set_param;
	((quicktime_codec_t*)vtrack->codec)->get_param = vcodecs[index].codec.get_param;

	((quicktime_codec_t*)vtrack->codec)->priv = vcodecs[index].init(vtrack);

	return 0;
}

int quicktime_init_acodec_core(int index, quicktime_audio_map_t *atrack)
{
	((quicktime_codec_t*)atrack->codec)->delete_acodec = acodecs[total_acodecs - 1].codec.delete_acodec;
	((quicktime_codec_t*)atrack->codec)->decode_audio = acodecs[total_acodecs - 1].codec.decode_audio;
	((quicktime_codec_t*)atrack->codec)->encode_audio = acodecs[total_acodecs - 1].codec.encode_audio;
	((quicktime_codec_t*)atrack->codec)->set_param = acodecs[total_acodecs - 1].codec.set_param;
	((quicktime_codec_t*)atrack->codec)->get_param = acodecs[total_acodecs - 1].codec.get_param;

	((quicktime_codec_t*)atrack->codec)->priv = acodecs[index].init(atrack);
	return 0;
}



