/* codecs.c
 * Copyright (C) 2001 QT4Linux and OpenQuicktime Teams
 *
 * This file is part of OpenQuicktime, a free QuickTime library.
 *
 * Based on QT4Linux by Adam Williams.
 *
 * OpenQuicktime is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * OpenQuicktime is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "colormodels.h"
#include "funcprotos.h"
#include "openquicktime.h"

static int quicktime_delete_vcodec_stub(quicktime_video_map_t *vtrack)
{
	printf("quicktime_delete_vcodec_stub called\n");
	return 0;
}

static int quicktime_delete_acodec_stub(quicktime_audio_map_t *atrack)
{
	printf("quicktime_delete_acodec_stub called\n");
	return 0;
}

static int quicktime_decode_video_stub(quicktime_t *file, 
				       unsigned char **row_pointers, 
				       int track)
{
	printf("quicktime_decode_video_stub called\n");
	return 1;
}

static int quicktime_encode_video_stub(quicktime_t *file, 
				       unsigned char **row_pointers, 
				       int track)
{
	printf("quicktime_encode_video_stub called\n");
	return 1;
}

static int quicktime_decode_audio_stub(quicktime_t *file, 
					int16_t *output_i, 
					float *output_f, 
					long samples, 
					int track, 
					int channel)
{
	printf("quicktime_decode_audio_stub called\n");
	return 1;
}

static int quicktime_encode_audio_stub(quicktime_t *file, 
				int16_t **input_i, 
				float **input_f, 
				int track, 
				long samples)
{
	printf("quicktime_encode_audio_stub called\n");
	return 1;
}


static int quicktime_reads_colormodel_stub(quicktime_t *file, 
		int colormodel, 
		int track)
{
	return (colormodel == BC_RGB888);
}

static int quicktime_writes_colormodel_stub(quicktime_t *file, 
		int colormodel, 
		int track)
{
  printf("quicktime_writes_colormodel_stub called\n");

	return (colormodel == BC_RGB888);
}

static int quicktime_set_param_stub (quicktime_t *file, int track,
				     const char* param, const void* data)
{
  printf("quicktime_set_param_stub called\n");
  return 1;
}

static int quicktime_get_param_stub (quicktime_t *file, int track,
				     const char* param, void* data)
{
  printf("quicktime_get_param_stub called\n");
  return 1;
}




int quicktime_codec_defaults(quicktime_codec_t *codec)
{
	codec->delete_vcodec = quicktime_delete_vcodec_stub;
	codec->delete_acodec = quicktime_delete_acodec_stub;
	codec->decode_video = quicktime_decode_video_stub;
	codec->encode_video = quicktime_encode_video_stub;
	codec->decode_audio = quicktime_decode_audio_stub;
	codec->encode_audio = quicktime_encode_audio_stub;
	codec->set_param = quicktime_set_param_stub;
	codec->get_param = quicktime_get_param_stub;
	codec->reads_colormodel = quicktime_reads_colormodel_stub;
	codec->writes_colormodel = quicktime_writes_colormodel_stub;
	return 0;
}

static int get_vcodec_index(char *compressor)
{
	int index;

/* Try already loaded codec */
	index = quicktime_find_vcodec(compressor);

//printf("get_vcodec_index %d\n", index);
/* Try external codec */
	if(index < 0)
	{
		index = quicktime_register_external_vcodec(compressor);
	}

	if(index < 0)
		return -1;
	return index;
}

static int get_acodec_index(char *compressor)
{
	int index;

/* Try already loaded codec */
	index = quicktime_find_acodec(compressor);

/* Try external codec */
	if(index < 0)
	{
		index = quicktime_register_external_acodec(compressor);
	}

	if(index < 0)
		return -1;
	return index;
}


int quicktime_init_vcodec(quicktime_t *file, quicktime_video_map_t *vtrack)
{
	char *compressor = vtrack->track->mdia.minf.stbl.stsd.table[0].format;
  	int index;
	vtrack->codec = calloc(1, sizeof(quicktime_codec_t));
	quicktime_codec_defaults((quicktime_codec_t*)vtrack->codec);

//printf("quicktime_init_vcodec %s\n", compressor);
	if((index = get_vcodec_index(compressor)) < 0) return -1;

/* Initialize the codec */
	return quicktime_init_vcodec_core(index, vtrack);
}

int quicktime_init_acodec(quicktime_t *file, quicktime_audio_map_t *atrack)
{
	char *compressor = atrack->track->mdia.minf.stbl.stsd.table[0].format;
	int index;

	atrack->codec = calloc(1, sizeof(quicktime_codec_t));
	quicktime_codec_defaults((quicktime_codec_t*)atrack->codec);

	if((index = get_acodec_index(compressor)) < 0) return -1;

/* Initialize the codec */
	return quicktime_init_acodec_core(index, atrack);
}


int quicktime_delete_vcodec(quicktime_t *file, quicktime_video_map_t *vtrack)
{
	((quicktime_codec_t*)vtrack->codec)->delete_vcodec(vtrack);
	free(vtrack->codec);
 	vtrack->codec = 0;
	return 0;
}

int quicktime_delete_acodec(quicktime_t *file, quicktime_audio_map_t *atrack)
{
	((quicktime_codec_t*)atrack->codec)->delete_acodec(atrack);
	free(atrack->codec);
	atrack->codec = 0;
	return 0;
}

int quicktime_supported_video(quicktime_t *file, int track)
{
	char *compressor = quicktime_video_compressor(file, track);
	if((get_vcodec_index(compressor)) < 0) return 0;
	return 1;
}

int quicktime_supported_audio(quicktime_t *file, int track)
{
	char *compressor = quicktime_audio_compressor(file, track);
	if((get_acodec_index(compressor)) < 0) return 0;
	return 1;
}

int quicktime_decode_video(quicktime_t    *file,
			   int             color_model, 
			   unsigned char **row_pointers, 
			   int             track)
{
        int result;
	quicktime_trak_t *trak = file->vtracks[track].track;
	int track_height = trak->tkhd.track_height;
	int track_width = trak->tkhd.track_width;

//printf("quicktime_decode_video 1\n");
	file->color_model = color_model;

//printf("quicktime_decode_video 1\n");
	result = ((quicktime_codec_t*)file->vtracks[track].codec)->decode_video(file, row_pointers, track);
	file->vtracks[track].current_position++;
//printf("quicktime_decode_video 2\n");
	return result;
}

int quicktime_encode_video(quicktime_t    *file, 
			   int             color_model, 
			   unsigned char **row_pointers, 
			   int             track)
{
	int result;
	file->color_model = color_model;
//printf("quicktime_encode_video 1 %p\n", ((quicktime_codec_t*)file->vtracks[track].codec)->encode_video);
	result = ((quicktime_codec_t*)file->vtracks[track].codec)->encode_video(file, row_pointers, track);
//printf("quicktime_encode_video 2\n");
	file->vtracks[track].current_position++;
	return result;
}


int quicktime_decode_audio(quicktime_t *file, 
				int16_t *output_i, 
				float *output_f, 
				long samples, 
				int channel)
{
	int quicktime_track, quicktime_channel;
	int result = 1;

	quicktime_channel_location(file, &quicktime_track, &quicktime_channel, channel);
	result = ((quicktime_codec_t*)file->atracks[quicktime_track].codec)->decode_audio(file, 
				output_i, 
				output_f, 
				samples, 
				quicktime_track, 
				quicktime_channel);
	file->atracks[quicktime_track].current_position += samples;

	return result;
}

/* Since all channels are written at the same time: */
/* Encode using the compressor for the first audio track. */
/* Which means all the audio channels must be on the same track. */

int quicktime_encode_audio(quicktime_t *file, int16_t **input_i, float **input_f, long samples)
{
	int result = 1;
	char *compressor = quicktime_audio_compressor(file, 0);

	result = ((quicktime_codec_t*)file->atracks[0].codec)->encode_audio(file, 
		input_i, 
		input_f,
		0, 
		samples);
	file->atracks[0].current_position += samples;

	return result;
}

int quicktime_reads_cmodel(quicktime_t *file, 
	int colormodel, 
	int track)
{
	int result = ((quicktime_codec_t*)file->vtracks[track].codec)->reads_colormodel(file, colormodel, track);
	return result;
}

int quicktime_writes_cmodel(quicktime_t *file, 
	int colormodel, 
	int track)
{
	return ((quicktime_codec_t*)file->vtracks[track].codec)->writes_colormodel(file, colormodel, track);
}

longest quicktime_samples_to_bytes(quicktime_trak_t *track, long samples)
{
	char *compressor = track->mdia.minf.stbl.stsd.table[0].format;
	int channels = track->mdia.minf.stbl.stsd.table[0].channels;

/* Default use the sample size specification for TWOS and RAW */
	return samples * channels * track->mdia.minf.stbl.stsd.table[0].sample_size / 8;
}

/* Compressors that can only encode a window at a time */
/* need to flush extra data here. */

int quicktime_flush_acodec(quicktime_t *file, int track)
{
	int result = 0;
	return result;
}

void quicktime_flush_vcodec(quicktime_t *file, int track)
{
}

int quicktime_codecs_flush(quicktime_t *file)
{
	int result = 0;
	int i;
	if(!file->wr) return result;

	if(file->total_atracks)
	{
		for(i = 0; i < file->total_atracks && !result; i++)
		{
			result += quicktime_flush_acodec(file, i);
		}
	}
	return result;
}


int quicktime_set_video_param(quicktime_t *file, int track,
			const char* param, const void* data)
{
  return ((quicktime_codec_t*)file->vtracks[track].codec)->set_param(file, track,param,data);
}

int quicktime_get_video_param(quicktime_t *file, int track,
			const char* param, void* data)
{
  return ((quicktime_codec_t*)file->vtracks[track].codec)->get_param(file, track,param,data);
}

int quicktime_set_audio_param(quicktime_t *file, int track,
			const char* param, const void* data)
{
  return ((quicktime_codec_t*)file->atracks[track].codec)->set_param(file, track,param,data);
}

int quicktime_get_audio_param(quicktime_t *file, int track,
			const char* param, void* data)
{
  return ((quicktime_codec_t*)file->atracks[track].codec)->get_param(file, track,param,data);
}
