/* codec_MP3.c
 * Copyright (C) 2001 QT4Linux and OpenQuicktime Teams
 *
 * This file is part of OpenQuicktime, a free QuickTime library.
 *
 * Based on QT4Linux by Adam Williams.
 *
 * OpenQuicktime is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.
 *
 * OpenQuicktime is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <pthread.h>

#include "openquicktime.h"
#include "private.h"

#include "audioencoder/mp3encoder.h"

#include "mpg123/mpg123.h"
#include "mpg123/mpglib.h"


/* Some static global variables */
static int Initialised = 0;           /* Initialising counter */

/* Each instance of the codec should have its own Param structure allocated */
typedef struct {
  MP3Encoder* Encoder;
  struct mpstr mp;
} Param;



/* Antoine: OK, we can have multiple instances of the codecs working at the
same time, but the mpg123lib (at least) has some static buffers that are used
in layer3.c, so we cannot have safely two different threads calling at the 
same time different instances of the mp3 encoder/decoder.
For the moment, I've simply added a mutex allowing only one thread to call
encodeMP3/decodeMP3 at the same time...
*/
static pthread_mutex_t decodelock;
static pthread_mutex_t encodelock;

int delete_MP3(quicktime_audio_map_t *atrack)
{
  Param *p = (Param*)(((quicktime_codec_t*)(atrack->codec))->priv);

  printf("Deleting external MP3 audio codec\n");

  if (p) {
    if(p->Encoder)
      delete_MP3Encoder(p->Encoder);
    free(p);
  }

  return --Initialised;
}


/* Convert the number of samples in a chunk into the number of bytes in that */
/* chunk.  The number of samples in a chunk should end on a block boundary. */
unsigned int MP3_samples_to_bytes(unsigned int samples, int channels)
{
  unsigned int result = /*17266/*50000/*/((samples*3753*8/*2*channels*/)/(41472));

  return  result;
}



/* Decode the chunk into the work buffer */
int decode_MP3(quicktime_t *file, int track, 
	       unsigned long inputsize, 
	       unsigned char* input,
	       unsigned long outputsize, 
	       unsigned char* output)
{
  Param *p = (Param*)(((quicktime_codec_t*)(file->atracks[track].codec))->priv);
  int size;
  int offset=0;
  int last_result = MP3_NEED_MORE;

  pthread_mutex_lock(&decodelock);

  //printf("inputsize %lu outputsize %lu\n", inputsize, outputsize);
  last_result = decodeMP3(&(p->mp),(char*)input, inputsize,(char*)output,outputsize,&size);
  //printf("  offset=%i size=%i last_result=%i\n",offset,size,last_result);
  offset+=size;
  
  while(last_result== MP3_OK && offset<outputsize)
    {// OK, here are some
      last_result = decodeMP3(&(p->mp),NULL,0,(char*)output+offset,outputsize-offset,&size);
      //printf("  offset=%i size=%i last_result=%i\n",offset,size,last_result);
      offset+=size;
    }
 
  pthread_mutex_unlock(&decodelock);

  return offset;
}


void init_encode_MP3(Param *p, quicktime_t *file, int track)
{
  int i;
  WAVEFORMATEX format;
  char *ptr;

  char beginning[30] = {0x00, 0x00, 0x00, 0x4C,
                        0x77, 0x61, 0x76, 0x65, 
			0x00, 0x00, 0x00, 0x0C,
			0x66, 0x72, 0x6D, 0x61,
			0x2E, 0x6D, 0x70, 0x33,
			0x00, 0x00, 0x00, 0x26,
			0x2E, 0x6D, 0x70, 0x33,
			0x55, 0x00};
  char endwav[8]= {0x0C, 0x00, 0x01, 0x00,
		   0x20, 0x01, 0x00, 0x00};
  char enda[20] = {0x00, 0x00, 0x00, 0x00,
		 0x00, 0x0A, 0x65, 0x6E,
		 0x64, 0x61, 0x00, 0x00,
		 0x00, 0x00, 0x00, 0x08,
		 0x00, 0x00, 0x00, 0x00};

  format.wFormatTag = 0x55;
  format.nChannels = file->atracks[track].channels;
  printf("channel %u\n", format.nChannels);
  format.nSamplesPerSec = (int)file->atracks[track].track->mdia.minf.stbl.stsd.table[0].sample_rate;
  format.nAvgBytesPerSec = 128000 + (file->atracks[track].track->mdia.minf.stbl.stsd.table[0].spatial_quality * 32000);
  format.nBlockAlign = 1;
  format.wBitsPerSample = file->atracks[track].track->mdia.minf.stbl.stsd.table[0].sample_size;
  
  p->Encoder = new_MP3Encoder(&format); 

  MP3Encoder_Start(p->Encoder);

  /* some magic stuff to make it work under Windows and Mac */
  file->atracks[track].track->mdia.minf.stbl.stsd.table[0].compression_id = 65534;
  file->atracks[track].track->mdia.minf.stbl.stsd.table[0].version = 1;
  file->atracks[track].track->mdia.mdhd.quality = 0;


  /* Adding some magic stuff to produce correct VBR MP3 mov */
  
  file->atracks[track].track->mdia.minf.stbl.stsc.table[0].samples = 1;

  file->atracks[track].track->mdia.minf.stbl.stsz.sample_size = 0;
  file->atracks[track].track->mdia.minf.stbl.stsz.entries_allocated = 2000;
  file->atracks[track].track->mdia.minf.stbl.stsz.table = (quicktime_stsz_table_t*)malloc(sizeof(quicktime_stsz_table_t) * file->atracks[track].track->mdia.minf.stbl.stsz.entries_allocated);
      
  file->atracks[track].track->tkhd.layer = 65535;
  file->atracks[track].track->tkhd.volume = 1.0;

  /* stupid hardcoded constant ... should be replaced */
  file->atracks[track].track->mdia.minf.stbl.stts.table[0].sample_duration = 1152;  

  /* Add wav header in the private sttd decriptor =) */
  file->atracks[track].track->mdia.minf.stbl.stsd.table[0].private_data_size = 0x4c;
  
  ptr = (char*)malloc(0x4C);
  file->atracks[track].track->mdia.minf.stbl.stsd.table[0].private_data = ptr;

  for(i=0; i<30; i++)
    ptr[i] = beginning[i];
  ptr+=30;

  *(short*)ptr = format.nChannels;
  ptr+=2;

  *(int*)ptr = format.nSamplesPerSec;
  ptr+=4;

  *(int*)ptr = /*format.nAvgBytesPerSec*/ 160000;
  ptr+=4;

  *(short*)ptr = format.nBlockAlign;
  ptr+=2;

  *(short*)ptr = 0;
  ptr+=2;

  for(i=0; i<8; i++)
    ptr[i] = endwav[i];
  ptr+=8;
  
  *(int*)ptr=0;
  ptr+=4;

  for(i=0; i<20; i++)
    ptr[i] = enda[i];


  /* end of magic :( */
  lame_print_config(&(p->Encoder->gf));

}

int encode_MP3(quicktime_t *file, int track, int inputsize,
	       unsigned char *input,
	       unsigned char *output)
{
  Param *p = (Param*)(((quicktime_codec_t*)(file->atracks[track].codec))->priv);
  int size_read, size_written;

  pthread_mutex_lock(&encodelock);

  if(!p->Encoder)
    init_encode_MP3(p, file, track);
      
  MP3Encoder_Convert(p->Encoder,(char*)input, 
		     (inputsize/(2*file->atracks[track].channels)),
		     (char*)output, inputsize,
		     &size_read, &size_written);
 
  //printf("inputsize %u, size_read %u, size_written %u\n", inputsize, size_read, size_written);

  /*
  *(int*)(file->atracks[track].track->mdia.minf.stbl.stsd.table[0].private_data + 50) +=size_written;
  */

  pthread_mutex_unlock(&encodelock);

  return size_written;
}


void *init_MP3(quicktime_audio_map_t *atrack)
{
  Param *p = NULL;  
  Initialised++;

  p = (Param*) malloc(sizeof(Param));
  printf("Initialising MP3 audio codec\n");
  p->Encoder = NULL;
  InitMP3(&(p->mp));
  
  pthread_mutex_init(&decodelock,NULL);
  pthread_mutex_init(&encodelock,NULL);

  return p;
}

int set_param_MP3(quicktime_t *file, int track,
		  const char* param, const void* data)
{
  fprintf(stderr,"set_param_MP3: unknownk parameter '%s'\n",param);
  return 1;
}

int get_param_MP3(quicktime_t *file, int track,
		  const char* param, void* data)
{
  fprintf(stderr,"get_param_MP3: unknownk parameter '%s'\n",param);
  return 1;
}

int quicktime_codec_register( quicktime_extern_audio_t*codec)
{
  int i;

  codec->init   = init_MP3;
  codec->encode = encode_MP3;
  codec->decode = decode_MP3;
  codec->delete_codec = delete_MP3;
  codec->set_param = set_param_MP3;
  codec->get_param = get_param_MP3;
  codec->sample_to_bytes = MP3_samples_to_bytes;

#ifdef REGISTER_MP3
  codec->fourcc[0]='.';
  codec->fourcc[1]='m';
  codec->fourcc[2]='p';
  codec->fourcc[3]= '3';
  codec->fourcc[4]=0;
#else
  codec->fourcc[0]='m';
  codec->fourcc[1]='s';
  codec->fourcc[2]=0;
  codec->fourcc[3]=0x55;
  codec->fourcc[4]=0;
#endif

  printf("Registering 0x%02X,0x%02X,0x%02X,0x%02X MP3 audio codec\n",
	 codec->fourcc[0],codec->fourcc[1],codec->fourcc[2],codec->fourcc[3]);

  return 1;
}
