// $Id: UnitTestRunner.m,v 1.13 2007/03/28 03:16:53 will_mason Exp $
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "UnitTestRunner.h"
#import "ThreadServices.h"
#import <ObjectiveLib/Text.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-runtime.h>
#else
#import <objc/objc-api.h>
#endif
#import <assert.h>
#import <stdlib.h>
#import <string.h>

static OLVector* testsInProcess;
static OLMutex testInProcessLock;
static OLOnceControl initializeOnce = OL_ONCE_INIT;

static void performInitialize()
{
    OLCreateMutex(&testInProcessLock);
    testsInProcess = [[OLVector alloc] initWithCapacity: 10];
}

@implementation UnitTestRunner

#if defined(OL_NO_OPENSTEP)
+ (id) initialize
#else
+ (void) initialize
#endif
{
    OLOnce(&initializeOnce, performInitialize);
#if defined(OL_NO_OPENSTEP)
	return self;
#endif
}

+ (void) popTestInProcess
{
    [testsInProcess popBack];
    OLUnlockMutex(&testInProcessLock);
}

+ (void) pushTestInProcess: (id)test
{
    OLLockMutex(&testInProcessLock);
    [testsInProcess pushBack: test];
}

+ (id) testInProcess
{
    return [testsInProcess empty] ? nil : [testsInProcess back];
}

- (id) init
{
    return [self initWithOut: stdout err: stderr];
}

- (id) initWithOut: (FILE*)o err: (FILE*)e
{
    [super initWithOut: o err: e];
    missingTests = [[OLVector alloc] initWithCapacity: 32];
    testsRun = 0;
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    [missingTests RELEASE];
    SUPER_FREE;
}

#if defined(__NEXT_RUNTIME__)
- (void) automaticallyLoadTests
{
    int classCount = objc_getClassList(NULL, 0);
    Class* classes;
    int i;
    Class unitTestClass;

    if (classCount == 0)
        return;

    unitTestClass = (Class)objc_getClass("UnitTest");
    assert(unitTestClass != NULL);
    if (unitTestClass == NULL)
        return;

    classes = malloc(classCount * sizeof(Class));
    objc_getClassList(classes, classCount);
    for (i = 0; i < classCount; i++)
    {
        if (strncmp(classes[i]->name, "NS", 2) &&
            classes[i]->name[0] != '_')
        {
            printf("%s\n", classes[i]->name);
        }
        if (strcmp(classes[i]->name, "CompositeUnitTest") != 0 &&
            strcmp(classes[i]->name, "UnitTestRunner") != 0 &&
            [self isClass: classes[i] aTypeOf: unitTestClass])
        {
            [self loadTestsForClass: classes[i]];
        }
    }
    free(classes);
}
#else
- (void) automaticallyLoadTests
{
    void* state;
    Class cur;
    Class unitTestClass = objc_get_class("UnitTest");

    assert(unitTestClass != NULL);
    if (unitTestClass == NULL)
        return;

    while ((cur = objc_next_class(&state)) != NULL)
    {
        if (strcmp(cur->name, "CompositeUnitTest") != 0 &&
            strcmp(cur->name, "UnitTestRunner") != 0 &&
            [self isClass: cur aTypeOf: unitTestClass])
        {
            [self loadTestsForClass: cur];
        }
    }
}
#endif

- (BOOL) isClass: (Class)target aTypeOf: (Class)cls
{
    Class current = target->super_class;

    while (current != NULL)
    {
        if (strcmp(current->name, cls->name) == 0)
            return YES;
        current = current->super_class;
    }
    return NO;
}

- (void) loadTestsForClass: (Class)cls
{
    id instance;

    if (cls != NULL && [self isClass: cls aTypeOf: [UnitTest class]])
    {
        instance = [[cls alloc] init];
        if (![instance preventAutoLoad])
            [self addUnitTest: instance];
        [instance RELEASE];
    }
}

- (void) loadTestsForClasses: (Class*)array
{
    Class cur = *array;

    while (cur != NULL)
    {
        [self loadTestsForClass: cur];
        cur = *++array;
    }
}

- (void) runAllTests
{
    [super runAllTests];
    testsRun += [self testCount];
}

- (BOOL) runTestWithName: (const char*)name
{
    OLText* missing;

    if (![super runTestWithName: name])
    {
        missing = [[OLText alloc] initWithCString: name];
        [missingTests pushBack: missing];
        [missing RELEASE];
        return NO;
    }
    else
    {
        testsRun++;
        return YES;
    }
}

- (void) runTestsWithNames: (OLVector*)names
{
    unsigned i;

    for (i = 0; i < [names size]; i++)
        [self runTestWithName: [[names at: i] cString]];
}

- (void) summarize
{
    int errs = [self errorCount];
    int missings = [missingTests size];
    FILE* stream = [self outStream];
    unsigned i;

    fprintf(stream, "\n%i %s run.\n", testsRun, (testsRun == 1) ? "test" : "tests");
    if (errs == 0)
    {
        fprintf(stream, "\nNo errors.\n");
    }
    else
    {
        fprintf(stream, "\nThere %s %i %s. The tests that contained errors were:\n",
                (errs == 1) ? "was" : "were",
                errs,
                (errs == 1) ? "error" : "errors");
        [self listFailedTestsToStream: stream delimiter: ' '];
        fprintf(stream, "\n");
    }
    if (missings != 0)
    {
        fprintf(stream, "\n%i %s not found. The missing %s:\n",
                missings,
                (missings == 1) ? "test was" : "tests were",
                (missings == 1) ? "test was" : "tests were");
        for (i = 0; i < missings; i++)
            fprintf(stream, "%s\n", [[missingTests at: i] cString]);
    }
    fprintf(stream, "\n");
}

@end
