//
// $Id: SocketAddressTest.m,v 1.13 2007/03/06 20:42:21 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "SocketAddressTest.h"
#import <ObjectiveLib/SocketAddress.h>
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#import <ObjectiveLib/Exception.h>
#import <ObjectiveLib/Text.h>
#else
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#endif
#import <unistd.h>
#include <limits.h>
#import <string.h>

@implementation SocketAddressTest

- (void) testCopyInternetAddresses
{
    OLInternetAddress* addr;
    OLInternetAddress* addr2;

    TRY

        addr = REAP([OLInternetAddress addressWithCurrentHostAndPort: 0]);
        addr2 = [addr copy];
        if (![addr isEqual: addr2])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" should be equal to \"%s\"",
                [REAP([addr description]) cString], [REAP([addr2 description]) cString]];
        }
        [addr2 RELEASE];

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "Could not resolve current host - %s: %s",
            EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    TRY

        addr = REAP([OLInternetAddress addressWithCurrentHostAndService: "telnet"]);
        addr2 = [addr copy];
        if (![addr isEqual: addr2])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" should be equal to \"%s\"",
                [REAP([addr description]) cString], [REAP([addr2 description]) cString]];
        }
        [addr2 RELEASE];

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "Could not resolve telnet service on current host - %s: %s",
            EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    TRY

        addr = REAP([OLInternetAddress addressWithHost: "gnustep.org" service: "telnet"]);
        addr2 = [addr copy];
        if (![addr isEqual: addr2])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" should not be equal to \"%s\"",
                [REAP([addr description]) cString], [REAP([addr2 description]) cString]];
        }
        [addr2 RELEASE];

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "Could not resolve telnet service on host gnustep.org - %s: %s",
            EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH
}

#if defined(OL_HAVE_UNIX_SOCKETS)

- (void) testCopyUnixAddress
{
    OLUnixAddress* addr;
    OLUnixAddress* addr2;

    addr = [[OLUnixAddress alloc] initWithPath: "test.socket"];
    addr2 = [addr copy];
    if (![addr isEqual: addr2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should be equal to \"%s\"",
            [addr path], [addr2 path]];
    }
    [addr RELEASE];
    [addr2 RELEASE];
}

#endif

- (void) testInternetAddresses
{
    OLInternetAddress* addr;
    OLInternetAddress* addr2;

    TRY

        addr = REAP([OLInternetAddress addressWithCurrentHostAndPort: 0]);
        [self logMessage: "Current host is %s", [REAP([addr description]) cString]];
        if ([addr port] != 0)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 0, but got %u", [addr port]];
        }
        addr2 = REAP([OLInternetAddress addressWithCurrentHostAndPort: 0]);
        if (![addr isEqual: addr2])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" should be equal to \"%s\"",
                [REAP([addr description]) cString], [REAP([addr2 description]) cString]];
        }

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "Could not resolve current host - %s: %s",
            EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    TRY

        addr = REAP([OLInternetAddress addressWithCurrentHostAndService: "telnet"]);
        [self logMessage: "Current telnet host is %s", [REAP([addr description]) cString]];
        if ([addr port] != 23)
            [self logMessage: "Expected 23, but got %u", [addr port]];

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "Could not resolve telnet service on current host - %s: %s",
            EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    TRY

        addr = REAP([OLInternetAddress addressWithHost: "gnustep.org" port: 0]);
        [self logMessage: "gnustep.org is %s", [REAP([addr description]) cString]];

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "Could not resolve host gnustep.org - %s: %s",
            EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    TRY

        addr = REAP([OLInternetAddress addressWithHost: "gnustep.org" service: "telnet"]);
        [self logMessage: "gnustep.org telnet is %s", [REAP([addr description]) cString]];
        if ([addr port] != 23)
            [self logMessage: "Expected 23, but got %u", [addr port]];
        addr2 = REAP([OLInternetAddress addressWithHost: "gnu.org" service: "telnet"]);
        if ([addr isEqual: addr2])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" should not be equal to \"%s\"",
                [REAP([addr description]) cString], [REAP([addr2 description]) cString]];
        }

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "Could not resolve telnet service on host gnustep.org - %s: %s",
            EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH
}

#if defined(OL_HAVE_UNIX_SOCKETS)

- (void) testUnixAddress
{
    OLUnixAddress* addr;
    OLUnixAddress* addr2;
    char buf[OL_PATH_MAX * 2];

    if (getcwd(buf, OL_PATH_MAX * 2) == NULL)
    {
        RAISE_EXCEPTION(@"UnitTestException",
            @"Couldn't get cwd due to buffer size limitation");
    }
    strcat(buf, "/test.socket");
    addr = [[OLUnixAddress alloc] initWithPath: "test.socket"];
    [self logMessage: "Got path for socket of %s", [addr path]];
    addr2 = [[OLUnixAddress alloc] initWithPath: buf];
    if (![addr isEqual: addr2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should be equal to \"%s\"",
            [addr path], [addr2 path]];
    }
    [addr RELEASE];
    addr = [[OLUnixAddress alloc] initWithPath: "doggyboy"];
    if ([addr isEqual: addr2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should not be equal to \"%s\"",
            [addr path], [addr2 path]];
    }
    [addr RELEASE];
    [addr2 RELEASE];
    addr = REAP([OLUnixAddress addressWithPath: "test.socket"]);
    addr2 = REAP([OLUnixAddress addressWithPath: buf]);
    if (![addr isEqual: addr2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should be equal to \"%s\"",
            [addr path], [addr2 path]];
    }
}

#endif

- (void) testWildcards
{
    OLInternetAddress* addr;

    addr = [[OLInternet4Address alloc] init];
    if (![addr isLocalWildcard])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should be a local wildcard address",
            [REAP([addr description]) cString]];
    }
    [addr RELEASE];
#if defined(OL_HAVE_INET6_SOCKETS)
    addr = [[OLInternet6Address alloc] init];
    if (![addr isLocalWildcard])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should be a local wildcard address",
            [REAP([addr description]) cString]];
    }
    [addr RELEASE];
#endif
    addr = [[OLInternet4Address alloc] initWithPort: 30001];
    if (![addr isLocalWildcard])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should be a local wildcard address",
            [REAP([addr description]) cString]];
    }
    if ([addr port] != 30001)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 30001, but got %u", [addr port]];
    }
    [addr RELEASE];
#if defined(OL_HAVE_INET6_SOCKETS)
    addr = [[OLInternet6Address alloc] initWithPort: 30001];
    if (![addr isLocalWildcard])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should be a local wildcard address",
            [REAP([addr description]) cString]];
    }
    if ([addr port] != 30001)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 30001, but got %u", [addr port]];
    }
    [addr RELEASE];
#endif
}

@end
