//
// $Id: InternetSocketTest.m,v 1.12 2007/03/28 03:16:52 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "InternetSocketTest.h"
#import "ThreadServices.h"
#import "Random.h"
#import <ObjectiveLib/Socket.h>
#import <ObjectiveLib/OutStream.h>
#import <ObjectiveLib/InStream.h>
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#import <ObjectiveLib/Text.h>
#import <ObjectiveLib/Exception.h>
#else
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#import <Foundation/NSAutoreleasePool.h>
#if defined(GNUSTEP)
#import <Foundation/NSThread.h>
#endif
#endif
#import <limits.h>
#include <stdlib.h>
#import <string.h>
#import <unistd.h>

#define BOOL_OPT_STR(msg) ( msg ? "YES" : "NO" )

static void* threadMain(void* arg)
{
#if defined(GNUSTEP)
    GSRegisterCurrentThread();
#endif 
    InternetSocketTest* test = arg;
    OLSocket* accpt;
    uint8_t* myData = NULL;
    unsigned i;
#if defined(OL_NO_OPENSTEP)
    OLReaper* reaper = [[OLReaper alloc] init];
#else
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
#endif

    TRY

        [test logMessage: "Starting server thread"];
        myData = malloc(1000000);
        accpt = REAP([REAP([OLInternetServerSocket socketWithLocalInternetAddress:
            REAP([OLInternetAddress addressWithCurrentHostAndPort: 19567])])
            acceptConnection]);
        [test logMessage: "Server accepted connection"];
        i = 0;
        while (i < 1000000)
            i += [[accpt inStream] readBytes: myData + i count: 1000000 - i];
        if (i != 1000000)
        {
            [test errInFile: __FILE__ line: __LINE__
                format: "Only wrote %u of 1000000 bytes", i];
        }
        if (memcmp(myData, test->communicatedData, 1000000) != 0)
        {
            [test errInFile: __FILE__ line: __LINE__
                format: "The memory regions should be equal"];
        }
        [test logMessage: "Server comparison done"];
        for (i = 0; i < 1000000; i++)
            test->communicatedData[i] = OLRandom() % 256;
        i = 0;
        while (i < 1000000)
            i += [[accpt outStream] writeBytes: test->communicatedData + i count: 1000000 - i];
        if (i != 1000000)
        {
            [test errInFile: __FILE__ line: __LINE__
                format: "Only wrote %u of 1000000 bytes", i];
        }
        [test logMessage: "Server data written"];

    CATCH

        [test errInFile: __FILE__ line: __LINE__
            format: "%s: %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    free(myData);
#if defined(OL_NO_OPENSTEP)
    [reaper RELEASE];
#else
    [pool RELEASE];
#endif
    [test logMessage: "Leaving server thread"];
    return NULL;
}

@implementation InternetSocketTest

- (void) testBind
{
    OLInternetServerSocket* server;
    OLInternetAddress* addr;

    server = [[OLInternetServerSocket alloc] init];

    TRY

        addr = REAP([OLInternetAddress addressWithCurrentHostAndPort: 5381]);
        [server bindToAddress: addr];
        if (![REAP([server localAddress]) isEqual: addr])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "\"%s\" should be equal to \"%s\"",
                [REAP([addr description]) cString],
                [REAP([REAP([server localAddress]) description]) cString]];
        }

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "%s: %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    [server RELEASE];
}

- (void) testCommunicate
{
    OLInternetClientSocket* client = nil;
    uint8_t* myData = NULL;
    int i;
    OLThread serverThread;

    TRY

        communicatedData = malloc(1000000);
        for (i = 0; i < 1000000; i++)
            communicatedData[i] = OLRandom() % 256;
        OLCreateThread(&serverThread, threadMain, self);
#if defined(OL_HAVE_SLEEP)
        sleep(1);
#elif defined(OL_HAVE__SLEEP)
        _sleep(1);
#endif
        [self logMessage: "Client connecting"];
        client = REAP([OLInternetClientSocket socketWithRemoteInternetAddress:
            REAP([OLInternetAddress addressWithCurrentHostAndPort: 19567])]);
        [self logMessage: "Client connected to %s",
            [REAP([REAP([client remoteAddress]) description]) cString]];
        i = 0;
        while (i < 1000000)
            i += [[client outStream] writeBytes: communicatedData + i count: 1000000 - i];
        if (i != 1000000)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Only wrote %i of 1000000 bytes", i];
        }
        [self logMessage: "Client data written"];
        myData = malloc(1000000);
        i = 0;
        while (i < 1000000)
            i += [[client inStream] readBytes: myData + i count: 1000000 - i];
        if (i != 1000000)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Only read %i of 1000000 bytes", i];
        }
        if (memcmp(myData, communicatedData, 1000000) != 0)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The memory regions should be equal"];
        }
        [self logMessage: "Client comparison done"];
        OLWaitForThread(serverThread);
        [self logMessage: "Server thread joined"];

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "%s: %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    free(myData);
    free(communicatedData);
}

- (void) testConvenienceAllocators
{
    OLInternetServerSocket* server;
    OLInternetClientSocket* client;
    unsigned qLen;

    server = REAP([OLInternetServerSocket socket]);
    qLen = [server queueLength];
    [self logMessage: "The default queue length is %u", qLen];

    TRY

        if (![(OLInternetAddress*)REAP([server localAddress]) isLocalWildcard])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The server is not bound to a local address"];
        }

    CATCH
    END_CATCH

    server = REAP([OLInternetServerSocket socketWithLocalInternetAddress:
        REAP([OLInternetAddress addressWithCurrentHostAndPort: 19568])]);
    if ([server queueLength] != qLen)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", qLen, [server queueLength]];
    }

    TRY

        if ([(OLInternetAddress*)REAP([server localAddress]) isLocalWildcard])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The server should be bound"];
        }

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "The server should be bound"];

    END_CATCH

    server = REAP([OLInternetServerSocket socketWithQueueLength: 5]);
    if ([server queueLength] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [server queueLength]];
    }

    TRY

        if (![(OLInternetAddress*)REAP([server localAddress]) isLocalWildcard])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The server is not bound to a local address"];
        }

    CATCH
    END_CATCH

    server = REAP([OLInternetServerSocket socketWithLocalInternetAddress:
        REAP([OLInternetAddress addressWithCurrentHostAndPort: 19569])
        queueLength: 5]);
    if ([server queueLength] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [server queueLength]];
    }

    TRY

        if ([(OLInternetAddress*)REAP([server localAddress]) isLocalWildcard])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The server should be bound"];
        }

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "The server should be bound"];

    END_CATCH

    client = REAP([OLInternetClientSocket socket]);

    TRY

        REAP([client remoteAddress]);
        [self errInFile: __FILE__ line: __LINE__
            format: "The client is not connected to a server"];

    CATCH
    END_CATCH

    client = REAP([OLInternetClientSocket socketWithRemoteInternetAddress:
        REAP([OLInternetAddress addressWithCurrentHostAndPort: 19569])]);

    TRY

        REAP([client remoteAddress]);

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "The client is connected"];

    END_CATCH
}

- (void) testSocketOptions
{
    OLInternetServerSocket* server;
    unsigned scratch;
    BOOL flipped;

    server = [[OLInternetServerSocket alloc] init];
    [self logMessage: "Internet Server Socket Options:"];

    TRY

        [self logMessage: " allows broadcast         : %s",
            BOOL_OPT_STR([server allowsBroadcast])];

    CATCH

        [self logMessage: " allows broadcast         : %s",
            EXCEPTION_MESSAGE];

    END_CATCH

    [self logMessage: " don't route              : %s",
        BOOL_OPT_STR([server dontRoute])];
    [self logMessage: " keep alive               : %s",
        BOOL_OPT_STR([server keepAlive])];
    scratch = [server linger];
    if (scratch != UINT_MAX)
        [self logMessage: " linger                   : %u", scratch];
    else
        [self logMessage: " linger                   : off"];
    [self logMessage: " out-of-band inline       : %s",
        BOOL_OPT_STR([server outOfBandInline])];
    [self logMessage: " receive buffer size      : %u",
        [server receiveBufferSize]];

    TRY
    
        scratch = [server receiveLowWaterMark];
        [self logMessage: " receive low water mark   : %u", scratch];

    CATCH

        [self logMessage: " receive low water mark   : %s",
            EXCEPTION_MESSAGE];

    END_CATCH
        
    TRY
    
        scratch = [server receiveTimeOut];
        [self logMessage: " receive time out         : %u", scratch];

    CATCH

        [self logMessage: " receive time out         : %s",
            EXCEPTION_MESSAGE];

    END_CATCH

    [self logMessage: " reuse address            : %s",
        BOOL_OPT_STR([server reuseAddress])];
    [self logMessage: " reuse port               : %s",
        BOOL_OPT_STR([server reusePort])];
    [self logMessage: " send buffer size         : %u",
        [server sendBufferSize]];

    TRY
    
        scratch = [server sendLowWaterMark];
        [self logMessage: " send low water mark      : %u", scratch];

    CATCH

        [self logMessage: " send low water mark      : %s",
            EXCEPTION_MESSAGE];

    END_CATCH

    TRY
    
        scratch = [server sendTimeOut];
        [self logMessage: " send time out            : %u", scratch];

    CATCH

        [self logMessage: " send time out            : %s",
            EXCEPTION_MESSAGE];

    END_CATCH

    [self logMessage: " socket type              : %u",
        [server socketType]];
    [self logMessage: " queue length             : %u",
        [server queueLength]];

    TRY

        flipped = [server allowsBroadcast] ? NO : YES;
        [server setAllowsBroadcast: flipped];
        if ([server allowsBroadcast] != flipped)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %s, but got %s",
                flipped ? "YES" : "NO", [server allowsBroadcast] ? "YES" : "NO"];
        }

    CATCH

        [self logMessage: "%s: %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    flipped = [server dontRoute] ? NO : YES;
    [server setDontRoute: flipped];
    if ([server dontRoute] != flipped)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %s, but got %s",
            flipped ? "YES" : "NO", [server dontRoute] ? "YES" : "NO"];
    }
    flipped = [server keepAlive] ? NO : YES;
    [server setKeepAlive: flipped];
    if ([server keepAlive] != flipped)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %s, but got %s",
            flipped ? "YES" : "NO", [server keepAlive] ? "YES" : "NO"];
    }
    [server setLinger: 99];
    if ([server linger] != 99)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 99, but got %u", [server linger]];
    }
    flipped = [server outOfBandInline] ? NO : YES;
    [server setOutOfBandInline: flipped];
    if ([server outOfBandInline] != flipped)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %s, but got %s",
            flipped ? "YES" : "NO", [server outOfBandInline] ? "YES" : "NO"];
    }
    [server setReceiveBufferSize: 711];
    if ([server receiveBufferSize] < 711)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 711, but got %u", [server receiveBufferSize]];
    }

    TRY

        [server setReceiveLowWaterMark: 711];
        if ([server receiveLowWaterMark] != 711)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 711, but got %u", [server receiveLowWaterMark]];
        }
        [server setReceiveTimeOut: 700];
        if ([server receiveTimeOut] != 700)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 700, but got %u", [server receiveTimeOut]];
        }

    CATCH

        [self logMessage: "%s: %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    flipped = [server reuseAddress] ? NO : YES;
    [server setReuseAddress: flipped];
    if ([server reuseAddress] != flipped)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %s, but got %s",
            flipped ? "YES" : "NO", [server reuseAddress] ? "YES" : "NO"];
    }
    if ([server hasReusePortSupport])
    {
        flipped = [server reusePort] ? NO : YES;
        [server setReusePort: flipped];
        if ([server reusePort] != flipped)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %s, but got %s",
                flipped ? "YES" : "NO", [server reusePort] ? "YES" : "NO"];
        }
    }
    [server setSendBufferSize: 711];
    if ([server sendBufferSize] < 711)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 711, but got %u", [server sendBufferSize]];
    }

    TRY

        [server setSendLowWaterMark: 711];
        if ([server sendLowWaterMark] != 711)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 711, but got %u", [server sendLowWaterMark]];
        }
        [server setSendTimeOut: 700];
        if ([server sendTimeOut] != 700)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 700, but got %u", [server sendTimeOut]];
        }

    CATCH

        [self logMessage: "%s: %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    [server RELEASE];
}

@end
