//
// $Id: ZlibOutStream.m,v 1.11 2007/03/06 20:42:20 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "ZlibOutStreamPackage.h"
#import "OutStreamPackage.h"
#import "Macros.h"
#import "RunTime.h"
#import "Exception.h"
#if defined(OL_NO_OPENSTEP)
#import "Text.h"
#else
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#endif
#import <stddef.h>
#import <zlib.h>
#import <string.h>
#import <stdlib.h>

#define FREE_MY_RESOURCES \
    [self close]; \
    objc_free(buffer); \
    objc_free(zstream)

voidpf __ol_zalloc(voidpf alloc, uInt items, uInt size)
{
    return objc_malloc(items * size);
}

void __ol_zfree(voidpf alloc, voidpf address)
{
    objc_free(address);
}

@implementation OLZlibOutStream

+ (id) streamWithOutStream: (OLOutStream*)underStream
{
    OL_BEGIN_AUTO_CTOR(OLZlibOutStream)
        initWithOutStream: underStream
    OL_END_AUTO_CTOR;
}

+ (id) streamWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level
{
    OL_BEGIN_AUTO_CTOR(OLZlibOutStream)
        initWithOutStream: underStream compressionLevel: level
    OL_END_AUTO_CTOR;
}

- (id) initWithOutStream: (OLOutStream*)underStream
{
    return [self initWithOutStream: underStream compressionLevel: Z_DEFAULT_COMPRESSION
        bufferSize: 1024 writeZlibHeader: YES];
}

- (id) initWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level
{
    return [self initWithOutStream: underStream compressionLevel: level bufferSize: 1024
        writeZlibHeader: YES];
}

- (id) initWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level bufferSize: (unsigned)size
{
    return [self initWithOutStream: underStream compressionLevel: level bufferSize: size
        writeZlibHeader: YES];
}

- (id) initWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level bufferSize: (unsigned)size writeZlibHeader: (BOOL)zlibHeader
{
    int rc;
    char msgBuf[512];
    int windowBits = zlibHeader ? MAX_WBITS : -MAX_WBITS;

    [super initWithOutStream: underStream];
    if (level != Z_DEFAULT_COMPRESSION && (level < 0 || level > 9))
    {
        isFinished = YES;
        RAISE_EXCEPTION(INVALID_ARGUMENT,
            @"Compression level must be in the range [0,9]");
    }
    zstream = objc_malloc(sizeof(struct z_stream_s));
    zstream->zalloc = __ol_zalloc;
    zstream->zfree = __ol_zfree;
    zstream->opaque = NULL;
    rc = deflateInit2(zstream, level, Z_DEFLATED, windowBits, MAX_MEM_LEVEL, Z_DEFAULT_STRATEGY);
    if (rc != Z_OK)
    {
        if (zstream->msg == NULL)
        {
            msgBuf[0] = 0;
        }
        else
        {
            strncpy(msgBuf, zstream->msg, 512);
            msgBuf[511] = 0;
        }
        buffer = NULL;
        bufferSize = 0;
        objc_free(zstream);
        zstream = NULL;
        isFinished = YES;
        RAISE_EXCEPTION(OLInputOutputException,
            @"Error initializing for deflation - %s", msgBuf);
    }
    buffer = objc_malloc(size);
    bufferSize = size;
    isFinished = NO;
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc
{
	FREE_MY_RESOURCES;
    SUPER_FREE;
}
#endif

- (void) close
{
    if (!isFinished)
        [self finish];
    [stream flush];
    [stream close];
}

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources
{
	FREE_MY_RESOURCES;
	[super freeStreamResources];
}
#endif

- (unsigned) writeBytes: (const uint8_t*)bytes count: (unsigned)num
{
    zstream->next_in = (Bytef*)bytes;
    zstream->avail_in = num;
    while (zstream->avail_in > 0)
    {
        zstream->next_out = buffer;
        zstream->avail_out = bufferSize;
        if (deflate(zstream, Z_NO_FLUSH) != Z_OK)
        {
            RAISE_EXCEPTION(OLInputOutputException,
                @"Error deflating - %s", zstream->msg);
        }
        [stream completelyWriteBytes: buffer count: bufferSize - zstream->avail_out];
    }
    return num;
}

@end

@implementation OLZlibOutStream (PackageMethods)

- (void) finish
{
    char msgBuf[512];
    int rc;

    // Regardless of what happens in this method we don't want to do it again
    isFinished = YES;

    zstream->next_in = NULL;
    zstream->avail_in = 0;
    do
    {
        zstream->next_out = buffer;
        zstream->avail_out = bufferSize;
        rc = deflate(zstream, Z_FINISH);
        [stream completelyWriteBytes: buffer count: bufferSize - zstream->avail_out];
    } while (rc == Z_OK);
    if (rc != Z_STREAM_END)
    {
        if (zstream->msg == NULL)
        {
            msgBuf[0] = 0;
        }
        else
        {
            strncpy(msgBuf, zstream->msg, 512);
            msgBuf[511] = 0;
        }
        deflateEnd(zstream);
        RAISE_EXCEPTION(OLInputOutputException,
            @"Error closing OLZlibOutStream - %s", msgBuf);
    }
    if (deflateEnd(zstream) != Z_OK)
    {
        RAISE_EXCEPTION(OLInputOutputException,
            @"Error closing OLZlibOutStream - %s", zstream->msg);
    }
}

@end

