//
// $Id: SocketAddress.m,v 1.33 2007/04/15 23:32:57 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "SocketAddressPackage.h"
#import "Synchronization.h"
#import "RunTime.h"
#import "Macros.h"
#import "ByteOrder.h"
#import "ConfigPrivate.h"
#import "VectorPackage.h"
#import "Functional.h"
#import "Algorithm.h"
#import "HashFunction.h"
#import "Text.h"
#import "Exception.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSException.h>
#import <Foundation/NSData.h>
#import <Foundation/NSPathUtilities.h>
#endif
#import <sys/types.h>
#import <sys/param.h>
#import <unistd.h>
#import <string.h>
#include <stdlib.h>
#import <errno.h>
#import <limits.h>
#if defined(OL_REALPATH_SUCKS)
#import <libgen.h>
#endif
#import <stdio.h>

#if defined(OL_WINDOWS_SOCKETS)
#import <winsock2.h>
#import <ws2tcpip.h>
#define MAXHOSTNAMELEN NI_MAXHOST
#else
#import <sys/socket.h>
#import <netdb.h>
#import <netinet/in.h>
#if defined(OL_HAVE_UNIX_SOCKETS)
#import <sys/un.h>
#endif
#endif

static BOOL __preferIPv6Addresses;
#if defined(OL_WINDOWS_SOCKETS)
static WSADATA __windowsAPIData;
OLOnceControl OLInitializeWindowsSocketsOnceControl = OL_ONCE_INIT;
void OLInitializeWindowsSockets()
{
    WORD versionsToTry[] =
    {
        MAKEWORD(1, 0),
        MAKEWORD(1, 1),
        MAKEWORD(2, 0),
        MAKEWORD(2, 1),
        MAKEWORD(2, 2)
    };
    int curVersion = sizeof(versionsToTry) / sizeof(WORD) - 1;
    int rc;

    do
    {
        rc = WSAStartup(versionsToTry[curVersion], &__windowsAPIData);
    } while (rc == WSAVERNOTSUPPORTED && --curVersion >= 0);
    if (rc != 0)
    {
        fprintf(stderr, "*** The Windows socket API could not be initialized.\n");
        if (rc == WSASYSNOTREADY)
            fprintf(stderr, "*** The system is not ready for network communication.\n");
        else if (rc == WSAVERNOTSUPPORTED)
            fprintf(stderr, "*** The socket library version is not supported.\n");
        else if (rc == WSAEINPROGRESS)
            fprintf(stderr, "*** A blocking socket operation is in progress.\n");
        else if (rc == WSAEPROCLIM)
            fprintf(stderr, "*** The socket library cannot support more tasks.\n");
        fflush(stderr);
        abort();
    }
    printf("WSAStartup: %s\nWSA System Status: %s\n", __windowsAPIData.szDescription, __windowsAPIData.szSystemStatus);
}
#endif

@interface OLCompareInternetAddressesByType : OLStreamableFunctor <OLBoolBinaryFunction>
{
}

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2;

@end

@interface OLInternetAddress (PrivateMethods)

+ (OLInternetAddress*) addressImplWithHost: (const char*)name service: (const char*)service port: (int)port;
+ (OLVector*) allAddressesImplWithHost: (const char*)name service: (const char*)service port: (int)port;
- (id) initImplWithName: (const char*)name socketType: (int)type;
- (int) addressFamily;

@end

@interface OLInternet4Address (PrivateMethods)

#if defined(OL_HAVE_GETADDRINFO)

- (id) initImplWithAddrinfo: (const struct addrinfo*)info port: (int) port;

#elif defined(OL_HAVE_GETHOSTBYNAME)

- (id) initImplWithHostent: (const struct hostent*)hent andServent: (const struct servent*)sent;

#endif
- (id) initLoopbackImplWithPort: (uint16_t)port;

@end

#if defined(OL_HAVE_INET6_SOCKETS)

@interface OLInternet6Address (PrivateMethods)

#if defined(OL_HAVE_GETADDRINFO)

- (id) initImplWithAddrinfo: (const struct addrinfo*)info port: (int)port;

#elif defined(OL_HAVE_GETHOSTBYNAME)

- (id) initImplWithHostent: (const struct hostent*)hent andServent: (const struct servent*)sent;

#endif

@end

#endif

@implementation OLSocketAddress

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    return nil;
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return nil;
}

#endif

- (OLText*) description
{
    SUBCLASS_MUST_IMPLEMENT;
    return nil;
}

- (unsigned) hash
{
    return OLHash((const uint8_t*)[self hostRepresentation], [self hostRepresentationLength]);
}

- (const struct sockaddr*) hostRepresentation
{
    SUBCLASS_MUST_IMPLEMENT;
    return NULL;
}

- (unsigned) hostRepresentationLength
{
    SUBCLASS_MUST_IMPLEMENT;
    return 0;
}

@end

@implementation OLInternetAddress

+ (OLInternetAddress*) addressWithCurrentHostAndPort: (uint16_t)port
{
    char nameBuf[MAXHOSTNAMELEN];

    OL_DURING

        if (gethostname(nameBuf, MAXHOSTNAMELEN - 1) != 0)
        {
            RAISE_EXCEPTION(OLSocketException,
                @"Could not retrieve the current host name");
        }
        OL_VALUERETURN([OLInternetAddress addressImplWithHost: nameBuf
            service: NULL port: port], OLInternetAddress*);

    OL_HANDLER

        OL_CATCH_RETURN([OLInternetAddress loopbackWithPort: port]);

    OL_ENDHANDLER
}

+ (OLInternetAddress*) addressWithCurrentHostAndService: (const char*)service
{
    char nameBuf[MAXHOSTNAMELEN];

    OL_DURING

        if (gethostname(nameBuf, MAXHOSTNAMELEN - 1) != 0)
        {
            RAISE_EXCEPTION(OLSocketException,
                @"Could not retrieve the current host name");
        }
        OL_VALUERETURN([OLInternetAddress addressImplWithHost: nameBuf
            service: service port: -1], OLInternetAddress*);

    OL_HANDLER

        OL_CATCH_RETURN([OLInternetAddress loopbackWithService: service]);

    OL_ENDHANDLER
}

+ (OLInternetAddress*) addressWithHost: (const char*)name port: (uint16_t)port
{
    return [OLInternetAddress addressImplWithHost: name service: NULL port: port];
}

+ (OLInternetAddress*) addressWithHost: (const char*)name service: (const char*)service
{
    return [OLInternetAddress addressImplWithHost: name service: service port: -1];
}

+ (OLVector*) allAddressesWithHost: (const char*)name port: (uint16_t)port
{
    return [OLInternetAddress allAddressesImplWithHost: name service: NULL port: port];
}

+ (OLVector*) allAddressesWithHost: (const char*)name service: (const char*)service;
{
    return [OLInternetAddress allAddressesImplWithHost: name service: service port: -1];
}

#if defined(OL_NO_OPENSTEP)
+ (id) initialize
#else
+ (void) initialize
#endif
{
    if (self == [OLInternetAddress class])
    {
        __preferIPv6Addresses = NO;
#if defined(OL_WINDOWS_SOCKETS)
        OLOnce(&OLInitializeWindowsSocketsOnceControl, OLInitializeWindowsSockets);
#endif
    }
#if defined(OL_NO_OPENSTEP)
    return self;
#endif
}

+ (OLInternetAddress*) loopbackWithPort: (uint16_t)port
{
    OLInternetAddress* loop;

#if defined(OL_HAVE_INET6_SOCKETS)
    if (__preferIPv6Addresses)
        loop = [[OLInternet6Address alloc] initLoopbackImplWithPort: port];
    else
#endif
        loop = [[OLInternet4Address alloc] initLoopbackImplWithPort: port];
    return OBJ_AUTORELEASE(loop);
}

+ (OLInternetAddress*) loopbackWithService: (const char*)service
{
    struct servent* sent = getservbyname(service, NULL);

    if (sent == NULL)
    {
        RAISE_EXCEPTION(OLSocketException,
            @"Error looking up service %s",
            service);
    }
    return [OLInternetAddress loopbackWithPort: sent->s_port];
}

+ (BOOL) preferIPv6Addresses
{
    return __preferIPv6Addresses;
}

+ (void) setPreferIPv6Addresses: (BOOL)state
{
    __preferIPv6Addresses = state;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    objc_free(canonicalName);
    SUPER_FREE;
}

- (const char*) canonicalName
{
#if defined(OL_HAVE_GETNAMEINFO)
    char nameBuf[NI_MAXHOST];
#elif defined(OL_HAVE_GETHOSTBYADDR)
    struct hostent* hent;
#endif

    if (canonicalName == NULL)
    {
#if defined(OL_HAVE_GETNAMEINFO)
        if (getnameinfo([self hostRepresentation], [self hostRepresentationLength],
                nameBuf, NI_MAXHOST, NULL, 0, NI_NAMEREQD) != 0)
        {
            RAISE_EXCEPTION(OLSocketException,
                @"Could not resolve host name");
        }
        canonicalName = objc_malloc(strlen(nameBuf) + 1);
        strcpy(canonicalName, nameBuf);
#elif defined(OL_HAVE_GETHOSTBYADDR)
        hent = gethostbyaddr((const char*)[self hostRepresentation],
            [self hostRepresentationLength], [self addressFamily]);
        if (hent == NULL)
        {
#if defined(OL_WINDOWS_SOCKETS)
            char msgBuf[8192];

            if (FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
                              NULL,
                              WSAGetLastError(),
                              0,
                              msgBuf,
                              8191,
                              NULL) == 0)
            {
                msgBuf[0] = 0;
            }
            RAISE_EXCEPTION(OLSocketException,
                @"Could not resolve host name - %s", msgBuf);
#else
            RAISE_EXCEPTION(OLSocketException,
                @"Could not resolve host name - %s",
                hstrerror(h_errno));
#endif
        }
        canonicalName = objc_malloc(strlen(hent->h_name) + 1);
        strcpy(canonicalName, hent->h_name);
#endif
    }
    return canonicalName;
}

- (BOOL) isLocalWildcard
{
    return NO;
}

- (uint16_t) port
{
    SUBCLASS_MUST_IMPLEMENT;
    return 0;
}

- (int) socketType
{
    return socketType;
}

@end

@implementation OLInternet4Address

- (id) init
{
    return [self initWithPort: 0];
}

- (id) initWithPort: (uint16_t)port
{
    [super initImplWithName: NULL socketType: SOCK_STREAM];
    hostRep = objc_malloc(sizeof(struct sockaddr_in));
#if defined(OL_HAVE_SIN_LEN)
    hostRep->sin_len = INET_ADDRSTRLEN;
#endif
    hostRep->sin_family = AF_INET;
    hostRep->sin_port = H16_TO_N(port);
    hostRep->sin_addr.s_addr = INADDR_ANY;
    memset(hostRep->sin_zero, 0, sizeof(hostRep->sin_zero));
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    objc_free(hostRep);
    SUPER_FREE;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLInternet4Address alloc]
        initWithSockaddr: hostRep socketType: socketType];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLInternet4Address allocWithZone: zone]
        initWithSockaddr: hostRep socketType: socketType];
}
#endif

- (OLText*) description
{
    uint8_t* addr = (uint8_t*)&hostRep->sin_addr.s_addr;
    OLText* desc;
    char buf[256];
    

    if (canonicalName != NULL)
    {
        desc = [[OLText alloc] initWithBytes: (const uint8_t*)canonicalName
            count: strlen(canonicalName) encoding: OL_ASCII_ENCODING];
    }
    else
    {
        if (hostRep->sin_port == 0)
        {
            sprintf(buf, "%u.%u.%hu.%hu",
                addr[0], addr[1], addr[2], addr[3]);
        }
        else
        {
            sprintf(buf, "%hu.%hu.%hu.%hu:%hu",
                addr[0], addr[1], addr[2], addr[3],
                N16_TO_H(hostRep->sin_port));
        }
        desc = [[OLText alloc] initWithBytes: (const uint8_t*)buf
            count: strlen(buf) encoding: OL_ASCII_ENCODING];
    }
    return OBJ_AUTORELEASE(desc);
}

- (const struct sockaddr*) hostRepresentation
{
    return (const struct sockaddr*)hostRep;
}

- (unsigned) hostRepresentationLength
{
    return sizeof(struct sockaddr_in);
}

- (BOOL) isEqual: (id)object
{
    OLInternet4Address* other;

    if (IS_KIND_OF(object, OLInternet4Address))
    {
        other = (OLInternet4Address*)object;
        return other->hostRep->sin_port == hostRep->sin_port &&
               !memcmp(&other->hostRep->sin_addr, &hostRep->sin_addr, sizeof(struct in_addr));
    }
    return NO;
}

- (BOOL) isLocalWildcard
{
    return hostRep->sin_addr.s_addr == INADDR_ANY;
}

- (uint16_t) port
{
    return N16_TO_H(hostRep->sin_port);
}

@end

#if defined(OL_HAVE_INET6_SOCKETS)

@implementation OLInternet6Address

- (id) init
{
    return [self initWithPort: 0];
}

- (id) initWithPort: (uint16_t)port
{
    [super initImplWithName: NULL socketType: SOCK_STREAM];
    hostRep = objc_malloc(sizeof(struct sockaddr_in6));
    memset(hostRep, 0, sizeof(struct sockaddr_in6));
#if defined(OL_HAVE_SIN6_LEN)
    hostRep->sin6_len = INET6_ADDRSTRLEN;
#endif
    hostRep->sin6_family = AF_INET6;
    hostRep->sin6_port = H16_TO_N(port);
    memcpy(&hostRep->sin6_addr, &in6addr_any, sizeof(struct in6_addr));
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    objc_free(hostRep);
    SUPER_FREE;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLInternet6Address alloc]
        initWithSockaddr6: hostRep socketType: socketType];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLInternet6Address allocWithZone: zone]
        initWithSockaddr6: hostRep socketType: socketType];
}
#endif

- (OLText*) description;
{
    uint16_t* addr = (uint16_t*)hostRep->sin6_addr.s6_addr;
    OLText* desc;
    char buf[256];

    if (canonicalName != NULL)
    {
        desc = [[OLText alloc] initWithBytes: (const uint8_t*)canonicalName
            count: strlen(canonicalName) encoding: "ASCII"];
    }
    else
    {
        if (hostRep->sin6_port == 0)
        {
            sprintf(buf, "%X:%X:%X:%X:%X:%X:%X:%X",
                N16_TO_H(addr[0]), N16_TO_H(addr[1]),
                N16_TO_H(addr[2]), N16_TO_H(addr[3]),
                N16_TO_H(addr[4]), N16_TO_H(addr[5]),
                N16_TO_H(addr[6]), N16_TO_H(addr[7]));
        }
        else
        {
            sprintf(buf, "%X:%X:%X:%X:%X:%X:%X:%X.%i",
                N16_TO_H(addr[0]), N16_TO_H(addr[1]),
                N16_TO_H(addr[2]), N16_TO_H(addr[3]),
                N16_TO_H(addr[4]), N16_TO_H(addr[5]),
                N16_TO_H(addr[6]), N16_TO_H(addr[7]),
                N16_TO_H(hostRep->sin6_port));
        }
        desc = [[OLText alloc] initWithBytes: (const uint8_t*)buf
            count: strlen(buf) encoding: OL_ASCII_ENCODING];
    }
    return OBJ_AUTORELEASE(desc);
}

- (const struct sockaddr*) hostRepresentation
{
    return (const struct sockaddr*)hostRep;
}

- (unsigned) hostRepresentationLength
{
    return sizeof(struct sockaddr_in6);
}

- (BOOL) isEqual: (id)object
{
    OLInternet6Address* other;

    if (IS_KIND_OF(object, OLInternet6Address))
    {
        other = (OLInternet6Address*)object;
        return other->hostRep->sin6_port == hostRep->sin6_port &&
               !memcmp(&other->hostRep->sin6_addr, &hostRep->sin6_addr, sizeof(struct in6_addr));
    }
    return NO;
}

- (BOOL) isLocalWildcard
{
    return memcmp(&in6addr_any, &hostRep->sin6_addr, sizeof(struct in6_addr)) == 0;
}

- (uint16_t) port
{
    return N16_TO_H(hostRep->sin6_port);
}

@end

#endif

#if defined(OL_HAVE_UNIX_SOCKETS)

@implementation OLUnixAddress

+ (id) addressWithPath: (const char*)pth
{
    OL_BEGIN_AUTO_CTOR(OLUnixAddress)
        initWithPath: pth
    OL_END_AUTO_CTOR;
}

- (id) initWithPath: (const char*)pth
{
    char resolved[OL_PATH_MAX + 1];
#if defined(OL_REALPATH_SUCKS)
    char* namePart;
    char* withoutName;
    char* pathCopy1;
    char* pathCopy2;
#endif

    [super init];
    hostRep = objc_malloc(sizeof(struct sockaddr_un));
#if defined(OL_REALPATH_SUCKS)
    pathCopy1 = objc_malloc(strlen(pth) + 1);
    strcpy(pathCopy1, pth);
    pathCopy2 = objc_malloc(strlen(pth) + 1);
    strcpy(pathCopy2, pth);
    namePart = basename(pathCopy1);
    withoutName = dirname(pathCopy2);
    if (realpath(withoutName, resolved) == NULL)
    {
        RAISE_EXCEPTION(GENERIC_EXCEPTION,
            @"Could not resolve full path - %s", strerror(errno));
    }
    strcpy(hostRep->sun_path, resolved);
    strcat(hostRep->sun_path, "/");
    strcat(hostRep->sun_path, namePart);
    objc_free(pathCopy1);
    objc_free(pathCopy2);
#else
    if (realpath(pth, resolved) == NULL)
    {
        RAISE_EXCEPTION(GENERIC_EXCEPTION,
            @"Could not resolve full path - %s", strerror(errno));
    }
    strcpy(hostRep->sun_path, resolved);
#endif
    hostRep->sun_family = AF_UNIX;
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    objc_free(hostRep);
    SUPER_FREE;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLUnixAddress alloc] initWithPath: hostRep->sun_path];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLUnixAddress allocWithZone: zone] initWithPath: hostRep->sun_path];
}
#endif

- (OLText*) description;
{
    return OBJ_AUTORELEASE([[OLText alloc] initWithCString: hostRep->sun_path]);
}

- (const struct sockaddr*) hostRepresentation
{
    return (const struct sockaddr*)hostRep;
}

- (unsigned) hostRepresentationLength
{
    return OL_SUN_LEN(hostRep);
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLUnixAddress) &&
           strcmp(hostRep->sun_path, ((OLUnixAddress*)object)->hostRep->sun_path) == 0;
}

- (const char*) path
{
    return hostRep->sun_path;
}

@end

#endif

@implementation OLInternetAddress (PrivateMethods)

+ (OLInternetAddress*) addressImplWithHost: (const char*)name service: (const char*)service port: (int)port
{
    OLVector* all = [OLInternetAddress allAddressesImplWithHost: name service: service port: port];
    OLInternetAddress* first = OBJ_AUTORELEASE(OBJ_RETAIN([all front]));

    OBJ_FREE_AUTO(all);
    return first;
}

+ (OLVector*) allAddressesImplWithHost: (const char*)name service: (const char*)service port: (int)port
{
    OLVector* all = OBJ_AUTORELEASE([[OLVector alloc] init]);
    OLInternetAddress* found = nil;
    OLCompareInternetAddressesByType* pred;
    OLArrayIterator* vbegin;
    OLArrayIterator* vend;
#if defined(OL_HAVE_GETADDRINFO)
    struct addrinfo* returned;
    struct addrinfo* cur;
    int rc;

    rc = getaddrinfo(name, service, NULL, &returned);
    if (rc != 0)
    {
        RAISE_EXCEPTION(OLSocketException,
            @"Error looking up host %s - %s",
            name, gai_strerror(rc));
    }
    for (cur = returned; cur != NULL; cur = cur->ai_next)
    {
        if (cur->ai_socktype == SOCK_STREAM)
        {
            if (cur->ai_family == PF_INET)
            {
                found = [[OLInternet4Address alloc] initImplWithAddrinfo: cur port: port];
                [all pushBack: found];
                OBJ_RELEASE(found);
            }
#if defined(OL_HAVE_INET6_SOCKETS)
            else if (cur->ai_family == PF_INET6)
            {
                found = [[OLInternet6Address alloc] initImplWithAddrinfo: cur port: port];
                [all pushBack: found];
                OBJ_RELEASE(found);
            }
#endif
        }
    }
    freeaddrinfo(returned);
#elif defined(OL_HAVE_GETHOSTBYNAME)
    struct hostent* hent;
    struct servent* sent;
    struct servent localSent;

    hent = gethostbyname(name);
    if (hent == NULL)
    {
#if defined(OL_WINDOWS_SOCKETS)
        char msgBuf[8192];

        if (FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
                          NULL,
                          WSAGetLastError(),
                          0,
                          msgBuf,
                          8191,
                          NULL) == 0)
        {
            msgBuf[0] = 0;
        }
        RAISE_EXCEPTION(OLSocketException,
            @"Error looking up host %s - %s",
            name, msgBuf);
#else
        RAISE_EXCEPTION(OLSocketException,
            @"Error looking up host %s - %s",
            name, hstrerror(h_errno));
#endif
    }
    if (service != NULL)
    {
        sent = getservbyname(service, NULL);
        if (sent == NULL)
        {
            RAISE_EXCEPTION(OLSocketException,
                @"Error looking up service %s",
                service);
        }
    }
    else
    {
        sent = &localSent;
        sent->s_port = H16_TO_N(port);
    }
    if (hent->h_addrtype == AF_INET)
    {
        found = [[OLInternet4Address alloc] initImplWithHostent: hent andServent: sent];
        [all pushBack: found];
        OBJ_RELEASE(found);
    }
#if defined(OL_HAVE_INET6_SOCKETS)
    else if (hent->h_addrtype == AF_INET6)
    {
        found = [[OLInternet6Address alloc] initImplWithHostent: hent andServent: sent];
        [all pushBack: found];
        OBJ_RELEASE(found);
    }
#endif
#endif
    if ([all empty])
    {
        OBJ_FREE_AUTO(all);
        RAISE_EXCEPTION(OLSocketException,
            @"No addresses found for host %s", name);
    }
    if ([all size] > 1)
    {
        pred = [[OLCompareInternetAddressesByType alloc] init];
        vbegin = [all beginImpl];
        vend = [all endImpl];
        [OLAlgorithm stableSortFrom: vbegin to: vend predicate: pred];
        OBJ_RELEASE(vend);
        OBJ_RELEASE(vbegin);
        OBJ_RELEASE(pred);
    }
    return all;
}

- (id) initImplWithName: (const char*)name socketType: (int)type
{
    [super init];
    if (name != NULL)
    {
        canonicalName = objc_malloc(strlen(name) + 1);
        strcpy(canonicalName, name);
    }
    else
    {
        canonicalName = NULL;
    }
    socketType = type;
    return self;
}

- (int) addressFamily
{
    return 0;
}

@end

@implementation OLInternet4Address (PackageMethods)

- (id) initWithSockaddr: (const struct sockaddr_in*)addr socketType: (int)sockType
{
    [super initImplWithName: NULL socketType: sockType];
    hostRep = objc_malloc(sizeof(struct sockaddr_in));
    memcpy(hostRep, addr, sizeof(struct sockaddr_in));
    return self;
}

@end

@implementation OLInternet4Address (PrivateMethods)

#if defined(OL_HAVE_GETADDRINFO)

- (id) initImplWithAddrinfo: (const struct addrinfo*)info port: (int) port
{
    [super initImplWithName: info->ai_canonname socketType: info->ai_socktype];
    hostRep = objc_malloc(sizeof(struct sockaddr_in));
    memcpy(hostRep, info->ai_addr, sizeof(struct sockaddr_in));
    if (port >= 0)
        hostRep->sin_port = H16_TO_N(port);
    return self;
}

#elif defined(OL_HAVE_GETHOSTBYNAME)

- (id) initImplWithHostent: (const struct hostent*)hent andServent: (const struct servent*)sent
{
    [super initImplWithName: hent->h_name socketType: hent->h_addrtype];
    hostRep = malloc(sizeof(struct sockaddr_in));
#if defined(OL_HAVE_SIN_LEN)
    hostRep->sin_len = INET_ADDRSTRLEN;
#endif
    hostRep->sin_family = hent->h_addrtype;
    memcpy(&hostRep->sin_addr, hent->h_addr_list[0], hent->h_length);
    hostRep->sin_port = sent->s_port;
    memset(hostRep->sin_zero, 0, sizeof(hostRep->sin_zero));
    return self;
}

#endif

- (id) initLoopbackImplWithPort: (uint16_t)port
{
    [self initWithPort: port];
    hostRep->sin_addr.s_addr = INADDR_LOOPBACK;
    return self;
}

- (int) addressFamily
{
    return AF_INET;
}

@end

#if defined(OL_HAVE_INET6_SOCKETS)

@implementation OLInternet6Address (PackageMethods)

- (id) initWithSockaddr6: (const struct sockaddr_in6*)addr socketType: (int)sockType
{
    [super initImplWithName: NULL socketType: sockType];
    hostRep = objc_malloc(sizeof(struct sockaddr_in6));
    memcpy(hostRep, addr, sizeof(struct sockaddr_in6));
    return self;
}

@end

@implementation OLInternet6Address (PrivateMethods)

#if defined(OL_HAVE_GETADDRINFO)

- (id) initImplWithAddrinfo: (const struct addrinfo*)info port: (int)port
{
    [super initImplWithName: info->ai_canonname socketType: info->ai_socktype];
    hostRep = objc_malloc(sizeof(struct sockaddr_in6));
    memcpy(hostRep, info->ai_addr, sizeof(struct sockaddr_in6));
    if (port >= 0)
        hostRep->sin6_port = H16_TO_N(port);
    return self;
}

#elif defined(OL_HAVE_GETHOSTBYNAME)

- (id) initImplWithHostent: (const struct hostent*)hent andServent: (const struct servent*)sent
{
    [super initImplWithName: hent->h_name socketType: hent->h_addrtype];
    hostRep = objc_malloc(sizeof(struct sockaddr_in6));
    memset(hostRep, 0, sizeof(struct sockaddr_in6));
#if defined(OL_HAVE_SIN6_LEN)
    hostRep->sin6_len = INET6_ADDRSTRLEN;
#endif
    hostRep->sin6_family = hent->h_addrtype;
    memcpy(&hostRep->sin6_addr, hent->h_addr_list[0], hent->h_length);
    hostRep->sin6_port = sent->s_port;
    return self;
}

#endif

- (id) initLoopbackImplWithPort: (uint16_t)port
{
    [self initWithPort: port];
    memcpy(&hostRep->sin6_addr, &in6addr_loopback, sizeof(struct in6_addr));
    return self;
}

- (int) addressFamily
{
    return AF_INET6;
}

@end

#endif

#if defined(OL_HAVE_UNIX_SOCKETS)

@implementation OLUnixAddress (PackageMethods)

- (id) initWithSockaddrUnix: (const struct sockaddr_un*)addr
{
    [super init];
    hostRep = objc_malloc(sizeof(struct sockaddr_un));
    memcpy(hostRep, addr, sizeof(struct sockaddr_un));
    return self;
}

@end

#endif

@implementation OLCompareInternetAddressesByType

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
#if defined(OL_HAVE_INET6_SOCKETS)
    if (IS_KIND_OF(arg1, OLInternet4Address))
    {
        return (!__preferIPv6Addresses && IS_KIND_OF(arg2, OLInternet6Address))
            ? YES : NO;
    }
    else
    {
        return (__preferIPv6Addresses && IS_KIND_OF(arg2, OLInternet4Address))
            ? YES : NO;
    }
#else
    return YES;
#endif
}

@end
