//
// $Id: ReferenceCountedStream.h,v 1.8 2007/03/08 19:59:21 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(REFERENCECOUNTEDSTREAM_OL_GUARD)
#define REFERENCECOUNTEDSTREAM_OL_GUARD

#include <ObjectiveLib/ObjectBase.h>

/**
 * @class OLReferenceCountedStream ReferenceCountedStream.h ObjectiveLib/ReferenceCountedStream.h
 *
 * A base class for maintaining reference counts of streams. Due to the fact that it is not
 * always possible or safe to copy stream instances, it is necessary, in order to maintain
 * the desired usage model, to count references to streams and only release them when no
 * one is watching anymore. This base class just maintains a simple reference count,
 * incrementing it on #init and #copy and decrementing it on #free, releasing the stream's
 * resources when the reference count drops to zero.
 *
 * @ingroup Streams
 */
@interface OLReferenceCountedStream : Object
{
@protected
    /**
     * The reference count of this stream
     */
    unsigned referenceCount;
}

/**
 * @name Initializers and Deallocators
 */
/* @{ */

/**
 * Initialize the stream with a reference count of one.
 *
 * @return a reference to this stream
 */
- (id) init;

/**
 * Decrement the reference count. If the reference count falls to zero, then the
 * stream's #freeStreamResources method will be called and the stream will be
 * forever free.
 *
 * @return a reference to this stream
 */
- (id) free;
/* @} */

/**
 * Make a copy of this stream. The reference count is incremented and a refernce to
 * this stream is returned.
 *
 * @return a reference to this stream
 */
- (id) copy;

/**
 * Free the stream's resources. This secondary freeing method is necessary to allow
 * the reference count to be properly manipulated from within the normal #free method.
 *
 * @note All sublcasses that allocate resources must implement this method to free
 * those resources and call [super freeStreamResources].
 */
- (void) freeStreamResources;

@end

#endif
