//
// $Id: GzipOutStream.m,v 1.13 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "GzipOutStream.h"
#import "OutStreamPackage.h"
#import "ZlibOutStreamPackage.h"
#import "Macros.h"
#import "ConfigPrivate.h"
#import "RunTime.h"
#import "ByteOrder.h"
#import "DataOutStream.h"
#import "Exception.h"
#import "Text.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSString.h>
#import <Foundation/NSPathUtilities.h>
#import <Foundation/NSData.h>
#import <Foundation/NSException.h>
#endif
#import <sys/types.h>
#import <sys/stat.h>
#import <zlib.h>
#import <string.h>
#import <limits.h>
#include <time.h>
#import <libgen.h>
#import <stdlib.h>

#define FREE_MY_RESOURCES \
    [self close]; \
    objc_free(commentData); \
    objc_free(fileNameData); \
    OBJ_RELEASE(extraFields)

@interface OLGzipOutStream (PrivateMethods)

- (void) writeHeader;
- (void) writeLE16: (OLOutStream*)strm value: (uint16_t)val;
- (void) writeLE32: (OLOutStream*)strm value: (uint32_t)val;

@end

@implementation OLGzipOutStream

+ (id) streamWithOutStream: (OLOutStream*)underStream
{
    OL_BEGIN_AUTO_CTOR(OLGzipOutStream)
        initWithOutStream: underStream
    OL_END_AUTO_CTOR;
}

+ (id) streamWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level
{
    OL_BEGIN_AUTO_CTOR(OLGzipOutStream)
        initWithOutStream: underStream compressionLevel: level
    OL_END_AUTO_CTOR;
}

- (id) initWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level bufferSize: (unsigned)size writeZlibHeader: (BOOL)zlibHeader
{
    [super initWithOutStream: underStream compressionLevel: level
        bufferSize: size writeZlibHeader: NO];
    crc = crc32(0, Z_NULL, 0);
    commentData = NULL;
    fileNameData = NULL;
    modTime = 0;
    extraFields = nil;
    headerWritten = NO;
    trailerWritten = NO;
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc
{
	FREE_MY_RESOURCES;
	SUPER_FREE;
}
#endif

- (void) addExtraField: (const char*)identifier withBytes: (const uint8_t*)buf count: (unsigned)num
{
    if (strlen(identifier) != 2)
    {
        RAISE_EXCEPTION(OLInputOutputException, @"Illegal extra field identifier");
    }
    if (extraFields == nil)
        extraFields = [[OLDataOutStream alloc] init];
    [extraFields writeBytes: (const uint8_t*)identifier count: 2];
    [self writeLE16: extraFields value: num];
    [extraFields writeBytes: buf count: num];
}

- (void) close
{
    if (!trailerWritten)
    {
        [self finish];
        [self writeLE32: stream value: crc];
        [self writeLE32: stream value: zstream->total_in];
        [super close];
        trailerWritten = YES;
    }
}

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources
{
	FREE_MY_RESOURCES;
	[super freeStreamResources];
}
#endif

- (void) markModificationTime
{
    modTime = time(NULL);
    if (modTime == UINT_MAX)
        modTime = 0;
}

- (void) setComment: (const char*)comment
{
    unsigned len = strlen(comment);

    objc_free(commentData);
    commentData = NULL;
    if (len != 0)
    {
        commentData = objc_malloc(len + 1);
        strcpy((char*)commentData, comment);
    }
}

- (void) setCommentText: (OLText*)comment
{
    objc_free(commentData);
    commentData = NULL;
    if ([comment length] != 0)
        commentData = [comment nullTerminatedBytesWithEncoding: "ISO-8859-1"];
}

- (void) setOriginalFileName: (const char*)name includeModificationTime: (BOOL)mtime
{
    struct stat statBuf;
    char* nameCopy;
    char* base;

    objc_free(fileNameData);
    fileNameData = NULL;
    modTime = 0;
    nameCopy = objc_malloc(strlen(name) + 1);
    strcpy(nameCopy, name);
    base = basename(nameCopy);
    fileNameData = objc_malloc(strlen(base) + 1);
    strcpy((char*)fileNameData, base);
    objc_free(nameCopy);
    if (mtime && stat(name, &statBuf) == 0)
        modTime = statBuf.st_mtime;
}

- (void) setOriginalFileNameText: (OLText*)name includeModificationTime: (BOOL)mtime
{
    uint8_t* bytes = [name nullTerminatedBytesWithEncoding: "ISO-8859-1"];
    [self setOriginalFileName: (char*)bytes includeModificationTime: mtime];
    objc_free(bytes);
}

- (unsigned) writeBytes: (const uint8_t*)bytes count: (unsigned)num
{
    unsigned written;

    if (!headerWritten)
    {
        [self writeHeader];
        headerWritten = YES;
    }
    written = [super writeBytes: bytes count: num];
    crc = crc32(crc, bytes, written);
    return written;
}

@end

@implementation OLGzipOutStream (PrivateMethods)

- (void) writeHeader
{
    //
    // Note: As much as you may want to, you can't include the header CRC here.
    // Older versions of gzip will choke. Believe me. I've tried it.
    //
    uint8_t flags = 0;

    [stream writeByte: GZ_ID1];
    [stream writeByte: GZ_ID2];
    [stream writeByte: Z_DEFLATED];
    if (fileNameData != NULL)
        flags |= GZ_FNAME;
    if (commentData != NULL)
        flags |= GZ_FCOMMENT;
    if (extraFields != nil)
        flags |= GZ_FEXTRA;
    [stream writeByte: flags];
    [self writeLE32: stream value: modTime];
    [stream writeByte: 0];
    [stream writeByte: GZ_OS_UNKNOWN];
    if (extraFields != nil)
    {
        [self writeLE16: stream value: [extraFields count]];
        [stream writeBytes: [extraFields bytes] count: [extraFields count]];
    }
    if (fileNameData != NULL)
        [stream writeBytes: fileNameData count: strlen((const char*)fileNameData) + 1];
    if (commentData != NULL)
        [stream writeBytes: commentData count: strlen((const char*)commentData) + 1];
}

- (void) writeLE16: (OLOutStream*)strm value: (uint16_t)val
{
    uint16_t swapped = H16_TO_L(val);

    [strm completelyWriteBytes: (uint8_t*)&swapped count: 2];
}

- (void) writeLE32: (OLOutStream*)strm value: (uint32_t)val
{
    uint32_t swapped = H32_TO_L(val);

    [strm completelyWriteBytes: (uint8_t*)&swapped count: 4];
}

@end
