//
// $Id: BzlibInStream.m,v 1.11 2007/04/16 01:48:15 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "BzlibInStream.h"
#import "Macros.h"
#import "RunTime.h"
#import "Exception.h"
#if defined(OL_NO_OPENSTEP)
#import "Text.h"
#else
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#endif
#import <bzlib.h>
#import <limits.h>
#import <stdlib.h>

#define FREE_MY_RESOURCES \
    [self close]; \
    objc_free(buffer)

extern void* __ol_bzalloc(void*, int, int);
extern void __ol_bzfree(void*, void*);

@interface OLBzlibInStream (PrivateMethods)

- (void) fillBuffer;

@end

@implementation OLBzlibInStream

+ (id) streamWithInStream: (OLInStream*)underStream
{
    OL_BEGIN_AUTO_CTOR(OLBzlibInStream)
        initWithInStream: underStream
    OL_END_AUTO_CTOR;
}

- (id) initWithInStream: (OLInStream*)underStream
{
    return [self initWithInStream: underStream bufferSize: 1024 smallMode: NO];
}

- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size
{
    return [self initWithInStream: underStream bufferSize: size smallMode: NO];
}

- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size smallMode: (BOOL)sm
{
    [super initWithInStream: underStream];
    bzstream = objc_malloc(sizeof(bz_stream));
    BZSTREAM->bzalloc = __ol_bzalloc;
    BZSTREAM->bzfree = __ol_bzfree;
    BZSTREAM->opaque = NULL;
    if (BZ2_bzDecompressInit(BZSTREAM, 0, (sm ? 1 : 0)) != BZ_OK)
    {
        objc_free(bzstream);
        bzstream = NULL;
        buffer = NULL;
        bufferSize = 0;
        RAISE_EXCEPTION(OLInputOutputException, @"Error initializing for decompression");
    }
    buffer = objc_malloc(size);
    bufferSize = size;
    BZSTREAM->avail_in = 0;
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) dealloc
{
    FREE_MY_RESOURCES;
    SUPER_FREE;
}
#endif

- (void) close
{
    if (bzstream != NULL)
    {
        BZ2_bzDecompressEnd(BZSTREAM);
        objc_free(bzstream);
        bzstream = NULL;
    }
    [stream close];
}

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources
{
    FREE_MY_RESOURCES;
    [super freeStreamResources];
}
#endif

- (unsigned) readBytes: (uint8_t*)dest count: (unsigned)max
{
    int rc;

    if (max != 0)
    {
        BZSTREAM->next_out = (char*)dest;
        BZSTREAM->avail_out = max;
        do
        {
            if (BZSTREAM->avail_in == 0)
            {
                [self fillBuffer];
                if (BZSTREAM->avail_in == 0)
                    return UINT_MAX;
            }
            rc = BZ2_bzDecompress(BZSTREAM);
            if (rc != BZ_OK)
            {
                if (rc == BZ_STREAM_END)
                {
                    if (BZSTREAM->avail_out == max)
                        return UINT_MAX;
                    break;
                }
                else
                {
                    RAISE_EXCEPTION(OLInputOutputException, @"Error decompressing");
                }
            }
        } while (BZSTREAM->avail_out != 0);
    }
    return max - BZSTREAM->avail_out;
}

@end

@implementation OLBzlibInStream (PrivateMethods)

- (void) fillBuffer
{
    unsigned cur = 0;
    unsigned total = 0;

    while (total < bufferSize)
    {
        cur = [stream readBytes: buffer + total count: bufferSize - total];
        if (cur == UINT_MAX)
            break;
        total += cur;
    }
    BZSTREAM->next_in = (char*)buffer;
    BZSTREAM->avail_in = total;
}

@end
