/*
 *  Copyright (C) 2002 Luca Deri <deri@ntop.org>
 *                      
 *  			  http://www.ntop.org/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#include "nprobe.h"
#include "version.c"


/* ************************************ */

HashBucket *theHash;
u_int bucketsLeft, globalFlowSequence=0, hashSize;
time_t initialSniffTime;
int idleTimeout = DUMP_TIMEOUT;
int lifetimeTimeout = 2*DUMP_TIMEOUT;
int scanCycle   = 2*DUMP_TIMEOUT;
#ifdef USE_SYSLOG
char nprobeId[255+1];
#endif
time_t lastExportTime;

/* ************************************ */

/*
 * A faster replacement for inet_ntoa().
 */
char* _intoa(struct in_addr addr, char* buf, u_short bufLen) {
  char *cp, *retStr;
  u_int byte;
  int n;

  cp = &buf[bufLen];
  *--cp = '\0';

  n = 4;
  do {
    byte = addr.s_addr & 0xff;
    *--cp = byte % 10 + '0';
    byte /= 10;
    if (byte > 0) {
      *--cp = byte % 10 + '0';
      byte /= 10;
      if (byte > 0)
	*--cp = byte + '0';
    }
    *--cp = '.';
    addr.s_addr >>= 8;
  } while (--n > 0);

  /* Convert the string to lowercase */
  retStr = (char*)(cp+1);

  return(retStr);
}

/* ****************************************************** */

void addPktToHash(u_short proto,
		  struct in_addr src,
		  u_short sport,
		  struct in_addr dst,
		  u_short dport,
		  u_int  len,
		  time_t actTime,
		  u_int8_t flags) {

  u_int i, n, idx = (src.s_addr+dst.s_addr) % hashSize;

  accessMutex(&theMutex, "addPktToHash");

  for(i = idx, n = 0; n < hashSize; n++) {
    if((theHash[i].proto == proto)
       && (((theHash[i].src.s_addr == src.s_addr) && (theHash[i].dst.s_addr == dst.s_addr)
	    && (theHash[i].sport == sport)  && (theHash[i].dport == dport))
	   || ((theHash[i].src.s_addr == dst.s_addr) && (theHash[i].dst.s_addr == src.s_addr)
	       && (theHash[i].sport == dport)  && (theHash[i].dport == sport)))) {
      if(theHash[i].src.s_addr == src.s_addr) {
	theHash[i].bytesSent += len, theHash[i].pktSent++;
	theHash[i].lastSeenSent = actTime;
	theHash[i].tcpFlags |= flags;
      } else {
	theHash[i].bytesRcvd += len, theHash[i].pktRcvd++;
	if(theHash[i].firstSeenRcvd == 0) theHash[i].firstSeenRcvd = actTime;
	theHash[i].lastSeenRcvd = actTime;
	theHash[i].tcpFlags |= flags;
      }
      break;
    } else if((theHash[i].proto == 0)
	      && (theHash[i].src.s_addr == 0)
	      && (theHash[i].dst.s_addr == 0)) {
      theHash[i].src.s_addr = src.s_addr, theHash[i].dst.s_addr = dst.s_addr;
      theHash[i].proto = proto;
      theHash[i].sport = sport, theHash[i].dport = dport;
      theHash[i].firstSeenSent = theHash[i].lastSeenSent = actTime;
      theHash[i].bytesSent += len, theHash[i].pktSent++;
      theHash[i].tcpFlags |= flags;
      bucketsLeft--;
      break;
    }

    i = (i+1) % hashSize;
  }

  if(n == hashSize) {
    traceEvent(TRACE_WARNING, "WARNING: hash full");
  }

  releaseMutex(&theMutex);
}

/* ****************************************************** */

void processPacket(u_char *_deviceId,
		   const struct pcap_pkthdr *h,
		   const u_char *p) {
  struct ether_header ehdr;
  u_int caplen = h->caplen;
  u_int length = h->len;
  unsigned short eth_type;
  u_int8_t flags = 0;
  struct ip ip;
  struct tcphdr tp;
  struct udphdr up;

  if(caplen >= sizeof(struct ether_header)) {
    memcpy(&ehdr, p, sizeof(struct ether_header));
    eth_type = ntohs(ehdr.ether_type);

    if(eth_type == ETHERTYPE_IP) {
      u_int plen, hlen;
      u_short sport, dport;

      memcpy(&ip, p+sizeof(struct ether_header), sizeof(struct ip));
      hlen = (u_int)ip.ip_hl * 4;
      NTOHL(ip.ip_dst.s_addr); NTOHL(ip.ip_src.s_addr);

      plen = length-sizeof(struct ether_header);

      switch(ip.ip_p) {
      case IPPROTO_TCP:
	if(plen < (hlen+sizeof(struct tcphdr))) return; /* packet too short */
	memcpy(&tp, p+sizeof(struct ether_header)+hlen, sizeof(struct tcphdr));
	sport = ntohs(tp.th_sport), dport = ntohs(tp.th_dport);
	flags = tp.th_flags;
	break;
      case IPPROTO_UDP:
	if(plen < (hlen+sizeof(struct udphdr))) return; /* packet too short */
	memcpy(&up, p+sizeof(struct ether_header)+hlen, sizeof(struct udphdr));
	sport = ntohs(up.uh_sport), dport = ntohs(up.uh_dport);
	break;
      default:
	sport = dport = 0;
      }


#ifdef DEBUG
      printf("%s->%s [len=%d]\n",
	     _intoa(ip.ip_src, buf, sizeof(buf)),
	     _intoa(ip.ip_dst, buf1, sizeof(buf1)),
	     plen);
#endif

      addPktToHash(ip.ip_p, ip.ip_src, sport,
		   ip.ip_dst, dport, plen,
		   h->ts.tv_sec, flags);
    }
  }
}

/* ****************************************************** */

void usage() {
  printf("Welcome to nprobe v.%s for %s\n"
	 "Built on %s\n"
	 "Copyright 2002 by Luca Deri <deri@ntop.org>\n\n", 
	 version, osName, buildDate);

#ifdef DEMO_MODE
  printf("************************************************************\n");
  printf("NOTE: This is a DEMO version limited to %d flows export.\n", MAX_DEMO_FLOWS);
  printf("************************************************************\n\n");
#endif

  printf("Usage: nprobe -n <host:port> [-i <interface>] [-t <dump timeout>]\n" 
	 "              [-d <idle timeout>] [-f <BPF filter>]\n"
#ifdef USE_SYSLOG
	 "              [-I <probe name>]\n"
#endif
	 "              [-w <hash size>]\n"
	 "\n\n");
  printf("-n <host:port>    | Address of the NetFlow collector\n");
#ifdef WIN32
  printf("-i <interface>    | Interface name from which packets are captured\n");
#else
  printf("-i <interface>    | Index of the interface from which packets are captured\n");
#endif

  printf("-t <dump timeout> | It specifies the maximum (seconds) flow lifetime\n"
	 "                  | [default=%d]\n", lifetimeTimeout);
  printf("-d <idle timeout> | It specifies the maximum (seconds) flow idle lifetime\n"
         "                  | [default=%d]\n", idleTimeout);
  printf("-s <scan cycle>   | It specifies how often (seconds) expired flows are emitted\n"
	 "                  | [default=%d]\n", scanCycle);
  printf("-f <BPF filter>   | BPF filter used to select captured packets\n"
	 "                  | [default=no filter]\n");
#ifdef USE_SYSLOG
  printf("-I <probe name>   | Log to syslog as <probe name>\n"
	 "                  | [default=stdout]\n");
#endif  
  printf("-w <hash size>    | Flows hash size\n"
	 "                  | [default=%d]\n", hashSize);
#ifdef WIN32
  {
    char *ifName = printAvailableInterfaces(-1);
  }
#endif

  exit(0);
}

/* ****************************************************** */

void initNetFlow(char* addr, int port) {
  int sockopt = 1;
  struct hostent *hostAddr = gethostbyname(addr);
  char buf[64];
  struct in_addr dstAddr;

  if(hostAddr == NULL) {
    traceEvent(TRACE_INFO, "Unable to resolve address '%s'\n", addr);
    exit(-1);
  }
  memcpy(&dstAddr.s_addr, hostAddr->h_addr_list[0], hostAddr->h_length);
  netFlowDest.sin_addr.s_addr = dstAddr.s_addr;
  netFlowDest.sin_family      = AF_INET;
  netFlowDest.sin_port        = (int)htons(port);

  netFlowOutSocket = socket(AF_INET, SOCK_DGRAM, 0);
  setsockopt(netFlowOutSocket, SOL_SOCKET, SO_REUSEADDR, (char *)&sockopt, sizeof(sockopt));

  dstAddr.s_addr = ntohl(dstAddr.s_addr);
  traceEvent(TRACE_INFO, "Exporting NetFlow V5 flows towards %s:%d", 
	     _intoa(dstAddr, buf, sizeof(buf)), port);
}

/* ****************************************************** */

void sendNetFlow(NetFlow5Record *theFlow) {
  int rc, len;

#ifdef DEMO_MODE
  if(globalFlowSequence > MAX_DEMO_FLOWS) return;
#endif

  len = (ntohs(theFlow->flowHeader.count)*sizeof(struct flow_ver5_rec)
	 +sizeof(struct flow_ver5_hdr));

  rc = sendto(netFlowOutSocket,
	      (void*)theFlow, len,
	      0, (struct sockaddr *)&netFlowDest,
	      sizeof(netFlowDest));

  if(rc != len) {
    traceEvent(TRACE_INFO, "Error while exporting flows (%s)", strerror(errno));
  }
}

/* ****************************************************** */

void initNetFlowHeader(NetFlow5Record *theFlow) {
  time_t actTime = time(NULL);

  memset(theFlow, 0, sizeof(NetFlow5Record));

  theFlow->flowHeader.version        = htons(5);
  theFlow->flowHeader.sysUptime      = htonl((actTime-initialSniffTime)*1000);
  theFlow->flowHeader.unix_secs      = htonl(actTime);
  theFlow->flowHeader.unix_nsecs     = htonl(0);
  theFlow->flowHeader.flow_sequence  = htonl(globalFlowSequence);
  theFlow->flowHeader.engine_type    = 0;
  theFlow->flowHeader.engine_id      = 0;
}

/* ****************************************************** */

void hash2NetFlow() {
  int i, numFlow = 0, totFlows = 0, numExports = 0;
  NetFlow5Record theFlow;
  time_t theTime = time(NULL);
  time_t expireTime, idleTime;

  expireTime = theTime-lifetimeTimeout;
  idleTime   = theTime-idleTimeout;

  initNetFlowHeader(&theFlow);

  for(i = 0; i<hashSize; i++) {
    if((theHash[i].src.s_addr != 0) && (theHash[i].dst.s_addr != 0)
       && ((theHash[i].lastSeenSent < idleTime)       /* flow expired: data not sent for a while */
	   || (theHash[i].firstSeenSent < expireTime) /* flow expired: flow active but too old   */)) {
      accessMutex(&theMutex, "hash2NetFlow");
      theFlow.flowRecord[numFlow].srcaddr  = htonl(theHash[i].src.s_addr);
      theFlow.flowRecord[numFlow].dstaddr  = htonl(theHash[i].dst.s_addr);
      theFlow.flowRecord[numFlow].input    = htons(0);
      theFlow.flowRecord[numFlow].output   = htons(255 /* unknown device */);
      theFlow.flowRecord[numFlow].dPkts    = htonl(theHash[i].pktSent);
      theFlow.flowRecord[numFlow].dOctets  = htonl(theHash[i].bytesSent);
      theFlow.flowRecord[numFlow].First    = htonl((theHash[i].firstSeenSent-initialSniffTime)*1000);
      theFlow.flowRecord[numFlow].Last     = htonl((theHash[i].lastSeenSent-initialSniffTime)*1000);
      theFlow.flowRecord[numFlow].srcport  = htons(theHash[i].sport);
      theFlow.flowRecord[numFlow].dstport  = htons(theHash[i].dport);
      theFlow.flowRecord[numFlow].prot     = theHash[i].proto;
      theFlow.flowRecord[numFlow].tcp_flags= theHash[i].tcpFlags;
      numFlow++, totFlows++, globalFlowSequence++;

      if(numFlow == V5FLOWS_PER_PAK) {
	theFlow.flowHeader.count = htons(numFlow);
	sendNetFlow(&theFlow);
	initNetFlowHeader(&theFlow);
	numFlow = 0, totExports++, numExports++;
      }

      if(theHash[i].pktRcvd > 0) {
	theFlow.flowRecord[numFlow].srcaddr  = htonl(theHash[i].dst.s_addr);
	theFlow.flowRecord[numFlow].dstaddr  = htonl(theHash[i].src.s_addr);
	theFlow.flowRecord[numFlow].input    = htons(0);
	theFlow.flowRecord[numFlow].output   = htons(255 /* unknown device */);
	theFlow.flowRecord[numFlow].dPkts    = htonl(theHash[i].pktRcvd);
	theFlow.flowRecord[numFlow].dOctets  = htonl(theHash[i].bytesRcvd);
	theFlow.flowRecord[numFlow].First    = htonl((theHash[i].firstSeenSent-initialSniffTime)*1000);
	theFlow.flowRecord[numFlow].Last     = htonl((theHash[i].lastSeenSent-initialSniffTime)*1000);
	theFlow.flowRecord[numFlow].srcport  = htons(theHash[i].dport);
	theFlow.flowRecord[numFlow].dstport  = htons(theHash[i].sport);
	theFlow.flowRecord[numFlow].prot     = theHash[i].proto;
	theFlow.flowRecord[numFlow].tcp_flags= theHash[i].tcpFlags;
	numFlow++, totFlows++, globalFlowSequence++;
      }

      memset(&theHash[i], 0, sizeof(HashBucket));
      bucketsLeft++;
      releaseMutex(&theMutex);

      if(numFlow == V5FLOWS_PER_PAK) {
	theFlow.flowHeader.count = htons(numFlow);
	sendNetFlow(&theFlow);
	initNetFlowHeader(&theFlow);
	numFlow = 0, totExports++, numExports++;
      }
    }
  }

  if(numFlow > 0) {
    /* Send remaining flows */
    theFlow.flowHeader.count = htons(numFlow);
    sendNetFlow(&theFlow);
    numFlow = 0, totExports++, numExports++;
  }

  traceEvent(TRACE_INFO, "Exported %d flows in %d (of %llu) exports.",
	     totFlows, numExports, totExports);

  /* Every five minutes we should give a life sign */
  if (theTime > lastExportTime + 300) {
    lastExportTime=theTime;	  
    traceEvent(TRACE_INFO, "Exported %llu packets.", totExports);
  }
}

/* ****************************************************** */

void* dumpHashes(void* unused) {
  while(1) {
    nprobe_sleep(scanCycle);        
    hash2NetFlow(); /* Hash dump */
  }  
}

/* ****************************************************** */

int main(int argc, char *argv[]) {
  char ebuf[PCAP_ERRBUF_SIZE];
  char *tmpDev = NULL, *netFilter = NULL;
  pcap_t *pcapPtr;
  int opt, mandatoryParamOk=0;
  pthread_t dumpThreadId;
  char *addr=NULL, *port=NULL;
#ifdef WIN32
  int optind=0;
#endif

  /* #define TIME_PROTECTION */

#ifdef TIME_PROTECTION
  struct tm expireDate;

#define EXPIRE_DAY    19
#define EXPIRE_MONTH  9
#define EXPIRE_YEAR   2002

  memset(&expireDate, 0, sizeof(expireDate));
  expireDate.tm_mday = EXPIRE_DAY;
  expireDate.tm_mon  = EXPIRE_MONTH-1;
  expireDate.tm_year = EXPIRE_YEAR-1900;

  if(time(NULL) > mktime(&expireDate)) {
    printf("Sorry: this copy of nProbe is expired.\n");
    exit(0);
  }
#endif


#ifdef WIN32
  initWinsock32();
#endif

  hashSize = HASH_SIZE;
  initialSniffTime = time(NULL);

  optarg = NULL;
  while((opt = getopt(argc, argv, "i:t:hn:f:d:s:I:w:")) != EOF) {
    switch (opt) {
    case 'i':
#ifdef WIN32
      tmpDev = printAvailableInterfaces(atoi(optarg));
#else
      tmpDev = strdup(optarg);
#endif
      break;
    case 't':
      lifetimeTimeout = atoi(optarg);
      break;
    case 'd':
      idleTimeout = atoi(optarg);
      break;
    case 's':
      scanCycle = atoi(optarg);
      break;
    case 'f':
      netFilter = strdup(optarg);
      break;
    case 'w':
      hashSize = atoi(optarg);
      if(hashSize < HASH_SIZE) {
	hashSize = HASH_SIZE;
	traceEvent(TRACE_INFO, "Minimum hash size if %d.", hashSize);
      }
      break;
#ifdef USE_SYSLOG
    case 'I':
      { 
	int len = sizeof(nprobeId);
	if(len >= (sizeof(nprobeId)-1)) len = sizeof(nprobeId);
	strncpy(nprobeId, optarg, len);
	nprobeId[len] = '\0';
	useSyslog = 1;
      }
      break;
#endif
    case 'n':
      {
	addr = strtok(optarg, ":");
	if(addr != NULL) {
	  port = strtok(NULL, ":");
	  if(port == NULL) usage();
	} else
	  usage();

	mandatoryParamOk++;
      }
      break;
    default:
      usage();
      break;
    }
  }

  if(mandatoryParamOk == 0) usage();

#ifndef WIN32
  if(useSyslog)
    openlog(nprobeId, LOG_PID ,LOG_DAEMON);
#endif


  theHash = (HashBucket*)calloc(sizeof(HashBucket), hashSize);
  bucketsLeft = hashSize;

  initNetFlow(addr, atoi(port));

  if(tmpDev == NULL) {
    tmpDev = pcap_lookupdev(ebuf);
    if(tmpDev == NULL) {
      traceEvent(TRACE_ERROR, "Unable to locate default interface (%s)\n", ebuf);
      exit(-1);
    }
  }

  pcapPtr = pcap_open_live(tmpDev, DEFAULT_SNAPLEN, 1 /* promiscous */, 100 /* ms */, ebuf);

  if(netFilter != NULL) {
    struct bpf_program fcode;
    struct in_addr netmask;

    netmask.s_addr = htonl(0xFFFFFF00);

    if((pcap_compile(pcapPtr, &fcode, netFilter, 1, netmask.s_addr) < 0)
       || (pcap_setfilter(pcapPtr, &fcode) < 0)) {
      traceEvent(TRACE_ERROR, "Unable to set filter %s (%s)\n", netFilter, ebuf);
    } else
      traceEvent(TRACE_INFO, "Packet capture filter set to \"%s\"", netFilter);
  }

  if(pcapPtr == NULL) {
    traceEvent(TRACE_ERROR, "Unable to open interface %s (%s)\n", tmpDev, ebuf);
    exit(-1);
  }

  traceEvent(TRACE_INFO, "The hash has %d buckets", hashSize);
  traceEvent(TRACE_INFO, "Flows older than %d seconds will be exported", lifetimeTimeout);
  traceEvent(TRACE_INFO, "Flows inactive for at least %d seconds will be exported", idleTimeout);
  traceEvent(TRACE_INFO, "Expired flows will be checked once every %d seconds", scanCycle);

  createMutex(&theMutex);
  createThread(&dumpThreadId, dumpHashes, NULL);

  traceEvent(TRACE_INFO, "Capturing packets from interface %s", tmpDev);
  pcap_loop(pcapPtr, -1, processPacket, NULL); /* loop 4ever */

  close(netFlowOutSocket);

  killThread(&dumpThreadId);
  deleteMutex(&theMutex);

  pcap_close(pcapPtr);
	
  free(theHash);

#ifndef WIN32
  if(useSyslog)
    closelog();
#endif

  return(0);
}

/************************************************************************************ */

void traceEvent(int eventTraceLevel, char* file,
		int line, char * format, ...) {
  va_list va_ap;
  va_start (va_ap, format);

  /* Fix courtesy of "Burton M. Strauss III" <BStrauss@acm.org> */
  if(eventTraceLevel <= traceLevel) {
    char buf[BUF_SIZE];
    char theDate[32];
    time_t theTime = time(NULL);    

    /* We have two paths - one if we're logging, one if we aren't
     *   Note that the no-log case is those systems which don't support it (WIN32),
     *                                those without the headers !defined(USE_SYSLOG)
     *                                those where it's parametrically off...
     */

    memset(buf, 0, BUF_SIZE);
    strftime(theDate, 32, "%d/%b/%Y %H:%M:%S", localtime(&theTime));

#if defined(WIN32) || !defined(USE_SYSLOG)

#ifdef DEBUG
    printf("%s [%s:%d] ", theDate, file, line);
#else
    printf("%s ", theDate);
#endif


#if defined(WIN32)
    /* Windows lacks vsnprintf */
    vsprintf(buf, format, va_ap);
#else /* WIN32 - vsnprintf */
    vsnprintf(buf, BUF_SIZE-1, format, va_ap);
#endif /* WIN32 - vsnprintf */

    printf("%s%s", buf, (format[strlen(format)-1] != '\n') ? "\n" : "");
    fflush(stdout);
#else /* !WIN32 */

    vsnprintf(buf, BUF_SIZE-1, format, va_ap);

#if defined(USE_SYSLOG)
    if(useSyslog)
      syslog(eventTraceLevel, "[%s:%d] %s", file, line, buf);
#endif

    if(!useSyslog) {
#ifdef DEBUG
      printf("%s [%s:%d] ", theDate, file, line);
#else
      printf("%s ", theDate);
#endif
      printf("%s%s", buf, (format[strlen(format)-1] != '\n') ? "\n" : "");
      fflush(stdout);
    }
#endif /* WIN32 || !USE_SYSLOG */
  }

  va_end (va_ap);
}

/* *********** MULTITHREAD STUFF *********** */

#ifndef WIN32
int createThread(pthread_t *threadId,
		 void *(*__start_routine) (void *),
		 char* userParm) {
  int rc;

  rc = pthread_create(threadId, NULL, __start_routine, userParm);
  return(rc);
}

/* ************************************ */

void killThread(pthread_t *threadId) {
  pthread_detach(*threadId);
}

/* ************************************ */

int _createMutex(PthreadMutex *mutexId, char* fileName, int fileLine) {
  int rc;

  memset(mutexId, 0, sizeof(PthreadMutex));

  rc = pthread_mutex_init(&(mutexId->mutex), NULL);

  if (rc != 0) {
    traceEvent(TRACE_ERROR,
               "ERROR: createMutex() call returned %d(%d) [%s:%d]\n",
               rc, errno, fileName, fileLine);
  } else {
    mutexId->isInitialized = 1;
  }

  return(rc);
}

/* ************************************ */

void _deleteMutex(PthreadMutex *mutexId, char* fileName, int fileLine) {
  if(mutexId == NULL) {
    traceEvent(TRACE_ERROR,
	       "ERROR: deleteMutex() call with a NULL mutex [%s:%d]\n",
	       fileName, fileLine);
    return;
  }

  if(!mutexId->isInitialized) {
    traceEvent(TRACE_ERROR,
	       "ERROR: deleteMutex() call with an UN-INITIALIZED mutex [%s:%d]\n",
	       fileName, fileLine);
    return;
  }

  pthread_mutex_unlock(&(mutexId->mutex));
  pthread_mutex_destroy(&(mutexId->mutex));

  memset(mutexId, 0, sizeof(PthreadMutex));
}

/* ************************************ */

int _accessMutex(PthreadMutex *mutexId, char* where,
		 char* fileName, int fileLine) {
  int rc;

  if(mutexId == NULL) {
    traceEvent(TRACE_ERROR,
	       "ERROR: accessMutex() call with a NULL mutex [%s:%d]\n",
	       fileName, fileLine);
    return(-1);
  }

  if(!mutexId->isInitialized) {
    traceEvent(TRACE_ERROR,
	       "ERROR: accessMutex() call with an UN-INITIALIZED mutex [%s:%d]\n",
	       fileName, fileLine);
    return(-1);
  }

#ifdef DEBUG
  traceEvent(TRACE_INFO, "Locking 0x%X @ %s [%s:%d]\n",
	     &(mutexId->mutex), where, fileName, fileLine);
#endif
  rc = pthread_mutex_lock(&(mutexId->mutex));

  if(rc != 0)
    traceEvent(TRACE_ERROR, "ERROR: lock failed 0x%X [%s:%d] (rc=%d)\n",
	       (void*)&(mutexId->mutex), fileName, fileLine, rc);
  else {
    /* traceEvent(TRACE_ERROR, "LOCKED 0x%X", &(mutexId->mutex)); */
    mutexId->numLocks++;
    mutexId->isLocked = 1;
    mutexId->lockTime = time(NULL);
    if(fileName != NULL) {
      strcpy(mutexId->lockFile, fileName);
      mutexId->lockLine = fileLine;
    }
  }

#ifdef DEBUG
  traceEvent(TRACE_INFO, "Locked 0x%X @ %s [%s:%d]\n",
	     &(mutexId->mutex), where, fileName, fileLine);
#endif
  return(rc);
}

/* ************************************ */

int _isMutexLocked(PthreadMutex *mutexId, char* fileName, int fileLine) {
  int rc;

  if(mutexId == NULL) {
    traceEvent(TRACE_ERROR,
	       "ERROR: isMutexLocked() call with a NULL mutex [%s:%d]\n",
	       fileName, fileLine);
    return(-1);
  }

  if(!mutexId->isInitialized) {
    traceEvent(TRACE_ERROR,
	       "ERROR: isMutexLocked() call with an UN-INITIALIZED mutex [%s:%d]\n",
	       fileName, fileLine);
    return(-1);
  }

#ifdef DEBUG
  traceEvent(TRACE_INFO, "Checking whether 0x%X is locked [%s:%d]\n",
	     &(mutexId->mutex), fileName, fileLine);
#endif

  rc = pthread_mutex_trylock(&(mutexId->mutex));

  /*
    Return code:

    0:    lock succesful
    EBUSY (mutex already locked)
  */

  if(rc == 0) {
    pthread_mutex_unlock(&(mutexId->mutex));
    return(0);
  } else
    return(1);
}

/* ************************************ */

int _releaseMutex(PthreadMutex *mutexId,
		  char* fileName, int fileLine) {
  int rc;

  if(mutexId == NULL) {
    traceEvent(TRACE_ERROR,
	       "ERROR: releaseMutex() call with a NULL mutex [%s:%d]\n",
	       fileName, fileLine);
    return(-1);
  }

  if(!mutexId->isInitialized) {
    traceEvent(TRACE_ERROR,
	       "ERROR: releaseMutex() call with an UN-INITIALIZED mutex [%s:%d]\n",
	       fileName, fileLine);
    return(-1);
  }

#ifdef DEBUG
  traceEvent(TRACE_INFO, "Unlocking 0x%X [%s:%d]\n",
	     &(mutexId->mutex), fileName, fileLine);
#endif
  rc = pthread_mutex_unlock(&(mutexId->mutex));

  if(rc != 0)
    traceEvent(TRACE_ERROR, "ERROR: unlock failed 0x%X [%s:%d]\n",
	       (void*)&(mutexId->mutex), fileName, fileLine);
  else {
    time_t lockDuration = time(NULL) - mutexId->lockTime;

    if((mutexId->maxLockedDuration < lockDuration)
       || (mutexId->maxLockedDurationUnlockLine == 0 /* Never set */)) {
      mutexId->maxLockedDuration = lockDuration;

      if(fileName != NULL) {
	strcpy(mutexId->maxLockedDurationUnlockFile, fileName);
	mutexId->maxLockedDurationUnlockLine = fileLine;
      }

#ifdef DEBUG
      if(mutexId->maxLockedDuration > 0) {
	traceEvent(TRACE_INFO, "DEBUG: semaphore 0x%X [%s:%d] locked for %d secs\n",
		   (void*)&(mutexId->mutex), fileName, fileLine,
		   mutexId->maxLockedDuration);
      }
#endif
    }

    mutexId->isLocked = 0;
    mutexId->numReleases++;
    if(fileName != NULL) {
      strcpy(mutexId->unlockFile, fileName);
      mutexId->unlockLine = fileLine;
    }
  }

#ifdef DEBUG
  traceEvent(TRACE_INFO, "Unlocked 0x%X [%s:%d]\n",
	     &(mutexId->mutex), fileName, fileLine);
#endif
  return(rc);
}

/* ************************************ */

#undef sleep

int nprobe_sleep(int secs) {
  int unsleptTime = secs;
  
  while((unsleptTime = sleep(unsleptTime)) > 0)
    ;

  return(secs);
}

/* ************************************ */

#else /* WIN32 */
int createThread(pthread_t *threadId,
		 void *(*__start_routine) (void *), char* userParm) {
  DWORD dwThreadId, dwThrdParam = 1;

  (*threadId) = CreateThread(NULL, /* no security attributes */
			     0,            /* use default stack size */
			     (LPTHREAD_START_ROUTINE)__start_routine, /* thread function */
			     userParm,     /* argument to thread function */
			     0,            /* use default creation flags */
			     &dwThreadId); /* returns the thread identifier */

  if(*threadId != NULL)
    return(1);
  else
    return(0);
}

/* ************************************ */

void killThread(pthread_t *threadId) {
  CloseHandle((HANDLE)*threadId);
}

/* ************************************ */

int _createMutex(PthreadMutex *mutexId, char* fileName, int fileLine) {

  memset(mutexId, 0, sizeof(PthreadMutex));

  mutexId->mutex = CreateMutex(NULL, FALSE, NULL);
  mutexId->isInitialized = 1;

#ifdef DEBUG
  if (fileName)
    traceEvent(TRACE_INFO,
	       "INFO: createMutex() call with %x mutex [%s:%d]", mutexId,
	       fileName, fileLine);
#endif

  return(1);
}

/* ************************************ */

void _deleteMutex(PthreadMutex *mutexId, char* fileName, int fileLine) {

#ifdef DEBUG
  if (fileName)
    traceEvent(TRACE_INFO,
	       "INFO: deleteMutex() call with %x(%c,%x) mutex [%s:%d]",
	       mutexId, (mutexId && mutexId->isInitialized) ? 'i' : '-',
	       mutexId ? mutexId->mutex : 0, fileName, fileLine);
#endif

  if(!mutexId->isInitialized) {
    traceEvent(TRACE_ERROR,
	       "ERROR: deleteMutex() call with a NULL mutex [%s:%d]",
	       fileName, fileLine);
    return;
  }

  ReleaseMutex(mutexId->mutex);
  CloseHandle(mutexId->mutex);

  memset(mutexId, 0, sizeof(PthreadMutex));
}

/* ************************************ */

int _accessMutex(PthreadMutex *mutexId, char* where,
		 char* fileName, int fileLine) {
#ifdef DEBUG
  traceEvent(TRACE_INFO, "Locking 0x%X @ %s [%s:%d]",
	     mutexId->mutex, where, fileName, fileLine);
#endif

  WaitForSingleObject(mutexId->mutex, INFINITE);

  mutexId->numLocks++;
  mutexId->isLocked = 1;
  mutexId->lockTime = time(NULL);

  if(fileName != NULL) {
    strcpy(mutexId->lockFile, fileName);
    mutexId->lockLine = fileLine;
  }

  return(1);
}

/* ************************************ */

int _tryLockMutex(PthreadMutex *mutexId, char* where,
		  char* fileName, int fileLine) {
#ifdef DEBUG
  traceEvent(TRACE_INFO, "Try to Lock 0x%X @ %s [%s:%d]",
	     mutexId->mutex, where, fileName, fileLine);
  fflush(stdout);
#endif

  if(WaitForSingleObject(mutexId->mutex, 0) == WAIT_FAILED)
    return(0);
  else {
    mutexId->numLocks++;
    mutexId->isLocked = 1;
    mutexId->lockTime = time(NULL);

    if(fileName != NULL) {
      strcpy(mutexId->lockFile, fileName);
      mutexId->lockLine = fileLine;
    }

    return(1);
  }
}

/* ************************************ */

int _releaseMutex(PthreadMutex *mutexId,
		  char* fileName, int fileLine) {

  time_t lockDuration;
  BOOL rc;

#ifdef DEBUG
  traceEvent(TRACE_INFO, "Unlocking 0x%X [%s:%d]",
	     mutexId->mutex, fileName, fileLine);
#endif
  rc = ReleaseMutex(mutexId->mutex);

  if((rc == 0) && (fileName)) {
    traceEvent(TRACE_ERROR, "ERROR while unlocking 0x%X [%s:%d] (LastError=%d)",
	       mutexId->mutex, fileName, fileLine, GetLastError());
  }

  lockDuration = time(NULL) - mutexId->lockTime;

  if((mutexId->maxLockedDuration < lockDuration)
     || (mutexId->maxLockedDurationUnlockLine == 0 /* Never set */)) {
    mutexId->maxLockedDuration = lockDuration;

    if(fileName != NULL) {
      strcpy(mutexId->maxLockedDurationUnlockFile, fileName);
      mutexId->maxLockedDurationUnlockLine = fileLine;
    }

#ifdef DEBUG
    traceEvent(TRACE_INFO, "INFO: semaphore 0x%X [%s:%d] locked for %d secs",
	       &(mutexId->mutex), fileName, fileLine,
	       mutexId->maxLockedDuration);
#endif
  }

  mutexId->isLocked = 0;
  mutexId->numReleases++;
  if(fileName != NULL) {
    strcpy(mutexId->unlockFile, fileName);
    mutexId->unlockLine = fileLine;
  }

  return(1);
}

/* **************************** */

unsigned long waitForNextEvent(unsigned long ulDelay /* ms */) {
  unsigned long ulSlice = 1000L; /* 1 Second */

  while (ulDelay > 0L) {
    if (ulDelay < ulSlice)
      ulSlice = ulDelay;
    Sleep(ulSlice);
    ulDelay -= ulSlice;
  }

  return ulDelay;
}

/* ******************************* */

void initWinsock32() {
  WORD wVersionRequested;
  WSADATA wsaData;
  int err;

  wVersionRequested = MAKEWORD(2, 0);
  err = WSAStartup( wVersionRequested, &wsaData );
  if( err != 0 ) {
    /* Tell the user that we could not find a usable */
    /* WinSock DLL.                                  */
    traceEvent(TRACE_ERROR, "FATAL ERROR: unable to initialise Winsock 2.x.");
    exit(-1);
  }
}

/* ******************************** */

short isWinNT() {
  DWORD dwVersion;
  DWORD dwWindowsMajorVersion;

  dwVersion=GetVersion();
  dwWindowsMajorVersion =  (DWORD)(LOBYTE(LOWORD(dwVersion)));
  if(!(dwVersion >= 0x80000000 && dwWindowsMajorVersion >= 4))
    return 1;
  else
    return 0;
}

/* ******************************** */

char* printAvailableInterfaces(int index) {
  char ebuf[PCAP_ERRBUF_SIZE];
  char *tmpDev = pcap_lookupdev(ebuf), *ifName;
  int ifIdx=0, defaultIdx, i;
  char intNames[32][256];

  if(tmpDev == NULL) {
    traceEvent(TRACE_INFO, "Unable to locate default interface (%s)", ebuf);
    exit(-1);
  }

  ifName = tmpDev;

  if(index == -1) printf("Available interfaces:\n");

  if(!isWinNT()) {
    for(i=0;; i++) {
      if(tmpDev[i] == 0) {
	if(ifName[0] == '\0')
	  break;
	else {
	  if(index == -1) printf("\t[index=%d] '%s'\n", ifIdx, ifName);

	  if(ifIdx < 32) {
	    strcpy(intNames[ifIdx], ifName);
	    if(defaultIdx == -1) {
	      if(strncmp(intNames[ifIdx], "PPP", 3) /* Avoid to use the PPP interface */
		 && strncmp(intNames[ifIdx], "ICSHARE", 6)) { /* Avoid to use the internet sharing interface */
		defaultIdx = ifIdx;
	      }
	    }
	  }

	  ifIdx++;
	  ifName = &tmpDev[i+1];
	}
      }
    }

    tmpDev = intNames[defaultIdx];
  } else {
    /* WinNT/2K */
    static char tmpString[128];
    int i, j;

    while(tmpDev[0] != '\0') {
      for(j=0, i=0; !((tmpDev[i] == 0) && (tmpDev[i+1] == 0)); i++) {
	if(tmpDev[i] != 0)
	  tmpString[j++] = tmpDev[i];
      }

      tmpString[j++] = 0;
      if(index == -1) traceEvent(TRACE_INFO, "\t[index=%d] '%s'\n", ifIdx, tmpString);
      tmpDev = &tmpDev[i+3];
      strcpy(intNames[ifIdx++], tmpString);
      defaultIdx = 0;
    }
    if(defaultIdx != -1)
      tmpDev = intNames[defaultIdx]; /* Default */
  }

  if(index == -1) 
    return(NULL);
  else if((index < 0) || (index > ifIdx)) {
    traceEvent(TRACE_ERROR, "Index=%d out of range\n", index);
    exit(-1);
  } else
    return(intNames[index]);
}

#endif
