#include "httpd.h"
#include "http_config.h"
#include "http_request.h"
#include "http_protocol.h"
#include "http_log.h"

#include "apr_strings.h"
#include "apr_file_info.h"

#include <sys/stat.h>

typedef struct {
	int mkdir_enabled;
	char *mkdir_base;
	int mkdir_maxdepth;
	int mkdir_methods;
} mkdir_dir_cfg_rec;

#define MKDIR_GET_DIR_CFG(r) \
   (mkdir_dir_cfg_rec *)ap_get_module_config((r)->per_dir_config,&mkdir_module)

const char *mkdir_set_methods_slot(cmd_parms *, void *, int, char *const *);
int mkdir_fixup(request_rec *);

command_rec mkdir_cmds[] = {
    AP_INIT_FLAG(
        "MkdirEnable",
        ap_set_flag_slot,    
        (void *)APR_OFFSETOF(mkdir_dir_cfg_rec, mkdir_enabled),
	OR_AUTHCFG,
        "Enable mod_mkdir"
        ),
    AP_INIT_TAKE1(
        "MkdirMaxDepth",
        ap_set_int_slot,    
        (void *)APR_OFFSETOF(mkdir_dir_cfg_rec, mkdir_maxdepth),
	OR_AUTHCFG,
        "Maximum depth of created directories"
        ),
    AP_INIT_TAKE_ARGV(
        "MkdirMethods",
        mkdir_set_methods_slot,    
        NULL,
	OR_AUTHCFG,
        "HTTP methods on which directories are created"
	),
    { NULL }

};


static void *
mkdir_dir_cfg_create(p, path)
	apr_pool_t *p;
	char *path;
{
	mkdir_dir_cfg_rec *dir;

	dir = (mkdir_dir_cfg_rec *)apr_palloc(p, sizeof(*dir));
	dir->mkdir_enabled = -1;
	dir->mkdir_base = apr_pstrdup(p, path);
	dir->mkdir_maxdepth = 1;
	dir->mkdir_methods = (AP_METHOD_BIT << M_PUT)
			   | (AP_METHOD_BIT << M_PROPFIND);

	return (void *)dir;
}

static void *
mkdir_dir_cfg_merge(p, base, add)
	apr_pool_t *p;
	void *base;
	void *add;
{
	mkdir_dir_cfg_rec *basedir;
	mkdir_dir_cfg_rec *adddir;
	mkdir_dir_cfg_rec *newdir;

	basedir = (mkdir_dir_cfg_rec *)base;
	adddir = (mkdir_dir_cfg_rec *)add;
	newdir = (mkdir_dir_cfg_rec *)apr_palloc(p, sizeof(*newdir));

	switch(adddir->mkdir_enabled) {
	case 0:
		newdir->mkdir_enabled = 0;
		newdir->mkdir_base = NULL;
		newdir->mkdir_maxdepth = 0;
		newdir->mkdir_methods = 0;
		break;
	case 1:
		newdir->mkdir_enabled = 1;
		newdir->mkdir_base = adddir->mkdir_base;
		newdir->mkdir_maxdepth = adddir->mkdir_maxdepth;
		newdir->mkdir_methods = adddir->mkdir_methods;
		break;
	case -1: /* FALLTHROUGH */
	default:
		newdir->mkdir_enabled = basedir->mkdir_enabled;
		newdir->mkdir_base = basedir->mkdir_base;
		newdir->mkdir_maxdepth = basedir->mkdir_maxdepth;
		newdir->mkdir_methods = basedir->mkdir_methods;
		break;
	}

	return (void *)newdir;
}


static void
mkdir_register_hooks(p)
	apr_pool_t *p;
{
	ap_hook_fixups(mkdir_fixup, NULL, NULL, APR_HOOK_MIDDLE);
	return;
}

module AP_MODULE_DECLARE_DATA mkdir_module =
{    
	STANDARD20_MODULE_STUFF,
	mkdir_dir_cfg_create,
	mkdir_dir_cfg_merge,
	NULL,
	NULL,
	mkdir_cmds,
	mkdir_register_hooks
};     

const char *
mkdir_set_methods_slot(params, mconfig, argc, argv)
	cmd_parms *params;
	void *mconfig;
	int argc;
	char *const *argv;
{
	int i, m;
	mkdir_dir_cfg_rec *dir;

	dir = (mkdir_dir_cfg_rec *)mconfig;
	dir->mkdir_methods = 0;

	for (i = 0; i < argc; i++) {
		m = ap_method_number_of(argv[i]);
		dir->mkdir_methods |= (AP_METHOD_BIT << m);
	}
	
	return NULL;
}

int
mkdir_fixup(r)
	request_rec *r;
{
	mkdir_dir_cfg_rec *dir;
	char *path;
	char *pcn;
	char *last;
	char *vpath = "";
	apr_finfo_t finfo;
	int i, depth;

	/* This happens on server shutdown */
	if (r == NULL)
		return OK;

	/* 
	 * This module does auto-mkdir, hence avoid 
	 * creating on MKCOL verb, since the Apache
	 * implementation of MKCOL would fail afterwards
	 */
	if (r->method_number == M_MKCOL)
		return DECLINED;

	dir = MKDIR_GET_DIR_CFG(r);
	if (dir->mkdir_enabled != 1 || 
	    dir->mkdir_maxdepth <= 0 ||
	    dir->mkdir_base == NULL)
		return OK;

	if ((AP_METHOD_BIT << r->method_number) & (dir->mkdir_methods) == 0)
		return OK;

	path = apr_pstrdup(r->pool, r->filename);

	ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, r->server,
		     "%s: %s base = \"%s\", maxdepth = %d, path = \"%s\"",
		     __func__, r->method, dir->mkdir_base, 
		     dir->mkdir_maxdepth, path);

	for (i = 0; path[i] && path[i] == dir->mkdir_base[i]; i++); 

	if (path[i] == '\0')
		return OK;

	vpath = apr_pstrdup(r->pool, path);
	vpath[i] = '\0';

	path = path + i;

	ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, 
		      "%s: %s dir = \"%s\", maxdepth = %d, path = \"%s\"",
		      __func__, r->method, vpath, 
		      dir->mkdir_maxdepth, path);
	
	last = NULL;
	depth = dir->mkdir_maxdepth;
	for (pcn = apr_strtok(path, "/", &last);
	     pcn != NULL;
	     pcn = apr_strtok(NULL, "/", &last))
	{
		vpath = apr_pstrcat(r->pool, vpath, "/", pcn, NULL);

		if (--depth < 0) {
			ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, r, 
				      "mod_mkdir will not mkdir \"%s\", "
				      "depth beyond MkdirMaxDepth %d", vpath,
				      dir->mkdir_maxdepth);
			return OK;
		}

		if (apr_stat(&finfo, vpath, APR_FINFO_TYPE, r->pool) != OK) {
			if (mkdir(vpath, 0755) != 0) {
				ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, 
					      "cannot mkdir(\"%s\")", vpath);
				return OK;
			} else {
				ap_log_rerror(APLOG_MARK, APLOG_NOTICE, 0, r, 
					      "Created directory \"%s\"",
					      vpath);
			}

		}

	}

	return OK;
}

