/*
 * MGTERM -- Terminal Emulator for MobileGear -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
/*
 * KON2 - Kanji ON Console -
 * Copyright (C) 1992-1996 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<string.h>
#include	<sys/types.h>
#include	<sys/ioctl.h>
#include	<assert.h>

#ifdef __CYGWIN__
#include	<termios.h>
#endif

#ifndef MGL1
#include	"mgl2.h"
#include	"mglcol.h"
#else
#include	"mgl.h"
#define NO_COLOR
#define mgl_getenv	getenv
#endif

#undef FALSE
#undef TRUE
typedef	enum {FALSE, TRUE} bool;

#define	FAILURE	(-1)
#define SUCCESS	(0)

#ifdef USE_MGL_MEMMOVE
#define memmove	mgl_memmove
#endif

int send_mouse = 0;

struct _con_info {
    short
	x,
	y,
	xmax,	/* 79 */
	ymin,	/* 0:볫Ϲ */
	ymax,	/* 29 */
	tab;	/* 8 */
    u_char
	fcol,	/* ե */
	bcol,	/* Хå */
	attr,	/* ʸ° */
	sb,	/* 1 byte code եֹ */
	db,	/* 2 byte code եֹ */
	knj1;	/* 饯 1 byte */
    void (*esc)(u_char);
    enum {
	CS_LEFT,
	CS_RIGHT,
	CS_GRAPH,
	CS_DBCS} trans, g[2];
    bool
	soft,
	ins,
	active,
	wrap,
	text_mode;
};

#define	CODE_2022	0	/* 2022 Τߤ˽*/
#define	CODE_EUC	1	/* EUC ˤ⽾ */
#define	CODE_SJIS	2	/* SJIS ˤ⽾ */

#define	G0_SET	0
#define	G1_SET	0x80

extern void	VtInit(void);
extern void	VtStart(int);
extern void	VtEmu(const char* buff, int nchars);
extern void	VtCleanup(void);

void TextClearBol(int c);
void TextClearEol(int c);
void TextClearEos(int c);
void TextDeleteChar(int c);
void TextInsertChar(int c);
void TextMoveDown(int top, int btm, int line);
void TextMoveUp(int top, int btm, int line);
void TextClearAll(void);
void TextWput(int ch1, int ch2);
void TextSput(int ch);
void ScrollUp(int l);
void ScrollDown(int l);
void hide_cursor(void);
void show_cursor(void);
void Beep(void);

#define	COL_FG		0
#define	COL_BG		1

#define	ATTR_ULINE	0x80	/* under line */
#define	ATTR_REVERSE	0x40	/* reverse */
#define	ATTR_HIGH	0x20	/* high */
#define	ATTR_ITALIC	0x10	/* italic */

#define	LATCH_S		0x0 /* single byte char */
#define	LATCH_1		0x20 /* double byte char 1st byte */
#define	LATCH_2		0x40 /* double byte char 2nd byte */

#define	CLEAN_S		0x80
#define	CODEIS_1	LATCH_1
#define	CODEIS_2	LATCH_2
#define	LANG_CODE	0x0F
/*
#define	LANG_DCODE	LANG_CODE|CODEIS_1
#define	LANG_SCODE	LANG_CODE
*/

struct	_con_info con;

#define	CHAR_NUL	'\x00'
#define	CHAR_BEL	'\x07'
#define	CHAR_BS		'\x08'
#define	CHAR_HT		'\x09'
#define	CHAR_LF		'\x0A'
#define	CHAR_VT		'\x0B'
#define	CHAR_FF		'\x0C'
#define	CHAR_CR		'\x0D'
#define	CHAR_SO		'\x0E'
#define	CHAR_SI		'\x0F'
#define	CHAR_XON	'\x11'
#define	CHAR_XOFF	'\x12'
#define	CHAR_CAN	'\x18'
#define	CHAR_SUB	'\x1A'
#define	CHAR_ESC	'\x1B'
#define	CHAR_DEL	'\x7F'
#define	CHAR_CSI	'\x9B'
#define	CHAR_SS2	'\x8E'

#define	LEN_REPORT	9

extern int screen_width;
extern int screen_height;
extern int font_size;
static int font_width;
static int font_height;

#define MAXCOLS		256
#define MAXLINES	64

int COLS=80;
int LINES=18;
extern int ptyfd;

/* cursor_x,y */
int cx,cy,cxs=-1;


#define CURSOR_BLOCK	0
#define CURSOR_UL	1
#define CURSOR_RECT	2
static int cursor_shape = CURSOR_BLOCK;

struct screen_data {
	char attr;	/* 0: no char 1: ANK 2:kanji 1st 4: kanji 2nd */
	unsigned char ch;
} *screen_data[MAXLINES];

#define S_NULL		0
#define S_ANK		1
#define S_KANJI1	2
#define S_KANJI2	4
#define	S_ULINE		ATTR_ULINE
#define	S_REVERSE	ATTR_REVERSE
#define	S_HIGH		ATTR_HIGH

struct attrStack {
    struct attrStack *prev;
    u_char x, y, attr, bcol, fcol;
};

static struct attrStack *saveAttr;

#ifndef MGL1
#define DEFAULT_COLOR_BLACK		COLOR_BLACK
#define DEFAULT_COLOR_WHITE		COLOR_WHITE
#define DEFAULT_COLOR_RED		packMC(MCH_RED,5,5)
#define DEFAULT_COLOR_GREEN		packMC(MCH_GREEN,5,5)
#define DEFAULT_COLOR_YELLOW		packMC(MCH_YELLOW,5,5)
#define DEFAULT_COLOR_BLUE		packMC(MCH_BLUE,5,5)
#define	DEFAULT_COLOR_MAGENTA		packMC(MCH_MAGENTA,5,5)
#define	DEFAULT_COLOR_CYAN		packMC(MCH_CYAN,5,5)

#define DEFAULT_COLOR_BLACK_BG		COLOR_BLACK
#define DEFAULT_COLOR_WHITE_BG		COLOR_WHITE
#define DEFAULT_COLOR_RED_BG		packMC(MCH_RED,15,15)
#define DEFAULT_COLOR_GREEN_BG		packMC(MCH_GREEN,15,15)
#define DEFAULT_COLOR_YELLOW_BG		packMC(MCH_YELLOW,15,15)
#define DEFAULT_COLOR_BLUE_BG		packMC(MCH_BLUE,15,15)
#define	DEFAULT_COLOR_MAGENTA_BG	packMC(MCH_MAGENTA,15,15)
#define	DEFAULT_COLOR_CYAN_BG		packMC(MCH_CYAN,15,15)
#else
#define DEFAULT_COLOR_BLACK		COLOR_BLACK
#define DEFAULT_COLOR_WHITE		COLOR_WHITE
#define DEFAULT_COLOR_RED		COLOR_LIGHTGRAY
#define DEFAULT_COLOR_GREEN		COLOR_LIGHTGRAY
#define DEFAULT_COLOR_YELLOW		COLOR_LIGHTGRAY
#define DEFAULT_COLOR_BLUE		COLOR_LIGHTGRAY
#define	DEFAULT_COLOR_MAGENTA		COLOR_LIGHTGRAY
#define	DEFAULT_COLOR_CYAN		COLOR_LIGHTGRAY

#define DEFAULT_COLOR_BLACK_BG		COLOR_BLACK
#define DEFAULT_COLOR_WHITE_BG		COLOR_WHITE
#define DEFAULT_COLOR_RED_BG		COLOR_DARKGRAY
#define DEFAULT_COLOR_GREEN_BG		COLOR_DARKGRAY
#define DEFAULT_COLOR_YELLOW_BG		COLOR_DARKGRAY
#define DEFAULT_COLOR_BLUE_BG		COLOR_DARKGRAY
#define	DEFAULT_COLOR_MAGENTA_BG	COLOR_DARKGRAY
#define	DEFAULT_COLOR_CYAN_BG		COLOR_DARKGRAY
#endif

#define	COLOR_FG			DEFAULT_COLOR_BLACK
#define	COLOR_BG			DEFAULT_COLOR_WHITE

static int	ctable[10] = {
	DEFAULT_COLOR_BLACK,
	DEFAULT_COLOR_RED,
	DEFAULT_COLOR_GREEN,
	DEFAULT_COLOR_YELLOW,
	DEFAULT_COLOR_BLUE,
	DEFAULT_COLOR_MAGENTA,
	DEFAULT_COLOR_CYAN,
	DEFAULT_COLOR_WHITE,
	COLOR_FG,
	COLOR_BG
};

static int	ctable_bg[10] = {
	DEFAULT_COLOR_BLACK,
	DEFAULT_COLOR_RED_BG,
	DEFAULT_COLOR_GREEN_BG,
	DEFAULT_COLOR_YELLOW_BG,
	DEFAULT_COLOR_BLUE_BG,
	DEFAULT_COLOR_MAGENTA_BG,
	DEFAULT_COLOR_CYAN_BG,
	DEFAULT_COLOR_WHITE,
	COLOR_FG,
	COLOR_BG
};

#ifndef NO_COLOR
struct cname_table {
	char *name;
	char *val;
} color_tab[] = {
{"Black"	,"#000000"},
{"Silver"	,"#C0C0C0"},
{"Gray"		,"#808080"},
{"White"	,"#FFFFFF"},
{"Maroon"	,"#800000"},
{"Red"		,"#FF0000"},
{"Purple"	,"#800080"},
{"Fuchsia"	,"#FF00FF"},
{"Green"	,"#008000"},
{"Lime"		,"#00FF00"},
{"Olive"	,"#808000"},
{"Yellow"	,"#FFFF00"},
{"Navy"		,"#000080"},
{"Blue"		,"#0000FF"},
{"Teal"		,"#008080"},
{"Aqua"		,"#00FFFF"},
#if 1
{"aliceblue"             ,"#F0F8FF"},
{"antiquewhite"          ,"#FAEBD7"},
{"aquamarine"            ,"#7FFFD4"},
{"azure"                 ,"#F0FFFF"},
{"beige"                 ,"#F5F5DC"},
{"bisque"                ,"#FFE4C4"},
{"blanchedalmond"        ,"#FFEBCD"},
{"blueviolet"            ,"#8A2BE2"},
{"brown"                 ,"#A52A2A"},
{"burlywood"             ,"#DEB887"},
{"cadetblue"             ,"#5F9EA0"},
{"chartreuse"            ,"#7FFF00"},
{"chocolate"             ,"#D2691E"},
{"coral"                 ,"#FF7F50"},
{"cornflowerblue"        ,"#6495ED"},
{"cornsilk"              ,"#FFF8DC"},
{"crimson"               ,"#DC143C"},
{"cyan"                  ,"#00FFFF"},
{"darkblue"              ,"#00008B"},
{"darkcyan"              ,"#008B8B"},
{"darkgoldenrod"         ,"#B8860B"},
{"darkgray"              ,"#A9A9A9"},
{"darkgreen"             ,"#006400"},
{"darkkhaki"             ,"#BDB76B"},
{"darkmagenta"           ,"#8B008B"},
{"darkolivegreen"        ,"#556B2F"},
{"darkorange"            ,"#FF8C00"},
{"darkorchid"            ,"#9932CC"},
{"darkred"               ,"#8B0000"},
{"darksalmon"            ,"#E9967A"},
{"darkseagreen"          ,"#8FBC8F"},
{"darkslateblue"         ,"#483D8B"},
{"darkslategray"         ,"#2F4F4F"},
{"darkturquoise"         ,"#00CED1"},
{"darkviolet"            ,"#9400D3"},
{"deeppink"              ,"#FF1493"},
{"deepskyblue"           ,"#00BFFF"},
{"dimgray"               ,"#696969"},
{"dodgerblue"            ,"#1E90FF"},
{"firebrick"             ,"#B22222"},
{"floralwhite"           ,"#FFFAF0"},
{"forestgreen"           ,"#228B22"},
{"gainsboro"             ,"#DCDCDC"},
{"ghostwhite"            ,"#F8F8FF"},
{"gold"                  ,"#FFD700"},
{"goldenrod"             ,"#DAA520"},
{"greenyellow"           ,"#ADFF2F"},
{"honeydew"              ,"#F0FFF0"},
{"hotpink"               ,"#FF69B4"},
{"indianred"             ,"#CD5C5C"},
{"indigo"                ,"#4B0082"},
{"ivory"                 ,"#FFFFF0"},
{"khaki"                 ,"#F0E68C"},
{"lavender"              ,"#E6E6FA"},
{"lavenderblush"         ,"#FFF0F5"},
{"lawngreen"             ,"#7CFC00"},
{"lemonchiffon"          ,"#FFFACD"},
{"lightblue"             ,"#ADD8E6"},
{"lightcoral"            ,"#F08080"},
{"lightcyan"             ,"#E0FFFF"},
{"lightgoldenrodyellow"  ,"#FAFAD2"},
{"lightgreen"            ,"#90EE90"},
{"lightgrey"             ,"#D3D3D3"},
{"lightpink"             ,"#FFB6C1"},
{"lightsalmon"           ,"#FFA07A"},
{"lightseagreen"         ,"#20B2AA"},
{"lightskyblue"          ,"#87CEFA"},
{"lightslategray"        ,"#778899"},
{"lightsteelblue"        ,"#B0C4DE"},
{"lightyellow"           ,"#FFFFE0"},
{"limegreen"             ,"#32CD32"},
{"linen"                 ,"#FAF0E6"},
{"magenta"               ,"#FF00FF"},
{"mediumaquamarine"      ,"#66CDAA"},
{"mediumblue"            ,"#0000CD"},
{"mediumorchid"          ,"#BA55D3"},
{"mediumpurple"          ,"#9370DB"},
{"mediumseagreen"        ,"#3CB371"},
{"mediumslateblue"       ,"#7B68EE"},
{"mediumspringgreen"     ,"#00FA9A"},
{"mediumturquoise"       ,"#48D1CC"},
{"mediumvioletred"       ,"#C71585"},
{"midnightblue"          ,"#191970"},
{"mintcream"             ,"#F5FFFA"},
{"mistyrose"             ,"#FFE4E1"},
{"moccasin"              ,"#FFE4B5"},
{"navajowhite"           ,"#FFDEAD"},
{"oldlace"               ,"#FDF5E6"},
{"olivedrab"             ,"#6B8E23"},
{"orange"                ,"#FFA500"},
{"orangered"             ,"#FF4500"},
{"orchid"                ,"#DA70D6"},
{"palegoldenrod"         ,"#EEE8AA"},
{"palegreen"             ,"#98FB98"},
{"paleturquoise"         ,"#AFEEEE"},
{"palevioletred"         ,"#DB7093"},
{"papayawhip"            ,"#FFEFD5"},
{"peachpuff"             ,"#FFDAB9"},
{"peru"                  ,"#CD853F"},
{"pink"                  ,"#FFC0CB"},
{"plum"                  ,"#DDA0DD"},
{"powderblue"            ,"#B0E0E6"},
{"rosybrown"             ,"#BC8F8F"},
{"royalblue"             ,"#4169E1"},
{"saddlebrown"           ,"#8B4513"},
{"salmon"                ,"#FA8072"},
{"sandybrown"            ,"#F4A460"},
{"seagreen"              ,"#2E8B57"},
{"seashell"              ,"#FFF5EE"},
{"sienna"                ,"#A0522D"},
{"skyblue"               ,"#87CEEB"},
{"slateblue"             ,"#6A5ACD"},
{"slategray"             ,"#708090"},
{"snow"                  ,"#FFFAFA"},
{"springgreen"           ,"#00FF7F"},
{"steelblue"             ,"#4682B4"},
{"tan"                   ,"#D2B48C"},
{"thistle"               ,"#D8BFD8"},
{"tomato"                ,"#FF6347"},
{"turquoise"             ,"#40E0D0"},
{"violet"                ,"#EE82EE"},
{"wheat"                 ,"#F5DEB3"},
{"whitesmoke"            ,"#F5F5F5"},
{"yellowgreen"           ,"#9ACD32"},
#endif
};
#endif

static int	scroll;			 /* Կ */

static
    void SaveAttr()
{
    struct attrStack *tmp;

    tmp = (struct attrStack *)malloc(sizeof(struct attrStack));
    if (saveAttr) tmp->prev = saveAttr;
    else tmp->prev = NULL;
    saveAttr = tmp;
    saveAttr->x = con.x;
    saveAttr->y = con.y;
    saveAttr->attr = con.attr;
    saveAttr->fcol = con.fcol;
    saveAttr->bcol = con.bcol;
}

static
    void RestoreAttr()
{
    struct attrStack *tmp;

    if (saveAttr) {
	con.x = saveAttr->x;
	con.y = saveAttr->y;
	con.attr = saveAttr->attr;
	con.fcol = saveAttr->fcol;
	con.bcol = saveAttr->bcol;
	tmp = saveAttr;
	saveAttr = tmp->prev;
	free(tmp);
    }
}

static int	get_font_color(int ground)
{
	if (con.attr&ATTR_REVERSE)
	{
		if (ground == COL_FG)
			return ctable_bg[con.bcol];
		else	/* COL_BG */
			return ctable[con.fcol];
	}
	else
	{
		if (ground == COL_FG)
			return ctable[con.fcol];
		else	/* COL_BG */
			return ctable_bg[con.bcol];
	}
}


static void	EscSetAttr(int col)
{
    static u_char table[] = {0, 1, 2, 3, 4, 5, 6, 7};
    u_char	swp;
    
    switch(col) {
    case 0:	/* off all attributes */
	con.bcol = 9;
	con.fcol = 8;
	con.attr = 0;
	break;
    case 1:	/*  */
	con.attr |= ATTR_HIGH;
	break;
    case 21:
	con.attr &= ~ATTR_HIGH;
	break;
    case 4:	/*  */
	con.attr |= ATTR_ULINE;
	break;
    case 24:
	con.attr &= ~ATTR_ULINE;
	break;
    case 7:	/* ȿž */
	con.attr |= ATTR_REVERSE;
	break;
    case 27:
	con.attr &= ~ATTR_REVERSE;
	break;
    case 38:
	con.attr |= ATTR_ULINE;
	con.fcol = 8;
	break;
    case 39:
	con.attr &= ~ATTR_ULINE;
	con.fcol = 8;
	break;
    case 49:
	con.bcol = 9;
	break;
    default:
	if (col >= 30 && col <= 37) {
	    con.fcol = table[col - 30];
	} else if (col >= 40 && col <= 47) {
	    con.bcol = table[col - 40];
	}
	break;
    }
}

static void	VtSetMode(int mode, bool sw)
{
    switch(mode) {
    case 4:
	con.ins = sw;
	break;
    case 25:
	break;
    case 1000:
	send_mouse = sw;
	break;
    }
}

static
    void EscReport(u_char mode, u_short arg)
{
    static char report[LEN_REPORT];
    
    switch(mode) {
    case 'n':
	if (arg == 6) {
	    int x = (con.x < con.xmax) ? con.x : con.xmax;
	    int y = (con.y < con.ymax) ? con.y : con.ymax;
	    sprintf(report, "\x1B[%d;%dR", y + 1, x + 1);
	} else if (arg == 5)
	    strcpy(report, "\x1B[0n\0");
	break;
    case 'c':
	if (arg == 0) strcpy(report, "\x1B[?6c\0");
	break;
    }
    write(ptyfd, report, strlen(report));
}

#define	MAX_NARG	8

static int right_char(n,mode) {
	int x = con.x;
	int i;
	if (mode) {
		for (i=0; i<n; i++) {
			if (screen_data[con.y][x].attr & S_KANJI1)
				x +=2;
			else
				x++;
		}
	} else {
		x += n;
	}
	return (x>COLS-1)?COLS-1:x;
}

static int left_char(n,mode) {
	int x = con.x;
	int i;
	
	if (mode) {
		for (i=0; i<n; i++) {
			if (x <= 0) break;
			x --;
			if (screen_data[con.y][x].attr & S_KANJI2)
			x --; 
		}
	} else {
		x -= n;
	}
	return (x<0)?0:x;
}

static
    void EscBracket(u_char ch)
{
    u_char	n;
    static u_short varg[MAX_NARG], narg, question;

    if (ch >= '0' && ch <= '9') {
	varg[narg] = (varg[narg] * 10) + (ch - '0');
    } else if (ch == ';') {
	/*  MAX_NARG ޤǤݡȤʤ!! */
	if (narg < MAX_NARG) narg ++;
	else con.esc = NULL;
    } else {
	con.esc = NULL;
	switch(ch) {
	case 'K':	/* Clear all or part of line */
	    if (varg[0] == 1)
		    TextClearBol(varg[0]);
	    else
		    TextClearEol(varg[0]);
	    break;
	case 'J':	/* Clear all or part of display */
	    TextClearEos(varg[0]);
	    break;
	case 'A':	/* up n rows */
	    con.y -= varg[0] ? varg[0]: 1;
	    if (con.y < con.ymin) {
		scroll -= con.y - con.ymin;
		con.y = con.ymin;
	    }
	    break;
	case 'e':	/* move cursor n rows down */
	case 'B':	/* down n rows */
	    con.y += varg[0] ? varg[0]: 1;
	    if (con.y > con.ymax) {
		scroll += con.y - con.ymin;
		con.y = con.ymax;
	    }
	    break;
	case 'a':	/* move cursor n columns to the right */
	case 'C':	/* right n columns */
	    con.x = right_char(varg[0] ? varg[0]: 1,0);
	    con.wrap = FALSE;
	    break;
	case 'D':	/* left n columns */
	    con.x = left_char(varg[0] ? varg[0]: 1,0);
	    con.wrap = FALSE;
	    break;
	case 'E':	/* cursor to start of line n lines down */
	    con.y += varg[0] ? varg[0]: 1;
	    con.x = 0;
	    if (con.y > con.ymax) {
		scroll += con.y - con.ymin;
		con.y = con.ymax;
	    }
	    break;
	case 'F':	/* cursor to start of line n lines up */
	    con.y -= varg[0] ? varg[0]: 1;
	    con.x = 0;
	    if (con.y < con.ymin) {
		scroll -= con.y - con.ymin;
		con.y = con.ymin;
	    }
	    break;
	case 'X':	/* erase n characters in line */
	    /* NOTYET */
	    break;
	case 'Z':	/* move n tabs backwards */
	    /* NOTYET */
	    break;
	/*case 'G':?????*/
	case '`':   /* move cursor to column n */
	    con.x = 0;
	    con.x = right_char(varg[0] ? varg[0] - 1: 0,0);
	    con.wrap = FALSE;
	    break;
	case 'P':	/* Delete n chars */
	    TextDeleteChar(varg[0] ? varg[0]: 1);
	    break;
	case '@':	/* Insert n chars */
	    TextInsertChar(varg[0] ? varg[0]: 1);
	    break;
	case 'T':	/* scroll down n lines */ 
	    TextMoveDown(con.ymin, con.ymax,
			 varg[0] ? varg[0] : 1);
	    break;
	case 'L':	/* Insert n lines */
	    TextMoveDown(con.y, con.ymax,
			 varg[0] ? varg[0] : 1);
	    break;
	case 'S':	/* scroll up n lines */
	    TextMoveUp(con.ymin, con.ymax,
		       varg[0] ? varg[0] : 1);
	    break;
	case 'M':	/* Delete n lines */
	    TextMoveUp(con.y, con.ymax,
		       varg[0] ? varg[0] : 1);
	    break;
	case 'H':	/* Cursor move */
	case 'f':
	    if (varg[1]) con.x = varg[1] - 1;
	    else con.x = 0;
	    if (con.x > con.xmax) con.x = con.xmax; /* takesi */
	    con.wrap = FALSE;
	case 'd':	/* move cursor to row n */
	    con.y = varg[0] ? varg[0] - 1: 0;
#if 0 /* FEP(uum) write message on system-line that is out of region (by 'H') */
	    if (con.y > con.ymax) con.y = con.ymax; /* takesi */
#else
	    if (con.y > LINES-1) con.y = LINES-1;
#endif
	    break;
	case 'm':	/* change attribute */
	    for (n = 0; n <= narg; n ++)
		EscSetAttr(varg[n]);
	    break;
	case 'r':
	    if ((varg[0] >= 1000) || (varg[1] >= 1000)) {
		break;
	    }
	    con.ymin = varg[0]? (varg[0] - 1): 0;
	    con.ymax = varg[1]? (varg[1] - 1): LINES-1;
	    if (con.ymax > LINES-1) con.ymax = LINES-1; /* by SUZ */
	    con.x = 0;
	    con.y = con.ymin;
	    con.wrap = FALSE;
	    if (con.ymin || con.ymax != LINES-1)
		con.soft = TRUE;
	    else
		con.soft = FALSE;
	    break;
	case 'l':
	    for (n = 0; n <= narg; n ++)
		VtSetMode(varg[n], FALSE);
	    break;
	case 'h':
	    for (n = 0; n <= narg; n ++)
		VtSetMode(varg[n], TRUE);
	    break;
	case '?':
	    question = TRUE;
	    con.esc = EscBracket;
	    break;
	case 's':	/* Save cursor position */
	    if ((varg[0] >= 1000) || (varg[1] >= 1000)) {
		break;
	    }
	    SaveAttr();
	    break;
	case 'u':	/* Restore saved cursor position */
	    RestoreAttr();
	    break;
	case 'n':
	case 'c':
	    if (question != TRUE)
		EscReport(ch, varg[0]);
	    break;
	case 'R':
	    break;
	}
	if (con.esc == NULL)
	    question = narg = varg[0] = varg[1] = 0;
    }
}

static
    void EscSetDCodeG0(u_char ch)
{
    switch(ch) {
    case '(': /* EscSetDCodeG0 */
    case ')': /* EscSetDCodeG1 */
	return;
    case '@':
	ch = 'B';
    default:
	con.trans = CS_DBCS;
    }
    con.esc = NULL;
}

static
    void EscSetSCodeG0(u_char ch)
{
    switch(ch) {
    case 'U':
	con.g[0] = CS_GRAPH;
	break;
    default:
	con.g[0] = CS_LEFT;
    }
    con.trans = con.g[0];
    con.esc = NULL;
}

static
    void EscSetSCodeG1(u_char ch)
{
    switch(ch) {
    case 'U':
	con.g[1] = CS_LEFT;
	break;
    case '0':
	con.g[1] = CS_GRAPH;
	break;
    case 'A':
    case 'J':
    case 'B':
	break;
    }
    con.trans = con.g[1];
    con.esc = NULL;
}

static
    void EscStart(u_char ch)
{
    con.esc = NULL;
    switch(ch) {
    case '[':
	con.esc = EscBracket;
	break;
    case '$':/* Set Double Byte Code */
	con.esc = EscSetDCodeG0;
	break;
    case '(':/* Set 94 to G0 */
    case ',':/* Set 96 to G0 */
	con.esc = EscSetSCodeG0;
	break;
    case ')':/* Set G1 */
	con.esc = EscSetSCodeG1;
	break;
    case 'E':
	con.x = 0;
	con.wrap = FALSE;
    case 'D':
	if (con.y == con.ymax) scroll ++;
	else con.y ++;
	break;
    case 'M':
	if (con.y == con.ymin) scroll --;
	else con.y --;
	break;	
    case 'c':
	con.fcol = 9;
	con.bcol = 8;
	con.attr = 0;
	con.knj1 = 0;
	con.wrap = FALSE;
	con.trans = CS_LEFT;
    case '*':
	con.x = con.y = 0;
	con.wrap = FALSE;
	TextClearAll();
	break;
    case '7':
	SaveAttr();
	break;
    case '8':
	RestoreAttr();
	con.wrap = FALSE;
	break;
    }
}

static inline
    bool iskanji(u_char c)
{
	return (c & 0x80);
}

void	VtEmu(const char *buff, int nchars)
{
    u_char	ch;

    hide_cursor();
    while (nchars-- > 0) {
	ch = *buff;
	buff ++;
	if (! ch)
	    continue;
	if (con.esc) {
	    con.esc(ch);
	} else switch (ch) {
	case CHAR_BEL:
	    Beep();
	    break;
	case CHAR_DEL:
	    break;
	case CHAR_BS:
	    if (con.x) con.x --;
#if 0
	    if (screen_data[con.y][con.x].attr & S_KANJI2) con.x --;
#endif
	    con.wrap = FALSE;
	    break;
	case CHAR_HT:
	    con.x += con.tab - (con.x % con.tab);
	    con.wrap = FALSE;
	    if (con.x > con.xmax) con.x -= con.xmax + 1;
	    else break;
	case CHAR_VT:
	case CHAR_FF:
	    con.trans = CS_LEFT;
	case CHAR_LF:
	    con.wrap = FALSE;
	    if (con.y == con.ymax) scroll ++;
	    else con.y ++;
	    break;
	case CHAR_CR:
	    con.x = 0;
	    con.wrap = FALSE;
	    break;
	case CHAR_ESC:
	    con.esc = EscStart;
	    continue;
	case CHAR_SO:
	    con.trans = con.g[1] | G1_SET;
	    continue;
	case CHAR_SI:
	    con.trans = con.g[0];
	    continue;
/*	case ' ': con.trans = CS_LEFT;*/
	default:
	   assert(con.x <= con.xmax);
	   assert(!con.wrap);
#if 0
	    if (con.x == con.xmax + 1) {
		con.wrap = TRUE;
		con.x --;
	    }
	    if (con.wrap) {
		con.x -= con.xmax;
		if (con.y == con.ymax) scroll ++;
		else con.y ++;
		con.wrap = FALSE;
		buff --;
		nchars ++;
		break;
	    }
#endif
	    if (con.knj1) {
		/*  2 ⡼ */
		if (con.knj1 & 0x80) {
		    if (con.knj1 == (u_char)CHAR_SS2) {
			/* handling 'kata-kana' */
			if (con.ins) TextInsertChar(1);
#ifdef JISX0201
			TextWput(con.knj1, ch);
			con.x ++;
			con.knj1 = 0;
#else  /* JISX0201 */
			TextSput(ch);
			con.x ++;
			con.knj1 = 0;
#endif /* JISX0201 */
		    }
		    con.knj1 &= 0x7F;
		    ch &= 0x7F;
		}
		if (con.ins) TextInsertChar(2);
		TextWput(con.knj1, ch);
		con.x += 2;
		con.knj1 = 0;
	    } else {
		if (con.trans == CS_DBCS
		    || (iskanji(ch) && con.trans == CS_LEFT)) {
		   /*  1 ⡼ */
		   if (con.x == con.xmax) con.wrap = TRUE;
		   con.knj1 = ch;
		} else {
		   /* ANK ⡼ */
		   if (con.ins) TextInsertChar(1);
		   TextSput(con.trans == CS_RIGHT ? ch | 0x80: ch);
		   con.x ++;
		}
	    }
	    if (con.x > con.xmax) con.wrap = TRUE;
	    if (con.wrap) {
		con.x = 0;
		if (con.y >= con.ymax) {
		    scroll ++;
		    con.y = con.ymax;
		}
		else con.y ++;
		con.wrap = FALSE;
	    }
	}
	if (scroll > 0) {
	    ScrollUp(scroll);
	} else if (scroll < 0) {
	    ScrollDown(- scroll);
	}
	scroll = 0;
    }
    show_cursor();
}

#include "mgl2.h"

extern struct textscreen *ts;

void	VtInit(void)
{
    con.text_mode = TRUE;
#if 0
    DefineCap("Coding", ConfigCoding,
	      "JISX0201.1976-0 JISX0208.1983-0 EUCJ"); 
#endif
}


#ifndef NO_COLOR
static int conv_color(char *val) {
	int r,g,b,c;
	int i;
	int ret = -1;

	if (*val != '#') {
		for (i=0; i< sizeof(color_tab)/sizeof(color_tab[0]); i++) {
			if (!strcasecmp(color_tab[i].name,val)) {
				val = color_tab[i].val;
				break;
			}
		}
	}
	if (*val == '#') {
		if (sscanf(val,"#%02x%02x%02x",&r,&g,&b) != 3)  {
			return ret;
		}
		r >>= 4;
		g >>= 4;
		b >>= 4;
		c = packRGB(r,g,b);
		c = mc_from_rgb(c);
		ret = c;
		return ret;
	}
	return ret;
}
#endif

void	VtStart(int reverse)
{
    int i,j;
    struct winsize text_win;
    void *p;
    char *ep;

    set_font(font_size,0);
    font_width = get_font_width()/2;
    font_height = get_font_height();
#ifdef DEBUG
printf("font_height = %d font_width = %d\n",font_height,font_width);
#endif

    COLS = screen_width / font_width;
    LINES = screen_height / font_height;
#if 1 /*def DEBUG*/
printf("COLS = %d LINES = %d\n",COLS,LINES);
printf("mod x = %d mod y = %d\n"
		,screen_width-(COLS*font_width)
		,screen_height-(LINES*font_height));
#endif
    p = (void *)malloc(sizeof(struct screen_data)*MAXCOLS*MAXLINES);
    for (i=0; i < MAXLINES; i++) {
	screen_data[i] = (struct screen_data *)(p
				+ sizeof(struct screen_data)*MAXCOLS*i);
    }

    /* xmax, ymax  kon.cfg ɤǤʤʬʤ*/
    con.x = con.y = 0;
    con.xmax = COLS-1;
    con.ymax = LINES-1;
    con.tab = 8;
    con.fcol = 8;
    con.bcol = 9;
    con.attr = 0;
    con.esc = NULL;
    con.g[0] = con.g[1] = CS_LEFT;
    con.trans = con.soft = con.ins = con.wrap = FALSE;
    con.active = TRUE;

    for (i=0; i< LINES; i++) for (j=0; j<COLS; j++) {
	screen_data[i][j].attr = S_NULL;
	screen_data[i][j].ch = 0;
    }


    text_win.ws_row = LINES;
    text_win.ws_col = COLS;
    ioctl(ptyfd, TIOCSWINSZ, &text_win);

#ifndef NO_COLOR
    for (i=0; i<8; i++) {
        int col;
    	char col_name[32];
	sprintf(col_name,"FGCOLOR%d",i);
    	if ((ep = mgl_getenv(col_name)) && (col = conv_color(ep)) >= 0) {
		ctable[i] = col;
	}
	sprintf(col_name,"BGCOLOR%d",i);
    	if ((ep = mgl_getenv(col_name)) && (col = conv_color(ep)) >= 0) {
		ctable_bg[i] = col;
	}
    }
    {
        int col;
    	if ((ep = mgl_getenv("FGCOLOR")) && (col = conv_color(ep)) >= 0) {
		ctable[8] = col;
		ctable_bg[8] = col;
	}
    	if ((ep = mgl_getenv("BGCOLOR")) && (col = conv_color(ep)) >= 0) {
		ctable[9] = col;
		ctable_bg[9] = col;
	}
    }
#endif
    if (reverse) {
        for (i=0; i< 10; i++) {
	    ctable[i]    = (ctable[i] & ~0xf) | (15 - (ctable[i] & 0xf));
	    ctable_bg[i] = (ctable_bg[i] & ~0xf) | (15 - (ctable_bg[i] & 0xf));
        }
    }
    ts_set_bgcolor(ts, ctable[9]);
    ts_clear(ts);
    if (ep = mgl_getenv("CURSOR_SHAPE")) {
	if (!strcmp("BLOCK",ep)) {
		cursor_shape = CURSOR_BLOCK;
	} else if (!strcmp("UL",ep)) {
		cursor_shape = CURSOR_UL;
	} else if (!strcmp("RECT",ep)) {
		cursor_shape = CURSOR_RECT;
	}
    }
}

void	VtChangeFont(int fsize)
{
    int i,j;
    struct winsize text_win;
    void *p;
    char *ep;

    hide_cursor();
    font_size = fsize;
    set_font(font_size,0);
    font_width = get_font_width()/2;
    font_height = get_font_height();

    COLS = screen_width / font_width;
    LINES = screen_height / font_height;

    for (i=0; i< LINES; i++) for (j=0; j<COLS; j++) {
	screen_data[i][j].attr = S_NULL;
	screen_data[i][j].ch = 0;
    }

    /* xmax, ymax  kon.cfg ɤǤʤʬʤ*/
    con.x = con.y = 0;
    con.xmax = COLS-1;
    con.ymax = LINES-1;
    con.tab = 8;
    con.fcol = 8;
    con.bcol = 9;
    con.attr = 0;
    con.esc = NULL;
    con.g[0] = con.g[1] = CS_LEFT;
    con.trans = con.soft = con.ins = con.wrap = FALSE;
    con.active = TRUE;

    for (i=0; i< LINES; i++) for (j=0; j<COLS; j++) {
	screen_data[i][j].attr = S_NULL;
	screen_data[i][j].ch = 0;
    }

    text_win.ws_row = LINES;
    text_win.ws_col = COLS;
    ioctl(ptyfd, TIOCSWINSZ, &text_win);

    ts_set_bgcolor(ts, ctable[9]);
    ts_clear(ts);
    show_cursor();
}

void Beep() {
	set_color(COLOR_REVERSE);
	fill_rect(ts->x,ts->y,ts->xs,ts->ys);
	refresh();
	fill_rect(ts->x,ts->y,ts->xs,ts->ys);
	refresh();
}

void ScrollUp(l) int l;  {
	int y;
#if 0
	printf("scroll up %d\n",l);
#endif
	if (l > LINES) l = LINES;
#if 1	/* scroll region fix */
	if (l > con.ymax - con.ymin + 1)
	    l = con.ymax - con.ymin + 1;

	for (y = con.ymin; y < con.ymax + 1 - l; y++) {
#else
	for (y = 0; y < LINES - l; y++) {
#endif
	    int x, x0, x1, yy;
	    struct screen_data *sp, *ssp;

	    x0 = 0; x1 = -1; yy = ts->y+y*font_height;
	    sp = screen_data[y]; ssp = screen_data[y+l];
	    for (x = 0; x < COLS; x++, sp++, ssp++) {
		if (sp->attr != S_NULL || ssp->attr != S_NULL) {
		    if (x1 >= 0) {
			if ((x - x1) * font_width > 48) {
			    int xx0, xx1;
			    xx0 = (ts->x+x0*font_width) & ~3;
			    xx1 = ((ts->x+x1*font_width - 1) & ~3) + 4;
#if 0
			    printf("(%d %d) ", xx0, xx1);
#endif
			    bitblt(NULL, xx0, yy,
				   NULL, xx0, yy + l * font_height,
				   xx1 - xx0 , font_height, 0);
			    x0 = x;
			}
			x1 = -1;
		    }
		    *sp = *ssp;
		} else if (x1 < 0) {
		    x1 = x;
		}
	    }
	{
	    int xx0 = (ts->x+x0*font_width) & ~3;
	    int xx1;
	    yy = ts->y+y*font_height;

	    if (x1 < 0) x1 = COLS;
	    xx1 = ((ts->x+x1*font_width - 1) & ~3) + 4;
#if 0
	    printf("(%d %d) ", xx0, xx1);
#endif
	    bitblt(NULL, xx0, yy,
		   NULL, xx0, yy + l * font_height,
		   xx1 - xx0, font_height, 0);
	}
#if 0
	putchar('\n');
#endif
	}
#if 0
	bitblt(NULL,ts->x,ts->y,NULL,ts->x,ts->y+l*font_height,ts->xs,ts->ys - l*font_height,0);
	memmove(&screen_data[0][0],&screen_data[l][0]
	   ,(void *)(&screen_data[LINES][0]) - (void *)(&screen_data[l][0]));
#endif
	set_color(get_font_color(COL_BG));
#if 1	/* scroll region fix */
	fill_rect(ts->x,ts->y+(con.ymax+1-l)*font_height,ts->xs,l*font_height);
	memset(&screen_data[con.ymax+1-l][0],0
		,(l)*sizeof(struct screen_data)*COLS);
#else
	fill_rect(ts->x,ts->y+ts->ys-l*font_height,ts->xs,l*font_height);
	memset(&screen_data[LINES-l][0],0
	   ,(void *)(&screen_data[l][0]) - (void *)(&screen_data[0][0]));
#endif
}

void ScrollDown(l) int l;  {
	if (l > LINES) l = LINES;
#if 0
printf("scroll down %d\n",l);
#endif
	TextMoveDown(con.ymin, con.ymax, l);
#if 0	/* use TextMoveDown() to scroll only in scroll region */
	bitblt(NULL,ts->x,ts->y+l*font_height,NULL,ts->x,ts->y,ts->xs,ts->ys - l*font_height,0);
	memmove(&screen_data[l][0],&screen_data[0][0]
	   ,(void *)(&screen_data[LINES][0]) - (void *)(&screen_data[l][0]));
	set_color(get_font_color(COL_BG));
	fill_rect(ts->x,ts->y,ts->xs,l*font_height);
	memset(&screen_data[0][0],0
	   ,(void *)(&screen_data[l][0]) - (void *)(&screen_data[0][0]));
#endif
}

void TextClearAll() {
	ts_clear(ts);
	memset(&screen_data[0][0],0
	   ,(void *)(&screen_data[LINES][0]) - (void *)(&screen_data[0][0]));
}

void TextClearEol(c) int c;  {
	set_color(get_font_color(COL_BG));
	if (con.x < COLS)
	fill_rect(ts->x+con.x*font_width,ts->y+con.y*font_height,ts->xs-con.x*font_width,font_height);
	memset(&screen_data[con.y][con.x],0
	   ,(void *)(&screen_data[con.y][COLS])
		-(void *)(&screen_data[con.y][con.x]));
}

void TextClearBol(c) int c;  {
	set_color(get_font_color(COL_BG));
	fill_rect(
	ts->x,
	ts->y+con.y*font_height,
	ts->x+(con.x+1)*font_width,
	font_height);
	memset(&screen_data[con.y][0],0
	   ,(void *)(&screen_data[con.y][con.x])
		-(void *)(&screen_data[con.y][0] + 1));
}

void TextClearEos(c) int c;  {
	set_color(get_font_color(COL_BG));
	if (con.x < COLS) {
	   fill_rect(ts->x+con.x*font_width,ts->y+con.y*font_height,ts->xs-con.x*font_width,font_height);
	   memset(&screen_data[con.y][con.x],0
	      ,(void *)(&screen_data[con.y][COLS])
		-(void *)(&screen_data[con.y][con.x]));
	}
	if (con.y+1 < LINES) {
	   fill_rect(ts->x,ts->y+(con.y+1)*font_height,ts->xs,ts->ys-(con.y+1)*font_height);
	   memset(&screen_data[con.y+1][0],0
	      ,(void *)(&screen_data[LINES][0])
		-(void *)(&screen_data[con.y+1][0]));
	}
}

void TextDeleteChar(c) int c; {
	int x1,x2,xs,xx;
	xx = right_char(c,0);
	x1 = ts->x + con.x*font_width;
	x2 = ts->x + (xx)*font_width;
	xs = ts->xs - (xx)*font_width;
	if (xs > 0) {
		bitblt(NULL,x1,ts->y+con.y*font_height,NULL,x2,ts->y+con.y*font_height,xs,font_height,0);
		memmove(&screen_data[con.y][con.x],&screen_data[con.y][xx]
	   		,(void *)(&screen_data[con.y][COLS]) 
			- (void *)(&screen_data[con.y][xx]));
	}
#if 0
printf("delete char %d\n",c);
#endif
}

void TextInsertChar(c) int c; {
	int x1,x2,xs;
	x1 = ts->x + con.x*font_width;
	x2 = ts->x + (con.x+c)*font_width;
	xs = ts->xs - (con.x+c)*font_width;
	if (xs > 0) {
		bitblt(NULL,x2,ts->y+con.y*font_height,NULL,x1,ts->y+con.y*font_height,xs,font_height,0);
		memmove(&screen_data[con.y][con.x+c],&screen_data[con.y][con.x]
	   		,(void *)(&screen_data[con.y][COLS]) 
			- (void *)(&screen_data[con.y][con.x+c]));
		set_color(get_font_color(COL_BG));
		/* kaga */
		fill_rect(x1,ts->y+con.y*font_height,font_width*c,font_height);
		memset(&screen_data[con.y][con.x],0,c);

	}
#if 0
printf("ins char %d\n",c);
#endif
}

void TextMoveUp(top,btm,line) int top, btm, line; {
	int y1,y2,y3,ys;
#if 0
printf("TextMoveUp t %d b %d l %d\n",top,btm,line);
#endif
	if (btm +1 > LINES) btm = LINES-1;
	if (line > btm - top +1) line = btm-top+1;
	y1 = ts->y + top*font_height;
	y2 = ts->y + (top+line)*font_height;
	y3 = ts->y + (btm+1 - line)*font_height;
	ys = (btm - top +1 -line)*font_height;
	if (ys > 0) {
		bitblt(NULL,ts->x,y1,NULL,ts->x,y2,ts->xs,ys,0);
		memmove(&screen_data[top][0],&screen_data[top+line][0]
			,(btm-line-top+1)*sizeof(struct screen_data)*COLS);
	}
	set_color(get_font_color(COL_BG));
	fill_rect(ts->x,y3,ts->xs,line*font_height);
	memset(&screen_data[btm+1-line][0],0
		,(line)*sizeof(struct screen_data)*COLS);
}

void TextMoveDown(top,btm,line) int top,btm,line; {
	int y1,y2,ys;
#if 0
printf("TextMoveDown t %d b %d l %d\n",top,btm,line);
#endif
	if (btm +1 > LINES) btm = LINES-1;
	if (line > btm - top +1) line = btm-top+1;
	y1 = ts->y + top*font_height;
	y2 = ts->y + (top+line)*font_height;
	ys = (btm - top +1 -line)*font_height;
	if (ys > 0) {
		bitblt(NULL,ts->x,y2,NULL,ts->x,y1,ts->xs,ys,0);
		memmove(&screen_data[top+line][0],&screen_data[top][0]
			,(btm-line-top+1)*sizeof(struct screen_data)*COLS);
	}
	set_color(get_font_color(COL_BG));
	fill_rect(ts->x,y1,ts->xs,line*font_height);
	memset(&screen_data[top][0],0
		,(line)*sizeof(struct screen_data)*COLS);
}

void TextWput(ch1,ch2) int ch1,ch2; {
	int attr=0;
	int x,y,xs,ys;
	x = ts->x+con.x*font_width;
	y = ts->y+con.y*font_height;
	xs = font_width*2;
	ys = font_height;

	if (con.attr & ATTR_ITALIC) attr |= FA_ITALIC;
	if (con.attr & ATTR_HIGH) attr |= FA_BOLD;
	set_font(font_size,attr);
	set_color(get_font_color(COL_BG));
	fill_rect(x,y,xs,ys);
	set_color(get_font_color(COL_FG));
#ifndef MGL1
	draw_font(x,y,(ch1<<8)|ch2,DIR_NORTH);
#else
	draw_font(x,y,(ch1<<8)|ch2);
#endif
	if (con.attr & ATTR_ULINE) {
		draw_line(x,y+ys-1,x+xs-1,y+ys-1);
	}
	screen_data[con.y][con.x].attr = S_KANJI1 | con.attr;
	screen_data[con.y][con.x].ch = ch1;
	screen_data[con.y][con.x+1].attr = S_KANJI2 | con.attr;
	screen_data[con.y][con.x+1].ch = ch2;
}

void TextSput(ch) int ch; {
	int attr=0;
	int x,y,xs,ys;
	x = ts->x+con.x*font_width;
	y = ts->y+con.y*font_height;
	xs = font_width;
	ys = font_height;

	if (con.attr & ATTR_ITALIC) attr |= FA_ITALIC;
	if (con.attr & ATTR_HIGH) attr |= FA_BOLD;
	set_font(font_size,attr);
	set_color(get_font_color(COL_BG));
	fill_rect(x,y,xs,ys);
	set_color(get_font_color(COL_FG));
#ifndef MGL1
	draw_font(x,y,ch&0x7f, DIR_NORTH);
#else
	draw_font(x,y,ch&0x7f);
#endif
	if (con.attr & ATTR_ULINE) {
		draw_line(x,y+ys-1,x+xs-1,y+ys-1);
	}
	screen_data[con.y][con.x].attr = S_ANK | con.attr;
	screen_data[con.y][con.x].ch = ch;
}

void hide_cursor() {
	if (cxs > 0) {
		set_color(COLOR_REVERSE);
		if (cursor_shape == CURSOR_BLOCK) {
			fill_rect(cx,cy,cxs,font_height);
		} else if (cursor_shape == CURSOR_UL) {
			draw_rect(cx,cy+font_height-2,cxs,2);
		} else if (cursor_shape == CURSOR_RECT) {
			draw_rect(cx,cy,cxs,font_height);
		}
		cxs = -1;
	}
}

void show_cursor() {

	cx = ts->x + con.x * font_width;
	cy = ts->y + con.y * font_height;
	cxs = font_width;
	if(screen_data[con.y][con.x].attr & S_KANJI1) {
		cxs *= 2;
	}
	set_color(COLOR_REVERSE);
	if (cursor_shape == CURSOR_BLOCK) {
		fill_rect(cx,cy,cxs,font_height);
	} else if (cursor_shape == CURSOR_UL) {
		draw_rect(cx,cy+font_height-2,cxs,2);
	} else if (cursor_shape == CURSOR_RECT) {
		draw_rect(cx,cy,cxs,font_height);
	}
#ifndef MGL1
	im_avoid_point(cx,cy,1);
	im_avoid_point(cx+cxs-1,cy+font_height-1,0);
#endif
}
