/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/fcntl.h>
#include <sys/time.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include "mgl2.h"
#include "mglcol.h"
#include "config.h"
#include "event_man.h"

struct screen *physical_screen,*current_screen;
struct screen *(*_create_memscreen[16])(int,int,char *,int);
int mgl_screen_type;		/* 2pp,4bpp .... etc */
int mgl_screen_realtype;	/* draw_engine */
int mgl_screen_width = 640;
int mgl_screen_height = 240;
char *mgl_screen_addr;
char *mgl_screenattr_addr;
int mgl_screen_offset;
int mgl_screen_realwidth;
int mgl_screen_bpp;
char mgl_screen_name[32];

int mgl_apli_type;
int mgl_client;
int mgl_noclient;
char mgl_display[32];
static int need_close;
static int need_atexit = 1;

int mgl_key_ok;
int mgl_key_cancel;
int mgl_key_up;
int mgl_key_down;
int mgl_key_left;
int mgl_key_right;
int mgl_key_A;
int mgl_key_B;

/* internal function */
int (*im_readp)(char *ibuf, int ibufsize);
#ifdef IM_STATIC
extern int im_read(char *ibuf, int ibufsize);
#endif
#ifdef IM_DYNAMIC
static void *im_handler;
void im_load(char *name);
#endif
#ifdef GP_DYNAMIC
static void *gp_handler;
int (*mgl_gp_proc)(int type,int x,int y);
void mgl_gp_load(char *name);
#endif

int (*mgl_calibrate_ctrl)(int,int *,int);

int mgl_calibrate_start(int *data,int max_data) {
	int ret = -1;
	if (mgl_calibrate_ctrl) {
		ret = (mgl_calibrate_ctrl(MGL_CALIBRATE_GET,data,max_data));
		if (ret >= 0) {
			(mgl_calibrate_ctrl(MGL_CALIBRATE_RESET,NULL,0));
		}
	}
	return ret;
}

int mgl_calibrate_end(int *data) {
	int ret = -1;
	if (mgl_calibrate_ctrl) {
		ret = mgl_calibrate_ctrl(MGL_CALIBRATE_SET, data,0);
	}
	return ret;
}

static struct sighandler {
  int sig;
  void (*handler)(int);
} orighandlers[] = {
  {SIGHUP, NULL},
  {SIGINT, NULL},
  {SIGABRT, NULL},
  {SIGTERM, NULL},
  {SIGPIPE, NULL},
  {SIGSEGV, NULL}
};

static struct term_funcs {
	void (*func)();
	void *arg;
} term_funcs[10];
static int term_funcs_cnt;

void mgl_add_term_func(void (*func)(),void *arg) {
	if (term_funcs_cnt < 10) {
		term_funcs[term_funcs_cnt].func = func;
		term_funcs[term_funcs_cnt].arg = arg;
		term_funcs_cnt++;
	}
}

static struct atfork_funcs {
	void (*func)();
	void *arg;
} atfork_funcs[10];
static int atfork_funcs_cnt;

void mgl_add_atfork_func(void (*func)(),void *arg) {
	if (atfork_funcs_cnt < 10) {
		atfork_funcs[atfork_funcs_cnt].func = func;
		atfork_funcs[atfork_funcs_cnt].arg = arg;
		atfork_funcs_cnt++;
	}
}

static void signal_exit(int);

int (*mgl_event_manager_initiator)();

/* եå̤Ѳǽˤޤ */

mgl_ignore_auto_close() {
	int i = atfork_funcs_cnt;
	need_close = 0;
	
	while(i > 0) {
		i--;
		atfork_funcs[i].func(atfork_funcs[i].arg);
	}
}

int open_graph(void) {

	int i,em_ok;
	char *p;

#ifdef SUPPORT_GENERIC_4COLOR
	extern void dec4_init();
#endif
#ifdef SUPPORT_GENERIC_192COLOR
	extern void dec192_init();
#endif
#ifdef SUPPORT_GENERIC_FULLCOLOR
	extern void dec3k_init();
#endif
#ifdef SUPPORT_GENERIC_16COLOR
	extern void dec16_init();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR
	extern void NATIVE_DRAW_ENGINE_INITIATOR();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR1
	extern void NATIVE_DRAW_ENGINE_INITIATOR1();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR2
	extern void NATIVE_DRAW_ENGINE_INITIATOR2();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR3
	extern void NATIVE_DRAW_ENGINE_INITIATOR3();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR4
	extern void NATIVE_DRAW_ENGINE_INITIATOR4();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR5
	extern void NATIVE_DRAW_ENGINE_INITIATOR5();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR6
	extern void NATIVE_DRAW_ENGINE_INITIATOR6();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR7
	extern void NATIVE_DRAW_ENGINE_INITIATOR7();
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR8
	extern void NATIVE_DRAW_ENGINE_INITIATOR8();
#endif
#ifdef EVENT_MANAGER_INITIATOR
	extern int EVENT_MANAGER_INITIATOR();
#endif
#ifdef EVENT_MANAGER_X_INITIATOR
	extern int EVENT_MANAGER_X_INITIATOR();
#endif
	extern int FONT_ENGINE_INITIATOR();
#ifdef SUPPORT_CLIENT
	extern int emc_init();
#endif
#ifdef SUPPORT_GENERIC_4COLOR
	dec4_init(STK_GENERIC_4COLOR);
#endif
#ifdef SUPPORT_GENERIC_192COLOR
	dec192_init(STK_GENERIC_192COLOR);
#endif
#ifdef SUPPORT_GENERIC_FULLCOLOR
	dec3k_init(STK_GENERIC_FULLCOLOR);
#endif
#ifdef SUPPORT_GENERIC_16COLOR
	dec16_init(STK_GENERIC_16COLOR);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR1
	NATIVE_DRAW_ENGINE_INITIATOR1(STK_MD_BASE+0);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR2
	NATIVE_DRAW_ENGINE_INITIATOR2(STK_MD_BASE+1);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR3
	NATIVE_DRAW_ENGINE_INITIATOR3(STK_MD_BASE+2);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR4
	NATIVE_DRAW_ENGINE_INITIATOR4(STK_MD_BASE+3);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR5
	NATIVE_DRAW_ENGINE_INITIATOR5(STK_MD_BASE+4);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR6
	NATIVE_DRAW_ENGINE_INITIATOR6(STK_MD_BASE+5);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR7
	NATIVE_DRAW_ENGINE_INITIATOR7(STK_MD_BASE+6);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR8
	NATIVE_DRAW_ENGINE_INITIATOR8(STK_MD_BASE+7);
#endif
#ifdef NATIVE_DRAW_ENGINE_INITIATOR
	NATIVE_DRAW_ENGINE_INITIATOR(STK_NATIVE);
#endif

	FONT_ENGINE_INITIATOR();

	if (!mgl_display[0] && !mgl_noclient) {
		char *p = getenv("MGL_DISPLAY");
		if (p) strcpy(mgl_display,p);
	}
	em_ok = 0;
#ifdef SUPPORT_CLIENT
	if (mgl_display[0]) {
		em_ok = !emc_init(1);
	}
#endif
	if (!mgl_display[0] && !em_ok) {
	    if (mgl_event_manager_initiator) {
		em_ok = !  mgl_event_manager_initiator(1);
	    } else {
#ifdef EVENT_MANAGER_X_INITIATOR
	        if (getenv("DISPLAY") && !em_ok) {
		    em_ok = ! EVENT_MANAGER_X_INITIATOR(1);
#ifdef EVENT_MANAGER_INITIATOR
	        } else if (!em_ok) {
		    em_ok = ! EVENT_MANAGER_INITIATOR(1);
#endif
	        }else if (!em_ok) {
		    em_ok = ! EVENT_MANAGER_X_INITIATOR(1);
		}
#else /* ! defined EVENT_MANAGER_X_INITIATOR */
#ifdef EVENT_MANAGER_INITIATOR
	        if (!em_ok) {
		    em_ok = ! EVENT_MANAGER_INITIATOR(1);
	        }
#endif
#endif
	   }
	}

	if (!em_ok) {
	    while(term_funcs_cnt > 0) {
		term_funcs_cnt--;
		term_funcs[term_funcs_cnt].func(term_funcs[term_funcs_cnt].arg);
	    }
	    if (mgl_fe && mgl_fe->_term) {
		mgl_fe->_term();
	    }
	    exit(1);
	}

	set_color(COLOR_WHITE);
	set_font(12,0);
	clear_screen();
	set_color(COLOR_BLACK);

	for (i = 0; i < sizeof(orighandlers) / sizeof(struct sighandler); i++) {
    	    orighandlers[i].handler = signal(orighandlers[i].sig, signal_exit);
	    if (orighandlers[i].handler == SIG_IGN) {
		signal(orighandlers[i].sig, orighandlers[i].handler);
	    }
	}
#ifdef GP_DYNAMIC
	p = mgl_getenv("MGLGP");
	if (p) mgl_gp_load(p);
#endif
#ifdef IM_DYNAMIC
	p = mgl_getenv("MGLIM");
	if (p) im_load(p);
#endif
#ifdef IM_STATIC
	im_readp = im_read;
#endif
	need_close = 1;
	if (need_atexit) {
		atexit(close_graph);
		need_atexit = 0;
	}
	return 1;
}

#if defined(SYM_PREFIX)
#define SYMPREFIX "_"
#else
#define SYMPREFIX
#endif

#if defined(IM_DYNAMIC) || defined(GP_DYNAMIC)
static void load_module(char *name,char *symname,void **handlerp, void **funcp) {
	char buf[256];

	if ((geteuid() == 0) && (strchr(name,'/'))) {
		fprintf(stderr,"security checked: -- im_load %s\n",name);
		return;
	}
	if (*handlerp) dlclose(*handlerp);
	*handlerp = 0;
	*funcp = 0;
	if (name) {
	    if (name[0] == '/') {
		buf[0] = 0;
	    } else {
		strcpy(buf,MGLDIR);
		strcat(buf,"/");
	    }
	    strcat(buf,name);
	    if (!(strlen(name) > 3 && !strcmp(name+strlen(name)-3,".so"))) {
		strcat(buf,".so");
	    }
	    *handlerp = dlopen(buf,1);

	    fprintf(stderr, "%x = dlopen(%s)\n", *handlerp, buf);

	    if (*handlerp) {
		*funcp = dlsym(*handlerp,symname);
	    }
	}
  return;
}
#endif

void im_load(char *name) {
#ifdef IM_DYNAMIC
	load_module(name,SYMPREFIX "im_read",&im_handler,(void **)(&im_readp));
#endif
}

void mgl_gp_load(char *name) {
#ifdef GP_DYNAMIC
	load_module(name,SYMPREFIX "gp_proc"
		,&gp_handler,(void **)(&mgl_gp_proc));
#endif
}

#if 0
void toggle_im_mode(void) {
    put_key('O'&0x1f);
    return;
}

int get_im_mode(void) {
    return im_mode;
}
#endif

/* եå̤λѤ򽪤ޤ */
void close_graph(void) {
	if (!need_close) return;
	need_close = 0;

	// printf("exec close_glaph(%d)...\n", term_funcs_cnt);
	while(term_funcs_cnt > 0) {
		term_funcs_cnt--;
		term_funcs[term_funcs_cnt].func(term_funcs[term_funcs_cnt].arg);
	}
	if (mgl_em && mgl_em->_term) {
		mgl_em->_term();
		mgl_em = NULL;
	}
	if (mgl_fe && mgl_fe->_term) {
		mgl_fe->_term();
		mgl_fe = NULL;
	}
	// printf("end of close_glaph()\n");

}

static void
signal_exit(n) 
int n;
{
  struct sighandler *sigp;

  close_graph();

  for (sigp = orighandlers; 
       sigp < orighandlers + sizeof(orighandlers) / sizeof(struct sighandler);
       sigp++) {
    if (sigp->sig == n && sigp->handler == SIG_IGN) {
	return;
    }
  }
  for (sigp = orighandlers; 
       sigp < orighandlers + sizeof(orighandlers) / sizeof(struct sighandler);
       sigp++) {
    signal(sigp->sig, sigp->handler);
  }
  kill(getpid(), n);		/* this signal is blocked */
}

#ifdef GP_DYNAMIC
static int gp_proc_hook(int type,int x,int y) {
	if (mgl_gp_proc) return mgl_gp_proc(type,x,y);
	return 0;
}
#endif

struct virtual_key *mgl_create_gesture_pad(int x,int y,int xs,int ys) {
	struct virtual_key *vk = NULL;

#ifdef GP_DYNAMIC
	if (!mgl_gp_proc) {
		return vk;
	}
	vk = create_virtual_key3(x,y,xs,ys,MK_V1,MK_V2,MK_V3);
	if (!vk) {
		return NULL;
	}
	vk->callback = gp_proc_hook;
#endif
	return vk;
}

/* ̤ɤĤ֤ޤ */
void clear_screen(void) {
	current_screen->de->_clear_screen(current_screen,current_screen->de);
}

/* ǡ x,y  */
void put_pixel(int x, int y, int col) {
	current_screen->de->_put_pixel(current_screen,x,y,col);
}

/* οɤߤޤop ˡBLT_MASKING ꤷ
   פ뿧ˤĤơƩޤ
*/
int get_pixel(int x, int y, int op) {
	return current_screen->de->_get_pixel(current_screen,x,y,op);
}

void get_pixstream(int x, int y,int *buf,int length,int dir,int op) {
	current_screen->de->_get_pixstream(current_screen,x,y,buf,length
		,dir,op,current_screen->de);
}

void put_pixstream(int x, int y,int *buf,int length,int dir) {
	current_screen->de->_put_pixstream(current_screen,x,y,buf,length
		,dir,current_screen->de);
}

void put_pixstream_rect(int x, int y,int *buf,int length,int dir,int op) {
	current_screen->de->_put_pixstream_rect(current_screen,x,y,buf
		,length,dir,op,current_screen->de);
}


/* 褹Ȥοꤷޤ
        ϡ0: 1:Ť졼 2:뤤졼 3:
        ط뵡ǽϡclear_screen , draw_pixel, draw_rect,fill_rect
        ,draw_string,draw_font  Ǥ
*/
void set_color(int col) {
	pen_color = col;
	current_screen->de->_set_color(current_screen,col);
	return;
}

/*  x,y  */
void draw_pixel(int x, int y) {
	current_screen->de->_draw_pixel(current_screen,x,y);
}

/* (x1,y1)  (x2,y2) ޤޤ */
int draw_line(int x1, int y1, int x2, int y2) {
	current_screen->de->_draw_line(current_screen,x1,y1,x2,y2
		,current_screen->de);
	return 0;
}

/* x,y  xs,ys ΥȢ񤭤ޤ */
void draw_rect(int x, int y, int xs, int ys) {
	xs -= 1;
	ys -= 1;
	draw_line(x,y,x,y+ys);
	draw_line(x+xs,y,x+xs,y+ys);
	draw_line(x,y,x+xs,y);
	draw_line(x,y+ys,x+xs,y+ys);
	return;
}

/* x,y  xs,ys ΥȢɤĤ֤ޤ */
void fill_rect(int x, int y, int xs, int ys) {
	int i;
	xs -= 1;
	for (i=0; i<ys; i++) {
		draw_line(x,y+i,x+xs,y+i);
	}
	return;
}

/* dst  dx,dy  src  sx,sy  xsize,ysize Υ᡼
        žޤ

        op ˡBLT_TILING ꤹȡsrc ϡ󥰤줿
        ̵¤礭ʥ꡼ȸʤޤ

        op  BLT_MASKING+ ꤹ ꤷƩȤ
        ޤ

        BLT_TILING  BLT_MASKING ꤻdx,sxХȶ
	(ɥå)ˤƱ֤ξ˹®ޤ
	(㡧ĥ롤ư̤ܿβ)

        dst  NULL ꤹȡ˻ѤƤ륹꡼
        ꤵ줿ΤȤޤ

        src  NULL ꤹȡ˻ѤƤ륹꡼
        ꤵ줿ΤȤޤ
*/
void bitblt(struct screen *dst, int dx, int dy,
            struct screen *src, int sx, int sy, int xsize, int ysize, int op) {
	if (dst == NULL) dst = current_screen;
	if (src == NULL) src = current_screen;

	dst->de->_bitblt(dst,dx,dy,src,sx,sy,xsize,ysize,op
				,dst->de);
}

/* ꥹ꡼ޤ
        xs,ys ǥӥåñ̤Υꤷޤ
        bitmap ϡNULL ΤȤưޤ
        ) ư줿ȤΤ free_screen  bitmap  free ޤ
*/
struct screen *create_memscreen(int xs, int ys, char *bitmap,int kind, int op) {
	struct screen *ret;
	if (_create_memscreen[kind & ST_KINDMASK]) {
		ret = _create_memscreen[kind & ST_KINDMASK](xs,ys,bitmap,op);
		return ret;
	}
	return NULL;
}

struct screen *conv_screen_from_v1(struct screen_v1 *s,int kind) {
	struct screen *s4=NULL;
	struct screen *sn;
	if (_create_memscreen[STK_GENERIC_4COLOR]) {
		s4 = _create_memscreen[STK_GENERIC_4COLOR](s->width
				,s->height,s->bitmap,0);
		if (kind == STK_GENERIC_4COLOR) return s4;
	}
	if (s4 && _create_memscreen[kind]) {
		sn = _create_memscreen[kind](s->width,s->height,NULL,0);
		bitblt(sn,0,0,s4,0,0,s->width,s->height,0);
		free_screen(s4);
		return sn;
	}
	return NULL;
}


/* org  x,y ΰ֤顢xs,ys Υ
        ʬ꡼ޤ
        ʬ꡼ϡorg Υӥåȥޥåפ򤽤Τޤ޻ѤޤΤǡ
        org  free ѤǤޤ
*/
struct screen *create_subscreen(struct screen *org, int x, int y,
                                int xs, int ys) {
	if (org == NULL) org = current_screen;
	return org->de->_create_subscreen(org,x,y,xs,ys,0);
}

/* create_memscreen create_subscreen Ǻ꡼
        free ޤ
*/
void free_screen(struct screen *s) {
	if (!s) return;
	s->de->_free_screen(s);
}

static struct screen *screen_stack[32];
static int screen_stack_pos;

/* 褹륹꡼ꤹȤȤˡΥ꡼
pen_color 򥻡֤ޤ */
void push_screen(struct screen *s) {
	if (s != current_screen) {
		s->_pen_color = current_screen->_pen_color;
		s->_pen_font = current_screen->_pen_font;
	}
	screen_stack[screen_stack_pos++] = current_screen;
	current_screen = s;
	return;
}

/* 褹륹꡼ pen_color 򸵤ᤷޤ */
void pop_screen(void) {
	current_screen = screen_stack[--screen_stack_pos];
	return;
}

struct event_manager *mgl_em;

int mgl_key_mode;

int mgl_keyboard_reopen = 0;

int mgl_set_keymode(mode) {
	int old = mgl_key_mode;
	int ret;
	ret = mgl_em->_set_keymode(mode & MGL_SK_KM_MASK);
	if (ret == 0) {
		mgl_key_mode = mode;
		mgl_modifier_status = 0;
		if (  (((old & MGL_SK_KM_MASK) == MGL_SK_RAW) 
		       && ((mgl_key_mode & MGL_SK_KM_MASK) != MGL_SK_RAW)) 
		    ||(((old & MGL_SK_KM_MASK) != MGL_SK_RAW) 
		       && ((mgl_key_mode & MGL_SK_KM_MASK) == MGL_SK_RAW)) )
				key_buf_cnt = 0; /* flush key buffer */
	}
	return ret;
}

#define OLD_MASK   (MGL_SKM_CAPS|MGL_SKM_SHIFT|MGL_SKM_CTRL|MGL_SKM_ALT|MGL_SKM_MENU)
#define NEW_MASK   (MGL_SKM_RSHIFT|MGL_SKM_RCTRL|MGL_SKM_RALT|MGL_SKM_RMENU)

int get_key(int time_out) {
	int ret;

retry:
	ret = mgl_em->_get_key(time_out);
	if (ret >= 0) {
		if ((mgl_key_mode & MGL_SK_KM_MASK) ==  MGL_SK_EXTRANSLATED) {
		    if (!(mgl_key_mode & MGL_SK_EXMODIFIER)&&(ret & NEW_MASK)) {
				int r = ret & (~MGL_SKM_MASK|OLD_MASK);
				if (ret & MGL_SKM_RSHIFT)
					r |= MGL_SKM_SHIFT;
				if (ret & MGL_SKM_RCTRL)
					r |= MGL_SKM_CTRL;
				if (ret & MGL_SKM_RALT)
					r |= MGL_SKM_ALT;
				if (ret & MGL_SKM_RMENU)
					r |= MGL_SKM_MENU;
				ret = r;
		    }
		} else if ((mgl_key_mode & MGL_SK_KM_MASK) ==  MGL_SK_RAW) {
			ret &= ~MGL_SKM_MASK;
		} else {
			if (ret & MGL_SKM_NOTICE)
				goto retry;
			ret &= ~MGL_SKM_MASK;
		}
	}
	return ret;
}

void refresh() {
        mgl_em->_refresh();
}

void set_icon(char *icon,char *name) {
        mgl_em->_set_icon(icon,name);
}

int key_select(int nfds, fd_set *readfds, int timeout) {
        return mgl_em->_key_select(nfds,readfds,timeout);
}

char *base76_chars =
"@*+.abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_0123456789%&-=^|;:~";
char base76_color2char[256];
int base76_char2color[256];
int base76_char2num[256];
int base76_initialized;

base76_init() {
    int i,j,c,c2,ch;
    int hue,sat,bri;
    int cnum = 0;
    for (i=0; i<256; i++) {
	base76_char2color[i] = (-1);
	base76_char2num[i] = (-1);
    }
    for (i=0; i<76; i++) {
	base76_char2num[base76_chars[i]]= i;
    }

    for (i=0; i<192; i++) {
	  c = CONV_FROM_COL192(i);
	  unpackMC(c,hue,sat,bri);
	  if ((sat == 0) && (hue != 0)) {
		continue;
	  }
	  if (sat > bri) {
		continue;
	  }
	  ch = base76_chars[cnum ++];
	  base76_color2char[i] = ch;
	  base76_char2color[ch] = c;
    }
    for (i=0; i<192; i++) {
	  c = CONV_FROM_COL192(i);
	  unpackMC(c,hue,sat,bri);
	  if ((sat == 0) && (hue != 0)) {
		hue = 0;
		c2 = packMC(hue,sat,bri);
		j = CONV_TO_COL192(c2);
	  	base76_color2char[i] = base76_color2char[j];
		continue;
	  }
	  if (sat > bri) {
		sat = bri;
		c2 = packMC(hue,sat,bri);
		j = CONV_TO_COL192(c2);
	  	base76_color2char[i] = base76_color2char[j];
		continue;
	  }
    }
    for (; i<256; i++) {
	base76_color2char[i] = base76_color2char[0];
    }
    base76_initialized = 1;
}

int write_screen_mgr(char *name, struct screen *ss,int opt) {
  char buf[MAXPATHLEN];
  char magic_buf[MGR_HEADER_SIZE + 1];
  int x,y,w,h;
  int fd;
  int size;
  struct screen *s;
  int i,j,c,c2,ch;

  if (!base76_initialized) base76_init();
  if (ss == NULL) ss = current_screen;

  strcpy(buf, name);
#if 1
  strcat(buf, ".mgr");
#endif

  /* file open */
  fd = open(buf, O_RDWR | O_CREAT | O_TRUNC, 0644);
  if(fd == -1){
	perror("open");
	return(-1);
  }

  if (opt == STK_GENERIC_192COLOR) {
	unsigned char scanline[128];
	int need_free = 0;
	push_screen(ss);
	c = get_pixel(0,0,0);
	pop_screen();
	if (c & COLOR_DITHER) {
	    s = create_memscreen(ss->width,ss->height,NULL,STK_GENERIC_192COLOR,0);
	    if (!s) {
		close(fd);
		unlink(buf);
		return (-1);
	    }
	    bitblt(s,0,0,ss,0,0,ss->width,ss->height,0);
	    need_free = 1;
	} else {
	    s = ss;
	}

	push_screen(s);

	/* write header */
	sprintf(magic_buf, "#MGR%04d%04d%04d", STK_GENERIC_192COLOR, s->width, s->height);
	if(write(fd, magic_buf, MGR_HEADER_SIZE) != MGR_HEADER_SIZE) {
		perror("write");
		close(fd);
		unlink(buf);
		pop_screen();
		if (need_free) free_screen(s);
		return(-1);
	}
	w = s->width;
	h = s->height;
	i = 0;
	
	for (y=0; y<h; y++) for (x=0; x<w; x++) {
		if ( (x % 72) == 0 ) {
			scanline[i++] = '\n';
		}
		c = get_pixel(x,y,0);
		scanline[i++] = base76_color2char[CONV_TO_COL192(c)];
		if (i>=1000) {
			if(write(fd, scanline, i) != i) {
				perror("write");
				close(fd);
				unlink(buf);
				pop_screen();
				if (need_free) free_screen(s);
				return(-1);
			}
			i=0;
		}
	}
	if (i >= 0) {
		if (scanline[i-1] != '\n')
			scanline[i++] = '\n';
		if(write(fd, scanline, i) != i) {
			perror("write");
			close(fd);
			unlink(buf);
			pop_screen();
			if (need_free) free_screen(s);
			return(-1);
		}
	}
	close(fd);
	pop_screen();
	if (need_free) free_screen(s);
	return (0);
  } else if (opt == STK_GENERIC_FULLCOLOR) {
	unsigned char scanline[1024];
	push_screen(ss);

	/* write header */
	sprintf(magic_buf, "#MGR%04d%04d%04d", STK_GENERIC_FULLCOLOR, ss->width, ss->height);
	if(write(fd, magic_buf, MGR_HEADER_SIZE) != MGR_HEADER_SIZE) {
		perror("write");
		close(fd);
		unlink(buf);
		pop_screen();
		return(-1);
	}
	w = ss->width;
	h = ss->height;
	i = 0;
	for (y=0; y<h; y++) for (x=0; x<w; x++) {
		if ( (x % 36) == 0 ) {
			scanline[i++] = '\n';
		}
		c = get_pixel(x,y,0);
		c &= 0xfff;
		if (c > 76*76) c = 0;

		scanline[i++] = base76_chars[c/76];
		scanline[i++] = base76_chars[c%76];
		if (i>=1000) {
			if(write(fd, scanline, i) != i) {
				perror("write");
				close(fd);
				unlink(buf);
				pop_screen();
				return(-1);
			}
			i=0;
		}
	}
	if (i >= 0) {
		if (scanline[i-1] != '\n')
			scanline[i++] = '\n';
		if(write(fd, scanline, i) != i) {
			perror("write");
			close(fd);
			unlink(buf);
			pop_screen();
			return(-1);
		}
	}
	close(fd);
	pop_screen();

	return (0);
  } else {
	s = create_memscreen(ss->width,ss->height,NULL,STK_GENERIC_4COLOR,0);
	if (!s) {
		close(fd);
		unlink(buf);
		return (-1);
	}
	bitblt(s,0,0,ss,0,0,ss->width,ss->height,0);

	push_screen(s);

	/* write header */
	sprintf(magic_buf, "#MGR%04d%04d%04d", STK_GENERIC_4COLOR, s->width, s->height);
	if(write(fd, magic_buf, MGR_HEADER_SIZE) != MGR_HEADER_SIZE) {
		perror("write");
		close(fd);
		unlink(buf);
		pop_screen();
		return(-1);
	}

	size = s->wbytes * s->height;

	/* write data */
	if(write(fd, s->bitmap, size) != size){
		perror("write");
		close(fd);
		unlink(buf);
		pop_screen();
		return(-1);
	}

	/* file close */
	close(fd);

	pop_screen();

	free_screen(s);
	return(0);
  }
}

struct screen * read_screen_mgr(char *name) {
  char buf[MGR_HEADER_SIZE + 1];
  int fd;
  int ver, w, h;
  int size;
  int x,y,i,rlen,c;
  struct screen *ret = NULL;

  if (!base76_initialized) base76_init();

  /* file open */
  fd = open(name, O_RDONLY);
  if(fd == -1){
    perror("open");
    return(NULL);
  }

  /* read header */
  if(read(fd, buf, MGR_HEADER_SIZE) != MGR_HEADER_SIZE){
    perror("read");
    close(fd);
    return(NULL);
  }
  buf[MGR_HEADER_SIZE] = '\0';
  if(sscanf(buf, "#MGR%04d%04d%04d", &ver, &w, &h) != 3) {
    close(fd);
    return(NULL);
  }

  if (ver == STK_GENERIC_4COLOR) {
	ret = create_memscreen(w, h, NULL,ver,0);
	if(!ret){
		close(fd);
		return(NULL);
	}

	size = (w + 3) / 4 * h;

	/* read data */
	if(read(fd, ret->bitmap, size) != size){
		perror("read");
		close(fd);
		free_screen(ret);
		return(NULL);
	}
  } else if (ver == STK_GENERIC_192COLOR) {
	unsigned char scanline[1024];

	ret = create_memscreen(w, h, NULL,ver,0);
	if(!ret) {
		close(fd);
		return(NULL);
	}
	push_screen(ret);
	i = rlen = 0;
	for (y = 0; y < h; y++) for (x=0; x < w; x++) {
re_read:
		if (i >= rlen) {
			rlen = read(fd, scanline, 1024);
			if (rlen < 0) { 
				perror("read");
				pop_screen();
				close(fd);
				free_screen(ret);
				return(NULL);
			}
			i = 0;
		}
		c = base76_char2color[scanline[i++]];
		while (c < 0) {
			if (i >= rlen) goto re_read;
			c = base76_char2color[scanline[i++]];
		}
		put_pixel(x,y,c);
	}
	pop_screen();
  } else if (ver == STK_GENERIC_FULLCOLOR) {
	unsigned char scanline[1024];
	int c1;

	ret = create_memscreen(w, h, NULL, ver, 0);
	if(!ret) {
		close(fd);
		return(NULL);
	}
	push_screen(ret);
	i = rlen = 0;
	for (y = 0; y < h; y++) for (x=0; x < w; x++) {
		c1 = -1;
re_read2:
		if (i >= rlen) {
			rlen = read(fd, scanline, 1024);
			if (rlen < 0) { 
				perror("read");
				pop_screen();
				close(fd);
				free_screen(ret);
				return(NULL);
			}
			i = 0;
		}
		c = base76_char2num[scanline[i++]];
		while (c < 0) {
			if (i >= rlen) goto re_read2;
			c = base76_char2num[scanline[i++]];
		}
		if (c1 < 0) {
			c1 = c;
			goto re_read2;
		}
		put_pixel(x,y,c1*76+c);
	}
	pop_screen();
  }
  /* file close */
  close(fd);
  return(ret);
}

struct screen * conv_screen_from_mgr(unsigned char *buf,int kind) {
  int ver, w, h;
  int size;
  int x,y,c,c1;
  struct screen *ret = NULL;

  if (!base76_initialized) base76_init();

  if(sscanf(buf, "#MGR%04d%04d%04d", &ver, &w, &h) != 3) {
    return(NULL);
  }
  buf += MGR_HEADER_SIZE;

  if (ver == 2) {
	ret = create_memscreen(w, h, NULL,kind,0);
	if(!ret) {
		return(NULL);
	}
	push_screen(ret);
	for (y = 0; y < h; y++) for (x=0; x < w; x++) {
		if (*buf == 0) goto err;
		c = base76_char2color[*buf++];
		while (c < 0) {
			if (*buf == 0) goto err;
			c = base76_char2color[*buf++];
		}
		put_pixel(x,y,c);
	}
	pop_screen();
  } else if (ver == 3) {

	ret = create_memscreen(w, h, NULL,STK_GENERIC_FULLCOLOR,0);
	if(!ret) {
		return(NULL);
	}
	push_screen(ret);
	for (y = 0; y < h; y++) for (x=0; x < w; x++) {
		if (*buf == 0) goto err;
		c1 = base76_char2num[*buf++];
		while (c1 < 0) {
			if (*buf == 0) goto err;
			c1 = base76_char2num[*buf++];
		}
		if (*buf == 0) goto err;
		c = base76_char2num[*buf++];
		while (c < 0) {
			if (*buf == 0) goto err;
			c = base76_char2num[*buf++];
		}
		put_pixel(x,y,c1*76+c);
	}
	pop_screen();
  }
  return(ret);
err:
  free_screen(ret);
  return NULL;
}

char *conv_screen_to_mgr(struct screen *s,char *buf,int len) {
  char *ret = NULL;
  int x,y,w,h;
  int c;

  if (!base76_initialized) base76_init();
  if (s == NULL) s = current_screen;

  w = s->width;
  h = s->height;

  if ((s->type & ST_KINDMASK) == STK_GENERIC_FULLCOLOR) {

	if (buf == NULL) {
		buf = malloc(MGR_HEADER_SIZE + w*h*2);
		if (!buf) return NULL;
	} else {
		if (len < MGR_HEADER_SIZE + w*h*2) {
			return NULL;
		}
	}
	ret = buf;

	/* write header */
	sprintf(buf, "#MGR%04d%04d%04d", 3, w, h);
	buf += MGR_HEADER_SIZE;

	push_screen(s);
	for (y=0; y<h; y++) for (x=0; x<w; x++) {
		c = get_pixel(x,y,0);
		c &= 0xfff;
		if (c > 76*76) c = 0;

		*buf++ = base76_chars[c/76];
		*buf++ = base76_chars[c%76];
	}
	pop_screen();
  } else {
	if (buf == NULL) {
		buf = malloc(MGR_HEADER_SIZE + w*h);
		if (!buf) return NULL;
	} else {
		if (len < MGR_HEADER_SIZE + w*h) {
			return NULL;
		}
	}
	ret = buf;

	/* write header */
	sprintf(buf, "#MGR%04d%04d%04d", 2, w,h);
	buf += MGR_HEADER_SIZE;

	push_screen(s);
	for (y=0; y<h; y++) for (x=0; x<w; x++) {
		c = get_pixel(x,y,0);
		*buf++ = base76_color2char[CONV_TO_COL192(c)];
	}
	pop_screen();
  }
  return ret;
}

void mgl2_clear_screen(struct screen *s) {
	s->de->_clear_screen(s,s->de);
}

void mgl2_put_pixel(struct screen *s,int x, int y, int col) {
	s->de->_put_pixel(s,x,y,col);
}

int mgl2_get_pixel(struct screen *s,int x, int y, int op) {
	return s->de->_get_pixel(s,x,y,op);
}

void mgl2_get_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir,int op) {
	s->de->_get_pixstream(s,x,y,buf,length ,dir,op,s->de);
}

void mgl2_put_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir) {
	s->de->_put_pixstream(s,x,y,buf,length ,dir,s->de);
}

void mgl2_put_pixstream_rect(struct screen *s,int x, int y,int *buf,int length,int dir,int op) {
	s->de->_put_pixstream_rect(s,x,y,buf ,length,dir,op,s->de);
}


void mgl2_set_color(struct screen *s,int col) {
	s->_pen_color.color = col;
	s->de->_set_color(s,col);
	return;
}

void mgl2_draw_pixel(struct screen *s,int x, int y) {
	s->de->_draw_pixel(s,x,y);
}

int mgl2_draw_line(struct screen *s,int x1, int y1, int x2, int y2) {
	s->de->_draw_line(s,x1,y1,x2,y2 ,s->de);
	return 0;
}

void mgl2_draw_rect(struct screen *s,int x, int y, int xs, int ys) {
	xs -= 1;
	ys -= 1;
	mgl2_draw_line(s,x,y,x,y+ys);
	mgl2_draw_line(s,x+xs,y,x+xs,y+ys);
	mgl2_draw_line(s,x,y,x+xs,y);
	mgl2_draw_line(s,x,y+ys,x+xs,y+ys);
	return;
}

void mgl2_fill_rect(struct screen *s,int x, int y, int xs, int ys) {
	int i;
	xs -= 1;
	for (i=0; i<ys; i++) {
		mgl2_draw_line(s,x,y+i,x+xs,y+i);
	}
	return;
}

static char *mgl_registry(char *name) {
	char *xx,*p;
	char buf[256];
	char fname[256];
	int pass;
	FILE *fp;

   for (pass=0; pass<2; pass++) {
	if (pass == 0) {
	    if (xx = getenv("HOME")) {
		sprintf(fname,"%s/.mgl/registry",xx);
	    } else {
		continue;
	    }
	} else {
		strcpy(fname,MGLDIR "/registry");
	}
	fp = fopen(fname,"r");
	if (!fp) continue;
//printf("mgl_registry: open %s\n",fname);
	while( fgets(buf,256,fp)) {
		char *xx = strchr(buf,'=');
		if (!xx) continue;
		*xx = 0;
		xx ++;

		if (strcmp(name,buf) == 0) {
			if (xx[strlen(xx)-1] == '\n')
				xx[strlen(xx)-1] = 0;
			p = malloc(strlen(xx)+1);
			strcpy(p,xx);
			if (!p) goto next;
//printf("match   : %s = %s\n",buf,p);
			fclose(fp);
			return p;
		}
	}
next:
	fclose(fp);
    }
    return NULL;
}

char *mgl_getenv(char *name) {
	char fname[256];
	char buf[256];
	int i,len,plen;
	char *p,*q;
#ifndef NO_PROGNAME
	char prog_name[256];
	extern char *__progname;
	char *xx = strchr(__progname,' ');
	prog_name[0] = 0;
	strncpy(prog_name,__progname,128);
	if (xx) {
		prog_name[xx - __progname] = 0;
	}
	xx = prog_name + strlen(prog_name);
	strcat(prog_name,"_");
	strcat(prog_name,name);
	p = getenv(prog_name);
	if (p) return p;
	*xx = '.';
	p = getenv(prog_name);
	if (p) return p;
	p = mgl_registry(prog_name);
	if (p) return p[0]?p:NULL;
#endif
	p = getenv(name);
	if (p) return p;
	p = mgl_registry(name);
	return (p&&p[0])?p:NULL;
}
