/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include "mgl2.h"
#include "config.h"
#include "mglkey.h"
#include "keymap.h"
#include "event_man.h"

int mgl_modifier_status;
int mgl_keymap_shift;
int mgl_button_shift;

int mgl_keymap[MGL_KEYMAP_SIZE][3];
int mgl_raw_keymap[MGL_NR_KEYS];

const struct mgl_encoding ec_base[] = {
{	MGL_KEY_ESC,		{'\e',		'\e'}	},
{	MGL_KEY_TAB,		{'\t',		'\t'}	},
{	MGL_KEY_Q,		{'q',		'Q'}	},
{	MGL_KEY_W,		{'w',		'W'}	},
{	MGL_KEY_E,		{'e',		'E'}	},
{	MGL_KEY_R,		{'r',		'R'}	},
{	MGL_KEY_T,		{'t',		'T'}	},
{	MGL_KEY_Y,		{'y',		'Y'}	},
{	MGL_KEY_U,		{'u',		'U'}	},
{	MGL_KEY_I,		{'i',		'I'}	},
{	MGL_KEY_O,		{'o',		'O'}	},
{	MGL_KEY_P,		{'p',		'P'}	},
{	MGL_KEY_ENTER,		{'\r',		'\r'}	},
{	MGL_KEY_CAPS,		{MGL_SKM_CAPS,	MGL_SKM_CAPS}},
{	MGL_KEY_A,		{'a',		'A'}},
{	MGL_KEY_S,		{'s',		'S'}},
{	MGL_KEY_D,		{'d',		'D'}},
{	MGL_KEY_F,		{'f',		'F'}},
{	MGL_KEY_G,		{'g',		'G'}},
{	MGL_KEY_H,		{'h',		'H'}},
{	MGL_KEY_J,		{'j',		'J'}},
{	MGL_KEY_K,		{'k',		'K'}},
{	MGL_KEY_L,		{'l',		'L'}},
{	MGL_KEY_SHIFT,		{MGL_SKM_SHIFT,	MGL_SKM_SHIFT}},
{	MGL_KEY_Z,		{'z',		'Z'}},
{	MGL_KEY_X,		{'x',		'X'}},
{	MGL_KEY_C,		{'c',		'C'}},
{	MGL_KEY_V,		{'v',		'V'}},
{	MGL_KEY_B,		{'b',		'B'}},
{	MGL_KEY_N,		{'n',		'N'}},
{	MGL_KEY_M,		{'m',		'M'}},
{	MGL_KEY_COMMA,		{',',		'<'}},
{	MGL_KEY_PERIOD,		{'.',		'>'}},
{	MGL_KEY_SLASH,		{'/',		'?'}},
{	MGL_KEY_CTRL,		{MGL_SKM_CTRL,	MGL_SKM_CTRL}},
{	MGL_KEY_ALT,		{MGL_SKM_ALT,	MGL_SKM_ALT}},
{	MGL_KEY_MENU,		{MGL_SKM_MENU,	MGL_SKM_MENU}},
{	MGL_KEY_SPACE,		{' ',		' '}},
{	MGL_KEY_F1,		{MK_F1,		MK_F1}	},
{	MGL_KEY_F2,		{MK_F2,		MK_F2}	},
{	MGL_KEY_F3,		{MK_F3,		MK_F3}	},
{	MGL_KEY_F4,		{MK_F4,		MK_F4}	},
{	MGL_KEY_F5,		{MK_F5,		MK_F5}	},
{	MGL_KEY_F6,		{MK_F6,		MK_F6}	},
{	MGL_KEY_F7,		{MK_F7,		MK_F7}	},
{	MGL_KEY_F8,		{MK_F8,		MK_F8}	},
{	MGL_KEY_F9,		{MK_F9,		MK_F9}	},
{	MGL_KEY_F10,		{MK_F10,	MK_F10}	},
{	MGL_KEY_UP,		{MK_UP,		MK_UP}	},
{	MGL_KEY_DOWN,		{MK_DOWN,	MK_DOWN}},
{	MGL_KEY_LEFT,		{MK_LEFT,	MK_LEFT}},
{	MGL_KEY_RIGHT,		{MK_RIGHT,	MK_RIGHT}},
{	MGL_KEY_INS,		{MK_INS,	MK_INS}},
{	MGL_KEY_DEL,		{0x7f,		0x7f}},
{	MGL_KEY_DEL_ALT,	{MK_DEL,	MK_DEL}},
{	MGL_KEY_PAGEUP,		{MK_PAGE_UP,	MK_PAGE_UP}},
{	MGL_KEY_PAGEDOWN,	{MK_PAGE_DOWN,	MK_PAGE_DOWN}},
{	MGL_KEY_HOME,		{MK_HOME,	MK_HOME}},
{	MGL_KEY_END,		{MK_END,	MK_END}},

};
/* other keys */

static const struct mgl_encoding ec_ext[] = {
{	MGL_EXKEY_0,		{MKE_INS,	MKE_0}},
{	MGL_EXKEY_1,		{MKE_END,	MKE_1}},
{	MGL_EXKEY_2,		{MKE_DOWN,	MKE_2}},
{	MGL_EXKEY_3,		{MKE_PAGE_DOWN,	MKE_3}},
{	MGL_EXKEY_4,		{MKE_LEFT,	MKE_4}},
{	MGL_EXKEY_5,		{MKE_BEGIN,	MKE_5}},
{	MGL_EXKEY_6,		{MKE_RIGHT,	MKE_6}},
{	MGL_EXKEY_7,		{MKE_HOME,	MKE_7}},
{	MGL_EXKEY_8,		{MKE_UP,	MKE_8}},
{	MGL_EXKEY_9,		{MKE_PAGE_UP,	MKE_9}},

{	MGL_EXKEY_NUMLOCK,	{MK_NUMLOCK,	MK_NUMLOCK}},
{	MGL_EXKEY_SCREENLOCK,	{MKE_SCRLOCK,	MKE_SCRLOCK}},
{	MGL_EXKEY_MULT,		{MKE_MULT,	MKE_MULT}},
{	MGL_EXKEY_MINUS,	{MKE_MINUS,	MKE_MINUS}},
{	MGL_EXKEY_PLUS,		{MKE_PLUS,	MKE_PLUS}},
{	MGL_EXKEY_DIV,		{MKE_DIV,	MKE_DIV}},

{	MGL_EXKEY_HOME,		{MKE_HOME,	MKE_HOME}},
{	MGL_EXKEY_PRINT,	{MKE_PRINT,	MKE_RETURN}},
{	MGL_EXKEY_F11,		{MK_F11,	MK_F11}},
{	MGL_EXKEY_F12,		{MK_F12,	MK_F12}},
{	MGL_EXKEY_RCTRL,	{MGL_SKM_RCTRL,	MGL_SKM_RCTRL}},
{	MGL_EXKEY_RALT,		{MGL_SKM_RALT,	MGL_SKM_RALT}},

{	MGL_EXKEY_PAUSE,	{MKE_PAUSE,	MKE_BREAK}},
{	MGL_EXKEY_RMENU,	{MKE_RMENU,	MKE_RMENU}},
{	MGL_EXKEY_SELECT,	{MKE_SELECT,	MKE_SELECT}},
{	MGL_EXKEY_KANA,		{MK_NONE,	MK_NONE}},
{	MGL_EXKEY_KANJI,	{MKE_KANJI,	MKE_KANJI}},
{	MGL_EXKEY_MUHENKAN,	{MKE_MUHENKAN,	MKE_MUHENKAN}},

};


const struct mgl_encoding ec_jp[] = {
{	MGL_KEY_HANZEN_J,	{'\e',		'\e'}	},
{	MGL_KEY_1,		{'1',		'!'}	},
{	MGL_KEY_2,		{'2',		'"'}	},
{	MGL_KEY_3,		{'3',		'#'}	},
{	MGL_KEY_4,		{'4',		'$'}	},
{	MGL_KEY_5,		{'5',		'%'}	},
{	MGL_KEY_6,		{'6',		'&'}	},
{	MGL_KEY_7,		{'7',		'\''}	},
{	MGL_KEY_8,		{'8',		'('}	},
{	MGL_KEY_9,		{'9',		')'}	},
{	MGL_KEY_0,		{'0',		'~'}	},
{	MGL_KEY_MINUS,		{'-',		'='}	},
{	MGL_KEY_ASCIICIRCUM_J,	{'^',		'~'}	},
{	MGL_KEY_BACKSLASH_BAR_J,{'\\',		'|'}	},
{	MGL_KEY_BACKSPACE,	{'\b',		'\b'}	},

{	MGL_KEY_AT_J,		{'@',		'`'}	},
{	MGL_KEY_LBRACKET_J,	{'[',		'{'}	},


{	MGL_KEY_SEMICOLON_J,	{';',		'+'}},
{	MGL_KEY_COLON_J,	{':',		'*'}},
{	MGL_KEY_RBRACKET_J,	{']',		'}'}},

{	MGL_KEY_BACKSLASH_UNDER,{'\\',		'_'}},


};

const struct mgl_encoding ec_us[] = {
{	MGL_KEY_1,		{'1',		'!'}	},
{	MGL_KEY_2,		{'2',		'@'}	},
{	MGL_KEY_3,		{'3',		'#'}	},
{	MGL_KEY_4,		{'4',		'$'}	},
{	MGL_KEY_5,		{'5',		'%'}	},
{	MGL_KEY_6,		{'6',		'^'}	},
{	MGL_KEY_7,		{'7',		'&'}	},
{	MGL_KEY_8,		{'8',		'*'}	},
{	MGL_KEY_9,		{'9',		'('}	},
{	MGL_KEY_0,		{'0',		')'}	},
{	MGL_KEY_MINUS,		{'-',		'_'}	},
{	MGL_KEY_EQUAL,		{'=',		'+'}	},
{	MGL_KEY_BACKSLASH_BAR_E,{'\\',		'|'}	},
{	MGL_KEY_GRAVE_E,	{'`',		'~'}	},

{	MGL_KEY_LBRACKET_E,	{'[',		'{'}	},
{	MGL_KEY_RBRACKET_E,	{']',		'}'}	},


{	MGL_KEY_SEMICOLON_E,	{';',		':'}},
{	MGL_KEY_APOSTROPHE_E,	{'\'',		'"'}},

/* for physical japanese keybard */
{	MGL_KEY_BACKSLASH_BAR_J,{'\\',		'|'}	},
{	MGL_KEY_BACKSLASH_UNDER,{'\\',		'_'}},
};


const struct mgl_symval sv_tab[] = {
{	"escape",	'\e'	},
{	"exclam",	'!'	},
{	"quotedbl",	'"'	},
{	"numbersign",	'#'	},
{	"dollar",	'$'	},
{	"percent",	'%'	},
{	"ampersand",	'&'	},
{	"apostrophe",	'\''	},
{	"asterisk",	'*'	},
{	"parenleft",	'('	},
{	"parenright",	')'	},
{	"asciitilde",	'~'	},
{	"minus",	'-'	},
{	"plus",		'+'	},
{	"equal",	'='	},
{	"asciicircum",	'^'	},
{	"asciitilde",	'~'	},
{	"backspace",	'\b'	},
{	"tab",		'\t'	},
{	"at",		'@'	},
{	"bracketleft",	'['	},
{	"bracketright",	']'	},
{	"braceleft",	'{'	},
{	"braceright",	'}'	},
{	"return",	'\r'	},
{	"semicolon",	';'	},
{	"colon",	':'	},
{	"comma",	','	},
{	"less",		'<'	},
{	"period",	'.'	},
{	"greater",	'>'	},
{	"slash",	'/'	},
{	"question",	'?'	},
{	"space",	' '	},
{	"bar",		'|'	},
{	"backslash",	'\\'	},
{	"underscore",	'_'	},
{	"grave",	'`'	},

{	"capslock",	MGL_SKM_CAPS	},
{	"shift",	MGL_SKM_SHIFT	},
{	"alt",		MGL_SKM_ALT	},
{	"ctrl",		MGL_SKM_CTRL	},
{	"menu",		MGL_SKM_MENU	},
{	"opt",		MGL_SKM_OPT	},
{	"up",		MK_UP	},
{	"down",		MK_DOWN	},
{	"left",		MK_LEFT	},
{	"right",	MK_RIGHT },
{	"pageup",	MK_PAGE_UP },
{	"pagedown",	MK_PAGE_DOWN },
{	"insert",	MK_INS },
{	"delete",	0x7f },
{	"delete_alt",	MK_DEL },
{	"home",		MK_HOME	},
{	"end",		MK_END	},
{	"F1",		MK_F1	},
{	"F2",		MK_F2	},
{	"F3",		MK_F3	},
{	"F4",		MK_F4	},
{	"F5",		MK_F5	},
{	"F6",		MK_F6	},
{	"F7",		MK_F7	},
{	"F8",		MK_F8	},
{	"F9",		MK_F9	},
{	"F10",		MK_F10	},
{	"F11",		MK_F11	},
{	"F12",		MK_F12	},
{	"F13",		MKE_F13	},
{	"F14",		MKE_F14	},
{	"F15",		MKE_F15	},
{	"F16",		MKE_F16	},
{	"F17",		MKE_F17	},
{	"F18",		MKE_F18	},
{	"F19",		MKE_F19	},
{	"F20",		MKE_F20	},
{	"F21",		MKE_F21	},
{	"F22",		MKE_F22	},
{	"F23",		MKE_F23	},
{	"F24",		MKE_F24	},
{	"KP_0",		MKE_0	},
{	"KP_1",		MKE_1	},
{	"KP_2",		MKE_2	},
{	"KP_3",		MKE_3	},
{	"KP_4",		MKE_4	},
{	"KP_5",		MKE_5	},
{	"KP_6",		MKE_6	},
{	"KP_7",		MKE_7	},
{	"KP_8",		MKE_8	},
{	"KP_9",		MKE_9	},
{	"KP_plus",	MKE_PLUS	},
{	"KP_minus",	MKE_MINUS	},
{	"KP_mult",	MKE_MULT	},
{	"KP_div",	MKE_DIV		},
{	"break",	MKE_BREAK	},
{	"pause",	MKE_PAUSE	},
{	"print",	MKE_PRINT	},
{	"hanzen",	MKE_HANZEN	},
{	"kanji",	MKE_KANJI	},
{	"henkan",	MKE_HENKAN	},
{	"muhenkan",	MKE_MUHENKAN	},
{	"numlock",	MK_NUMLOCK	},
{	"scrlock",	MKE_SCRLOCK	},
{	"select",	MKE_SELECT	},
{	"KP_delete",	MKE_DEL		},
{	"KP_end",	MKE_END		},
{	"r_alt",	MGL_SKM_RALT},
{	"r_ctrl",	MGL_SKM_RCTRL},
{	"r_shift",	MGL_SKM_RSHIFT},
{	"r_menu",	MGL_SKM_RMENU	},
{	"KP_up",	MKE_UP	},
{	"KP_down",	MKE_DOWN	},
{	"KP_left",	MKE_LEFT	},
{	"KP_right",	MKE_RIGHT	},
{	"KP_pageup",	MKE_PAGE_UP	},
{	"KP_pagedown",	MKE_PAGE_DOWN	},
{	"KP_begin",	MKE_BEGIN	},
{	"KP_return",	MKE_RETURN	},
{	"KP_execute",	MKE_EXECUTE	},
{	"KP_home",	MKE_HOME	},
{	"KP_insert",	MKE_INS	},

{	"backlight_toggle",MKE_BACKLIGHT_TOGGLE},
{	"brightness_up",MKE_BRIGHTNESS_UP},
{	"brightness_down",MKE_BRIGHTNESS_DOWN},
{	"contrast_up",	MKE_CONTRAST_UP},
{	"contrast_down",MKE_CONTRAST_DOWN},
{	"power_off",	MKE_POWER_OFF},
{	"dial",		MKE_DIAL},
{	"play",		MKE_PLAY},
{	"stop",		MKE_STOP},
{	"record",	MKE_RECORD},
{	"switch_window",MKE_SWITCH_WINDOW},
{	"switch_focus",MKE_SWITCH_FOCUS},
{	"Console_1",	MKE_CONSOLE_1 },
{	"Console_2",	MKE_CONSOLE_2 },
{	"Console_3",	MKE_CONSOLE_3 },
{	"Console_4",	MKE_CONSOLE_4 },
{	"Console_5",	MKE_CONSOLE_5 },
{	"Console_6",	MKE_CONSOLE_6 },
{	"Console_7",	MKE_CONSOLE_7 },
{	"Console_8",	MKE_CONSOLE_8 },
{	"Console_9",	MKE_CONSOLE_9 },
{	"none",		MK_NONE },
};

int sym_trans_tab[MKE_SIZE];

int local_translate;

static const int mgl_local_action_map[26] = {
	MK_NONE,		/* a */
	MKE_BRIGHTNESS_UP,	/* b */
	MKE_CONTRAST_UP,	/* c */
	MK_NONE,		/* d */
	MKE_CONSOLE_3,		/* e */
	MK_NONE,		/* f */
	MK_NONE,		/* g */
	MK_NONE,		/* h */
	MKE_CONSOLE_8,		/* i */
	MK_NONE,		/* j */
	MK_NONE,		/* k */
	MK_NONE,		/* l */
	MKE_BACKLIGHT_TOGGLE,	/* m */
	MKE_BRIGHTNESS_DOWN,	/* n */
	MKE_CONSOLE_9,		/* o */
	MK_NONE,		/* p */
	MKE_CONSOLE_1,		/* q */
	MKE_CONSOLE_4,		/* r */
	MK_NONE,		/* s */
	MKE_CONSOLE_5,		/* t */
	MKE_CONSOLE_7,		/* u */
	MKE_CONTRAST_DOWN,	/* v */
	MKE_CONSOLE_2,		/* w */
	MK_NONE,		/* x */
	MKE_CONSOLE_6,		/* y */
	MK_NONE,		/* z */
};

sym_trans_tab_init() {
	int i;
	for (i=0; i< MKE_SIZE; i++) {
		sym_trans_tab[i] = MK_NONE;
	}
	sym_trans_tab[MKE_DEL - MKE_BASE] = MK_DEL;
	sym_trans_tab[MKE_INS - MKE_BASE] = MK_INS;
	sym_trans_tab[MKE_END - MKE_BASE] = MK_END;
	sym_trans_tab[MKE_HOME - MKE_BASE] = MK_HOME;
	sym_trans_tab[MKE_UP - MKE_BASE] = MK_UP;
	sym_trans_tab[MKE_DOWN - MKE_BASE] = MK_DOWN;
	sym_trans_tab[MKE_LEFT - MKE_BASE] = MK_LEFT;
	sym_trans_tab[MKE_RIGHT - MKE_BASE] = MK_RIGHT;
	sym_trans_tab[MKE_PAGE_UP - MKE_BASE] = MK_PAGE_UP;
	sym_trans_tab[MKE_PAGE_DOWN - MKE_BASE] = MK_PAGE_DOWN;

	sym_trans_tab[MKE_HANZEN - MKE_BASE] = '\e';
	sym_trans_tab[MKE_0 - MKE_BASE] = '0';
	sym_trans_tab[MKE_1 - MKE_BASE] = '1';
	sym_trans_tab[MKE_2 - MKE_BASE] = '2';
	sym_trans_tab[MKE_3 - MKE_BASE] = '3';
	sym_trans_tab[MKE_4 - MKE_BASE] = '4';
	sym_trans_tab[MKE_5 - MKE_BASE] = '5';
	sym_trans_tab[MKE_6 - MKE_BASE] = '6';
	sym_trans_tab[MKE_7 - MKE_BASE] = '7';
	sym_trans_tab[MKE_8 - MKE_BASE] = '8';
	sym_trans_tab[MKE_9 - MKE_BASE] = '9';
	sym_trans_tab[MKE_MULT - MKE_BASE] = '*';
	sym_trans_tab[MKE_DIV - MKE_BASE] = '/';
	sym_trans_tab[MKE_PLUS - MKE_BASE] = '+';
	sym_trans_tab[MKE_MINUS - MKE_BASE] = '-';
}

int mgl_get_keysym(char *name) {
	int i,c;
	for (i=0; i< sizeof(sv_tab)/sizeof(sv_tab[0]); i++) {
		if (!strcmp(sv_tab[i].sym,name)) {
		     c = sv_tab[i].val;
		     if ((c >= MKE_BASE) && (c < MKE_BASE + MKE_SIZE))
			  return sym_trans_tab[c - MKE_BASE];
		     return c;
		}
	}
}

int mgl_set_keysym(char *name, int code) {
	int i,c;
	for (i=0; i< sizeof(sv_tab)/sizeof(sv_tab[0]); i++) {
		if (!strcmp(sv_tab[i].sym,name)) {
		    c = sv_tab[i].val;
		    if ((c >= MKE_BASE) && (c < MKE_BASE + MKE_SIZE)) {
			sym_trans_tab[sv_tab[i].val - MKE_BASE] = code;
			return 0;
		    }
		    return -1;
		}
	}
	return -1;
}

static char *find_val(int val) {
	int i;
	static char buf[2];
	for (i=0; i< sizeof(sv_tab)/sizeof(sv_tab[0]); i++) {
		if (sv_tab[i].val == val)
			return sv_tab[i].sym;
	}
	if ( (('a' <= val) && ('z' >= val ))
		    || (('A' <= val) && ('Z' >= val ))
		    || (('0' <= val) && ('9' >= val ))
		    || ('_' == val)  ) {
			buf[0] = val;
			buf[1] = 0;
			return buf;
	}
	return 0;
}

static int find_sym(char *name) {
	int i;
	for (i=0; i< sizeof(sv_tab)/sizeof(sv_tab[0]); i++) {
		if (!strcmp(sv_tab[i].sym,name)) 
			return sv_tab[i].val;
	}
	if (strlen(name) == 1) {
		if ( (('a' <= *name) && ('z' >= *name ))
		    || (('A' <= *name) && ('Z' >= *name ))
		    || (('0' <= *name) && ('9' >= *name ))
		    || ('_' == *name)  )
			return *name;
	}
		
	return -1;
}

void mgl_get_keymap(int code, char *sym1, char *sym2, char *sym3) {
	char *p;
	strcpy(sym1,"");
	strcpy(sym2,"");
	strcpy(sym3,"");
	if ((code < 0) || (code >= MGL_KEYMAP_SIZE)) return;

	if (p = find_val(mgl_keymap[code][0])) {
		strcpy(sym1,p);
	}
	if (p = find_val(mgl_keymap[code][1])) {
		strcpy(sym2,p);
	}
	if (p = find_val(mgl_keymap[code][2])) {
		strcpy(sym3,p);
	}
}

int mgl_set_keymap(int key,char *sym1,char *sym2,char *sym3) {
	int v1=MK_NONE,v2=MK_NONE,v3=MK_NONE;

	if ((key < 0) || (key >= MGL_KEYMAP_SIZE)) return -1;
	v1 = find_sym(sym1);

	if (sym2) v2 = find_sym(sym2);
	else v2 = v1;

	if (sym3) v3 = find_sym(sym3);

	if ((v1 < 0) || (v2 < 0) || (v3 < 0)) return -1;
	
	mgl_keymap[key][0] = v1;
	mgl_keymap[key][1] = v2;
	mgl_keymap[key][2] = v3;
	return 0;
}


void mk_init() {
	int i;
	for (i=0; i<MGL_KEYMAP_SIZE; i++) {
		mgl_keymap[i][0] = -1;
		mgl_keymap[i][1] = -1;
		mgl_keymap[i][2] = -1;
	}
	for (i=0; i< sizeof(ec_base)/sizeof(ec_base[0]);i++) {
		mgl_keymap[ec_base[i].kc][0] = ec_base[i].tc[0];
		mgl_keymap[ec_base[i].kc][1] = ec_base[i].tc[1];
	}
	for (i=0; i< sizeof(ec_base)/sizeof(ec_ext[0]);i++) {
		mgl_keymap[ec_ext[i].kc][0] = ec_ext[i].tc[0];
		mgl_keymap[ec_ext[i].kc][1] = ec_ext[i].tc[1];
	}
	sym_trans_tab_init();
	mgl_key_ok 	= MGL_KEY_ENTER;
	mgl_key_cancel 	= MGL_KEY_ESC;
	mgl_key_up 	= MGL_KEY_UP;
	mgl_key_down 	= MGL_KEY_DOWN;
	mgl_key_left 	= MGL_KEY_LEFT;
	mgl_key_right 	= MGL_KEY_RIGHT;
	mgl_key_A 	= MGL_KEY_ALT;
	mgl_key_B 	= MGL_KEY_SPACE;
}

static int mk_set_type(char *sym) {
	int i;
	if (!strcmp("jp",sym)) {
	    for (i=0; i< sizeof(ec_jp)/sizeof(ec_jp[0]);i++) {
		mgl_keymap[ec_jp[i].kc][0] = ec_jp[i].tc[0];
		mgl_keymap[ec_jp[i].kc][1] = ec_jp[i].tc[1];
		mgl_keymap[ec_jp[i].kc][2] = MK_NONE;
	    }
	    return 0;
	}
	if (!strcmp("us",sym)) {
	    for (i=0; i< sizeof(ec_us)/sizeof(ec_us[0]);i++) {
		mgl_keymap[ec_us[i].kc][0] = ec_us[i].tc[0];
		mgl_keymap[ec_us[i].kc][1] = ec_us[i].tc[1];
		mgl_keymap[ec_us[i].kc][2] = MK_NONE;
	    }
	    return 0;
	}
	return -1;
}

static int mk_swap_key(int key_a,int key_b) {
	int tmp;

	if ((key_a < 0) || (key_a >= MGL_NR_KEYS)) return -1;
	if ((key_b < 0) || (key_b >= MGL_NR_KEYS)) return -1;

	tmp = mgl_keymap[key_a][0];
	mgl_keymap[key_a][0] = mgl_keymap[key_b][0];
	mgl_keymap[key_b][0] = tmp;
	tmp = mgl_keymap[key_a][1];
	mgl_keymap[key_a][1] = mgl_keymap[key_b][1];
	mgl_keymap[key_b][1] = tmp;
	tmp = mgl_keymap[key_a][2];
	mgl_keymap[key_a][2] = mgl_keymap[key_b][2];
	mgl_keymap[key_b][2] = tmp;
	return 0;
}


static int mk_set_alias(char *sym,int key) {
	if (key < 0 || key >= MGL_NR_KEYS) return -1;

	if (!strcmp("ok",sym)) mgl_key_ok = key;
	else if (!strcmp("cancel",sym)) mgl_key_cancel = key;
	else if (!strcmp("up",sym)) mgl_key_up = key;
	else if (!strcmp("down",sym)) mgl_key_down = key;
	else if (!strcmp("right",sym)) mgl_key_right = key;
	else if (!strcmp("left",sym)) mgl_key_left = key;
	else if (!strcmp("A",sym)) mgl_key_A = key;
	else if (!strcmp("B",sym)) mgl_key_B = key;
	else return -1;
	return 0;
}

int mk_load_keymap(FILE *fp) {
	char buf[256];
	char sym1[32];
	char sym2[32];
	char sym3[32];
	char *p;
	int n,c,c2;
	int line=0;

	mk_init();
	while (fgets(buf,256,fp)) {
		line++;
		p = buf;
		while ((*p == ' ') || (*p == '\t')) p++;
		if ((*p == '#') || (*p == '\n') || (*p == 0)) continue;

		if (!strncmp("localtranslate",p,strlen("localtranslate"))) {
				local_translate = 1;
		} else if (sscanf(p,"swap %d %d",&c,&c2) == 2) {
			if (mk_swap_key(c,c2)) {
				fprintf(stderr,"keyswap error at %d\n",line);
				return -1;
			}
		}
		else if (sscanf(p,"keyboardtype = %30s",&sym1) == 1) {
			if (mk_set_type(sym1)) {
				fprintf(stderr,"keyboardtype error at %d\n",line);
				return -1;
			}
			
		}
		else if ((n = sscanf(p,"keycode %d = %30s %30s %30s"
			,&c,sym1,sym2,sym3))==2 || (n == 3) || (n == 4)) {
			if (n == 2) strcpy(sym2,sym1);
			if (n == 3) strcpy(sym3,"none");
			if (mgl_set_keymap(c,sym1,sym2,sym3)) {
				fprintf(stderr,"keycode  error at %d\n",line);
				return -1;
			}
		}
		else if (sscanf(p,"alias %30s = %d",sym1,&c)==2) {
			if (mk_set_alias(sym1,c)) {
				fprintf(stderr,"alias error at %d\n",line);
				return -1;
			}
		} else {
				fprintf(stderr,"syntax error at %d\n",line);
				return -1;
		}
	}
	return 0;
}

int mk_init_keymap(char *opt, char *keymap_postfix) {
	char buf[5][64];
	int i,n = 0;
	int ok = 0;

	mk_init();
	if (opt) 
		n = sscanf(opt,"%60s %60s %60s %60s %60s"
			,buf[0],buf[1],buf[2],buf[3],buf[4]);
	if (n == 0) {
		mk_set_type("jp");
		ok = 1;
	} else {
		for (i=0; i<n; i++) {
			if (!strcmp("jp",buf[i])) {
				mk_set_type("jp");
				ok = 1;
			}
			if (!strcmp("us",buf[i])) {
				mk_set_type("us");
				ok = 1;
			}
			if (!strcmp("localtranslate",buf[i])) {
				local_translate = 1;
			}
			if (!strcmp("swapctrlcaps",buf[i])) {
				mk_swap_key(MGL_KEY_CTRL,MGL_KEY_CAPS);
			}
			if (!strcmp("swapdelbs",buf[i])) {
				mk_swap_key(MGL_KEY_DEL,MGL_KEY_BACKSPACE);
			}
			if (!strcmp("loadkeymap",buf[i])) {
				char fname[256];
				FILE *fp = NULL;
				int r = -1;

				if (keymap_postfix && keymap_postfix[0]) {
					strcpy(fname,MGLDIR);
					strcat(fname,"/keymap");
					strcat(fname,".");
					strcat(fname,keymap_postfix);
					if (fp = fopen(fname,"r")) {
						r = mk_load_keymap(fp);
						fclose(fp);
					}
				}
				if (r) {
					strcpy(fname,MGLDIR);
					strcat(fname,"/keymap");
					if (fp = fopen(fname,"r")) {
						r = mk_load_keymap(fp);
						fclose(fp);
					}
				}
				if (r) ok = 1;
			}
		}
	}
	return (ok)?0:-1;
}

static int search_rawcode_sub(int sym1,int sym2,const struct mgl_encoding *e,int n,int mode) {
	int i;
	if (mode == 0) {
	    for (i=0; i<n; i++) {
		if ((sym1 == e[i].tc[0]) && (sym2 == e[i].tc[1])) {
			return e[i].kc;
		}
	    }
	} else {
	    for (i=0; i<n; i++) {
		if ((sym1 == e[i].tc[0])) {
			return e[i].kc;
		}
	    }
	}
	return 0;
}

static int search_rawcode(int sym1,int sym2,int jp, int *found) {
	int c;
	int mode;

	if ((sym1 == MK_NONE) && (sym2 == MK_NONE)) return 0;

	for (mode=0; mode<2; mode++) {
	    c = search_rawcode_sub(sym1,sym2, ec_base
		, sizeof(ec_base)/sizeof(ec_base[0]),mode);
	    if (c && !found[c]) return c;
	    if (jp) {
		c = search_rawcode_sub(sym1,sym2, ec_jp
			, sizeof(ec_jp)/sizeof(ec_jp[0]),mode);
		if (c && !found[c]) return c;
	    } else {
		c = search_rawcode_sub(sym1,sym2, ec_us
			, sizeof(ec_us)/sizeof(ec_us[0]),mode);
		if (c && !found[c]) return c;
	    }
	    c = search_rawcode_sub(sym1,sym2, ec_ext
		, sizeof(ec_ext)/sizeof(ec_ext[0]),mode);
	    if (c && !found[c]) return c;
	}
	return 0;
}

// #define swap(a,b) { int tmp; tmp = a; a = b; b = tmp; }

#ifndef MGL_RAW_KEYMAP 

void mk_create_raw_keymap() {
	int i,j,n;
	int tmp;
	int jp=1;
	int c;
	int sym1,sym2,sym3;
	int raw[MGL_NR_KEYS];
	int found[MGL_NR_KEYS];
	int keymap[MGL_NR_KEYS][3];

	for (i=0; i< MGL_NR_KEYS; i++) {
		raw[i] = 0;
		found[i] = 0;
		keymap[i][0] = MK_NONE;
		keymap[i][1] = MK_NONE;
		keymap[i][2] = MK_NONE;
	}
	for (i=0; i< MGL_NR_KEYS; i++) {
		if((mgl_keymap[i][0] == '2') && (mgl_keymap[i][1] == '@')) {
			jp = 0;
		}
	}
	/* try to assign MGL keycode */
	for (i=0; i< MGL_NR_KEYS; i++) {
		sym1 = mgl_keymap[i][0];
		sym2 = mgl_keymap[i][1];
		sym3 = mgl_keymap[i][2];
		c = search_rawcode(sym1,sym2,jp,found);
		if (c) {
			raw[i] = c;
			found[c] = i;
			keymap[c][0] = sym1;
			keymap[c][1] = sym2;
			keymap[c][2] = sym3;
		}
	}
	/* try to assign the same keycode (except 0) */
	for (i=1; i< MGL_NR_KEYS; i++) {
		if ((mgl_keymap[i][0] != MK_NONE) && (!raw[i])) {
			if (!found[i]) {
				raw[i] = i;
				found[i] = i;
				keymap[i][0] = mgl_keymap[i][0];
				keymap[i][1] = mgl_keymap[i][1];
				keymap[i][2] = mgl_keymap[i][2];
			}
		}
	}
	/* try to assign the other keycode */
	for (i=0; i< MGL_NR_KEYS; i++) {
		if ((mgl_keymap[i][0] != MK_NONE) && (!raw[i])) {
			for (j=1; j<MGL_NR_KEYS; j++) {
				if (!found[j]) {
					raw[i] = j;
					found[j] = i;
					keymap[j][0] = mgl_keymap[i][0];
					keymap[j][1] = mgl_keymap[i][1];
					keymap[j][2] = mgl_keymap[i][2];
					break;
				}
			}
		}
	}
	for (i=0; i< MGL_NR_KEYS; i++) {
		mgl_raw_keymap[i] = raw[i];
		mgl_keymap[i][0] = keymap[i][0];
		mgl_keymap[i][1] = keymap[i][1];
		mgl_keymap[i][2] = keymap[i][2];
	}
}
#else /* defined MGL_RAW_KEYMAP */

#if MGL_RAW_KEYMAP == PSION5
#include "Linux/rawkeymap_psion5.c"
#endif

#endif /* MGL_RAW_KEYMAP */

int  (*mgl_key_proc_action)(int);
void (*mgl_mouse_proc)(int,int,int);

int mgl_key_conv_raw(int c) {
	int v,vv;
	int old_c = c;

	mgl_keymap_shift = (mgl_modifier_status
		 & (MGL_SKM_SHIFT|MGL_SKM_RSHIFT))?1:0;

	c  = mgl_raw_keymap[c & 0x7f] | (c & 0x80) ;

	// fprintf(stderr, "conv_raw_native() 0x%02x(%d) -> 0x%02x\r\n", old_c, old_c, c);

	if ((mgl_modifier_status &  (MGL_SKM_ALT|MGL_SKM_RALT)) 
		&& (mgl_keymap[c & 0x7f][2] != MK_NONE))
			mgl_keymap_shift = 2;

	if (c & 0x80) { /* release */
		v = mgl_keymap[c & 0x7f][mgl_keymap_shift];
		if ((v >= 0) && (v & MGL_SKM_MASK)) {
			mgl_modifier_status &= ~v;
		}
		if (mgl_modifier_status & (MGL_SKM_MENU|MGL_SKM_RMENU)) {
			v = mgl_keymap[c & 0x7f][0];
			switch (v) {
			case ' ':
				if (mgl_mouse_proc) mgl_mouse_proc(2,0,0);
				return 0;
			}
		}
	} else { /* press */
		v = mgl_keymap[c & 0x7f][mgl_keymap_shift];
		if ((v >= 0) && (v & MGL_SKM_MASK)) {
			mgl_modifier_status |= v;
		}
#if 0 /* for test only */
		if (v == MGL_SKM_CTRL) {
			if (mgl_modifier_status & MGL_SKM_SHIFT) {
				mgl_modifier_status ^= MGL_SKS_CAPS_LOCKED;
			}
		}
#endif
		if (v == MGL_SKM_CAPS) {
			mgl_modifier_status ^= MGL_SKS_CAPS_LOCKED;
		}
		if (v == MK_NUMLOCK) {
			mgl_modifier_status ^= MGL_SKS_NUM_LOCKED;
		}
		else if ((v >= MKE_BASE) && (v < MKE_BASE + MKE_SIZE)) {
			if (mgl_key_proc_action) mgl_key_proc_action(v);
			vv = sym_trans_tab[v - MKE_BASE];
			if (((v==MKE_SWITCH_WINDOW)||(v == MKE_SWITCH_FOCUS))
			      && (vv != MK_NONE)) {
				put_key(vv);
			}
		}
		if (mgl_modifier_status & (MGL_SKM_MENU|MGL_SKM_RMENU)) {
			int dx,dy;
			v = mgl_keymap[c & 0x7f][0];
			if (('a' <= v) && (v <= 'z')) {
				int act = mgl_local_action_map[v - 'a'];
				if ((act != MK_NONE) && (mgl_key_proc_action)){
					mgl_key_proc_action(act);
					return 0;
				}
			} else
			switch (v) {
			case '1':
				mgl_button_shift = 0;
				return 0;
			case '2':
				mgl_button_shift = 1;
				return 0;
			case '3':
				mgl_button_shift = 2;
				return 0;
			case '4':
				if (mgl_mouse_proc) mgl_mouse_proc(3,0,0);
				return 0;
			case ' ':
				if (mgl_mouse_proc) mgl_mouse_proc(0,0,0);
				return 0;
			case MK_UP:
				dx = 0; dy = -8;
				if (mgl_modifier_status
					 & (MGL_SKM_SHIFT|MGL_SKM_RSHIFT)) {
					dy = -1;
				}
				if (mgl_mouse_proc) mgl_mouse_proc(1,dx,dy);
				return 0;
			case MK_DOWN:
				dx = 0; dy = 8;
				if (mgl_modifier_status
					 & (MGL_SKM_SHIFT|MGL_SKM_RSHIFT)) {
					dy = 1;
				}
				if (mgl_mouse_proc) mgl_mouse_proc(1,dx,dy);
				return 0;
			case MK_LEFT:
				dx = -8; dy = 0;
				if (mgl_modifier_status
					 & (MGL_SKM_SHIFT|MGL_SKM_RSHIFT)) {
					dx = -1;
				}
				if (mgl_mouse_proc) mgl_mouse_proc(1,dx,dy);
				return 0;
			case MK_RIGHT:
				dx = 8; dy = 0;
				if (mgl_modifier_status
					 & (MGL_SKM_SHIFT|MGL_SKM_RSHIFT)) {
					dx = 1;
				}
				if (mgl_mouse_proc) mgl_mouse_proc(1,dx,dy);
				return 0;
				
			}
		}
	}
	return c;
}

int mgl_key_trans(int c,int release) {
	int v;

	if ((c <= 0) || (c >= MGL_KEYMAP_SIZE)) return (-1);

	if (release) {
		v = mgl_keymap[c][mgl_keymap_shift];
		if ((v >= 0 ) && (v & MGL_SKM_MASK)) {
			return (mgl_modifier_status & MGL_SKM_MASK)
				 | MGL_SKM_NOTICE;
		}
		return (-1);
	} else { /* press */
		v = mgl_keymap[c][mgl_keymap_shift];
		if ((v >= MKE_BASE) && (v < MKE_BASE + MKE_SIZE)) {
			if ((v == MKE_SWITCH_WINDOW) || 
			   (v == MKE_SWITCH_FOCUS)) {
				/* already sended by key_conv_raw() */
				return (-1);
			}
			v = sym_trans_tab[v - MKE_BASE];
		}
		if (v < 0) return (-1);
		if (v & MGL_SKM_MASK) {
			return (mgl_modifier_status & MGL_SKM_MASK)
				 | MGL_SKM_NOTICE;
		}
		if (mgl_modifier_status & (MGL_SKM_CTRL|MGL_SKM_RCTRL)) {
			if (v < 0x7f)
				return ((v & 0x1f) | 
					(mgl_modifier_status & MGL_SKM_MASK));
		}
		if (mgl_modifier_status & MGL_SKS_CAPS_LOCKED) {
			if (('a' <= v ) && (v <= 'z')) {
				v = v + 'A' - 'a'; 
			} else if (('A' <= v ) && (v <= 'Z')) {
				v = v + 'a' - 'A'; 
			}
		}
		return (v | (mgl_modifier_status & MGL_SKM_MASK));
	}
}

int mgl_put_key_raw(int c,int release) {
	int v;

	if (mgl_key_mode & MGL_SK_RAW) {
		if ((c > 0) && (c < MGL_NR_KEYS)) {
			put_key( (c & 0x7f) | (release?0x80:0) );
			return 0;
		}
		return -1;
	}
	if ((c <= 0) || (c >= MGL_KEYMAP_SIZE)) {
		return -1;
	}
	mgl_keymap_shift = (mgl_modifier_status
		 & (MGL_SKM_SHIFT|MGL_SKM_RSHIFT))?1:0;

	if ((mgl_modifier_status &  (MGL_SKM_ALT|MGL_SKM_RALT)) 
		&& (mgl_keymap[c][2] != MK_NONE))
			mgl_keymap_shift = 2;

	if (release) { /* release */
		v = mgl_keymap[c][mgl_keymap_shift];
		if ((v >= 0) && (v & MGL_SKM_MASK)) {
			mgl_modifier_status &= ~v;
		}
	} else { /* press */
		v = mgl_keymap[c][mgl_keymap_shift];
		if ((v >= 0) && (v & MGL_SKM_MASK)) {
			mgl_modifier_status |= v;
		}
		if (v == MGL_SKM_CAPS) {
			mgl_modifier_status ^= MGL_SKS_CAPS_LOCKED;
		}
		if (v == MK_NUMLOCK) {
			mgl_modifier_status ^= MGL_SKS_NUM_LOCKED;
		}
	}
	c = mgl_key_trans(c,release);
	if (c >= 0) {
		put_key(c);
		return 0;
	}
	return -1;
}

mk_dump_keymap(char *s) {
	char sym1[32],sym2[32],sym3[32];
	char *p;
	int i;
	for (i=0; i<MGL_NR_KEYS; i++) {
		mgl_get_keymap(i, sym1, sym2, sym3);
		if (!strcmp(sym1,"none") && !strcmp(sym2,"none")) 
			continue;
		printf("%s %3d = %-16s %-16s %-16s\n",s,i,sym1,sym2,sym3);
	}
	fflush(stdout);
}

#if 0
main() {
	mk_load_keymap(stdin);
	//mk_dump_keymap();
}
#endif
