/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/mman.h>
#include <sys/time.h>

#include "config.h"
#include "mgl2.h"
#include "event_man.h"

#ifdef USE_LOCAL_MEMSET
#define memset	mgl_memset
#endif
#ifdef USE_LOCAL_MEMMOVE
#define memmove	mgl_memmove
#endif

extern struct screen *physical_screen;
extern int (*im_readp)(char *ibuf, int ibufsize);

int im_mode;
struct font_engine *mgl_fe;

int delayed_key;
long long dk_limit;

struct key_buf key_buf[100];
int key_buf_cnt=0;

void put_key(int c) {
      if (delayed_key) {
          if(key_buf_cnt < 100) {
		key_buf[key_buf_cnt].code = delayed_key;
		key_buf[key_buf_cnt].x = -1;
		key_buf[key_buf_cnt].y = -1;
		key_buf_cnt++;
          }
	  delayed_key = 0;
      }
      if ((c >= 0)  && (key_buf_cnt < 100)) {
		key_buf[key_buf_cnt].code = c;
		key_buf[key_buf_cnt].x = -1;
		key_buf[key_buf_cnt].y = -1;
		key_buf_cnt++;
      }
}

void put_key_delay(int c) {
	if (delayed_key) {
		put_key(c);
	} else {
		delayed_key = c;
		dk_limit = millitime() + 30;
	}
}

void put_key_xy(struct virtual_key *v,int code,int x,int y) {
	if (delayed_key) {
          if(key_buf_cnt < 100) {
		key_buf[key_buf_cnt].code = delayed_key;
		key_buf[key_buf_cnt].x = -1;
		key_buf[key_buf_cnt].y = -1;
		key_buf_cnt++;
          }
	  delayed_key = 0;
	}
	while(v) {
		x -= v->x;
		y -= v->y;
		v = v->parent;
	}
	key_buf[key_buf_cnt].code = code;
	key_buf[key_buf_cnt].x = x;
	key_buf[key_buf_cnt].y = y;
	key_buf_cnt++;
}

int get_key_nodelay() {
	int c;
	if (!key_buf_cnt) return -1;

	c = key_buf[0].code;
	if (key_buf[0].x >= 0) {
		vk_x = key_buf[0].x;
		vk_y = key_buf[0].y;
	}
	memmove(key_buf,key_buf+1,(key_buf_cnt-1)*sizeof(key_buf[0]));
	key_buf_cnt--;
	return c;
}

void unget_key(int c) {
	if (key_buf_cnt < 100) {
	    memmove(key_buf+1,key_buf,(key_buf_cnt)*sizeof(key_buf[0]));
	    key_buf[0].code = c;
	    key_buf[0].x = -1;
	    key_buf[0].y = -1;
	    key_buf_cnt++;
	}
}

struct virtual_key *vk_root,*vk_free;
struct virtual_key *vk_cur;
int vk_pressed;
int vk_x,vk_y;
struct virtual_key *vkp;
extern void vk_init();
extern void vk_reverse();
extern int vk_search();
extern int vk_match(struct virtual_key *p,int x,int y);

struct virtual_key *
create_virtual_key(x,y,xs,ys,keycode) {
	return create_virtual_key3(x,y,xs,ys,MK_V0,MK_V0,keycode);
}

struct virtual_key *
create_virtual_key3(x,y,xs,ys,down,move,up) {
	struct virtual_key *vk;

	if (vk_free) {
		vk = vk_free;
		vk_free = vk->link;
	} else {
		vk = (struct virtual_key *)malloc(sizeof(struct virtual_key));
	}
	if (!vk) return vk;
	vk->up = up;
	vk->down = down;
	vk->move = move;
	vk->x = x;
	vk->y = y;
	vk->xs = xs;
	vk->ys = ys;
	vk->link = vk->parent = vk->child = NULL;
	vk->callback = NULL;
	return vk;
}

void vk_attach(parent,vk) struct virtual_key *parent,*vk; {
	if (!parent) parent = vk_root;
	vk->parent = parent;
	vk->link = parent->child;
 	parent->child = vk;
}

void vk_detach(vk,clean_mode) struct virtual_key *vk; {
	struct virtual_key *p;

	if (vk == vk_cur) {
		vk_cur = NULL;
		vk_pressed = 0;
	}
	p = vk->parent;
	if (p) {
		if (p->child ==vk) {
			p->child = vk->link;
			vk->parent = NULL;
			vk->link = NULL;
		} else {
			p = p->child;
			while (p->link) {
			   if (p->link == vk) {
				p->link = vk->link;
				vk->parent = NULL;
				vk->link = NULL;
				break;
			    }
			    p = p->link;
			}
		}
	}
	if (clean_mode) {
		while ((struct virtual_key volatile *)(vk->child)) {
			vk_detach(vk->child,clean_mode);
		}
		vk->link = vk_free;
		vk_free = vk;
	}
}

int vk_search(p,x,y) struct virtual_key *p; {
	struct virtual_key *v;
	v = p->child;
	while (v) {
		if (vk_search(v,x - p->x,y - p->y)) return 1;
		v = v->link;
	}
	if ((p->down == MK_V0) && (p->up == MK_V0) && (p->move == MK_V0)) return 0;
	if ( (p->x <= x)&&(x <p->x+p->xs)&&(p->y <= y)&&(y < p->y+p->ys)) {
		vkp = p;
		return 1;
	}
	return 0;
}

int vk_match(p,x,y) struct virtual_key *p; {
	struct virtual_key *pp;
	struct virtual_key *v;

	pp = p->parent;
	while (pp) {
		x -= pp->x;
		y -= pp->y;
		pp = pp->parent;
	}
	v = p->child;
	while (v) {
		if (vk_search(v,x - p->x,y - p->y)) return 0;
		v = v->link;
	}
	if ( (p->x <= x)&&(x <p->x+p->xs)&&(p->y <= y)&&(y < p->y+p->ys)) {
		return 1;
	}
	return 0;
}

int vk_rev = 0;

void vk_reverse(p,mode) struct virtual_key *p; {
	struct virtual_key *pp;
	int x,y;
	int sv;

	if (vk_rev && !mode) return;
	if (!vk_rev && mode) return;
	vk_rev = !vk_rev;

	x = p->x;
	y = p->y;
	pp = p->parent;
	while (pp) {
		x += pp->x;
		y += pp->y;
		pp = pp->parent;
	}

	push_screen(physical_screen);
	sv = pen_color;
	pen_color = COLOR_REVERSE;

	fill_rect(x,y,p->xs,p->ys);

	pen_color = sv;
	pop_screen();
refresh();
}

static void vk_put_key_xy(struct virtual_key *vkp,int ev,int x, int y) {
	if (vkp->callback) {
		if ((vkp->callback)(ev,x,y)) {
			put_key_xy(vkp,ev,x,y);
		}
	} else {
		put_key_xy(vkp,ev,x,y);
	}
}


void vk_mouse_down(int x,int y);
void vk_mouse_move(int x,int y) {

	if (!vk_cur){
	    if (mgl_key_mode & MGL_SK_EXMOUSE_EVENT){
		if (vk_search(vk_root,x,y)) {
		    if (vkp->move != MK_V0) {
			vk_cur = vkp;
			vk_put_key_xy(vkp,vkp->move,x,y);
		    }
		}
	    }
	    return;
	}

	if (vk_cur && !vk_pressed) {
		if ( (mgl_key_mode & MGL_SK_EXMOUSE_EVENT) &&
		    (vk_cur->move != MK_V0) && vk_match(vk_cur,x,y)) {
			vk_put_key_xy(vk_cur,vk_cur->move,x,y);
		}
		return;
	}
	if (vk_match(vk_cur,x,y)) {
		if (vk_cur->down == MK_V0) {
			vk_reverse(vk_cur,0);
		} else if (vk_cur->move != MK_V0) {
			vk_put_key_xy(vk_cur,vk_cur->move,x,y);
		}
	} else {
		if (vk_cur->down == MK_V0) {
			vk_reverse(vk_cur,1);
		}
	   	if (vk_search(vk_root,x,y)) {
			vk_mouse_down(x,y);
		}
	}
}

void vk_mouse_up(int x,int y) {
	if (!vk_pressed) {
		vk_mouse_move(x,y);
		return;
	}
	if (!vk_cur) return;
	if (vk_cur->down == MK_V0) {
		vk_reverse(vk_cur,1);
		if (vk_match(vk_cur,x,y)) {
			vk_put_key_xy(vk_cur,vk_cur->up,x,y);
		}
	} else {
		vk_put_key_xy(vk_cur,vk_cur->up,x,y);
	}
	mgl_button_shift = 0; /* XXX */
	vk_pressed = 0;
	return;
}

void vk_mouse_down(int x,int y) {
	if (vk_cur && vk_pressed) { /* same as release */
		if (vk_cur->down == MK_V0) {
		    vk_reverse(vk_cur,1);
		    if (vk_match(vk_cur,x,y)) {
			vk_put_key_xy(vk_cur,vk_cur->up,x,y);
		    }
		} else {
		    vk_put_key_xy(vk_cur,vk_cur->up,x,y);
		}
		vk_pressed = 0;
	}
	if (vk_search(vk_root,x,y)) {
		if ((vkp->up == MK_V0)&&(vkp->down != MK_V0)) {
			vk_put_key_xy(vkp,vkp->down,x,y);
		} else if ((vkp->up != MK_V0)&&(vkp->down != MK_V0)) {
			vk_cur = vkp;
			vk_put_key_xy(vkp,vkp->down,x,y);
		} else if ((vkp->up != MK_V0)&&(vkp->down == MK_V0)) {
			vk_cur = vkp;
			vk_reverse(vk_cur,0);
		}
	}
	vk_pressed = 1;
}

#define SM_SHIFT  0
#define SM_WEIGHT   (6  << SM_SHIFT)
static int sm_x;
static int sm_y;
#define ABS(x)	(((x)<0)?-(x):(x))

void vk_mouse_smooth(int *xp,int *yp) {
	int new_x,new_y;
	int x,y;
	int d;
	x = *xp << SM_SHIFT;
	y = *yp << SM_SHIFT;

	d = ABS(x - sm_x);
	new_x = ((x * d) + (SM_WEIGHT * sm_x) + (d + SM_WEIGHT)/2) / (d + SM_WEIGHT);

	d = ABS(y - sm_y);
	new_y = ((y * d) + (SM_WEIGHT * sm_y) + (d + SM_WEIGHT)/2) / (d + SM_WEIGHT);

	sm_x = new_x;
	sm_y = new_y;
	*xp = new_x >> SM_SHIFT;
	*yp = new_y >> SM_SHIFT;
}

void vk_mouse_smooth_init(int *xp,int *yp) {
	sm_x = *xp << SM_SHIFT;
	sm_y = *yp << SM_SHIFT;
}

void vk_init() {
	vk_root = create_virtual_key(0,0,SCREEN_WIDTH,SCREEN_HEIGHT,MK_V0);
	vk_x = vk_y = 0;
}

int bgmode;
struct screen *bg_screen;
struct screen *fg_screen;

int create_physical_screen(int kind,int xs,int ys,char *bitmap) {
	if (bitmap == NULL) { /* create bg_screen only */
		if (!bg_screen) {
			bg_screen = _create_memscreen[kind](xs,ys,NULL,0);
		}
		if (!physical_screen) {
			physical_screen = _create_memscreen[kind](xs,ys
					,bg_screen->bitmap,0);
			bgmode = 1;
		}
	} else {
		if (!fg_screen) {
			fg_screen = _create_memscreen[kind](xs,ys,bitmap,0);
		}
		if (!physical_screen) {
			physical_screen = _create_memscreen[kind](xs,ys
					,fg_screen->bitmap,0);
			bgmode = 0;
		}
	}
}

int screen_switch(mode) {
	int xs,ys;
	int kind;

	xs = physical_screen->width;
	ys = physical_screen->height;

	if (!bg_screen) {
		kind =  physical_screen->type & ST_KINDMASK;
		bg_screen = _create_memscreen[kind](xs,ys,NULL,0);
	}
	if ((bgmode==0) && mode) {
		bitblt(bg_screen,0,0,fg_screen,0,0,xs,ys,0);
		bgmode = !bgmode;
		physical_screen->bitmap = bg_screen->bitmap;
		physical_screen->wbytes = bg_screen->wbytes;
		physical_screen->plbytes = bg_screen->plbytes;
	} else if ((bgmode==1) && !mode) {
		if (!fg_screen) return bgmode;
		bitblt(fg_screen,0,0,bg_screen,0,0,xs,ys,0);
		physical_screen->bitmap = fg_screen->bitmap;
		physical_screen->wbytes = fg_screen->wbytes;
		physical_screen->plbytes = fg_screen->plbytes;
		bgmode = !bgmode;
	}
	return bgmode;
}

long long millitime(void) {
	struct timeval n;
	gettimeofday(&n,0);
	return (n.tv_sec *1000 + n.tv_usec/1000);
}

void toggle_im_mode(void) {
//printf("im_mode_toggled %d\n",im_mode);
    put_key('O'&0x1f);
    return;
}

int get_im_mode(void) {
    return im_mode;
}

/*
key_buf ζͭ
ݤʤȤ򤷤ƤΤǡ

***** "" ꤹ
im_mode  OFF ˤơ
 
^O Хåեꤳࡣ

im_mode  OFF ξ֤ǡ""Ф塢
^O 򽦤Τǡim_mode  ON ˤʤ롣

                                Хåե(key_buf)
 im_mode ON Ǵ          ʸ
         
        
         
 im_mode OFF                        ꤷ + ^O
         
     ^O 򽦤          
         
 im_mode ON

ɲäν

^O 򽦤Τ˥ǥ쥤򤤤줿

  mgterm ǡɽȿǤޤˡget_key_im Ƥޤ
  ʸåץǡȤʤȤ꤬ä

*/

int get_key_im(int time_out) {
	int save_keymode;
	int sav_color;
	struct fe_context fc;
	int ret,key;
	int i;
	char buf[256];

	if (!im_readp) return get_key(time_out);
	if ((mgl_key_mode & MGL_SK_KM_MASK) == MGL_SK_RAW)
		 return get_key(time_out);
retry:
	while (!im_mode || key_buf_cnt) {
		ret = key = get_key(time_out);
		if ((ret >= 0) &&
		    ((mgl_key_mode & MGL_SK_KM_MASK) == MGL_SK_EXTRANSLATED)) {
			key &= ~MGL_SKM_MASK;
		}
		if (key == ('O'&0x1f)) {
			im_mode = !im_mode;
			break;
		} else {
			return ret;
		}
	}
	fc = *pen_font;
	sav_color = pen_color;
	save_keymode = mgl_key_mode;
	mgl_set_keymode(MGL_SK_TRANSLATED);
	while ((ret = (im_readp)(buf,256))==0)
		;
	mgl_set_keymode(save_keymode);
	if (ret>0) {
		im_mode = 0;
		for (i=0; i< ret; i++) {
		    if (buf[i] != ('O'&0x1f))
		        put_key(buf[i]&0xff);
		}
		if (buf[ret-1] != ('O'&0x1f))
			put_key_delay('O'&0x1f);
	}
	set_color(sav_color);
	*pen_font = fc;
	goto retry;
}

static int ia_rect;
static int ia_minx;
static int ia_maxx;
static int ia_miny;
static int ia_maxy;

static int ii_rect;
static int ii_minx;
static int ii_maxx;
static int ii_miny;
static int ii_maxy;


/* application -> im */
void mgl_im_avoid_point(int x, int y, int clean) {
	if (clean || !ia_rect) {
		ia_minx = x;
		ia_maxx = x;
		ia_miny = y;
		ia_maxy = y;
		ia_rect = 1;
	} else {
		if (x < ia_minx) ia_minx = x;
		if (x > ia_maxx) ia_maxx = x;
		if (y < ia_miny) ia_miny = y;
		if (y > ia_maxy) ia_maxy = y;
	}
	return;
}

/* application -> im */
void mgl_im_impart_point(int x, int y, int clean) {
	if (clean || !ii_rect) {
		ii_minx = x;
		ii_maxx = x;
		ii_miny = y;
		ii_maxy = y;
		ii_rect = 1;
	} else {
		if (x < ii_minx) ii_minx = x;
		if (x > ii_maxx) ii_maxx = x;
		if (y < ii_miny) ii_miny = y;
		if (y > ii_maxy) ii_maxy = y;
	}
	return;
}

/* api for im */
int mgl_im_view_point(int xs,int ys,int *xp,int *yp) {
	int ret = 1; /* 0: forced 1: coperative 2: impared */
	int x,y;
	int d = 0;

	if (mgl_apli_type == AT_MINIAPLI) {
		d = 4;
	}

	/* check imparted */
	if (ii_rect) {
//printf("%d %d - %d-%d xs = %d ys = %d\n",ii_minx,ii_miny,ii_maxx,ii_maxy,
//xs,ys);
	}
	if (ii_rect && (ii_maxx+1-ii_minx >= xs) 
		&& (ii_maxy+1-ii_miny >= ys) ) {
//printf("imparted! \n");
		x = ii_minx;
		y = ii_miny;
		ret = 2;
	}
	else if (!ia_rect) {	/* any */
		x = 0;
		y = SCREEN_HEIGHT - (16+12+8);
		if (y < 0) y = 0;
	} else {
		/* left size */
		if (ia_maxx+1 + xs < SCREEN_WIDTH - d) {
			x = ia_maxx + 1;
			y = ia_miny;
			if (y + ys >= SCREEN_HEIGHT - d)
				y = SCREEN_HEIGHT - d - ys;
		}
		/* down size */
		else if (ia_maxy+1 + ys < SCREEN_HEIGHT - d) {
			x = 0;
			if (ia_minx + xs < SCREEN_WIDTH - d)
				x = ia_minx;
			y = ia_maxy+1;
		}
		/* upside */
		else if ( ia_miny - ys >= d) {
			x = 0;
			if (ia_minx + xs < SCREEN_WIDTH - d)
				x = ia_minx;
			y = ia_miny - ys;
		}
		else {
			x = 0;
			y = SCREEN_HEIGHT - ys;
			ret = 0;
		}
	}
	*xp = x;
	*yp = y;
	return ret;
}
