/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999, 2000, 2001
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *	SATO Kazumi (sato@netbsd.org)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/fcntl.h>
#include <sys/time.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <sys/ioctl.h>

#include <errno.h>
#include <sys/stat.h>

#include "config.h"
#include "mgl2.h"
#include "mglcol.h"
#include "event_man.h"
#include "mglkey.h"
#include "keymap.h"

static int mgl_keyboard_fd = 0;
static int do_mouse_smooth = 0;

static int vt_get_active(int fd);
static int scr_self = -1;
static long long redraw_time = 0LL;

#ifdef USE_LOCAL_MEMSET
#define memset	mgl_memset
#endif
#ifdef USE_LOCAL_MEMMOVE
#define memmove	mgl_memmove
#endif

#include "mgl_md.h"

extern int mgl_screen_type;
extern int mgl_screen_realtype;
extern char *mgl_screen_addr;
extern int mgl_screen_offset;
extern int mgl_screen_realwidth;
extern int mgl_screen_bpp;
extern char mgl_screen_name[32];

static int share_fb = 0;
static int depth = 8;
static int shift_bits = 0;
static int rotated = 0;

static int emcons_key_mode = 0;

static char keymap_postfix[64];
static int mouse_tick = 10;	/* 10 ms */
static int repeat_tick = 100;	/* 100 ms */
static int repeat_tick1 = 300;	/* 300 ms */
static int last_key,last_key_is_pressed;
static long long last_key_time = MGL_SKM_NOTICE;

#ifdef MGL_MACHINE
#if ( MGL_MACHINE == MGL_MACHINE_MAC_ADB )
#include "md_mac.h"
#elif ( MGL_MACHINE == MGL_MACHINE_SVGA )
#include "md_svga.h"
#elif ( MGL_MACHINE == MGL_MACHINE_VGA )
#include "md_vga.h"
#elif ( MGL_MACHINE == MGL_MACHINE_HPCMIPS )
#include "md_hpcmips.h"
#elif ( MGL_MACHINE == MGL_MACHINE_LINUXFB ) || ( MGL_MACHINE == MGL_MACHINE_MG_LINUX )
#include "md_linuxfb.h"
#elif ( MGL_MACHINE == MGL_MACHINE_CYGWIN )
#include "md_cygwin.h"
#elif ( MGL_MACHINE == MGL_MACHINE_NONE )
#include "md_template.h"
#endif
#endif

#ifndef NO_VIRTUAL_CONSOLE
#ifndef MD_KEY_SELECT
#ifdef __linux__
#define _LINUX_TYPES_H
#include <linux/vt.h>
#include <linux/kd.h>
#endif
#ifdef __FreeBSD__
#include <machine/console.h>
#endif
#if defined(__NetBSD__) && !defined(VT_PROCESS)
#include <dev/wscons/wsconsio.h>
#include <dev/wscons/wsdisplay_usl_io.h>
#ifndef VT_PROCESS
#warning no VT_PROCESS
#endif
#endif
#endif
#endif

#ifndef VT_TRUE
#define VT_TRUE	1
#endif

static char *mouse_icon="\
#MGR000200160016
@@++++++++++++++
@G@+++++++++++++
@GG@++++++++++++
@GGG@+++++++++++
@GGGG@++++++++++
@GGGGG@+++++++++
@GGGGGG@++++++++
@GGGGGG@@+++++++
@GGGGG@+++++++++
@G@@G@++++++++++
@@++@G@+++++++++
++++@GG@++++++++
++++@GGG@+++++++
++++@GGG@@++++++
++++@G@@++++++++
++++@@++++++++++
";

static int mouse_pattern[16][16];

static int mouse_x,mouse_y;
static int mouse_stat = 1;
static int mouse_down = 0;

static void emcons_refresh();

static int mouse_hide() {
    if (show_mouse) {
	if (!mouse_stat) return;
	rec_point(current_screen,mouse_x,mouse_y);
	rec_point(current_screen,mouse_x+15,mouse_y+15);
	mouse_stat = 0;
	emcons_refresh();
    }
}

static int mouse_show() {
    if (show_mouse) {
	if (mouse_stat) return;
	rec_point(current_screen,mouse_x,mouse_y);
	rec_point(current_screen,mouse_x+15,mouse_y+15);
	mouse_stat = 1;
	emcons_refresh();
    }
}

static inline int mouse_move(int new_x,int new_y) {
    if ((new_x == mouse_x) && (new_y == mouse_y)) return;
    if (show_mouse) {
	int old_stat;
	old_stat = mouse_stat;
	if (old_stat) mouse_hide();
	mouse_x = new_x;
	mouse_y = new_y;
	if (old_stat) mouse_show();
    }
}

static struct termios ttysave;
static struct termios ttycurrent;
static int can_disp = 0;

static struct event_manager emcons;

static int emcons_get_key(int);
static int _emcons_get_key(int);
static int emcons_key_select(int,fd_set *,int);
static void emcons_term();
static void emcons_set_icon(char *icon,char *name);
static int emcons_set_keymode(int);
static int _emcons_set_keymode(int);
static int emcons_proc_action(int v);
static void emcons_mouse_proc(int v,int x,int y);

static int emcons_bgmode;

/* internal function */
int (*im_readp)(char *ibuf, int ibufsize);

extern void put_key(int c);
extern void put_key_xy(struct virtual_key *v,int c,int x,int y);

static int setup_physical_screen();
extern void rp_wrap();

extern long long millitime(void);

extern int r_rect;

static void set_mouse_pattern(struct screen *s) {
	int y;
	push_screen(s);
	for (y=0; y<16; y++) {
		if (!rotated) 
		    get_pixstream(0,y,mouse_pattern[y],16,DIR_NORTH
				,BLT_MASKING|COLOR_LIGHTGRAY);
		else
		    get_pixstream(15,y,mouse_pattern[y],16,DIR_SOUTH
				,BLT_MASKING|COLOR_LIGHTGRAY);
	}
	pop_screen();
}

static int vt_get_active(int fd) {
#ifdef VT_GETACTIVE
	int scr = -1;
	if (ioctl(fd, VT_GETACTIVE, &scr) < 0) {
		scr = -1; /* XXX why ? */
	}
	return scr;
#else
#ifdef VT_GETSTATE
	struct vt_stat s;
	if (ioctl(fd, VT_GETSTATE, &s) < 0) {
		s.v_active = -1;
	}
	return s.v_active;
#else
	return -1;
#endif
#endif
}

#ifdef VT_PROCESS
static void set_tty(int mode);
static void switch_request(sig) int sig; {
	int scr;

	signal(SIGUSR1, switch_request);  /* should use sigaction()? */
	if (emcons_bgmode) {
		mgl_modifier_status = 0; /* force modifyer key release */
		emcons_bgmode = 0;
		MD_GRAPH_MODE();
		signal(SIGINT,SIG_IGN);	/* reset for bug ??? */
		set_tty(1);
		ioctl(mgl_keyboard_fd, KDSETMODE, KD_GRAPHICS);
		ioctl(mgl_keyboard_fd,VT_RELDISP,VT_ACKACQ);
		rec_point(current_screen,0,0);
		rec_point(current_screen,SCREEN_WIDTH-1,SCREEN_HEIGHT-1);
printf("recv signal to forground (%d)\r\n",scr_self);
#ifdef	MD_MOUSE_RELEASE
		MD_MOUSE_CATCH();
#endif
		/* delayed redraw automatically 
		 * there is no effect on shared_fb mode.
		 */
		redraw_time = millitime() + 100;
		emcons_refresh();
	} else {
		last_key_is_pressed = 0; /* force key release */
#ifdef	MD_MOUSE_RELEASE
		MD_MOUSE_RELEASE();
#endif
		MD_TEXT_MODE();
		ioctl(mgl_keyboard_fd, KDSETMODE, KD_TEXT);
		ioctl(mgl_keyboard_fd,VT_RELDISP,VT_TRUE);
		while (scr_self != -1 && (scr = vt_get_active(mgl_keyboard_fd)) == scr_self)
			;
		/* 
		 * scr == 0    : scr is NULL screen -> skip setting text mode
		 *   Currently this occur only in suspend/resume 
		 *     on NetBSD/hpcmips.
		 */
		if (scr == 0) {
			MD_GRAPH_MODE();
			ioctl(mgl_keyboard_fd, KDSETMODE, KD_GRAPHICS);
		}
printf("recv signal to background (%d -> %d)\r\n",scr_self,scr);
		emcons_bgmode = 1;
	}
}
#endif

static void init_tty() {
	tcgetattr(mgl_keyboard_fd, &ttysave);
	ttysave.c_iflag |= INLCR | ICRNL;
	ttysave.c_lflag |= ICANON | ECHO;
	ttysave.c_oflag |= OPOST | ONLCR;
	ttycurrent = ttysave;
	ttycurrent.c_iflag = IGNPAR | IGNBRK;
	ttycurrent.c_oflag = ONLCR;
	ttycurrent.c_cflag = CREAD | CS8;
	ttycurrent.c_lflag &= ~(ICANON | IEXTEN | ECHO | ISIG);
	ttycurrent.c_cc[VTIME] = 0;
	ttycurrent.c_cc[VMIN] = 1;
	tcsetattr(mgl_keyboard_fd, TCSANOW, &ttycurrent);
}

static void set_tty(int mode) {
	if (mode) {
		tcsetattr(mgl_keyboard_fd, TCSANOW, &ttycurrent);
	} else {
		tcsetattr(mgl_keyboard_fd, TCSANOW, &ttysave);
	}
}

static int create_keymap();
static void mgl2_signal_init();

int emcons_init(int debug) {
	struct screen *s;
#ifdef VT_PROCESS
 	struct vt_mode smode;
#endif
	if (MD_INIT() < 0) {
printf("error in MD_INIT\n");
		return -1;
	}
	setup_physical_screen(debug);
	mouse_x = SCREEN_WIDTH-16;
	mouse_y = SCREEN_HEIGHT-16;
	if (!rotated)
		MD_SET_MOUSEXY(mouse_x,mouse_y);
	else
		MD_SET_MOUSEXY(mouse_y,SCREEN_WIDTH-1-mouse_x);

	emcons._key_select = emcons_key_select;
	emcons._get_key = emcons_get_key;
	emcons._refresh = emcons_refresh;
	emcons._term = emcons_term;
	emcons._set_icon = emcons_set_icon;
	emcons._set_keymode = emcons_set_keymode;
	mgl_em = &emcons;
#ifdef VT_PROCESS
	ioctl(mgl_keyboard_fd, VT_WAITACTIVE, 0);
	ioctl(mgl_keyboard_fd, KDSETMODE, KD_GRAPHICS);
	scr_self = vt_get_active(mgl_keyboard_fd);
	signal(SIGUSR1,switch_request);

	smode.mode = VT_PROCESS;
	smode.waitv = 0;	/* dmy for FreeBSD */
	smode.relsig = SIGUSR1;
	smode.acqsig = SIGUSR1;
	smode.frsig  = SIGUSR1;	/* dmy for FreeBSD */
	if (ioctl(mgl_keyboard_fd,VT_SETMODE, &smode)) {
		perror("VT_SETMODE");
	}
#endif
	init_tty();

	vk_init();
	if (!share_fb)
		rp_wrap(physical_screen);
	can_disp = 1;

	s = conv_screen_from_mgr(mouse_icon,STK_NATIVE);
	set_mouse_pattern(s);
	free_screen(s);

#ifdef USE_KEYMAP
	local_translate = 1;
	mgl2_signal_init();

	if (create_keymap() < 0) {
		local_translate = 0;
		mk_init_keymap(mgl_getenv("MGL_KEYMAP"),keymap_postfix);
	} else {
		mgl_key_proc_action = emcons_proc_action;
		mgl_mouse_proc = emcons_mouse_proc;
	}
#else
	mk_init_keymap(mgl_getenv("MGL_KEYMAP"),keymap_postfix);
#endif
	emcons_set_keymode(MGL_SK_TRANSLATED);
	return 0;
}


extern struct screen *physical_screen,*current_screen;

static int setup_physical_screen(int debug) {
	int fd;
	int r;
	char *v;
	int de;

	de = STK_NATIVE;
	if(!_create_memscreen[STK_NATIVE]) {
		if ((depth > 8) && _create_memscreen[STK_GENERIC_FULLCOLOR]) {
			de = STK_GENERIC_FULLCOLOR;
			mgl_screen_bpp = 16;
		} else if ((depth == 8) && _create_memscreen[STK_GENERIC_192COLOR]) {
			de = STK_GENERIC_192COLOR;
			mgl_screen_bpp = 8;
		} else if ((depth == 4) && _create_memscreen[STK_GENERIC_16COLOR]) {
			de = STK_GENERIC_16COLOR;
			mgl_screen_bpp = 4;
		} else {
			de = STK_GENERIC_4COLOR;
			mgl_screen_bpp = 2;
		}
		_create_memscreen[STK_NATIVE] = _create_memscreen[de];
		mgl_screen_realtype = mgl_screen_type = de;
	}

	physical_screen = _create_memscreen[STK_NATIVE] (
		SCREEN_WIDTH,SCREEN_HEIGHT,mgl_screen_addr,mgl_screen_offset);
	if (mgl_screen_realwidth) {
		physical_screen->wbytes = mgl_screen_realwidth * 
					   mgl_screen_bpp / 8;
	} else {
		mgl_screen_realwidth = physical_screen->wbytes * 
					   8 / mgl_screen_bpp;
	}
	current_screen = physical_screen;

	/* set shift_bits (for use "same_format accel.") */
	switch(depth) {
	case 2:
	case 16:
		shift_bits = 2;
		break;

	case 4:
		shift_bits = 1;
		break;

	default:
		shift_bits = 0;
		break;
	}

	return 0;
}

static int emcons_set_keymode(int mode) {
	if (local_translate && (mode == MGL_SK_TRANSLATED)) {
		mode = MGL_SK_EXTRANSLATED;
	}
	return _emcons_set_keymode(mode);
}

static int _emcons_set_keymode(int mode) {
	int tmp;
	int ret;

#ifdef MD_SET_KEYMODE
	ret = MD_SET_KEYMODE(mode);
#else
	if (mode & (MGL_SK_RAW|MGL_SK_EXTRANSLATED)) {
		mgl_modifier_status = 0;
#if defined(__FreeBSD__)
		ret = ioctl(mgl_keyboard_fd,KDSKBMODE, K_CODE);
#elif defined(__NetBSD__)
		tmp = WSKBD_RAW;
		ret = ioctl(mgl_keyboard_fd,WSKBDIO_SETMODE, &tmp);
#elif defined(__linux__)
		ret = ioctl(mgl_keyboard_fd,KDSKBMODE, K_MEDIUMRAW);
#endif
	} else {
		last_key_is_pressed = 0;
#if defined(__FreeBSD__)
		ret = ioctl(mgl_keyboard_fd,KDSKBMODE, K_XLATE);
#elif defined (__NetBSD__)
		tmp = WSKBD_TRANSLATED;
		ret = ioctl(mgl_keyboard_fd,WSKBDIO_SETMODE, &tmp);
#elif defined(__linux__)
		ret = ioctl(mgl_keyboard_fd,KDSKBMODE, K_XLATE);
#endif
	}
#endif
	if (ret == 0) {
		emcons_key_mode = mode;
	}
	return ret;
}

static int key_conv_raw(int c) {
#ifdef USE_KEYMAP
	return mgl_key_conv_raw(c);
#else
#ifdef MD_KEY_CONV_RAW
	c = MD_KEY_CONV_RAW(c);
#elif defined(__NetBSD__)
	switch(c & 0x7f) {
	case  72: c =  95 | (c & 0x80); break;	/* Up */
	case  80: c = 100 | (c & 0x80); break;	/* Down */
	case  75: c =  97 | (c & 0x80); break; 	/* Left */
	case  77: c =  98 | (c & 0x80); break;	/* Right */
	case  82: c = 102 | (c & 0x80); break;	/* Ins */
	case  71: c =  94 | (c & 0x80); break;	/* Home */
	case  79: c =  99 | (c & 0x80); break;	/* End */
	case  73: c =  96 | (c & 0x80); break;	/* PgUp */
	case  81: c = 101 | (c & 0x80); break;	/* PgDown */
	case  93: c = 105 | (c & 0x80); break;	/* Menu */
	case 0xe0: c = 0; break;
	};
#elif defined(__linux__)
	switch(c & 0x7f) {
	case 103: c =  95 | (c & 0x80); break;	/* Up */
	case 108: c = 100 | (c & 0x80); break;	/* Down */
	case 105: c =  97 | (c & 0x80); break; 	/* Left */
	case 106: c =  98 | (c & 0x80); break;	/* Right */
	case 110: c = 102 | (c & 0x80); break;	/* Ins */
	case 111: c =  83 | (c & 0x80); break;	/* Del */
	case  83: c = 103 | (c & 0x80); break;	/* Del (Keypad) */
	case 102: c =  94 | (c & 0x80); break;	/* Home */
	case 107: c =  99 | (c & 0x80); break;	/* End */
	case 104: c =  96 | (c & 0x80); break;	/* PgUp */
	case 109: c = 101 | (c & 0x80); break;	/* PgDown */
	case 124: c = 125 | (c & 0x80); break;	/* \|  */
	case  89: c = 115 | (c & 0x80); break;	/* \_ */
	case  97: c =  96 | (c & 0x80); break;	/* R_Ctrl */
	case 100: c =  93 | (c & 0x80); break;	/* R_ALT */
	case 125: c = 105 | (c & 0x80); break;	/* Menu */
	};
#endif
	return c;
#endif
}

static void emcons_term(void) {
#ifdef VT_PROCESS
	struct vt_mode smode;
#endif

/* printf("start emcons_term.\n"); fflush(stdout); */
	_emcons_set_keymode(MGL_SK_TRANSLATED);

	if (!can_disp) return;

	set_color(COLOR_BLACK);
	clear_screen();
	emcons_refresh();

	can_disp = 0;

	set_tty(0);

#ifdef VT_PROCESS

	ioctl(mgl_keyboard_fd, KDSETMODE, KD_TEXT);
	smode.mode = VT_AUTO;
	ioctl(mgl_keyboard_fd, VT_SETMODE, &smode);

	/* printf("set text mode....\n"); */

#endif

	MD_TERM();

/* printf("end of emcons.\n"); fflush(stdout); */
}


static int emcons_get_key(int time_out) {
	long long lim,t;
	int k;

	if (!local_translate 
		|| ((MGL_SK_KM_MASK & mgl_key_mode) != MGL_SK_TRANSLATED)) {
		return _emcons_get_key(time_out);
	}
	lim = millitime() + (long long)time_out * 100;
	for (;;) {
		k = _emcons_get_key(time_out);
		if ((k & MGL_SKM_NOTICE) == 0) return k;
		t = millitime();
		if ((time_out >= 0) && (t >= lim)) return (-1);
	}
}

static int _emcons_get_key(int time_out) {
	fd_set fds;
	int c;

	if (key_buf_cnt) {
		return get_key_nodelay();
	}
	FD_ZERO( &fds );
        FD_SET(mgl_keyboard_fd, &fds);
	if (time_out >= 0) {
		time_out *= 100;
	}
	emcons_key_select(mgl_keyboard_fd+1, &fds, time_out);
	if (key_buf_cnt) {
		return get_key_nodelay();
	}
	return (-1);
}

static int altgr_unified = 0;
static int altgr_code = 0;
static int altgr_stat;

static int getch(void) {
	int ret;
	char buf[10];
	ret = read(mgl_keyboard_fd,buf,1);
	if (ret != 1) return (-1);

	if (altgr_unified 
		&& (emcons_key_mode & (MGL_SK_RAW|MGL_SK_EXTRANSLATED))) {
		if ((buf[0] & 0x7f) == altgr_code) {
			if (buf[0] & 0x80) {
				altgr_stat = 0;
			} else {
				altgr_stat = 0x40;
			}
			return (-1);
		} else {
 			return (buf[0] & 0xff) | altgr_stat ;
		}
	} else {
 		return (buf[0] & 0xff);
	}
}

#ifdef USE_KEYMAP
static void emcons_mouse_proc(int mode,int dx,int dy) {
	if (mode == 0) { /* down */
		vk_mouse_down(mouse_x,mouse_y);
	} else if (mode == 1) { /* move */
		int x = mouse_x + dx;
		int y = mouse_y + dy;

		if (do_mouse_smooth) vk_mouse_smooth_init(&x,&y);
		mouse_move(x,y);
		vk_mouse_move(x,y);
	} else if (mode == 2) { /* up */
		vk_mouse_up(mouse_x,mouse_y);
	} else if (mode == 3) { /* toggle mouse */
		if (show_mouse) mouse_hide();
		show_mouse = !show_mouse;
		if (show_mouse) mouse_show();
	}
}

static __mgl2_sig_handler_t mgl2_signal_tab[MKE_SIGNAL_SIZE];
#define signal_tab(x)   mgl2_signal_tab[(x) - MKE_PROC_BASE]

static void mgl2_signal_init() {
    int i;
    for(i = 0; i < MKE_SIGNAL_SIZE; i++){
	mgl2_signal_tab[i] = MD_PROC_IGN;
    }
#ifdef MD_PROC_CONSOLE
    for(i = MKE_CONSOLE; i <= MKE_CONSOLE_10; i++){
	signal_tab(i) = MD_PROC_CONSOLE;
    }
#endif
#ifdef MD_PROC_BACKLIGHT
    signal_tab(MKE_BACKLIGHT_TOGGLE) = MD_PROC_BACKLIGHT;
#endif
#ifdef MD_PROC_BRIGHTNESS
    signal_tab(MKE_BRIGHTNESS_UP)   = MD_PROC_BRIGHTNESS;
    signal_tab(MKE_BRIGHTNESS_DOWN) = MD_PROC_BRIGHTNESS;
#endif
#ifdef MD_PROC_CONTRAST
    signal_tab(MKE_CONTRAST_UP)   = MD_PROC_CONTRAST;
    signal_tab(MKE_CONTRAST_DOWN) = MD_PROC_CONTRAST;
#endif
#ifdef MD_PROC_POWER_OFF
    signal_tab(MKE_POWER_OFF) = MD_PROC_POWER_OFF;
#endif
#ifdef MD_PROC_PLAY
    signal_tab(MKE_STOP)   = MD_PROC_PLAY;
    signal_tab(MKE_PLAY)   = MD_PROC_PLAY;
    signal_tab(MKE_RECORD) = MD_PROC_PLAY;
#endif
    /* signal_tab(MKE_SWITCH_WINDOW) = MD_PROC_WINDOW; */
    /* signal_tab(MKE_SWITCH_FOCUS) = MD_PROC_FOCUS; */
}

__mgl2_sig_handler_t mgl2_signal(int sgnum, __mgl2_sig_handler_t handler) {

    __mgl2_sig_handler_t old_handler;

    if(sgnum < MKE_PROC_BASE) return MD_PROC_ERR;
    if((sgnum - MKE_PROC_BASE) >= MKE_SIGNAL_SIZE) return MD_PROC_ERR;
    old_handler = signal_tab(sgnum);

    if(handler != MD_PROC_DFL){
	signal_tab(sgnum) = handler;
    } else {
	switch(sgnum){
#ifdef MD_PROC_CONSOLE
	 case MKE_CONSOLE:
	 case MKE_CONSOLE_1:
	 case MKE_CONSOLE_2:
	 case MKE_CONSOLE_3:
	 case MKE_CONSOLE_4:
	 case MKE_CONSOLE_5:
	 case MKE_CONSOLE_6:
	 case MKE_CONSOLE_7:
	 case MKE_CONSOLE_8:
	 case MKE_CONSOLE_9:
	 case MKE_CONSOLE_10:
	    signal_tab(sgnum) = MD_PROC_CONSOLE;
	    break;
#endif
#ifdef MD_PROC_BACKLIGHT
	 case MKE_BACKLIGHT_TOGGLE:
	    signal_tab(sgnum) = MD_PROC_BACKLIGHT;
	    break;
#endif
#ifdef MD_PROC_BRIGHTNESS
	 case MKE_BRIGHTNESS_UP:
	 case MKE_BRIGHTNESS_DOWN:
	    signal_tab(sgnum) = MD_PROC_BRIGHTNESS;
	    break;
#endif
#ifdef MD_PROC_CONTRAST
	 case MKE_CONTRAST_UP:
	 case MKE_CONTRAST_DOWN:
	    signal_tab(sgnum) = MD_PROC_CONTRAST;
	    break;
#endif
#ifdef MD_PROC_POWER_OFF
	 case MKE_POWER_OFF:
	    signal_tab(sgnum) = MD_PROC_POWER_OFF;
	    break;
#endif
#ifdef MD_PROC_PLAY
	 case MKE_STOP:
	 case MKE_PLAY:
	 case MKE_RECORD:
	    signal_tab(sgnum) = MD_PROC_PLAY;
	    break;
#endif
	 default:
	    signal_tab(sgnum) = MD_PROC_IGN;
	    break;
	};
    }
    return old_handler;
}

static int emcons_proc_action(int v) {

    if(v < MKE_PROC_BASE) return -1;
    if((v - MKE_PROC_BASE) >= MKE_SIGNAL_SIZE) return -1;
    if(signal_tab(v) == MD_PROC_IGN) return -1;

    return (*signal_tab(v))(v);
}

#undef signal_tab
#endif

static int key_trans(int c) {
	return mgl_key_trans(c & 0x7f, c & 0x80);
}

static int do_local_translate(int c,int ret,int auto_repeat) {
	if (c < 0) return 0;

	if (emcons_key_mode & MGL_SK_RAW) {
		if (auto_repeat) {
			last_key_is_pressed = (c & 0x80)?0:1;
			if (last_key_is_pressed) {
				last_key = c;
				last_key_time = millitime();
			}
		}
		c = key_conv_raw(c);
		if (c) {
			put_key(c);
			return ret;
		}
	} else if (emcons_key_mode & MGL_SK_EXTRANSLATED) {
		if (auto_repeat) {
			last_key_is_pressed = (c & 0x80)?0:1;
			if (last_key_is_pressed) {
				last_key = c;
				last_key_time = millitime();
			}
		}
		c = key_conv_raw(c);
		c = key_trans(c);
			if (c >= 0) {
			put_key(c);
			return ret;
		}
	}
	return 0;
}

#ifndef MD_KEY_SELECT
#define MD_KEY_SELECT	_key_select
#if defined (__FreeBSD__)	/* cons25 type */
static int _key_select(int nfds, fd_set *readfds, int timeout) {
    int  ret;
    int c,r;
    struct timeval to,*top;
    fd_set fds;

    if (key_buf_cnt) {
        FD_ZERO(readfds);
        FD_SET(0, readfds);
	return 1;
    }

    if (timeout < 0) {
	top = 0;
    } else {
	top = &to;
	to.tv_sec = timeout / 1000;
	to.tv_usec = (timeout % 1000) * 1000;
    }

    FD_CLR(0, readfds);
    FD_SET(mgl_keyboard_fd, readfds);
    if (mgl_keyboard_fd+1 > nfds)
	nfds = mgl_keyboard_fd+1;
    ret = select(nfds, readfds, 0, 0, top);
    if (ret > 0 && FD_ISSET(mgl_keyboard_fd , readfds)) {
	FD_CLR(mgl_keyboard_fd, readfds);
        FD_SET(0, readfds);
	c = getch();
	if (emcons_key_mode & (MGL_SK_RAW|MGL_SK_EXTRANSLATED)) {
		return do_local_translate(c,ret,0);
	} else if (c != 033) {
		put_key(c);
		return ret;
	}
retry:
	FD_ZERO(&fds);
        FD_SET(mgl_keyboard_fd, &fds);
	to.tv_sec = 0;
	to.tv_usec = 300000;
    	r = select(mgl_keyboard_fd+1, &fds, 0, 0, &to);
        if (r > 0 && FD_ISSET(mgl_keyboard_fd , &fds)) {
		FD_CLR(mgl_keyboard_fd, &fds);
        	FD_SET(0, &fds);
		c = getch();
		if (c == 033) {
			put_key(033);
			goto retry;
		}
		if (c != '[') {
			put_key(033);
			put_key(c);
			return ret;
		}
		c = getch();
		switch (c) {
		case 'M':  c = MK_F1; break;
		case 'N':  c = MK_F2; break;
		case 'O':  c = MK_F3; break;
		case 'P':  c = MK_F4; break;
		case 'Q':  c = MK_F5; break;
		case 'R':  c = MK_F6; break;
		case 'S':  c = MK_F7; break;
		case 'T':  c = MK_F8; break;
		case 'U':  c = MK_F9; break;
		case 'V':  c = MK_F10; break;
		case 'W':  c = MK_F11; break; /* */
		case 'X':  c = MK_F12; break; /* */
		case 'A':  c = MK_UP; break;
		case 'B':  c = MK_DOWN; break;
		case 'C':  c = MK_RIGHT; break;
		case 'D':  c = MK_LEFT; break;
		case 'I':  c = MK_PAGE_UP; break;
		case 'G':  c = MK_PAGE_DOWN; break;
		case 'F':  c = MK_END; break; /* */
		case 'H':  c = MK_HOME; break; /* */
		case 'L':  c = MK_INS; break;
		case 'E':  c = MK_DEL; break;
		default:
			put_key(033);
			put_key('[');
			put_key(c);
			return ret;
		}
	}
	put_key(c);
    }
    return ret;
}
#elif defined (__NetBSD__) || defined(__CYGWIN__)	/* vt100 type ? */
static int _key_select(int nfds, fd_set *readfds, int timeout) {
    int  ret;
    int c,r,n;
    struct timeval to,*top;
    fd_set fds;

    if (key_buf_cnt) {
        FD_ZERO(readfds);
        FD_SET(0, readfds);
	return 1;
    }
    if (timeout < 0) {
	top = 0;
    } else {
	top = &to;
	to.tv_sec = timeout / 1000;
	to.tv_usec = (timeout % 1000) * 1000;
    }
    FD_CLR(0, readfds);
    FD_SET(mgl_keyboard_fd, readfds);
    if (mgl_keyboard_fd+1 > nfds)
	nfds = mgl_keyboard_fd+1;
    ret = select(nfds, readfds, 0, 0, top);
    if (ret > 0 && FD_ISSET(mgl_keyboard_fd , readfds)) {
	FD_CLR(mgl_keyboard_fd, readfds);
	FD_SET(0, readfds);
	c = getch();
	if (emcons_key_mode & (MGL_SK_RAW|MGL_SK_EXTRANSLATED)) {
		return do_local_translate(c,ret,1);
	} else if (c != 033) {
		put_key(c);
		return ret;
	}
retry:
	FD_ZERO(&fds);
        FD_SET(mgl_keyboard_fd, &fds);
	to.tv_sec = 0;
	to.tv_usec = 300000;
    	r = select(mgl_keyboard_fd+1, &fds, 0, 0, &to);
        if (r > 0 && FD_ISSET(mgl_keyboard_fd , &fds)) {
            FD_CLR(mgl_keyboard_fd, &fds);
            FD_SET(0, &fds);
	    c = getch();
	    if (c == 033) {
	 	put_key(033);
		goto retry;
	    }
	    if (c != '[') {
		put_key(033);
		put_key(c);
		return ret;
	    }
	    c = getch();
	    n = 0;
	    while (('0' <= c) && (c <= '9')) {
		n *= 10;
		n += c - '0';
		c = getch();
	    }
	    if (n > 0) {
		if (c != '~') {
			put_key(033);
			put_key('[');
			if (n > 10) put_key((n/10)+'0');
			put_key((n%10)+'0');
		} else 
		switch(n) {
			case 11: c = MK_F1; break;
			case 12: c = MK_F2; break;
			case 13: c = MK_F3; break;
			case 14: c = MK_F4; break;
			case 15: c = MK_F5; break;
			case 17: c = MK_F6; break;
			case 18: c = MK_F7; break;
			case 19: c = MK_F8; break;
			case 20: c = MK_F9; break;
			case 21: c = MK_F10; break;
			case 23: c = MK_F11; break;
			case 24: c = MK_F12; break;
			case  2: c = MK_INS; break;
			case  3: c = MK_DEL; break;
			case  5: c = MK_PAGE_UP; break;
			case  6: c = MK_PAGE_DOWN; break;
			default:
				put_key(033);
				put_key('[');
				if (n > 10) put_key((n/10)+'0');
				put_key((n%10)+'0');
		}
	    } else {
		switch(c) {
		case 'A':  c = MK_UP; break;
		case 'B':  c = MK_DOWN; break;
		case 'C':  c = MK_RIGHT; break;
		case 'D':  c = MK_LEFT; break;
		default:
			put_key(033);
			put_key('[');
		}
	    }
	}
	put_key(c);
    }
    return ret;
}
#elif defined (__linux__)
static int _key_select(int nfds, fd_set *readfds, int timeout) {
    int  ret;
    int c,r,n;
    struct timeval to,*top;
    fd_set fds;

    if (key_buf_cnt) {
        FD_ZERO(readfds);
        FD_SET(0, readfds);
	return 1;
    }
    if (timeout < 0) {
	top = 0;
    } else {
	top = &to;
	to.tv_sec = timeout / 1000;
	to.tv_usec = (timeout % 1000) * 1000;
    }
    FD_CLR(0, readfds);
    FD_SET(mgl_keyboard_fd, readfds);
    if (mgl_keyboard_fd+1 > nfds)
	nfds = mgl_keyboard_fd+1;
    ret = select(nfds, readfds, 0, 0, top);

    if (ret > 0 && FD_ISSET(mgl_keyboard_fd , readfds)) {
	FD_CLR(mgl_keyboard_fd, readfds);
	FD_SET(0, readfds);
	c = getch();
	if (emcons_key_mode & (MGL_SK_RAW|MGL_SK_EXTRANSLATED)) {
		return do_local_translate(c,ret,0);
	}
	if (c != 033) {
		put_key(c);
		return ret;
	}
retry:
	FD_ZERO(&fds);
        FD_SET(mgl_keyboard_fd, &fds);
	to.tv_sec = 0;
	to.tv_usec = 300000;
    	r = select(mgl_keyboard_fd+1, &fds, 0, 0, &to);
        if (r > 0 && FD_ISSET(mgl_keyboard_fd , &fds)) {
            FD_CLR(mgl_keyboard_fd, &fds);
	    FD_SET(0, &fds);

	    c = getch();
	    if (c == 033) {
		put_key(033);
		goto retry;
	    }
	    if (c != '[') {
		put_key(033);
		put_key(c);
		return ret;
	    }
	    c = getch();
	    n = 0;
	    while (('0' <= c) && (c <= '9')) {
		n *= 10;
		n += c - '0';
		c = getch();
	    }
	    if (n > 0) {
		if (c != '~') {
			put_key(033);
			put_key('[');
			if (n > 10) put_key((n/10)+'0');
			put_key((n%10)+'0');
		} else 
		switch(n) {
			case 11: c = MK_F1; break;
			case 12: c = MK_F2; break;
			case 13: c = MK_F3; break;
			case 14: c = MK_F4; break;
			case 15: c = MK_F5; break;
			case 17: c = MK_F6; break;
			case 18: c = MK_F7; break;
			case 19: c = MK_F8; break;
			case 20: c = MK_F9; break;
			case 21: c = MK_F10; break;
			case 23: c = MK_F11; break;
			case 24: c = MK_F12; break;
			case  2: c = MK_INS; break;
			case  3: c = MK_DEL; break;
			case  5: c = MK_PAGE_UP; break;
			case  6: c = MK_PAGE_DOWN; break;
			default:
				put_key(033);
				put_key('[');
				if (n > 10) put_key((n/10)+'0');
				put_key((n%10)+'0');
		}
	    } else {
		switch(c) {
		case 'A':  c = MK_UP; break;
		case 'B':  c = MK_DOWN; break;
		case 'C':  c = MK_RIGHT; break;
		case 'D':  c = MK_LEFT; break;
#if 0 /* ???? */
		case 'W':  c = MK_F11; break;
		case 'X':  c = MK_F12; break;
		case 'I':  c = MK_PAGE_UP; break;
		case 'G':  c = MK_PAGE_DOWN; break;
		case 'F':  c = MK_END; break;
		case 'H':  c = MK_HOME; break;
#endif
		case '[':
			c = getch();
			switch (c) {
			case 'A':  c = MK_F1; break;
			case 'B':  c = MK_F2; break;
			case 'C':  c = MK_F3; break;
			case 'D':  c = MK_F4; break;
			case 'E':  c = MK_F5; break;
			default:
				put_key(033);
				put_key('[');
				put_key('[');
			}
		default:
			put_key(033);
			put_key('[');
		}
	    }
	}
	put_key(c);
    }
    return ret;
}
#endif
#endif

static int emcons_key_select(nfds, readfds, timeout)
int    nfds;
fd_set *readfds;
int timeout;
{
	int ret;
	fd_set local;
	int ticks;
	long long s,e,t,tt;
	int x,y,b;

	t = s = millitime();
	if ((redraw_time != 0LL) && (s > redraw_time)) {
		rec_point(physical_screen,0,0);
		rec_point(physical_screen,SCREEN_WIDTH-1,SCREEN_HEIGHT-1);
		redraw_time = 0LL;
	}
	emcons_refresh();	/* auto matically refresh */
	if (mouse_down) 
		ticks = mouse_tick;
	else
		ticks = repeat_tick;

	if (timeout < 0) {
		e = 0x7fffffffffffffffLL;
	} else if (timeout < repeat_tick) {
		e = s + timeout;
		ticks = timeout;
	} else {
		e = s + timeout;
	}
	for (;;) {
	    if (!emcons_bgmode && can_disp && MD_MOUSE_EVENT()) {
		if (!rotated) {
		     x = MD_MOUSE_X();
		     y = MD_MOUSE_Y();
		} else {
		     x = MD_MOUSE_Y();
		     y = SCREEN_HEIGHT - 1 - MD_MOUSE_X();
		}
		b = MD_MOUSE_BUTTON();

		if (b  ^ mouse_down) {
			mouse_down = b;
			if (!b) {
#if 0
				if (!((mgl_modifier_status
				    & (MGL_SKM_MENU|MGL_SKM_RMENU)))) {
					vk_mouse_up(mouse_x,mouse_y);
				}
#else
				if (do_mouse_smooth) vk_mouse_smooth(&x,&y);
		    		mouse_move(x,y);
				if ((mgl_modifier_status
				    & (MGL_SKM_MENU|MGL_SKM_RMENU))) {
					vk_mouse_move(x,y);
				} else {
					vk_mouse_up(x,y);
				}
#endif
			} else if((emcons_key_mode & MGL_SK_EXTRANSLATED)
				&& (mgl_modifier_status & MGL_SKM_MENU)) {
				if (do_mouse_smooth) vk_mouse_smooth(&x,&y);
		    		mouse_move(x,y);
				vk_mouse_move(x,y);
			} else {
				if (do_mouse_smooth) vk_mouse_smooth_init(&x,&y);
		    		mouse_move(x,y);
				if (mgl_modifier_status
				     & (MGL_SKM_MENU|MGL_SKM_RMENU)) {
					vk_mouse_move(x,y);
				} else {
					vk_mouse_down(x,y);
				}
			}
		} else {
				if (do_mouse_smooth) vk_mouse_smooth(&x,&y);
		    		mouse_move(x,y);
				vk_mouse_move(x,y);
		}
	    }
	    local = *readfds;
	    if (delayed_key && (t > dk_limit)) {
		put_key(-1);
	    }
	    ret = MD_KEY_SELECT(nfds,&local,ticks);
	    tt = millitime();
	    if ((tt - t) > 5000) {
		/* auto matically refresh after suspend */
		rec_point(physical_screen,0,0);
		rec_point(physical_screen,SCREEN_WIDTH-1,SCREEN_HEIGHT-1);
		emcons_refresh();
	    }
	    t = tt;
	    if (t > e) break;
	    if (ret != 0) break;
	    if (!delayed_key && last_key_is_pressed ) {
		if (last_key_is_pressed == 1) {
		    if (tt -last_key_time > repeat_tick1) {
			do_local_translate(last_key,0,0);
			last_key_time = tt;
			last_key_is_pressed ++;
			ret = 1;
			break;
		    }
		} else {
		    if (tt -last_key_time > repeat_tick) {
			do_local_translate(last_key,0,0);
			last_key_time = tt;
			ret = 1;
			break;
		    }
		}
	    }
	}
	*readfds = local;
	return ret;
}

extern int r_rect;
extern int r_minx;
extern int r_maxx;
extern int r_miny;
extern int r_maxy;

static void emcons_refresh() {
	int c,x,y;
	int xs;
	int buf[1024];

	if (share_fb) return;
	if (emcons_bgmode) return;
	if (!can_disp) return;
//printf("refresh %d (%d,%d) - (%d,%d)\n",r_rect,r_minx,r_miny,r_maxx,r_maxy);
	if (!r_rect) {
		return;
	}
	if (r_rect == 2) {
		r_minx = 0;
		r_miny = 0;
		r_maxx = physical_screen->width-1;
		r_maxy = physical_screen->height-1;
	}
	/* Check */
	if(r_minx < 0){
		r_minx = 0;
	}
	if(r_miny < 0){
		r_miny = 0;
	}
	if(r_maxx > physical_screen->width-1) {
		r_maxx = physical_screen->width-1;
	}
	if(r_maxy > physical_screen->height-1) {
		r_maxy = physical_screen->height-1;
	}
	if ((r_minx > r_maxx) || (r_miny > r_maxy)) {
		r_rect = r_minx = r_miny = r_maxx = r_maxy = 0;
		return;
	}

     if (same_format && !(physical_screen->off_x & 0x3)) {
	char *src;
	int wbytes;
	int xoffset,nbytes,src_offset;

	r_minx = r_minx & ~0x3;
	r_maxx = ((r_maxx+4) & ~0x3 ) - 1;
	nbytes = r_maxx - r_minx + 1;

	src_offset = r_minx + physical_screen->off_x;
	xoffset = r_minx;

	if(depth > 8) {
	        nbytes <<= shift_bits;
	        src_offset <<= shift_bits;
	        xoffset <<= shift_bits;
	} else {
	        nbytes >>= shift_bits;
	        src_offset >>= shift_bits;
	        xoffset >>= shift_bits;
	}

	wbytes = physical_screen->wbytes;
	src = (physical_screen->type & ST_SUBSCREEN)
		?((struct screen *)(physical_screen->bitmap))->bitmap
		:(char *)physical_screen->bitmap;
	src += (r_miny + physical_screen->off_y) * wbytes + src_offset;

	for (y=r_miny; y<=r_maxy; y++) {
		MD_PUT_SCANSEGMENT(y,xoffset,src,nbytes);
		src += wbytes;
	}
    } else
     if (!rotated) {
	r_minx = r_minx & ~0x7;
	r_maxx = ((r_maxx+8) & ~0x7 ) - 1;

	xs = r_maxx - r_minx + 1;

	push_screen(physical_screen);
	for (y=r_miny; y<=r_maxy; y++) {
		get_pixstream(r_minx,y,buf,xs,0,0);
		if (show_mouse)
		  if (mouse_stat && (mouse_y <= y) && (y <= mouse_y + 15)) {
		    for (x=r_minx; x<=r_maxx; x++) {
			if ((mouse_x <= x) && (x <= mouse_x + 15)) {
				c = mouse_pattern[y - mouse_y][x - mouse_x];
				if (!(c & COLOR_TRANSPARENT)) {
					buf[x-r_minx] = c;
				}
			}
		    }
		  }
		switch(depth) {
		case 2:
			MD_PUT_PIXSTREAM2(r_minx,y,buf,xs,0,0);
			break;
		case 4:
			MD_PUT_PIXSTREAM4(r_minx,y,buf,xs,0,0);
			break;
		case 8:
			MD_PUT_PIXSTREAM8(r_minx,y,buf,xs,0,0);
			break;
		case 15:
		case 16:
			MD_PUT_PIXSTREAM16(r_minx,y,buf,xs,0,0);
			break;
		}
	}
	pop_screen();
    } else {
	int minx,maxx,miny,maxy;
	int mx,my;

	minx = (physical_screen->height - 1 - r_maxy) & ~0x7;
	maxx = ((physical_screen->height - 1 - r_miny+8) & ~0x7) -1;
	miny = r_minx;
	maxy = r_maxx;
	xs = maxx - minx + 1;
	r_maxy = physical_screen->height - 1 - minx;
	r_miny = physical_screen->height - 1 - maxx;

	mx = physical_screen->height - 1 - mouse_y-15;
	my = mouse_x;
	push_screen(physical_screen);
	for (y=miny; y<=maxy; y++) {
		get_pixstream(r_minx++,r_maxy,buf,xs,DIR_WEST,0);
		if (show_mouse)
		  if (mouse_stat && (my <= y) && (y <= my + 15)) {
		    for (x=minx; x<=maxx; x++) {
			if ((mx <= x) && (x <= mx + 15)) {
				c = mouse_pattern[y - my][x - mx];
				if (!(c & COLOR_TRANSPARENT)) {
					buf[x-minx] = c;
				}
			}
		    }
		  }
		switch(depth) {
		case 2:
			MD_PUT_PIXSTREAM2(minx,y,buf,xs,0,0);
			break;
		case 4:
			MD_PUT_PIXSTREAM4(minx,y,buf,xs,0,0);
			break;
		case 8:
			MD_PUT_PIXSTREAM8(minx,y,buf,xs,0,0);
			break;
		case 15:
		case 16:
			MD_PUT_PIXSTREAM16(minx,y,buf,xs,0,0);
			break;
		}
	}
	pop_screen();
    }
	r_rect = r_minx = r_miny = r_maxx = r_maxy = 0;
}

static void emcons_set_icon(char *icon,char *name) {
}

#ifdef USE_KEYMAP
static void modify_keymap_for_mglsvr() {
	int i,j;
	int no_switch_window=1;
	int no_switch_focus=1;

	for (i=0; i<128; i++) {
		for (j=0; j<3; j++) {
			if (mgl_keymap[i][j] == MKE_SWITCH_WINDOW) {
				no_switch_window=0;
			}
			if (mgl_keymap[i][j] == MKE_SWITCH_FOCUS) {
				no_switch_focus=0;
			}
		}
	}
	for (i=0; i<128; i++) {
		for (j=0; j<3; j++) {
			if (no_switch_window && (mgl_keymap[i][j] == MK_F9)
			   && !((j == 1) && (mgl_keymap[i][0] == MKE_SWITCH_WINDOW))) {
				mgl_keymap[i][j] = MKE_SWITCH_WINDOW;
			}
			if (no_switch_focus && (mgl_keymap[i][j] == MK_F10)
			   && !((j == 1) && (mgl_keymap[i][0] == MKE_SWITCH_FOCUS))) {
				mgl_keymap[i][j] = MKE_SWITCH_FOCUS;
			}
		}
	}
}

#ifdef __linux__
#include "Linux/keymap.c"
#endif
#ifdef __FreeBSD__
#include "FreeBSD/keymap.c"
#endif
#ifdef __NetBSD__
#include "NetBSD/keymap.c"
#endif

#endif /* USE_KEYMAP */
