/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/mman.h>
#include <sys/time.h>

#include "config.h"
#include "mgl2.h"
#include "event_man.h"

#include "em_comm.h"
#include "mglcol.h"
#include "mglkey.h"
#include "keymap.h"

#ifdef __CYGWIN__
#include <windows.h>
HANDLE hFileMapping;
#endif

#ifdef USE_LOCAL_MEMSET
#define memset	mgl_memset
#endif
#ifdef USE_LOCAL_MEMMOVE
#define memmove	mgl_memmove
#endif

#define COLOR_FOCUS	   packMC(0,15,15)

static int emc_sock_fd;
static int emc_screen_fd;
static char *emc_screen_addr;

static struct screen *emc_screen;
extern struct screen *physical_screen,*current_screen;

static int emcli_key_mode;

extern int mgl_client;
extern int mgl_screen_offset;
extern int mgl_screen_realwidth;
extern int mgl_screen_bpp;
extern int mgl_screen_type;
extern int mgl_screen_realtype;
extern char mgl_display[];
extern char *mgl_screenattr_addr;
#ifdef USE_SENDIMAGE
static int send_image;
#else
const static int send_image = 0;
#endif

static void emc_refresh();
extern void put_key_xy(struct virtual_key *v,int c,int x,int y);

static int emc_send(struct emc_packet *p) {
	int r;
	if (emc_sock_fd <= 0) return 0;

#ifdef DEBUG_COMM
	emc_packet_print("Send",p);
#endif

	r = send(emc_sock_fd,p,sizeof(struct emc_packet),0);
	return r;
}

static int emc_recv(struct emc_packet *p) {
	int r;
	if (emc_sock_fd <= 0) return 0;
	r = recv(emc_sock_fd,p,sizeof(*p),0);
	if (r <= 0) {
		perror("emc_read");
		fprintf(stderr,"connection closed\n");
		exit(2);
	}
#ifdef DEBUG_COMM
	emc_packet_print("Recv",p);
#endif
	return r;
}

static void emc_execute_event() {
	struct emc_packet ep;
	int i,r,x,y;

	r = emc_recv(&ep);
	if (r < 0) {
		exit(1);
	}
	switch (ep.kind) {
	case EVENT_FOCUS:	/* on */
		if (mgl_apli_type == AT_MINIAPLI) {
			push_screen(physical_screen);
			if (ep.args[0] == FOCUS_ON) {
				set_color(COLOR_FOCUS);
			} else {
				set_color(COLOR_DARKGRAY);
			}
			draw_rect(0,0,SCREEN_WIDTH,SCREEN_HEIGHT);
			draw_rect(1,1,SCREEN_WIDTH-2,SCREEN_HEIGHT-2);
			emc_refresh();
			pop_screen();
		}
		break;
	case EVENT_KEY:		/* ncodes, code1,code2... */
		for (i=1; i<= ep.args[0]; i++) {
			if ((ep.args[i] >= 0) 
				&& (ep.args[i] & MGL_SKM_MASK)) {
				mgl_modifier_status = ep.args[i] & MGL_SKM_MASK;
			}
			put_key(ep.args[i]);
		}
		break;
	case EVENT_MMOVE:	/* x,y */
		x = ep.args[0]; y = ep.args[1];
		vk_mouse_move(x,y);
		break;
	case EVENT_MRELEASE:	/* x,y */
		x = ep.args[0]; y = ep.args[1];
		vk_mouse_up(x,y);
		break;
	case EVENT_MPRESS:	/* x,y */
		x = ep.args[0]; y = ep.args[1];
		mgl_button_shift = ep.args[2];
		vk_mouse_down(x,y);
		break;
	case REQ_DISPMODE:	/* hide */
		if (ep.args[0] == 1) { /* hide */
		    if (!bgmode) {
			screen_switch(1);
		    }
		} else {
		    if (bgmode) {
			if (fg_screen) {
				free_screen(fg_screen);
			}
			fg_screen = emc_screen->de->_create_subscreen(emc_screen
				,ep.args[1],ep.args[2]
				,bg_screen->width,bg_screen->height
				,CSS_AS_MEMSCREEN);
			screen_switch(0);
			r_rect = 2;
		    }
		}
		ep.kind = RPLY_DISPMODE;
		r = emc_send(&ep);
		if (r <= 0) {
			perror("send");
			exit(2);
		}
		if (!bgmode) {
			emc_refresh();
		}
		break;
	default:
		fprintf(stderr, "emc_execute_event(0x%04x): unknown request.\n", ep.kind);
	}
}

extern long long millitime();
static int    _key_select(int nfds, fd_set *readfds, int timeout);

static int emc_key_select(nfds, readfds, timeout)
int    nfds;
fd_set *readfds;
int timeout;
{
	int ret;
	fd_set local;
	int ticks;
	long long s,e,t;
	int x,y,b;

	t = s = millitime();
	emc_refresh();	/* auto matically refresh */
	if (timeout < 0) {
		e = 0x7fffffffffffffffLL;
		ticks = 300;
	} else if (timeout < 300) {
		e = s + timeout;
		ticks = timeout;
	} else {
		e = s + timeout;
		ticks = 300;
	}
	for (;;) {
		local = *readfds;
		if (delayed_key && (t > dk_limit)) {
			put_key(-1);
		}
		ret = _key_select(nfds,&local,ticks);
		t = millitime();
		if (t > e) break;
		if (ret != 0) break;
	}
	*readfds = local;
	return ret;
}


static int _key_select(int nfds, fd_set *readfds, int timeout) {
    int  ret;
    struct timeval to,*top;
    int xfd = -1;

    xfd = emc_sock_fd;
    if (key_buf_cnt) {
	timeout = 0;
    }
    FD_CLR(0, readfds);
    if (xfd > 0) {
	if (xfd+1 > nfds) {
		nfds = xfd+1;
	}
	FD_SET(xfd, readfds);
    }

    top = &to;
    to.tv_sec = timeout / 1000;
    to.tv_usec = (timeout % 1000) * 1000;

    ret = select(nfds, readfds, 0, 0, top);

    if ((ret < 0)) {
        /* error or time out */
        return ret;
    }
    if ((xfd > 0) && FD_ISSET(xfd,readfds)) {
	if (ret > 0) ret--;
        FD_CLR(xfd, readfds);
	emc_execute_event();
    }
    if (key_buf_cnt) {
	ret += 1;
        FD_SET(0, readfds);
    }
    return ret;
}

static int emc_get_key_common(int timeout) {
	long long s;
	fd_set fds;
	int c,r;

	if (!key_buf_cnt) {
		if (timeout > 0) timeout *= 100;
		FD_ZERO(&fds);
		FD_SET(0,&fds);
		r = emc_key_select(1,&fds,timeout);
	}
	if (key_buf_cnt) {
		int ret = get_key_nodelay();
		if (ret >= 0) {
			int sym = ret & ~MGL_SKM_MASK;
			if ((sym >= MKE_BASE) && (sym < MKE_BASE+MKE_SIZE)) {
				sym = sym_trans_tab[sym];
				if (sym >= 0) {
					ret = (ret & MGL_SKM_MASK)
						| (sym & ~MGL_SKM_MASK);
				} else {
					ret = sym;
				}
			}
		}
		return ret;
	}
	return (-1);
}

static int emc_get_key(int timeout) {
	struct emc_packet ep;
	int r;

#if 0
	if (im_mode) {
		im_mode = !im_mode;
		ep.kind = SET_IM_MODE;
		ep.args[0] = 0;
		r = emc_send(&ep);
		if (r < 0) {
			exit(1);
		}
	}
#endif
	return emc_get_key_common(timeout);
}


extern int r_rect;
extern int r_minx;
extern int r_maxx;
extern int r_miny;
extern int r_maxy;

static void emc_refresh() {
	struct emc_packet e;

//printf("refresh %d (%d,%d) - (%d,%d)\n",r_rect,r_minx,r_miny,r_maxx,r_maxy);
	if (bgmode) {
		return;
	}
	if (!r_rect) {
		return;
	}
	if (r_rect == 2) {
		r_minx = 0;
		r_miny = 0;
		r_maxx = physical_screen->width-1;
		r_maxy = physical_screen->height-1;
	}
	/* Check */
	if(r_minx < 0){
		r_minx = 0;
	}
	if(r_miny < 0){
		r_miny = 0;
	}
	if(r_maxx > physical_screen->width-1){
		r_maxx = physical_screen->width-1;
	}
	if(r_maxy > physical_screen->height-1){
		r_maxy = physical_screen->height-1;
	}

	e.kind = REQ_REFRESH;
	e.args[0] = r_minx;
	e.args[1] = r_miny;
	e.args[2] = r_maxx;
	e.args[3] = r_maxy;
	if (emc_send(&e) != sizeof(struct emc_packet)) {
		fprintf(stderr,"cannt send packet\n");
		exit(2);
	}
#ifdef USE_SENDIMAGE
	if (send_image) {
		int r,n,l;
		char *p = NULL;
		struct screen *s;
		int xs = (r_maxx - r_minx + 1);
		int ys = (r_maxy - r_miny + 1);
		int sz = xs * ys * mgl_screen_bpp / 8;
		char buf[sz];

		if (0 /*xs == SCREEN_WIDTH*/) {
			p = emc_screen_addr 
				+ r_miny * SCREEN_WIDTH * mgl_screen_bpp / 8;
		} else {
			p = buf;
			s = _create_memscreen[STK_NATIVE](xs,ys,p,0);
			if (s) {
				bitblt(s,0,0,physical_screen,r_minx,r_miny
					,xs,ys,0);
				free_screen(s);
			} else {
printf("create memscreen error\n");
			}
		}
		n = 0;
		while (n < sz) {
			l = 8192;
			if ((sz - n) < l) l = sz - n;
			r = send(emc_sock_fd,p + n,l,0);
			if (r != l) {
				perror("send image");
				exit(2);
			}
			n += l;
		}
//printf("send %d %d,%d n = %d\n",sz,xs,ys,n);
	}
#endif
	r_rect = r_minx = r_miny = r_maxx = r_maxy = 0;
}

static int emc_set_keymode(int mode) {
	struct emc_packet e;

	e.kind = SET_KEY_MODE;
	e.args[0] = mode;
	emc_send(&e);
	emcli_key_mode = mode;
	return 0;
}

static void emc_term() {
	struct emc_packet e;
	e.kind = EXIT_NOTIFY;
//fprintf(stderr,"EXIT_NOTIFY\n");
	emc_send(&e);
	close(emc_sock_fd);
	close(emc_screen_fd);
}

static void emc_set_icon(char *icon,char *name) {
	struct emc_packet ep;
	int r,len;

	len = strlen(icon);
	if ((len > 1024) || strncmp(icon,"#MGR0002",8)) {
		fprintf(stderr,"set_icon format error\n");
		return;
	}
	ep.kind = SET_ICON;
	ep.args[0] = len;
	if (name) {
		strncpy((char *)(&ep.args[1]),name,16);
	} else {
		memset(&ep.args[1],0,16);
	}
	r = emc_send(&ep);
	if (r != sizeof(ep)) {
		perror("send icon");
		exit(2);
	}
	r = send(emc_sock_fd,icon,len,0);
	if (r != len) {
		perror("send icon");
		exit(2);
	}
}

static struct event_manager emc_em = {
	emc_get_key, emc_key_select,
	emc_refresh,emc_set_icon,emc_term,emc_set_keymode
};

static int find_server(char *dir) {
	struct stat sb;
	char fname[256];
	sprintf(fname,"%s/mgl.sock.%s",dir,mgl_display);
//printf("stat %s\n",fname);
	if (!stat(fname,&sb)) return 1;
	return 0;
}

int emc_init(int debug) {
	char emc_sock_name[32];
	char emc_screen_name[32];
	char emc_screen_name_alt[32];
	char emc_work_dir[256];
	char curdir[256];
	char *p;
	int sd,fd,fd_alt;
	int flag,i,r;
	struct sockaddr_un addr;
	struct emc_packet ep;
	int fbsize = MAX_SCREEN_SIZE;
	int fbsize_alt = MAX_SCREENATTR_SIZE;
	char *v,*v_alt;
	int xs,ys;

	for (i=0; i<128; i++) {
		mgl_keymap[i][0] = MK_NONE;
		mgl_keymap[i][1] = MK_NONE;
		mgl_keymap[i][3] = MK_NONE;
	}
	if (p = getenv("HOME")) {
		sprintf(emc_work_dir,"%s/.mgl",p);
		if (find_server(emc_work_dir)) goto found;
	}
	strcpy(emc_work_dir,"/var/run");
	if (!find_server(emc_work_dir)) {
//printf("cannot found server\n");
		return (-1);
	}
found:
	sprintf(emc_sock_name,"mgl.sock.%s",mgl_display);
	getcwd(curdir,sizeof(curdir));

	r = chdir(emc_work_dir);
	if (r < 0) {
		printf("cannot chdir to ${HOME}/.mgl or /var/run\n");
		goto bad1;
	}
	chown(emc_sock_name,getuid(),getgid());
	chmod(emc_sock_name,0600);
	emc_sock_fd = sd = socket( AF_LOCAL, SOCK_STREAM, 0);
	if (sd < 0) {
		if (debug) perror("socket(2)");
		goto bad2;
	}
	addr.sun_family = AF_UNIX;
	strcpy(addr.sun_path,emc_sock_name);

	r = connect(sd,(struct sockaddr *)&addr, sizeof(struct sockaddr_un));
	if (r != 0) {
		if (debug) perror("connect(2)");
		goto bad2;
	}
	ep.kind = REQ_CONNECT;
	if (mgl_apli_type) {
		ep.args[0] = mgl_apli_type;
	} else {
		ep.args[0] = AT_MAIN;	/* want */
	}
	r = emc_send(&ep);
	if (r <= 0) {
		if (debug) perror("send request open");
		goto bad3;
	}
next_recv:
	r = emc_recv(&ep);
	if (r <= 0) {
		if (debug) perror("recv reply request open");
		goto bad3;
	}
	if (ep.kind == PUT_KEYMAP) {
		mgl_keymap[ep.args[0]][0] = ep.args[1];
		mgl_keymap[ep.args[0]][1] = ep.args[2];
		mgl_keymap[ep.args[0]][2] = ep.args[3];
		goto next_recv;
	}
	if ((ep.kind != RPLY_CONNECT) || (ep.args[0] <= 0)) {
		if (debug) fprintf(stderr,"request open refused\n");
		goto bad3;
	}
	SCREEN_WIDTH = ep.args[0];
	SCREEN_HEIGHT = ep.args[1];
	xs = ep.args[2];
	ys = ep.args[3];
	_create_memscreen[STK_NATIVE] = 
			_create_memscreen[ep.args[4] & 0xff];
	mgl_screen_realtype = ep.args[4] & 0xff;
	mgl_screen_type = (ep.args[4] >> 8) & 0xff;
	mgl_screen_bpp = (ep.args[4] >> 16) & 0xff;
#ifdef USE_SENDIMAGE
	send_image = (ep.args[4] >> 24) & 1;
	if (send_image && (mgl_screen_bpp < 8)) {
		/* need 8bpp or 16bpp */
		goto bad3;
	}
#endif

	mgl_screen_offset = ep.args[5];
	mgl_screen_realwidth = xs;
//printf("type %d bpp %d realtype %d realwidth %d \n"
//		,mgl_screen_type,mgl_screen_bpp
//		,mgl_screen_realtype,mgl_screen_realwidth);
	fbsize = xs * ys * mgl_screen_bpp / 8 + mgl_screen_offset;

	if (*((char *)(&ep.args[6]))) {
		strncpy(emc_screen_name,(char *)(&ep.args[6]),16);
		emc_screen_name[16] = 0;
		sprintf(emc_screen_name_alt,"mgl.screen.%s",mgl_display);
	} else {
		sprintf(emc_screen_name,"mgl.screen.%s",mgl_display);
		emc_screen_name_alt[0] = 0;
	}

	ep.kind = ACCEPT_SIZE;
	ep.args[0] = 1;	/* yes */
	r = emc_send(&ep);
	if (r <= 0) {
		if (debug) perror("send request open");
		goto bad3;
	}
	chown(emc_screen_name,getuid(),getgid());
	chmod(emc_screen_name,0600);

	if (emc_screen_name_alt[0]) {
#ifndef __CYGWIN__
		chown(emc_screen_name_alt,getuid(),getgid());
		chmod(emc_screen_name_alt,0600);
		fd_alt = open(emc_screen_name_alt,O_RDWR);
		if (fd_alt < 0) goto bad3;
		v_alt = mmap(NULL,fbsize_alt,PROT_READ|PROT_WRITE,MAP_SHARED
			,fd_alt,0);
		if (v_alt == MAP_FAILED) {
			perror("mmap alt_screen");
			close(fd_alt);
			goto bad3;
		}
#else
		/* XXX
		   Win32 ζͭ API ȤWin9x Ǥʤȥ
		*/
		hFileMapping = CreateFileMapping((HANDLE) 0xffffffff,
						 NULL,
						 PAGE_READWRITE,
						 0,
						 fbsize_alt,
						 "mgl2_screen");
		if(hFileMapping == NULL) {
			perror("mmap alt_screen");
			goto bad3;
		}
		v_alt = (char *) MapViewOfFile(hFileMapping, 
					       FILE_MAP_ALL_ACCESS,
					       0,
					       0,
					       0);
#endif /* !__CYGWIN__ */
		mgl_screenattr_addr = v_alt;
	} else {
		fbsize += fbsize_alt;
	}
	if (send_image) {
	    v = malloc(fbsize - mgl_screen_offset);
	} else {
	    fd = open(emc_screen_name,O_RDWR);
	    if (fd < 0) {
		if (debug) perror("open screen");
		goto bad3;
	    }
#ifdef VIRTUAL_FRAME_ADDR
	    v = mmap((void *)VIRTUAL_FRAME_ADDR,fbsize
		,PROT_READ|PROT_WRITE,MAP_SHARED|MAP_FIXED
		,fd,mgl_screen_offset);
printf("cli:try to map %08x -> %08x\n",VIRTUAL_FRAME_ADDR,v);
#else
  #ifndef __CYGWIN__
	    v = mmap(NULL,fbsize,PROT_READ|PROT_WRITE,MAP_SHARED
		,fd,mgl_screen_offset);
  #else
	    hFileMapping = CreateFileMapping((HANDLE) 0xffffffff,
					     NULL,
					     PAGE_READWRITE,
					     0,
					     fbsize,
					     "mgl2_screen");
	    if (hFileMapping == NULL) {
		perror("mmap screen");
		goto bad4;
	    }
	    v = (char *) MapViewOfFile(hFileMapping,
				       FILE_MAP_ALL_ACCESS,
				       0,
				       0,
				       0);
	    v += mgl_screen_offset;	   
  #endif /* !__CYGWIN__ */
#endif
//printf("mmap addr = %08x fbsize = %d offset = %d\n",v,fbsize,mgl_screen_offset);
	    if (v == MAP_FAILED) {
		perror("mmap screen");
		goto bad4;
	    }
	}
	emc_screen_addr = v;
	if (!emc_screen_name_alt[0]) {
		mgl_screenattr_addr = v + fbsize - fbsize_alt;
	}
	emc_screen = _create_memscreen[STK_NATIVE](
		xs,ys,emc_screen_addr,0);

	create_physical_screen(STK_NATIVE,SCREEN_WIDTH,SCREEN_HEIGHT,NULL);
	rp_wrap(physical_screen);
	chdir(curdir);

	current_screen = physical_screen;
	emc_screen_fd = fd;
	emc_sock_fd = sd;

	mgl_em = &emc_em;

	vk_init();
	mgl_client = 1;
	return 0;

bad5:
	munmap(v,fbsize);
bad4:
	close(fd);
bad3:
	close(sd);
	emc_sock_fd = 0;
bad2:
	chdir(curdir);
bad1:
	return (-1);
}
