/**
 * @file libcomprex/file.c File structures
 * 
 * $Id: file.c,v 1.37 2003/01/01 06:22:35 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>
#include <libcomprex/file.h>

CxFile *
cxNewFile(void)
{
	CxFile *file;

	file = (CxFile *)cxNewFsNode();

	MEM_CHECK(file->u.file = (CxFileData *)malloc(sizeof(CxFileData)));
	memset(file->u.file, 0, sizeof(CxFileData));

	cxSetFsNodeType(file, CX_FSNODETYPE_FILE);

	return file;
}

void
cxDestroyFile(CxFile *file)
{
	CxArchive *archive;

	if (file == NULL || file->refCount == 0)
		return;

	if ((file->refCount - 1) > 0)
		return;

	if (file->u.file->physPath != NULL)
		free(file->u.file->physPath);

	archive = cxGetFileArchive(file);

	/* XXX This shouldn't be NULL, but check anyway. */
	if (archive != NULL && archive->module != NULL &&
		archive->module->ops.archive->destroyFile != NULL)
	{
		archive->module->ops.archive->destroyFile(file);
	}

	if (file->moduleData != NULL)
	{
		free(file->moduleData);
		file->moduleData = NULL;
	}

	if (archive != NULL)
	{
		if (cxGetFileParent(file) != NULL)
			cxDirRemoveFile(cxGetFileParent(file), file);

		cxDestroyArchive(archive);
	}

	free(file->u.file);

	cxDestroyFsNode(file);
}

void
cxSetFileArchive(CxFile *file, CxArchive *archive)
{
	cxSetFsNodeArchive(file, archive);
}

void
cxSetFileParent(CxFile *file, CxDirectory *parent)
{
	cxSetFsNodeParent(file, parent);
}

void
cxSetFileName(CxFile *file, const char *name)
{
	cxSetFsNodeName(file, name);
}

void
cxSetFilePhysicalPath(CxFile *file, const char *path)
{
	if (file == NULL)
		return;

	if (file->u.file->physPath != NULL)
		free(file->u.file->physPath);

	file->u.file->physPath = (path == NULL ? NULL : strdup(path));
}

void
cxSetFileMode(CxFile *file, mode_t mode)
{
	cxSetFsNodeMode(file, mode);
}

void
cxSetFileUid(CxFile *file, uid_t uid)
{
	cxSetFsNodeUid(file, uid);
}

void
cxSetFileGid(CxFile *file, gid_t gid)
{
	cxSetFsNodeGid(file, gid);
}

void
cxSetFileCompressedSize(CxFile *file, unsigned int size)
{
	if (file == NULL)
		return;

	file->u.file->compressedSize = size;
}

void
cxSetFileSize(CxFile *file, unsigned int size)
{
	if (file == NULL)
		return;

	file->u.file->uncompressedSize = size;
}

void
cxSetFileDate(CxFile *file, time_t date)
{
	cxSetFsNodeDate(file, date);
}

void
cxSetFileLocal(CxFile *file, char isLocal)
{
	cxSetFsNodeLocal(file, isLocal);
}

CxArchive *
cxGetFileArchive(CxFile *file)
{
	return cxGetFsNodeArchive(file);
}

CxDirectory *
cxGetFileParent(CxFile *file)
{
	return cxGetFsNodeParent(file);
}

const char *
cxGetFileName(CxFile *file)
{
	return cxGetFsNodeName(file);
}

const char *
cxGetFilePath(CxFile *file)
{
	return cxGetFsNodePath(file);
}

const char *
cxGetFilePhysicalPath(CxFile *file)
{
	if (file == NULL)
		return NULL;

	return file->u.file->physPath;
}

mode_t
cxGetFileMode(CxFile *file)
{
	return cxGetFsNodeMode(file);
}

uid_t
cxGetFileUid(CxFile *file)
{
	return cxGetFsNodeUid(file);
}

gid_t
cxGetFileGid(CxFile *file)
{
	return cxGetFsNodeGid(file);
}

unsigned int
cxGetFileCompressedSize(CxFile *file)
{
	if (file == NULL)
		return 0;

	return file->u.file->compressedSize;
}

unsigned int
cxGetFileSize(CxFile *file)
{
	if (file == NULL)
		return 0;

	return file->u.file->uncompressedSize;
}

time_t
cxGetFileDate(CxFile *file)
{
	return cxGetFsNodeDate(file);
}

char
cxIsFileLocal(CxFile *file)
{
	return cxIsFsNodeLocal(file);
}

CxFile *
cxGetNextFile(CxFile *file)
{
	if (file == NULL)
		return NULL;

	return file->next;
}

CxFile *
cxGetPreviousFile(CxFile *file)
{
	if (file == NULL)
		return NULL;

	return file->prev;
}

