/**
 * @file gnuext.h GNU Tar extensions
 * 
 * $Id: gnuext.h,v 1.2 2003/01/01 06:22:33 chipx86 Exp $
 *
 * @Copyright (C) 1999-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _TAR_GNUEXT_H_
#define _TAR_GNUEXT_H_

/*
 * From the tar 1.13's tar.h:
 * 
 * In GNU tar, SYMTYPE is for to symbolic links, and CONTTYPE is for
 * contiguous files, so maybe disobeying the `reserved' comment in POSIX
 * header description.  I suspect these were meant to be used this way, and
 * should not have really been `reserved' in the published standards.
 *
 * *BEWARE* *BEWARE* *BEWARE* that the following information is still
 * boiling, and may change.  Even if the OLDGNU format description should be
 * accurate, the so-called GNU format is not yet fully decided.  It is
 * surely meant to use only extensions allowed by POSIX, but the sketch
 * below repeats some ugliness from the OLDGNU format, which should rather
 * go away.  Sparse files should be saved in such a way that they do *not*
 * require two passes at archive creation time.  Huge files get some POSIX
 * fields to overflow, alternate solutions have to be sought for this.
 */

/**
 * Descriptor for a single file hole.
 */
typedef struct
{
	char offset[12];   /* Byte offset:  0 */
	char numbytes[12]; /*              12 */
	                   /*              24 */
} GnuSparse;

/*
 * Sparse files are not supported in POSIX ustar format.  For sparse files
 * with a POSIX header, a GNU extra header is provided which holds overall
 * sparse information and a few sparse descriptors.  When an old GNU header
 * replaces both the POSIX header and the GNU extra header, it holds some
 * sparse descriptors too.  Whether POSIX or not, if more sparse descriptors
 * are still needed, they are put into as many successive sparse headers as
 * necessary.  The following constants tell how many sparse descriptors fit
 * in each kind of header able to hold them.
 */
#define TAR_SPARSES_IN_OLDGNU_HEADER   4
#define TAR_SPARSES_IN_EXTRA_HEADER   16
#define TAR_SPARSES_IN_SPARSE_HEADER  21

/*
 * The GNU extra header.
 *
 * This contains some information GNU tar needs, but not foreseen in
 * POSIX header format.  It is only used after a POSIX header (and
 * never with old GNU headers), and immediately follows this POSIX
 * header, when typeflag is a letter rather than a digit, so signaling
 * a GNU extension.
 */
typedef struct
{
	char      atime[12];
	char      ctime[12];
	char      offset[12];
	char      realSize[12];
	char      longNames[4];
	char      padding[68];
	GnuSparse sp[TAR_SPARSES_IN_EXTRA_HEADER];
	char      isExtended;

} GnuHeader;


/**
 * Extension header for sparse files.
 *
 * Used immediately after the GNU extra header, and used only if all
 * sparse information cannot fit into that extra header.  There might
 * even be many such extension headers, one after the other, until all
 * sparse information has been recorded.
 */
typedef struct
{
	GnuSparse sp[TAR_SPARSES_IN_SPARSE_HEADER];
	char      isExtended;

} SparseHeader;


/**
 * Extension header for sparse files
 *
 * Uused immediately after the GNU extra header, and used only if all
 * sparse information cannot fit into that extra header.  There might
 * even be many such extension headers, one after the other, until all
 * sparse information has been recorded.
 */
typedef struct
{
	char      padding1[345];
	char      atime[12];
	char      ctime[12];
	char      offset[12];
	char      longNames[4];
	char      padding2;
	GnuSparse sp[TAR_SPARSES_IN_OLDGNU_HEADER];
	char      isExtended;
	char      realSize[12];

} GnuOldHeader;

/*
 * TAR_OLDGNU_MAGIC uses both magic and version fields, which are contiguous.
 * Found in an archive, it indicates an old GNU header format, which will be
 * hopefully become obsolescent.  With OLDGNU_MAGIC, uname and gname are
 * valid, though the header is not truly POSIX conforming.
 */
#define TAR_OLDGNU_MAGIC  "ustar  "  /* 7 chars and a null */

/*
 * The standards committee allows only capital A through capital Z for
 * user-defined expansion.
 */

typedef enum
{
	/**
	 * This is a dir entry that contains the names of files that
	 * were in the dir at the time the dump was made.
	 */
	GNUTYPE_DUMPDIR = 'D',

	/** Identifies the *next* file on the tape as having a long linkname. */
	GNUTYPE_LONGLINK = 'K',

	/** Identifies the *next* file on the tape as having a long name. */
	GNUTYPE_LONGNAME = 'L',

	/** This is the continuation of a file that began on another volume. */
	GNUTYPE_MULTIVOL = 'M',

	/** For storing filenames that do not fit into the main header. */
	GNUTYPE_NAMES = 'N',

	/** This is for sparse files. */
	GNUTYPE_SPARSE = 'S',

	/** This file is a tape/volume header.  Ignore it on extraction. */
	GNUTYPE_VOLHDR = 'V'

} GnuFileType;


#endif /* _TAR_GNUEXT_H_ */

