/*-
 * Copyright (c) 2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_time.c,v 1.3 2011/01/23 18:42:35 simon Exp $";
#endif /* !lint */


#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"

#include "ipa_time.h"

#include "ipa_log.h"
#include "ipa_main.h"

/*
 * This file contains different time manipulation functions.  I do not
 * use standard library time manipulation functions, they cannot be used
 * directly and should be wrapped to fix time zone and summer time related
 * errors, since ipa(8) does not need information about time zone and
 * summer time, I found that implementing own functions is better,
 * than writing wrappers for standard library time functions.
 *
 * May be for(;;) loops are not optimal, but for most cases
 * they are not executed and the ideas of algorithms are easy.
 */

unsigned int	wakeup_time;		/* wakeup_time parameter. */
unsigned int	freeze_time;		/* freeze_time parameter. */
unsigned int	sleep_after_dump;	/* sleep_after_dump parameter. */
unsigned int	sensitive_time;		/* sensitive_time parameter. */
signed char	debug_time;		/* debug_time parameter. */
signed char	debug_worktime;		/* debug_worktime parameter. */

unsigned int	worktimes_check_sec;	/* When to call worktimes_check(). */

const struct worktime *global_worktime;	/* global { worktime } */

struct worktime worktime_default;	/* 00:00-24:00 for all days. */
static struct tint_set tint_set_default;

const struct tevent *global_update_tevent; /* global { update_time } */
const struct tevent *global_append_tevent; /* global { append_time } */

const char *const wdays[] = {
	"Sunday", "Monday", "Tuesday", "Wednesday",
	"Thursday", "Friday", "Saturday"
};

const char *const active_msg[] = {
	"inactive",			/* 0 */
	"active"			/* 1 */
};

ipa_mzone	*tevent_mzone;		/* Mzone for all struct tevent{}. */
struct tevents_list tevents_list;	/* List of all tevents. */

ipa_mzone	*tint_mzone;		/* Mzone for all struct tint{}. */
struct tint_sets tint_sets;		/* List of all tint_sets. */

ipa_mzone	*worktime_mzone;	/* Mzone for all worktimes. */
struct worktimes_list worktimes_list;	/* List of all worktimes. */

time_t		curtime;		/* Current time. */
ipa_tm		curdate, curdate_new;	/* Current human localdate. */
unsigned int	cursec, cursec_new;	/* Current time in seconds. */
unsigned int	curwday;		/* Current week day. */
char		newday_flag;		/* Set if new day came. */

#ifdef WITH_ANY_LIMITS
/*
 * Non-reentrant functions for convert ipa_tm{} value to human readable
 * value and return a pointer to allocated buffer with this value.
 * If an error occurred, then a pointer to static string with error
 * message is returned.
 */
static const char *
tm_str_buf(const ipa_tm *tm, unsigned int i)
{
	static char buf[2][TM_STR_SIZE];

	if (snprintf(buf[i], TM_STR_SIZE, "%d.%02d.%02d/%02d:%02d:%02d",
	    tm->tm_year, tm->tm_mon, tm->tm_mday,
	    tm->tm_hour, tm->tm_min, tm->tm_sec) < 0)
		return ("(tm_str_buf: snprintf failed)");
	return (buf[i]);
}

const char *
tm_str(const ipa_tm *tm)
{
	return (tm_str_buf(tm, 0));
}

const char *
tm_str2(const ipa_tm *tm)
{
	return (tm_str_buf(tm, 1));
}

/*
 * Return last month day in the given year/mon.
 */
static int
last_mday(int year, int mon)
{
	static int const last_mday_arr[] =
	    /*   Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec */
	    {  0, 31,  0, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

	if (mon == 2) {
		/* February */
		return ((year % 4 == 0 &&
		    (year % 100 != 0 || year % 400 == 0)) ? 29 : 28);
	}
	return (last_mday_arr[mon]);
}

/*
 * Check ipa_tm date for correct values, ignore nonexistent
 * time (for example when time zone is changed as +1h, then some
 * time does not exist).
 */
int
check_ipa_tm(const ipa_tm *tm)
{
	const int mon = tm->tm_mon;
	const int mday = tm->tm_mday;

	if (mon == 0 || mon > MONTHES_IN_YEAR || mday > 31 ||
	    mday == 0 || mday > last_mday(tm->tm_year, mon) ||
	    tm->tm_hour > HOURS_IN_DAY - 1 ||
	    tm->tm_min > MINUTES_IN_HOUR - 1 ||
	    tm->tm_sec > SECONDS_IN_MINUTE - 1)
		return (-1);
	return (0);
}

/*
 * If time in tm is 24:00:00, then convert it to 00:00:00 of the next day.
 */
void
fix_240000(ipa_tm *tm)
{
	/* When tm_hour is 24, then tm_min and tm_sec == 0. */
	if (tm->tm_hour == HOURS_IN_DAY) {
		/* 24:00:00 --> next day 00:00:00 */
		tm->tm_hour = 0;
		if (tm->tm_mday != last_mday(tm->tm_year, tm->tm_mon))
			tm->tm_mday++;
		else {
			tm->tm_mday = 1;
			if (tm->tm_mon != MONTHES_IN_YEAR)
				tm->tm_mon++;
			else {
				tm->tm_mon = 1;
				tm->tm_year++;
			}
		}
	}
}

/*
 * Compare two ipa_tm tm1 and tm2 structures:
 * if (tm1 == tm2)
 *	return (0);
 * if (tm1 > tm2)
 *	return (1);
 * if (tm1 < tm2)
 *	return (-1);
 * This function correctly works with 24:00:00 time.
 */
int
cmp_ipa_tm(const ipa_tm *tm1, const ipa_tm *tm2)
{
	/*
	 * We do not use mktime(3), because there can be problem
	 * with time zone and summer time.
	 */
	if (tm1->tm_year > tm2->tm_year)
		return (1);
	if (tm1->tm_year == tm2->tm_year) {
		if (tm1->tm_mon > tm2->tm_mon)
			return (1);
		if (tm1->tm_mon == tm2->tm_mon) {
			if (tm1->tm_mday > tm2->tm_mday)
				return (1);
			if (tm1->tm_mday == tm2->tm_mday) {
				if (tm1->tm_hour > tm2->tm_hour)
					return (1);
				if (tm1->tm_hour == tm2->tm_hour) {
					if (tm1->tm_min > tm2->tm_min)
						return (1);
					if (tm1->tm_min == tm2->tm_min) {
						if (tm1->tm_sec > tm2->tm_sec)
							return (1);
						if (tm1->tm_sec == tm2->tm_sec)
							return (0);
					}
				}
			}
		}
	}
	return (-1);
}

/*
 * Return difference in seconds between tm1 and tm0: tm1 - tm0,
 * tm1 is expected to be greater than tm0.
 */
unsigned int
ipa_tm_diff(const ipa_tm *tm1_arg, const ipa_tm *tm0_arg)
{
	ipa_tm tm1, tm0;
	unsigned int result, sec1, sec0;

	tm1 = *tm1_arg;
	tm0 = *tm0_arg;

	sec1 = tm1.tm_hour * SECONDS_IN_HOUR +
	    tm1.tm_min * SECONDS_IN_MINUTE + tm1.tm_sec;
	sec0 = tm0.tm_hour * SECONDS_IN_HOUR +
	    tm0.tm_min * SECONDS_IN_MINUTE + tm0.tm_sec;

	if (sec1 >= sec0)
		result = sec1 - sec0;
	else {
		result = HOURS_IN_DAY * SECONDS_IN_HOUR - sec0;
		result += sec1;
		if (tm0.tm_mday != last_mday(tm0.tm_year, tm0.tm_mon))
			tm0.tm_mday++;
		else {
			tm0.tm_mday = 1;
			if (tm0.tm_mon != MONTHES_IN_YEAR)
				tm0.tm_mon++;
			else {
				tm0.tm_mon = 1;
				tm0.tm_year++;
			}
		}
	}

	for (; tm0.tm_year < tm1.tm_year; tm0.tm_year++) {
		for (; tm0.tm_mon <= MONTHES_IN_YEAR; tm0.tm_mon++) {
			result += (last_mday(tm0.tm_year, tm0.tm_mon) -
			    tm0.tm_mday + 1) * SECONDS_IN_DAY;
			tm0.tm_mday = 1;
		}
		tm0.tm_mon = 1;
	}

	for (; tm0.tm_mon < tm1.tm_mon; tm0.tm_mon++) {
		result += (last_mday(tm0.tm_year, tm0.tm_mon) -
		    tm0.tm_mday + 1) * SECONDS_IN_DAY;
		tm0.tm_mday = 1;
	}

	result += (tm1.tm_mday - tm0.tm_mday) * SECONDS_IN_DAY;
	return (result);
}
#endif /* WITH_ANY_LIMITS */

#ifdef WITH_LIMITS
/*
 * Add some seconds to ipa_tm variable ignoring any time zone issues
 * without using any standard time functions from the library.
 */
static void
ipa_tm_add(ipa_tm *tm, unsigned int addsec)
{
	int t, lmday, year, mon, mday;

	year = tm->tm_year;
	mon = tm->tm_mon;
	mday = tm->tm_mday;

	t = addsec / SECONDS_IN_DAY;
	addsec %= SECONDS_IN_DAY;
	if (t != 0) {
		for (;;) {
			lmday = last_mday(year, mon);
			if (mday + t > lmday) {
				t -= lmday - mday;
				if (mon != MONTHES_IN_YEAR)
					mon++;
				else {
					mon = 1;
					year++;
				}
				mday = 1;
				if (--t == 0)
					break;
			} else {
				mday += t;
				break;
			}
		}
	}
	if (addsec != 0) {
		t = tm->tm_hour * SECONDS_IN_HOUR +
		    tm->tm_min * SECONDS_IN_MINUTE + tm->tm_sec;
		if (t + addsec >= SECONDS_IN_DAY) {
			t += addsec - SECONDS_IN_DAY;
			if (mday != last_mday(year, mon))
				mday++;
			else {
				if (mon != MONTHES_IN_YEAR)
					mon++;
				else {
					mon = 1;
					year++;
				}
				mday = 1;
			}
		} else
			t += addsec;
		tm->tm_hour = t / SECONDS_IN_HOUR;
		t %= SECONDS_IN_HOUR;
		tm->tm_min = t / SECONDS_IN_MINUTE;
		tm->tm_sec = t % SECONDS_IN_MINUTE;
	}

	tm->tm_year = year;
	tm->tm_mon = mon;
	tm->tm_mday = mday;
}

/*
 * Convert given time in tm according to +upto.
 * tm->tm_wday field must have correct value for the given date.
 */
static void
ipa_tm_upto(ipa_tm *tm, char upto)
{
	if (upto == TEXP_UPTO_SIMPLE)
		return;

	/* +M or +m -> +h -> +D -> +W */

	if (upto == TEXP_UPTO_MONTH) {
		/* Up to the end of month. */
		if (tm->tm_mon != MONTHES_IN_YEAR)
			tm->tm_mon++;
		else {
			tm->tm_mon = 1;
			tm->tm_year++;
		}
		tm->tm_mday = 1;
		tm->tm_hour = tm->tm_min = tm->tm_sec = 0;
		return;
	}

	/* Up to the end of minute. */
	tm->tm_sec = 0;
	if (tm->tm_min == MINUTES_IN_HOUR - 1) {
		tm->tm_min = 0;
		tm->tm_hour++;
		fix_240000(tm);
	} else
		tm->tm_min++;
	if (upto == TEXP_UPTO_MINUTE)
		return;

	/* Up to the end of hour. */
	if (tm->tm_min != 0) {
		tm->tm_min = 0;
		tm->tm_hour++;
		fix_240000(tm);
	}
	if (upto == TEXP_UPTO_HOUR)
		return;

	/* Up to the end of day. */
	if (tm->tm_hour != 0) {
		tm->tm_hour = HOURS_IN_DAY;
		fix_240000(tm);
	}
	if (upto == TEXP_UPTO_DAY)
		return;

	/* Up to the end of week. */
	if (tm->tm_wday != 0)
		ipa_tm_add(tm, SECONDS_IN_DAY * (DAYS_IN_WEEK - tm->tm_wday));
}

/*
 * Convert given time in tm according to texp.
 */
void
ipa_tm_texp(ipa_tm *tm, const struct texp *texp)
{
	const char right_side = texp->side;

	if (!right_side)
		ipa_tm_upto(tm, texp->upto);
	if (texp->seconds != 0)
		ipa_tm_add(tm, texp->seconds);
	if (right_side)
		ipa_tm_upto(tm, texp->upto);
}

/*
 * Set tm_wday in the given structure.  mktime(3) ignores tm_wday
 * and tm_wday is set on success.  It is assumed, that 00:00:01 time
 * exists in all time zones.
 */
int
set_wday(ipa_tm *tmp)
{
	struct tm tm;

	tm = *tmp;
	tm.tm_year -= 1900;
	tm.tm_mon--;
	tm.tm_hour = tm.tm_min = 0;
	tm.tm_sec = 1;
	tm.tm_isdst = -1;
	if (mktime(&tm) == (time_t)-1) {
		logmsg(IPA_LOG_ERR, "set_wday: mktime failed");
		return (-1);
	}
	tmp->tm_wday = tm.tm_wday;
	return (0);
}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
/*
 * Subtract some seconds from ipa_tm variable ignoring any time zone issues
 * without using any standard time functions from the library.
 */
void
ipa_tm_sub(ipa_tm *tm, unsigned int subsec)
{
	int t, year, mon, mday;

	year = tm->tm_year;
	mon = tm->tm_mon;
	mday = tm->tm_mday;

	t = subsec / SECONDS_IN_DAY;
	subsec %= SECONDS_IN_DAY;
	if (t != 0) {
		for (;;) {
			if (mday > t) {
				mday -= t;
				break;
			} else {
				if (mon != 1)
					mon--;
				else {
					mon = MONTHES_IN_YEAR;
					year--;
				}
				t -= mday;
				mday = last_mday(year, mon);
				if (t == 0)
					break;
			}
		}
	}
	if (subsec != 0) {
		t = tm->tm_hour * SECONDS_IN_HOUR +
		    tm->tm_min * SECONDS_IN_MINUTE + tm->tm_sec;
		if (t >= subsec)
			t -= subsec;
		else {
			t = SECONDS_IN_DAY - (subsec - t);
			if (mday > 1)
				mday--;
			else {
				if (mon != 1)
					mon--;
				else {
					mon = MONTHES_IN_YEAR;
					year--;
				}
				mday = last_mday(year, mon);
			}
		}
		tm->tm_hour = t / SECONDS_IN_HOUR;
		t %= SECONDS_IN_HOUR;
		tm->tm_min = t / SECONDS_IN_MINUTE;
		tm->tm_sec = t % SECONDS_IN_MINUTE;
	}

	tm->tm_year = year;
	tm->tm_mon = mon;
	tm->tm_mday = mday;
}
#endif /* WITH_THRESHOLDS */

/*
 * Non-reentrant function for converting number of seconds to hours,
 * minutes and seconds and return a pointer to allocated buffer with
 * this value.  If an error occurred, then static string with error
 * message is returned.
 */
const char *
time_str(unsigned int t)
{
	static char buf[TIME_STR_SIZE];

	unsigned int h, m, s;
	int rv;

	h = t / SECONDS_IN_HOUR;
	t %= SECONDS_IN_HOUR;
	m = t / SECONDS_IN_MINUTE;
	s = t % SECONDS_IN_MINUTE;
	if (h == 0) {
		if (m == 0)
			rv = snprintf(buf, sizeof(buf), "%us", s);
		else if (s == 0)
			rv = snprintf(buf, sizeof(buf), "%um", m);
		else
			rv = snprintf(buf, sizeof(buf), "%um %02us", m, s);
	} else
		rv = snprintf(buf, sizeof(buf), "%uh %02um %02us", h, m, s);
	return (rv < 0 ? "(time_str: snprintf failed)" : buf);
}

/*
 * Non-reentrant function for converting number of seconds to time.
 */
const char *
sec_str(unsigned int sec)
{
	static char buf[SEC_STR_SIZE];

	unsigned int h, m, s;

	if (sec == EVENT_NOT_SCHEDULED)
		return ("xx:xx:xx");
	if (sec > SECONDS_IN_DAY)
		return ("??:??:??");

	h = sec / SECONDS_IN_HOUR;
	sec %= SECONDS_IN_HOUR;
	m = sec / SECONDS_IN_MINUTE;
	s = sec % SECONDS_IN_MINUTE;

	if (snprintf(buf, sizeof(buf), "%02u:%02u:%02u", h, m, s) < 0)
		return ("(sec_str: snprintf failed)");
	return (buf);
}

/*
 * Non-reentrant function for converting time interval to string.
 */
const char *
tint_str(const struct tint *tint)
{
	static char buf[sizeof("xx:xx-xx:xx")];

	unsigned int h1, m1, h2, m2;

	h1 = tint->sec1 / SECONDS_IN_HOUR;
	m1 = (tint->sec1 % SECONDS_IN_HOUR) / SECONDS_IN_MINUTE;
	h2 = tint->sec2 / SECONDS_IN_HOUR;
	m2 = (tint->sec2 % SECONDS_IN_HOUR) / SECONDS_IN_MINUTE;

	if (snprintf(buf, sizeof(buf), "%02u:%02u-%02u:%02u",
	    h1, m1, h2, m2) < 0)
		return ("(tint_str: snprintf failed)");
	return (buf);
}

/*
 * Convert seconds since midnight to hours, minutes and seconds.
 */
void
sec_to_time(unsigned int sec, ipa_tm *tm)
{
	int h, m, s;

	h = sec / SECONDS_IN_HOUR;
	sec %= SECONDS_IN_HOUR;
	m = sec / SECONDS_IN_MINUTE;
	s = sec % SECONDS_IN_MINUTE;

	tm->tm_hour = h;
	tm->tm_min = m;
	tm->tm_sec = s;
}

void
init_worktime_default(void)
{
	static struct tint tint;

	unsigned int wday;

	tint.sec1 = 0;
	tint.sec2 = SECONDS_IN_DAY;

	STAILQ_INIT(&tint_set_default.list);
	STAILQ_INSERT_HEAD(&tint_set_default.list, &tint, link);

	STAILQ_INSERT_HEAD(&tint_sets, &tint_set_default, link);

	for (wday = 0; wday < DAYS_IN_WEEK; ++wday)
		worktime_default.tint_list[wday] = &tint_set_default.list;

	/* See comment in parse_generic_worktime(). */
	worktime_default.wt_flags = WT_FLAG_ACTIVE;
}

/*
 * Release memory help by tint_set, including struct tint_set{}.
 */
void
free_tint_set(struct tint_set *set)
{
	struct tint *tint, *tint_next;

	STAILQ_FOREACH_SAFE(tint, &set->list, link, tint_next)
		mzone_free(tint_mzone, tint);
	mem_free(set, m_anon);
}

/*
 * Free all worktimes and worktime mzone.
 */
void
free_worktimes(void)
{
	struct tint_set *set, *set_next;

	/* Skip first tint, since it is tint_set_default. */
	STAILQ_REMOVE_HEAD(&tint_sets, link);

	/* Free other tint_sets. */
	STAILQ_FOREACH_SAFE(set, &tint_sets, link, set_next)
		free_tint_set(set);

	mzone_deinit(tint_mzone);
	mzone_deinit(worktime_mzone);
}

/*
 * Find next time interval for worktime starting from tint_start.
 */
static void
find_next_tint(struct worktime *wt, const struct tint *tint_start,
    int report_skip)
{
	const struct tint *tint;

	for (tint = tint_start; tint != NULL; tint = STAILQ_NEXT(tint, link)) {
		if (cursec < tint->sec1) {
			/* x [   ] */
			WT_SET_INACTIVE(wt);
			if (worktimes_check_sec > tint->sec1)
				worktimes_check_sec = tint->sec1;
			if (debug_worktime)
				logdbg("find_next_tint: %s left before start "
				    "of %s %s worktime interval",
				    time_str(tint->sec1 - cursec),
				    wdays[curwday], tint_str(tint));
			break;
		}
		if (tint->sec1 <= cursec && cursec < tint->sec2) {
			/* [ x ] */
			WT_SET_ACTIVE(wt);
			if (worktimes_check_sec > tint->sec2)
				worktimes_check_sec = tint->sec2;
			if (debug_worktime)
				logdbg("find_next_tint: %s left before end "
				    "of %s %s worktime interval",
				    time_str(tint->sec2 - cursec),
				    wdays[curwday], tint_str(tint));
			break;
		}
		/* [   ] x */
		if (report_skip)
			logmsgx(IPA_LOG_WARNING, "find_next_tint: skipping "
			    "%s %s worktime interval", wdays[curwday],
			    tint_str(tint));
		else if (debug_worktime)
			logdbg("find_next_tint: skipping %s %s worktime "
			    "interval", wdays[curwday], tint_str(tint));
	}
	if (tint == NULL) {
		/* There is no more time intervals for current week day. */
		WT_SET_INACTIVE(wt);
		wt->active_sec = wt->inactive_sec = EVENT_NOT_SCHEDULED;
		if (debug_worktime)
			logdbg("find_next_tint: there is no more worktime "
			    "intervals for %s (current week day)",
			    wdays[curwday]);
	} else {
		wt->active_sec = tint->sec1;
		wt->inactive_sec = tint->sec2;
	}

	/* Real tint or NULL for debugging purpose. */
	wt->curtint = tint;
}

/*
 * Check if current week day is set in worktime, find current
 * active or inactive time interval.
 */
void
worktimes_newday(int report_skip)
{
	const struct tint_list *list;
	struct worktime *wt;

	worktimes_check_sec = EVENT_NOT_SCHEDULED;

	SLIST_FOREACH(wt, &worktimes_list, link) {
		list = wt->tint_list[curwday];
		if (!STAILQ_EMPTY(list)) {
			/* Current week day is set in worktime. */
			find_next_tint(wt, STAILQ_FIRST(list), report_skip);
			continue;
		} else {
			/* Current week day is not set in worktime. */
			WT_SET_INACTIVE(wt);
			/* Set NULL for debugging purpose. */
			wt->curtint = NULL;
			wt->active_sec = wt->inactive_sec = EVENT_NOT_SCHEDULED;
			if (debug_worktime)
				logdbg("worktimes_newday: %s (current week "
				    "day) is not set in worktime",
				    wdays[curwday]);
		}
	}

	if (debug_worktime)
		logdbg("worktimes_newday: worktimes_check_sec %s",
		    sec_str(worktimes_check_sec));
}

static unsigned int
worktime_check(struct worktime *wt)
{
	unsigned int delta;

	if (WT_IS_ACTIVE(wt)) {
		/* Is active. */
		if (wt->inactive_sec > cursec)
			return (wt->inactive_sec);
		/* It's time to make it inactive. */
		delta = cursec - wt->inactive_sec;
		find_next_tint(wt, STAILQ_NEXT(wt->curtint, link), 1);
	} else {
		/* Is inactive. */
		if (wt->active_sec > cursec)
			return (wt->active_sec);
		/* It's time to make it active. */
		delta = cursec - wt->active_sec;
		find_next_tint(wt, wt->curtint, 1);
	}

	if (delta > sensitive_time)
		logmsgx(IPA_LOG_WARNING, "worktime_check: worktime interval "
		    "%s %s became %s too late: delta %s is greater than "
		    "sensitive_time %us", wdays[curwday],
		    WT_IS_ACTIVE(wt) ? "inactive" : "active",
		    tint_str(wt->curtint), time_str(delta), sensitive_time);

	return (EVENT_NOT_SCHEDULED);
}

/*
 * Check if current time interval is still active, if it is
 * time to make inactive time interval active.
 */
void
worktimes_check(void)
{
	struct worktime *wt;
	unsigned int rv;

	worktimes_check_sec = EVENT_NOT_SCHEDULED;

	SLIST_FOREACH(wt, &worktimes_list, link) {
		rv = worktime_check(wt);
		if (worktimes_check_sec > rv)
			worktimes_check_sec = rv;
	}

	if (debug_worktime)
		logdbg("worktimes_checks: worktimes_check_sec %s",
		    sec_str(worktimes_check_sec));
}

/*
 * Try to find already registered worktime with the same settings as
 * in wt1.  If such worktime exist, then return its pointer, else
 * return original one.
 */
const struct worktime *
find_worktime(struct worktime *wt1)
{
	const struct tint_list * const *list1;
	const struct tint_list * const *list2;
	struct worktime	*wt2;
	unsigned int wday;

	/* Check if there is already the same worktime. */
	SLIST_FOREACH(wt2, &worktimes_list, link) {
		list1 = wt1->tint_list;
		list2 = wt2->tint_list;
		for (wday = 0; wday < DAYS_IN_WEEK; ++wday) {
			if (*list1 != *list2)
				break;
			++list1;
			++list2;
		}
		if (wday == DAYS_IN_WEEK) {
			/* The same worktime was found --> free original one. */
			mzone_free(worktime_mzone, wt1);
			return (wt2);
		}
	}

	/* New worktime, add it to the list. */
	SLIST_INSERT_HEAD(&worktimes_list, wt1, link);
	return (wt1);
}

#ifdef WITH_ANY_LIMITS
/*
 * Return 0 if wt2 is a subset of wt1, else return -1.
 * Time intervals in limits' or thresholds' worktimes must be
 * subsets of time intervals in their rule's worktime.
 */
int
check_worktime_subset(const struct worktime *wt1, const struct worktime *wt2)
{
	const struct tint *tint1, *tint2;
	const struct tint_list * const *list1;
	const struct tint_list * const *list2;
	unsigned int wday;

	if (wt1 == NULL || wt2 == NULL || wt1 == wt2)
		return (0);

	list1 = wt1->tint_list;
	list2 = wt2->tint_list;

	for (wday = 0; wday < DAYS_IN_WEEK; ++list1, ++list2, ++wday) {
		if (*list1 == *list2)
			/* The same tint_list. */
			continue;

		if (STAILQ_EMPTY(*list2))
			/* This day in wt2 is not for accounting. */
			continue;

		if (STAILQ_EMPTY(*list1))
			/*
			 * This day in wt1 is not for accounting, but this
			 * day in wt2 is for accounting.
			 */
			return (-1);

		/* [ ] -- intervals in tint1, { } -- intervals in tint1. */
		tint2 = STAILQ_FIRST(*list2);
		STAILQ_FOREACH(tint1, *list1, link)
			for (;tint2 != NULL; tint2 = STAILQ_NEXT(tint2, link)) {
				if (tint2->sec1 >= tint1->sec1 &&
				    tint2->sec2 <= tint1->sec2)
					/* [ {  } ] */
					continue;
				if (tint2->sec1 > tint1->sec2)
					/* [   ] {   } */
					break;
				/* {   } [   ] or {   }[   ] */
				return (-1);
			}
		if (tint2 != NULL)
			/* [   ] {   } */
			return (-1);
	}

	return (0);
}
#endif /* WITH_ANY_LIMITS */

/*
 * Initialize time related values.
 */
int
init_time_values(void)
{
	ipa_tm tm;
	time_t t;

	if (time(&t) == (time_t)-1) {
		logmsg(IPA_LOG_ERR, "init_time_values: time failed");
		return (-1);
	}
	if (localtime_r(&t, (struct tm *)&tm) == NULL) {
		logmsg(IPA_LOG_ERR, "init_time_values: localtime_r failed");
		return (-1);
	}
	tm.tm_year += 1900;
	tm.tm_mon++;
	cursec = TIME_TO_SEC(&tm);
	curdate = tm;
	curwday = tm.tm_wday;
	curtime = t;
	newday_flag = 0;
	return (0);
}

/*
 * Get new time related values and verify that everything is correct
 * with time and local date.
 */
int
new_time_values(void)
{
	ipa_tm tm;
	double d;
	time_t t;
	int rv;

	/* Get new current time. */
	if (time(&t) == (time_t)-1) {
		logmsg(IPA_LOG_ERR, "new_time_values: time failed");
		return (-1);
	}
	if (localtime_r(&t, (struct tm *)&tm) == NULL) {
		logmsg(IPA_LOG_ERR, "new_time_values: localtime_r failed");
		return (-1);
	}

	tm.tm_year += 1900;
	tm.tm_mon++;
	rv = 0;

	/* Check whether new time is less than old time. */
	d = difftime(t, curtime);
	if (d < 0.0) {
		logmsgx(IPA_LOG_WARNING, "time goes back: delta %.0fs", d);
		rv = 1;
	}

	/*
	 * Check whether time changes too quickly, note that ipa
	 * does not sleep more than 1 day.
	 */
	if (d > SECONDS_IN_DAY) {
		logmsgx(IPA_LOG_WARNING, "time changes too quickly, worked "
		    "or slept too much: delta %.0fs is greater than 1 day", d);
		rv = 1;
	}

#ifdef HAVE_TM_GMTOFF
	/* Check whether offset from UTC was changed. */
	if (tm.tm_gmtoff != curdate.tm_gmtoff) {
		logmsgx(IPA_LOG_WARNING, "offset from UTC changed from "
		    "%ld to %ld seconds", (long)curdate.tm_gmtoff,
		    (long)tm.tm_gmtoff);
		rv = 1;
	}
#endif

	/* Check whether Daylight Saving flag was changed. */
	if (tm.tm_isdst != curdate.tm_isdst) {
		logmsgx(IPA_LOG_WARNING, "Daylight Saving flag changed "
		    "from %d to %d", curdate.tm_isdst, tm.tm_isdst);
		rv = 1;
	}

	if (rv != 0)
		logmsgx(IPA_LOG_WARNING, "some time related problem occurred");

	curtime = t;
	curdate_new = tm;
	cursec_new = TIME_TO_SEC(&tm);

	return (rv);
}
