/*-
 * Copyright (c) 2004, 2009 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_ctl.c,v 1.4.2.3 2012/07/12 20:25:57 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/uio.h>
#include <sys/un.h>

#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <limits.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"
#include "pathnames.h"

#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_time.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

#ifndef PF_LOCAL
# ifdef PF_UNIX
#  define PF_LOCAL PF_UNIX
# else
#  ifdef AF_LOCAL
#   define PF_LOCAL AF_LOCAL
#  else
#   ifdef AF_UNIX
#    define PF_LOCAL AF_UNIX
#   else
#    error "cannot define PF_LOCAL"
#   endif
#  endif
# endif
#endif

#ifndef SUN_LEN
# define SUN_LEN(su) (sizeof(struct sockaddr_un))
#endif

#define CTL_BACKLOG	5		/* An argument for listen(). */

/* Ctl related parameters. */
char		*ctl_socket_path;
mode_t		ctl_socket_perm;
size_t		ctl_query_max_size;
unsigned int	ctl_timeout;
signed char	ctl_enable;
signed char	debug_ctl;

char		*ctl_socket_path_default = IPA_CTL_SOCKET;

#ifdef WITH_RULES
ipa_mzone	*ictl_mzone;		/* Mzone for all struct ictl{}. */
#endif

ipa_mem_type	*m_ctl;			/* Memory allocations for ctl. */

#ifdef CTL_CHECK_CREDS

# if !defined(__FreeBSD__) && !defined(__NetBSD__)
#  error "--enable-ctl-creds can be used only on FreeBSD or NetBSD"
# endif

/* Ctl ACL related parameters. */
const struct ctl_acl_class *ctl_dump_acl;
const struct ctl_acl_class *ctl_freeze_acl;
const struct ctl_acl_class *ctl_stat_acl;
const struct ctl_acl_class *global_ctl_rule_acl;

struct ctl_acl_classes ctl_acl_classes;	/* List of all ctl ACL classes. */
ipa_mzone	*ctl_acl_elem_mzone;	/* Mzone for all ctl ACL elements. */
ipa_mzone	*ctl_acl_class_mzone;	/* Mzone for all ctl ACL classes. */

# ifdef __FreeBSD__
static union {
	struct cmsghdr	cmsghdr;
	char		control[CMSG_SPACE(sizeof(struct cmsgcred))];
} query_cmsg;
# endif

# ifdef WITH_PTHREAD

#  ifndef GETPW_R_BUFSIZE
#   define GETPW_R_BUFSIZE 1024
#  endif
#  ifndef GETGR_R_BUFSIZE
#   define GETGR_R_BUFSIZE 1024
#  endif

static char	*pwd_buf;
static char	*grp_buf;
static size_t	pwd_buf_size;
static size_t	grp_buf_size;

# endif /* WITH_PTHREAD */

#endif /* CTL_CHECK_CREDS */

int		ctl_listenfd = -1;	/* Listen socket descriptor. */
static int	ctl_connfd = -1;	/* Connected socket descriptor. */
static char	ctl_socket_bound;	/* Set if socket file was bound. */

static struct ctl_cmdq cmdq;		/* Control command query. */
static struct ctl_cmdq_val cmdq_val;	/* Two values. */
static void	*cmdq_aux = NULL;	/* Auxiliary query. */

static struct ctl_cmda cmda;		/* Control command answer. */
static void	*cmda_aux;		/* Auxiliary answer. */
static size_t	cmda_aux_size;		/* Auxiliary answer size. */
static char	cmda_aux_allocated = 0;	/* Set if cmda_aux allocated. */

static union {
	struct ctl_cmda_all_status all;
	struct ctl_cmda_rule_status rule;
#ifdef WITH_AUTORULES
	struct ctl_cmda_autorule_status autorule;
#endif
#ifdef WITH_LIMITS
	struct ctl_cmda_limit_status limit;
#endif
#ifdef WITH_THRESHOLDS
	struct ctl_cmda_threshold_status threshold;
#endif
} cmda_status;

#ifdef WITH_ANY_LIMITS
static struct ctl_cmda_set cmda_set;
#endif

static struct msghdr query_msghdr;	/* Message with command query. */
static struct iovec query_iov[1];	/* iovec with query data. */

static struct iovec answer_iov[2];	/* iovec with answer data. */

static struct rule *q_rule;		/* Pointer to rule in query. */

#ifdef WITH_AUTORULES
static const char *q_rule_name;		/* Name of rule in query. */
static struct autorule *q_autorule;	/* Pointer to autorule in query. */
#endif

#ifdef WITH_LIMITS
static struct limit *q_limit;		/* Pointer to limit in query. */
#endif

#ifdef WITH_THRESHOLDS
static struct threshold *q_threshold;	/* Pointer to threshold in query. */
#endif

static const char * const cmdq_msg[] = {
	"status",		/* 0 | CTL_CMD_STATUS	*/
	"memory",		/* 1 | CTL_CMD_MEMORY	*/
	"dump",			/* 2 | CTL_CMD_DUMP	*/
	"freeze",		/* 3 | CTL_CMD_FREEZE	*/
	"restart",		/* 4 | CTL_CMD_RESTART	*/
	"expire",		/* 5 | CTL_CMD_EXPIRE	*/
	"set",			/* 6 | CTL_CMD_SET	*/
	"create",		/* 7 | CTL_CMD_CREATE	*/
	"delete"		/* 8 | CTL_CMD_DELETE	*/
};

/*
 * Flags map for control command query.
 */
static const struct {
	unsigned int	fl_req;		/* Flags that are required. */
	unsigned int	fl_abs;		/* Flags that must be absent. */
} cmdq_flags[] = {
	{ /* CTL_CMD_STATUS */
		0,
		CTL_CFLAG_VALUE1|CTL_CFLAG_VALUE1_INC|CTL_CFLAG_VALUE1_DEC|
		CTL_CFLAG_VALUE2|CTL_CFLAG_VALUE2_INC|CTL_CFLAG_VALUE2_DEC
	},
	{ /* CTL_CMD_MEMORY */
		0,
		CTL_ALL_CFLAGS & ~CTL_CFLAG_WAIT
	},
	{ /* CTL_CMD_DUMP */
		0,
		CTL_ALL_CFLAGS & ~CTL_CFLAG_WAIT
	},
	{ /* CTL_CMD_FREEZE */
		0,
		CTL_ALL_CFLAGS & ~CTL_CFLAG_WAIT
	},
	{ /* CTL_CMD_RESTART */
		CTL_CFLAG_RULE|CTL_CFLAG_LIMIT,
		CTL_CFLAG_AUTORULE|CTL_CFLAG_THRESHOLD|
		CTL_CFLAG_VALUE1|CTL_CFLAG_VALUE1_INC|CTL_CFLAG_VALUE1_DEC|
		CTL_CFLAG_VALUE2|CTL_CFLAG_VALUE2_INC|CTL_CFLAG_VALUE2_DEC
	},
	{ /* CTL_CMD_EXPIRE */
		CTL_CFLAG_RULE|CTL_CFLAG_LIMIT,
		CTL_CFLAG_AUTORULE|CTL_CFLAG_THRESHOLD|
		CTL_CFLAG_VALUE1|CTL_CFLAG_VALUE1_INC|CTL_CFLAG_VALUE1_DEC|
		CTL_CFLAG_VALUE2|CTL_CFLAG_VALUE2_INC|CTL_CFLAG_VALUE2_DEC
	},
	{ /* CTL_CMD_SET */
		CTL_CFLAG_RULE,
		CTL_CFLAG_AUTORULE
	},
	{ /* CTL_CMD_CREATE */
		CTL_CFLAG_AUTORULE,
		CTL_CFLAG_RULE|CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD|
		CTL_CFLAG_VALUE1|CTL_CFLAG_VALUE1_INC|CTL_CFLAG_VALUE1_DEC|
		CTL_CFLAG_VALUE2|CTL_CFLAG_VALUE2_INC|CTL_CFLAG_VALUE2_DEC
	},
	{ /* CTL_CMD_DELETE */
		CTL_CFLAG_RULE,
		CTL_CFLAG_AUTORULE|CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD|
		CTL_CFLAG_VALUE1|CTL_CFLAG_VALUE1_INC|CTL_CFLAG_VALUE1_DEC|
		CTL_CFLAG_VALUE2|CTL_CFLAG_VALUE2_INC|CTL_CFLAG_VALUE2_DEC
	}
};

/*
 * Several similar make_*_active() functions: if worktime is inactive,
 * then set rule, limit or threshold active or inactive, depending
 * on the active argument.
 */
static int
make_rule_active(struct rule *rule, int active)
{
	if (WT_IS_INACTIVE(rule->worktime))
		if (mod_set_rule_active(rule, active) < 0) {
			logbt("make_rule_active");
			return (-1);
		}
	return (0);
}

#ifdef WITH_LIMITS
static int
make_limit_active(const struct rule *rule, struct limit *limit, int active)
{
	if (WT_IS_INACTIVE(limit->worktime))
		if (mod_set_limit_active(rule, limit, active) < 0) {
			logbt("make_limit_active");
			return (-1);
		}
	return (0);
}
#endif

#ifdef WITH_THRESHOLDS
static int
make_threshold_active(const struct rule *rule, struct threshold *threshold,
    int active)
{
	if (WT_IS_INACTIVE(threshold->worktime))
		if (mod_set_threshold_active(rule, threshold, active) < 0) {
			logbt("make_threshold_active");
			return (-1);
		}
	return (0);
}
#endif

/*
 * Create Unix domain socket and update relevant variables.
 */
int
init_ctl(void)
{
	struct sockaddr_un servaddr;
	struct timeval tv;
	mode_t oumask;
	int fd, rv;
#ifdef CTL_CHECK_CREDS
# if defined(WITH_PTHREAD) && \
    (defined(_SC_GETPW_R_SIZE_MAX) || defined(_SC_GETGR_R_SIZE_MAX))
	long size;
# endif
# ifdef __NetBSD__
	int optval;
# endif
#endif

#ifdef CTL_CHECK_CREDS
# ifdef WITH_PTHREAD
	pwd_buf = grp_buf = NULL;
#endif
# ifdef __FreeBSD__
	query_msghdr.msg_control = query_cmsg.control;
	query_msghdr.msg_controllen = sizeof(query_cmsg.control);
# endif
# ifdef __NetBSD__
	/* CMSG_SPACE() uses function on NetBSD 2.0. */
	query_msghdr.msg_control = mem_calloc(1,
	    CMSG_SPACE(SOCKCREDSIZE(NGROUPS_MAX)), m_ctl);
	if (query_msghdr.msg_control == NULL) {
		logmsgx(IPA_LOG_ERR, "init_ctl: mem_calloc failed");
		return (-1);
	}
	query_msghdr.msg_controllen = CMSG_SPACE(SOCKCREDSIZE(NGROUPS_MAX));
# endif
# ifdef WITH_PTHREAD
#  ifdef _SC_GETPW_R_SIZE_MAX
	size = sysconf(_SC_GETPW_R_SIZE_MAX);
	if (size < 0)
		pwd_buf_size = GETPW_R_BUFSIZE;
	else
		pwd_buf_size = size;
#  else
	pwd_buf_size = GETPW_R_BUFSIZE;
#  endif
	pwd_buf = mem_malloc(pwd_buf_size, m_ctl);
	if (pwd_buf == NULL) {
		logmsgx(IPA_LOG_ERR, "init_ctl: mem_malloc failed");
		return (-1);
	}
#  ifdef _SC_GETGR_R_SIZE_MAX
	size = sysconf(_SC_GETGR_R_SIZE_MAX);
	if (size < 0)
		grp_buf_size = GETPW_R_BUFSIZE;
	else
		grp_buf_size = size;
#  else
	grp_buf_size = GETGR_R_BUFSIZE;
#  endif
	grp_buf = mem_malloc(grp_buf_size, m_ctl);
	if (grp_buf == NULL) {
		logmsgx(IPA_LOG_ERR, "init_ctl: mem_malloc failed");
		return (-1);
	}
# endif /* WITH_PTHREAD */
#else
	query_msghdr.msg_control = NULL;
	query_msghdr.msg_controllen = 0;
#endif /* CTL_CHECK_CREDS */

	logmsgx(IPA_LOG_INFO, "creating control socket %s", ctl_socket_path);

	fd = ctl_listenfd = socket(PF_LOCAL, SOCK_STREAM, 0);
	if (fd < 0) {
		logmsg(IPA_LOG_ERR, "init_ctl: socket");
		return (-1);
	}

	memset(&servaddr, 0, sizeof(servaddr));
	servaddr.sun_family = PF_LOCAL;
	strncpy(servaddr.sun_path, ctl_socket_path,
	    sizeof(servaddr.sun_path) - 1);

	oumask = umask(S_IWUSR|S_IXUSR|S_IRWXG|S_IRWXO);
	rv = bind(fd, (struct sockaddr *)&servaddr, SUN_LEN(&servaddr));
	(void)umask(oumask);
	if (rv < 0) {
		logmsg(IPA_LOG_ERR, "init_ctl: bind(%s)", ctl_socket_path);
		ctl_socket_bound = 0;
		return (-1);
	}
	ctl_socket_bound = 1;

	if (chmod(ctl_socket_path, ctl_socket_perm) < 0) {
		logmsg(IPA_LOG_ERR, "init_ctl: chmod(%s, 0%03o)",
		    ctl_socket_path, (unsigned int)ctl_socket_perm);
		return (-1);
	}
	if (chown(ctl_socket_path, myuid, mygid) < 0) {
		logmsg(IPA_LOG_ERR, "init_ctl: chown(%s, %lu, %lu)",
		    ctl_socket_path, (unsigned long)myuid,
		    (unsigned long)mygid);
		return (-1);
	}
	if (fd_set_nonblock(fd) < 0) {
		logmsgx(IPA_LOG_ERR, "init_ctl: cannot mark "
		    "listen socket as non-blocking");
		return (-1);
	}

	tv.tv_sec = ctl_timeout;
	tv.tv_usec = 0;
	if (setsockopt(fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv)) < 0 ||
	    setsockopt(fd, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv)) < 0) {
		logmsg(IPA_LOG_ERR, "init_ctl: "
		    "sesockopt(SO_RCVTIMEO/SO_SNDTIMEO");
		return (-1);
	}

#ifdef CTL_CHECK_CREDS
# ifdef __NetBSD__
	optval = 1;
	if (setsockopt(fd, 0, LOCAL_CREDS, &optval, sizeof(optval)) < 0) {
		logmsg(IPA_LOG_ERR, "init_ctl: setsockopt(LOCAL_CREDS)");
		return (-1);
	}
# endif
#endif /* CTL_CHECK_CREDS */

	/*
	 * Even if ctl_socket_path was created by bind() with incorrect,
	 * permission bits, all clients will get "Connection refused",
	 * because socket is not in SO_ACCEPTCONN state.
	 */
	if (listen(fd, CTL_BACKLOG) < 0) {
		logmsg(IPA_LOG_ERR, "init_ctl: listen(%s, %d)",
		    ctl_socket_path, CTL_BACKLOG);
		return (-1);
	}

	/* Initialize all fields of query_msghdr. */
	query_iov[0].iov_base = (char *)&cmdq;
	query_iov[0].iov_len = sizeof(cmdq);
	query_msghdr.msg_name = NULL;
	query_msghdr.msg_namelen = 0;
	query_msghdr.msg_iov = query_iov;
	query_msghdr.msg_iovlen = 1;

	/* Initialize answer_iov. */
	answer_iov[0].iov_base = (char *)&cmda;
	answer_iov[0].iov_len = sizeof(cmda);
	return (0);
}

#ifdef CTL_CHECK_CREDS
/*
 * Release memory used by all ACLs.
 */
static int
free_ctl_acls(void)
{
	const struct ctl_acl_class *class;
	struct ctl_acl_elem *elem;

	STAILQ_FOREACH(class, &ctl_acl_classes, link) {
		STAILQ_FOREACH(elem, &class->list, link) {
			mem_free(elem->user, m_ctl);
			mem_free(elem->group, m_ctl);
		}
		mem_free(class->class_name, m_ctl);
	}

	mzone_deinit(ctl_acl_elem_mzone);
	mzone_deinit(ctl_acl_class_mzone);

	return (0);
}
#endif /* CTL_CHECK_CREDS */

/*
 * Close listening Unix domain socket and unlink it.
 */
int
deinit_ctl(void)
{
	int rv;

	rv = 0;
	if (ctl_listenfd >= 0) {
		if (close(ctl_listenfd) < 0) {
			logmsg(IPA_LOG_ERR, "deinit_ctl: close");
			rv = -1;
		}
		if (ctl_socket_bound) {
			logmsgx(IPA_LOG_INFO, "unlinking control socket %s",
			    ctl_socket_path);
			if (unlink(ctl_socket_path) < 0) {
				logmsg(IPA_LOG_ERR, "deinit_ctl: unlink(%s)",
				    ctl_socket_path);
				rv = -1;
			}
		}
		ctl_listenfd = -1;
	}
	if (ctl_connfd >= 0) {
		if (close(ctl_connfd) < 0) {
			logmsg(IPA_LOG_ERR, "deinit_ctl: close");
			if (errno == EBADF)
				return (-1);
		}
		ctl_connfd = -1;
	}

	if (cmda_aux_allocated) {
		cmda_aux_allocated = 0;
		mem_free(cmda_aux, m_ctl);
	}

	if (ctl_socket_path != ctl_socket_path_default)
		mem_free(ctl_socket_path, m_parser);

#ifdef CTL_CHECK_CREDS
	free_ctl_acls();
# ifdef __NetBSD__
	mem_free(query_msghdr.msg_control, m_ctl);
# endif
# ifdef WITH_PTHREAD
	mem_free(pwd_buf, m_ctl);
	mem_free(grp_buf, m_ctl);
# endif
#endif /* CTL_CHECK_CREDS */

	return (rv);
}

/*
 * "status"
 */
static int
ctl_func_status(void)
{
#ifdef WITH_LIMITS
	const struct limit *limit;
#endif
#ifdef WITH_SUBLIMITS
	const struct sublimit *sublimit;
#endif
#ifdef WITH_THRESHOLDS
	const struct threshold *threshold;
#endif
	unsigned int qflags;

	qflags = cmdq.flags;
	if (!(qflags & CTL_CFLAG_WAIT))
		return (0);

#ifdef WITH_AUTORULES
# define status cmda_status.autorule
	if (qflags & CTL_CFLAG_AUTORULE) {
		cmda_aux = &status;
		cmda_aux_size = sizeof(status);
		status.nrules = q_autorule->nrules;
		status.active = AUTORULE_IS_ACTIVE(q_autorule);
	} else
# undef status
#endif
#define status cmda_status.rule
	if ((qflags & (CTL_CFLAG_RULE|CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD)) ==
	    CTL_CFLAG_RULE) {
		cmda_aux = &status;
		cmda_aux_size = sizeof(status);
		status.nlimits = status.nsublimits = status.nthresholds = 0;
#ifdef WITH_LIMITS
		STAILQ_FOREACH(limit, &q_rule->limits, link) {
			status.nlimits++;
# ifdef WITH_SUBLIMITS
			STAILQ_FOREACH(sublimit, &limit->sublimits, link)
				status.nsublimits++;
# endif
		}
#endif
#ifdef WITH_THRESHOLDS
		STAILQ_FOREACH(threshold, &q_rule->thresholds, link)
			status.nthresholds++;
#endif
		status.dynamic = RULE_IS_DYNAMIC(q_rule);
		status.active = RULE_IS_ACTIVE(q_rule);
	} else
#undef status
#ifdef WITH_LIMITS
#define status cmda_status.limit
	if (qflags & CTL_CFLAG_LIMIT) {
		cmda_aux = &status;
		cmda_aux_size = sizeof(status);
		status.tm.tm_year = 0;
		if (LIMIT_IS_REACHED(q_limit)) {
			if (q_limit->expire.expire.upto != TEXP_UPTO_NOTSET)
				status.tm = q_limit->event_tm;
			status.reached = 1;
		} else {
			if (q_limit->restart.restart.upto != TEXP_UPTO_NOTSET)
				status.tm = q_limit->event_tm;
			status.reached = 0;
		}
		status.value1 = q_limit->lim;
		if (q_limit->cnt_neg == 0) {
			status.value2 = q_limit->cnt;
			status.value2_sign = 0;
		} else {
			status.value2 = q_limit->cnt_neg;
			status.value2_sign = 1;
		}
		status.nsublimits = 0;
# ifdef WITH_SUBLIMITS
		STAILQ_FOREACH(sublimit, &q_limit->sublimits, link)
			status.nsublimits++;
# endif
		status.value_type = q_limit->cnt_type;
		status.dynamic = RULE_IS_DYNAMIC(q_rule);
		status.active = LIMIT_IS_ACTIVE(q_limit);
	} else
#undef status
#endif
#ifdef WITH_THRESHOLDS
#define status cmda_status.threshold
	if (qflags & CTL_CFLAG_THRESHOLD) {
		cmda_aux = &status;
		cmda_aux_size = sizeof(status);
		status.value1 = q_threshold->thr;
		if (q_threshold->cnt_neg == 0) {
			status.value2 = q_threshold->cnt;
			status.value2_sign = 0;
		} else {
			status.value2 = q_threshold->cnt_neg;
			status.value2_sign = 1;
		}
		status.deviation = q_threshold->thr_max - q_threshold->thr;
		status.tm[0] = q_threshold->tm_started;
		status.tm[1] = q_threshold->tm_updated;
		status.value_type = q_threshold->cnt_type;
		status.dynamic = RULE_IS_DYNAMIC(q_rule);
		status.active = THRESHOLD_IS_ACTIVE(q_threshold);
	} else
#undef status
#endif
#define status cmda_status.all
	{
		cmda_aux = &status;
		cmda_aux_size = sizeof(status);
		status.nac_mods = nac_mods;
		status.ndb_mods = ndb_mods;
		status.nautorules = nautorules;
		status.nstatrules = nstatrules;
		status.ndynrules = ndynrules;
		status.nstatlimits = nstatlimits;
		status.ndynlimits = ndynlimits;
		status.nstatsublimits = nstatsublimits;
		status.ndynsublimits = ndynsublimits;
		status.nstatthresholds = nstatthresholds;
		status.ndynthresholds = ndynthresholds;
	}
#undef status

	return (1);
}

/*
 * "memory"
 */
static int
ctl_func_memory(void)
{
	if (cmdq.flags & CTL_CFLAG_WAIT) {
		if (memfunc_get_stat(&cmda_aux, &cmda_aux_size) < 0) {
			logbt("ctl_func_memory");
			cmda.result = CTL_ANS_SYSTEM_ERROR;
			return (1);
		}
		cmda_aux_allocated = 1;
	}
	return (1);
}

/*
 * "dump"
 */
static int
ctl_func_dump(void)
{
	need_check_flag = 1;
	set_rules_for_check();
	main_check_sec = cursec;
	dump_flag = 1;
	logmsgx(IPA_LOG_INFO, "saving current statistics to database...");
	return (0);
}

/*
 * "freeze"
 */
static int
ctl_func_freeze(void)
{
	freeze_flag = 1;
	return (1);
}

#ifdef WITH_LIMITS
static int
do_func_expire(struct rule *rule, struct limit *limit)
{
	struct ipa_limit_state newstate;

	if (LIMIT_IS_NOTREACHED(limit)) {
		logmsgx(IPA_LOG_INFO, "rule %s, limit %s: do_func_expire: "
		    "cannot expire limit, it is not reached",
		    rule->name, limit->name);
		cmda.result = CTL_ANS_CANNOT_EXPIRE;
		return (1);
	}

	if (make_rule_active(rule, 1) < 0)
		goto failed;
	if (make_limit_active(rule, limit, 1) < 0)
		goto failed;

	if (debug_ctl || rule->debug_limit || rule->debug_limit_init)
		logdbg("rule %s, limit %s: do_func_expire: expiring limit",
		    rule->name, limit->name);

	newstate.lim = limit->lim;
	newstate.cnt = limit->cnt;
	newstate.event_date_set = limit->event_date_set;
	memcpy(newstate.event_date, limit->event_date,
	    sizeof(newstate.event_date));

	newstate.event_date_set |= IPA_LIMIT_EVENT_EXPIRE_SET;
	newstate.event_date[IPA_LIMIT_EVENT_EXPIRE] = curdate;

	if (db_set_limit_state(rule, limit, &newstate, 0) < 0)
		goto failed;

	if (expire_limit(rule, limit) < 0)
		goto failed;

	if (make_limit_active(rule, limit, 0) < 0)
		goto failed;
	if (make_rule_active(rule, 0) < 0)
		goto failed;

	return (1);

failed:
	logbt("do_func_expire");
	return (-1);
}

/*
 * "-r -l expire"
 */
static int
ctl_func_expire(void)
{
	return (do_func_expire(q_rule, q_limit));
}

static int
do_func_restart(struct rule *rule, struct limit *limit)
{
	struct ipa_limit_state newstate;

	if (LIMIT_IS_REACHED(limit)) {
		logmsgx(IPA_LOG_INFO, "rule %s, limit %s: do_func_restart: "
		    "cannot restart limit, it is reached", rule->name,
		    limit->name);
		cmda.result = CTL_ANS_CANNOT_RESTART;
		return (1);
	}

	if (make_rule_active(rule, 1) < 0)
		goto failed;
	if (make_limit_active(rule, limit, 1) < 0)
		goto failed;

	if (debug_ctl || rule->debug_limit || rule->debug_limit_init)
		logdbg("rule %s, limit %s: do_func_restart: restarting limit",
		    rule->name, limit->name);

	newstate.lim = limit->lim;
	newstate.cnt = limit->cnt;
	newstate.event_date_set = limit->event_date_set;
	memcpy(newstate.event_date, limit->event_date,
	    sizeof(newstate.event_date));

	newstate.event_date_set |= IPA_LIMIT_EVENT_RESTART_SET;
	newstate.event_date[IPA_LIMIT_EVENT_RESTART] = curdate;

	if (db_set_limit_state(rule, limit, &newstate, 0) < 0)
		goto failed;

	if (restart_limit(rule, limit) < 0)
		goto failed;

	if (make_limit_active(rule, limit, 0) < 0)
		goto failed;
	if (make_rule_active(rule, 0) < 0)
		goto failed;

	return (1);

failed:
	logbt("do_func_restart");
	return (-1);
}

/*
 * "-r -l restart"
 */
static int
ctl_func_restart(void)
{
	return (do_func_restart(q_rule, q_limit));
}

#ifdef WITH_SUBLIMITS
static int
do_func_set_sublimit(struct rule *rule, struct limit *limit,
    struct sublimit *sublimit, unsigned int qflags, char debug_limit)
{
	if ((qflags & CTL_CFLAG_VALUE1) && sublimit->lim_pc != 0)
		sublimit->lim = uint64_per_cent(&limit->lim, sublimit->lim_pc);

	if (limit->cnt >= sublimit->lim) {
		if (SUBLIMIT_IS_NOTREACHED(sublimit)) {
			/*
			 * Sublimit is not reached, but with new
			 * settings became reached.
			 */
			if (debug_limit)
				logdbg("rule %s, limit %s, sublimit %s: "
				    "do_func_set_sublimit: sublimit becomes "
				    "reached", rule->name, limit->name,
				    sublimit->name);
			if (reach_sublimit(rule, limit, sublimit) < 0) {
				logbt("do_func_set_sublimit");
				return (-1);
			}
		}
	} else {
		if (SUBLIMIT_IS_REACHED(sublimit)) {
			/*
			 * Sublimit is reached, but with new setting
			 * it became not reached.
			 */
			if (debug_limit)
				logdbg("rule %s, limit %s, sublimit %s: "
				    "do_func_set_sublimit: sublimit becomes "
				    "not reached", rule->name, limit->name,
				    sublimit->name);
			SUBLIMIT_SET_NOTREACHED(sublimit);
		}
	}
	return (0);
}
#endif /* WITH_SUBLIMITS */

static int
do_func_set_limit(struct rule *rule, struct limit *limit,
    const struct ctl_cmdq *query, const struct ctl_cmdq_val *values)
{
	struct ipa_limit_state newstate;
	uint64_t value;
#ifdef WITH_SUBLIMITS
	struct sublimit *sublimit;
#endif
	unsigned int qflags;
	char debug_limit;

	if (make_rule_active(rule, 1) < 0)
		goto failed;
	if (make_limit_active(rule, limit, 1) < 0)
		goto failed;

	qflags = query->flags;
	debug_limit = debug_ctl || rule->debug_limit || rule->debug_limit_init;

	if (qflags & CTL_CFLAG_VALUE1) {
		uint64_t lim;

		if (!limit->load_limit) {
			logmsgx(IPA_LOG_INFO, "rule %s, limit %s: "
			    "do_func_set_limit: cannot change \"limit\" "
			    "parameter, load_limit = no", rule->name,
			    limit->name);
			goto cannot_modify;
		}
		lim = limit->lim;
		value = values->value1;
		if (qflags & CTL_CFLAG_VALUE1_INC) {
			if (lim <= UINT64_MAX - value)
				lim += value;
			else {
				logmsgx(IPA_LOG_ERR, "rule %s, limit %s: "
				    "do_func_set_limit: \"limit\" parameter "
				    "overflowed", rule->name, limit->name);
				goto cannot_modify;
			}
		} else if (qflags & CTL_CFLAG_VALUE1_DEC) {
			if (lim >= value)
				lim -= value;
			else {
				logmsgx(IPA_LOG_ERR, "rule %s, limit %s: "
				    "do_func_set_limit: \"limit\" parameter "
				    "underflowed", rule->name, limit->name);
				goto cannot_modify;
			}
		} else
			lim = value;
		if (value == 0 &&
		    limit->expire.expire.upto == TEXP_UPTO_SIMPLE &&
		    limit->expire.expire.seconds == 0) {
			logmsgx(IPA_LOG_ERR, "rule %s, limit %s: "
			    "do_func_set_limit: cannot set \"limit\" "
			    "parameter to zero, since \"expire\" parameter "
			    "is equal to 0s", rule->name, limit->name);
			goto cannot_modify;
		}
		limit->lim = lim;
#ifdef WITH_SUBLIMITS
		STAILQ_FOREACH(sublimit, &limit->sublimits, link)
			if (sublimit->lim_pc == 0 && sublimit->lim > lim)
				logmsgx(IPA_LOG_WARNING, "rule %s, limit %s, "
				    "sublimit %s: do_func_set_limit: sublimit "
				    "now is greater than parent limit %"PRIu64,
				    rule->name, limit->name, sublimit->name,
				    lim);
#endif
		if (debug_limit)
			logdbg("rule %s, limit %s: do_func_set_limit: "
			    "\"limit\" parameter was changed to %"PRIu64,
			    rule->name, limit->name, lim);
	}

	if (qflags & CTL_CFLAG_VALUE2) {
		value = values->value2;
		if (qflags & CTL_CFLAG_VALUE2_INC) {
			if (limit_add_chunk(rule, limit, &value) < 0)
				goto failed;
		} else if (qflags & CTL_CFLAG_VALUE2_DEC) {
			if (limit_sub_chunk(rule, limit, &value) < 0)
				goto failed;
		} else {
			limit->cnt = value;
			limit->cnt_neg = 0;
		}
		if (debug_limit) {
			uint64_t cnt;
			const char *sign;

			if (limit->cnt_neg == 0) {
				cnt = limit->cnt;
				sign = "";
			} else {
				cnt = limit->cnt_neg;
				sign = "-";
			}
			logdbg("rule %s, limit %s: do_func_set_limit: "
			    "counter was changed to %s%"PRIu64, rule->name,
			    limit->name, sign, cnt);
		}
	}

	newstate.lim = limit->lim;
	newstate.cnt = limit->cnt;
	newstate.event_date_set = limit->event_date_set;
	memcpy(newstate.event_date, limit->event_date,
	    sizeof(newstate.event_date));

	if (LIMIT_IS_REACHED(limit) && limit->cnt < limit->lim) {
		/*
		 * Limit is reached, but with new settings it became
		 * not reached.
		 */
		if (debug_limit)
			logdbg("rule %s, limit %s: do_func_set_limit: "
			    "limit becomes not reached", rule->name,
			    limit->name);
		newstate.event_date_set &= ~(IPA_LIMIT_EVENT_REACH_SET|
		    IPA_LIMIT_EVENT_REACH_EXEC_SET|IPA_LIMIT_EVENT_EXPIRE_SET);
		LIMIT_SET_NOTREACHED(limit);
		if (limit->restart.restart.upto != TEXP_UPTO_NOTSET) {
			/* limit { restart } */
			limit->event_tm =
			    limit->event_date[IPA_LIMIT_EVENT_START];
			if (set_wday(&limit->event_tm) < 0) {
				logmsgx(IPA_LOG_ERR, "rule %s, limit %s: "
				    "do_func_set_limit: set_wday failed",
				    rule->name, limit->name);
				return (-1);
			}
			/* These lines were copied from new_limit_state(). */
			ipa_tm_texp(&limit->event_tm, &limit->restart.restart);
			newstate.event_date_set |= IPA_LIMIT_EVENT_RESTART_SET;
			newstate.event_date[IPA_LIMIT_EVENT_RESTART] =
			    limit->event_tm;
			limit_set_event_sec(limit);
			if (rule->debug_limit || rule->debug_limit_init)
				logdbg("rule %s, limit %s: limit will be "
				    "restarted at %s", rule->name, limit->name,
				    tm_str(&limit->event_tm));
		} else
			limit->event_sec = EVENT_NOT_SCHEDULED;
	}

	if (db_set_limit_state(rule, limit, &newstate, 0) < 0)
		goto failed;

	if (LIMIT_IS_NOTREACHED(limit)) {
		/* Limit is not reached. */
		if (limit->cnt >= limit->lim) {
			/* But with new settings it became reached. */
			if (debug_limit)
				logdbg("rule %s, limit %s: do_func_set_limit: "
				    "limit becomes reached", rule->name,
				    limit->name);
			if (reach_limit(rule, limit) < 0)
				goto failed;
		}
	}

#ifdef WITH_SUBLIMITS
	STAILQ_FOREACH(sublimit, &limit->sublimits, link)
		if (do_func_set_sublimit(rule, limit, sublimit, qflags,
		    debug_limit) < 0)
			goto failed;
#endif

	if (qflags & CTL_CFLAG_WAIT) {
		cmda_set.value1 = limit->lim;
		if (limit->cnt_neg == 0) {
			cmda_set.value2 = limit->cnt;
			cmda_set.value2_sign = 0;
		} else {
			cmda_set.value2 = limit->cnt_neg;
			cmda_set.value2_sign = 1;
		}
		cmda_set.value_type = limit->cnt_type;
		cmda_aux = &cmda_set;
		cmda_aux_size = sizeof(cmda_set);
	}

	if (WT_IS_ACTIVE(limit->worktime)) {
		/*
		 * Limit is really active, update rule->check_sec and
		 * main_check_sec.
		 */
		if (rule->check_sec > limit->event_sec) {
			rule->check_sec = limit->event_sec;
			if (main_check_sec > rule->check_sec)
				main_check_sec = rule->check_sec;
		}
	}

	goto done;

cannot_modify:
	logmsgx(IPA_LOG_INFO, "rule %s, limit %s: do_func_set_limit: "
	    "\"limit\" parameter was not modified", rule->name, limit->name);
	cmda.result = CTL_ANS_CANNOT_MODIFY;

done:
	if (make_limit_active(rule, limit, 0) < 0)
		goto failed;
	if (make_rule_active(rule, 0) < 0)
		goto failed;
	return (1);

failed:
	logbt("do_func_set_limit");
	return (-1);
}
#else
# define ctl_func_expire NULL
# define ctl_func_restart NULL
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
static int
do_func_set_threshold(struct rule *rule, struct threshold *threshold,
    const struct ctl_cmdq *query, const struct ctl_cmdq_val *values)
{
	struct ipa_threshold_state newstate;
	uint64_t value;
	unsigned int qflags;
	char debug_threshold;

	if (make_rule_active(rule, 1) < 0)
		goto failed;
	if (make_threshold_active(rule, threshold, 1) < 0)
		goto failed;

	qflags = query->flags;
	debug_threshold = debug_ctl || rule->debug_threshold ||
	    rule->debug_threshold_init;

	if (qflags & CTL_CFLAG_VALUE1) {
		uint64_t thr;

		if (!threshold->load_thr) {
			logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "do_func_set_threshold: cannot change "
			    "\"threshold\" parameter, load_threshold = no",
			    rule->name, threshold->name);
			goto cannot_modify;
		}
		thr = threshold->thr;
		value = values->value1;
		if (qflags & CTL_CFLAG_VALUE1_INC) {
			if (thr <= UINT64_MAX - value)
				thr += value;
			else {
				logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
				    "do_func_set_threshold: \"threshold\" "
				    "parameter overflowed", rule->name,
				    threshold->name);
				goto cannot_modify;
			}
		} else if (qflags & CTL_CFLAG_VALUE1_DEC) {
			if (thr >= value)
				thr -= value;
			else {
				logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
				    "do_func_set_threshold: \"threshold\" "
				    "parameter underflowed", rule->name,
				    threshold->name);
				goto cannot_modify;
			}
		} else
			thr = value;
		threshold->thr = thr;
		if (debug_threshold)
			logdbg("rule %s, threshold %s: do_func_set_threshold: "
			    "\"threshold\" parameter was changed to %"PRIu64,
			    rule->name, threshold->name, thr);
	}

	if (qflags & CTL_CFLAG_VALUE2) {
		value = values->value2;
		if (qflags & CTL_CFLAG_VALUE2_INC) {
			if (threshold_add_chunk(rule, threshold, &value) < 0)
				goto failed;
		} else if (qflags & CTL_CFLAG_VALUE2_DEC) {
			if (threshold_sub_chunk(rule, threshold, &value) < 0)
				goto failed;
		} else if (threshold->cnt > 0 || threshold->cnt_neg == 0) {
			if (threshold->cnt > value) {
				value = threshold->cnt - value;
				if (threshold_sub_chunk(rule, threshold,
				    &value) < 0)
					goto failed;
			} else {
				value -= threshold->cnt;
				if (threshold_add_chunk(rule, threshold,
				    &value) < 0)
					goto failed;
			}
		} else {
			uint64_t value2;

			value2 = threshold->cnt_neg;
			if (threshold_add_chunk(rule, threshold, &value2) < 0 ||
			    threshold_add_chunk(rule, threshold, &value) < 0)
				goto failed;
		}
		if (debug_threshold) {
			uint64_t cnt;
			const char *sign;

			if (threshold->cnt_neg == 0) {
				cnt = threshold->cnt;
				sign = "";
			} else {
				cnt = threshold->cnt_neg;
				sign = "-";
			}
			logdbg("rule %s, threshold %s: do_func_set_threshold: "
			    "counter was changed to %s%"PRIu64, rule->name,
			    threshold->name, sign, cnt);
		}
	}

	set_thr_min_max(threshold);

	newstate.thr = threshold->thr;
	newstate.cnt = threshold->cnt;
	newstate.tm_from = threshold->tm_started;
	newstate.tm_updated = curdate;

	if (db_set_threshold_state(rule, threshold, &newstate) < 0)
		goto failed;

	if (qflags & CTL_CFLAG_WAIT) {
		cmda_set.value1 = threshold->thr;
		if (threshold->cnt_neg == 0) {
			cmda_set.value2 = threshold->cnt;
			cmda_set.value2_sign = 0;
		} else {
			cmda_set.value2 = threshold->cnt_neg;
			cmda_set.value2_sign = 1;
		}
		cmda_set.value_type = threshold->cnt_type;
		cmda_aux = &cmda_set;
		cmda_aux_size = sizeof(cmda_set);
	}

	goto done;

cannot_modify:
	logmsgx(IPA_LOG_INFO, "rule %s, threshold %s: do_func_set_threshold: "
	    "\"threshold\" parameter was not modified", rule->name,
	    threshold->name);
	cmda.result = CTL_ANS_CANNOT_MODIFY;

done:
	if (make_threshold_active(rule, threshold, 0) < 0)
		goto failed;
	if (make_rule_active(rule, 0) < 0)
		goto failed;
	return (1);

failed:
	logbt("do_func_set_threshold");
	return (-1);
}
#endif /* WITH_THRESHOLDS */

static int
do_func_set_rule(struct rule *rule, const struct ctl_cmdq *query,
    const struct ctl_cmdq_val *values)
{
	uint64_t value;
	unsigned int qflags;

	if (make_rule_active(rule, 1) < 0)
		goto failed;

	/*
	 * If a rule is really inactive, then a new record should be
	 * appended, as it is defined in ipa_mod(3).  If rule->newstat
	 * is non-zero, then this function is called from exec_ictl_list()
	 * before newday().
	 */
	if (WT_IS_INACTIVE(rule->worktime) || rule->newstat) {
		/* Append a new record for the rule. */
		if (db_append_rule(rule, &uint64_zero, 1) < 0)
			goto failed;
	}

	qflags = query->flags;
	value = values->value2;

	if (qflags & CTL_CFLAG_VALUE2_INC) {
		if (rule_add_chunk(rule, &value) < 0)
			goto failed;
	} else { /* qflags & CTL_CFLAG_VALUE2_DEC */
		if (rule_sub_chunk(rule, &value) < 0)
			goto failed;
	}

	/*
	 * If a rule is really inactive, then update its statistics
	 * right now.  Comment for rule->newstat see above.
	 */
	if (WT_IS_INACTIVE(rule->worktime) || rule->newstat) {
		if (db_update_rule(rule, &rule->cnt) < 0)
			goto failed;
		if (rule->newstat)
			rule->cnt = 0;
	} else {
#ifdef WITH_LIMITS
		if (!STAILQ_EMPTY(&rule->limits)) {
			/* Rule has limits and it is really active. */
			main_check_sec = rule->check_sec = cursec;
		}
		/*
		 * Other rules can have ac_gather_* parameters and can
		 * get statistics from this rule and can have limits.
		 * We do not force checking those rules, since ac_gather_*
		 * parameters are asynchronous in respect to the rule{}
		 * sections (by definition).
		 */
#endif
	}

	if (make_rule_active(rule, 0) < 0)
		goto failed;
	return (1);

failed:
	logbt("do_func_set_rule");
	return (-1);
}

/*
 * "-r [-l|t] set ..."
 */
static int
ctl_func_set(void)
{
#ifdef WITH_LIMITS
	if (q_limit != NULL)
		return (do_func_set_limit(q_rule, q_limit, &cmdq, &cmdq_val));
#endif
#ifdef WITH_THRESHOLDS
	if (q_threshold != NULL)
		return (do_func_set_threshold(q_rule, q_threshold, &cmdq,
		    &cmdq_val));
#endif
	return (do_func_set_rule(q_rule, &cmdq, &cmdq_val));
}

/*
 * "-a -r create"
 */
static int
ctl_func_create(void)
{
#ifdef WITH_AUTORULES
	const char *name;
	struct autorule *autorule;
	struct rule *rule;

	autorule = q_autorule;
	if (!STAILQ_EMPTY(autorule->ac_list)) {
		logmsgx(IPA_LOG_WARNING, "ctl_func_create: autorule %s "
		    "uses non \"null\" accounting system",
		    autorule->name);
		goto failed;
	}
	name = q_rule_name;
	if (name == NULL) {
		logmsgx(IPA_LOG_ERR, "ctl_func_create: illegal rule name");
		goto failed;
	}
	rule = rule_by_name(name);
	if (rule != NULL) {
		logmsgx(IPA_LOG_WARNING, "ctl_func_create: rule %s already "
		    "exists", name);
		goto failed;
	}
	if (create_rule("<ctl>", autorule->no, name, (char *)NULL) < 0) {
		logbt("ctl_func_create");
		return (-1);
	}
	rule = rule_by_name(name);
	rule->rule_flags |= RULE_FLAG_IPACTL;
	return (1);

failed:
	cmda.result = CTL_ANS_CANNOT_CREATE;
	return (1);
#else
	/* Should not be called if autorules are disabled. */
	logmsgx(IPA_LOG_ERR, "ctl_func_create: this function "
	    "should not be called");
	return (-1);
#endif
}

/*
 * "-a -r delete"
 */
static int
ctl_func_delete(void)
{
#ifdef WITH_AUTORULES
	struct rule *rule;

	rule = q_rule;
	if (!(rule->rule_flags & RULE_FLAG_IPACTL)) {
		logmsgx(IPA_LOG_ERR, "ctl_func_delete: rule %s was not "
		    "created by user request", rule->name);
		goto failed;
	}
	if (delete_rule("<ctl>", rule->no) < 0) {
		logbt("ctl_func_delete");
		return (-1);
	}
	return (1);

failed:
	cmda.result = CTL_ANS_CANNOT_DELETE;
	return (1);
#else
	logmsgx(IPA_LOG_ERR, "ctl_func_delete: rule %s is not dynamic",
	    q_rule->name);
	cmda.result = CTL_ANS_CANNOT_DELETE;
	return (1);
#endif
}

/*
 * Table of functions, which implement execution of control messages.
 * Return codes:
 *   0 -- everything is Ok, send answer from ipa_main();
 *   1 -- everything is Ok, send answer immediately;
 *  -1 -- some error occurred, send CTL_ANS_SYSTEM_ERROR answer immediately.
 */
static int (* const ctl_func_tbl[])(void) = {
	ctl_func_status,	/* 0 | CTL_CMD_STATUS	*/
	ctl_func_memory,	/* 1 | CTL_CMD_MEMORY	*/
	ctl_func_dump,		/* 2 | CTL_CMD_DUMP	*/
	ctl_func_freeze,	/* 3 | CTL_CMD_FREEZE	*/
	ctl_func_restart,	/* 4 | CTL_CMD_RESTART	*/
	ctl_func_expire,	/* 5 | CTL_CMD_EXPIRE	*/
	ctl_func_set,		/* 6 | CTL_CMD_SET	*/
	ctl_func_create,	/* 7 | CTL_CMD_CREATE	*/
	ctl_func_delete		/* 8 | CTL_CMD_DELETE	*/
};

/*
 * Get one or two names from the given buffer.
 */
static int
ctl_get_names(const char *names, size_t names_size, const char **pp1,
    const char **pp2)
{
	if (names_size == 0)
		*pp1 = *pp2 = "";
	else {
		char *ptr;

		if (names[names_size - 1] != '\0')
			return (-1);
		if (validate_name(names) < 0)
			return (-1);
		*pp1 = names;
		ptr = strchr(names, '\0');
		if (ptr == &names[names_size - 1])
			*pp2 = "";
		else {
			if (validate_name(ptr + 1) < 0)
				return (-1);
			*pp2 = ptr + 1;
		}
	}
	return (0);
}

#ifdef CTL_CHECK_CREDS
static void
log_matched_cred(const struct ctl_acl_elem *elem)
{
	const char *action, *type, *name;

	if (elem->user != NULL) {
		type = "user";
		name = elem->user;
	} else {
		type = "group";
		name = elem->group;
	}
	action = elem->allowed ?
	    "ctl_check_creds: accept command" :
	    "ctl_check_creds: deny command";
	if (q_rule != NULL)
		logdbg("%s \"%s\" from %s %s for rule %s",
		    action, cmdq_msg[cmdq.cmd], type, name, q_rule->name);
	else
		logdbg("%s \"%s\" from %s %s",
		    action, cmdq_msg[cmdq.cmd], type, name);
}

static void
log_unmatched_cred(void)
{
	if (q_rule != NULL)
		logdbg("ctl_check_creds: deny command \"%s\" for rule %s",
		    cmdq_msg[cmdq.cmd], q_rule->name);
	else
		logdbg("ctl_check_creds: deny command \"%s\"",
		    cmdq_msg[cmdq.cmd]);
}

/*
 * Compare credentials in received message with one ctl_acl_elem..
 * Return codes:
 *   0 -- deny access;
 *   1 -- allow access;
 *   2 -- did not match;
 *  -1 -- some error occurred.
 */
static int
ctl_check_cred(const struct ctl_acl_elem *elem, const void *cred)
{
#ifdef WITH_PTHREAD
	struct group grp;
	struct passwd pwd;
#endif
#ifdef __FreeBSD__
	const struct cmsgcred *cmsgcred;
#endif
#ifdef __NetBSD__
	const struct sockcred *sockcred;
#endif

#ifdef __FreeBSD__
	cmsgcred = (const struct cmsgcred *)cred;
#endif
#ifdef __NetBSD__
	sockcred = (const struct sockcred *)cred;
#endif

	if (elem->user != NULL) {
		const struct passwd *pwd_res;
		const char *user;

		/* Check real UID. */
		user = elem->user;
#ifdef WITH_PTHREAD
		errno = getpwnam_r(user, &pwd, pwd_buf, pwd_buf_size, &pwd_res);
		if (errno != 0) {
			logmsg(IPA_LOG_ERR, "ctl_check_cred: "
			    "getpwnam_r(%s)", user);
			return (-1);
		}
		if (pwd_res == NULL) {
			logmsgx(IPA_LOG_WARNING, "ctl_check_cred: "
			    "cannot find user %s", user);
			return (0);
		}
#else
		errno = 0;
		pwd_res = getpwnam(user);
		if (pwd_res == NULL) {
			if (errno != 0) {
				logmsg(IPA_LOG_ERR, "ctl_check_cred: "
				    "getpwnam(%s)", user);
				return (-1);
			}
			logmsgx(IPA_LOG_WARNING, "ctl_check_cred: "
			    "cannot find user %s", user);
			return (0);
		}
#endif /* WITH_PTHREAD */

#ifdef __FreeBSD__
		if (pwd_res->pw_uid != cmsgcred->cmcred_uid)
			return (2);
#endif
#ifdef __NetBSD__
		if (pwd_res->pw_uid != sockcred->sc_uid)
			return (2);
#endif
	} else {
		const struct group *grp_res;
		const char *group;
		int i;

		/* Check real GID. */
		group = elem->group;
#ifdef WITH_PTHREAD
		errno = getgrnam_r(group, &grp, grp_buf, grp_buf_size,
		    &grp_res);
		if (errno != 0) {
			logmsg(IPA_LOG_ERR, "ctl_check_cred: "
			    "getgrnam(%s)", group);
			return (-1);
		}
		if (grp_res == NULL) {
			logmsgx(IPA_LOG_WARNING, "ctl_check_cred: "
			    "cannot find group %s", group);
			return (0);
		}
#else
		errno = 0;
		grp_res = getgrnam(elem->group);
		if (grp_res == NULL) {
			if (errno != 0) {
				logmsg(IPA_LOG_ERR, "ctl_check_cred: "
				    "getgrnam(%s)", group);
				return (-1);
			}
			logmsgx(IPA_LOG_WARNING, "ctl_check_cred: "
			    "cannot find group %s", group);
			return (0);
		}
#endif /* WITH_PTHREAD */

#ifdef __FreeBSD__
		/*
		 * Is there standard that cmcred_groups should
		 * contain cmcred_gid?
		 */
		if (cmsgcred->cmcred_gid == grp_res->gr_gid)
			goto done;
		/* Start with first group, since zero group is EGID. */
		for (i = 1; i < cmsgcred->cmcred_ngroups; ++i)
			if (cmsgcred->cmcred_groups[i] == grp_res->gr_gid)
				goto done;
#endif
#ifdef __NetBSD__
		/*
		 * Is there standard that sc_groups should
		 * contain sc_gid?
		 */
		if (sockcred->sc_gid == grp_res->gr_gid)
			goto done;
		for (i = 0; i < sockcred->sc_ngroups; ++i)
			if (sockcred->sc_groups[i] == grp_res->gr_gid)
				goto done;
#endif
	}
done:
	return (elem->allowed);
}

/*
 * Compare credentials in received message with
 * all elementrs in acl_class.
 * Return codes:
 *   0 -- deny access;
 *   1 -- allow access;
 *  -1 -- some error occurred.
 */
static int
ctl_check_creds(const struct ctl_acl_class *acl_class)
{
	const struct ctl_acl_elem *elem;
	const struct msghdr *msghdr;
	const struct cmsghdr *cmsghdr;
	int rv;

	if (acl_class == NULL) {
		/* Deny by default. */
		goto deny;
	}

	msghdr = &query_msghdr;
	if (msghdr->msg_flags & MSG_CTRUNC) {
		logmsgx(IPA_LOG_WARNING, "ctl_check_creds: "
		    "control data was truncated");
		return (-1);
	}
	if (msghdr->msg_controllen <= sizeof(struct cmsghdr)) {
		logmsgx(IPA_LOG_WARNING, "ctl_check_creds: sender's "
		    "credentials are not returned: msg_controllen %lu <= %lu",
		    (unsigned long)msghdr->msg_controllen,
		    (unsigned long)sizeof(struct cmsghdr));
		return (-1);
	}

	cmsghdr = CMSG_FIRSTHDR(msghdr);
#ifdef __FreeBSD__
	if (cmsghdr->cmsg_len != CMSG_LEN(sizeof(struct cmsgcred))) {
		logmsgx(IPA_LOG_WARNING, "ctl_check_creds: incorrect control "
		    "message with credentials, cmsg_len %lu != %lu",
		    (unsigned long)cmsghdr->cmsg_len,
		    (unsigned long)CMSG_LEN(sizeof(struct cmsgcred)));
		return (-1);
	}
#endif
#ifdef __NetBSD__
	if (cmsghdr->cmsg_len < CMSG_LEN(SOCKCREDSIZE(1))) {
		logmsgx(IPA_LOG_WARNING, "ctl_check_creds: incorrect control "
		    "message with credentials, cmsg_len %lu < %lu bytes",
		    (unsigned long)cmsghdr->cmsg_len,
		    (unsigned long)CMSG_LEN(SOCKCREDSIZE(1)));
		return (-1);
	}
#endif
	if (cmsghdr->cmsg_level != SOL_SOCKET) {
		logmsgx(IPA_LOG_WARNING, "ctl_check_creds: control level "
		    "%d != SOL_SOCKET", cmsghdr->cmsg_level);
		return (-1);
	}
	if (cmsghdr->cmsg_type != SCM_CREDS) {
		logmsgx(IPA_LOG_WARNING, "ctl_check_creds: control type "
		    "%d != SCM_CREDS", cmsghdr->cmsg_type);
		return (-1);
	}

	STAILQ_FOREACH(elem, &acl_class->list, link) {
		rv = ctl_check_cred(elem, (const void *)CMSG_DATA(cmsghdr));
		if (rv < 0) {
			logbt("ctl_check_creds");
			return (-1);
		}
		if (rv <= 1)
			break;
	}

	if (elem != NULL) {
		if (debug_ctl)
			log_matched_cred(elem);
		return (elem->allowed);
	}

deny:
	if (debug_ctl)
		log_unmatched_cred();
	return (0);
}
#endif /* CTL_CHECK_CREDS */

/*
 * Send answer to connected socket and close it.
 */
int
ctl_send_answer(void)
{
	size_t msg_size;
	ssize_t nsent;
	int fd, iovcnt;

	if (!(cmdq.flags & CTL_CFLAG_WAIT))
		return (0);

	if (debug_ctl)
		logdbg("ctl_send_answer: sending message with answer");

	if (cmda_aux != NULL) {
		cmda.size = cmda_aux_size;
		answer_iov[1].iov_base = (char *)cmda_aux;
		answer_iov[1].iov_len = cmda_aux_size;
		iovcnt = 2;
	} else {
		cmda.size = 0;
		iovcnt = 1;
	}

	fd = ctl_connfd;
	nsent = writev(fd, answer_iov, iovcnt);
	msg_size = cmda_aux_size + sizeof(cmda);
	if (nsent != (ssize_t)msg_size) {
		if (nsent < 0) {
			logmsg(IPA_LOG_ERR, "ctl_send_answer: writev");
			switch (errno) {
			case EBADF:
			case EINVAL:
				return (-1);
			}
		} else
			logmsgx(IPA_LOG_WARNING, "ctl_send_answer: writev: "
			    "short send (%ld of %lu bytes)", (long)nsent,
			    (unsigned long)msg_size);
	}
	if (close(fd) < 0) {
		logmsg(IPA_LOG_ERR, "ctl_send_answer: close");
		if (errno == EBADF)
			return (-1);
	}
	ctl_connfd = -1;
	if (cmda_aux_allocated) {
		cmda_aux_allocated = 0;
		mem_free(cmda_aux, m_ctl);
	}
	cmda_aux = NULL;
	cmda_aux_size = 0;
	return (0);
}

/*
 * Mark a file descriptor as blocking.
 */
static int
fd_set_block(int fd)
{
	int val;

	val = fcntl(fd, F_GETFL, 0);
	if (val < 0) {
		logmsg(IPA_LOG_ERR, "fd_set_block: fcntl(%d, F_GETFL)", fd);
		return (-1);
	}
	if (fcntl(fd, F_SETFL, val & ~O_NONBLOCK) < 0) {
		logmsg(IPA_LOG_ERR, "fd_set_block: fcntl(%d, F_SETFL)", fd);
		return (-1);
	}
	return (0);
}

/*
 * Receive control query from a client.
 * Return codes:
 *   0 -- nothing was received;
 *   1 -- message was received;
 *  -1 -- some error occurred.
 */
static int
ctl_recv_query(void)
{
	ssize_t nread;
	int fd, rv;

	/*
	 * Accept connection, ctl_listenfd is non-blocking, so accept()
	 * will not block, but it can return EWOULDBLOCK, if a client
	 * closed connection before accept() invocation.
	 */
	fd = ctl_connfd = accept(ctl_listenfd, (struct sockaddr *)NULL,
	    (socklen_t *)NULL);
	if (fd < 0) {
		switch (errno) {
		case EAGAIN:
#if EAGAIN != EWOULDBLOCK
		case EWOULDBLOCK:
#endif
#ifdef ECONNABORTED
		case ECONNABORTED:
#endif
#ifdef EPROTO
		case EPROTO:
#endif
			/* Ignore connection, it was interrupted by a client. */
			logmsg(IPA_LOG_WARNING, "ctl_recv_query: accept");
			return (0);
		default:
			logmsg(IPA_LOG_ERR, "ctl_recv_query: accept");
			return (-1);
		}
	}

	rv = 0;
	if (debug_ctl)
		logdbg("ctl_recv_query: receiving control message");

	if (fd_set_block(fd) < 0) {
		logmsgx(IPA_LOG_ERR, "ctl_recv_query: cannot mark "
		    "connected socket as blocking");
		goto done;
	}

	nread = recvmsg(fd, &query_msghdr, MSG_WAITALL);
	if (nread < 0) {
		logmsg(IPA_LOG_ERR, "ctl_recv_query: recvmsg");
		switch (errno) {
		case EBADF:
		case EINVAL:
		case EMSGSIZE:
		case ENOTSOCK:
			rv = -1;
		}
		goto done;
	}
	if (nread != sizeof(cmdq)) {
		logmsgx(IPA_LOG_WARNING, "ctl_recv_query: recvmsg: "
		    "short read (%ld of %lu bytes)", (long)nread,
		    (unsigned long)sizeof(cmdq));
		goto done;
	}

	if (cmdq.ver != CTL_PROTOCOL_VERSION) {
		logmsgx(IPA_LOG_ERR, "ctl_recv_query: received query with "
		    "incorrect protocol version %u, my protocol version %u",
		    cmdq.ver, CTL_PROTOCOL_VERSION);
		goto done;
	}
	if (cmdq.size > 0) {
		size_t qsize;

		qsize = cmdq.size;
		if (qsize > ctl_query_max_size) {
			logmsgx(IPA_LOG_WARNING, "ctl_recv_query: too big "
			    "auxiliary control query (%lu > %lu bytes)",
			    (unsigned long)qsize, (unsigned long)cmdq.size);
			goto done;
		}
		cmdq_aux = mem_malloc(qsize, m_ctl);
		if (cmdq_aux == NULL) {
			logmsgx(IPA_LOG_ERR, "ctl_recv_query: "
			    "mem_malloc failed");
			cmda.result = CTL_ANS_SYSTEM_ERROR;
			rv = 1;
			goto done;
		}

		nread = recv(fd, cmdq_aux, qsize, MSG_WAITALL);
		if (nread < 0) {
			logmsg(IPA_LOG_ERR, "ctl_recv_query: recv");
			switch (errno) {
			case EBADF:
			case EINVAL:
			case ENOTSOCK:
				rv = -1;
			}
			goto done;
		}
		if (nread != qsize) {
			logmsgx(IPA_LOG_WARNING, "ctl_recv_query: recv: "
			    "short read (%ld of %lu bytes)", (long)nread,
			    (unsigned long)qsize);
			goto done;
		}
	}

	rv = 1;
done:
	if (!(cmdq.flags & CTL_CFLAG_WAIT) || rv != 1) {
		if (close(fd) < 0) {
			logmsg(IPA_LOG_ERR, "ctl_recv_query: close");
			if (errno == EBADF)
				return (-1);
		}
		ctl_connfd = -1;
	}
	return (rv);
}

/*
 * Process command query from a client.
 * Return code:
 *   0 -- do not send answer;
 *   1 -- send answer immediately;
 *  -1 -- some error occurred.
 */
static int
ctl_proc_query(void)
{
#ifdef CTL_CHECK_CREDS
	const struct ctl_acl_class *acl_class;
#endif
	const char *names, *name1, *name2;
	size_t qsize, names_size;
	unsigned int cmd, qflags;
	int rv;

	cmd = cmdq.cmd;
	qflags = cmdq.flags;
	if (cmd > CTL_MAX_CMD) {
		logmsgx(IPA_LOG_WARNING, "ctl_proc_query: received unknown "
		    "command number %u", cmd);
		cmda.result = CTL_ANS_WRONG_COMMAND;
		return (1);
	}
	if (debug_ctl)
		logdbg("ctl_proc_query: received \"%s\" command",
		    cmdq_msg[cmd]);

	rv = 0;
	if (qflags & ~CTL_ALL_CFLAGS) {
		/* Wrong flag. */
		rv = 1;
	} else if ((cmdq_flags[cmd].fl_req & qflags) !=
	    cmdq_flags[cmd].fl_req) {
		/* Some required flag is absent. */
		rv = 1;
	} else if (cmdq_flags[cmd].fl_abs & qflags) {
		/* Some flag that must be absent is specified. */
		rv = 1;
	}
	switch (cmd) {
	case CTL_CMD_STATUS:
		if ((qflags & (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD)) &&
		    !(qflags & CTL_CFLAG_RULE)) {
			/* "status" for limit or threshold, but no rule. */
			rv = 1;
		}
		break;
	case CTL_CMD_SET:
		if (!(qflags & (CTL_CFLAG_VALUE1|CTL_CFLAG_VALUE2))) {
			/* "set", but no values. */
			rv = 1;
		}
		if (!(qflags & (CTL_CFLAG_LIMIT|CTL_CFLAG_THRESHOLD))) {
			/* "set" for rule. */
			if ((qflags & CTL_CFLAG_VALUE1) ||
			    !(qflags & CTL_CFLAG_VALUE2)) {
				/* Has value1 or does not have value2. */
				rv = 1;
			}
			if (!(qflags &
			    (CTL_CFLAG_VALUE2_INC|CTL_CFLAG_VALUE2_DEC))) {
				/* Cannot set counter for rule. */
				rv = 1;
			}
		}
		break;
	}
	if (rv != 0) {
		logmsgx(IPA_LOG_WARNING, "ctl_proc_query: received \"%s\" "
		    "command with illegal combinations of flags 0x%08x",
		    cmdq_msg[cmd], qflags);
		cmda.result = CTL_ANS_WRONG_COMMAND;
		return (1);
	}

	qsize = cmdq.size;
	names = cmdq_aux;
	names_size = qsize;
	if (cmd == CTL_CMD_SET) {
		if (qsize < sizeof(cmdq_val)) {
			logmsgx(IPA_LOG_ERR, "ctl_proc_query: too small "
			    "auxiliary data in command \"%s\"", cmdq_msg[cmd]);
			cmda.result = CTL_ANS_WRONG_COMMAND;
			return (1);
		}
		memcpy(&cmdq_val, cmdq_aux, sizeof(cmdq_val));
		names += sizeof(cmdq_val);
		names_size -= sizeof(cmdq_val);
	}

	if (ctl_get_names(names, names_size, &name1, &name2) < 0) {
		logmsgx(IPA_LOG_WARNING, "ctl_proc_query: received \"%s\" "
		    "command with illegal name", cmdq_msg[cmd]);
		cmda.result = CTL_ANS_ILLEGAL_NAME;
		return (1);
	}

	if (qflags & CTL_CFLAG_RULE) {
		q_rule = rule_by_name(name1);
		if (q_rule == NULL) {
			cmda.result = CTL_ANS_UNKNOWN_RULE;
			return (1);
		}
	} else
		q_rule = NULL;

	if (qflags & CTL_CFLAG_AUTORULE) {
#ifdef WITH_AUTORULES
		q_autorule = autorule_by_name(name1);
		if (q_autorule == NULL) {
			cmda.result = CTL_ANS_UNKNOWN_AUTORULE;
			return (1);
		}
		q_rule_name = name2;
#else
		cmda.result = CTL_ANS_UNKNOWN_AUTORULE;
		return (1);
#endif
	} else {
#ifdef WITH_AUTORULES
		q_autorule = NULL;
#endif
	}

#ifdef CTL_CHECK_CREDS
	switch (cmd) {
	case CTL_CMD_STATUS:
		if (q_rule != NULL)
			acl_class = q_rule->ctl_rule_acl;
		else
			acl_class = ctl_stat_acl;
		break;
	case CTL_CMD_MEMORY:
		acl_class = ctl_stat_acl;
		break;
	case CTL_CMD_DUMP:
		acl_class = ctl_dump_acl;
		break;
	case CTL_CMD_FREEZE:
		acl_class = ctl_freeze_acl;
		break;
	case CTL_CMD_RESTART:
	case CTL_CMD_EXPIRE:
	case CTL_CMD_SET:
		acl_class = q_rule->ctl_rule_acl;
		break;
	case CTL_CMD_CREATE:
		acl_class = q_autorule->ctl_rule_acl;
		break;
	case CTL_CMD_DELETE:
		acl_class = q_rule->ctl_rule_acl;
		break;
	default:
		acl_class = NULL;
	}
	switch (ctl_check_creds(acl_class)) {
	case 0:
		cmda.result = CTL_ANS_DENIED;
		return (1);
	case -1:
		cmda.result = CTL_ANS_SYSTEM_ERROR;
		return (-1);
	}
#endif

	if (qflags & CTL_CFLAG_LIMIT) {
#ifdef WITH_LIMITS
		q_limit = limit_by_name(q_rule, name2);
		if (q_limit == NULL) {
			cmda.result = CTL_ANS_UNKNOWN_LIMIT;
			return (1);
		}
#else
		cmda.result = CTL_ANS_UNKNOWN_LIMIT;
		return (1);
#endif
	} else {
#ifdef WITH_LIMITS
		q_limit = NULL;
#endif
	}

	if (qflags & CTL_CFLAG_THRESHOLD) {
#ifdef WITH_THRESHOLDS
		q_threshold = threshold_by_name(q_rule, name2);
		if (q_threshold == NULL) {
			cmda.result = CTL_ANS_UNKNOWN_THRESHOLD;
			return (1);
		}
#else
		cmda.result = CTL_ANS_UNKNOWN_THRESHOLD;
		return (1);
#endif
	} else {
#ifdef WITH_THRESHOLDS
		q_threshold = NULL;
#endif
	}

	rv = ctl_func_tbl[cmd]();
	if (rv < 0) {
		logmsgx(IPA_LOG_ERR, "ctl_recv_query: execution of control "
		    "command failed");
		cmda.result = CTL_ANS_SYSTEM_ERROR;
	}
	return (rv);
}

/*
 * Receive a command query from a client, process and send answer.
 */
int
ctl_query(void)
{
	int rv;

	cmda.result = CTL_ANS_DONE;
	rv = ctl_recv_query();
	if (rv == 1) {
		if (cmda.result == CTL_ANS_DONE)
			rv = ctl_proc_query();
		if (rv == 1)
			rv = ctl_send_answer();
	}

	if (cmdq_aux != NULL) {
		mem_free(cmdq_aux, m_ctl);
		cmdq_aux = NULL;
	}

	if (rv < 0)
		logbt("ctl_query");

	return (rv);
}

#ifdef WITH_RULES
static int
run_ictl(const struct ictl *ictl)
{
	unsigned int qflags;
	int rv;

	if (debug_ctl)
		logdbg("running ictl \"%s\"", ictl->str);
	rv = -1;
	switch (ictl->cmdq.cmd) {
# ifdef WITH_LIMITS
	case CTL_CMD_RESTART:
		rv = do_func_restart(ictl->rule.ptr, ictl->limit.ptr);
		break;
	case CTL_CMD_EXPIRE:
		rv = do_func_expire(ictl->rule.ptr, ictl->limit.ptr);
		break;
# endif
	case CTL_CMD_SET:
		qflags = ictl->cmdq.flags;
		if ((qflags & (CTL_CFLAG_RULE|CTL_CFLAG_LIMIT|
		    CTL_CFLAG_THRESHOLD)) == CTL_CFLAG_RULE)
			rv = do_func_set_rule(ictl->rule.ptr, &ictl->cmdq,
			    &ictl->values);
# ifdef WITH_LIMITS
		else if (qflags & CTL_CFLAG_LIMIT)
			rv = do_func_set_limit(ictl->rule.ptr, ictl->limit.ptr,
			    &ictl->cmdq, &ictl->values);
# endif
# ifdef WITH_THRESHOLDS
		else if (qflags & CTL_CFLAG_THRESHOLD)
			rv = do_func_set_threshold(ictl->rule.ptr,
			    ictl->threshold.ptr, &ictl->cmdq, &ictl->values);
# endif
		break;
	}
	if (rv < 0) {
		logbt("run_ictl");
		logmsgx(IPA_LOG_ERR, "command ictl \"%s\" failed", ictl->str);
		return (-1);
	}
	return (0);
}

int
run_ictl_list(const struct ictl_list *ictl_list)
{
	const struct ictl *ictl;
	unsigned int result;

	result = cmda.result;
	STAILQ_FOREACH(ictl, ictl_list, link)
		if (run_ictl(ictl) < 0) {
			logbt("run_ictl_list");
			return (-1);
		}
	cmda.result = result;
	return (0);
}

void
free_ictl_list(struct ictl_list *ictl_list)
{
	struct ictl *ictl, *ictl_next;

	STAILQ_FOREACH_SAFE(ictl, ictl_list, link, ictl_next) {
		mem_free(ictl->str, m_ctl);
		mzone_free(ictl_mzone, ictl);
	}
	STAILQ_INIT(ictl_list);
}

static int
setup_ictl1(struct ictl *ictl)
{
	struct rule *rule;
# ifdef WITH_LIMITS
	struct limit *limit;
# endif
# ifdef WITH_THRESHOLDS
	struct threshold *threshold;
# endif
# ifdef WITH_ANY_LIMITS
	unsigned int qflags;
# endif

	rule = rule_by_name(ictl->rule.name);
	if (rule == NULL) {
		logconfe("cannot find rule");
		return (-1);
	}
	mem_free(ictl->rule.name, m_ctl);
	ictl->rule.ptr = rule;
# ifdef WITH_ANY_LIMITS
	qflags = ictl->cmdq.flags;
# endif
# ifdef WITH_LIMITS
	if (qflags & CTL_CFLAG_LIMIT) {
		limit = limit_by_name(rule, ictl->limit.name);
		if (limit == NULL) {
			logconfe("cannot find limit");
			return (-1);
		}
		if (qflags & CTL_CFLAG_VALUE1) {
			if (!limit->load_limit) {
				logconfe("cannot change \"limit\" "
				    "parameter, load_limit = no");
				return (-1);
			}
			if (!(qflags &
			    (CTL_CFLAG_VALUE1_INC|CTL_CFLAG_VALUE1_DEC)) &&
			    ictl->values.value1 == 0 &&
			    limit->expire.expire.upto == TEXP_UPTO_SIMPLE &&
			    limit->expire.expire.seconds == 0) {
				logconfe("cannot set \"limit\" parameter "
				    "to zero, since \"expire\" parameter "
				    "is equal to 0s");
				return (-1);
			}
		}
		mem_free(ictl->limit.name, m_ctl);
		ictl->limit.ptr = limit;
	}
# endif
# ifdef WITH_THRESHOLDS
	if (qflags & CTL_CFLAG_THRESHOLD) {
		threshold = threshold_by_name(rule, ictl->threshold.name);
		if (threshold == NULL) {
			logconfe("cannot find threshold");
			return (-1);
		}
		if (qflags & CTL_CFLAG_VALUE1) {
			if (!threshold->load_thr) {
				logconfe("cannot change \"threshold\" "
				    "parameter, load_threshold = no");
				return (-1);
			}
		}
		mem_free(ictl->threshold.name, m_ctl);
		ictl->threshold.ptr = threshold;
	}
# endif
	return (0);
}

/*
 * Check that rule's, limit's and threshold's names specified in "ictl"
 * parameter exist and change corresponding pointers to rule, limit and
 * threshold structures.
 */
static int
setup_ictl(const struct cmds *cmds, const char *format, ...)
{
	va_list ap;
	const struct ictl_list *ictl_list;
	struct ictl *ictl;

	ictl_list = &cmds->ictl_list;
	STAILQ_FOREACH(ictl, ictl_list, link)
		if (setup_ictl1(ictl) < 0) {
			logconfe("ictl command: \"%s\", from section:",
			    ictl->str);
			va_start(ap, format);
			vlogconfe(format, ap);
			va_end(ap);
			return (-1);
		}
	return (0);
}

#ifdef WITH_SUBLIMITS
static int
setup_sublimit_ictl(struct sublimit *sublimit, const char *rule_name,
    const char *limit_name)
{
	const char *sublimit_name, *sect_name;
	struct cmds_limit *cmds_limit;
	unsigned int x;
	int rv;

	sublimit_name = sublimit->name;
	for (x = 0; x < 2; ++x) {
		sect_name = rc_sect_name[x];
		cmds_limit = &sublimit->rc[x];
		rv =
		    setup_ictl(&cmds_limit->cmds, "rule %s { "
		        "limit %s { sublimit %s { %s {}}}}",
		        rule_name, limit_name, sublimit_name, sect_name) +
		    setup_ictl(&cmds_limit->cmds_reached, "rule %s { "
		        "limit %s { sublimit %s { %s { if_reached {}}}}}",
		        rule_name, limit_name, sublimit_name, sect_name) +
		    setup_ictl(&cmds_limit->cmds_not_reached, "rule %s { "
		        "limit %s { sublimit %s { %s { if_not_reached {}}}}}",
		        rule_name, limit_name, sublimit_name, sect_name);
		if (rv < 0)
			return (-1);
	}
	if (setup_ictl(&sublimit->reach, "rule %s { "
	    "limit %s { sublimit %s { reach {}}}}",
	    rule_name, limit_name, sublimit_name) < 0)
		return (-1);
	return (0);
}
#endif

#ifdef WITH_LIMITS
static int
setup_limit_ictl(struct limit *limit, const char *rule_name)
{
	const char *limit_name, *sect_name;
	struct cmds_limit *cmds_limit;
# ifdef WITH_SUBLIMITS
	struct sublimit *sublimit;
# endif
	unsigned int x;
	int rv;

	limit_name = limit->name;
	for (x = 0; x < 2; ++x) {
		sect_name = rc_sect_name[x];
		cmds_limit = &limit->rc[x];
		rv =
		    setup_ictl(&cmds_limit->cmds,
		        "rule %s { limit %s { %s {}}}",
		        rule_name, limit_name, sect_name) +
		    setup_ictl(&cmds_limit->cmds_reached,
		        "rule %s { limit %s { %s { if_reached {}}}}",
		        rule_name, limit_name, sect_name) +
		    setup_ictl(&cmds_limit->cmds_not_reached,
		        "rule %s { limit %s { %s { if_not_reached {}}}}",
		        rule_name, limit_name, sect_name);
		if (rv < 0)
			return (-1);
	}
	rv =
	    setup_ictl(&limit->reach,
	        "rule %s { limit %s { reach {}}}", rule_name, limit_name) +
	    setup_ictl(&limit->restart.cmds,
	        "rule %s { limit %s { restart {}}}", rule_name, limit_name) +
	    setup_ictl(&limit->expire.cmds,
	        "rule %s { limit %s { expire {}}}", rule_name, limit_name);
	if (rv < 0)
		return (-1);
# ifdef WITH_SUBLIMITS
	STAILQ_FOREACH(sublimit, &limit->sublimits, link)
		if (setup_sublimit_ictl(sublimit, rule_name, limit_name) < 0)
			return (-1);
# endif
	return (0);
}
#endif /* WITH_LIMITS */

#ifdef WITH_THRESHOLDS
static int
setup_threshold_ictl(struct threshold *threshold, const char *rule_name)
{
	const char *threshold_name;
	unsigned int x;
	int rv;

	threshold_name = threshold->name;
	for (x = 0; x < 2; ++x) {
		rv =
		    setup_ictl(&threshold->rc[x],
		        "rule %s { threshold %s { %s {}}}",
		        rule_name, threshold_name, rc_sect_name[x]);
		if (rv < 0)
			return (-1);
	}
	rv =
	    setup_ictl(&threshold->below_thr,
	        "rule %s { threshold %s { below_threshold {}}}",
	        rule_name, threshold_name) +
	    setup_ictl(&threshold->equal_thr,
	        "rule %s { threshold %s { equal_threshold {}}}",
	        rule_name, threshold_name) +
	    setup_ictl(&threshold->above_thr,
	        "rule %s { threshold %s { above_threshold {}}}",
	        rule_name, threshold_name);
	return (rv < 0 ? -1 : 0);
}
#endif /* WITH_THRESHOLDS */

static int
setup_rule_ictl(struct rule *rule)
{
	const char *rule_name, *sect_name;
	struct cmds_rule *cmds_rule;
#ifdef WITH_LIMITS
	struct limit *limit;
#endif
#ifdef WITH_THRESHOLDS
	struct threshold *threshold;
#endif
	unsigned int x;
	int rv;

	rule_name = rule->name;
	for (x = 0; x < 2; ++x) {
		sect_name = rc_sect_name[x];
		cmds_rule = &rule->rc[x];
		rv =
# ifdef WITH_LIMITS
		    setup_ictl(&cmds_rule->cmds_all_reached,
		        "rule %s { %s { if_all_reached {}}}",
		        rule_name, sect_name) +
		    setup_ictl(&cmds_rule->cmds_all_not_reached,
		        "rule %s { %s { if_all_not_reached {}}}",
		        rule_name, sect_name) +
		    setup_ictl(&cmds_rule->cmds_any_reached,
		        "rule %s { %s { if_any_reached {}}}",
		        rule_name, sect_name) +
		    setup_ictl(&cmds_rule->cmds_any_not_reached,
		        "rule %s { %s { if_any_not_reached {}}}",
		        rule_name, sect_name) +
# endif
		    setup_ictl(&cmds_rule->cmds,
		        "rule %s { %s {}}",
		        rule_name, sect_name);
		if (rv < 0)
			return (-1);
	}
# ifdef WITH_LIMITS
	STAILQ_FOREACH(limit, &rule->limits, link)
		if (setup_limit_ictl(limit, rule_name) < 0)
			return (-1);
# endif
# ifdef WITH_THRESHOLDS
	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (setup_threshold_ictl(threshold, rule_name) < 0)
			return (-1);
# endif
	return (0);
}

int
setup_all_ictl(void)
{
	struct rule *rule;

	if (setup_ictl(&cmds_global[RC_STARTUP], "%s {}",
	    rc_sect_name[RC_STARTUP]) < 0)
		return (-1);

	TAILQ_FOREACH(rule, &rules_list, list)
		if (setup_rule_ictl(rule) < 0)
			return (-1);

	return (0);
}
#endif /* WITH_RULES */
