/*-
 * Copyright (c) 1999-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa.c,v 1.4.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <locale.h>
#include <pwd.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <unistd.h>

#ifdef HAVE_PATHS_H
# include <paths.h>
#endif
#ifndef _PATH_DEVNULL
# ifdef __GNUC__
#  warning "defining _PATH_DEVNULL to \"/dev/null\""
# endif
# define _PATH_DEVNULL "/dev/null"
#endif

#ifdef WITH_PTHREAD
# include <pthread.h>
#endif

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "pathnames.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_time.h"

#include "ipa_conf.h"
#include "ipa_ctl.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

#if defined(HAVE_CLOSEFROM) && \
    (STDIN_FILENO > 2 || STDOUT_FILENO > 2 || STDERR_FILENO > 2)
# undef HAVE_CLOSEFROM
#endif

static const char *envprogname;

static const char *pid_file_name = IPA_PID_FILE; /* -p pid_file */
static int	pid_file_fd;		/* File descriptor of PID-file. */

static int	nofile_max;		/* Maximum number of opened files. */

struct sig_descr {
	const char	*signame;	/* Command name. */
	int		signo;		/* Signal number. */
};

static const struct sig_descr sig_descr_tbl[] = {
	{ "shutdown",	 SIGTERM },
	{ "reconfigure", SIGHUP	 },
	{ "kill",	 SIGKILL },
	{ "test",	 0	 },
	{ NULL,		 0	 }
};

#define PID_FILE_PERM (S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) /* 0644 */

static void	exit_err(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);
static void	exit_errx(const char *, ...) ATTR_NORETURN
		    ATTR_FORMAT(printf, 1, 2);

/*
 * Close and unlink PID-file.
 */
static void
unlink_pid_file(void)
{
	if (close(pid_file_fd) < 0)
		logmsg(IPA_LOG_ERR, "unlink_pid_file: close(%s)",
		    pid_file_name);
	if (unlink(pid_file_name) < 0)
		logmsg(IPA_LOG_ERR, "unlink_pid_file: unlink(%s)",
		    pid_file_name);
}

/*
 * Output the program name, a message, an error message and exit.
 */
static void
exit_err(const char *format, ...)
{
	va_list ap;
	int errno_save;

	errno_save = errno;
	fflush(stdout);
	fprintf(stderr, "%s: ", envprogname);
	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);
	if (errno_save != 0)
		fprintf(stderr, ": %s", strerror(errno_save));
	fprintf(stderr, "\n");
	exit(EXIT_FAILURE);
}

/*
 * Output the program name, a message and exit.
 */
static void
exit_errx(const char *format, ...)
{
	va_list ap;

	fflush(stdout);
	fprintf(stderr, "%s: ", envprogname);
	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	exit(EXIT_FAILURE);
}

/*
 * Output version number and settings (-v and -h switches).
 */
static void
show_version(void)
{
	printf(IPA_NAME ", version " PACKAGE_VERSION "\nRuntime settings: "
#ifdef SYM_PREFIX
	"symbol prefix \"" SYM_PREFIX "\", "
#else
	"symbol prefix is not used, "
#endif
#ifdef WITH_PTHREAD
	"thread-safe mode, "
#else
	"single-threaded mode, "
#endif
#ifdef USE_LIBLTDL
	"using libtool's ltdl library, "
#else
	"using dlopen interface, "
#endif
#ifdef WITH_MEMFUNC_DEBUG
	"memfunc debugging is enabled, "
#else
	"memfunc debugging is disabled, "
#endif
	"using shell %s, using null device %s.\nSupports:"
#ifdef WITH_AUTORULES
	" autorules"
#endif
#ifdef WITH_RULES
# ifdef WITH_AUTORULES
	","
# endif
	" rules"
#endif
#ifdef WITH_LIMITS
	", limits"
#endif
#ifdef WITH_SUBLIMITS
	", sublimits"
#endif
#ifdef WITH_THRESHOLDS
	", thresholds"
#endif
#ifdef CTL_CHECK_CREDS
	"; checking ipactl's messages credentials is enabled"
#else
	"; checking ipactl's messages credentials is disabled"
#endif
	".\n", shell_path_default, _PATH_DEVNULL);
}

/*
 * Output the help message (-h switch).
 */
static void
usage(void)
{
	show_version();
	printf("\
Usage: %s [options]\n\
 Utility for accounting\n\
 Options are:\n\
  -c dir\t\tSet the directory " IPA_NAME " should chroot(2) into\n\
\t\t\timmediately, working directory is not changed\n\
  -d\t\t\tDo not run in the background (debug mode)\n\
  -f conf_file\t\tUse given config-file instead of using default\n\
\t\t\tconfiguration file %s\n", envprogname, IPA_CONF_FILE);
	printf("\
  -i log_ident\t\tUse given log-ident (the default log-ident is \
\"" IPA_LOG_IDENT "\")\n\
  -k signal\t\tSend signal to the running copy of ipa, signal\n\
\t\t\tcan be: reconfigure, shutdown, test or kill\n\
  -o log_file\t\tUse given log-file instead of using syslog\n\
  -p pid_file\t\tUse given PID-file instead of using default\n\
\t\t\tPID-file %s\n\
  -t\t\t\tCheck the configuration file (two switches mimic\n\
\t\t\treal configuration regime) and exit\n", IPA_PID_FILE);
	printf("\
  -u user\t\tChange UID of the running copy of " IPA_NAME " to user\n\
  -g group\t\tChange GID of the running copy of " IPA_NAME " to group\n\
  -x"
#ifdef WITH_RULES
" [-r rule"
# ifdef WITH_LIMITS
" [-l limit"
#  ifdef WITH_SUBLIMITS
" [-s sublimit]"
#  endif
"]"
# endif
# ifdef WITH_THRESHOLDS
" [-t threshold]"
# endif
"]\n     section"
# ifdef WITH_ANY_LIMITS
" [subsection]\n\t"
# endif
#else
" section"
#endif /* WITH_RULES */
"\t\tRun commands from the given section and exit\n\
  -h\t\t\tOutput this help message and exit\n\
  -v\t\t\tShow version number and exit\n");
}

/*
 * Ignore SIGTTOU, SIGTTIN, SIGTSTP and SIGPIPE and install sigmask
 * to block signals which can be received.  By default SIGCHLD is ignored.
 */
static int
sig_init_1(void)
{
	struct sigaction sigact;

	sigact.sa_handler = SIG_IGN;
	sigemptyset(&sigact.sa_mask);
	sigact.sa_flags = 0;

	if (sigaction(SIGTTOU, &sigact, (struct sigaction *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_1: sigaction(SIGTTOU)");
		return (-1);
	}
	if (sigaction(SIGTTIN, &sigact, (struct sigaction *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_1: sigaction(SIGTTIN)");
		return (-1);
	}
	if (sigaction(SIGTSTP, &sigact, (struct sigaction *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_1: sigaction(SIGTSTP)");
		return (-1);
	}

	/*
	 * Need this for control socket and for asynchronous running
	 * commands (in case when ipa exits before asynchronous
	 * running commands).
	 */
	if (sigaction(SIGPIPE, &sigact, (struct sigaction *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_1: sigaction(SIGPIPE)");
		return (-1);
	}

	/* Set sigmask. */
	sigemptyset(&sigmask);
	if (debug)
		sigaddset(&sigmask, SIGINT);
	sigaddset(&sigmask, SIGHUP);
	sigaddset(&sigmask, SIGTERM);

	/*
	 * Signals SIGTERM (and SIGINT) and SIGHUP
	 * before next function are not handled properly.
	 */
	if (Sigprocmask(SIG_SETMASK, &sigmask, (sigset_t *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_1: "
		    SIGPROCMASK_NAME "(SIG_SETMASK)");
		return (-1);
	}

	return (0);
}

/*
 * Add SIGCHLD to sigmask, initialize zeromask and set signal handlers.
 */
static int
sig_init_2(void)
{
	struct sigaction sigact;

	/* Add SIGCHLD to sigmask. */
	sigaddset(&sigmask, SIGCHLD);
	if (Sigprocmask(SIG_SETMASK, &sigmask, (sigset_t *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_2: "
		    SIGPROCMASK_NAME "(SIG_SETMASK)");
		return (-1);
	}

	/* Empty zeromask. */
	sigemptyset(&zeromask);

	/* Install signal handlers. */
	sigact.sa_flags = 0;
	sigact.sa_mask = sigmask;

	/* Set SIGTERM (and SIGINT) handlers. */
	sigact.sa_handler = sig_term;
	if (debug)
		if (sigaction(SIGINT, &sigact, (struct sigaction *)NULL) < 0) {
			logmsg(IPA_LOG_ERR, "sig_init_2: sigaction(SIGINT)");
			return (-1);
		}
	if (sigaction(SIGTERM, &sigact, (struct sigaction *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_2: sigaction(SIGTERM)");
		return (-1);
	}

	/* Set SIGHUP handler. */
	sigact.sa_handler = sig_hup;
	if (sigaction(SIGHUP, &sigact, (struct sigaction *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_2: sigaction(SIGHUP)");
		return (-1);
	}

	/* Set SIGCHLD handler. */
	sigact.sa_handler = sig_chld;
	sigact.sa_flags = SA_NOCLDSTOP;
	if (sigaction(SIGCHLD, &sigact, (struct sigaction *)NULL) < 0) {
		logmsg(IPA_LOG_ERR, "sig_init_2: sigaction(SIGCHLD)");
		return (-1);
	}

	return (0);
}

/*
 * Send a signal signo to the running copy of ipa(8).
 * Check if the file pid_file_name is locked and if it is locked,
 * then send a signal to the PID returned in l_pid field of
 * struct flock{}.
 */
static void
sendsig(int signo)
{
	struct flock lock;
	int fd;

	fd = open(pid_file_name, O_RDONLY);
	if (fd < 0)
		exit_err("sendsig: open(%s, O_RDONLY)", pid_file_name);

	lock.l_start = 0;		/* Zero offset from... */
	lock.l_len = 0;			/* Entire file. */
	lock.l_type = F_WRLCK;		/* Should be valid value. */
	lock.l_whence = SEEK_SET;	/* ... the start of the file. */

	if (fcntl(fd, F_GETLK, &lock) < 0) {
#ifdef __CYGWIN__
		if (errno == ENOSYS)
			exit_errx("on Cygwin the -k option cannot be "
			    "implemented in a safe way");
#endif
		exit_err("sendsig: fcntl(%s, F_GETLK)", pid_file_name);
	}

	if (signo != 0) {
		switch (lock.l_type) {
		case F_UNLCK:
			exit_errx("sendsig: file %s is not locked, do not "
			    "send any signal", pid_file_name);
			/* NOTREACHED */
		case F_RDLCK:
			exit_errx("sendsig: file %s is lock, but lock is "
			    "shared, do not send any signal", pid_file_name);
		}
		printf("Sending signal %d to PID %ld...", signo,
		    (long)lock.l_pid);
		if (kill(lock.l_pid, signo) < 0) {
			printf("\n");
			exit_err("sendsig: kill(PID %ld, %d)",
			    (long)lock.l_pid, signo);
		}
		printf(" done\n");
	} else {
		if (lock.l_type == F_UNLCK)
			exit_err("file %s exists, but is not locked",
			    pid_file_name);
		printf("File %s is locked (%s) by PID %ld\n",
		    pid_file_name, lock.l_type == F_WRLCK ?
		    "exclusive" : "shared", (long)lock.l_pid);
	}
}

/*
 * Run in the background.
 */
static int
bg_init(void)
{
	pid_t pid;

	pid = fork();
	if (pid == (pid_t)-1) {
		logmsg(IPA_LOG_ERR, "bg_init: fork");
		return (-1);
	}

	if (pid != 0) {
		/* Parent exits. */
		exit(EXIT_SUCCESS);
	}

	/* Child continues. */
	if (setsid() == (pid_t)-1)
		logmsg(IPA_LOG_ERR, "bg_init: setsid");
	return (0);
}

/*
 * Create pipe, if any of pipe's ends is stdin, stdout or stderr,
 * then create another pipe and so on.  Then mark the read-end of
 * a pipe as non-blocking, dup2() the write-end of a pipe to fd_write
 * and mark fd_write as non-blocking, also mark the read-end of
 * a pipe as non-blocking and return its value in *fd_read.  Read
 * also comment for fd_init().
 */
static int
pipe_stream(int fd_write, int *fd_read)
{
	unsigned int i, j;
	int fd0, fd1, fd[6];

	/*
	 * Pipe's ends should not be equal to file descriptor of stdin,
	 * stdout or stderr.  SUSv3 claims that pipe(2) returns lowest
	 * available descriptors, so three (each with two descriptors)
	 * attempts are enough to get desired descriptor.
	 */
	for (i = j = 0;; j += 2, ++i) {
		if (pipe(fd + j) < 0) {
			logmsg(IPA_LOG_ERR, "pipe_stream: pipe");
			return (-1);
		}
		fd0 = fd[j];
		fd1 = fd[j + 1];
		if (fd0 != STDIN_FILENO && fd0 != STDOUT_FILENO &&
		    fd0 != STDERR_FILENO && fd1 != STDIN_FILENO &&
		    fd1 != STDOUT_FILENO && fd1 != STDERR_FILENO)
			break;
		if (i == 2) {
			logmsgx(IPA_LOG_ERR, "pipe_stream: something is wrong "
			    "in algorithm");
			return (-1);
		}
	}

	/* Close "incorrectly" opened file descriptors. */
	while (j > 0) {
		--j;
		if (close(fd[j]) < 0) {
			logmsg(IPA_LOG_ERR, "pipe_stream: close(%d)", fd[j]);
			return (-1);
		}
	}

	if (dup2(fd1, fd_write) < 0) {
		logmsg(IPA_LOG_ERR, "pipe_stream: dup2(%d, %d)", fd1, fd_write);
		return (-1);
	}
	if (close(fd1) < 0) {
		logmsg(IPA_LOG_ERR, "pipe_stream: close(%d)", fd1);
		return (-1);
	}
	if (fd_set_nonblock(fd0) < 0 || fd_set_nonblock(fd_write) < 0) {
		logbt("pipe_stream");
		return (-1);
	}

	*fd_read = fd0;
	return (0);
}

/*
 * Close not used file descriptors and create pipes for standard
 * streams stdout and stderr (if needed).  If this function returns -1,
 * then function which called fd_init() should not call log_stdxxx()
 * functions since syslog() or log file could use one of STDxxx_FILENO
 * descriptors.  Also any log message should not be sent before this
 * function invocation.
 */
static int
fd_init(void)
{
	long val;
	int fd;

	errno = 0;
	val = sysconf(_SC_OPEN_MAX);
	if (val < 0) {
		if (errno != 0) {
			logmsg(IPA_LOG_ERR, "fd_init: sysconf(_SC_OPEN_MAX)");
			return (-1);
		}
#ifndef HAVE_CLOSEFROM
		logmsgx(IPA_LOG_ERR, "fd_init: infinite limit for "
		    "_SC_OPEN_MAX is not supported");
		return (-1);
#endif
	}
	nofile_max = (int)val;

#ifdef HAVE_CLOSEFROM
	if (!debug) {
		(void)close(STDIN_FILENO);
		(void)close(STDERR_FILENO);
	}
	(void)close(STDOUT_FILENO);
	for (fd = 3; fd < pid_file_fd; ++fd)
		(void)close(fd);
	(void)closefrom(fd + 1);
#else
# if LONG_MAX > INT_MAX
	if (val > INT_MAX) {
		logmsgx(IPA_LOG_ERR, "fd_init: too big value %ld for 'int' "
		    "type from sysconf(_SC_OPEN_MAX)", val);
		return (-1);
	}
# endif
	for (fd = 0; fd < (int)val; ++fd)
		if (fd != pid_file_fd &&
		    !(debug && (fd == STDIN_FILENO || fd == STDERR_FILENO)))
			(void)close(fd);
#endif /* HAVE_CLOSEFROM */

	if (!debug) {
		fd = open(_PATH_DEVNULL, O_RDWR);
		if (fd < 0) {
			logmsg(IPA_LOG_ERR, "fd_init: open(%s, O_RDWR)",
			    _PATH_DEVNULL);
			return (-1);
		}
		if (fd != STDIN_FILENO) {
			/* stdin -> /dev/null */
			if (dup2(fd, STDIN_FILENO) < 0) {
				logmsg(IPA_LOG_ERR, "fd_init: dup2(%d, %d)",
				    fd, STDIN_FILENO);
				return (-1);
			}
			if (close(fd) < 0) {
				logmsg(IPA_LOG_ERR, "fd_init: close(%d)", fd);
				return (-1);
			}
		}
		if (pipe_stream(STDERR_FILENO, &stderr_read) < 0) {
			logmsgx(IPA_LOG_ERR, "fd_init: pipe_stream for "
			    "STDERR_FILENO failed");
			return (-1);
		}
	}

	if (pipe_stream(STDOUT_FILENO, &stdout_read) < 0) {
		logmsgx(IPA_LOG_ERR, "fd_init: pipe_stream for "
		    "STDOUT_FILENO failed");
		return (-1);
	}

	return (0);
}

/*
 * Create/open PID-file and check if it is already locked, since it is
 * better to create PID-file before writing something to the log, we need
 * to choose file descriptor for PID-file not equal to stdxxx.
 */
static void
create_pid_file(void)
{
	struct flock lock;
	int fd[4];
	unsigned int i;

	/*
	 * pid_file_fd should not be equal to file descriptor of stdin,
	 * stdout or stderr.  SUSv3 claims that open(2) returns lowest
	 * available descriptor, so four attempts are enough to get
	 * desired file descriptor.
	 */
	for (i = 0;; ++i) {
		fd[i] = open(pid_file_name, O_RDWR|O_CREAT, PID_FILE_PERM);
		if (fd[i] < 0)
			exit_err("create_pid_file: cannot create or open "
			    "PID-file: open(%s, O_RDWR|O_CREAT, 0%03o)",
			    pid_file_name, PID_FILE_PERM);
		if (fd[i] != STDIN_FILENO && fd[i] != STDOUT_FILENO &&
		    fd[i] != STDERR_FILENO) {
			pid_file_fd = fd[i];
			break;
		}
		if (i == 3)
			exit_err("create_pid_file: something is wrong "
			    "in algorithm");
	}

	/* Close "incorrectly" opened file descriptors. */
	while (i > 0) {
		--i;
		if (close(fd[i]) < 0)
			exit_err("create_pid_file: close(%d)", fd[i]);
	}

	lock.l_start = 0;		/* Zero offset from ... */
	lock.l_len = 0;			/* Entire file. */
	lock.l_type = F_WRLCK;		/* Should be a valid value. */
	lock.l_whence = SEEK_SET;	/* ... the start of the file. */

	if (fcntl(pid_file_fd, F_GETLK, &lock) < 0) {
#ifdef __CYGWIN__
		if (errno == ENOSYS)
			return;
#endif
		exit_err("create_pid_file: fcntl(%s, F_GETLK)",
		    pid_file_name);
	}

	switch (lock.l_type) {
	case F_UNLCK:
		break;
	case F_WRLCK:
	case F_RDLCK:
		exit_errx("create_pid_file: file %s is already locked (%s) by "
		    "PID %ld", pid_file_name, lock.l_type == F_WRLCK ?
		    "exclusive" : "shared", (long)lock.l_pid);
		/* NOTREACHED */
	default:
		exit_errx("create_pid_file: unknown lock type %d for file %s",
		    lock.l_type, pid_file_name);
	}
}

/*
 * Lock the file pid_file_name to prevent multiple copies of itself
 * from running and store PID in this file.
 */
static int
lock_pid_file(void)
{
	struct flock lock;
	ssize_t	n;
	int len;
	char num_str[22];

	lock.l_start = 0;		/* Zero offset from... */
	lock.l_len = 0;			/* Entire file. */
	lock.l_type = F_WRLCK;		/* Write lock. */
	lock.l_whence = SEEK_SET;	/* ... the start of the file. */

	if (fcntl(pid_file_fd, F_SETLK, &lock) < 0) {
		switch (errno) {
		case EACCES:
		case EAGAIN:
			logmsg(IPA_LOG_ERR, "lock_pid_file: cannot acquire "
			    "exclusive lock on file %s", pid_file_name);
			break;
		default:
			logmsg(IPA_LOG_ERR, "lock_pid_file: fcntl(%s, F_SETLK)",
			    pid_file_name);
		}
		return (-1);
	}

	if (ftruncate(pid_file_fd, (off_t)0) < 0) {
		logmsg(IPA_LOG_ERR, "lock_pid_file: ftruncate(%s, 0)",
		    pid_file_name);
		goto failed;
	}

	len = snprintf(num_str, sizeof(num_str), "%ld\n", (long)getpid());
	if (len < 0) {
		logmsg(IPA_LOG_ERR, "lock_pid_file: snprintf");
		goto failed;
	}
	if (len >= sizeof(num_str)) {
		logmsgx(IPA_LOG_ERR, "lock_pid_file: not enough space in "
		    "snprintf");
		goto failed;
	}

	n = write(pid_file_fd, num_str, len);
	if (n < 0) {
		logmsg(IPA_LOG_ERR, "lock_pid_file: write(%s)", pid_file_name);
		goto failed;
	}
	if (n != len) {
		logmsgx(IPA_LOG_ERR, "lock_pid_file: write(%s): short write, "
		    "%ld of %lu bytes", pid_file_name, (long)n,
		    (unsigned long)len);
		goto failed;
	}

	return (0);

failed:
	unlink_pid_file();
	return (-1);
}

#ifdef WITH_RULES

# define X_OPTSTRING_RULE "r:"

# ifdef WITH_LIMITS
#  define X_OPTSTRING_LIMIT "l:"
# else
#  define X_OPTSTRING_LIMIT ""
# endif

# ifdef WITH_SUBLIMITS
#  define X_OPTSTRING_SUBLIMIT "s:"
# else
#  define X_OPTSTRING_SUBLIMIT ""
# endif

# ifdef WITH_THRESHOLDS
#  define X_OPTSTRING_THRESHOLD "t:"
# else
#  define X_OPTSTRING_THRESHOLD ""
# endif

# define X_OPTSTRING ":" \
	X_OPTSTRING_RULE X_OPTSTRING_LIMIT X_OPTSTRING_SUBLIMIT \
	X_OPTSTRING_THRESHOLD

#else /* !WITH_RULES */

# define X_OPTSTRING ""

#endif /* WITH_RULES */

/*
 * Implement "-x ..." option.
 */
static int
run_opt_cmds(int argc, char *argv[])
{
	enum {
		SECT_STARTUP =		   0,
		SECT_SHUTDOWN,		/* 1 */
#if defined(WITH_RULES) && defined(WITH_LIMITS)
		SECT_REACH,		/* 2 */
		SECT_IF_REACHED,	/* 3 */
		SECT_IF_NOT_REACHED,	/* 4 */
#endif
		SECT_NOTSET
	};
	const char *section;
	const struct cmds *cmds;
#ifdef WITH_RULES
	const char *rule_name = NULL;
	const struct rule *rule;
	const struct cmds_rule *cmds_rule;
# ifdef WITH_LIMITS
	const char *subsection = NULL;
	const char *limit_name = NULL;
	const struct limit *limit;
	const struct cmds_limit *cmds_limit;
	unsigned int subsection_code;
# endif
# ifdef WITH_SUBLIMITS
	const char *sublimit_name = NULL;
	const struct sublimit *sublimit;
# endif
# ifdef WITH_THRESHOLDS
	const char *threshold_name = NULL;
	const struct threshold *threshold;
# endif
#endif /* WITH_RULES */
	unsigned int section_code;
	int opt;

	xvlogmsgx = vlogmsgx_stderr;

	while ((opt = getopt(argc, argv, X_OPTSTRING)) != -1)
		switch (opt) {
#ifdef WITH_RULES
		case ':':
			exit_errx("-x: option requires an argument -- %c",
			    optopt);
			/* NOTREACHED */
# ifdef WITH_LIMITS
		case 'l':
			if (limit_name != NULL)
				goto failed_duped;
			limit_name = optarg;
			if (validate_name(limit_name) < 0)
				exit_errx("-x: illegal %s name \"%s\"",
				    "limit", limit_name);
			break;
# endif
		case 'r':
			if (rule_name != NULL)
				goto failed_duped;
			rule_name = optarg;
			if (validate_name(rule_name) < 0)
				exit_errx("-x: illegal %s name \"%s\"",
				    "rule", rule_name);
			break;
# ifdef WITH_SUBLIMITS
		case 's':
			if (sublimit_name != NULL)
				goto failed_duped;
			sublimit_name = optarg;
			if (validate_name(sublimit_name) < 0)
				exit_errx("-x: illegal %s name \"%s\"",
				    "sublimit", sublimit_name);
			break;
# endif
# ifdef WITH_THRESHOLDS
		case 't':
			if (threshold_name != NULL)
				goto failed_duped;
			threshold_name = optarg;
			if (validate_name(threshold_name) < 0)
				exit_errx("-x: illegal %s name \"%s\"",
				    "threshold", threshold_name);
			break;
# endif
#endif /* WITH_RULES */
		case '?':
			exit_errx("-x: illegal option -- %c", optopt);
			/* NOTREACHED */
		default:
			exit_errx("-x: unexpected option -- %c", optopt);
		}

	if (optind == argc)
		exit_errx("-x: section name is absent");

	section = argv[optind++];

#ifdef WITH_RULES
# if defined(WITH_LIMITS) && defined(WITH_THRESHOLDS)
	if (limit_name != NULL && threshold_name != NULL)
		exit_errx("-x: -l and -t options cannot be used together");
# endif
# if defined(WITH_SUBLIMITS) && defined(WITH_THRESHOLDS)
	if (sublimit_name != NULL && threshold_name != NULL)
		exit_errx("-x: -s and -t options cannot be used together");
# endif
	if (rule_name != NULL) {
		rule = rule_by_name(rule_name);
		if (rule == NULL)
			exit_errx("-x: cannot find rule %s in "
			    "the configuration file", rule_name);
	}
# ifdef WITH_LIMITS
	if (limit_name != NULL) {
		if (rule_name == NULL)
			exit_errx("-x: the -l option should be used "
			    "with the -r option");
		limit = limit_by_name(rule, limit_name);
		if (limit == NULL)
			exit_errx("-x: cannot find limit %s for rule %s in "
			    "the configuration file", limit_name, rule_name);
	}
# endif /* WITH_LIMITS */
# ifdef WITH_SUBLIMITS
	if (sublimit_name != NULL) {
		if (limit_name == NULL)
			exit_errx("-x: the -s option should be used "
			    "with the -l option");
		sublimit = sublimit_by_name(limit, sublimit_name);
		if (sublimit == NULL)
			exit_errx("-x: cannot find sublimit %s for "
			    "limit %s in rule %s in the configuration file",
			    sublimit_name, limit_name, rule_name);
	}
# endif /* WITH_SUBLIMITS */
# ifdef WITH_THRESHOLDS
	if (threshold_name != NULL) {
		if (rule_name == NULL)
			exit_errx("-x: the -t option should be used "
			    "with the -r option");
		threshold = threshold_by_name(rule, threshold_name);
		if (threshold == NULL)
			exit_errx("-x: cannot find threshold %s for "
			    "rule %s in the configuration file",
			    threshold_name, rule_name);
	}
# endif /* WITH_THRESHOLDS */
#endif /* WITH_RULES */

	if (strcmp(section, "startup") == 0)
		section_code = SECT_STARTUP;
	else if (strcmp(section, "shutdown") == 0)
		section_code = SECT_SHUTDOWN;
	else
		section_code = SECT_NOTSET;

#ifdef WITH_RULES
# ifdef WITH_LIMITS
	subsection_code = SECT_NOTSET;
# endif
	if (optind != argc) {
# ifdef WITH_LIMITS
		if (optind + 1 != argc)
			goto failed_many;
		subsection = argv[optind];
		if (limit_name != NULL) {
			if (strcmp(subsection, "if_reached") == 0)
				subsection_code = SECT_IF_REACHED;
			else if (strcmp(subsection, "if_not_reached") == 0)
				subsection_code = SECT_IF_NOT_REACHED;
			else
				goto failed_subs;
		}
#  ifdef WITH_THRESHOLDS
		else if (threshold_name != NULL)
			goto failed_many;
#  endif
# else
		goto failed_many;
# endif /* WITH_LIMITS */
	}
# ifdef WITH_LIMITS
	if (section_code == SECT_NOTSET)
		if (strcmp(section, "reach") == 0)
			section_code = SECT_REACH;
# endif
# ifdef WITH_SUBLIMITS
	if (sublimit_name != NULL) {
		switch (section_code) {
		case SECT_STARTUP:
			cmds_limit = &sublimit->rc[RC_STARTUP];
			break;
		case SECT_SHUTDOWN:
			cmds_limit = &sublimit->rc[RC_SHUTDOWN];
			break;
		case SECT_REACH:
			if (subsection != NULL)
				goto failed_many;
			return (run_cmds_cons(&sublimit->reach,
			    "rule %s { limit %s { sublimit %s { reach {}}}}",
			    rule_name, limit_name, sublimit_name));
		default: /* SECT_NOTSET */
			goto failed_sect;
		}
		switch (subsection_code) {
		case SECT_IF_REACHED:
			cmds = &cmds_limit->cmds_reached;
			break;
		case SECT_IF_NOT_REACHED:
			cmds = &cmds_limit->cmds_not_reached;
			break;
		default: /* SECT_NOTSET */
			return (run_cmds_cons(&cmds_limit->cmds,
			    "rule %s { limit %s { sublimit %s { %s {}}}}",
			    rule_name, limit_name, sublimit_name, section));
		}
		return (run_cmds_cons(cmds,
		    "rule %s { limit %s { sublimit %s { %s { %s {}}}}}",
		    rule_name, limit_name, sublimit_name, section, subsection));
	}
# endif /* WITH_SUBLIMITS */
# ifdef WITH_LIMITS
	if (limit_name != NULL) {
		switch (section_code) {
		case SECT_STARTUP:
			cmds_limit = &limit->rc[RC_STARTUP];
			break;
		case SECT_SHUTDOWN:
			cmds_limit = &limit->rc[RC_SHUTDOWN];
			break;
		default:
			if (subsection != NULL)
				goto failed_many;
			if (section_code == SECT_REACH)
				cmds = &limit->reach;
			else if (strcmp(section, "expire") == 0)
				cmds = &limit->expire.cmds;
			else if (strcmp(section, "restart") == 0)
				cmds = &limit->restart.cmds;
			else
				goto failed_sect;
			return (run_cmds_cons(cmds,
			    "rule %s { limit %s { %s {}}}",
			    rule_name, limit_name, section));
		}
		switch (subsection_code) {
		case SECT_IF_REACHED:
			cmds = &cmds_limit->cmds_reached;
			break;
		case SECT_IF_NOT_REACHED:
			cmds = &cmds_limit->cmds_not_reached;
			break;
		default: /* SECT_NOTSET */
			return (run_cmds_cons(&cmds_limit->cmds,
			    "rule %s { limit %s { %s {}}}",
			    rule_name, limit_name, section));
		}
		return (run_cmds_cons(cmds,
		    "rule %s { limit %s { %s { %s {}}}}",
		    rule_name, limit_name, section, subsection));
	}
# endif /* WITH_LIMITS */
# ifdef WITH_THRESHOLDS
	if (threshold_name != NULL) {
		switch (section_code) {
		case SECT_STARTUP:
			cmds = &threshold->rc[RC_STARTUP];
			break;
		case SECT_SHUTDOWN:
			cmds = &threshold->rc[RC_SHUTDOWN];
			break;
		default:
			if (strcmp(section, "below_threshold") == 0)
				cmds = &threshold->below_thr;
			else if (strcmp(section, "equal_threshold") == 0)
				cmds = &threshold->equal_thr;
			else if (strcmp(section, "above_threshold") == 0)
				cmds = &threshold->above_thr;
			else
				goto failed_sect;
		}
		return (run_cmds_cons(cmds,
		    "rule %s { threshold %s { %s {}}}",
		    rule_name, threshold_name, section));
	}
# endif /* WITH_THRESHOLDS */
	if (rule_name != NULL) {
		switch (section_code) {
		case SECT_STARTUP:
			cmds_rule = &rule->rc[RC_STARTUP];
			break;
		case SECT_SHUTDOWN:
			cmds_rule = &rule->rc[RC_SHUTDOWN];
			break;
		default:
			goto failed_sect;
		}
# ifdef WITH_LIMITS
		if (subsection != NULL) {
			if (strcmp(subsection, "if_any_reached") == 0)
				cmds = &cmds_rule->cmds_any_reached;
			else if (strcmp(subsection, "if_any_not_reached") == 0)
				cmds = &cmds_rule->cmds_any_not_reached;
			else if (strcmp(subsection, "if_all_reached") == 0)
				cmds = &cmds_rule->cmds_all_reached;
			else if (strcmp(subsection, "if_all_not_reached") == 0)
				cmds = &cmds_rule->cmds_all_not_reached;
			else
				goto failed_subs;
			return (run_cmds_cons(cmds,
			    "rule %s { %s { %s {}}}",
			    rule_name, section, subsection));
		}
# endif /* WITH_LIMITS */
		return (run_cmds_cons(&cmds_rule->cmds,
		    "rule %s { %s {}}", rule_name, section));
	}
#endif /* WITH_RULES */

	if (optind != argc)
		goto failed_many;

	switch (section_code) {
	case SECT_STARTUP:
		cmds = &cmds_global[RC_STARTUP];
		break;
	case SECT_SHUTDOWN:
		cmds = &cmds_global[RC_SHUTDOWN];
		break;
	default:
		goto failed_sect;
	}
	return (run_cmds_cons(cmds, "%s {}", section));

#ifdef WITH_RULES
failed_duped:
	exit_errx("-x: duplicated option -- %c", optopt);
#endif

failed_many:
	exit_errx("-x: too many sections for the given options");

failed_sect:
	exit_errx("-x: unknown section \"%s\" for the given options", section);

#if defined(WITH_RULES) && defined(WITH_LIMITS)
failed_subs:
	exit_errx("-x: unknown subsection \"%s\" for the given options",
	    subsection);
#endif
}

/*
 * Implement "-u user" and "-g group" options.
 */
static void
change_user(const char *u_name, const char *g_name)
{
	const struct passwd *pwd;
	const struct group *grp;
	char *endptr;
	uid_t uid;
	gid_t gid;

	if (u_name == NULL && g_name == NULL)
		return;

	xvlogmsgx = vlogmsgx_stderr;

	if (g_name != NULL) {
		/* -g group */
		errno = 0;
		grp = getgrnam(g_name);
		if (grp == NULL) {
			if (errno != 0)
				exit_err("change_user: getgrnam(%s)", g_name);
			if (!isdigit((unsigned char)*g_name))
				exit_errx("cannot find group %s", g_name);
			errno = 0;
			gid = (gid_t)strtoul(g_name, &endptr, 10);
			if (errno != 0)
				exit_err("change_user: strtoul");
			if (g_name == endptr || *endptr != '\0')
				exit_errx("cannot find group %s", g_name);
		} else
			gid = grp->gr_gid;
		if (setgroups(1, &gid) < 0)
			exit_err("change_user: setgroups(1, [%lu])",
			    (unsigned long)gid);
	}

	if (u_name != NULL) {
		/* -u user */
		errno = 0;
		pwd = getpwnam(u_name);
		if (pwd == NULL) {
			if (errno != 0)
				exit_err("change_user: getpwnam(%s)", u_name);
			if (!isdigit((unsigned char)*u_name))
				exit_errx("cannot find user %s", u_name);
			if (g_name == NULL)
				exit_errx("argument in the -u option is not "
				    "a valid user name");
			errno = 0;
			uid = (uid_t)strtoul(u_name, &endptr, 10);
			if (errno != 0)
				exit_err("change_user: strtoul");
			if (u_name == endptr || *endptr != '\0')
				exit_errx("cannot find user %s", u_name);
		} else {
			uid = pwd->pw_uid;
			if (g_name == NULL) {
				gid = pwd->pw_gid;
				if (setsuppgids(pwd->pw_name, gid) < 0)
					exit_errx("change_user: cannot set "
					    "all groups for user %s", u_name);
			}
		}
	}

	if (setgid(gid) < 0)
		exit_err("change_user: setgid(%lu)", (unsigned long)gid);

	if (u_name != NULL && setuid(uid) < 0)
		exit_err("change_user: setuid(%lu)", (unsigned long)uid);

	/*
	 * To be sure, that descriptors used by getpw*() and getgr*()
	 * functions are closed.
	 */
	endpwent();
	endgrent();
}

int
main(int argc, char *argv[])
{
	const char *k_optarg;		/* -k signal */
	const char *u_name;		/* -u user */
	const char *g_name;		/* -g group */
	int opt;			/* Current option. */
	int rv;				/* Return value. */
	int action;			/* Return code from ipa_main(). */
	char t_flag;			/* Set if -t. */
	char x_flag;			/* Set if -x. */

	/* Save the program name. */
	envprogname = strrchr(argv[0], '/');
	if (envprogname != NULL)
		++envprogname;
	else
		envprogname = argv[0];

	k_optarg = u_name = g_name = NULL;
	t_flag = x_flag = 0;
	opterr = 0;
	while ((opt = getopt(argc, argv, ":c:df:g:hi:k:o:p:tu:vx")) != -1)
		switch (opt) {
		case ':':
			exit_errx("option requires an argument -- %c", optopt);
			/* NOTREACHED */
		case '?':
			exit_errx("illegal option -- %c", optopt);
			/* NOTREACHED */
		case 'c':
			if (*optarg != '/')
				exit_errx("path in the -%c option should be "
				    "absolute", 'c');
			if (chroot(optarg) < 0)
				exit_err("main: chroot(%s)", optarg);
			break;
		case 'd':
			debug = 1;
			break;
		case 'f':
			ipa_conf_file = optarg;
			break;
		case 'g':
			g_name = optarg;
			break;
		case 'h':
			usage();
			return (EXIT_SUCCESS);
		case 'i':
			log_ident = optarg;
			break;
		case 'k':
			k_optarg = optarg;
			break;
		case 'o':
			log_file = optarg;
			if (*log_file != '/')
				exit_errx("path in the -%c option should be "
				    "absolute", 'o');
			break;
		case 'p':
			pid_file_name = optarg;
			if (*pid_file_name != '/')
				exit_errx("path in the -%c option should be "
				    "absolute", 'c');
			break;
		case 't':
			if (t_flag)
				mimic_real_config = 1;
			else
				t_flag = 1;
			break;
		case 'u':
			u_name = optarg;
			break;
		case 'v':
			show_version();
			return (EXIT_SUCCESS);
		case 'x':
			x_flag = 1;
			goto getopt_done;
		default:
			exit_errx("unexpected option -- %c", optopt);
		}

	if (optind < argc)
		exit_errx("non-switch argument \"%s\"", argv[optind]);

getopt_done:
	change_user(u_name, g_name);

	if (k_optarg != NULL) {
		const struct sig_descr *sig;

		for (sig = sig_descr_tbl; sig->signame != NULL; ++sig)
			if (strcmp(k_optarg, sig->signame) == 0) {
				sendsig(sig->signo);
				return (EXIT_SUCCESS);
			}
		exit_errx("illegal argument \"%s\" for the -k option",
		    k_optarg);
	}

#ifdef HAVE_DL_INIT
	if (dl_init() != 0)
		exit_errx("main: dl_init failed");
#endif

	if (setlocale(LC_ALL, "") == NULL)
		exit_err("setlocale");

	memfunc_pre_init();

	myuid = getuid();
	mygid = getgid();

	if (t_flag) {
		if (configure(TEST_PARSING) < 0)
			return (EXIT_FAILURE);
		show_config();
		return (EXIT_SUCCESS);
	}

	if (x_flag) {
		if (configure(CMD_PARSING) < 0)
			return (EXIT_FAILURE);
		return (run_opt_cmds(argc, argv) < 0 ?
		    EXIT_FAILURE : EXIT_SUCCESS);
	}

	create_pid_file();

	tzset();

	init_log();
	open_log();

	if (fd_init() < 0) {
		logbt("main");
		return (EXIT_FAILURE);
	}

	logmsgx(IPA_LOG_INFO, IPA_NAME " (version " PACKAGE_VERSION ") "
	    "started by UID %lu, GID %lu", (unsigned long)myuid,
	    (unsigned long)mygid);
#ifdef WITH_PTHREAD
	logmsgx(IPA_LOG_INFO, "thread-safe mode");
#endif
#ifdef WITH_MEMFUNC_DEBUG
	logmsgx(IPA_LOG_INFO, "memfunc debugging is enabled");
#endif

	rv = EXIT_FAILURE;

	if (!debug) {
		if (bg_init() < 0) {
			logmsgx(IPA_LOG_ERR, "cannot run in the background");
			goto out_exit;
		}
		logmsgx(IPA_LOG_INFO, "started in background mode");
	} else
		logmsgx(IPA_LOG_INFO, "started in foreground mode");

	if (lock_pid_file() < 0) {
		logmsgx(IPA_LOG_ERR, "cannot lock PID-file");
		goto out_exit;
	}

	logmsgx(IPA_LOG_INFO, "maximum number of open descriptors is %d%s",
	    nofile_max, nofile_max >= 0 ? "" : " (no limit)");

	if (configure(CONFIG_PARSING) < 0) {
		logmsgx(IPA_LOG_ERR, "main: configure failed");
		goto out_exit_unlink;
	}

	/* Startup. */
	if (sig_init_1() < 0)
		goto out_deinit;

	logmsgx(IPA_LOG_INFO, "startup... all signals are blocked");

	if (init_all() < 0)
		goto out_deinit;

	if (run_all_cmds(RC_STARTUP) < 0)
		goto out_deinit;

	/*
	 * If second stage of signal initialization succeeded,
	 * then begin to work.
	 */
	if (sig_init_2() == 0) {
		/* Working... */
		for (;;) {
			action = ipa_main();
			if (action < 0) {
				/* Some error occurred. */
				break;
			}
			if (action == 0) {
				/* Normal exit. */
				rv = EXIT_SUCCESS;
				break;
			}

			/* Reconfiguration. */
			rv = EXIT_SUCCESS;
			if (deinit_all() < 0)
				rv = EXIT_FAILURE;
			if (free_all() < 0)
				rv = EXIT_FAILURE;
			if (rv != EXIT_SUCCESS)
				goto out_exit_unlink;
			rv = EXIT_FAILURE;
			if (configure(RECONFIG_PARSING) < 0)
				goto out_exit_unlink;

			/*
			 * Now initialize everything and start to work
			 * with new configuration.
			 */
			logmsgx(IPA_LOG_INFO, "using new configuration");
			if (init_all() < 0)
				goto out_deinit;
		}
	}

	/* Shutdown. */
	logmsgx(IPA_LOG_INFO, "shutdown... all signals are ignored");
	if (run_all_cmds(RC_SHUTDOWN) < 0)
		rv = EXIT_FAILURE;

out_deinit:
	/* Deiniting. */
	if (deinit_all() < 0)
		rv = EXIT_FAILURE;
	if (free_all() < 0)
		rv = EXIT_FAILURE;

out_exit_unlink:
	/* Unlink PID file, since it is locked by this process. */
	unlink_pid_file();

out_exit:
	/* Exiting. */
#ifdef HAVE_DL_EXIT
	if (dl_exit() != 0) {
		logmsgx(IPA_LOG_ERR, "main: dl_exit failed");
		rv = EXIT_FAILURE;
	}
#endif
	log_stdall();
	logmsgx(IPA_LOG_INFO, IPA_NAME " (version " PACKAGE_VERSION ") "
	    "exited, return code is %d", rv);

	close_log();
	return (rv);
}
