/*-
 * Copyright (c) 2003-2004 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *  @(#)$Id: memfunc.h,v 1.2 2011/01/23 18:42:35 simon Exp $
 */

#ifndef IPA_MEMFUNC_H
#define IPA_MEMFUNC_H

#include <limits.h>

/*
 * Macros for bitmap manipulation, all bitmaps are arrays of BITMAP_TYPE.
 * sizeof(BITMAP_TYPE) must be <= sizeof(int).  BITMAP_TYPE can be
 * 'unsigned int', 'unsigned short' or 'unsigned char', bigger size of
 * BITMAP_TYPE speeds up search of free items, but wastes memory for
 * not needed items in actual arrays in marrays.
 */

/* Type of bitmap word, do not forget to update BITMAP_ALL_BITS. */
#define BITMAP_TYPE \
	unsigned int

/* All bits in one bitword (it is not portable to use (BITMAP_TYPE)-1). */
#define BIT_WORD_ALL_BITS \
	UINT_MAX

/* Number of bits in BITMAP_TYPE. */
#define BIT_WORD_NBITS \
	(sizeof(BITMAP_TYPE) * CHAR_BIT)

/* Return size of bitmap in bytes. */
#define BITMAP_SIZE(n) \
	(sizeof(BITMAP_TYPE) * (((n) + BIT_WORD_NBITS - 1) / BIT_WORD_NBITS))

/* Mask of a bit. */
#define BIT_MASK(idx) \
	(1 << ((idx) & (BIT_WORD_NBITS - 1)))

/* Offset of bitword with bit for the given index. */
#define BIT_SLOT(idx) \
	((idx) / BIT_WORD_NBITS)

/* Return pointer to a bitword in map with the given index. */
#define BIT_WORD(map, idx) \
	((map) + BIT_SLOT(idx))

/* Set bit. */
#define BIT_SET(bitword, bitmask) \
	(*(bitword) |= (bitmask))

/* Clear bit. */
#define BIT_CLEAR(bitword, bitmask) \
	(*(bitword) &= ~(bitmask))

/* Test bit. */
#define BIT_TEST(bitword, bitmask) \
	(*(bitword) & (bitmask))

extern ipa_memfunc memfunc;

extern int	memfunc_init(void);
extern void	memfunc_pre_init(void);
extern void	memfunc_deinit_1(int);
extern void	memfunc_deinit_2(int);

extern void	(*mvlogmsgx)(const char *, va_list);

extern ipa_mem_type *mem_type_new_local(const char *, const char *,
		    unsigned int);

extern void	*mem_malloc(size_t, ipa_mem_type *);
extern void	*mem_calloc(size_t, size_t, ipa_mem_type *);
extern void	*mem_realloc(void *, size_t, ipa_mem_type *);
extern char	*mem_strdup(const char *, ipa_mem_type *);
extern void	mem_free(void *, ipa_mem_type *);

extern int	mem_asprintf(ipa_mem_type *, char **, const char *, ...)
		    ATTR_FORMAT(printf, 3, 4);
extern int	mem_vasprintf(ipa_mem_type *, char **, const char *, va_list)
		    ATTR_FORMAT(printf, 3, 0);

extern ipa_marray *marray_init(const char *, const char *, unsigned int,
		    void **, size_t, unsigned int, unsigned int);
extern void	marray_deinit(ipa_marray *);
extern int	marray_alloc(ipa_marray *, unsigned int *, int);
extern void	marray_free(ipa_marray *, unsigned int);
extern void	marray_minimize(ipa_marray *);
extern int	marray_check_index(ipa_marray *, unsigned int);
extern unsigned int marray_nused(ipa_marray *);

#define marray_is_empty(x) (marray_nused(x) == 0)

extern ipa_mzone *mzone_init(const char *, const char *, unsigned int, size_t,
		    unsigned int, unsigned int);
extern void	mzone_deinit(ipa_mzone *);
extern void	*mzone_alloc(ipa_mzone *);
extern void	mzone_free(ipa_mzone *, void *);
extern unsigned int mzone_nused(ipa_mzone *);

#define mzone_is_empty(x) (mzone_nused(x) == 0)

extern int	memfunc_get_stat(void **, size_t *);

#endif /* !IPA_MEMFUNC_H */
